# Data files used for this analysis:
# Blindcoding.csv
# Blindcoding_original_coder.csv
# bias_block10.csv
# DetourAndBiasData.csv
# DetourRawPhase3.csv

#########################################################
#                 set working directory                 #
#########################################################

setwd()

#########################################################
#                     load packages                     #
#########################################################

library("cogdat")
library("lme4")
library("lmerTest")
library("markovchain")
library("MASS")
library("ggplot2")
library("psy")

#########################################################
#      Kohens kappa for inter-observer reliability      #
#########################################################

# load and transpose data from independent rater
data_independent_coder <- read.csv("Data/Blindcoding.csv", 
                                   header = FALSE, colClasses = "character")
proc_data_independent_coder <- processDat(data_independent_coder)
# check data
str(proc_data_independent_coder)

### Process dataset
# remove unneeded column and change column 1s name to PIT
proc_data_independent_coder$Date <- NULL
colnames(proc_data_independent_coder)[1] <- "PIT"

# change all columns to factors
proc_data_independent_coder <- as.data.frame(unclass(proc_data_independent_coder))

# change correct and latencies to numeric
proc_data_independent_coder$correct <- as.numeric(
                                        as.character(proc_data_independent_coder$correct))
proc_data_independent_coder$latency <- as.numeric(
                                        as.character(proc_data_independent_coder$latency))
proc_data_independent_coder$latency.trial <- as.numeric(
                                        as.character(proc_data_independent_coder$latency.trial))
# change date to date-format
proc_data_independent_coder$date <- as.Date(proc_data_independent_coder$date, "%d/%m/%Y")

# check data
summary(proc_data_independent_coder)

# load and transpose data from original coder
data_original_coder <- read.csv("Data/Blindcoding_original_coder.csv", 
                                header = FALSE, colClasses = "character")
proc_data_original_coder <- processDat(data_original_coder)
# check data
str(proc_data_original_coder)

### Process dataset
# remove unneeded column and change column 1s name to PIT
proc_data_original_coder$Date <- NULL
colnames(proc_data_original_coder)[1] <- "PIT"

# change all columns to factors
proc_data_original_coder <- as.data.frame(unclass(proc_data_original_coder))

# change correct and latencies to numeric
proc_data_original_coder$correct <- as.numeric(
                                      as.character(proc_data_original_coder$correct))
proc_data_original_coder$latency <- as.numeric(
                                      as.character(proc_data_original_coder$latency))
proc_data_original_coder$latency.trial <- as.numeric(
                                      as.character(proc_data_original_coder$latency.trial))
# change date to date-format
proc_data_original_coder$date <- as.Date(proc_data_original_coder$date, "%d/%m/%Y")

# check data
summary(proc_data_original_coder)


##### calculate interobserver reliability (cohens kappa for 2 raters) 
##### for first 60 trials of Stimulus training batch 1

# create dataframe with both original and independent raters data
kappa <- as.data.frame(cbind(proc_data_independent_coder$correct, 
                             proc_data_original_coder$correct, 
                             proc_data_original_coder$researcher))
# change column names (original coder: rater 1: BS, rater 2: ML)
colnames(kappa) <- c("independent.coder","original.coder", "rater")
# check if correct data were selected
head(kappa)
# data from ML only
SM <- subset(kappa, kappa$rater == 2)
# interobserver reliability between independent rater SH and original coder ML
ckappa(SM)
# cohens kappa = 0.8683205

#########################################################
#           Visual discrimination learning              #
#########################################################

#### functions
# calculate Bias Index
bias_index <- function(N_right, N_left) {
  BI <- (N_right - N_left) / (N_right + N_left)
  return(BI)
}

# calculates the sum of the probability for making a right or left choice at time t+1
choice.bias <- function(markovchain_prop){
  trMatrix_est_side_sum <- data.frame(matrix(0, ncol = 2, nrow = 4))
  # change row and column names
  colnames(trMatrix_est_side_sum) <- c("left","right")
  rownames(trMatrix_est_side_sum) <- c("left non-rewarded","left rewarded","right non-rewarded","right rewarded")
  
  # calculate sums for probability of right/left choice at t
  for(i in 1:4){
    trMatrix_est_side_sum[i,1] <- markovchain_prop$estimate@transitionMatrix[i,1] + markovchain_prop$estimate@transitionMatrix[i,2]
    trMatrix_est_side_sum[i,2] <- markovchain_prop$estimate@transitionMatrix[i,3] + markovchain_prop$estimate@transitionMatrix[i,4]
  }
  
  # calculate upper CIs
  for(i in 1:4){
    trMatrix_est_side_sum$upper_left[i] <- markovchain_prop$upperEndpointMatrix[i,1] + markovchain_prop$upperEndpointMatrix[i,2]
    trMatrix_est_side_sum$upper_right[i] <- markovchain_prop$upperEndpointMatrix[i,3] + markovchain_prop$upperEndpointMatrix[i,4]
  }
  
  # calculate lower CIs
  for(i in 1:4){
    trMatrix_est_side_sum$lower_left[i] <- markovchain_prop$lowerEndpointMatrix[i,1] + markovchain_prop$lowerEndpointMatrix[i,2]
    trMatrix_est_side_sum$lower_right[i] <- markovchain_prop$lowerEndpointMatrix[i,3] + markovchain_prop$lowerEndpointMatrix[i,4]
  }
  
  # return table
  return(trMatrix_est_side_sum)
  
}
# calculate mean difference between two groups
mean_diff <- function(data){
  grp1 <- data[data$learnt == "y", ]
  grp2 <- data[data$learnt == "n", ]
  
  mean_diff <- mean(grp2$BI_ST1_crit)-mean(grp1$BI_ST1_crit)
  
  return(mean_diff)
}


##### Block Bias Index

# load data
bias_block10 <- read.csv("Data/bias_block10.csv")
# check data
str(bias_block10)

# calculate Bias Index for each block of 10 trials
bias_block10$BI <- bias_index(bias_block10$Nright, bias_block10$Nleft)
# remove NAs
bias_block10 <- bias_block10[complete.cases(bias_block10$BI), ]

# density function
plot(density(bias_block10$BI))

#### Did lizards develop a bias or did they have a pre-existing bias?
model <- lmer(BI ~ block + (1 + block|PIT), data = bias_block10)
summary(model)
#              Estimate Std. Error  df       t value  Pr(>|t|)
# (Intercept)  0.069935   0.115934 18.232000   0.603    0.554
# block        0.003594   0.017275 14.784000   0.208    0.838
# the estimate is close to 0 indicating and not significant 
# indicating that there was no change over time


##### Plot Figure 2A

# get predicted values from the model
bias_block10$pred_val<-predict(model)
# raw values
plot(bias_block10$pred_val~bias_block10$block)

# use block and the predicted values from data frame bias_block10
plot1<-ggplot(bias_block10, aes(block, pred_val)) + 
  # draw linear function in black
  stat_smooth(method="lm", colour=c("#000000")) + 
  # define limits of y-axis
  coord_cartesian(ylim = c(-0.4, 0.4))  +
  # define limits, breaks and labels on x-axis
  scale_x_discrete(breaks=c(1:9), labels=c(1:9), limits=c(1:9)) +
  # axis names
  labs(x = "Block of 10 trials", y = "Bias Index") +
  # white background
  theme_bw() +
  # no grid lines, x-axis label moved further away from tick labels and font size 12pt
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(), axis.title.x = element_text(vjust = -30), text = element_text(size=12))
# save plot
ggsave(filename="Figures/BlockBI_ggplot.tiff", scale = 1, width = 10, height = 10, units = "cm", dpi = 500)



##### Lerners/Non-learners Bias Index

# load data
bias_data <- read.csv("Data/DetourAndBiasData.csv")
# check data
str(bias_data)
# remove spelling errors
names(bias_data)[11] <- c("learnt")
names(bias_data)[6] <- c("association.time")
names(bias_data)[7] <- c("association.frequ")

# calculate BI for Stimulu training batch 1
bias_data$BI_ST1_crit <- bias_index(bias_data$SD1_N_crit_r, bias_data$SD1_N_crit_l)

#### Do "initial learners" and "non-learners" differ in their Bias Index?

# empty vector
means <- c()
for(i in 1:1000){
  # split data into learners and non-learners
  splt <- split(bias_data, bias_data$learnt)
  # pick random rows to sample
  random <- do.call(rbind, lapply(splt, function(x) x[sample(nrow(x), replace = TRUE),]))
  # calculate mean difference for each sample
  means[i] <- mean_diff(random)
}

# Get the 95% confidence intervals of the distribution of contrasts
CI <- quantile(t, c(0.025, 0.975))
#      2.5%     97.5% 
# -0.671535 -0.138180 
# CI does not overlapp 0, difference is significant




##### Plot Figure 2B

# subset into learners and non-learners
none_learner_BI <- subset(bias_data$BI_ST1_crit, bias_data$learnt=="n")
learner_BI <- subset(bias_data$BI_ST1_crit, bias_data$learnt=="y")

mean <- aggregate(bias_data$BI_ST1_crit, list(bias_data$learnt), mean)
BI_up_ci_n  <-mean(none_learner_BI) + (1.96 * (sd(none_learner_BI) / sqrt(length(none_learner_BI))))
BI_low_ci_n <-mean(none_learner_BI) - (1.96 * (sd(none_learner_BI) / sqrt(length(none_learner_BI))))
BI_up_ci_l  <-mean(learner_BI) + (1.96 * (sd(learner_BI) / sqrt(length(learner_BI))))
BI_low_ci_l <-mean(learner_BI) - (1.96 * (sd(learner_BI) / sqrt(length(learner_BI))))
mean$ci_up<-rbind(BI_up_ci_n, BI_up_ci_l)
mean$ci_low<-rbind(BI_low_ci_n, BI_low_ci_l)

# use the calculated mean values for learners and non-learners
plot2<-ggplot(mean, aes(x = Group.1, y = x)) +
  # draw a bar chart in light grey with black borders
  geom_bar(stat = "identity", fill = "lightgrey", colour="black") +
  # add error bars based on the 95% confidence intervals with no whiskers
  geom_errorbar(aes(ymax = ci_up, ymin = ci_low), width = 0, data = mean) +
  # add a line at 0
  geom_hline(yintercept = 0) +
  # set limits of the y-axis
  coord_cartesian(ylim = c(-0.8, 0.2)) +
  # axis labels
  labs(x = "", y = "Bias Index") +
  # force order of categories on x-axis and rename
  scale_x_discrete(name ="", limits = c("y","n"), labels = c("Learner", "Non-learner")) +
  # set y-axis limits, breaks and labels
  scale_y_continuous(breaks=c(-0.8,-0.6,-0.4,-0.2,0,0.2,0.4), labels=c(-0.8,-0.6,-0.4,-0.2,0,0.2,0.4), limits=c(-0.8,0.4)) +
  # white background
  theme_bw() +
  # no grid lines and font size 12pt
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(), text = element_text(size = 12))
# save plot
ggsave(filename="Figures/BILearnerNonLearner_ggplot.tiff", scale = 1, width = 10, height = 10, units = "cm", dpi = 500)




##### Plot Figure 2C
##### probability to make a left/right choice after being rewarded or not

# load data
DataLR <- read.csv("Data/DataDF.csv")
# check data
str(DataLR)

# merge side choice and outcome together in a single column
DataLR$SideXOutc <- paste0( DataLR [ ,"Side"], DataLR [ ,"Outcome"] )

# Run the descrete time marcov chain (DTMC)
trMatrix <- markovchainFit(DataLR$SideXOutc)

# create probability table with the function above
group_prop <- choice.bias(trMatrix)

# check raw counts to evaluate how many events of each state are present
# and gauge our confidence in the matrices
createSequenceMatrix(DataLR$SideXOutc)

# add column with rownames for plot-table
group_prop$t <- c("left non-rewarded","left rewarded","right non-rewarded","right rewarded")
# get left and right choices together in table to plot with barplot
right <- xtabs(group_prop$right ~ group_prop$t)
UP    <- xtabs(group_prop$upper_right ~ group_prop$t)
LOW   <- xtabs(group_prop$lower_right ~ group_prop$t)
# number of events the Markov Chain is based on
events <- c(128, 162, 223, 134)

### individual datapoints to plot on top of bars

# retrieve animal IDs
PIT <- as.character(unique(DataLR$PIT))
# dimensions for array to save individual data
dim_pit <- length(unique(PIT))
dim_col <- length(colnames(group_prop)) - 1
dim_row <- length(rownames(group_prop))
# retrieve names for dimesions
dim_col_names <- c("left", "right", "upper_left", "upper_right", "lower_left", "lower_right")
dim_row_names <- c("left non-rewarded","left rewarded","right non-rewarded","right rewarded")
# get propabilities for each individual separately
individual_prop<-by(DataLR$SideXOutc, DataLR$PIT, markovchainFit)
# create array with appropriate dimensions and dimension-names
individual_prop_array <-array(0, dim = c(dim_row, dim_col, dim_pit), 
                              dimnames = list(dim_row_names, dim_col_names, PIT))
# fill array with row sum probabilities
for(l in 1:length(PIT)){
  for(i in 1:6){
    for(k in 1:4){
      individual_prop_array[k,i,PIT[l]] <- choice.bias(individual_prop[[PIT[l]]])[k,i]
    }
  }
}

## Plot as tiff, 800x600, point size 25 and no compression
tiff('Figures/prop_side_choice3.tiff', width = 800, height = 600, 
     pointsize = 25, compression = 'none')
# change margins and increase linewidth
par(lwd = 3, mar = c(4,4,1,1))
# plot bars next to each other with no x-axis in colour lightgrey and horizontal numbering
xx <- barplot(right, ylim = c(0,1.1), lwd = 3, las = 2, xaxt = "n", col="lightgrey")
# add y-axis lable
mtext("Probability of right choice at trial t+1", side = 2, line = 2.5)
# add x-axis
axis(1, at = -1:30, lwd = 3, lwd.ticks = 0, labels = FALSE)
# add x-axis lables
xlable <- c('left',' ','right',' ')
text(cex = 1, x = xx+0.6 , y = -0.1, xlable, xpd = TRUE, srt = 0)
xlable2 <- c('-','+','-','+')
text(cex = 1, x = x , y = -0.05, xlable2, xpd = TRUE, srt = 0)
mtext("Choice at trial t", side = 1, line = 2.5)
# add 95% confidence intervals
arrows(x0 = xx, x1 = xx, y0 = right, y1 = UP, angle = 90, length = 0.05)
arrows(x0 = xx, x1 = xx, y0 = right, y1 = LOW, angle = 90, length = 0.05)
# add events
text(cex = 1, x = xx, y = 0.05, events, xpd = TRUE, srt = 0)
# add a 50% line, dotted
abline(0, 0, 0.5, lty = 2)
# add new plot on top of previous plot for individual datapoints
par(new = TRUE, lwd = 3, mar = c(4,4,1,1))
# plot empty plot
barplot(right, ylim = c(0,1.1), lwd = 3, las = 2, col = NA, 
        border = NA, xlab = "", ylab = "", xaxt = "n", yaxt = "n")
# add individual data points for each PIT
for(i in 1:length(PIT)){
  points(individual_prop_array[,2,PIT[i]] ~ xx, col = rgb(0, 0, 0, alpha = 0.4))
}
# end plotting
dev.off()



#########################################################
#                 Response inhibition                   #
#########################################################

# read data file
detour <- read.csv("Data/DetourAndBiasData.csv")
# check data formats
str(detour)
# change to factor
detour$PIT <- factor(detour$PIT)
# remove spelling errors
names(detour)[11] <- c("learnt")
names(detour)[6] <- c("association.time")
names(detour)[7] <- c("association.frequ")
# check data
summary(detour)
# remove NAs
detour_noNA <- detour[complete.cases(detour$ttc), ]

#### Is association time correlated with the time to reach criterion in phase 2,
#### the number of correct detours in phase 3 or sex 
model_association <- lmer(log(association.time) ~ ttc + Sex + score + (1|association.frequ), 
                          data = detour_noNA)
# check if residual distribution is normal
qqnorm(resid(model_association))
qqline(resid(model_association))

summary(model_association)
# no significant correlations


#### Are faster learners better or worse at inhibiting direct responses towards a visible reward?
#### Is sex a predictor for trials to criterion (phase 2) or score (phase 3)? 
#### Is there a difference in inhibitory control (cylinder task) 
#### between "initial learners" and "non-learners"?
model_ttc_nb <- glm.nb(ttc ~ score + Sex + Sex:score + learnt:score, 
                       data = detour_noNA)
summary(model_ttc_nb)
# "initial learners" and "non-learners" differ in their performance during phase 3


#### Are lizards that received more trials during the discrimination learning task
#### better at detouring the transparent cylidner?

corr_trial_score <- cor.test(detour_noNA$sum,detour_noNA$score, method='pearson')
# t = -0.2113, df = 14, p-value = 0.8357
# rs=-0.05638334 

##### Plot Figure 2D

# use the score and categorical variable learnt (yes or no) from data.frame detour_noNA
plot3<-ggplot(detour_noNA, aes(x=learnt, y=score)) + 
  # draw a boxplot in light grey
  geom_boxplot(fill="lightgrey") +
  # set y-axis limits
  coord_cartesian(ylim = c(0, 10))  +
  # axis labels
  labs(x = "", y = "Number of correct detours (score)") +
  # force order of categories on x-axis and rename
  scale_x_discrete(name ="", limits=c("y","n"), labels=c("Learner", "Non-learner")) +
  # set y-axis limits, breaks and labels
  scale_y_continuous(breaks=c(0:10), labels=c(0:10), limits=c(0:10)) +
  # white background
  theme_bw() +
  # no grid lines and font size 12pt
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(), text = element_text(size=12))
# finish plotting
ggsave(filename="Figures/ScoreLearnerNonlearner_ggplot.tiff", scale = 1, width = 10, height = 10, units = "cm", dpi = 500)


##### Plot Figure 3

# read detour data file
detour_cum_plot_data <- read.csv("Data/DetourRawPhase3.csv", 
                                 header = FALSE, colClasses = "character")
# transpose raw data and remove empty column date
proc_detour_cum_plot_data <- processDat(detour_cum_plot_data)
proc_detour_cum_plot_data$date <- NULL
# check data
str(proc_detour_cum_plot_data)

# colomn ID is chaned to PIT and made factor
colnames(proc_detour_cum_plot_data)[1] <- "PIT"
proc_detour_cum_plot_data$PIT <- factor(proc_detour_cum_plot_data$PIT)
# sex is changed to factor
proc_detour_cum_plot_data$Sex <- factor(proc_detour_cum_plot_data$Sex)
# column channel and treatment are removed (not needed)
proc_detour_cum_plot_data$Chan <- NULL
proc_detour_cum_plot_data$trt  <- NULL
# trial is changed to numeric
proc_detour_cum_plot_data$trial <- as.numeric(as.character(proc_detour_cum_plot_data$trial))
# stage is changed to factor
proc_detour_cum_plot_data$stage <- factor(proc_detour_cum_plot_data$stage)
# column date2 is changed to date and date type
colnames(proc_detour_cum_plot_data)[5] <- "date"
proc_detour_cum_plot_data$date <- as.Date(proc_detour_cum_plot_data$date, "%d/%m/%Y")
# correct is changed to numeric to be able to calculate commulative sums
proc_detour_cum_plot_data$correct <- as.numeric(as.character(proc_detour_cum_plot_data$correct))
# latencies are changed to numeric
proc_detour_cum_plot_data$latency <- as.numeric(as.character(proc_detour_cum_plot_data$latency))
proc_detour_cum_plot_data$latency.trial <- as.numeric(as.character(proc_detour_cum_plot_data$latency.trial))
# one empty entry in column researcher is changed to NA and researcher is changed to factor
proc_detour_cum_plot_data$researcher[which(proc_detour_cum_plot_data$researcher=='')] <- NA
proc_detour_cum_plot_data$researcher <- factor(proc_detour_cum_plot_data$researcher)
# check changes
summary(proc_detour_cum_plot_data)

# calculate commulative sums of correct detours for each animal based on PIT 
# and save in data frame
proc_detour_cum_plot_data$cumsum <- NA
pit <- unique(proc_detour_cum_plot_data$PIT)

for(i in 1:length(unique(proc_detour_cum_plot_data$PIT))) {
  proc_detour_cum_plot_data$cumsum[
    which(proc_detour_cum_plot_data$PIT == pit[i] )] <- 
    cumsum(subset(proc_detour_cum_plot_data$correct,
                  proc_detour_cum_plot_data$PIT == pit[i]))
}

# remove NAs to plot only data from animals that finished the cylinder task
proc_detour_cum_plot_data_noNA <- proc_detour_cum_plot_data[complete.cases(
                                                  proc_detour_cum_plot_data$date), ]
proc_detour_cum_plot_data_noNA$PIT <- factor(proc_detour_cum_plot_data_noNA$PIT)

# calculate sample mean
mean <- aggregate(proc_detour_cum_plot_data_noNA$cumsum, 
                  list(proc_detour_cum_plot_data_noNA$trial), mean)
# calculate standard deviation
sd   <- aggregate(proc_detour_cum_plot_data_noNA$cumsum, 
                  list(proc_detour_cum_plot_data_noNA$trial), sd)
# calculate sample size
N    <- aggregate(proc_detour_cum_plot_data_noNA$cumsum, 
                  list(proc_detour_cum_plot_data_noNA$trial), length)

# calculate conficence interval
upperCI <- mean$x + (1.96 * (sd$x / sqrt(N$x)))
lowerCI <- mean$x - (1.96 * (sd$x / sqrt(N$x)))
# trial 1 to 10 for plotting
trial <- c(1,2,3,4,5,6,7,8,9,10)
# headings for individual plots
pit2 <- unique(proc_detour_cum_plot_data_noNA$PIT)

# save plot as tiff, 2000x2000, point size 30 and no compression
tiff('Figures/Cummulative_correct_all2.tiff', width = 2000, height = 2000, 
     pointsize = 30, compression = 'none')
# increse line width and increse right margin to include axis lable in plot
par(lwd = 3, mar = c(1,2,4,2), mfrow = c(4,4), oma = c(4,4,1,1))
# parplot with no axes, linewidth 2 and axis lables horizontal
# limit of y-axis adjusted to axis drawn further below so as to show corret numbers (axis length based on second plot)
for(i in 1:length(unique(proc_detour_cum_plot_data_noNA$PIT))) {
  # plot of cummulative sums of correct detours
  plot(proc_detour_cum_plot_data_noNA$cumsum[which(proc_detour_cum_plot_data_noNA$PIT == pit2[i] )],
       ylab=" ", type = "n", ylim = c(0,10), xaxp = c(1,10,9))
  # writes text in the margins: axis title on the left on margin line
  mtext(pit2[i], side = 3, line = 0.5)
  # add group mean
  lines(mean$x, col = "grey40", lty = 'longdash')
  # add cumulative sums to same plot
  par(new=TRUE)
  # plot of cummulative sums of correct detours
  plot(proc_detour_cum_plot_data_noNA$cumsum[which(proc_detour_cum_plot_data_noNA$PIT == pit2[i] )],
       lwd = 3, ylab = " ", type = "b", ylim = c(0,10), xaxp = c(1,10,9))
}
# add text to outer margins
mtext("Trial", side = 1, line = 2.5, outer = TRUE, cex = 1.5)
mtext("Cummulative sum of correct detours", side = 2, line = 2, outer=TRUE, cex=1.5)
# end saving plot
dev.off()

