package fed.fdbs;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Types;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import fed.parser.ParseException;
import fed.parser.Parser;

/**
 * This class contains three SQL Statement objects of the three Connection objects 
 * contained in the FedConnection. This class also holds that FedConnection object.
 *
 */
public class FedStatement implements FedStatementInterface {
    private Statement stmt1;
    private Statement stmt2;
    private Statement stmt3;
    private FedConnection fconn;

    
    /**
     * Create a FedStatement object and set the SQL Statement objects from the SQL Connection objects
     * and the parent FedConnection object.
     * It also creates the metadata and operation tables if they don't exist.
     * 
     * @param conn1
     * @param conn2
     * @param conn3
     * @param daddy
     * @throws FedException 
     */
    public FedStatement (Connection c1, Connection c2, Connection c3, FedConnection daddy) throws FedException {
        try {
            // Create the statements
            stmt1 = c1.createStatement();
            stmt2 = c2.createStatement();
            stmt3 = c3.createStatement();
            fconn = daddy;
            
            // Set fetch size of the statements
            setFetchSize(FedConfig.fetchsize);
            
            // Start catalog manager
            FedCatalog.setStatement(stmt1);
            FedCatalog.createMetadataTable();
            FedCatalog.createOperationTable();
            
        } catch (SQLException e) {
            throw new FedException(e);
        }
    }
    
    
    /**
     * Polymorphic execute method to call executeQuery or executeUpdate based on the SQL
     * statement/command received as a parameter. It validates the statement if the parser validation
     * is activated in the configuration properties.
     * It also calls the FedLogger if the statement is a logger-comment or a SET ECHO command.
     * 
     * @param sql
     * @return An Object instance of Integer or FedResultSet
     * @throws FedException
     * @throws ParseException 
     */
    public Object execute (String sql) throws FedException, ParseException {
        // Ignore statement if...
        if (sql == null || sql.length() < 3 || sql.toUpperCase().startsWith("ALTER")) {
            return null;
        }
        
        // Statement type: logger comment or echo on/off
        if (sql.startsWith("[[")) {
            FedLogger.info(sql.substring(2));
            return null;
        } 
        else if (sql.toUpperCase().startsWith("SET ECHO")) {
            FedLogger.setEcho(sql.substring(8).toUpperCase().trim());
            return null;
        }
        
        FedLogger.trace("Statement: " + sql);
        
        // Parser validation
        if (FedConfig.validate) {
            new Parser(sql).validate();
        }

        // execute statement
        return sql.toUpperCase().startsWith("SELECT") ? executeQuery(sql) : executeUpdate(sql);
    }
    
    
    /**
     * Set the number of rows that should be fetched from the databases when more rows are needed 
     * for FedResultSet objects generated by this FedStatement. 
     * If the value specified is zero, then the hint is ignored. The default value is zero.
     * 
     * @param rows
     * @throws FedException
     */
    public void setFetchSize (int rows) throws FedException {
        if (rows >= 0) { 
            FedLogger.trace("Setting fetch size to " + rows);
        
            try {
                if (stmt1 != null) stmt1.setFetchSize(rows);
                if (stmt2 != null) stmt2.setFetchSize(rows);
                if (stmt3 != null) stmt3.setFetchSize(rows);
            } catch (SQLException e) {
                throw new FedException(e);
            }
        }
    }

    /** Deprecated method. Use execute(String sql) instead.
     *  
     * @see fedjdbc.FedStatementInterface#executeQuery(java.lang.String)
     */
    @Override
    @Deprecated
    public FedResultSet executeQuery (String sql) throws FedException {
        String SQL   = sql.toUpperCase().trim();
        String table = SQL.substring(SQL.indexOf("FROM") + 4);
        
        table = table.contains("WHERE") ? table.substring(0, table.indexOf("WHERE")).trim() : table.trim();  
        table = table.contains("GROUP BY") ? table.substring(0, table.indexOf("GROUP BY")).trim() : table.trim();
        
        // Statement: SELECT
        try {
            ResultSet rs  = FedCatalog.getMetadata(table);
            ResultSet rs1 = null; 
            ResultSet rs2 = null; 
            ResultSet rs3 = null; 
            boolean funct = false;

            if (table.contains(",")) {
                // It's a join...
                boolean distributed = false;
                String table1 = table.split(",")[0].trim();
                String table2 = table.split(",")[1].trim();
                String sql_temp = sql;
                
                FedCatalog.dropTempTables();
                
                rs = FedCatalog.getMetadata(table1);
                if (rs != null && rs.next()) { 
                    // ...and it's distributed (first table)
                    Object right = rs.getObject("right");
                    distributed  = true;
                    sql_temp = FedCatalog.getTempQuery(sql_temp, table1);
                    
                    // Clone table1 on db1
                    FedCatalog.createTempTable(table1);
                    
                    // Insert values from db2.table1 into db1.tempTable1
                    FedCatalog.insertIntoTempTable(table1, stmt2.executeQuery("SELECT * FROM " + table1));
                    
                    // Insert values from db3.table1 into db1.tempTable1
                    if (right != null) {
                        FedCatalog.insertIntoTempTable(table1, stmt3.executeQuery("SELECT * FROM " + table1));
                    }
                }
                
                rs = FedCatalog.getMetadata(table2);
                if (rs != null && rs.next()) { 
                    // ...and it's distributed (second table)
                    Object right = rs.getObject("right");
                    distributed  = true;
                    sql_temp = FedCatalog.getTempQuery(sql_temp, table2);
                    
                    // Clone table2 on db1
                    FedCatalog.createTempTable(table2);
                    
                    // Insert values from db2.table2 into db1.tempTable2
                    FedCatalog.insertIntoTempTable(table2, stmt2.executeQuery("SELECT * FROM " + table2));
                    
                    // Insert values from db3.table2 into db1.tempTable2
                    if (right != null) {
                        FedCatalog.insertIntoTempTable(table2, stmt3.executeQuery("SELECT * FROM " + table2));
                    }
                }
                
                if (distributed) {                  
                    sql = sql_temp;
                    rs  = null;
                    rs2 = null;
                    rs3 = null;
                }
            }
            
            if (rs != null && rs.next()) {
                rs2 = stmt2.executeQuery(sql);
                rs3 = rs.getObject("right") != null ? stmt3.executeQuery(sql) : null;
                
                // Check if it's a COUNT(*) or SUM(column) function
                String columns = SQL.substring(SQL.indexOf("SELECT") + 6, SQL.indexOf("FROM")).replaceAll(" ", "").trim();

                funct = columns.contains("COUNT(*)") || columns.contains("SUM(");
            }
            rs1 = stmt1.executeQuery(sql);
            
            if (funct && SQL.contains("GROUP BY")) {
                // Aggregate functions
                Map<String, Integer> temptable = new HashMap<>();
                String attribute = rs1.getMetaData().getColumnName(1);
                String function  = rs1.getMetaData().getColumnName(2).replace("(*)", "#$#")
                                    .replace("(", "_$_").replace(".", "$#$").replace(")", "_#_");
                
                while (rs1 != null && rs1.next()) {
                    int opt = Integer.parseInt(rs1.getObject(2).toString());
                    temptable.put(rs1.getString(1), opt);
                }
                while (rs2 != null && rs2.next()) {
                    int opt = Integer.parseInt(rs2.getObject(2).toString());
                    String key = rs2.getString(1);
                    int value = temptable.containsKey(key) ? temptable.get(key) + opt : opt; 
                    temptable.put(key, value);
                }
                while (rs3 != null && rs3.next()) {
                    int opt = Integer.parseInt(rs3.getObject(2).toString());
                    String key = rs3.getString(1);
                    int value = temptable.containsKey(key) ? temptable.get(key) + opt : opt; 
                    temptable.put(key, value);
                }
                
                // Recreate the operations table
                stmt1.executeUpdate("DROP TABLE " + FedCatalog.OPERATION);
                stmt1.executeUpdate("CREATE TABLE " + FedCatalog.OPERATION + " (" 
                                     + attribute + " VARCHAR(20), " + function + " INTEGER)");
                
                for (String key : temptable.keySet()) {
                    stmt1.executeUpdate("INSERT INTO " + FedCatalog.OPERATION + " VALUES ('" 
                                          + key + "', " + temptable.get(key) + ")");
                }

                // Select results from the operations table
                rs1 = stmt1.executeQuery("SELECT * FROM " + FedCatalog.OPERATION);
                rs2 = null;
                rs3 = null;
                funct = false;
            }
            
            return new FedResultSet(rs1, rs2, rs3, funct);
        } catch (SQLException e) {
            throw new FedException(e);
        }
    }
    
    /** Deprecated method. Use execute(String sql) instead.
     *  
     * @see fedjdbc.FedStatementInterface#executeUpdate(java.lang.String)
     */
    @Override
    @Deprecated
    public int executeUpdate (String sql) throws FedException {
        return executeUpdate(sql, true);
    }
    
    private int executeUpdate (String sql, boolean show_trace) throws FedException {
        String SQL = sql.toUpperCase().trim();

        try {
            // Create and Drop Statements
            if (SQL.startsWith("CREATE") || SQL.startsWith("DROP") ) {
                String table = SQL.substring(SQL.indexOf("TABLE") + 5).trim();
                
                if (SQL.startsWith("CREATE")) {
                    // Statement: CREATE TABLE ...
                    table = table.substring(0, table.indexOf("(")).trim();
                    
                    if (SQL.contains("HORIZONTAL")) {
                        // ... IN HORIZONTAL PARTITION
                        String val = SQL.substring(SQL.indexOf("HORIZONTAL") + 10).trim();
                        val = val.substring(1, val.length() - 2).trim();
                        
                        String fcolumn = val.substring(0, val.indexOf("(")).trim();
                        val = val.substring(val.indexOf("(") + 1).trim();
                        
                        Integer left  = null;
                        Integer right = null;
                        
                        if (val.contains(",")) {
                            left  = Integer.parseInt(val.split(",")[0]);
                            right = Integer.parseInt(val.split(",")[1]);
                        }
                        else {
                            left  = Integer.parseInt(val);
                        }
                        
                        try {
                            // Get Oracle valid statement
                            sql = sql.substring(0, SQL.indexOf("HORIZONTAL")).trim();
                            
                            // Create table in db2
                            FedLogger.trace("Creating " + table.toUpperCase() + " table in " + FedConfig.getDb2Name() + "...");
                            stmt2.executeUpdate(sql);

                            if (right != null) {
                                // Create table in db3
                                FedLogger.trace("Creating " + table.toUpperCase() + " table in " + FedConfig.getDb3Name() + "...");
                                stmt3.executeUpdate(sql);
                            }
                            
                            // Store metadata of the horizontal partitioned table in db1
                            FedLogger.trace("Storing metadata for " + table.toUpperCase() + " table...");
                            stmt1.executeUpdate("INSERT INTO " + FedCatalog.METADATA + " VALUES ('" 
                                        + table + "', '" + fcolumn + "', " + left + ", " + right + ")");
                        }
                        catch (SQLException e) {
                            throw new FedException(e);
                        }
                    }

                    // Create table in db1
                    FedLogger.trace("Creating " + table.toUpperCase() + " table in " + FedConfig.getDb1Name() + "...");
                    return stmt1.executeUpdate(sql);
                }
                else if (SQL.startsWith("DROP")) {
                    // Statement: DROP TABLE
                    if (table.contains("CASCADE")) {
                        table = table.substring(0, table.indexOf("CASCADE")).trim();
                    }
                    
                    ResultSet rs = FedCatalog.getMetadata(table);
                    
                    if (rs != null && rs.next()) {
                        Object right = rs.getObject("right");
                        
                        // The table is horizontal partitioned, delete it from metadata
                        FedLogger.trace("Deleting metadata for " + table.toUpperCase() + " table...");
                        stmt1.executeUpdate("DELETE FROM " + FedCatalog.METADATA + " WHERE ftable = '" + table + "'");
                        
                        // Drop table in db2
                        FedLogger.trace("Dropping " + table.toUpperCase() + " table in " + FedConfig.getDb2Name() + "...");
                        stmt2.executeUpdate(sql);
                        
                        if (right != null) {
                            // Drop table in db3
                            FedLogger.trace("Dropping " + table.toUpperCase() + " table in " + FedConfig.getDb3Name() + "...");
                            stmt3.executeUpdate(sql);
                        }
                    }
                    // Drop table in db1
                    FedLogger.trace("Dropping " + table.toUpperCase() + " table in " + FedConfig.getDb1Name() + "...");
                    return stmt1.executeUpdate(sql);
                }
            }
            else if (SQL.startsWith("INSERT")) {
                // Statement: INSERT INTO 
                String table = SQL.substring(SQL.indexOf("INTO") + 4, SQL.indexOf("VALUES")).trim();
                String fcolumn = "";
                Integer left = null;
                Object right = null;
                boolean is_distributed = false;
                
                ResultSet rs = FedCatalog.getMetadata(table);
                
                if (rs != null && rs.next()) {
                    is_distributed = true;
                    
                    // Get metadata of the distributed table
                    fcolumn = rs.getString("fcolumn");
                    left    = rs.getInt("left");
                    right   = rs.getObject("right");
                    
                    // Insert into db2 
                    stmt2.executeUpdate(sql);
                    stmt2.executeUpdate("DELETE FROM " + table + " WHERE " + fcolumn + "<" + left);

                    if (right != null) {
                        // Insert into db3
                        stmt2.executeUpdate("DELETE FROM " + table + " WHERE " + fcolumn + ">=" + right);
                        stmt3.executeUpdate(sql);
                        stmt3.executeUpdate("DELETE FROM " + table + " WHERE " + fcolumn + "<" + right);
                    }
                }

                // Insert into db1
                int row_count = stmt1.executeUpdate(sql);
                if (is_distributed) {
                    stmt1.executeUpdate("DELETE FROM " + table + " WHERE " + fcolumn + ">=" + left);
                }

                if (show_trace) FedLogger.info(row_count + " row inserted");
                return row_count;
            }
            else if (SQL.startsWith("DELETE")) {
                // Statement: DELETE FROM
                String table = SQL.substring(SQL.indexOf("FROM") + 4).trim();
                
                if (table.contains("WHERE")) {
                    table = table.substring(0, table.indexOf("WHERE")).trim();
                }
                
                int row_count = 0;
                ResultSet rs = FedCatalog.getMetadata(table);
                
                if (rs != null && rs.next()) {
                    Object right = rs.getObject("right");
                    
                    // Delete from db2 
                    row_count += stmt2.executeUpdate(sql);
                    
                    if (right != null) {
                        // Delete from db3
                        row_count += stmt3.executeUpdate(sql);
                    }
                }
                
                // Delete from db1
                return row_count + stmt1.executeUpdate(sql);
            }
            else if (SQL.startsWith("UPDATE")) {
                // Statement: UPDATE
                String table = SQL.substring(SQL.indexOf("UPDATE") + 6, SQL.indexOf("SET")).trim();
                String fcolumn = "";
                Integer left = null;
                Object right = null;
                boolean do_refactor = false;
                
                int up1 = 0;
                int up2 = 0;
                int up3 = 0;
                
                ResultSet rs = FedCatalog.getMetadata(table);
                
                if (rs != null && rs.next()) {
                    String attribute = SQL.substring(SQL.indexOf("SET") + 3, SQL.indexOf("=")).trim();
                    
                    // Get metadata of the distributed table
                    fcolumn = rs.getString("fcolumn");
                    left    = rs.getInt("left");
                    right = rs.getObject("right");
                    
                    do_refactor = attribute.equals(fcolumn); 
                    
                    // Update in db2 
                    up2 = stmt2.executeUpdate(sql);
                    
                    if (right != null) {
                        // Update in db3
                        up3 = stmt3.executeUpdate(sql);
                    }
                }
                
                // Update in db1
                up1 = stmt1.executeUpdate(sql);
                
                if (do_refactor) {
                    // The row(s) shall be moved to other partition
                    if (up1 > 0) {
                        // Refactor db1
                        String select = "SELECT * FROM " + table + " WHERE " + fcolumn + ">=" + left; 
                        List<String> statements = getRefactorStatements(stmt1.executeQuery(select), table);
                        
                        for (String dst : statements) {
                            if (dst.startsWith("DELETE")) 
                                stmt1.executeUpdate(dst);
                            else 
                                this.executeUpdate(dst, false);
                        }
                    }
                    if (up2 > 0) {
                        // Refactor db2
                        String select = "SELECT * FROM " + table + " WHERE " + fcolumn + "<" + left 
                                        + (right != null ? " OR " + fcolumn + ">=" + right : ""); 
                        List<String> statements = getRefactorStatements(stmt2.executeQuery(select), table);
                        
                        for (String dst : statements) {
                            if (dst.startsWith("DELETE")) 
                                stmt2.executeUpdate(dst);
                            else 
                                this.executeUpdate(dst, false);
                        }
                    }
                    if (up3 > 0) {
                        // Refactor db3
                        String select = "SELECT * FROM " + table + " WHERE " + fcolumn + "<" + right; 
                        List<String> statements = getRefactorStatements(stmt3.executeQuery(select), table);
                        
                        for (String dst : statements) {
                            if (dst.startsWith("DELETE")) 
                                stmt3.executeUpdate(dst);
                            else 
                                this.executeUpdate(dst, false);
                        }
                    }
                }
                
                return up1 + up2 + up3;
            }
            else if (SQL.equals("COMMIT")) {
                // Command: COMMIT
                fconn.commit();
                
                // Return success
                return FedCatalog.SUCCESS;
            }
            else if (SQL.equals("ROLLBACK")) {
                // Command: ROLLBACK
                fconn.rollback();
                
                // Return success
                return FedCatalog.SUCCESS;
            }
        } catch (SQLException e) {
            throw new FedException(e);
        }
        
        // Nothing was executed
        FedLogger.error("Statement not recognized: " + sql);
        return FedCatalog.NO_EXECUTION;
    }


    private List<String> getRefactorStatements (ResultSet rsrf, String table) throws SQLException {
        /* TODO: Want to fork it?
         * - Implement delete at once, everything not satisfying the distribution constraint
         * - Implement inserts with PreparedStatement to improve performance
         */
        
        List<String> statements = new ArrayList<>();
        
        while (rsrf != null && rsrf.next()) {
            ResultSetMetaData meta = rsrf.getMetaData();
            String delete = "DELETE FROM " + table + " WHERE ";
            String insert = "INSERT INTO " + table + " VALUES (";
            
            for (int i = 1, colcount = meta.getColumnCount(); i <= colcount; i++) {
                boolean is_int = meta.getColumnType(i) != Types.VARCHAR; 
                delete += meta.getColumnName(i) + "=" + (is_int ? rsrf.getInt(i) : rsrf.getString(i) == null ? "null" : 
                                                                                            "'" + rsrf.getString(i) + "'")
                          + (i+1 <= colcount ? " AND " : "");
                insert += (is_int ? rsrf.getInt(i) : rsrf.getString(i) == null ? "null" : 
                                                                                 "'" + rsrf.getString(i) + "'")
                        + (i+1 <= colcount ? ", " : ")");
            }
            
            statements.add(delete);
            statements.add(insert);
        }
        
        return statements;
    }

    /* (non-Javadoc)
     * @see fedjdbc.FedStatementInterface#getConnection()
     */
    @Override
    public FedConnection getConnection () throws FedException {
        return fconn;
    }

    /* (non-Javadoc)
     * @see fedjdbc.FedStatementInterface#close()
     */
    @Override
    public void close () throws FedException {
        try {
            if (stmt1 != null) stmt1.close();
            if (stmt2 != null) stmt2.close();
            if (stmt3 != null) stmt3.close();
        } catch (SQLException e) {
            throw new FedException(e);
        }
    }

}
