/*
 * Copyright (c) 2010-2013, 2015 ARM Limited
 * All rights reserved
 *
 * The license below extends only to copyright in the software and shall
 * not be construed as granting a license to any other intellectual
 * property including but not limited to intellectual property relating
 * to a hardware implementation of the functionality of the software
 * licensed hereunder.  You may use the software subject to the license
 * terms below provided that you ensure that this notice is replicated
 * unmodified and in its entirety in all distributions of the software,
 * modified or unmodified, in source code or in binary form.
 *
 * Copyright (c) 2003-2005 The Regents of The University of Michigan
 * Copyright (c) 2007-2008 The Florida State University
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met: redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer;
 * redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution;
 * neither the name of the copyright holders nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * Authors: Korey Sewell
 *          Stephen Hines
 *          Ali Saidi
 *          Giacomo Gabrielli
 */

#include "arch/arm/linux/process.hh"

#include "arch/arm/isa_traits.hh"
#include "arch/arm/linux/linux.hh"
#include "base/trace.hh"
#include "cpu/thread_context.hh"
#include "kern/linux/linux.hh"
#include "sim/process.hh"
#include "sim/syscall_desc.hh"
#include "sim/syscall_emul.hh"
#include "sim/system.hh"

using namespace std;
using namespace ArmISA;

/// Target uname() handler.
static SyscallReturn
unameFunc32(SyscallDesc *desc, int callnum, Process *process,
            ThreadContext *tc)
{
    int index = 0;
    TypedBufferArg<Linux::utsname> name(process->getSyscallArg(tc, index));

    strcpy(name->sysname, "Linux");
    strcpy(name->nodename, "m5.eecs.umich.edu");
    strcpy(name->release, "3.0.0");
    strcpy(name->version, "#1 Mon Aug 18 11:32:15 EDT 2003");
    strcpy(name->machine, "armv7l");

    name.copyOut(tc->getMemProxy());
    return 0;
}

/// Target uname() handler.
static SyscallReturn
unameFunc64(SyscallDesc *desc, int callnum, Process *process,
            ThreadContext *tc)
{
    int index = 0;
    TypedBufferArg<Linux::utsname> name(process->getSyscallArg(tc, index));

    strcpy(name->sysname, "Linux");
    strcpy(name->nodename, "gem5");
    strcpy(name->release, "3.7.0+");
    strcpy(name->version, "#1 SMP Sat Dec  1 00:00:00 GMT 2012");
    strcpy(name->machine, "armv8l");

    name.copyOut(tc->getMemProxy());
    return 0;
}

/// Target set_tls() handler.
static SyscallReturn
setTLSFunc32(SyscallDesc *desc, int callnum, Process *process,
             ThreadContext *tc)
{
    int index = 0;
    uint32_t tlsPtr = process->getSyscallArg(tc, index);

    tc->getMemProxy().writeBlob(ArmLinuxProcess32::commPage + 0x0ff0,
                                 (uint8_t *)&tlsPtr, sizeof(tlsPtr));
    tc->setMiscReg(MISCREG_TPIDRURO,tlsPtr);
    return 0;
}

static SyscallReturn
setTLSFunc64(SyscallDesc *desc, int callnum, Process *process,
             ThreadContext *tc)
{
    int index = 0;
    uint32_t tlsPtr = process->getSyscallArg(tc, index);

    tc->setMiscReg(MISCREG_TPIDRRO_EL0, tlsPtr);
    return 0;
}

static SyscallDesc syscallDescs32[] = {
    /*  0 */ SyscallDesc("syscall", unimplementedFunc),
    /*  1 */ SyscallDesc("exit", exitFunc),
    /*  2 */ SyscallDesc("fork", unimplementedFunc),
    /*  3 */ SyscallDesc("read", readFunc),
    /*  4 */ SyscallDesc("write", writeFunc),
    /*  5 */ SyscallDesc("open", openFunc<ArmLinux32>),
    /*  6 */ SyscallDesc("close", closeFunc),
    /*  7 */ SyscallDesc("unused#7", unimplementedFunc),
    /*  8 */ SyscallDesc("creat", unimplementedFunc),
    /*  9 */ SyscallDesc("link", unimplementedFunc),
    /* 10 */ SyscallDesc("unlink", unlinkFunc),
    /* 11 */ SyscallDesc("execve", unimplementedFunc),
    /* 12 */ SyscallDesc("chdir", unimplementedFunc),
    /* 13 */ SyscallDesc("time", timeFunc<ArmLinux32>),
    /* 14 */ SyscallDesc("mknod", unimplementedFunc),
    /* 15 */ SyscallDesc("chmod", chmodFunc<ArmLinux32>),
    /* 16 */ SyscallDesc("lchown", chownFunc),
    /* 17 */ SyscallDesc("unused#17", unimplementedFunc),
    /* 18 */ SyscallDesc("unused#18", unimplementedFunc),
    /* 19 */ SyscallDesc("lseek", lseekFunc),
    /* 20 */ SyscallDesc("getpid", getpidFunc),
    /* 21 */ SyscallDesc("mount", unimplementedFunc),
    /* 22 */ SyscallDesc("umount", unimplementedFunc),
    /* 23 */ SyscallDesc("setuid", setuidFunc),
    /* 24 */ SyscallDesc("getuid", getuidFunc),
    /* 25 */ SyscallDesc("stime", unimplementedFunc),
    /* 26 */ SyscallDesc("ptrace", unimplementedFunc),
    /* 27 */ SyscallDesc("alarm", unimplementedFunc),
    /* 28 */ SyscallDesc("unused#28", unimplementedFunc),
    /* 29 */ SyscallDesc("pause", unimplementedFunc),
    /* 30 */ SyscallDesc("utime", unimplementedFunc),
    /* 31 */ SyscallDesc("unused#31", unimplementedFunc),
    /* 32 */ SyscallDesc("unused#32", unimplementedFunc),
    /* 33 */ SyscallDesc("access", accessFunc),
    /* 34 */ SyscallDesc("nice", unimplementedFunc),
    /* 35 */ SyscallDesc("unused#35", unimplementedFunc),
    /* 36 */ SyscallDesc("sync", unimplementedFunc),
    /* 37 */ SyscallDesc("kill", ignoreFunc),
    /* 38 */ SyscallDesc("rename", renameFunc),
    /* 39 */ SyscallDesc("mkdir", mkdirFunc),
    /* 40 */ SyscallDesc("rmdir", unimplementedFunc),
    /* 41 */ SyscallDesc("dup", dupFunc),
    /* 42 */ SyscallDesc("pipe", pipePseudoFunc),
    /* 43 */ SyscallDesc("times", timesFunc<ArmLinux32>),
    /* 44 */ SyscallDesc("unused#44", unimplementedFunc),
    /* 45 */ SyscallDesc("brk", brkFunc),
    /* 46 */ SyscallDesc("setgid", unimplementedFunc),
    /* 47 */ SyscallDesc("getgid", getgidFunc),
    /* 48 */ SyscallDesc("unused#48", unimplementedFunc),
    /* 49 */ SyscallDesc("geteuid", geteuidFunc),
    /* 50 */ SyscallDesc("getegid", getegidFunc),
    /* 51 */ SyscallDesc("acct", unimplementedFunc),
    /* 52 */ SyscallDesc("umount2", unimplementedFunc),
    /* 53 */ SyscallDesc("unused#53", unimplementedFunc),
    /* 54 */ SyscallDesc("ioctl", ioctlFunc<ArmLinux32>),
    /* 55 */ SyscallDesc("fcntl", fcntlFunc),
    /* 56 */ SyscallDesc("unused#56", unimplementedFunc),
    /* 57 */ SyscallDesc("setpgid", unimplementedFunc),
    /* 58 */ SyscallDesc("unused#58", unimplementedFunc),
    /* 59 */ SyscallDesc("unused#59", unimplementedFunc),
    /* 60 */ SyscallDesc("umask", umaskFunc),
    /* 61 */ SyscallDesc("chroot", unimplementedFunc),
    /* 62 */ SyscallDesc("ustat", unimplementedFunc),
    /* 63 */ SyscallDesc("dup2", unimplementedFunc),
    /* 64 */ SyscallDesc("getppid", getppidFunc),
    /* 65 */ SyscallDesc("getpgrp", unimplementedFunc),
    /* 66 */ SyscallDesc("setsid", unimplementedFunc),
    /* 67 */ SyscallDesc("sigaction",unimplementedFunc),
    /* 68 */ SyscallDesc("unused#68", unimplementedFunc),
    /* 69 */ SyscallDesc("unused#69", unimplementedFunc),
    /* 70 */ SyscallDesc("setreuid", unimplementedFunc),
    /* 71 */ SyscallDesc("setregid", unimplementedFunc),
    /* 72 */ SyscallDesc("sigsuspend", unimplementedFunc),
    /* 73 */ SyscallDesc("sigpending", unimplementedFunc),
    /* 74 */ SyscallDesc("sethostname", ignoreFunc),
    /* 75 */ SyscallDesc("setrlimit", ignoreFunc),
    /* 76 */ SyscallDesc("getrlimit", getrlimitFunc<ArmLinux32>),
    /* 77 */ SyscallDesc("getrusage", getrusageFunc<ArmLinux32>),
    /* 78 */ SyscallDesc("gettimeofday", gettimeofdayFunc<ArmLinux32>),
    /* 79 */ SyscallDesc("settimeofday", unimplementedFunc),
    /* 80 */ SyscallDesc("getgroups", unimplementedFunc),
    /* 81 */ SyscallDesc("setgroups", unimplementedFunc),
    /* 82 */ SyscallDesc("reserved#82", unimplementedFunc),
    /* 83 */ SyscallDesc("symlink", unimplementedFunc),
    /* 84 */ SyscallDesc("unused#84", unimplementedFunc),
    /* 85 */ SyscallDesc("readlink", readlinkFunc),
    /* 86 */ SyscallDesc("uselib", unimplementedFunc),
    /* 87 */ SyscallDesc("swapon", unimplementedFunc),
    /* 88 */ SyscallDesc("reboot", unimplementedFunc),
    /* 89 */ SyscallDesc("readdir", unimplementedFunc),
    /* 90 */ SyscallDesc("mmap", mmapFunc<ArmLinux32>),
    /* 91 */ SyscallDesc("munmap", munmapFunc),
    /* 92 */ SyscallDesc("truncate", truncateFunc),
    /* 93 */ SyscallDesc("ftruncate", ftruncateFunc),
    /* 94 */ SyscallDesc("fchmod", unimplementedFunc),
    /* 95 */ SyscallDesc("fchown", unimplementedFunc),
    /* 96 */ SyscallDesc("getpriority", unimplementedFunc),
    /* 97 */ SyscallDesc("setpriority", unimplementedFunc),
    /* 98 */ SyscallDesc("unused#98", unimplementedFunc),
    /* 99 */ SyscallDesc("statfs", unimplementedFunc),
    /* 100 */ SyscallDesc("fstatfs", unimplementedFunc),
    /* 101 */ SyscallDesc("unused#101", unimplementedFunc),
    /* 102 */ SyscallDesc("socketcall", unimplementedFunc),
    /* 103 */ SyscallDesc("syslog", unimplementedFunc),
    /* 104 */ SyscallDesc("setitimer", unimplementedFunc),
    /* 105 */ SyscallDesc("getitimer", unimplementedFunc),
    /* 106 */ SyscallDesc("stat",  statFunc<ArmLinux32>),
    /* 107 */ SyscallDesc("lstat", unimplementedFunc),
    /* 108 */ SyscallDesc("fstat", fstatFunc<ArmLinux32>),
    /* 109 */ SyscallDesc("unused#109", unimplementedFunc),
    /* 110 */ SyscallDesc("unused#101", unimplementedFunc),
    /* 111 */ SyscallDesc("vhangup", unimplementedFunc),
    /* 112 */ SyscallDesc("unused#112", unimplementedFunc),
    /* 113 */ SyscallDesc("syscall", unimplementedFunc),
    /* 114 */ SyscallDesc("wait4", unimplementedFunc),
    /* 115 */ SyscallDesc("swapoff", unimplementedFunc),
    /* 116 */ SyscallDesc("sysinfo", sysinfoFunc<ArmLinux32>),
    /* 117 */ SyscallDesc("ipc", unimplementedFunc),
    /* 118 */ SyscallDesc("fsync", unimplementedFunc),
    /* 119 */ SyscallDesc("sigreturn", unimplementedFunc),
    /* 120 */ SyscallDesc("clone", cloneFunc<ArmLinux32>),
    /* 121 */ SyscallDesc("setdomainname", unimplementedFunc),
    /* 122 */ SyscallDesc("uname", unameFunc32),
    /* 123 */ SyscallDesc("unused#123", unimplementedFunc),
    /* 124 */ SyscallDesc("adjtimex", unimplementedFunc),
    /* 125 */ SyscallDesc("mprotect", ignoreFunc),
    /* 126 */ SyscallDesc("sigprocmask", ignoreFunc, SyscallDesc::WarnOnce),
    /* 127 */ SyscallDesc("unused#127", unimplementedFunc),
    /* 128 */ SyscallDesc("init_module", unimplementedFunc),
    /* 129 */ SyscallDesc("delete_module", unimplementedFunc),
    /* 130 */ SyscallDesc("unused#130", unimplementedFunc),
    /* 131 */ SyscallDesc("quotactl", unimplementedFunc),
    /* 132 */ SyscallDesc("getpgid", unimplementedFunc),
    /* 133 */ SyscallDesc("fchdir", unimplementedFunc),
    /* 134 */ SyscallDesc("bdflush", unimplementedFunc),
    /* 135 */ SyscallDesc("sysfs", unimplementedFunc),
    /* 136 */ SyscallDesc("personality", unimplementedFunc),
    /* 137 */ SyscallDesc("reserved#138", unimplementedFunc),
    /* 138 */ SyscallDesc("setfsuid", unimplementedFunc),
    /* 139 */ SyscallDesc("setfsgid", unimplementedFunc),
    /* 140 */ SyscallDesc("llseek", _llseekFunc),
    /* 141 */ SyscallDesc("getdents", unimplementedFunc),
    /* 142 */ SyscallDesc("newselect", unimplementedFunc),
    /* 143 */ SyscallDesc("flock", unimplementedFunc),
    /* 144 */ SyscallDesc("msync", unimplementedFunc),
    /* 145 */ SyscallDesc("readv", unimplementedFunc),
    /* 146 */ SyscallDesc("writev", writevFunc<ArmLinux32>),
    /* 147 */ SyscallDesc("getsid", unimplementedFunc),
    /* 148 */ SyscallDesc("fdatasync", unimplementedFunc),
    /* 149 */ SyscallDesc("sysctl", unimplementedFunc),
    /* 150 */ SyscallDesc("mlock", unimplementedFunc),
    /* 151 */ SyscallDesc("munlock", unimplementedFunc),
    /* 152 */ SyscallDesc("mlockall", unimplementedFunc),
    /* 153 */ SyscallDesc("munlockall", unimplementedFunc),
    /* 154 */ SyscallDesc("sched_setparam", unimplementedFunc),
    /* 155 */ SyscallDesc("sched_getparam", unimplementedFunc),
    /* 156 */ SyscallDesc("sched_setscheduler", unimplementedFunc),
    /* 157 */ SyscallDesc("sched_getscheduler", unimplementedFunc),
    /* 158 */ SyscallDesc("sched_yield", unimplementedFunc),
    /* 159 */ SyscallDesc("sched_get_priority_max", unimplementedFunc),
    /* 160 */ SyscallDesc("sched_get_priority_min", unimplementedFunc),
    /* 161 */ SyscallDesc("sched_rr_get_interval", unimplementedFunc),
    /* 162 */ SyscallDesc("nanosleep", ignoreFunc, SyscallDesc::WarnOnce),
    /* 163 */ SyscallDesc("mremap", mremapFunc<ArmLinux32>), // ARM-specific
    /* 164 */ SyscallDesc("setresuid", unimplementedFunc),
    /* 165 */ SyscallDesc("getresuid", unimplementedFunc),
    /* 166 */ SyscallDesc("unused#166", unimplementedFunc),
    /* 167 */ SyscallDesc("unused#167", unimplementedFunc),
    /* 168 */ SyscallDesc("poll", unimplementedFunc),
    /* 169 */ SyscallDesc("nfsservctl", unimplementedFunc),
    /* 170 */ SyscallDesc("setresgid", unimplementedFunc),
    /* 171 */ SyscallDesc("getresgid", unimplementedFunc),
    /* 172 */ SyscallDesc("prctl", unimplementedFunc),
    /* 173 */ SyscallDesc("rt_sigreturn", unimplementedFunc),
    /* 174 */ SyscallDesc("rt_sigaction", ignoreFunc, SyscallDesc::WarnOnce),
    /* 175 */ SyscallDesc("rt_sigprocmask", ignoreFunc, SyscallDesc::WarnOnce),
    /* 176 */ SyscallDesc("rt_sigpending", unimplementedFunc),
    /* 177 */ SyscallDesc("rt_sigtimedwait", unimplementedFunc),
    /* 178 */ SyscallDesc("rt_sigqueueinfo", ignoreFunc),
    /* 179 */ SyscallDesc("rt_sigsuspend", unimplementedFunc),
    /* 180 */ SyscallDesc("pread64", unimplementedFunc),
    /* 181 */ SyscallDesc("pwrite64", unimplementedFunc),
    /* 182 */ SyscallDesc("chown", unimplementedFunc),
    /* 183 */ SyscallDesc("getcwd", getcwdFunc),
    /* 184 */ SyscallDesc("capget", unimplementedFunc),
    /* 185 */ SyscallDesc("capset", unimplementedFunc),
    /* 186 */ SyscallDesc("sigaltstack", unimplementedFunc),
    /* 187 */ SyscallDesc("sendfile", unimplementedFunc),
    /* 188 */ SyscallDesc("unused#188", unimplementedFunc),
    /* 189 */ SyscallDesc("unused#189", unimplementedFunc),
    /* 190 */ SyscallDesc("vfork", unimplementedFunc),
    /* 191 */ SyscallDesc("getrlimit", getrlimitFunc<ArmLinux32>),
    /* 192 */ SyscallDesc("mmap2", mmapFunc<ArmLinux32>),
    /* 193 */ SyscallDesc("truncate64", unimplementedFunc),
    /* 194 */ SyscallDesc("ftruncate64", ftruncate64Func),
    /* 195 */ SyscallDesc("stat64", stat64Func<ArmLinux32>),
    /* 196 */ SyscallDesc("lstat64", lstat64Func<ArmLinux32>),
    /* 197 */ SyscallDesc("fstat64", fstat64Func<ArmLinux32>),
    /* 198 */ SyscallDesc("lchown", unimplementedFunc),
    /* 199 */ SyscallDesc("getuid", getuidFunc),
    /* 200 */ SyscallDesc("getgid", getgidFunc),
    /* 201 */ SyscallDesc("geteuid", geteuidFunc),
    /* 202 */ SyscallDesc("getegid", getegidFunc),
    /* 203 */ SyscallDesc("setreuid", unimplementedFunc),
    /* 204 */ SyscallDesc("setregid", unimplementedFunc),
    /* 205 */ SyscallDesc("getgroups", unimplementedFunc),
    /* 206 */ SyscallDesc("setgroups", unimplementedFunc),
    /* 207 */ SyscallDesc("fchown", unimplementedFunc),
    /* 208 */ SyscallDesc("setresuid", unimplementedFunc),
    /* 209 */ SyscallDesc("getresuid", unimplementedFunc),
    /* 210 */ SyscallDesc("setresgid", unimplementedFunc),
    /* 211 */ SyscallDesc("getresgid", unimplementedFunc),
    /* 212 */ SyscallDesc("chown", unimplementedFunc),
    /* 213 */ SyscallDesc("setuid", unimplementedFunc),
    /* 214 */ SyscallDesc("setgid", unimplementedFunc),
    /* 215 */ SyscallDesc("setfsuid", unimplementedFunc),
    /* 216 */ SyscallDesc("setfsgid", unimplementedFunc),
    /* 217 */ SyscallDesc("getdents64", unimplementedFunc),
    /* 218 */ SyscallDesc("pivot_root", unimplementedFunc),
    /* 219 */ SyscallDesc("mincore", unimplementedFunc),
    /* 220 */ SyscallDesc("madvise", unimplementedFunc),
    /* 221 */ SyscallDesc("fcntl64", fcntl64Func),
    /* 222 */ SyscallDesc("unused#222", unimplementedFunc),
    /* 223 */ SyscallDesc("unknown#223", unimplementedFunc),
    /* 224 */ SyscallDesc("gettid", unimplementedFunc),
    /* 225 */ SyscallDesc("readahead", unimplementedFunc),
    /* 226 */ SyscallDesc("setxattr", unimplementedFunc),
    /* 227 */ SyscallDesc("lsetxattr", unimplementedFunc),
    /* 228 */ SyscallDesc("fsetxattr", unimplementedFunc),
    /* 229 */ SyscallDesc("getxattr", unimplementedFunc),
    /* 230 */ SyscallDesc("lgetxattr", unimplementedFunc),
    /* 231 */ SyscallDesc("fgetxattr", unimplementedFunc),
    /* 232 */ SyscallDesc("listxattr", unimplementedFunc),
    /* 233 */ SyscallDesc("llistxattr", unimplementedFunc),
    /* 234 */ SyscallDesc("flistxattr", unimplementedFunc),
    /* 235 */ SyscallDesc("removexattr", unimplementedFunc),
    /* 236 */ SyscallDesc("lremovexattr", unimplementedFunc),
    /* 237 */ SyscallDesc("fremovexattr", unimplementedFunc),
    /* 238 */ SyscallDesc("tkill", unimplementedFunc),
    /* 239 */ SyscallDesc("sendfile64", unimplementedFunc),
    /* 240 */ SyscallDesc("futex", ignoreFunc, SyscallDesc::WarnOnce),
    /* 241 */ SyscallDesc("sched_setaffinity", unimplementedFunc),
    /* 242 */ SyscallDesc("sched_getaffinity", unimplementedFunc),
    /* 243 */ SyscallDesc("io_setup", unimplementedFunc),
    /* 244 */ SyscallDesc("io_destroy", unimplementedFunc),
    /* 245 */ SyscallDesc("io_getevents", unimplementedFunc),
    /* 246 */ SyscallDesc("io_submit", unimplementedFunc),
    /* 247 */ SyscallDesc("io_cancel", unimplementedFunc),
    /* 248 */ SyscallDesc("exit_group", exitGroupFunc),
    /* 249 */ SyscallDesc("lookup_dcookie", unimplementedFunc),
    /* 250 */ SyscallDesc("epoll_create", unimplementedFunc),
    /* 251 */ SyscallDesc("epoll_ctl", unimplementedFunc),
    /* 252 */ SyscallDesc("epoll_wait", unimplementedFunc),
    /* 253 */ SyscallDesc("remap_file_pages", unimplementedFunc),
    /* 254 */ SyscallDesc("unused#254", unimplementedFunc),
    /* 255 */ SyscallDesc("unused#255", unimplementedFunc),
    /* 256 */ SyscallDesc("set_tid_address", unimplementedFunc),
    /* 257 */ SyscallDesc("timer_create", unimplementedFunc),
    /* 258 */ SyscallDesc("timer_settime", unimplementedFunc),
    /* 259 */ SyscallDesc("timer_gettime", unimplementedFunc),
    /* 260 */ SyscallDesc("timer_getoverrun", unimplementedFunc),
    /* 261 */ SyscallDesc("timer_delete", unimplementedFunc),
    /* 262 */ SyscallDesc("clock_settime", unimplementedFunc),
    /* 263 */ SyscallDesc("clock_gettime", clock_gettimeFunc<ArmLinux32>),
    /* 264 */ SyscallDesc("clock_getres", clock_getresFunc<ArmLinux32>),
    /* 265 */ SyscallDesc("clock_nanosleep", unimplementedFunc),
    /* 266 */ SyscallDesc("statfs64", unimplementedFunc),
    /* 267 */ SyscallDesc("fstatfs64", unimplementedFunc),
    /* 268 */ SyscallDesc("tgkill", unimplementedFunc),
    /* 269 */ SyscallDesc("utimes", unimplementedFunc),
    /* 270 */ SyscallDesc("arm_fadvise64_64", unimplementedFunc),
    /* 271 */ SyscallDesc("pciconfig_iobase", unimplementedFunc),
    /* 272 */ SyscallDesc("pciconfig_read", unimplementedFunc),
    /* 273 */ SyscallDesc("pciconfig_write", unimplementedFunc),
    /* 274 */ SyscallDesc("mq_open", unimplementedFunc),
    /* 275 */ SyscallDesc("mq_unlink", unimplementedFunc),
    /* 276 */ SyscallDesc("mq_timedsend", unimplementedFunc),
    /* 277 */ SyscallDesc("mq_timedreceive", unimplementedFunc),
    /* 278 */ SyscallDesc("mq_notify", unimplementedFunc),
    /* 279 */ SyscallDesc("mq_getsetattr", unimplementedFunc),
    /* 280 */ SyscallDesc("waitid", unimplementedFunc),
    /* 281 */ SyscallDesc("socket", unimplementedFunc),
    /* 282 */ SyscallDesc("bind", unimplementedFunc),
    /* 283 */ SyscallDesc("connect", unimplementedFunc),
    /* 284 */ SyscallDesc("listen", unimplementedFunc),
    /* 285 */ SyscallDesc("accept", unimplementedFunc),
    /* 286 */ SyscallDesc("getsockname", unimplementedFunc),
    /* 287 */ SyscallDesc("getpeername", unimplementedFunc),
    /* 288 */ SyscallDesc("socketpair", unimplementedFunc),
    /* 289 */ SyscallDesc("send", unimplementedFunc),
    /* 290 */ SyscallDesc("sendto", unimplementedFunc),
    /* 291 */ SyscallDesc("recv", unimplementedFunc),
    /* 292 */ SyscallDesc("recvfrom", unimplementedFunc),
    /* 293 */ SyscallDesc("shutdown", unimplementedFunc),
    /* 294 */ SyscallDesc("setsockopt", unimplementedFunc),
    /* 295 */ SyscallDesc("getsockopt", unimplementedFunc),
    /* 296 */ SyscallDesc("sendmsg", unimplementedFunc),
    /* 297 */ SyscallDesc("rcvmsg", unimplementedFunc),
    /* 298 */ SyscallDesc("semop", unimplementedFunc),
    /* 299 */ SyscallDesc("semget", unimplementedFunc),
    /* 300 */ SyscallDesc("semctl", unimplementedFunc),
    /* 301 */ SyscallDesc("msgsend", unimplementedFunc),
    /* 302 */ SyscallDesc("msgrcv", unimplementedFunc),
    /* 303 */ SyscallDesc("msgget", unimplementedFunc),
    /* 304 */ SyscallDesc("msgctl", unimplementedFunc),
    /* 305 */ SyscallDesc("shmat", unimplementedFunc),
    /* 306 */ SyscallDesc("shmdt", unimplementedFunc),
    /* 307 */ SyscallDesc("shmget", unimplementedFunc),
    /* 308 */ SyscallDesc("shmctl", unimplementedFunc),
    /* 309 */ SyscallDesc("add_key", unimplementedFunc),
    /* 310 */ SyscallDesc("request_key", unimplementedFunc),
    /* 311 */ SyscallDesc("keyctl", unimplementedFunc),
    /* 312 */ SyscallDesc("semtimedop", unimplementedFunc),
    /* 313 */ SyscallDesc("unused#313", unimplementedFunc),
    /* 314 */ SyscallDesc("ioprio_set", unimplementedFunc),
    /* 315 */ SyscallDesc("ioprio_get", unimplementedFunc),
    /* 316 */ SyscallDesc("inotify_init", unimplementedFunc),
    /* 317 */ SyscallDesc("inotify_add_watch", unimplementedFunc),
    /* 318 */ SyscallDesc("inotify_rm_watch", unimplementedFunc),
    /* 319 */ SyscallDesc("mbind", unimplementedFunc),
    /* 320 */ SyscallDesc("get_mempolicy", unimplementedFunc),
    /* 321 */ SyscallDesc("set_mempolicy", unimplementedFunc),
    /* 322 */ SyscallDesc("openat", unimplementedFunc),
    /* 323 */ SyscallDesc("mkdirat", unimplementedFunc),
    /* 324 */ SyscallDesc("mknodat", unimplementedFunc),
    /* 325 */ SyscallDesc("fchownat", unimplementedFunc),
    /* 326 */ SyscallDesc("futimesat", unimplementedFunc),
    /* 327 */ SyscallDesc("fstatat64", unimplementedFunc),
    /* 328 */ SyscallDesc("unlinkat", unimplementedFunc),
    /* 329 */ SyscallDesc("renameat", unimplementedFunc),
    /* 330 */ SyscallDesc("linkat", unimplementedFunc),
    /* 331 */ SyscallDesc("symlinkat", unimplementedFunc),
    /* 332 */ SyscallDesc("readlinkat", unimplementedFunc),
    /* 333 */ SyscallDesc("fchmodat", unimplementedFunc),
    /* 334 */ SyscallDesc("faccessat", unimplementedFunc),
    /* 335 */ SyscallDesc("pselect6", unimplementedFunc),
    /* 336 */ SyscallDesc("ppoll", unimplementedFunc),
    /* 337 */ SyscallDesc("unshare", unimplementedFunc),
    /* 338 */ SyscallDesc("set_robust_list", unimplementedFunc),
    /* 339 */ SyscallDesc("get_robust_list", unimplementedFunc),
    /* 340 */ SyscallDesc("splice", unimplementedFunc),
    /* 341 */ SyscallDesc("arm_sync_file_range", unimplementedFunc),
    /* 342 */ SyscallDesc("tee", unimplementedFunc),
    /* 343 */ SyscallDesc("vmsplice", unimplementedFunc),
    /* 344 */ SyscallDesc("move_pages", unimplementedFunc),
    /* 345 */ SyscallDesc("getcpu", unimplementedFunc),
    /* 346 */ SyscallDesc("epoll_pwait", unimplementedFunc),
    /* 347 */ SyscallDesc("sys_kexec_load", unimplementedFunc),
    /* 348 */ SyscallDesc("sys_utimensat", unimplementedFunc),
    /* 349 */ SyscallDesc("sys_signalfd", unimplementedFunc),
    /* 350 */ SyscallDesc("sys_timerfd_create", unimplementedFunc),
    /* 351 */ SyscallDesc("sys_eventfd", unimplementedFunc),
    /* 352 */ SyscallDesc("sys_fallocate", unimplementedFunc),
    /* 353 */ SyscallDesc("sys_timerfd_settime", unimplementedFunc),
    /* 354 */ SyscallDesc("sys_timerfd_gettime", unimplementedFunc),
    /* 355 */ SyscallDesc("sys_signalfd4", unimplementedFunc),
    /* 356 */ SyscallDesc("sys_eventfd2", unimplementedFunc),
    /* 357 */ SyscallDesc("sys_epoll_create1", unimplementedFunc),
    /* 358 */ SyscallDesc("sys_dup3", unimplementedFunc),
    /* 359 */ SyscallDesc("sys_pipe2", unimplementedFunc),
    /* 360 */ SyscallDesc("sys_inotify_init1", unimplementedFunc),
    /* 361 */ SyscallDesc("sys_preadv", unimplementedFunc),
    /* 362 */ SyscallDesc("sys_pwritev", unimplementedFunc),
    /* 363 */ SyscallDesc("sys_rt_tgsigqueueinfo", unimplementedFunc),
    /* 364 */ SyscallDesc("sys_perf_event_open", unimplementedFunc),
    /* 365 */ SyscallDesc("sys_recvmmsg", unimplementedFunc),
};

static SyscallDesc syscallDescs64[] = {
    /*    0 */ SyscallDesc("io_setup", unimplementedFunc),
    /*    1 */ SyscallDesc("io_destroy", unimplementedFunc),
    /*    2 */ SyscallDesc("io_submit", unimplementedFunc),
    /*    3 */ SyscallDesc("io_cancel", unimplementedFunc),
    /*    4 */ SyscallDesc("io_getevents", unimplementedFunc),
    /*    5 */ SyscallDesc("setxattr", unimplementedFunc),
    /*    6 */ SyscallDesc("lsetxattr", unimplementedFunc),
    /*    7 */ SyscallDesc("fsetxattr", unimplementedFunc),
    /*    8 */ SyscallDesc("getxattr", unimplementedFunc),
    /*    9 */ SyscallDesc("lgetxattr", unimplementedFunc),
    /*   10 */ SyscallDesc("fgetxattr", unimplementedFunc),
    /*   11 */ SyscallDesc("listxattr", unimplementedFunc),
    /*   12 */ SyscallDesc("llistxattr", unimplementedFunc),
    /*   13 */ SyscallDesc("flistxattr", unimplementedFunc),
    /*   14 */ SyscallDesc("removexattr", unimplementedFunc),
    /*   15 */ SyscallDesc("lremovexattr", unimplementedFunc),
    /*   16 */ SyscallDesc("fremovexattr", unimplementedFunc),
    /*   17 */ SyscallDesc("getcwd", getcwdFunc),
    /*   18 */ SyscallDesc("lookup_dcookie", unimplementedFunc),
    /*   19 */ SyscallDesc("eventfd2", unimplementedFunc),
    /*   20 */ SyscallDesc("epoll_create1", unimplementedFunc),
    /*   21 */ SyscallDesc("epoll_ctl", unimplementedFunc),
    /*   22 */ SyscallDesc("epoll_pwait", unimplementedFunc),
    /*   23 */ SyscallDesc("dup", dupFunc),
    /*   24 */ SyscallDesc("dup3", unimplementedFunc),
    /*   25 */ SyscallDesc("fcntl64", fcntl64Func),
    /*   26 */ SyscallDesc("inotify_init1", unimplementedFunc),
    /*   27 */ SyscallDesc("inotify_add_watch", unimplementedFunc),
    /*   28 */ SyscallDesc("inotify_rm_watch", unimplementedFunc),
    /*   29 */ SyscallDesc("ioctl", ioctlFunc<ArmLinux64>),
    /*   30 */ SyscallDesc("ioprio_set", unimplementedFunc),
    /*   31 */ SyscallDesc("ioprio_get", unimplementedFunc),
    /*   32 */ SyscallDesc("flock", unimplementedFunc),
    /*   33 */ SyscallDesc("mknodat", unimplementedFunc),
    /*   34 */ SyscallDesc("mkdirat", unimplementedFunc),
    /*   35 */ SyscallDesc("unlinkat", unlinkatFunc<ArmLinux64>),
    /*   36 */ SyscallDesc("symlinkat", unimplementedFunc),
    /*   37 */ SyscallDesc("linkat", unimplementedFunc),
    /*   38 */ SyscallDesc("renameat", renameatFunc<ArmLinux64>),
    /*   39 */ SyscallDesc("umount2", unimplementedFunc),
    /*   40 */ SyscallDesc("mount", unimplementedFunc),
    /*   41 */ SyscallDesc("pivot_root", unimplementedFunc),
    /*   42 */ SyscallDesc("nfsservctl", unimplementedFunc),
    /*   43 */ SyscallDesc("statfs64", unimplementedFunc),
    /*   44 */ SyscallDesc("fstatfs64", unimplementedFunc),
    /*   45 */ SyscallDesc("truncate64", unimplementedFunc),
    /*   46 */ SyscallDesc("ftruncate64", ftruncate64Func),
    /*   47 */ SyscallDesc("fallocate", unimplementedFunc),
    /*   48 */ SyscallDesc("faccessat", faccessatFunc<ArmLinux64>),
    /*   49 */ SyscallDesc("chdir", unimplementedFunc),
    /*   50 */ SyscallDesc("fchdir", unimplementedFunc),
    /*   51 */ SyscallDesc("chroot", unimplementedFunc),
    /*   52 */ SyscallDesc("fchmod", unimplementedFunc),
    /*   53 */ SyscallDesc("fchmodat", unimplementedFunc),
    /*   54 */ SyscallDesc("fchownat", unimplementedFunc),
    /*   55 */ SyscallDesc("fchown", unimplementedFunc),
    /*   56 */ SyscallDesc("openat", openatFunc<ArmLinux64>),
    /*   57 */ SyscallDesc("close", closeFunc),
    /*   58 */ SyscallDesc("vhangup", unimplementedFunc),
    /*   59 */ SyscallDesc("pipe2", unimplementedFunc),
    /*   60 */ SyscallDesc("quotactl", unimplementedFunc),
    /*   61 */ SyscallDesc("getdents64", unimplementedFunc),
    /*   62 */ SyscallDesc("llseek", lseekFunc),
    /*   63 */ SyscallDesc("read", readFunc),
    /*   64 */ SyscallDesc("write", writeFunc),
    /*   65 */ SyscallDesc("readv", unimplementedFunc),
    /*   66 */ SyscallDesc("writev", writevFunc<ArmLinux64>),
    /*   67 */ SyscallDesc("pread64", unimplementedFunc),
    /*   68 */ SyscallDesc("pwrite64", unimplementedFunc),
    /*   69 */ SyscallDesc("preadv", unimplementedFunc),
    /*   70 */ SyscallDesc("pwritev", unimplementedFunc),
    /*   71 */ SyscallDesc("sendfile64", unimplementedFunc),
    /*   72 */ SyscallDesc("pselect6", unimplementedFunc),
    /*   73 */ SyscallDesc("ppoll", unimplementedFunc),
    /*   74 */ SyscallDesc("signalfd4", unimplementedFunc),
    /*   75 */ SyscallDesc("vmsplice", unimplementedFunc),
    /*   76 */ SyscallDesc("splice", unimplementedFunc),
    /*   77 */ SyscallDesc("tee", unimplementedFunc),
    /*   78 */ SyscallDesc("readlinkat", readlinkatFunc<ArmLinux64>),
    /*   79 */ SyscallDesc("fstatat64", fstatat64Func<ArmLinux64>),
    /*   80 */ SyscallDesc("fstat64", fstat64Func<ArmLinux64>),
    /*   81 */ SyscallDesc("sync", unimplementedFunc),
    /*   82 */ SyscallDesc("fsync", unimplementedFunc),
    /*   83 */ SyscallDesc("fdatasync", unimplementedFunc),
    /*   84 */ SyscallDesc("sync_file_range", unimplementedFunc),
    /*   85 */ SyscallDesc("timerfd_create", unimplementedFunc),
    /*   86 */ SyscallDesc("timerfd_settime", unimplementedFunc),
    /*   87 */ SyscallDesc("timerfd_gettime", unimplementedFunc),
    /*   88 */ SyscallDesc("utimensat", unimplementedFunc),
    /*   89 */ SyscallDesc("acct", unimplementedFunc),
    /*   90 */ SyscallDesc("capget", unimplementedFunc),
    /*   91 */ SyscallDesc("capset", unimplementedFunc),
    /*   92 */ SyscallDesc("personality", unimplementedFunc),
    /*   93 */ SyscallDesc("exit", exitFunc),
    /*   94 */ SyscallDesc("exit_group", exitGroupFunc),
    /*   95 */ SyscallDesc("waitid", unimplementedFunc),
    /*   96 */ SyscallDesc("set_tid_address", unimplementedFunc),
    /*   97 */ SyscallDesc("unshare", unimplementedFunc),
    /*   98 */ SyscallDesc("futex", unimplementedFunc),
    /*   99 */ SyscallDesc("set_robust_list", unimplementedFunc),
    /*  100 */ SyscallDesc("get_robust_list", unimplementedFunc),
    /*  101 */ SyscallDesc("nanosleep", ignoreFunc, SyscallDesc::WarnOnce),
    /*  102 */ SyscallDesc("getitimer", unimplementedFunc),
    /*  103 */ SyscallDesc("setitimer", unimplementedFunc),
    /*  104 */ SyscallDesc("kexec_load", unimplementedFunc),
    /*  105 */ SyscallDesc("init_module", unimplementedFunc),
    /*  106 */ SyscallDesc("delete_module", unimplementedFunc),
    /*  107 */ SyscallDesc("timer_create", unimplementedFunc),
    /*  108 */ SyscallDesc("timer_gettime", unimplementedFunc),
    /*  109 */ SyscallDesc("timer_getoverrun", unimplementedFunc),
    /*  110 */ SyscallDesc("timer_settime", unimplementedFunc),
    /*  111 */ SyscallDesc("timer_delete", unimplementedFunc),
    /*  112 */ SyscallDesc("clock_settime", unimplementedFunc),
    /*  113 */ SyscallDesc("clock_gettime", clock_gettimeFunc<ArmLinux64>),
    /*  114 */ SyscallDesc("clock_getres", unimplementedFunc),
    /*  115 */ SyscallDesc("clock_nanosleep", unimplementedFunc),
    /*  116 */ SyscallDesc("syslog", unimplementedFunc),
    /*  117 */ SyscallDesc("ptrace", unimplementedFunc),
    /*  118 */ SyscallDesc("sched_setparam", unimplementedFunc),
    /*  119 */ SyscallDesc("sched_setscheduler", unimplementedFunc),
    /*  120 */ SyscallDesc("sched_getscheduler", unimplementedFunc),
    /*  121 */ SyscallDesc("sched_getparam", unimplementedFunc),
    /*  122 */ SyscallDesc("sched_setaffinity", unimplementedFunc),
    /*  123 */ SyscallDesc("sched_getaffinity", unimplementedFunc),
    /*  124 */ SyscallDesc("sched_yield", unimplementedFunc),
    /*  125 */ SyscallDesc("sched_get_priority_max", unimplementedFunc),
    /*  126 */ SyscallDesc("sched_get_priority_min", unimplementedFunc),
    /*  127 */ SyscallDesc("sched_rr_get_interval", unimplementedFunc),
    /*  128 */ SyscallDesc("restart_syscall", unimplementedFunc),
    /*  129 */ SyscallDesc("kill", ignoreFunc),
    /*  130 */ SyscallDesc("tkill", unimplementedFunc),
    /*  131 */ SyscallDesc("tgkill", unimplementedFunc),
    /*  132 */ SyscallDesc("sigaltstack", unimplementedFunc),
    /*  133 */ SyscallDesc("rt_sigsuspend", unimplementedFunc),
    /*  134 */ SyscallDesc("rt_sigaction", ignoreFunc),
    /*  135 */ SyscallDesc("rt_sigprocmask", ignoreFunc, SyscallDesc::WarnOnce),
    /*  136 */ SyscallDesc("rt_sigpending", unimplementedFunc),
    /*  137 */ SyscallDesc("rt_sigtimedwait", unimplementedFunc),
    /*  138 */ SyscallDesc("rt_sigqueueinfo", ignoreFunc),
    /*  139 */ SyscallDesc("rt_sigreturn", unimplementedFunc),
    /*  140 */ SyscallDesc("setpriority", unimplementedFunc),
    /*  141 */ SyscallDesc("getpriority", unimplementedFunc),
    /*  142 */ SyscallDesc("reboot", unimplementedFunc),
    /*  143 */ SyscallDesc("setregid", unimplementedFunc),
    /*  144 */ SyscallDesc("setgid", unimplementedFunc),
    /*  145 */ SyscallDesc("setreuid", unimplementedFunc),
    /*  146 */ SyscallDesc("setuid", unimplementedFunc),
    /*  147 */ SyscallDesc("setresuid", unimplementedFunc),
    /*  148 */ SyscallDesc("getresuid", unimplementedFunc),
    /*  149 */ SyscallDesc("setresgid", unimplementedFunc),
    /*  150 */ SyscallDesc("getresgid", unimplementedFunc),
    /*  151 */ SyscallDesc("setfsuid", unimplementedFunc),
    /*  152 */ SyscallDesc("setfsgid", unimplementedFunc),
    /*  153 */ SyscallDesc("times", timesFunc<ArmLinux64>),
    /*  154 */ SyscallDesc("setpgid", unimplementedFunc),
    /*  155 */ SyscallDesc("getpgid", unimplementedFunc),
    /*  156 */ SyscallDesc("getsid", unimplementedFunc),
    /*  157 */ SyscallDesc("setsid", unimplementedFunc),
    /*  158 */ SyscallDesc("getgroups", unimplementedFunc),
    /*  159 */ SyscallDesc("setgroups", unimplementedFunc),
    /*  160 */ SyscallDesc("uname", unameFunc64),
    /*  161 */ SyscallDesc("sethostname", ignoreFunc),
    /*  162 */ SyscallDesc("setdomainname", unimplementedFunc),
    /*  163 */ SyscallDesc("getrlimit", getrlimitFunc<ArmLinux64>),
    /*  164 */ SyscallDesc("setrlimit", ignoreFunc),
    /*  165 */ SyscallDesc("getrusage", getrusageFunc<ArmLinux64>),
    /*  166 */ SyscallDesc("umask", unimplementedFunc),
    /*  167 */ SyscallDesc("prctl", unimplementedFunc),
    /*  168 */ SyscallDesc("getcpu", unimplementedFunc),
    /*  169 */ SyscallDesc("gettimeofday", gettimeofdayFunc<ArmLinux64>),
    /*  170 */ SyscallDesc("settimeofday", unimplementedFunc),
    /*  171 */ SyscallDesc("adjtimex", unimplementedFunc),
    /*  172 */ SyscallDesc("getpid", getpidFunc),
    /*  173 */ SyscallDesc("getppid", getppidFunc),
    /*  174 */ SyscallDesc("getuid", getuidFunc),
    /*  175 */ SyscallDesc("geteuid", geteuidFunc),
    /*  176 */ SyscallDesc("getgid", getgidFunc),
    /*  177 */ SyscallDesc("getegid", getegidFunc),
    /*  178 */ SyscallDesc("gettid", unimplementedFunc),
    /*  179 */ SyscallDesc("sysinfo", sysinfoFunc<ArmLinux64>),
    /*  180 */ SyscallDesc("mq_open", unimplementedFunc),
    /*  181 */ SyscallDesc("mq_unlink", unimplementedFunc),
    /*  182 */ SyscallDesc("mq_timedsend", unimplementedFunc),
    /*  183 */ SyscallDesc("mq_timedreceive", unimplementedFunc),
    /*  184 */ SyscallDesc("mq_notify", unimplementedFunc),
    /*  185 */ SyscallDesc("mq_getsetattr", unimplementedFunc),
    /*  186 */ SyscallDesc("msgget", unimplementedFunc),
    /*  187 */ SyscallDesc("msgctl", unimplementedFunc),
    /*  188 */ SyscallDesc("msgrcv", unimplementedFunc),
    /*  189 */ SyscallDesc("msgsnd", unimplementedFunc),
    /*  190 */ SyscallDesc("semget", unimplementedFunc),
    /*  191 */ SyscallDesc("semctl", unimplementedFunc),
    /*  192 */ SyscallDesc("semtimedop", unimplementedFunc),
    /*  193 */ SyscallDesc("semop", unimplementedFunc),
    /*  194 */ SyscallDesc("shmget", unimplementedFunc),
    /*  195 */ SyscallDesc("shmctl", unimplementedFunc),
    /*  196 */ SyscallDesc("shmat", unimplementedFunc),
    /*  197 */ SyscallDesc("shmdt", unimplementedFunc),
    /*  198 */ SyscallDesc("socket", unimplementedFunc),
    /*  199 */ SyscallDesc("socketpair", unimplementedFunc),
    /*  200 */ SyscallDesc("bind", unimplementedFunc),
    /*  201 */ SyscallDesc("listen", unimplementedFunc),
    /*  202 */ SyscallDesc("accept", unimplementedFunc),
    /*  203 */ SyscallDesc("connect", unimplementedFunc),
    /*  204 */ SyscallDesc("getsockname", unimplementedFunc),
    /*  205 */ SyscallDesc("getpeername", unimplementedFunc),
    /*  206 */ SyscallDesc("sendto", unimplementedFunc),
    /*  207 */ SyscallDesc("recvfrom", unimplementedFunc),
    /*  208 */ SyscallDesc("setsockopt", unimplementedFunc),
    /*  209 */ SyscallDesc("getsockopt", unimplementedFunc),
    /*  210 */ SyscallDesc("shutdown", unimplementedFunc),
    /*  211 */ SyscallDesc("sendmsg", unimplementedFunc),
    /*  212 */ SyscallDesc("recvmsg", unimplementedFunc),
    /*  213 */ SyscallDesc("readahead", unimplementedFunc),
    /*  214 */ SyscallDesc("brk", brkFunc),
    /*  215 */ SyscallDesc("munmap", munmapFunc),
    /*  216 */ SyscallDesc("mremap", mremapFunc<ArmLinux64>),
    /*  217 */ SyscallDesc("add_key", unimplementedFunc),
    /*  218 */ SyscallDesc("request_key", unimplementedFunc),
    /*  219 */ SyscallDesc("keyctl", unimplementedFunc),
    /*  220 */ SyscallDesc("clone", unimplementedFunc),
    /*  221 */ SyscallDesc("execve", unimplementedFunc),
    /*  222 */ SyscallDesc("mmap2", mmapFunc<ArmLinux64>),
    /*  223 */ SyscallDesc("fadvise64_64", unimplementedFunc),
    /*  224 */ SyscallDesc("swapon", unimplementedFunc),
    /*  225 */ SyscallDesc("swapoff", unimplementedFunc),
    /*  226 */ SyscallDesc("mprotect", ignoreFunc),
    /*  227 */ SyscallDesc("msync", unimplementedFunc),
    /*  228 */ SyscallDesc("mlock", unimplementedFunc),
    /*  229 */ SyscallDesc("munlock", unimplementedFunc),
    /*  230 */ SyscallDesc("mlockall", unimplementedFunc),
    /*  231 */ SyscallDesc("munlockall", unimplementedFunc),
    /*  232 */ SyscallDesc("mincore", unimplementedFunc),
    /*  233 */ SyscallDesc("madvise", unimplementedFunc),
    /*  234 */ SyscallDesc("remap_file_pages", unimplementedFunc),
    /*  235 */ SyscallDesc("mbind", unimplementedFunc),
    /*  236 */ SyscallDesc("get_mempolicy", unimplementedFunc),
    /*  237 */ SyscallDesc("set_mempolicy", unimplementedFunc),
    /*  238 */ SyscallDesc("migrate_pages", unimplementedFunc),
    /*  239 */ SyscallDesc("move_pages", unimplementedFunc),
    /*  240 */ SyscallDesc("rt_tgsigqueueinfo", unimplementedFunc),
    /*  241 */ SyscallDesc("perf_event_open", unimplementedFunc),
    /*  242 */ SyscallDesc("accept4", unimplementedFunc),
    /*  243 */ SyscallDesc("recvmmsg", unimplementedFunc),
    /*  244 */ SyscallDesc("unused#244", unimplementedFunc),
    /*  245 */ SyscallDesc("unused#245", unimplementedFunc),
    /*  246 */ SyscallDesc("unused#246", unimplementedFunc),
    /*  247 */ SyscallDesc("unused#247", unimplementedFunc),
    /*  248 */ SyscallDesc("unused#248", unimplementedFunc),
    /*  249 */ SyscallDesc("unused#249", unimplementedFunc),
    /*  250 */ SyscallDesc("unused#250", unimplementedFunc),
    /*  251 */ SyscallDesc("unused#251", unimplementedFunc),
    /*  252 */ SyscallDesc("unused#252", unimplementedFunc),
    /*  253 */ SyscallDesc("unused#253", unimplementedFunc),
    /*  254 */ SyscallDesc("unused#254", unimplementedFunc),
    /*  255 */ SyscallDesc("unused#255", unimplementedFunc),
    /*  256 */ SyscallDesc("unused#256", unimplementedFunc),
    /*  257 */ SyscallDesc("unused#257", unimplementedFunc),
    /*  258 */ SyscallDesc("unused#258", unimplementedFunc),
    /*  259 */ SyscallDesc("unused#259", unimplementedFunc),
    /*  260 */ SyscallDesc("wait4", unimplementedFunc),
    /*  261 */ SyscallDesc("prlimit64", unimplementedFunc),
    /*  262 */ SyscallDesc("fanotify_init", unimplementedFunc),
    /*  263 */ SyscallDesc("fanotify_mark", unimplementedFunc),
    /*  264 */ SyscallDesc("name_to_handle_at", unimplementedFunc),
    /*  265 */ SyscallDesc("open_by_handle_at", unimplementedFunc),
    /*  266 */ SyscallDesc("clock_adjtime", unimplementedFunc),
    /*  267 */ SyscallDesc("syncfs", unimplementedFunc),
    /*  268 */ SyscallDesc("setns", unimplementedFunc),
    /*  269 */ SyscallDesc("sendmmsg", unimplementedFunc),
    /*  270 */ SyscallDesc("process_vm_readv", unimplementedFunc),
    /*  271 */ SyscallDesc("process_vm_writev", unimplementedFunc),
    /*  272 */ SyscallDesc("unused#272", unimplementedFunc),
    /*  273 */ SyscallDesc("unused#273", unimplementedFunc),
    /*  274 */ SyscallDesc("unused#274", unimplementedFunc),
    /*  275 */ SyscallDesc("unused#275", unimplementedFunc),
    /*  276 */ SyscallDesc("unused#276", unimplementedFunc),
    /*  277 */ SyscallDesc("unused#277", unimplementedFunc),
    /*  278 */ SyscallDesc("unused#278", unimplementedFunc),
    /*  279 */ SyscallDesc("unused#279", unimplementedFunc),
    /*  280 */ SyscallDesc("unused#280", unimplementedFunc),
    /*  281 */ SyscallDesc("unused#281", unimplementedFunc),
    /*  282 */ SyscallDesc("unused#282", unimplementedFunc),
    /*  283 */ SyscallDesc("unused#283", unimplementedFunc),
    /*  284 */ SyscallDesc("unused#284", unimplementedFunc),
    /*  285 */ SyscallDesc("unused#285", unimplementedFunc),
    /*  286 */ SyscallDesc("unused#286", unimplementedFunc),
    /*  287 */ SyscallDesc("unused#287", unimplementedFunc),
    /*  288 */ SyscallDesc("unused#288", unimplementedFunc),
    /*  289 */ SyscallDesc("unused#289", unimplementedFunc),
    /*  290 */ SyscallDesc("unused#290", unimplementedFunc),
    /*  291 */ SyscallDesc("unused#291", unimplementedFunc),
    /*  292 */ SyscallDesc("unused#292", unimplementedFunc),
    /*  293 */ SyscallDesc("unused#293", unimplementedFunc),
    /*  294 */ SyscallDesc("unused#294", unimplementedFunc),
    /*  295 */ SyscallDesc("unused#295", unimplementedFunc),
    /*  296 */ SyscallDesc("unused#296", unimplementedFunc),
    /*  297 */ SyscallDesc("unused#297", unimplementedFunc),
    /*  298 */ SyscallDesc("unused#298", unimplementedFunc),
    /*  299 */ SyscallDesc("unused#299", unimplementedFunc),
    /*  300 */ SyscallDesc("unused#300", unimplementedFunc),
    /*  301 */ SyscallDesc("unused#301", unimplementedFunc),
    /*  302 */ SyscallDesc("unused#302", unimplementedFunc),
    /*  303 */ SyscallDesc("unused#303", unimplementedFunc),
    /*  304 */ SyscallDesc("unused#304", unimplementedFunc),
    /*  305 */ SyscallDesc("unused#305", unimplementedFunc),
    /*  306 */ SyscallDesc("unused#306", unimplementedFunc),
    /*  307 */ SyscallDesc("unused#307", unimplementedFunc),
    /*  308 */ SyscallDesc("unused#308", unimplementedFunc),
    /*  309 */ SyscallDesc("unused#309", unimplementedFunc),
    /*  310 */ SyscallDesc("unused#310", unimplementedFunc),
    /*  311 */ SyscallDesc("unused#311", unimplementedFunc),
    /*  312 */ SyscallDesc("unused#312", unimplementedFunc),
    /*  313 */ SyscallDesc("unused#313", unimplementedFunc),
    /*  314 */ SyscallDesc("unused#314", unimplementedFunc),
    /*  315 */ SyscallDesc("unused#315", unimplementedFunc),
    /*  316 */ SyscallDesc("unused#316", unimplementedFunc),
    /*  317 */ SyscallDesc("unused#317", unimplementedFunc),
    /*  318 */ SyscallDesc("unused#318", unimplementedFunc),
    /*  319 */ SyscallDesc("unused#319", unimplementedFunc),
    /*  320 */ SyscallDesc("unused#320", unimplementedFunc),
    /*  321 */ SyscallDesc("unused#321", unimplementedFunc),
    /*  322 */ SyscallDesc("unused#322", unimplementedFunc),
    /*  323 */ SyscallDesc("unused#323", unimplementedFunc),
    /*  324 */ SyscallDesc("unused#324", unimplementedFunc),
    /*  325 */ SyscallDesc("unused#325", unimplementedFunc),
    /*  326 */ SyscallDesc("unused#326", unimplementedFunc),
    /*  327 */ SyscallDesc("unused#327", unimplementedFunc),
    /*  328 */ SyscallDesc("unused#328", unimplementedFunc),
    /*  329 */ SyscallDesc("unused#329", unimplementedFunc),
    /*  330 */ SyscallDesc("unused#330", unimplementedFunc),
    /*  331 */ SyscallDesc("unused#331", unimplementedFunc),
    /*  332 */ SyscallDesc("unused#332", unimplementedFunc),
    /*  333 */ SyscallDesc("unused#333", unimplementedFunc),
    /*  334 */ SyscallDesc("unused#334", unimplementedFunc),
    /*  335 */ SyscallDesc("unused#335", unimplementedFunc),
    /*  336 */ SyscallDesc("unused#336", unimplementedFunc),
    /*  337 */ SyscallDesc("unused#337", unimplementedFunc),
    /*  338 */ SyscallDesc("unused#338", unimplementedFunc),
    /*  339 */ SyscallDesc("unused#339", unimplementedFunc),
    /*  340 */ SyscallDesc("unused#340", unimplementedFunc),
    /*  341 */ SyscallDesc("unused#341", unimplementedFunc),
    /*  342 */ SyscallDesc("unused#342", unimplementedFunc),
    /*  343 */ SyscallDesc("unused#343", unimplementedFunc),
    /*  344 */ SyscallDesc("unused#344", unimplementedFunc),
    /*  345 */ SyscallDesc("unused#345", unimplementedFunc),
    /*  346 */ SyscallDesc("unused#346", unimplementedFunc),
    /*  347 */ SyscallDesc("unused#347", unimplementedFunc),
    /*  348 */ SyscallDesc("unused#348", unimplementedFunc),
    /*  349 */ SyscallDesc("unused#349", unimplementedFunc),
    /*  350 */ SyscallDesc("unused#350", unimplementedFunc),
    /*  351 */ SyscallDesc("unused#351", unimplementedFunc),
    /*  352 */ SyscallDesc("unused#352", unimplementedFunc),
    /*  353 */ SyscallDesc("unused#353", unimplementedFunc),
    /*  354 */ SyscallDesc("unused#354", unimplementedFunc),
    /*  355 */ SyscallDesc("unused#355", unimplementedFunc),
    /*  356 */ SyscallDesc("unused#356", unimplementedFunc),
    /*  357 */ SyscallDesc("unused#357", unimplementedFunc),
    /*  358 */ SyscallDesc("unused#358", unimplementedFunc),
    /*  359 */ SyscallDesc("unused#359", unimplementedFunc),
    /*  360 */ SyscallDesc("unused#360", unimplementedFunc),
    /*  361 */ SyscallDesc("unused#361", unimplementedFunc),
    /*  362 */ SyscallDesc("unused#362", unimplementedFunc),
    /*  363 */ SyscallDesc("unused#363", unimplementedFunc),
    /*  364 */ SyscallDesc("unused#364", unimplementedFunc),
    /*  365 */ SyscallDesc("unused#365", unimplementedFunc),
    /*  366 */ SyscallDesc("unused#366", unimplementedFunc),
    /*  367 */ SyscallDesc("unused#367", unimplementedFunc),
    /*  368 */ SyscallDesc("unused#368", unimplementedFunc),
    /*  369 */ SyscallDesc("unused#369", unimplementedFunc),
    /*  370 */ SyscallDesc("unused#370", unimplementedFunc),
    /*  371 */ SyscallDesc("unused#371", unimplementedFunc),
    /*  372 */ SyscallDesc("unused#372", unimplementedFunc),
    /*  373 */ SyscallDesc("unused#373", unimplementedFunc),
    /*  374 */ SyscallDesc("unused#374", unimplementedFunc),
    /*  375 */ SyscallDesc("unused#375", unimplementedFunc),
    /*  376 */ SyscallDesc("unused#376", unimplementedFunc),
    /*  377 */ SyscallDesc("unused#377", unimplementedFunc),
    /*  378 */ SyscallDesc("unused#378", unimplementedFunc),
    /*  379 */ SyscallDesc("unused#379", unimplementedFunc),
    /*  380 */ SyscallDesc("unused#380", unimplementedFunc),
    /*  381 */ SyscallDesc("unused#381", unimplementedFunc),
    /*  382 */ SyscallDesc("unused#382", unimplementedFunc),
    /*  383 */ SyscallDesc("unused#383", unimplementedFunc),
    /*  384 */ SyscallDesc("unused#384", unimplementedFunc),
    /*  385 */ SyscallDesc("unused#385", unimplementedFunc),
    /*  386 */ SyscallDesc("unused#386", unimplementedFunc),
    /*  387 */ SyscallDesc("unused#387", unimplementedFunc),
    /*  388 */ SyscallDesc("unused#388", unimplementedFunc),
    /*  389 */ SyscallDesc("unused#389", unimplementedFunc),
    /*  390 */ SyscallDesc("unused#390", unimplementedFunc),
    /*  391 */ SyscallDesc("unused#391", unimplementedFunc),
    /*  392 */ SyscallDesc("unused#392", unimplementedFunc),
    /*  393 */ SyscallDesc("unused#393", unimplementedFunc),
    /*  394 */ SyscallDesc("unused#394", unimplementedFunc),
    /*  395 */ SyscallDesc("unused#395", unimplementedFunc),
    /*  396 */ SyscallDesc("unused#396", unimplementedFunc),
    /*  397 */ SyscallDesc("unused#397", unimplementedFunc),
    /*  398 */ SyscallDesc("unused#398", unimplementedFunc),
    /*  399 */ SyscallDesc("unused#399", unimplementedFunc),
    /*  400 */ SyscallDesc("unused#400", unimplementedFunc),
    /*  401 */ SyscallDesc("unused#401", unimplementedFunc),
    /*  402 */ SyscallDesc("unused#402", unimplementedFunc),
    /*  403 */ SyscallDesc("unused#403", unimplementedFunc),
    /*  404 */ SyscallDesc("unused#404", unimplementedFunc),
    /*  405 */ SyscallDesc("unused#405", unimplementedFunc),
    /*  406 */ SyscallDesc("unused#406", unimplementedFunc),
    /*  407 */ SyscallDesc("unused#407", unimplementedFunc),
    /*  408 */ SyscallDesc("unused#408", unimplementedFunc),
    /*  409 */ SyscallDesc("unused#409", unimplementedFunc),
    /*  410 */ SyscallDesc("unused#410", unimplementedFunc),
    /*  411 */ SyscallDesc("unused#411", unimplementedFunc),
    /*  412 */ SyscallDesc("unused#412", unimplementedFunc),
    /*  413 */ SyscallDesc("unused#413", unimplementedFunc),
    /*  414 */ SyscallDesc("unused#414", unimplementedFunc),
    /*  415 */ SyscallDesc("unused#415", unimplementedFunc),
    /*  416 */ SyscallDesc("unused#416", unimplementedFunc),
    /*  417 */ SyscallDesc("unused#417", unimplementedFunc),
    /*  418 */ SyscallDesc("unused#418", unimplementedFunc),
    /*  419 */ SyscallDesc("unused#419", unimplementedFunc),
    /*  420 */ SyscallDesc("unused#420", unimplementedFunc),
    /*  421 */ SyscallDesc("unused#421", unimplementedFunc),
    /*  422 */ SyscallDesc("unused#422", unimplementedFunc),
    /*  423 */ SyscallDesc("unused#423", unimplementedFunc),
    /*  424 */ SyscallDesc("unused#424", unimplementedFunc),
    /*  425 */ SyscallDesc("unused#425", unimplementedFunc),
    /*  426 */ SyscallDesc("unused#426", unimplementedFunc),
    /*  427 */ SyscallDesc("unused#427", unimplementedFunc),
    /*  428 */ SyscallDesc("unused#428", unimplementedFunc),
    /*  429 */ SyscallDesc("unused#429", unimplementedFunc),
    /*  430 */ SyscallDesc("unused#430", unimplementedFunc),
    /*  431 */ SyscallDesc("unused#431", unimplementedFunc),
    /*  432 */ SyscallDesc("unused#432", unimplementedFunc),
    /*  433 */ SyscallDesc("unused#433", unimplementedFunc),
    /*  434 */ SyscallDesc("unused#434", unimplementedFunc),
    /*  435 */ SyscallDesc("unused#435", unimplementedFunc),
    /*  436 */ SyscallDesc("unused#436", unimplementedFunc),
    /*  437 */ SyscallDesc("unused#437", unimplementedFunc),
    /*  438 */ SyscallDesc("unused#438", unimplementedFunc),
    /*  439 */ SyscallDesc("unused#439", unimplementedFunc),
    /*  440 */ SyscallDesc("unused#440", unimplementedFunc),
    /*  441 */ SyscallDesc("unused#441", unimplementedFunc),
    /*  442 */ SyscallDesc("unused#442", unimplementedFunc),
    /*  443 */ SyscallDesc("unused#443", unimplementedFunc),
    /*  444 */ SyscallDesc("unused#444", unimplementedFunc),
    /*  445 */ SyscallDesc("unused#445", unimplementedFunc),
    /*  446 */ SyscallDesc("unused#446", unimplementedFunc),
    /*  447 */ SyscallDesc("unused#447", unimplementedFunc),
    /*  448 */ SyscallDesc("unused#448", unimplementedFunc),
    /*  449 */ SyscallDesc("unused#449", unimplementedFunc),
    /*  450 */ SyscallDesc("unused#450", unimplementedFunc),
    /*  451 */ SyscallDesc("unused#451", unimplementedFunc),
    /*  452 */ SyscallDesc("unused#452", unimplementedFunc),
    /*  453 */ SyscallDesc("unused#453", unimplementedFunc),
    /*  454 */ SyscallDesc("unused#454", unimplementedFunc),
    /*  455 */ SyscallDesc("unused#455", unimplementedFunc),
    /*  456 */ SyscallDesc("unused#456", unimplementedFunc),
    /*  457 */ SyscallDesc("unused#457", unimplementedFunc),
    /*  458 */ SyscallDesc("unused#458", unimplementedFunc),
    /*  459 */ SyscallDesc("unused#459", unimplementedFunc),
    /*  460 */ SyscallDesc("unused#460", unimplementedFunc),
    /*  461 */ SyscallDesc("unused#461", unimplementedFunc),
    /*  462 */ SyscallDesc("unused#462", unimplementedFunc),
    /*  463 */ SyscallDesc("unused#463", unimplementedFunc),
    /*  464 */ SyscallDesc("unused#464", unimplementedFunc),
    /*  465 */ SyscallDesc("unused#465", unimplementedFunc),
    /*  466 */ SyscallDesc("unused#466", unimplementedFunc),
    /*  467 */ SyscallDesc("unused#467", unimplementedFunc),
    /*  468 */ SyscallDesc("unused#468", unimplementedFunc),
    /*  469 */ SyscallDesc("unused#469", unimplementedFunc),
    /*  470 */ SyscallDesc("unused#470", unimplementedFunc),
    /*  471 */ SyscallDesc("unused#471", unimplementedFunc),
    /*  472 */ SyscallDesc("unused#472", unimplementedFunc),
    /*  473 */ SyscallDesc("unused#473", unimplementedFunc),
    /*  474 */ SyscallDesc("unused#474", unimplementedFunc),
    /*  475 */ SyscallDesc("unused#475", unimplementedFunc),
    /*  476 */ SyscallDesc("unused#476", unimplementedFunc),
    /*  477 */ SyscallDesc("unused#477", unimplementedFunc),
    /*  478 */ SyscallDesc("unused#478", unimplementedFunc),
    /*  479 */ SyscallDesc("unused#479", unimplementedFunc),
    /*  480 */ SyscallDesc("unused#480", unimplementedFunc),
    /*  481 */ SyscallDesc("unused#481", unimplementedFunc),
    /*  482 */ SyscallDesc("unused#482", unimplementedFunc),
    /*  483 */ SyscallDesc("unused#483", unimplementedFunc),
    /*  484 */ SyscallDesc("unused#484", unimplementedFunc),
    /*  485 */ SyscallDesc("unused#485", unimplementedFunc),
    /*  486 */ SyscallDesc("unused#486", unimplementedFunc),
    /*  487 */ SyscallDesc("unused#487", unimplementedFunc),
    /*  488 */ SyscallDesc("unused#488", unimplementedFunc),
    /*  489 */ SyscallDesc("unused#489", unimplementedFunc),
    /*  490 */ SyscallDesc("unused#490", unimplementedFunc),
    /*  491 */ SyscallDesc("unused#491", unimplementedFunc),
    /*  492 */ SyscallDesc("unused#492", unimplementedFunc),
    /*  493 */ SyscallDesc("unused#493", unimplementedFunc),
    /*  494 */ SyscallDesc("unused#494", unimplementedFunc),
    /*  495 */ SyscallDesc("unused#495", unimplementedFunc),
    /*  496 */ SyscallDesc("unused#496", unimplementedFunc),
    /*  497 */ SyscallDesc("unused#497", unimplementedFunc),
    /*  498 */ SyscallDesc("unused#498", unimplementedFunc),
    /*  499 */ SyscallDesc("unused#499", unimplementedFunc),
    /*  500 */ SyscallDesc("unused#500", unimplementedFunc),
    /*  501 */ SyscallDesc("unused#501", unimplementedFunc),
    /*  502 */ SyscallDesc("unused#502", unimplementedFunc),
    /*  503 */ SyscallDesc("unused#503", unimplementedFunc),
    /*  504 */ SyscallDesc("unused#504", unimplementedFunc),
    /*  505 */ SyscallDesc("unused#505", unimplementedFunc),
    /*  506 */ SyscallDesc("unused#506", unimplementedFunc),
    /*  507 */ SyscallDesc("unused#507", unimplementedFunc),
    /*  508 */ SyscallDesc("unused#508", unimplementedFunc),
    /*  509 */ SyscallDesc("unused#509", unimplementedFunc),
    /*  510 */ SyscallDesc("unused#510", unimplementedFunc),
    /*  511 */ SyscallDesc("unused#511", unimplementedFunc),
    /*  512 */ SyscallDesc("unused#512", unimplementedFunc),
    /*  513 */ SyscallDesc("unused#513", unimplementedFunc),
    /*  514 */ SyscallDesc("unused#514", unimplementedFunc),
    /*  515 */ SyscallDesc("unused#515", unimplementedFunc),
    /*  516 */ SyscallDesc("unused#516", unimplementedFunc),
    /*  517 */ SyscallDesc("unused#517", unimplementedFunc),
    /*  518 */ SyscallDesc("unused#518", unimplementedFunc),
    /*  519 */ SyscallDesc("unused#519", unimplementedFunc),
    /*  520 */ SyscallDesc("unused#520", unimplementedFunc),
    /*  521 */ SyscallDesc("unused#521", unimplementedFunc),
    /*  522 */ SyscallDesc("unused#522", unimplementedFunc),
    /*  523 */ SyscallDesc("unused#523", unimplementedFunc),
    /*  524 */ SyscallDesc("unused#524", unimplementedFunc),
    /*  525 */ SyscallDesc("unused#525", unimplementedFunc),
    /*  526 */ SyscallDesc("unused#526", unimplementedFunc),
    /*  527 */ SyscallDesc("unused#527", unimplementedFunc),
    /*  528 */ SyscallDesc("unused#528", unimplementedFunc),
    /*  529 */ SyscallDesc("unused#529", unimplementedFunc),
    /*  530 */ SyscallDesc("unused#530", unimplementedFunc),
    /*  531 */ SyscallDesc("unused#531", unimplementedFunc),
    /*  532 */ SyscallDesc("unused#532", unimplementedFunc),
    /*  533 */ SyscallDesc("unused#533", unimplementedFunc),
    /*  534 */ SyscallDesc("unused#534", unimplementedFunc),
    /*  535 */ SyscallDesc("unused#535", unimplementedFunc),
    /*  536 */ SyscallDesc("unused#536", unimplementedFunc),
    /*  537 */ SyscallDesc("unused#537", unimplementedFunc),
    /*  538 */ SyscallDesc("unused#538", unimplementedFunc),
    /*  539 */ SyscallDesc("unused#539", unimplementedFunc),
    /*  540 */ SyscallDesc("unused#540", unimplementedFunc),
    /*  541 */ SyscallDesc("unused#541", unimplementedFunc),
    /*  542 */ SyscallDesc("unused#542", unimplementedFunc),
    /*  543 */ SyscallDesc("unused#543", unimplementedFunc),
    /*  544 */ SyscallDesc("unused#544", unimplementedFunc),
    /*  545 */ SyscallDesc("unused#545", unimplementedFunc),
    /*  546 */ SyscallDesc("unused#546", unimplementedFunc),
    /*  547 */ SyscallDesc("unused#547", unimplementedFunc),
    /*  548 */ SyscallDesc("unused#548", unimplementedFunc),
    /*  549 */ SyscallDesc("unused#549", unimplementedFunc),
    /*  550 */ SyscallDesc("unused#550", unimplementedFunc),
    /*  551 */ SyscallDesc("unused#551", unimplementedFunc),
    /*  552 */ SyscallDesc("unused#552", unimplementedFunc),
    /*  553 */ SyscallDesc("unused#553", unimplementedFunc),
    /*  554 */ SyscallDesc("unused#554", unimplementedFunc),
    /*  555 */ SyscallDesc("unused#555", unimplementedFunc),
    /*  556 */ SyscallDesc("unused#556", unimplementedFunc),
    /*  557 */ SyscallDesc("unused#557", unimplementedFunc),
    /*  558 */ SyscallDesc("unused#558", unimplementedFunc),
    /*  559 */ SyscallDesc("unused#559", unimplementedFunc),
    /*  560 */ SyscallDesc("unused#560", unimplementedFunc),
    /*  561 */ SyscallDesc("unused#561", unimplementedFunc),
    /*  562 */ SyscallDesc("unused#562", unimplementedFunc),
    /*  563 */ SyscallDesc("unused#563", unimplementedFunc),
    /*  564 */ SyscallDesc("unused#564", unimplementedFunc),
    /*  565 */ SyscallDesc("unused#565", unimplementedFunc),
    /*  566 */ SyscallDesc("unused#566", unimplementedFunc),
    /*  567 */ SyscallDesc("unused#567", unimplementedFunc),
    /*  568 */ SyscallDesc("unused#568", unimplementedFunc),
    /*  569 */ SyscallDesc("unused#569", unimplementedFunc),
    /*  570 */ SyscallDesc("unused#570", unimplementedFunc),
    /*  571 */ SyscallDesc("unused#571", unimplementedFunc),
    /*  572 */ SyscallDesc("unused#572", unimplementedFunc),
    /*  573 */ SyscallDesc("unused#573", unimplementedFunc),
    /*  574 */ SyscallDesc("unused#574", unimplementedFunc),
    /*  575 */ SyscallDesc("unused#575", unimplementedFunc),
    /*  576 */ SyscallDesc("unused#576", unimplementedFunc),
    /*  577 */ SyscallDesc("unused#577", unimplementedFunc),
    /*  578 */ SyscallDesc("unused#578", unimplementedFunc),
    /*  579 */ SyscallDesc("unused#579", unimplementedFunc),
    /*  580 */ SyscallDesc("unused#580", unimplementedFunc),
    /*  581 */ SyscallDesc("unused#581", unimplementedFunc),
    /*  582 */ SyscallDesc("unused#582", unimplementedFunc),
    /*  583 */ SyscallDesc("unused#583", unimplementedFunc),
    /*  584 */ SyscallDesc("unused#584", unimplementedFunc),
    /*  585 */ SyscallDesc("unused#585", unimplementedFunc),
    /*  586 */ SyscallDesc("unused#586", unimplementedFunc),
    /*  587 */ SyscallDesc("unused#587", unimplementedFunc),
    /*  588 */ SyscallDesc("unused#588", unimplementedFunc),
    /*  589 */ SyscallDesc("unused#589", unimplementedFunc),
    /*  590 */ SyscallDesc("unused#590", unimplementedFunc),
    /*  591 */ SyscallDesc("unused#591", unimplementedFunc),
    /*  592 */ SyscallDesc("unused#592", unimplementedFunc),
    /*  593 */ SyscallDesc("unused#593", unimplementedFunc),
    /*  594 */ SyscallDesc("unused#594", unimplementedFunc),
    /*  595 */ SyscallDesc("unused#595", unimplementedFunc),
    /*  596 */ SyscallDesc("unused#596", unimplementedFunc),
    /*  597 */ SyscallDesc("unused#597", unimplementedFunc),
    /*  598 */ SyscallDesc("unused#598", unimplementedFunc),
    /*  599 */ SyscallDesc("unused#599", unimplementedFunc),
    /*  600 */ SyscallDesc("unused#600", unimplementedFunc),
    /*  601 */ SyscallDesc("unused#601", unimplementedFunc),
    /*  602 */ SyscallDesc("unused#602", unimplementedFunc),
    /*  603 */ SyscallDesc("unused#603", unimplementedFunc),
    /*  604 */ SyscallDesc("unused#604", unimplementedFunc),
    /*  605 */ SyscallDesc("unused#605", unimplementedFunc),
    /*  606 */ SyscallDesc("unused#606", unimplementedFunc),
    /*  607 */ SyscallDesc("unused#607", unimplementedFunc),
    /*  608 */ SyscallDesc("unused#608", unimplementedFunc),
    /*  609 */ SyscallDesc("unused#609", unimplementedFunc),
    /*  610 */ SyscallDesc("unused#610", unimplementedFunc),
    /*  611 */ SyscallDesc("unused#611", unimplementedFunc),
    /*  612 */ SyscallDesc("unused#612", unimplementedFunc),
    /*  613 */ SyscallDesc("unused#613", unimplementedFunc),
    /*  614 */ SyscallDesc("unused#614", unimplementedFunc),
    /*  615 */ SyscallDesc("unused#615", unimplementedFunc),
    /*  616 */ SyscallDesc("unused#616", unimplementedFunc),
    /*  617 */ SyscallDesc("unused#617", unimplementedFunc),
    /*  618 */ SyscallDesc("unused#618", unimplementedFunc),
    /*  619 */ SyscallDesc("unused#619", unimplementedFunc),
    /*  620 */ SyscallDesc("unused#620", unimplementedFunc),
    /*  621 */ SyscallDesc("unused#621", unimplementedFunc),
    /*  622 */ SyscallDesc("unused#622", unimplementedFunc),
    /*  623 */ SyscallDesc("unused#623", unimplementedFunc),
    /*  624 */ SyscallDesc("unused#624", unimplementedFunc),
    /*  625 */ SyscallDesc("unused#625", unimplementedFunc),
    /*  626 */ SyscallDesc("unused#626", unimplementedFunc),
    /*  627 */ SyscallDesc("unused#627", unimplementedFunc),
    /*  628 */ SyscallDesc("unused#628", unimplementedFunc),
    /*  629 */ SyscallDesc("unused#629", unimplementedFunc),
    /*  630 */ SyscallDesc("unused#630", unimplementedFunc),
    /*  631 */ SyscallDesc("unused#631", unimplementedFunc),
    /*  632 */ SyscallDesc("unused#632", unimplementedFunc),
    /*  633 */ SyscallDesc("unused#633", unimplementedFunc),
    /*  634 */ SyscallDesc("unused#634", unimplementedFunc),
    /*  635 */ SyscallDesc("unused#635", unimplementedFunc),
    /*  636 */ SyscallDesc("unused#636", unimplementedFunc),
    /*  637 */ SyscallDesc("unused#637", unimplementedFunc),
    /*  638 */ SyscallDesc("unused#638", unimplementedFunc),
    /*  639 */ SyscallDesc("unused#639", unimplementedFunc),
    /*  640 */ SyscallDesc("unused#640", unimplementedFunc),
    /*  641 */ SyscallDesc("unused#641", unimplementedFunc),
    /*  642 */ SyscallDesc("unused#642", unimplementedFunc),
    /*  643 */ SyscallDesc("unused#643", unimplementedFunc),
    /*  644 */ SyscallDesc("unused#644", unimplementedFunc),
    /*  645 */ SyscallDesc("unused#645", unimplementedFunc),
    /*  646 */ SyscallDesc("unused#646", unimplementedFunc),
    /*  647 */ SyscallDesc("unused#647", unimplementedFunc),
    /*  648 */ SyscallDesc("unused#648", unimplementedFunc),
    /*  649 */ SyscallDesc("unused#649", unimplementedFunc),
    /*  650 */ SyscallDesc("unused#650", unimplementedFunc),
    /*  651 */ SyscallDesc("unused#651", unimplementedFunc),
    /*  652 */ SyscallDesc("unused#652", unimplementedFunc),
    /*  653 */ SyscallDesc("unused#653", unimplementedFunc),
    /*  654 */ SyscallDesc("unused#654", unimplementedFunc),
    /*  655 */ SyscallDesc("unused#655", unimplementedFunc),
    /*  656 */ SyscallDesc("unused#656", unimplementedFunc),
    /*  657 */ SyscallDesc("unused#657", unimplementedFunc),
    /*  658 */ SyscallDesc("unused#658", unimplementedFunc),
    /*  659 */ SyscallDesc("unused#659", unimplementedFunc),
    /*  660 */ SyscallDesc("unused#660", unimplementedFunc),
    /*  661 */ SyscallDesc("unused#661", unimplementedFunc),
    /*  662 */ SyscallDesc("unused#662", unimplementedFunc),
    /*  663 */ SyscallDesc("unused#663", unimplementedFunc),
    /*  664 */ SyscallDesc("unused#664", unimplementedFunc),
    /*  665 */ SyscallDesc("unused#665", unimplementedFunc),
    /*  666 */ SyscallDesc("unused#666", unimplementedFunc),
    /*  667 */ SyscallDesc("unused#667", unimplementedFunc),
    /*  668 */ SyscallDesc("unused#668", unimplementedFunc),
    /*  669 */ SyscallDesc("unused#669", unimplementedFunc),
    /*  670 */ SyscallDesc("unused#670", unimplementedFunc),
    /*  671 */ SyscallDesc("unused#671", unimplementedFunc),
    /*  672 */ SyscallDesc("unused#672", unimplementedFunc),
    /*  673 */ SyscallDesc("unused#673", unimplementedFunc),
    /*  674 */ SyscallDesc("unused#674", unimplementedFunc),
    /*  675 */ SyscallDesc("unused#675", unimplementedFunc),
    /*  676 */ SyscallDesc("unused#676", unimplementedFunc),
    /*  677 */ SyscallDesc("unused#677", unimplementedFunc),
    /*  678 */ SyscallDesc("unused#678", unimplementedFunc),
    /*  679 */ SyscallDesc("unused#679", unimplementedFunc),
    /*  680 */ SyscallDesc("unused#680", unimplementedFunc),
    /*  681 */ SyscallDesc("unused#681", unimplementedFunc),
    /*  682 */ SyscallDesc("unused#682", unimplementedFunc),
    /*  683 */ SyscallDesc("unused#683", unimplementedFunc),
    /*  684 */ SyscallDesc("unused#684", unimplementedFunc),
    /*  685 */ SyscallDesc("unused#685", unimplementedFunc),
    /*  686 */ SyscallDesc("unused#686", unimplementedFunc),
    /*  687 */ SyscallDesc("unused#687", unimplementedFunc),
    /*  688 */ SyscallDesc("unused#688", unimplementedFunc),
    /*  689 */ SyscallDesc("unused#689", unimplementedFunc),
    /*  690 */ SyscallDesc("unused#690", unimplementedFunc),
    /*  691 */ SyscallDesc("unused#691", unimplementedFunc),
    /*  692 */ SyscallDesc("unused#692", unimplementedFunc),
    /*  693 */ SyscallDesc("unused#693", unimplementedFunc),
    /*  694 */ SyscallDesc("unused#694", unimplementedFunc),
    /*  695 */ SyscallDesc("unused#695", unimplementedFunc),
    /*  696 */ SyscallDesc("unused#696", unimplementedFunc),
    /*  697 */ SyscallDesc("unused#697", unimplementedFunc),
    /*  698 */ SyscallDesc("unused#698", unimplementedFunc),
    /*  699 */ SyscallDesc("unused#699", unimplementedFunc),
    /*  700 */ SyscallDesc("unused#700", unimplementedFunc),
    /*  701 */ SyscallDesc("unused#701", unimplementedFunc),
    /*  702 */ SyscallDesc("unused#702", unimplementedFunc),
    /*  703 */ SyscallDesc("unused#703", unimplementedFunc),
    /*  704 */ SyscallDesc("unused#704", unimplementedFunc),
    /*  705 */ SyscallDesc("unused#705", unimplementedFunc),
    /*  706 */ SyscallDesc("unused#706", unimplementedFunc),
    /*  707 */ SyscallDesc("unused#707", unimplementedFunc),
    /*  708 */ SyscallDesc("unused#708", unimplementedFunc),
    /*  709 */ SyscallDesc("unused#709", unimplementedFunc),
    /*  710 */ SyscallDesc("unused#710", unimplementedFunc),
    /*  711 */ SyscallDesc("unused#711", unimplementedFunc),
    /*  712 */ SyscallDesc("unused#712", unimplementedFunc),
    /*  713 */ SyscallDesc("unused#713", unimplementedFunc),
    /*  714 */ SyscallDesc("unused#714", unimplementedFunc),
    /*  715 */ SyscallDesc("unused#715", unimplementedFunc),
    /*  716 */ SyscallDesc("unused#716", unimplementedFunc),
    /*  717 */ SyscallDesc("unused#717", unimplementedFunc),
    /*  718 */ SyscallDesc("unused#718", unimplementedFunc),
    /*  719 */ SyscallDesc("unused#719", unimplementedFunc),
    /*  720 */ SyscallDesc("unused#720", unimplementedFunc),
    /*  721 */ SyscallDesc("unused#721", unimplementedFunc),
    /*  722 */ SyscallDesc("unused#722", unimplementedFunc),
    /*  723 */ SyscallDesc("unused#723", unimplementedFunc),
    /*  724 */ SyscallDesc("unused#724", unimplementedFunc),
    /*  725 */ SyscallDesc("unused#725", unimplementedFunc),
    /*  726 */ SyscallDesc("unused#726", unimplementedFunc),
    /*  727 */ SyscallDesc("unused#727", unimplementedFunc),
    /*  728 */ SyscallDesc("unused#728", unimplementedFunc),
    /*  729 */ SyscallDesc("unused#729", unimplementedFunc),
    /*  730 */ SyscallDesc("unused#730", unimplementedFunc),
    /*  731 */ SyscallDesc("unused#731", unimplementedFunc),
    /*  732 */ SyscallDesc("unused#732", unimplementedFunc),
    /*  733 */ SyscallDesc("unused#733", unimplementedFunc),
    /*  734 */ SyscallDesc("unused#734", unimplementedFunc),
    /*  735 */ SyscallDesc("unused#735", unimplementedFunc),
    /*  736 */ SyscallDesc("unused#736", unimplementedFunc),
    /*  737 */ SyscallDesc("unused#737", unimplementedFunc),
    /*  738 */ SyscallDesc("unused#738", unimplementedFunc),
    /*  739 */ SyscallDesc("unused#739", unimplementedFunc),
    /*  740 */ SyscallDesc("unused#740", unimplementedFunc),
    /*  741 */ SyscallDesc("unused#741", unimplementedFunc),
    /*  742 */ SyscallDesc("unused#742", unimplementedFunc),
    /*  743 */ SyscallDesc("unused#743", unimplementedFunc),
    /*  744 */ SyscallDesc("unused#744", unimplementedFunc),
    /*  745 */ SyscallDesc("unused#745", unimplementedFunc),
    /*  746 */ SyscallDesc("unused#746", unimplementedFunc),
    /*  747 */ SyscallDesc("unused#747", unimplementedFunc),
    /*  748 */ SyscallDesc("unused#748", unimplementedFunc),
    /*  749 */ SyscallDesc("unused#749", unimplementedFunc),
    /*  750 */ SyscallDesc("unused#750", unimplementedFunc),
    /*  751 */ SyscallDesc("unused#751", unimplementedFunc),
    /*  752 */ SyscallDesc("unused#752", unimplementedFunc),
    /*  753 */ SyscallDesc("unused#753", unimplementedFunc),
    /*  754 */ SyscallDesc("unused#754", unimplementedFunc),
    /*  755 */ SyscallDesc("unused#755", unimplementedFunc),
    /*  756 */ SyscallDesc("unused#756", unimplementedFunc),
    /*  757 */ SyscallDesc("unused#757", unimplementedFunc),
    /*  758 */ SyscallDesc("unused#758", unimplementedFunc),
    /*  759 */ SyscallDesc("unused#759", unimplementedFunc),
    /*  760 */ SyscallDesc("unused#760", unimplementedFunc),
    /*  761 */ SyscallDesc("unused#761", unimplementedFunc),
    /*  762 */ SyscallDesc("unused#762", unimplementedFunc),
    /*  763 */ SyscallDesc("unused#763", unimplementedFunc),
    /*  764 */ SyscallDesc("unused#764", unimplementedFunc),
    /*  765 */ SyscallDesc("unused#765", unimplementedFunc),
    /*  766 */ SyscallDesc("unused#766", unimplementedFunc),
    /*  767 */ SyscallDesc("unused#767", unimplementedFunc),
    /*  768 */ SyscallDesc("unused#768", unimplementedFunc),
    /*  769 */ SyscallDesc("unused#769", unimplementedFunc),
    /*  770 */ SyscallDesc("unused#770", unimplementedFunc),
    /*  771 */ SyscallDesc("unused#771", unimplementedFunc),
    /*  772 */ SyscallDesc("unused#772", unimplementedFunc),
    /*  773 */ SyscallDesc("unused#773", unimplementedFunc),
    /*  774 */ SyscallDesc("unused#774", unimplementedFunc),
    /*  775 */ SyscallDesc("unused#775", unimplementedFunc),
    /*  776 */ SyscallDesc("unused#776", unimplementedFunc),
    /*  777 */ SyscallDesc("unused#777", unimplementedFunc),
    /*  778 */ SyscallDesc("unused#778", unimplementedFunc),
    /*  779 */ SyscallDesc("unused#779", unimplementedFunc),
    /*  780 */ SyscallDesc("unused#780", unimplementedFunc),
    /*  781 */ SyscallDesc("unused#781", unimplementedFunc),
    /*  782 */ SyscallDesc("unused#782", unimplementedFunc),
    /*  783 */ SyscallDesc("unused#783", unimplementedFunc),
    /*  784 */ SyscallDesc("unused#784", unimplementedFunc),
    /*  785 */ SyscallDesc("unused#785", unimplementedFunc),
    /*  786 */ SyscallDesc("unused#786", unimplementedFunc),
    /*  787 */ SyscallDesc("unused#787", unimplementedFunc),
    /*  788 */ SyscallDesc("unused#788", unimplementedFunc),
    /*  789 */ SyscallDesc("unused#789", unimplementedFunc),
    /*  790 */ SyscallDesc("unused#790", unimplementedFunc),
    /*  791 */ SyscallDesc("unused#791", unimplementedFunc),
    /*  792 */ SyscallDesc("unused#792", unimplementedFunc),
    /*  793 */ SyscallDesc("unused#793", unimplementedFunc),
    /*  794 */ SyscallDesc("unused#794", unimplementedFunc),
    /*  795 */ SyscallDesc("unused#795", unimplementedFunc),
    /*  796 */ SyscallDesc("unused#796", unimplementedFunc),
    /*  797 */ SyscallDesc("unused#797", unimplementedFunc),
    /*  798 */ SyscallDesc("unused#798", unimplementedFunc),
    /*  799 */ SyscallDesc("unused#799", unimplementedFunc),
    /*  800 */ SyscallDesc("unused#800", unimplementedFunc),
    /*  801 */ SyscallDesc("unused#801", unimplementedFunc),
    /*  802 */ SyscallDesc("unused#802", unimplementedFunc),
    /*  803 */ SyscallDesc("unused#803", unimplementedFunc),
    /*  804 */ SyscallDesc("unused#804", unimplementedFunc),
    /*  805 */ SyscallDesc("unused#805", unimplementedFunc),
    /*  806 */ SyscallDesc("unused#806", unimplementedFunc),
    /*  807 */ SyscallDesc("unused#807", unimplementedFunc),
    /*  808 */ SyscallDesc("unused#808", unimplementedFunc),
    /*  809 */ SyscallDesc("unused#809", unimplementedFunc),
    /*  810 */ SyscallDesc("unused#810", unimplementedFunc),
    /*  811 */ SyscallDesc("unused#811", unimplementedFunc),
    /*  812 */ SyscallDesc("unused#812", unimplementedFunc),
    /*  813 */ SyscallDesc("unused#813", unimplementedFunc),
    /*  814 */ SyscallDesc("unused#814", unimplementedFunc),
    /*  815 */ SyscallDesc("unused#815", unimplementedFunc),
    /*  816 */ SyscallDesc("unused#816", unimplementedFunc),
    /*  817 */ SyscallDesc("unused#817", unimplementedFunc),
    /*  818 */ SyscallDesc("unused#818", unimplementedFunc),
    /*  819 */ SyscallDesc("unused#819", unimplementedFunc),
    /*  820 */ SyscallDesc("unused#820", unimplementedFunc),
    /*  821 */ SyscallDesc("unused#821", unimplementedFunc),
    /*  822 */ SyscallDesc("unused#822", unimplementedFunc),
    /*  823 */ SyscallDesc("unused#823", unimplementedFunc),
    /*  824 */ SyscallDesc("unused#824", unimplementedFunc),
    /*  825 */ SyscallDesc("unused#825", unimplementedFunc),
    /*  826 */ SyscallDesc("unused#826", unimplementedFunc),
    /*  827 */ SyscallDesc("unused#827", unimplementedFunc),
    /*  828 */ SyscallDesc("unused#828", unimplementedFunc),
    /*  829 */ SyscallDesc("unused#829", unimplementedFunc),
    /*  830 */ SyscallDesc("unused#830", unimplementedFunc),
    /*  831 */ SyscallDesc("unused#831", unimplementedFunc),
    /*  832 */ SyscallDesc("unused#832", unimplementedFunc),
    /*  833 */ SyscallDesc("unused#833", unimplementedFunc),
    /*  834 */ SyscallDesc("unused#834", unimplementedFunc),
    /*  835 */ SyscallDesc("unused#835", unimplementedFunc),
    /*  836 */ SyscallDesc("unused#836", unimplementedFunc),
    /*  837 */ SyscallDesc("unused#837", unimplementedFunc),
    /*  838 */ SyscallDesc("unused#838", unimplementedFunc),
    /*  839 */ SyscallDesc("unused#839", unimplementedFunc),
    /*  840 */ SyscallDesc("unused#840", unimplementedFunc),
    /*  841 */ SyscallDesc("unused#841", unimplementedFunc),
    /*  842 */ SyscallDesc("unused#842", unimplementedFunc),
    /*  843 */ SyscallDesc("unused#843", unimplementedFunc),
    /*  844 */ SyscallDesc("unused#844", unimplementedFunc),
    /*  845 */ SyscallDesc("unused#845", unimplementedFunc),
    /*  846 */ SyscallDesc("unused#846", unimplementedFunc),
    /*  847 */ SyscallDesc("unused#847", unimplementedFunc),
    /*  848 */ SyscallDesc("unused#848", unimplementedFunc),
    /*  849 */ SyscallDesc("unused#849", unimplementedFunc),
    /*  850 */ SyscallDesc("unused#850", unimplementedFunc),
    /*  851 */ SyscallDesc("unused#851", unimplementedFunc),
    /*  852 */ SyscallDesc("unused#852", unimplementedFunc),
    /*  853 */ SyscallDesc("unused#853", unimplementedFunc),
    /*  854 */ SyscallDesc("unused#854", unimplementedFunc),
    /*  855 */ SyscallDesc("unused#855", unimplementedFunc),
    /*  856 */ SyscallDesc("unused#856", unimplementedFunc),
    /*  857 */ SyscallDesc("unused#857", unimplementedFunc),
    /*  858 */ SyscallDesc("unused#858", unimplementedFunc),
    /*  859 */ SyscallDesc("unused#859", unimplementedFunc),
    /*  860 */ SyscallDesc("unused#860", unimplementedFunc),
    /*  861 */ SyscallDesc("unused#861", unimplementedFunc),
    /*  862 */ SyscallDesc("unused#862", unimplementedFunc),
    /*  863 */ SyscallDesc("unused#863", unimplementedFunc),
    /*  864 */ SyscallDesc("unused#864", unimplementedFunc),
    /*  865 */ SyscallDesc("unused#865", unimplementedFunc),
    /*  866 */ SyscallDesc("unused#866", unimplementedFunc),
    /*  867 */ SyscallDesc("unused#867", unimplementedFunc),
    /*  868 */ SyscallDesc("unused#868", unimplementedFunc),
    /*  869 */ SyscallDesc("unused#869", unimplementedFunc),
    /*  870 */ SyscallDesc("unused#870", unimplementedFunc),
    /*  871 */ SyscallDesc("unused#871", unimplementedFunc),
    /*  872 */ SyscallDesc("unused#872", unimplementedFunc),
    /*  873 */ SyscallDesc("unused#873", unimplementedFunc),
    /*  874 */ SyscallDesc("unused#874", unimplementedFunc),
    /*  875 */ SyscallDesc("unused#875", unimplementedFunc),
    /*  876 */ SyscallDesc("unused#876", unimplementedFunc),
    /*  877 */ SyscallDesc("unused#877", unimplementedFunc),
    /*  878 */ SyscallDesc("unused#878", unimplementedFunc),
    /*  879 */ SyscallDesc("unused#879", unimplementedFunc),
    /*  880 */ SyscallDesc("unused#880", unimplementedFunc),
    /*  881 */ SyscallDesc("unused#881", unimplementedFunc),
    /*  882 */ SyscallDesc("unused#882", unimplementedFunc),
    /*  883 */ SyscallDesc("unused#883", unimplementedFunc),
    /*  884 */ SyscallDesc("unused#884", unimplementedFunc),
    /*  885 */ SyscallDesc("unused#885", unimplementedFunc),
    /*  886 */ SyscallDesc("unused#886", unimplementedFunc),
    /*  887 */ SyscallDesc("unused#887", unimplementedFunc),
    /*  888 */ SyscallDesc("unused#888", unimplementedFunc),
    /*  889 */ SyscallDesc("unused#889", unimplementedFunc),
    /*  890 */ SyscallDesc("unused#890", unimplementedFunc),
    /*  891 */ SyscallDesc("unused#891", unimplementedFunc),
    /*  892 */ SyscallDesc("unused#892", unimplementedFunc),
    /*  893 */ SyscallDesc("unused#893", unimplementedFunc),
    /*  894 */ SyscallDesc("unused#894", unimplementedFunc),
    /*  895 */ SyscallDesc("unused#895", unimplementedFunc),
    /*  896 */ SyscallDesc("unused#896", unimplementedFunc),
    /*  897 */ SyscallDesc("unused#897", unimplementedFunc),
    /*  898 */ SyscallDesc("unused#898", unimplementedFunc),
    /*  899 */ SyscallDesc("unused#899", unimplementedFunc),
    /*  900 */ SyscallDesc("unused#900", unimplementedFunc),
    /*  901 */ SyscallDesc("unused#901", unimplementedFunc),
    /*  902 */ SyscallDesc("unused#902", unimplementedFunc),
    /*  903 */ SyscallDesc("unused#903", unimplementedFunc),
    /*  904 */ SyscallDesc("unused#904", unimplementedFunc),
    /*  905 */ SyscallDesc("unused#905", unimplementedFunc),
    /*  906 */ SyscallDesc("unused#906", unimplementedFunc),
    /*  907 */ SyscallDesc("unused#907", unimplementedFunc),
    /*  908 */ SyscallDesc("unused#908", unimplementedFunc),
    /*  909 */ SyscallDesc("unused#909", unimplementedFunc),
    /*  910 */ SyscallDesc("unused#910", unimplementedFunc),
    /*  911 */ SyscallDesc("unused#911", unimplementedFunc),
    /*  912 */ SyscallDesc("unused#912", unimplementedFunc),
    /*  913 */ SyscallDesc("unused#913", unimplementedFunc),
    /*  914 */ SyscallDesc("unused#914", unimplementedFunc),
    /*  915 */ SyscallDesc("unused#915", unimplementedFunc),
    /*  916 */ SyscallDesc("unused#916", unimplementedFunc),
    /*  917 */ SyscallDesc("unused#917", unimplementedFunc),
    /*  918 */ SyscallDesc("unused#918", unimplementedFunc),
    /*  919 */ SyscallDesc("unused#919", unimplementedFunc),
    /*  920 */ SyscallDesc("unused#920", unimplementedFunc),
    /*  921 */ SyscallDesc("unused#921", unimplementedFunc),
    /*  922 */ SyscallDesc("unused#922", unimplementedFunc),
    /*  923 */ SyscallDesc("unused#923", unimplementedFunc),
    /*  924 */ SyscallDesc("unused#924", unimplementedFunc),
    /*  925 */ SyscallDesc("unused#925", unimplementedFunc),
    /*  926 */ SyscallDesc("unused#926", unimplementedFunc),
    /*  927 */ SyscallDesc("unused#927", unimplementedFunc),
    /*  928 */ SyscallDesc("unused#928", unimplementedFunc),
    /*  929 */ SyscallDesc("unused#929", unimplementedFunc),
    /*  930 */ SyscallDesc("unused#930", unimplementedFunc),
    /*  931 */ SyscallDesc("unused#931", unimplementedFunc),
    /*  932 */ SyscallDesc("unused#932", unimplementedFunc),
    /*  933 */ SyscallDesc("unused#933", unimplementedFunc),
    /*  934 */ SyscallDesc("unused#934", unimplementedFunc),
    /*  935 */ SyscallDesc("unused#935", unimplementedFunc),
    /*  936 */ SyscallDesc("unused#936", unimplementedFunc),
    /*  937 */ SyscallDesc("unused#937", unimplementedFunc),
    /*  938 */ SyscallDesc("unused#938", unimplementedFunc),
    /*  939 */ SyscallDesc("unused#939", unimplementedFunc),
    /*  940 */ SyscallDesc("unused#940", unimplementedFunc),
    /*  941 */ SyscallDesc("unused#941", unimplementedFunc),
    /*  942 */ SyscallDesc("unused#942", unimplementedFunc),
    /*  943 */ SyscallDesc("unused#943", unimplementedFunc),
    /*  944 */ SyscallDesc("unused#944", unimplementedFunc),
    /*  945 */ SyscallDesc("unused#945", unimplementedFunc),
    /*  946 */ SyscallDesc("unused#946", unimplementedFunc),
    /*  947 */ SyscallDesc("unused#947", unimplementedFunc),
    /*  948 */ SyscallDesc("unused#948", unimplementedFunc),
    /*  949 */ SyscallDesc("unused#949", unimplementedFunc),
    /*  950 */ SyscallDesc("unused#950", unimplementedFunc),
    /*  951 */ SyscallDesc("unused#951", unimplementedFunc),
    /*  952 */ SyscallDesc("unused#952", unimplementedFunc),
    /*  953 */ SyscallDesc("unused#953", unimplementedFunc),
    /*  954 */ SyscallDesc("unused#954", unimplementedFunc),
    /*  955 */ SyscallDesc("unused#955", unimplementedFunc),
    /*  956 */ SyscallDesc("unused#956", unimplementedFunc),
    /*  957 */ SyscallDesc("unused#957", unimplementedFunc),
    /*  958 */ SyscallDesc("unused#958", unimplementedFunc),
    /*  959 */ SyscallDesc("unused#959", unimplementedFunc),
    /*  960 */ SyscallDesc("unused#960", unimplementedFunc),
    /*  961 */ SyscallDesc("unused#961", unimplementedFunc),
    /*  962 */ SyscallDesc("unused#962", unimplementedFunc),
    /*  963 */ SyscallDesc("unused#963", unimplementedFunc),
    /*  964 */ SyscallDesc("unused#964", unimplementedFunc),
    /*  965 */ SyscallDesc("unused#965", unimplementedFunc),
    /*  966 */ SyscallDesc("unused#966", unimplementedFunc),
    /*  967 */ SyscallDesc("unused#967", unimplementedFunc),
    /*  968 */ SyscallDesc("unused#968", unimplementedFunc),
    /*  969 */ SyscallDesc("unused#969", unimplementedFunc),
    /*  970 */ SyscallDesc("unused#970", unimplementedFunc),
    /*  971 */ SyscallDesc("unused#971", unimplementedFunc),
    /*  972 */ SyscallDesc("unused#972", unimplementedFunc),
    /*  973 */ SyscallDesc("unused#973", unimplementedFunc),
    /*  974 */ SyscallDesc("unused#974", unimplementedFunc),
    /*  975 */ SyscallDesc("unused#975", unimplementedFunc),
    /*  976 */ SyscallDesc("unused#976", unimplementedFunc),
    /*  977 */ SyscallDesc("unused#977", unimplementedFunc),
    /*  978 */ SyscallDesc("unused#978", unimplementedFunc),
    /*  979 */ SyscallDesc("unused#979", unimplementedFunc),
    /*  980 */ SyscallDesc("unused#980", unimplementedFunc),
    /*  981 */ SyscallDesc("unused#981", unimplementedFunc),
    /*  982 */ SyscallDesc("unused#982", unimplementedFunc),
    /*  983 */ SyscallDesc("unused#983", unimplementedFunc),
    /*  984 */ SyscallDesc("unused#984", unimplementedFunc),
    /*  985 */ SyscallDesc("unused#985", unimplementedFunc),
    /*  986 */ SyscallDesc("unused#986", unimplementedFunc),
    /*  987 */ SyscallDesc("unused#987", unimplementedFunc),
    /*  988 */ SyscallDesc("unused#988", unimplementedFunc),
    /*  989 */ SyscallDesc("unused#989", unimplementedFunc),
    /*  990 */ SyscallDesc("unused#990", unimplementedFunc),
    /*  991 */ SyscallDesc("unused#991", unimplementedFunc),
    /*  992 */ SyscallDesc("unused#992", unimplementedFunc),
    /*  993 */ SyscallDesc("unused#993", unimplementedFunc),
    /*  994 */ SyscallDesc("unused#994", unimplementedFunc),
    /*  995 */ SyscallDesc("unused#995", unimplementedFunc),
    /*  996 */ SyscallDesc("unused#996", unimplementedFunc),
    /*  997 */ SyscallDesc("unused#997", unimplementedFunc),
    /*  998 */ SyscallDesc("unused#998", unimplementedFunc),
    /*  999 */ SyscallDesc("unused#999", unimplementedFunc),
    /* 1000 */ SyscallDesc("unused#1000", unimplementedFunc),
    /* 1001 */ SyscallDesc("unused#1001", unimplementedFunc),
    /* 1002 */ SyscallDesc("unused#1002", unimplementedFunc),
    /* 1003 */ SyscallDesc("unused#1003", unimplementedFunc),
    /* 1004 */ SyscallDesc("unused#1004", unimplementedFunc),
    /* 1005 */ SyscallDesc("unused#1005", unimplementedFunc),
    /* 1006 */ SyscallDesc("unused#1006", unimplementedFunc),
    /* 1007 */ SyscallDesc("unused#1007", unimplementedFunc),
    /* 1008 */ SyscallDesc("unused#1008", unimplementedFunc),
    /* 1009 */ SyscallDesc("unused#1009", unimplementedFunc),
    /* 1010 */ SyscallDesc("unused#1010", unimplementedFunc),
    /* 1011 */ SyscallDesc("unused#1011", unimplementedFunc),
    /* 1012 */ SyscallDesc("unused#1012", unimplementedFunc),
    /* 1013 */ SyscallDesc("unused#1013", unimplementedFunc),
    /* 1014 */ SyscallDesc("unused#1014", unimplementedFunc),
    /* 1015 */ SyscallDesc("unused#1015", unimplementedFunc),
    /* 1016 */ SyscallDesc("unused#1016", unimplementedFunc),
    /* 1017 */ SyscallDesc("unused#1017", unimplementedFunc),
    /* 1018 */ SyscallDesc("unused#1018", unimplementedFunc),
    /* 1019 */ SyscallDesc("unused#1019", unimplementedFunc),
    /* 1020 */ SyscallDesc("unused#1020", unimplementedFunc),
    /* 1021 */ SyscallDesc("unused#1021", unimplementedFunc),
    /* 1022 */ SyscallDesc("unused#1022", unimplementedFunc),
    /* 1023 */ SyscallDesc("unused#1023", unimplementedFunc),
    /* 1024 */ SyscallDesc("open", openFunc<ArmLinux64>),
    /* 1025 */ SyscallDesc("link", unimplementedFunc),
    /* 1026 */ SyscallDesc("unlink", unlinkFunc),
    /* 1027 */ SyscallDesc("mknod", unimplementedFunc),
    /* 1028 */ SyscallDesc("chmod", chmodFunc<ArmLinux64>),
    /* 1029 */ SyscallDesc("chown", unimplementedFunc),
    /* 1030 */ SyscallDesc("mkdir", mkdirFunc),
    /* 1031 */ SyscallDesc("rmdir", unimplementedFunc),
    /* 1032 */ SyscallDesc("lchown", unimplementedFunc),
    /* 1033 */ SyscallDesc("access", accessFunc),
    /* 1034 */ SyscallDesc("rename", renameFunc),
    /* 1035 */ SyscallDesc("readlink", readlinkFunc),
    /* 1036 */ SyscallDesc("symlink", unimplementedFunc),
    /* 1037 */ SyscallDesc("utimes", unimplementedFunc),
    /* 1038 */ SyscallDesc("stat64", stat64Func<ArmLinux64>),
    /* 1039 */ SyscallDesc("lstat64", lstat64Func<ArmLinux64>),
    /* 1040 */ SyscallDesc("pipe", pipePseudoFunc),
    /* 1041 */ SyscallDesc("dup2", unimplementedFunc),
    /* 1042 */ SyscallDesc("epoll_create", unimplementedFunc),
    /* 1043 */ SyscallDesc("inotify_init", unimplementedFunc),
    /* 1044 */ SyscallDesc("eventfd", unimplementedFunc),
    /* 1045 */ SyscallDesc("signalfd", unimplementedFunc),
    /* 1046 */ SyscallDesc("sendfile", unimplementedFunc),
    /* 1047 */ SyscallDesc("ftruncate", ftruncateFunc),
    /* 1048 */ SyscallDesc("truncate", truncateFunc),
    /* 1049 */ SyscallDesc("stat", statFunc<ArmLinux64>),
    /* 1050 */ SyscallDesc("lstat", unimplementedFunc),
    /* 1051 */ SyscallDesc("fstat", fstatFunc<ArmLinux64>),
    /* 1052 */ SyscallDesc("fcntl", fcntlFunc),
    /* 1053 */ SyscallDesc("fadvise64", unimplementedFunc),
    /* 1054 */ SyscallDesc("newfstatat", unimplementedFunc),
    /* 1055 */ SyscallDesc("fstatfs", unimplementedFunc),
    /* 1056 */ SyscallDesc("statfs", unimplementedFunc),
    /* 1057 */ SyscallDesc("lseek", lseekFunc),
    /* 1058 */ SyscallDesc("mmap", mmapFunc<ArmLinux64>),
    /* 1059 */ SyscallDesc("alarm", unimplementedFunc),
    /* 1060 */ SyscallDesc("getpgrp", unimplementedFunc),
    /* 1061 */ SyscallDesc("pause", unimplementedFunc),
    /* 1062 */ SyscallDesc("time", timeFunc<ArmLinux64>),
    /* 1063 */ SyscallDesc("utime", unimplementedFunc),
    /* 1064 */ SyscallDesc("creat", unimplementedFunc),
    /* 1065 */ SyscallDesc("getdents", unimplementedFunc),
    /* 1066 */ SyscallDesc("futimesat", unimplementedFunc),
    /* 1067 */ SyscallDesc("select", unimplementedFunc),
    /* 1068 */ SyscallDesc("poll", unimplementedFunc),
    /* 1069 */ SyscallDesc("epoll_wait", unimplementedFunc),
    /* 1070 */ SyscallDesc("ustat", unimplementedFunc),
    /* 1071 */ SyscallDesc("vfork", unimplementedFunc),
    /* 1072 */ SyscallDesc("oldwait4", unimplementedFunc),
    /* 1073 */ SyscallDesc("recv", unimplementedFunc),
    /* 1074 */ SyscallDesc("send", unimplementedFunc),
    /* 1075 */ SyscallDesc("bdflush", unimplementedFunc),
    /* 1076 */ SyscallDesc("umount", unimplementedFunc),
    /* 1077 */ SyscallDesc("uselib", unimplementedFunc),
    /* 1078 */ SyscallDesc("_sysctl", unimplementedFunc),
    /* 1079 */ SyscallDesc("fork", unimplementedFunc)
};

static SyscallDesc privSyscallDescs32[] = {
    /*  1 */ SyscallDesc("breakpoint", unimplementedFunc),
    /*  2 */ SyscallDesc("cacheflush", unimplementedFunc),
    /*  3 */ SyscallDesc("usr26", unimplementedFunc),
    /*  4 */ SyscallDesc("usr32", unimplementedFunc),
    /*  5 */ SyscallDesc("set_tls", setTLSFunc32)
};

// Indices 1, 3 and 4 are unallocated.
static SyscallDesc privSyscallDescs64[] = {
    /*  1 */ SyscallDesc("unallocated", unimplementedFunc),
    /*  2 */ SyscallDesc("cacheflush", unimplementedFunc),
    /*  3 */ SyscallDesc("unallocated", unimplementedFunc),
    /*  4 */ SyscallDesc("unallocated", unimplementedFunc),
    /*  5 */ SyscallDesc("set_tls", setTLSFunc64)
};

ArmLinuxProcess32::ArmLinuxProcess32(ProcessParams * params,
        ObjectFile *objFile, ObjectFile::Arch _arch)
    : ArmProcess32(params, objFile, _arch)
{
    SyscallTable table;

    table.descs = syscallDescs32;
    table.size = sizeof(syscallDescs32) / sizeof(SyscallDesc);
    table.base = 0;
    syscallTables.push_back(table);
    table.base = 0x900000;
    syscallTables.push_back(table);

    table.descs = privSyscallDescs32;
    table.size = sizeof(privSyscallDescs32) / sizeof(SyscallDesc);
    table.base = 0xf0001;
    syscallTables.push_back(table);
}

ArmLinuxProcess64::ArmLinuxProcess64(ProcessParams * params,
        ObjectFile *objFile, ObjectFile::Arch _arch)
    : ArmProcess64(params, objFile, _arch)
{
    SyscallTable table;

    table.descs = syscallDescs64;
    table.size = sizeof(syscallDescs64) / sizeof(SyscallDesc);
    table.base = 0;
    syscallTables.push_back(table);
    table.base = 0x900000;
    syscallTables.push_back(table);

    table.descs = privSyscallDescs64;
    table.size = sizeof(privSyscallDescs64) / sizeof(SyscallDesc);
    table.base = 0x1001;
    syscallTables.push_back(table);
}

const Addr ArmLinuxProcess32::commPage = 0xffff0000;

SyscallDesc*
ArmLinuxProcessBits::getLinuxDesc(int callnum)
{
    // Angel SWI syscalls are unsupported in this release
    if (callnum == 0x123456)
        panic("Attempt to execute an ANGEL_SWI system call (newlib-related)");
    for (unsigned i = 0; i < syscallTables.size(); i++) {
        SyscallDesc *desc = syscallTables[i].getDesc(callnum);
        if (desc)
            return desc;
    }
    return NULL;
}

SyscallDesc *
ArmLinuxProcessBits::SyscallTable::getDesc(int callnum) const
{
    int offset = callnum - base;
    if (offset < 0 || offset >= size)
        return NULL;
    return &descs[offset];
}

SyscallDesc*
ArmLinuxProcess32::getDesc(int callnum)
{
    return getLinuxDesc(callnum);
}

SyscallDesc*
ArmLinuxProcess64::getDesc(int callnum)
{
    return getLinuxDesc(callnum);
}

void
ArmLinuxProcess32::initState()
{
    ArmProcess32::initState();
    allocateMem(commPage, PageBytes);
    ThreadContext *tc = system->getThreadContext(contextIds[0]);

    uint8_t swiNeg1[] = {
        0xff, 0xff, 0xff, 0xef  // swi -1
    };

    // Fill this page with swi -1 so we'll no if we land in it somewhere.
    for (Addr addr = 0; addr < PageBytes; addr += sizeof(swiNeg1)) {
        tc->getMemProxy().writeBlob(commPage + addr,
                                    swiNeg1, sizeof(swiNeg1));
    }

    uint8_t memory_barrier[] =
    {
        0x5f, 0xf0, 0x7f, 0xf5, // dmb
        0x0e, 0xf0, 0xa0, 0xe1  // return
    };
    tc->getMemProxy().writeBlob(commPage + 0x0fa0, memory_barrier,
                                sizeof(memory_barrier));

    uint8_t cmpxchg[] =
    {
        0x9f, 0x3f, 0x92, 0xe1,  // ldrex    r3, [r2]
        0x00, 0x30, 0x53, 0xe0,  // subs     r3, r3, r0
        0x91, 0x3f, 0x82, 0x01,  // strexeq  r3, r1, [r2]
        0x01, 0x00, 0x33, 0x03,  // teqeq    r3, #1
        0xfa, 0xff, 0xff, 0x0a,  // beq 1b
        0x00, 0x00, 0x73, 0xe2,  // rsbs r0, r3, #0
        0x5f, 0xf0, 0x7f, 0xf5,  // dmb
        0x0e, 0xf0, 0xa0, 0xe1   // return
    };
    tc->getMemProxy().writeBlob(commPage + 0x0fc0, cmpxchg, sizeof(cmpxchg));

    uint8_t get_tls[] =
    {
                                // read user read-only thread id register
        0x70, 0x0f, 0x1d, 0xee, // mrc p15, 0, r0, c13, c0, 3
        0x0e, 0xf0, 0xa0, 0xe1  // return
    };
    tc->getMemProxy().writeBlob(commPage + 0x0fe0, get_tls, sizeof(get_tls));
}

void
ArmLinuxProcess64::initState()
{
    ArmProcess64::initState();
    // The 64 bit equivalent of the comm page would be set up here.
}
