import os
import platform
import shutil
import logging
from subprocess import call

from downward.experiment import FastDownwardExperiment
from downward.reports.absolute import AbsoluteReport
from downward.reports.comparison import ComparisonReport
from downward.reports.scatter import ScatterPlotReport
from lab.environments import BaselSlurmEnvironment, LocalEnvironment
from lab import tools

from tools.my_reports import FlexibleAggregationReport, DomainsWithDifferentValuesOfAttributeReport, AlgorithmMatrixReport, TasksWithSpecificValueOfAttributeReport, SuiteReport, DomainwiseReport
import exit_reason_parser_attributes

try:
    from relativescatter import RelativeScatterPlotReport
    matplotlib = True
except ImportError:
    print 'matplotlib not availabe, scatter plots not available'
    matplotlib = False

class MyExperiment(FastDownwardExperiment):
    REMOTE = (platform.node() == "login-infai.scicore.unibas.ch") or platform.node().endswith(".cluster.bc2.ch")
    SCP_LOGIN = 'grid'
    # Note: do not use ~ here because we need both home-paths for scp steps
    REMOTE_EXP_PATH = '/infai/sieverss/experiments/'
    LOCAL_EXP_PATH = '/home/silvan/experiments/'
    REVISION_CACHE = os.path.expanduser('~/experiments/revision_cache')

    # Default settings
    # ./suites.py optimal_strips (all ipcs up to 14)
    DEFAULT_GRID_SUITE = [
        'airport', 'barman-opt11-strips', 'barman-opt14-strips', 'blocks',
        'childsnack-opt14-strips', 'depot', 'driverlog',
        'elevators-opt08-strips', 'elevators-opt11-strips',
        'floortile-opt11-strips', 'floortile-opt14-strips', 'freecell',
        'ged-opt14-strips', 'grid', 'gripper', 'hiking-opt14-strips',
        'logistics00', 'logistics98', 'miconic', 'movie', 'mprime', 'mystery',
        'nomystery-opt11-strips', 'openstacks-opt08-strips',
        'openstacks-opt11-strips', 'openstacks-opt14-strips',
        'openstacks-strips', 'parcprinter-08-strips',
        'parcprinter-opt11-strips', 'parking-opt11-strips',
        'parking-opt14-strips', 'pathways-noneg', 'pegsol-08-strips',
        'pegsol-opt11-strips', 'pipesworld-notankage', 'pipesworld-tankage',
        'psr-small', 'rovers', 'satellite', 'scanalyzer-08-strips',
        'scanalyzer-opt11-strips', 'sokoban-opt08-strips',
        'sokoban-opt11-strips', 'storage', 'tetris-opt14-strips',
        'tidybot-opt11-strips', 'tidybot-opt14-strips', 'tpp',
        'transport-opt08-strips', 'transport-opt11-strips',
        'transport-opt14-strips', 'trucks-strips', 'visitall-opt11-strips',
        'visitall-opt14-strips', 'woodworking-opt08-strips',
        'woodworking-opt11-strips', 'zenotravel'
    ]
    DEFAULT_LOCAL_SUITE = ['depot:p01.pddl']
    DEFAULT_ATTRIBUTES = [
        'cost',
        'coverage',
        'error',
        #'evaluations_until_last_jump',
        'expansions_until_last_jump',
        'generated_until_last_jump',
        'initial_h_value',
        'memory',
        'run_dir',
        'search_time',
        'total_time',
        #'unsolvable',
        'unsolvable_incomplete',
        #'evaluations',
        #'expansions',
        #'generated',
    ]
    SCORE_ATTRIBUTES = [
        #'score_evaluations',
        'score_expansions',
        'score_memory',
        'score_search_time',
        'score_total_time',
    ]

    def __init__(self, repo_name, exp_name=None, configs=None, revisions=None,
                 grid_suite=None, test_suite=None, build_options=None,
                 driver_options=None, extra_options=None, **kwargs):
        script_path = tools.get_script_path()
        exp_name = exp_name or os.path.splitext(os.path.basename(script_path))[0]
        project_name = os.path.basename(os.path.dirname(script_path))
        name=os.path.join(project_name, exp_name)

        remote_exp_path = os.path.join(self.REMOTE_EXP_PATH, name)
        local_exp_path = os.path.join(self.LOCAL_EXP_PATH, name)

        repo = os.path.expanduser('~/repos/' + repo_name)
        if self.REMOTE:
            path = remote_exp_path
            environment = BaselSlurmEnvironment(
                email='silvan.sievers@unibas.ch',
                extra_options=extra_options,
                export=["PATH", "DOWNWARD_BENCHMARKS"])
            self.suite = grid_suite or self.DEFAULT_GRID_SUITE
        else:
            path = local_exp_path
            environment = LocalEnvironment(processes=4)
            self.suite = test_suite or self.DEFAULT_LOCAL_SUITE

        FastDownwardExperiment.__init__(self,
                                        path=path,
                                        environment=environment,
                                        revision_cache=self.REVISION_CACHE,
                                        **kwargs)

        benchmarks_dir = os.path.join(repo, 'benchmarks')
        if not os.path.isdir(benchmarks_dir):
            benchmarks_dir = os.path.expanduser('~/repos/downward/benchmarks')
        assert os.path.isdir(benchmarks_dir)
        self.add_suite(benchmarks_dir, self.suite)
        self.attributes = self.DEFAULT_ATTRIBUTES

        # Add default parsers to the experiment.
        self.add_parser(self.LAB_STATIC_PROPERTIES_PARSER)
        self.add_parser(self.LAB_DRIVER_PARSER)
        self.add_parser(self.EXITCODE_PARSER)
        self.add_parser(self.TRANSLATOR_PARSER)
        self.add_parser(self.SINGLE_SEARCH_PARSER)

        # Add custom parser
        self.search_parsers = []
        self.add_search_parser('../exit-reason-parser.py', exit_reason_parser_attributes.get())

        for nick, cmdline in configs.items():
            for rev in revisions:
                self.add_algorithm('%s-%s' % (rev, nick), repo, rev, cmdline,
                                   build_options, driver_options)

        self.add_step('build', self.build)
        self.add_step('start', self.start_runs)
        self.add_fetcher(name='fetch')

        if not self.REMOTE:
            # Copy the properties file from the evaluation directory to local directory
            self.add_step('scp-eval-dir',
                              call,
                              ['mkdir -p %s-eval && scp -r %s:%s-eval/properties %s-eval/properties'
                              % (local_exp_path, self.SCP_LOGIN, remote_exp_path, local_exp_path)], shell=True)

        # Remove the experiment (and evaluation) directory
        self.add_step('remove-exp-dir', shutil.rmtree,
                          self.path, ignore_errors=True)
        self.add_step('remove-eval-dir', shutil.rmtree,
                          self.eval_dir, ignore_errors=True)

    def add_search_parser(self, path_to_parser, attributes):
        self.add_parser(path_to_parser)
        self.search_parsers.append(path_to_parser)
        self.attributes.extend(attributes)

    def add_score_attributes(self):
        self.attributes.extend(self.SCORE_ATTRIBUTES)

    def add_absolute_report(self, name=None, publish=True, **kwargs):
        kwargs.setdefault('attributes', self.attributes)
        format = kwargs.get('format', None)
        if format is not None and format == 'tex':
            ending = 'tex'
        else:
            ending = 'html'
        name = name or 'abs'
        outfile = os.path.join(self.eval_dir, '%s-%s.%s' % (self.name, name, ending))
        name += '-%s' % ending
        self.add_report(AbsoluteReport(**kwargs),
                        name='report-%s' % name, outfile=outfile)
        if publish:
            self.add_step('publish-%s' % name, call, ['publish', outfile])

    def add_comparison_report(self, algorithm_pairs, name=None,
                              publish=True, **kwargs):
        kwargs.setdefault('attributes', self.attributes)
        format = kwargs.get('format', None)
        if format is not None and format == 'tex':
            ending = 'tex'
        else:
            ending = 'html'
        name = name or 'comp'
        outfile = os.path.join(self.eval_dir, '%s-%s.%s' % (self.name, name, ending))
        self.add_report(ComparisonReport(algorithm_pairs=algorithm_pairs, **kwargs),
                        name='report-%s' % name, outfile=outfile)
        if publish:
            self.add_step('publish-%s' % name, call, ['publish', outfile])

    def add_scatter_plot_report(self, attributes, relative=False, name=None, **kwargs):
        """Add step creating (relative) scatter plots for all revision pairs.

        Create a scatter plot for each combination of attribute,
        configuration and revisions pair. filter_algorithm should be used
        to provide exactly two algorithms that should be compared.
        """
        if matplotlib:
            format = kwargs.get('format', None)
            if format is not None:
                if format == 'tex':
                    ending = 'tex'
                elif format == 'png':
                    ending = 'png'
            else:
                ending = 'html'
            name = name or 'scatter'
            outfile = os.path.join(self.eval_dir, '%s-%s.%s' % (self.name, name, ending))
            if relative:
                report_class = RelativeScatterPlotReport
            else:
                report_class = ScatterPlotReport

            self.add_report(
                report_class(
                    attributes=attributes,
                    #get_category=lambda run1, run2: run1["domain"],
                    **kwargs),
                name='report-%s' % name,
                outfile=outfile)

    def add_tasks_with_specific_value_of_attribute_report(
        self, specific_value, name='', **kwargs):
        if name is not '':
            name += '-'
        name += 'tasks-with-non-none-value-of-attribute'
        outfile = os.path.join(self.eval_dir, '%s-%s.txt' % (self.name, name))
        self.add_report(TasksWithSpecificValueOfAttributeReport(specific_value, **kwargs),
                        name='report-%s' % name,
                        outfile=outfile)

    def add_flexible_aggregation_report(self,
                                        name='',
                                        **kwargs):
        kwargs.setdefault('attributes', self.attributes)
        if name is not '':
            name += '-'
        name += 'flexible-aggregation'
        outfile = os.path.join(self.eval_dir, '%s-%s.txt' % (self.name, name))
        self.add_report(FlexibleAggregationReport(**kwargs),
                        name='report-%s' % name,
                        outfile=outfile)

    def add_domains_with_different_values_of_attribute_report(
        self, name='', **kwargs):
        if name is not '':
            name += '-'
        name += 'domains-with-different-values-of-attribute'
        outfile = os.path.join(self.eval_dir, '%s-%s.txt' % (self.name, name))
        self.add_report(DomainsWithDifferentValuesOfAttributeReport(**kwargs),
                        name='report-%s' % name,
                        outfile=outfile)

    def add_algorithm_matrix_report(
        self,
        name='',
        algorithm_matrix=None,
        row_names=[],
        column_names=[],
        attributes=[],
        row_names_start_algorithm_names=False,
        column_names_end_algorithm_names=False,
        attribute_function_pairs=[('coverage', sum)],
        **kwargs):
        if name is not '':
            name += '-'
        name += 'algorithm-matrix'
        outfile = os.path.join(self.eval_dir, '%s-%s.txt' % (self.name, name))
        kwargs.setdefault('attributes', ['coverage'])
        self.add_report(AlgorithmMatrixReport(
                                algorithm_matrix=algorithm_matrix,
                                row_names=row_names,
                                column_names=column_names,
                                row_names_start_algorithm_names=row_names_start_algorithm_names,
                                column_names_end_algorithm_names=column_names_end_algorithm_names,
                                attribute_function_pairs=attribute_function_pairs,
                                **kwargs),
                        name='report-%s' % name,
                        outfile=outfile)

    def add_suite_report(
        self,
        name='',
        **kwargs):
        if name is not '':
            name += '-'
        name += 'suite'
        outfile = os.path.join(self.eval_dir, '%s-%s.txt' % (self.name, name))
        self.add_report(SuiteReport(**kwargs),
                        name='report-%s' % name,
                        outfile=outfile)

    def add_domain_wise_report(
        self,
        name='',
        **kwargs):
        if name is not '':
            name += '-'
        name += 'domain-wise'
        outfile = os.path.join(self.eval_dir, '%s-%s.txt' % (self.name, name))
        self.add_report(DomainwiseReport(**kwargs),
                        name='report-%s' % name,
                        outfile=outfile)
