#include "symmetries.h"

#include "symmetry_generator.h"

#include "../abstraction.h"

#include "../../globals.h"
#include "../../utils/timer.h"
#include "../../utils/system.h"

#include <cassert>
#include <iostream>
#include <limits>

using namespace std;
using namespace utils;

namespace mas_new_lr {
// TODO: copied and renamed from shrink_strategy.h
typedef __gnu_cxx::slist<AbstractStateRef> EquivClass;
typedef std::vector<EquivClass> EquivRel;

Symmetries::Symmetries(const Options &options)
    : gc(options),
      symmetries_for_shrinking(SymmetriesForShrinking(options.get_enum("symmetries_for_shrinking"))),
      symmetries_for_merging(SymmetriesForMerging(options.get_enum("symmetries_for_merging"))),
      external_merging(ExternalMerging(options.get_enum("external_merging"))),
      internal_merging(InternalMerging(options.get_enum("internal_merging"))),
      bliss_time(0) {
}

bool Symmetries::find_and_apply_symmetries(const vector<Abstraction *> &abstractions,
                                           vector<pair<int, int> > &merge_order) {
    // We must make sure that all abstractions distances have been computed
    // because of the nasty possible side effect of pruning irrelevant
    // states and because the application of an equivalence realtion to an
    // abstraction requires distances to be computed.
    for (size_t i = 0; i < abstractions.size(); ++i) {
        if (abstractions[i])
            abstractions[i]->compute_distances();
    }
    bliss_time = gc.compute_generators(abstractions);
    if (get_num_generators() == 0 || is_bliss_limit_reached()) {
        return false;
    }


    int chosen_generator_for_merging = -1;
    int smallest_generator_affected_abstrations_size = numeric_limits<int>::max();
    int smallest_generator_mapped_abstractions_size = numeric_limits<int>::max();
    int largest_generator_affected_abstrations_size = 0;
    int largest_generator_mapped_abstractions_size = 0;
    vector<int> atomic_generators;
    vector<int> local_generators;

    // go over the generators and classify them into atomic, local or general
    // ones. also store information about the smallest/largest generator
    // with respect to overall affected abstractions or mapped abstractions,
    // depending on the chosen setting.
    for (size_t generator_index = 0; static_cast<int>(generator_index) < get_num_generators(); ++generator_index) {
        const SymmetryGenerator *generator = get_symmetry_generator(generator_index);
        const vector<int> &internally_affected_abstractions = generator->get_internally_affected_abstractions();
        const vector<int> &mapped_abstractions = generator->get_mapped_abstractions();
        const vector<int> &overall_affected_abstractions = generator->get_overall_affected_abstractions();

        int number_overall_affected_abstractions = overall_affected_abstractions.size();
        if (number_overall_affected_abstractions < 1) {
            cerr << "Something is wrong! The generator is the identity generator." << endl;
            exit_with(ExitCode::CRITICAL_ERROR);
        }
        if (number_overall_affected_abstractions == 1) {
            atomic_generators.push_back(generator_index);
        } else {
            if (external_merging == MERGE_FOR_ATOMIC) {
                if (symmetries_for_merging == SMALLEST
                        && number_overall_affected_abstractions
                        < smallest_generator_affected_abstrations_size) {
                    smallest_generator_affected_abstrations_size
                            = number_overall_affected_abstractions;
                    chosen_generator_for_merging = generator_index;
                } else if (symmetries_for_merging == LARGEST
                              && number_overall_affected_abstractions
                              > largest_generator_affected_abstrations_size) {
                    largest_generator_affected_abstrations_size
                            = number_overall_affected_abstractions;
                    chosen_generator_for_merging = generator_index;
                }
            }
        }

        int number_mapped_abstractions = mapped_abstractions.size();
        if (number_mapped_abstractions == 0) {
            // note that this also includes atomic generators
            local_generators.push_back(generator_index);
        } else {
            if (external_merging == MERGE_FOR_LOCAL) {
                if (symmetries_for_merging == SMALLEST
                        && number_mapped_abstractions
                        < smallest_generator_mapped_abstractions_size) {
                    smallest_generator_mapped_abstractions_size
                            = number_mapped_abstractions;
                    chosen_generator_for_merging = generator_index;
                }
                if (symmetries_for_merging == LARGEST
                        && number_mapped_abstractions
                        > largest_generator_mapped_abstractions_size) {
                    largest_generator_mapped_abstractions_size
                            = number_mapped_abstractions;
                    chosen_generator_for_merging = generator_index;
                }
            }
        }

        // dump generator properties
        cout << "Generator " << generator_index << endl;
        for (size_t i = 0; i < mapped_abstractions.size(); ++i) {
            int abs_index = mapped_abstractions[i];
            int to_index = generator->get_value(abs_index);
            cout << abstractions[abs_index]->description() << " mapped to " <<
                    abstractions[to_index]->description();
            if (generator->internally_affects(abs_index))
                cout << " (and also internally affected)";
            cout << endl;
        }
        for (size_t i = 0; i < internally_affected_abstractions.size(); ++i) {
            int abs_index = internally_affected_abstractions[i];
            assert(!generator->maps(abs_index));
            cout << abstractions[abs_index]->description() << " internally affected" << endl;
        }
    }

    // apply symmetries if possible
    bool applied_symmetries = false;
    if (symmetries_for_shrinking == ATOMIC || symmetries_for_shrinking == LOCAL) {
        cerr << "shrinking with symmetries is not implemented!" << endl;
        exit_with(ExitCode::UNSUPPORTED);
    }

    if (symmetries_for_merging != NO_MERGING && chosen_generator_for_merging != -1) {
        vector<vector<int> > cycles;
        vector<int> merge_linear_abstractions;
        const SymmetryGenerator *generator =
                get_symmetry_generator(chosen_generator_for_merging);

        // Always include all mapped abstractions
        if (internal_merging == NON_LINEAR
                || external_merging == MERGE_FOR_LOCAL) {
            // if the internal merge strategy is non linear or we only want
            // to merge every cycle (non linearly), we need to
            // compute the actual cycles of abstraction mappings.
            generator->compute_cycles(cycles);
        } else if (internal_merging == LINEAR) {
            // if the internal merge strategy is linear, we simply collect
            // all mapped abstractions (i.e. the same abstractions as above,
            // but we do not compute cycle information)
            const vector<int> &mapped_abstractions =
                    generator->get_mapped_abstractions();
            merge_linear_abstractions.insert(merge_linear_abstractions.end(),
                                             mapped_abstractions.begin(),
                                             mapped_abstractions.end());
        }

        // If merging for least/most number of overall affected abstactions,
        // also include the non-mapped, i.e. internally affected abstractions
        // (always as to be linearly merged abstractions)
        if (external_merging == MERGE_FOR_ATOMIC) {
            const vector<int> &internally_affected_abstractions =
                    generator->get_internally_affected_abstractions();
            merge_linear_abstractions.insert(merge_linear_abstractions.end(),
                                             internally_affected_abstractions.begin(),
                                             internally_affected_abstractions.end());
        }

        // compute a merge tree
        assert(merge_order.empty());
        int number_of_abstractions = abstractions.size();
        int number_of_merges = 0;
        vector<int> merge_linear_indices;
        for (size_t cycle_no = 0; cycle_no < cycles.size(); ++cycle_no) {
            // go over the cycles and compute a non-linear merge order.
            const vector<int> &cycle = cycles[cycle_no];
            size_t abs_index_1 = cycle[0];
            for (size_t i = 1; i < cycle.size(); ++i) {
                size_t abs_index_2 = cycle[i];
                merge_order.push_back(make_pair(abs_index_1, abs_index_2));
                abs_index_1 = number_of_abstractions + number_of_merges;
                ++number_of_merges;
            }
            if (external_merging == MERGE_FOR_ATOMIC) {
                // number_of_abstractions + number_of_merges always is the *next*
                // position where a new merged abstraction will be stored at.
                // here, we need the *last* position where the abstraction
                // resulting from merging the cycle was stored, hence the -1.
                merge_linear_indices.push_back(number_of_abstractions + number_of_merges - 1);
            }
        }

        if (external_merging == MERGE_FOR_ATOMIC) {
            // merge_linear_indices possibly contains abstractions that have been
            // non-linearly merged from information about cycles.
            // here we add abstractions that need to be merged linearly anyways
            merge_linear_indices.insert(merge_linear_indices.end(),
                                        merge_linear_abstractions.begin(),
                                        merge_linear_abstractions.end());

            // go over all abstractions that (now) need to be merged linearly
            size_t abs_index_1 = merge_linear_indices[0];
            for (size_t i = 1; i < merge_linear_indices.size(); ++i) {
                size_t abs_index_2 = merge_linear_indices[i];
                merge_order.push_back(make_pair(abs_index_1, abs_index_2));
                abs_index_1 = number_of_abstractions + number_of_merges;
                ++number_of_merges;
            }
        }

        cout << "current number of abstractions " << number_of_abstractions << endl;
        cout << "chosen internal merge order: " << endl;
        for (size_t i = 0; i < merge_order.size(); ++i) {
            cout << merge_order[i].first << ", " << merge_order[i].second << endl;
        }
    }

    return applied_symmetries;
}

const SymmetryGenerator* Symmetries::get_symmetry_generator(int ind) const {
    assert(ind >= 0 && ind < get_num_generators());
    return gc.get_symmetry_generators()[ind];
}
}
