% This function solves the differential equation for the bond graph model
% of SERCA, based on the Tran et al. (2009) model. The inputs are the
% simulation duration and initial conditions

function [VOI, STATES, ALGEBRAIC, CONSTANTS] = SERCA_ode(tspan,init_states)
    % Initialise constants and state variables
    [INIT_STATES, CONSTANTS] = initConsts;
    if exist('init_states','var')
        INIT_STATES = init_states;
    end

    % Set numerical accuracy options for ODE solver
    options = odeset('RelTol', 1e-10, 'AbsTol', 1e-15, 'MaxStep', 1);

    % Solve model with ODE solver
    [VOI, STATES] = ode15s(@(VOI, STATES)computeRates(VOI, STATES, CONSTANTS), tspan, INIT_STATES, options);

    % Compute algebraic variables
    [RATES, ALGEBRAIC] = computeRates(VOI, STATES, CONSTANTS);
    ALGEBRAIC = computeAlgebraic(ALGEBRAIC, CONSTANTS, STATES, VOI);
end

function [LEGEND_STATES, LEGEND_ALGEBRAIC, LEGEND_VOI, LEGEND_CONSTANTS] = createLegends()
    LEGEND_STATES = ''; LEGEND_ALGEBRAIC = ''; LEGEND_VOI = ''; LEGEND_CONSTANTS = '';
    LEGEND_VOI = strpad('t in component environment (second)');
    LEGEND_CONSTANTS(:,1) = strpad('R in component environment (J_per_K_per_mol)');
    LEGEND_CONSTANTS(:,2) = strpad('T in component environment (kelvin)');
    LEGEND_CONSTANTS(:,3) = strpad('F in component environment (C_per_mol)');
    LEGEND_CONSTANTS(:,4) = strpad('K_1 in component environment (dimensionless)');
    LEGEND_CONSTANTS(:,5) = strpad('K_10 in component environment (dimensionless)');
    LEGEND_CONSTANTS(:,6) = strpad('K_2 in component environment (dimensionless)');
    LEGEND_CONSTANTS(:,7) = strpad('K_2a in component environment (dimensionless)');
    LEGEND_CONSTANTS(:,8) = strpad('K_4 in component environment (dimensionless)');
    LEGEND_CONSTANTS(:,9) = strpad('K_5 in component environment (dimensionless)');
    LEGEND_CONSTANTS(:,10) = strpad('K_6 in component environment (dimensionless)');
    LEGEND_CONSTANTS(:,11) = strpad('K_8 in component environment (dimensionless)');
    LEGEND_CONSTANTS(:,12) = strpad('K_9 in component environment (dimensionless)');
    LEGEND_CONSTANTS(:,13) = strpad('K_Cai in component environment (dimensionless)');
    LEGEND_CONSTANTS(:,14) = strpad('K_Casr in component environment (dimensionless)');
    LEGEND_CONSTANTS(:,15) = strpad('K_H in component environment (dimensionless)');
    LEGEND_CONSTANTS(:,16) = strpad('K_MgADP in component environment (dimensionless)');
    LEGEND_CONSTANTS(:,17) = strpad('K_MgATP in component environment (dimensionless)');
    LEGEND_CONSTANTS(:,18) = strpad('K_P in component environment (dimensionless)');
    LEGEND_CONSTANTS(:,19) = strpad('kappa_101 in component environment (dimensionless)');
    LEGEND_CONSTANTS(:,20) = strpad('kappa_12 in component environment (dimensionless)');
    LEGEND_CONSTANTS(:,21) = strpad('kappa_22a in component environment (dimensionless)');
    LEGEND_CONSTANTS(:,22) = strpad('kappa_24 in component environment (dimensionless)');
    LEGEND_CONSTANTS(:,23) = strpad('kappa_45 in component environment (dimensionless)');
    LEGEND_CONSTANTS(:,24) = strpad('kappa_56 in component environment (dimensionless)');
    LEGEND_CONSTANTS(:,25) = strpad('kappa_68 in component environment (dimensionless)');
    LEGEND_CONSTANTS(:,26) = strpad('kappa_89 in component environment (dimensionless)');
    LEGEND_CONSTANTS(:,27) = strpad('kappa_910 in component environment (dimensionless)');
    LEGEND_CONSTANTS(:,28) = strpad('n_Cai in component environment (dimensionless)');
    LEGEND_CONSTANTS(:,29) = strpad('n_Casr in component environment (dimensionless)');
    LEGEND_CONSTANTS(:,30) = strpad('n_H in component environment (dimensionless)');
    LEGEND_STATES(:,1) = strpad('x_Ca_i in component environment (dimensionless)');
    LEGEND_STATES(:,2) = strpad('x_Ca_sr in component environment (dimensionless)');
    LEGEND_STATES(:,3) = strpad('x_H in component environment (dimensionless)');
    LEGEND_STATES(:,4) = strpad('x_MgADP in component environment (dimensionless)');
    LEGEND_STATES(:,5) = strpad('x_MgATP in component environment (dimensionless)');
    LEGEND_STATES(:,6) = strpad('x_P1 in component environment (dimensionless)');
    LEGEND_STATES(:,7) = strpad('x_Pi in component environment (dimensionless)');
    LEGEND_STATES(:,8) = strpad('x_P2 in component environment (dimensionless)');
    LEGEND_STATES(:,9) = strpad('x_P2a in component environment (dimensionless)');
    LEGEND_STATES(:,10) = strpad('x_P4 in component environment (dimensionless)');
    LEGEND_STATES(:,11) = strpad('x_P5 in component environment (dimensionless)');
    LEGEND_STATES(:,12) = strpad('x_P10 in component environment (dimensionless)');
    LEGEND_STATES(:,13) = strpad('x_P6 in component environment (dimensionless)');
    LEGEND_STATES(:,14) = strpad('x_P8 in component environment (dimensionless)');
    LEGEND_STATES(:,15) = strpad('x_P9 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,12) = strpad('e_1 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,32) = strpad('e_2 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,36) = strpad('e_3 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,2) = strpad('e_4 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,73) = strpad('e_5 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,8) = strpad('e_6 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,4) = strpad('e_7 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,33) = strpad('e_8 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,40) = strpad('e_9 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,34) = strpad('e_10 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,6) = strpad('e_11 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,38) = strpad('e_12 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,14) = strpad('e_13 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,58) = strpad('e_14 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,72) = strpad('e_15 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,88) = strpad('e_16 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,10) = strpad('e_17 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,35) = strpad('e_18 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,50) = strpad('e_19 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,16) = strpad('e_20 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,48) = strpad('e_21 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,49) = strpad('e_22 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,60) = strpad('e_23 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,51) = strpad('e_24 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,18) = strpad('e_25 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,20) = strpad('e_26 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,52) = strpad('e_27 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,107) = strpad('e_28 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,105) = strpad('e_29 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,119) = strpad('e_30 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,54) = strpad('e_31 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,57) = strpad('e_32 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,22) = strpad('e_33 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,116) = strpad('e_34 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,113) = strpad('e_35 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,56) = strpad('e_36 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,137) = strpad('e_37 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,68) = strpad('e_38 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,136) = strpad('e_39 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,29) = strpad('e_40 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,70) = strpad('e_41 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,27) = strpad('e_42 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,85) = strpad('e_43 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,69) = strpad('e_44 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,148) = strpad('e_45 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,145) = strpad('e_46 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,80) = strpad('e_47 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,83) = strpad('e_48 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,31) = strpad('e_49 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,93) = strpad('e_50 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,86) = strpad('e_51 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,87) = strpad('e_52 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,25) = strpad('e_53 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,97) = strpad('f_1 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,46) = strpad('f_2 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,43) = strpad('f_3 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,120) = strpad('f_4 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,77) = strpad('f_5 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,79) = strpad('f_6 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,146) = strpad('f_7 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,94) = strpad('f_8 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,44) = strpad('f_9 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,135) = strpad('f_10 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,162) = strpad('f_11 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,92) = strpad('f_12 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,95) = strpad('f_13 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,76) = strpad('f_14 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,78) = strpad('f_15 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,91) = strpad('f_16 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,45) = strpad('f_17 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,160) = strpad('f_18 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,133) = strpad('f_19 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,118) = strpad('f_20 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,115) = strpad('f_21 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,65) = strpad('f_22 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,63) = strpad('f_23 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,67) = strpad('f_24 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,64) = strpad('f_25 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,128) = strpad('f_26 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,111) = strpad('f_27 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,112) = strpad('f_28 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,114) = strpad('f_29 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,125) = strpad('f_30 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,126) = strpad('f_31 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,129) = strpad('f_32 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,131) = strpad('f_33 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,127) = strpad('f_34 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,130) = strpad('f_35 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,117) = strpad('f_36 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,140) = strpad('f_37 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,142) = strpad('f_38 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,143) = strpad('f_39 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,158) = strpad('f_40 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,141) = strpad('f_41 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,144) = strpad('f_42 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,157) = strpad('f_43 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,155) = strpad('f_44 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,152) = strpad('f_45 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,154) = strpad('f_46 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,153) = strpad('f_47 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,101) = strpad('f_48 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,156) = strpad('f_49 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,102) = strpad('f_50 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,103) = strpad('f_51 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,104) = strpad('f_52 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,106) = strpad('f_53 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,1) = strpad('mu_Ca_i in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,123) = strpad('v_Ca_i in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,3) = strpad('mu_Ca_sr in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,149) = strpad('v_Ca_sr in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,5) = strpad('mu_H in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,163) = strpad('v_H in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,7) = strpad('mu_MgADP in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,81) = strpad('v_MgADP in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,9) = strpad('mu_MgATP in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,47) = strpad('v_MgATP in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,11) = strpad('mu_P1 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,100) = strpad('v_P1 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,13) = strpad('mu_Pi in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,98) = strpad('v_Pi in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,89) = strpad('Af_R10_1 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,39) = strpad('Ar_R10_1 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,90) = strpad('v_R10_1 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,37) = strpad('Af_R1_2 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,41) = strpad('Ar_R1_2 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,42) = strpad('v_R1_2 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,59) = strpad('Af_R5_6 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,74) = strpad('Ar_R5_6 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,75) = strpad('v_R5_6 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,15) = strpad('mu_P2 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,121) = strpad('v_P2 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,17) = strpad('mu_P2a in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,66) = strpad('v_P2a in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,19) = strpad('mu_P4 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,132) = strpad('v_P4 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,21) = strpad('mu_P5 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,134) = strpad('v_P5 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,61) = strpad('Af_R2_2a in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,23) = strpad('Ar_R2_2a in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,62) = strpad('v_R2_2a in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,109) = strpad('Af_R2_4 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,53) = strpad('Ar_R2_4 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,110) = strpad('v_R2_4 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,55) = strpad('Af_R4_5 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,122) = strpad('Ar_R4_5 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,124) = strpad('v_R4_5 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,24) = strpad('mu_P10 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,108) = strpad('v_P10 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,26) = strpad('mu_P6 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,147) = strpad('v_P6 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,28) = strpad('mu_P8 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,161) = strpad('v_P8 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,30) = strpad('mu_P9 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,159) = strpad('v_P9 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,71) = strpad('Af_R6_8 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,138) = strpad('Ar_R6_8 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,139) = strpad('v_R6_8 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,150) = strpad('Af_R8_9 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,82) = strpad('Ar_R8_9 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,151) = strpad('v_R8_9 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,84) = strpad('Af_R9_10 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,96) = strpad('Ar_R9_10 in component environment (dimensionless)');
    LEGEND_ALGEBRAIC(:,99) = strpad('v_R9_10 in component environment (dimensionless)');
    LEGEND_RATES(:,1) = strpad('d/dt x_Ca_i in component environment (dimensionless)');
    LEGEND_RATES(:,2) = strpad('d/dt x_Ca_sr in component environment (dimensionless)');
    LEGEND_RATES(:,3) = strpad('d/dt x_H in component environment (dimensionless)');
    LEGEND_RATES(:,4) = strpad('d/dt x_MgADP in component environment (dimensionless)');
    LEGEND_RATES(:,5) = strpad('d/dt x_MgATP in component environment (dimensionless)');
    LEGEND_RATES(:,6) = strpad('d/dt x_P1 in component environment (dimensionless)');
    LEGEND_RATES(:,7) = strpad('d/dt x_Pi in component environment (dimensionless)');
    LEGEND_RATES(:,8) = strpad('d/dt x_P2 in component environment (dimensionless)');
    LEGEND_RATES(:,9) = strpad('d/dt x_P2a in component environment (dimensionless)');
    LEGEND_RATES(:,10) = strpad('d/dt x_P4 in component environment (dimensionless)');
    LEGEND_RATES(:,11) = strpad('d/dt x_P5 in component environment (dimensionless)');
    LEGEND_RATES(:,12) = strpad('d/dt x_P10 in component environment (dimensionless)');
    LEGEND_RATES(:,13) = strpad('d/dt x_P6 in component environment (dimensionless)');
    LEGEND_RATES(:,14) = strpad('d/dt x_P8 in component environment (dimensionless)');
    LEGEND_RATES(:,15) = strpad('d/dt x_P9 in component environment (dimensionless)');
    LEGEND_STATES  = LEGEND_STATES';
    LEGEND_ALGEBRAIC = LEGEND_ALGEBRAIC';
    LEGEND_RATES = LEGEND_RATES';
    LEGEND_CONSTANTS = LEGEND_CONSTANTS';
end

function [STATES, CONSTANTS] = initConsts()
    VOI = 0; CONSTANTS = []; STATES = []; ALGEBRAIC = [];
    CONSTANTS(:,1) = 8.314;
    CONSTANTS(:,2) = 310;
    CONSTANTS(:,3) = 96485;
    CONSTANTS(:,4) = 5263.6085;
    CONSTANTS(:,5) = 410.6048;
    CONSTANTS(:,6) = 3803.6518;
    CONSTANTS(:,7) = 3110.4445;
    CONSTANTS(:,8) = 16520516.1239;
    CONSTANTS(:,9) = 0.82914;
    CONSTANTS(:,10) = 993148.433;
    CONSTANTS(:,11) = 37.7379;
    CONSTANTS(:,12) = 2230.2717;
    CONSTANTS(:,13) = 1.9058;
    CONSTANTS(:,14) = 31.764;
    CONSTANTS(:,15) = 1862.5406;
    CONSTANTS(:,16) = 5.8126e-07;
    CONSTANTS(:,17) = 244.3021;
    CONSTANTS(:,18) = 0.014921;
    CONSTANTS(:,19) = 0.049926;
    CONSTANTS(:,20) = 0.00053004;
    CONSTANTS(:,21) = 8326784.0537;
    CONSTANTS(:,22) = 1567.7476;
    CONSTANTS(:,23) = 1567.7476;
    CONSTANTS(:,24) = 3063.4006;
    CONSTANTS(:,25) = 130852.3839;
    CONSTANTS(:,26) = 11612934.8748;
    CONSTANTS(:,27) = 11612934.8748;
    CONSTANTS(:,28) = 2;
    CONSTANTS(:,29) = 2;
    CONSTANTS(:,30) = 2;
    STATES(:,1) = 0.0057;
    STATES(:,2) = 2.28;
    STATES(:,3) = 0.004028;
    STATES(:,4) = 1.3794;
    STATES(:,5) = 3.8;
    STATES(:,6) = 0.11111;
    STATES(:,7) = 38;
    STATES(:,8) = 0.11111;
    STATES(:,9) = 0.11111;
    STATES(:,10) = 0.11111;
    STATES(:,11) = 0.11111;
    STATES(:,12) = 0.11111;
    STATES(:,13) = 0.11111;
    STATES(:,14) = 0.11111;
    STATES(:,15) = 0.11111;
    CONSTANTS(:,31) = 0.000000;
    CONSTANTS(:,32) = 0.000000;
    CONSTANTS(:,33) = 0.000000;
    CONSTANTS(:,34) = 0.000000;
    CONSTANTS(:,35) = 0.000000;
    CONSTANTS(:,36) = 0.000000;
    if (isempty(STATES)), warning('Initial values for states not set');, end
end

function [RATES, ALGEBRAIC] = computeRates(VOI, STATES, CONSTANTS)
    global algebraicVariableCount;
    statesSize = size(STATES);
    statesColumnCount = statesSize(2);
    if ( statesColumnCount == 1)
        STATES = STATES';
        ALGEBRAIC = zeros(1, algebraicVariableCount);
        utilOnes = 1;
    else
        statesRowCount = statesSize(1);
        ALGEBRAIC = zeros(statesRowCount, algebraicVariableCount);
        RATES = zeros(statesRowCount, statesColumnCount);
        utilOnes = ones(statesRowCount, 1);
    end
    RATES(:,1) = CONSTANTS(:,31);
    RATES(:,2) = CONSTANTS(:,32);
    RATES(:,3) = CONSTANTS(:,33);
    RATES(:,4) = CONSTANTS(:,34);
    RATES(:,5) = CONSTANTS(:,35);
    RATES(:,7) = CONSTANTS(:,36);
    ALGEBRAIC(:,15) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,6).*STATES(:,8));
    ALGEBRAIC(:,16) = ALGEBRAIC(:,15);
    ALGEBRAIC(:,49) = ALGEBRAIC(:,16);
    ALGEBRAIC(:,5) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,15).*STATES(:,3));
    ALGEBRAIC(:,6) = ALGEBRAIC(:,5);
    ALGEBRAIC(:,34) = ALGEBRAIC(:,6);
    ALGEBRAIC(:,51) = ALGEBRAIC(:,34);
    ALGEBRAIC(:,60) = ALGEBRAIC(:,49)+ALGEBRAIC(:,51);
    ALGEBRAIC(:,61) = ALGEBRAIC(:,60);
    ALGEBRAIC(:,17) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,7).*STATES(:,9));
    ALGEBRAIC(:,18) = ALGEBRAIC(:,17);
    ALGEBRAIC(:,23) = ALGEBRAIC(:,18);
    ALGEBRAIC(:,62) =  CONSTANTS(:,21).*(exp(ALGEBRAIC(:,61)./( CONSTANTS(:,1).*CONSTANTS(:,2))) - exp(ALGEBRAIC(:,23)./( CONSTANTS(:,1).*CONSTANTS(:,2))));
    ALGEBRAIC(:,64) = ALGEBRAIC(:,62);
    ALGEBRAIC(:,66) = ALGEBRAIC(:,64);
    RATES(:,9) = ALGEBRAIC(:,66);
    ALGEBRAIC(:,11) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,4).*STATES(:,6));
    ALGEBRAIC(:,12) = ALGEBRAIC(:,11);
    ALGEBRAIC(:,32) = ALGEBRAIC(:,12);
    ALGEBRAIC(:,9) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,17).*STATES(:,5));
    ALGEBRAIC(:,10) = ALGEBRAIC(:,9);
    ALGEBRAIC(:,36) = ALGEBRAIC(:,32)+ALGEBRAIC(:,10);
    ALGEBRAIC(:,37) = ALGEBRAIC(:,36);
    ALGEBRAIC(:,40) = ALGEBRAIC(:,16);
    ALGEBRAIC(:,41) = ALGEBRAIC(:,40);
    ALGEBRAIC(:,42) =  CONSTANTS(:,20).*(exp(ALGEBRAIC(:,37)./( CONSTANTS(:,1).*CONSTANTS(:,2))) - exp(ALGEBRAIC(:,41)./( CONSTANTS(:,1).*CONSTANTS(:,2))));
    ALGEBRAIC(:,43) = ALGEBRAIC(:,42);
    ALGEBRAIC(:,46) = ALGEBRAIC(:,43);
    ALGEBRAIC(:,24) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,5).*STATES(:,12));
    ALGEBRAIC(:,25) = ALGEBRAIC(:,24);
    ALGEBRAIC(:,88) = ALGEBRAIC(:,25);
    ALGEBRAIC(:,89) = ALGEBRAIC(:,88);
    ALGEBRAIC(:,33) = ALGEBRAIC(:,12);
    ALGEBRAIC(:,13) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,18).*STATES(:,7));
    ALGEBRAIC(:,14) = ALGEBRAIC(:,13);
    ALGEBRAIC(:,38) = ALGEBRAIC(:,33)+ALGEBRAIC(:,14);
    ALGEBRAIC(:,39) = ALGEBRAIC(:,38);
    ALGEBRAIC(:,90) =  CONSTANTS(:,19).*(exp(ALGEBRAIC(:,89)./( CONSTANTS(:,1).*CONSTANTS(:,2))) - exp(ALGEBRAIC(:,39)./( CONSTANTS(:,1).*CONSTANTS(:,2))));
    ALGEBRAIC(:,92) = ALGEBRAIC(:,90);
    ALGEBRAIC(:,94) = ALGEBRAIC(:,92);
    ALGEBRAIC(:,97) =  - ALGEBRAIC(:,46)+ALGEBRAIC(:,94);
    ALGEBRAIC(:,100) = ALGEBRAIC(:,97);
    RATES(:,6) = ALGEBRAIC(:,100);
    ALGEBRAIC(:,91) = ALGEBRAIC(:,90);
    ALGEBRAIC(:,30) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,12).*STATES(:,15));
    ALGEBRAIC(:,31) = ALGEBRAIC(:,30);
    ALGEBRAIC(:,83) = ALGEBRAIC(:,31);
    ALGEBRAIC(:,84) = ALGEBRAIC(:,83);
    ALGEBRAIC(:,35) = ALGEBRAIC(:,6);
    ALGEBRAIC(:,86) = ALGEBRAIC(:,35);
    ALGEBRAIC(:,87) = ALGEBRAIC(:,25);
    ALGEBRAIC(:,93) = ALGEBRAIC(:,86)+ALGEBRAIC(:,87);
    ALGEBRAIC(:,96) = ALGEBRAIC(:,93);
    ALGEBRAIC(:,99) =  CONSTANTS(:,27).*(exp(ALGEBRAIC(:,84)./( CONSTANTS(:,1).*CONSTANTS(:,2))) - exp(ALGEBRAIC(:,96)./( CONSTANTS(:,1).*CONSTANTS(:,2))));
    ALGEBRAIC(:,102) = ALGEBRAIC(:,99);
    ALGEBRAIC(:,104) = ALGEBRAIC(:,102);
    ALGEBRAIC(:,106) = ALGEBRAIC(:,104) - ALGEBRAIC(:,91);
    ALGEBRAIC(:,108) = ALGEBRAIC(:,106);
    RATES(:,12) = ALGEBRAIC(:,108);
    ALGEBRAIC(:,44) = ALGEBRAIC(:,42);
    ALGEBRAIC(:,48) = ALGEBRAIC(:,16);
    ALGEBRAIC(:,1) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,13).*STATES(:,1));
    ALGEBRAIC(:,2) = ALGEBRAIC(:,1);
    ALGEBRAIC(:,56) = ALGEBRAIC(:,2);
    ALGEBRAIC(:,105) =  CONSTANTS(:,28).*ALGEBRAIC(:,56);
    ALGEBRAIC(:,107) = ALGEBRAIC(:,48)+ALGEBRAIC(:,105);
    ALGEBRAIC(:,109) = ALGEBRAIC(:,107);
    ALGEBRAIC(:,19) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,8).*STATES(:,10));
    ALGEBRAIC(:,20) = ALGEBRAIC(:,19);
    ALGEBRAIC(:,52) = ALGEBRAIC(:,20);
    ALGEBRAIC(:,53) = ALGEBRAIC(:,52);
    ALGEBRAIC(:,110) =  CONSTANTS(:,22).*(exp(ALGEBRAIC(:,109)./( CONSTANTS(:,1).*CONSTANTS(:,2))) - exp(ALGEBRAIC(:,53)./( CONSTANTS(:,1).*CONSTANTS(:,2))));
    ALGEBRAIC(:,112) = ALGEBRAIC(:,110);
    ALGEBRAIC(:,115) = ALGEBRAIC(:,112);
    ALGEBRAIC(:,63) = ALGEBRAIC(:,62);
    ALGEBRAIC(:,65) = ALGEBRAIC(:,63);
    ALGEBRAIC(:,118) = (ALGEBRAIC(:,44) - ALGEBRAIC(:,115)) - ALGEBRAIC(:,65);
    ALGEBRAIC(:,121) = ALGEBRAIC(:,118);
    RATES(:,8) = ALGEBRAIC(:,121);
    ALGEBRAIC(:,111) = ALGEBRAIC(:,110);
    ALGEBRAIC(:,54) = ALGEBRAIC(:,20);
    ALGEBRAIC(:,55) = ALGEBRAIC(:,54);
    ALGEBRAIC(:,21) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,9).*STATES(:,11));
    ALGEBRAIC(:,22) = ALGEBRAIC(:,21);
    ALGEBRAIC(:,57) = ALGEBRAIC(:,22);
    ALGEBRAIC(:,50) = ALGEBRAIC(:,34);
    ALGEBRAIC(:,113) =  CONSTANTS(:,30).*ALGEBRAIC(:,50);
    ALGEBRAIC(:,116) = ALGEBRAIC(:,113);
    ALGEBRAIC(:,119) = ALGEBRAIC(:,57)+ALGEBRAIC(:,116);
    ALGEBRAIC(:,122) = ALGEBRAIC(:,119);
    ALGEBRAIC(:,124) =  CONSTANTS(:,23).*(exp(ALGEBRAIC(:,55)./( CONSTANTS(:,1).*CONSTANTS(:,2))) - exp(ALGEBRAIC(:,122)./( CONSTANTS(:,1).*CONSTANTS(:,2))));
    ALGEBRAIC(:,126) = ALGEBRAIC(:,124);
    ALGEBRAIC(:,128) = ALGEBRAIC(:,111) - ALGEBRAIC(:,126);
    ALGEBRAIC(:,132) = ALGEBRAIC(:,128);
    RATES(:,10) = ALGEBRAIC(:,132);
    ALGEBRAIC(:,58) = ALGEBRAIC(:,22);
    ALGEBRAIC(:,59) = ALGEBRAIC(:,58);
    ALGEBRAIC(:,7) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,16).*STATES(:,4));
    ALGEBRAIC(:,8) = ALGEBRAIC(:,7);
    ALGEBRAIC(:,26) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,10).*STATES(:,13));
    ALGEBRAIC(:,27) = ALGEBRAIC(:,26);
    ALGEBRAIC(:,72) = ALGEBRAIC(:,27);
    ALGEBRAIC(:,73) = ALGEBRAIC(:,8)+ALGEBRAIC(:,72);
    ALGEBRAIC(:,74) = ALGEBRAIC(:,73);
    ALGEBRAIC(:,75) =  CONSTANTS(:,24).*(exp(ALGEBRAIC(:,59)./( CONSTANTS(:,1).*CONSTANTS(:,2))) - exp(ALGEBRAIC(:,74)./( CONSTANTS(:,1).*CONSTANTS(:,2))));
    ALGEBRAIC(:,76) = ALGEBRAIC(:,75);
    ALGEBRAIC(:,125) = ALGEBRAIC(:,124);
    ALGEBRAIC(:,129) = ALGEBRAIC(:,125);
    ALGEBRAIC(:,131) = ALGEBRAIC(:,129) - ALGEBRAIC(:,76);
    ALGEBRAIC(:,134) = ALGEBRAIC(:,131);
    RATES(:,11) = ALGEBRAIC(:,134);
    ALGEBRAIC(:,77) = ALGEBRAIC(:,75);
    ALGEBRAIC(:,78) = ALGEBRAIC(:,77);
    ALGEBRAIC(:,70) = ALGEBRAIC(:,27);
    ALGEBRAIC(:,71) = ALGEBRAIC(:,70);
    ALGEBRAIC(:,28) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,11).*STATES(:,14));
    ALGEBRAIC(:,29) = ALGEBRAIC(:,28);
    ALGEBRAIC(:,68) = ALGEBRAIC(:,29);
    ALGEBRAIC(:,3) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,14).*STATES(:,2));
    ALGEBRAIC(:,4) = ALGEBRAIC(:,3);
    ALGEBRAIC(:,136) =  CONSTANTS(:,29).*ALGEBRAIC(:,4);
    ALGEBRAIC(:,137) = ALGEBRAIC(:,68)+ALGEBRAIC(:,136);
    ALGEBRAIC(:,138) = ALGEBRAIC(:,137);
    ALGEBRAIC(:,139) =  CONSTANTS(:,25).*(exp(ALGEBRAIC(:,71)./( CONSTANTS(:,1).*CONSTANTS(:,2))) - exp(ALGEBRAIC(:,138)./( CONSTANTS(:,1).*CONSTANTS(:,2))));
    ALGEBRAIC(:,141) = ALGEBRAIC(:,139);
    ALGEBRAIC(:,144) =  - ALGEBRAIC(:,141)+ALGEBRAIC(:,78);
    ALGEBRAIC(:,147) = ALGEBRAIC(:,144);
    RATES(:,13) = ALGEBRAIC(:,147);
    ALGEBRAIC(:,69) = ALGEBRAIC(:,29);
    ALGEBRAIC(:,85) = ALGEBRAIC(:,35);
    ALGEBRAIC(:,145) =  CONSTANTS(:,30).*ALGEBRAIC(:,85);
    ALGEBRAIC(:,148) = ALGEBRAIC(:,69)+ALGEBRAIC(:,145);
    ALGEBRAIC(:,150) = ALGEBRAIC(:,148);
    ALGEBRAIC(:,80) = ALGEBRAIC(:,31);
    ALGEBRAIC(:,82) = ALGEBRAIC(:,80);
    ALGEBRAIC(:,151) =  CONSTANTS(:,26).*(exp(ALGEBRAIC(:,150)./( CONSTANTS(:,1).*CONSTANTS(:,2))) - exp(ALGEBRAIC(:,82)./( CONSTANTS(:,1).*CONSTANTS(:,2))));
    ALGEBRAIC(:,153) = ALGEBRAIC(:,151);
    ALGEBRAIC(:,101) = ALGEBRAIC(:,99);
    ALGEBRAIC(:,156) = ALGEBRAIC(:,153) - ALGEBRAIC(:,101);
    ALGEBRAIC(:,159) = ALGEBRAIC(:,156);
    RATES(:,15) = ALGEBRAIC(:,159);
    ALGEBRAIC(:,140) = ALGEBRAIC(:,139);
    ALGEBRAIC(:,142) = ALGEBRAIC(:,140);
    ALGEBRAIC(:,152) = ALGEBRAIC(:,151);
    ALGEBRAIC(:,155) = ALGEBRAIC(:,152);
    ALGEBRAIC(:,158) = ALGEBRAIC(:,142) - ALGEBRAIC(:,155);
    ALGEBRAIC(:,161) = ALGEBRAIC(:,158);
    RATES(:,14) = ALGEBRAIC(:,161);
   RATES = RATES';
end

% Calculate algebraic variables
function ALGEBRAIC = computeAlgebraic(ALGEBRAIC, CONSTANTS, STATES, VOI)
    statesSize = size(STATES);
    statesColumnCount = statesSize(2);
    if ( statesColumnCount == 1)
        STATES = STATES';
        utilOnes = 1;
    else
        statesRowCount = statesSize(1);
        utilOnes = ones(statesRowCount, 1);
    end
    ALGEBRAIC(:,15) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,6).*STATES(:,8));
    ALGEBRAIC(:,16) = ALGEBRAIC(:,15);
    ALGEBRAIC(:,49) = ALGEBRAIC(:,16);
    ALGEBRAIC(:,5) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,15).*STATES(:,3));
    ALGEBRAIC(:,6) = ALGEBRAIC(:,5);
    ALGEBRAIC(:,34) = ALGEBRAIC(:,6);
    ALGEBRAIC(:,51) = ALGEBRAIC(:,34);
    ALGEBRAIC(:,60) = ALGEBRAIC(:,49)+ALGEBRAIC(:,51);
    ALGEBRAIC(:,61) = ALGEBRAIC(:,60);
    ALGEBRAIC(:,17) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,7).*STATES(:,9));
    ALGEBRAIC(:,18) = ALGEBRAIC(:,17);
    ALGEBRAIC(:,23) = ALGEBRAIC(:,18);
    ALGEBRAIC(:,62) =  CONSTANTS(:,21).*(exp(ALGEBRAIC(:,61)./( CONSTANTS(:,1).*CONSTANTS(:,2))) - exp(ALGEBRAIC(:,23)./( CONSTANTS(:,1).*CONSTANTS(:,2))));
    ALGEBRAIC(:,64) = ALGEBRAIC(:,62);
    ALGEBRAIC(:,66) = ALGEBRAIC(:,64);
    ALGEBRAIC(:,11) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,4).*STATES(:,6));
    ALGEBRAIC(:,12) = ALGEBRAIC(:,11);
    ALGEBRAIC(:,32) = ALGEBRAIC(:,12);
    ALGEBRAIC(:,9) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,17).*STATES(:,5));
    ALGEBRAIC(:,10) = ALGEBRAIC(:,9);
    ALGEBRAIC(:,36) = ALGEBRAIC(:,32)+ALGEBRAIC(:,10);
    ALGEBRAIC(:,37) = ALGEBRAIC(:,36);
    ALGEBRAIC(:,40) = ALGEBRAIC(:,16);
    ALGEBRAIC(:,41) = ALGEBRAIC(:,40);
    ALGEBRAIC(:,42) =  CONSTANTS(:,20).*(exp(ALGEBRAIC(:,37)./( CONSTANTS(:,1).*CONSTANTS(:,2))) - exp(ALGEBRAIC(:,41)./( CONSTANTS(:,1).*CONSTANTS(:,2))));
    ALGEBRAIC(:,43) = ALGEBRAIC(:,42);
    ALGEBRAIC(:,46) = ALGEBRAIC(:,43);
    ALGEBRAIC(:,24) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,5).*STATES(:,12));
    ALGEBRAIC(:,25) = ALGEBRAIC(:,24);
    ALGEBRAIC(:,88) = ALGEBRAIC(:,25);
    ALGEBRAIC(:,89) = ALGEBRAIC(:,88);
    ALGEBRAIC(:,33) = ALGEBRAIC(:,12);
    ALGEBRAIC(:,13) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,18).*STATES(:,7));
    ALGEBRAIC(:,14) = ALGEBRAIC(:,13);
    ALGEBRAIC(:,38) = ALGEBRAIC(:,33)+ALGEBRAIC(:,14);
    ALGEBRAIC(:,39) = ALGEBRAIC(:,38);
    ALGEBRAIC(:,90) =  CONSTANTS(:,19).*(exp(ALGEBRAIC(:,89)./( CONSTANTS(:,1).*CONSTANTS(:,2))) - exp(ALGEBRAIC(:,39)./( CONSTANTS(:,1).*CONSTANTS(:,2))));
    ALGEBRAIC(:,92) = ALGEBRAIC(:,90);
    ALGEBRAIC(:,94) = ALGEBRAIC(:,92);
    ALGEBRAIC(:,97) =  - ALGEBRAIC(:,46)+ALGEBRAIC(:,94);
    ALGEBRAIC(:,100) = ALGEBRAIC(:,97);
    ALGEBRAIC(:,91) = ALGEBRAIC(:,90);
    ALGEBRAIC(:,30) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,12).*STATES(:,15));
    ALGEBRAIC(:,31) = ALGEBRAIC(:,30);
    ALGEBRAIC(:,83) = ALGEBRAIC(:,31);
    ALGEBRAIC(:,84) = ALGEBRAIC(:,83);
    ALGEBRAIC(:,35) = ALGEBRAIC(:,6);
    ALGEBRAIC(:,86) = ALGEBRAIC(:,35);
    ALGEBRAIC(:,87) = ALGEBRAIC(:,25);
    ALGEBRAIC(:,93) = ALGEBRAIC(:,86)+ALGEBRAIC(:,87);
    ALGEBRAIC(:,96) = ALGEBRAIC(:,93);
    ALGEBRAIC(:,99) =  CONSTANTS(:,27).*(exp(ALGEBRAIC(:,84)./( CONSTANTS(:,1).*CONSTANTS(:,2))) - exp(ALGEBRAIC(:,96)./( CONSTANTS(:,1).*CONSTANTS(:,2))));
    ALGEBRAIC(:,102) = ALGEBRAIC(:,99);
    ALGEBRAIC(:,104) = ALGEBRAIC(:,102);
    ALGEBRAIC(:,106) = ALGEBRAIC(:,104) - ALGEBRAIC(:,91);
    ALGEBRAIC(:,108) = ALGEBRAIC(:,106);
    ALGEBRAIC(:,44) = ALGEBRAIC(:,42);
    ALGEBRAIC(:,48) = ALGEBRAIC(:,16);
    ALGEBRAIC(:,1) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,13).*STATES(:,1));
    ALGEBRAIC(:,2) = ALGEBRAIC(:,1);
    ALGEBRAIC(:,56) = ALGEBRAIC(:,2);
    ALGEBRAIC(:,105) =  CONSTANTS(:,28).*ALGEBRAIC(:,56);
    ALGEBRAIC(:,107) = ALGEBRAIC(:,48)+ALGEBRAIC(:,105);
    ALGEBRAIC(:,109) = ALGEBRAIC(:,107);
    ALGEBRAIC(:,19) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,8).*STATES(:,10));
    ALGEBRAIC(:,20) = ALGEBRAIC(:,19);
    ALGEBRAIC(:,52) = ALGEBRAIC(:,20);
    ALGEBRAIC(:,53) = ALGEBRAIC(:,52);
    ALGEBRAIC(:,110) =  CONSTANTS(:,22).*(exp(ALGEBRAIC(:,109)./( CONSTANTS(:,1).*CONSTANTS(:,2))) - exp(ALGEBRAIC(:,53)./( CONSTANTS(:,1).*CONSTANTS(:,2))));
    ALGEBRAIC(:,112) = ALGEBRAIC(:,110);
    ALGEBRAIC(:,115) = ALGEBRAIC(:,112);
    ALGEBRAIC(:,63) = ALGEBRAIC(:,62);
    ALGEBRAIC(:,65) = ALGEBRAIC(:,63);
    ALGEBRAIC(:,118) = (ALGEBRAIC(:,44) - ALGEBRAIC(:,115)) - ALGEBRAIC(:,65);
    ALGEBRAIC(:,121) = ALGEBRAIC(:,118);
    ALGEBRAIC(:,111) = ALGEBRAIC(:,110);
    ALGEBRAIC(:,54) = ALGEBRAIC(:,20);
    ALGEBRAIC(:,55) = ALGEBRAIC(:,54);
    ALGEBRAIC(:,21) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,9).*STATES(:,11));
    ALGEBRAIC(:,22) = ALGEBRAIC(:,21);
    ALGEBRAIC(:,57) = ALGEBRAIC(:,22);
    ALGEBRAIC(:,50) = ALGEBRAIC(:,34);
    ALGEBRAIC(:,113) =  CONSTANTS(:,30).*ALGEBRAIC(:,50);
    ALGEBRAIC(:,116) = ALGEBRAIC(:,113);
    ALGEBRAIC(:,119) = ALGEBRAIC(:,57)+ALGEBRAIC(:,116);
    ALGEBRAIC(:,122) = ALGEBRAIC(:,119);
    ALGEBRAIC(:,124) =  CONSTANTS(:,23).*(exp(ALGEBRAIC(:,55)./( CONSTANTS(:,1).*CONSTANTS(:,2))) - exp(ALGEBRAIC(:,122)./( CONSTANTS(:,1).*CONSTANTS(:,2))));
    ALGEBRAIC(:,126) = ALGEBRAIC(:,124);
    ALGEBRAIC(:,128) = ALGEBRAIC(:,111) - ALGEBRAIC(:,126);
    ALGEBRAIC(:,132) = ALGEBRAIC(:,128);
    ALGEBRAIC(:,58) = ALGEBRAIC(:,22);
    ALGEBRAIC(:,59) = ALGEBRAIC(:,58);
    ALGEBRAIC(:,7) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,16).*STATES(:,4));
    ALGEBRAIC(:,8) = ALGEBRAIC(:,7);
    ALGEBRAIC(:,26) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,10).*STATES(:,13));
    ALGEBRAIC(:,27) = ALGEBRAIC(:,26);
    ALGEBRAIC(:,72) = ALGEBRAIC(:,27);
    ALGEBRAIC(:,73) = ALGEBRAIC(:,8)+ALGEBRAIC(:,72);
    ALGEBRAIC(:,74) = ALGEBRAIC(:,73);
    ALGEBRAIC(:,75) =  CONSTANTS(:,24).*(exp(ALGEBRAIC(:,59)./( CONSTANTS(:,1).*CONSTANTS(:,2))) - exp(ALGEBRAIC(:,74)./( CONSTANTS(:,1).*CONSTANTS(:,2))));
    ALGEBRAIC(:,76) = ALGEBRAIC(:,75);
    ALGEBRAIC(:,125) = ALGEBRAIC(:,124);
    ALGEBRAIC(:,129) = ALGEBRAIC(:,125);
    ALGEBRAIC(:,131) = ALGEBRAIC(:,129) - ALGEBRAIC(:,76);
    ALGEBRAIC(:,134) = ALGEBRAIC(:,131);
    ALGEBRAIC(:,77) = ALGEBRAIC(:,75);
    ALGEBRAIC(:,78) = ALGEBRAIC(:,77);
    ALGEBRAIC(:,70) = ALGEBRAIC(:,27);
    ALGEBRAIC(:,71) = ALGEBRAIC(:,70);
    ALGEBRAIC(:,28) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,11).*STATES(:,14));
    ALGEBRAIC(:,29) = ALGEBRAIC(:,28);
    ALGEBRAIC(:,68) = ALGEBRAIC(:,29);
    ALGEBRAIC(:,3) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,14).*STATES(:,2));
    ALGEBRAIC(:,4) = ALGEBRAIC(:,3);
    ALGEBRAIC(:,136) =  CONSTANTS(:,29).*ALGEBRAIC(:,4);
    ALGEBRAIC(:,137) = ALGEBRAIC(:,68)+ALGEBRAIC(:,136);
    ALGEBRAIC(:,138) = ALGEBRAIC(:,137);
    ALGEBRAIC(:,139) =  CONSTANTS(:,25).*(exp(ALGEBRAIC(:,71)./( CONSTANTS(:,1).*CONSTANTS(:,2))) - exp(ALGEBRAIC(:,138)./( CONSTANTS(:,1).*CONSTANTS(:,2))));
    ALGEBRAIC(:,141) = ALGEBRAIC(:,139);
    ALGEBRAIC(:,144) =  - ALGEBRAIC(:,141)+ALGEBRAIC(:,78);
    ALGEBRAIC(:,147) = ALGEBRAIC(:,144);
    ALGEBRAIC(:,69) = ALGEBRAIC(:,29);
    ALGEBRAIC(:,85) = ALGEBRAIC(:,35);
    ALGEBRAIC(:,145) =  CONSTANTS(:,30).*ALGEBRAIC(:,85);
    ALGEBRAIC(:,148) = ALGEBRAIC(:,69)+ALGEBRAIC(:,145);
    ALGEBRAIC(:,150) = ALGEBRAIC(:,148);
    ALGEBRAIC(:,80) = ALGEBRAIC(:,31);
    ALGEBRAIC(:,82) = ALGEBRAIC(:,80);
    ALGEBRAIC(:,151) =  CONSTANTS(:,26).*(exp(ALGEBRAIC(:,150)./( CONSTANTS(:,1).*CONSTANTS(:,2))) - exp(ALGEBRAIC(:,82)./( CONSTANTS(:,1).*CONSTANTS(:,2))));
    ALGEBRAIC(:,153) = ALGEBRAIC(:,151);
    ALGEBRAIC(:,101) = ALGEBRAIC(:,99);
    ALGEBRAIC(:,156) = ALGEBRAIC(:,153) - ALGEBRAIC(:,101);
    ALGEBRAIC(:,159) = ALGEBRAIC(:,156);
    ALGEBRAIC(:,140) = ALGEBRAIC(:,139);
    ALGEBRAIC(:,142) = ALGEBRAIC(:,140);
    ALGEBRAIC(:,152) = ALGEBRAIC(:,151);
    ALGEBRAIC(:,155) = ALGEBRAIC(:,152);
    ALGEBRAIC(:,158) = ALGEBRAIC(:,142) - ALGEBRAIC(:,155);
    ALGEBRAIC(:,161) = ALGEBRAIC(:,158);
    ALGEBRAIC(:,45) = ALGEBRAIC(:,43);
    ALGEBRAIC(:,47) =  - ALGEBRAIC(:,45);
    ALGEBRAIC(:,67) = ALGEBRAIC(:,63);
    ALGEBRAIC(:,79) = ALGEBRAIC(:,77);
    ALGEBRAIC(:,81) = ALGEBRAIC(:,79);
    ALGEBRAIC(:,95) = ALGEBRAIC(:,92);
    ALGEBRAIC(:,98) = ALGEBRAIC(:,95);
    ALGEBRAIC(:,103) = ALGEBRAIC(:,102);
    ALGEBRAIC(:,114) = ALGEBRAIC(:,112);
    ALGEBRAIC(:,117) =  CONSTANTS(:,28).*ALGEBRAIC(:,114);
    ALGEBRAIC(:,120) =  - ALGEBRAIC(:,117);
    ALGEBRAIC(:,123) = ALGEBRAIC(:,120);
    ALGEBRAIC(:,127) = ALGEBRAIC(:,125);
    ALGEBRAIC(:,130) = ALGEBRAIC(:,127);
    ALGEBRAIC(:,133) =  CONSTANTS(:,30).*ALGEBRAIC(:,130);
    ALGEBRAIC(:,135) = ALGEBRAIC(:,133) - ALGEBRAIC(:,67);
    ALGEBRAIC(:,143) = ALGEBRAIC(:,140);
    ALGEBRAIC(:,146) =  CONSTANTS(:,29).*ALGEBRAIC(:,143);
    ALGEBRAIC(:,149) = ALGEBRAIC(:,146);
    ALGEBRAIC(:,154) = ALGEBRAIC(:,152);
    ALGEBRAIC(:,157) =  CONSTANTS(:,30).*ALGEBRAIC(:,154);
    ALGEBRAIC(:,160) =  - ALGEBRAIC(:,157)+ALGEBRAIC(:,103);
    ALGEBRAIC(:,162) = ALGEBRAIC(:,135)+ALGEBRAIC(:,160);
    ALGEBRAIC(:,163) = ALGEBRAIC(:,162);
end

% Pad out or shorten strings to a set length
function strout = strpad(strin)
    req_length = 160;
    insize = size(strin,2);
    if insize > req_length
        strout = strin(1:req_length);
    else
        strout = [strin, blanks(req_length - insize)];
    end
end

