% This function simulates the bond graph of the updated Terkildsen et al.
% model. It takes the simulation duration and initial conditions as inputs.

function [VOI, STATES, ALGEBRAIC, CONSTANTS] = solveModel(tspan,init_states)
    % Initialise constants and state variables
    [INIT_STATES, CONSTANTS] = initConsts;
    if exist('init_states','var')
        INIT_STATES = init_states;
    end
    % Set numerical accuracy options for ODE solver
    options = odeset('RelTol', 1e-10, 'AbsTol', 1e-15, 'MaxStep', 1);

    % Solve model with ODE solver
    [VOI, STATES] = ode15s(@(VOI, STATES)computeRates(VOI, STATES, CONSTANTS), tspan, INIT_STATES, options);

    % Compute algebraic variables
    [RATES, ALGEBRAIC] = computeRates(VOI, STATES, CONSTANTS);
    ALGEBRAIC = computeAlgebraic(ALGEBRAIC, CONSTANTS, STATES, VOI);
end

function [LEGEND_STATES, LEGEND_ALGEBRAIC, LEGEND_VOI, LEGEND_CONSTANTS] = createLegends()
    LEGEND_STATES = ''; LEGEND_ALGEBRAIC = ''; LEGEND_VOI = ''; LEGEND_CONSTANTS = '';
    LEGEND_VOI = strpad('t in component environment (second)');
    LEGEND_CONSTANTS(:,1) = strpad('R in component environment (J_per_K_per_mol)');
    LEGEND_CONSTANTS(:,2) = strpad('T in component environment (kelvin)');
    LEGEND_CONSTANTS(:,3) = strpad('F in component environment (C_per_mol)');
    LEGEND_CONSTANTS(:,4) = strpad('C_m in component environment (fF)');
    LEGEND_CONSTANTS(:,5) = strpad('K_1 in component environment (per_fmol)');
    LEGEND_CONSTANTS(:,6) = strpad('K_10 in component environment (per_fmol)');
    LEGEND_CONSTANTS(:,7) = strpad('K_11 in component environment (per_fmol)');
    LEGEND_CONSTANTS(:,8) = strpad('K_12 in component environment (per_fmol)');
    LEGEND_CONSTANTS(:,9) = strpad('K_13 in component environment (per_fmol)');
    LEGEND_CONSTANTS(:,10) = strpad('K_14 in component environment (per_fmol)');
    LEGEND_CONSTANTS(:,11) = strpad('K_15 in component environment (per_fmol)');
    LEGEND_CONSTANTS(:,12) = strpad('K_2 in component environment (per_fmol)');
    LEGEND_CONSTANTS(:,13) = strpad('K_3 in component environment (per_fmol)');
    LEGEND_CONSTANTS(:,14) = strpad('K_4 in component environment (per_fmol)');
    LEGEND_CONSTANTS(:,15) = strpad('K_5 in component environment (per_fmol)');
    LEGEND_CONSTANTS(:,16) = strpad('K_6 in component environment (per_fmol)');
    LEGEND_CONSTANTS(:,17) = strpad('K_7 in component environment (per_fmol)');
    LEGEND_CONSTANTS(:,18) = strpad('K_8 in component environment (per_fmol)');
    LEGEND_CONSTANTS(:,19) = strpad('K_9 in component environment (per_fmol)');
    LEGEND_CONSTANTS(:,20) = strpad('K_H in component environment (per_fmol)');
    LEGEND_CONSTANTS(:,21) = strpad('K_Ke in component environment (per_fmol)');
    LEGEND_CONSTANTS(:,22) = strpad('K_Ki in component environment (per_fmol)');
    LEGEND_CONSTANTS(:,23) = strpad('K_MgADP in component environment (per_fmol)');
    LEGEND_CONSTANTS(:,24) = strpad('K_MgATP in component environment (per_fmol)');
    LEGEND_CONSTANTS(:,25) = strpad('K_Nae in component environment (per_fmol)');
    LEGEND_CONSTANTS(:,26) = strpad('K_Nai in component environment (per_fmol)');
    LEGEND_CONSTANTS(:,27) = strpad('K_P in component environment (per_fmol)');
    LEGEND_CONSTANTS(:,28) = strpad('kappa_1 in component environment (fmol_per_sec)');
    LEGEND_CONSTANTS(:,29) = strpad('kappa_10 in component environment (fmol_per_sec)');
    LEGEND_CONSTANTS(:,30) = strpad('kappa_11 in component environment (fmol_per_sec)');
    LEGEND_CONSTANTS(:,31) = strpad('kappa_12 in component environment (fmol_per_sec)');
    LEGEND_CONSTANTS(:,32) = strpad('kappa_13 in component environment (fmol_per_sec)');
    LEGEND_CONSTANTS(:,33) = strpad('kappa_14 in component environment (fmol_per_sec)');
    LEGEND_CONSTANTS(:,34) = strpad('kappa_15 in component environment (fmol_per_sec)');
    LEGEND_CONSTANTS(:,35) = strpad('kappa_2 in component environment (fmol_per_sec)');
    LEGEND_CONSTANTS(:,36) = strpad('kappa_3 in component environment (fmol_per_sec)');
    LEGEND_CONSTANTS(:,37) = strpad('kappa_4 in component environment (fmol_per_sec)');
    LEGEND_CONSTANTS(:,38) = strpad('kappa_5 in component environment (fmol_per_sec)');
    LEGEND_CONSTANTS(:,39) = strpad('kappa_6 in component environment (fmol_per_sec)');
    LEGEND_CONSTANTS(:,40) = strpad('kappa_7 in component environment (fmol_per_sec)');
    LEGEND_CONSTANTS(:,41) = strpad('kappa_8 in component environment (fmol_per_sec)');
    LEGEND_CONSTANTS(:,42) = strpad('kappa_9 in component environment (fmol_per_sec)');
    LEGEND_CONSTANTS(:,43) = strpad('z_5 in component environment (dimensionless)');
    LEGEND_CONSTANTS(:,44) = strpad('z_8 in component environment (dimensionless)');
    LEGEND_CONSTANTS(:,47) = strpad('zF_5 in component environment (C_per_mol)');
    LEGEND_CONSTANTS(:,48) = strpad('zF_8 in component environment (C_per_mol)');
    LEGEND_STATES(:,1) = strpad('x_H in component environment (fmol)');
    LEGEND_STATES(:,2) = strpad('x_Ke in component environment (fmol)');
    LEGEND_STATES(:,3) = strpad('x_Ki in component environment (fmol)');
    LEGEND_STATES(:,4) = strpad('x_MgADP in component environment (fmol)');
    LEGEND_STATES(:,5) = strpad('x_MgATP in component environment (fmol)');
    LEGEND_STATES(:,6) = strpad('x_Nae in component environment (fmol)');
    LEGEND_STATES(:,7) = strpad('x_Nai in component environment (fmol)');
    LEGEND_STATES(:,8) = strpad('x_P7 in component environment (fmol)');
    LEGEND_STATES(:,9) = strpad('x_P_i in component environment (fmol)');
    LEGEND_STATES(:,10) = strpad('q_mem in component environment (fC)');
    LEGEND_STATES(:,11) = strpad('x_P14 in component environment (fmol)');
    LEGEND_STATES(:,12) = strpad('x_P15 in component environment (fmol)');
    LEGEND_STATES(:,13) = strpad('x_P1 in component environment (fmol)');
    LEGEND_STATES(:,14) = strpad('x_P2 in component environment (fmol)');
    LEGEND_STATES(:,15) = strpad('x_P3 in component environment (fmol)');
    LEGEND_STATES(:,16) = strpad('x_P4 in component environment (fmol)');
    LEGEND_STATES(:,17) = strpad('x_P5 in component environment (fmol)');
    LEGEND_STATES(:,18) = strpad('x_P6 in component environment (fmol)');
    LEGEND_STATES(:,19) = strpad('x_P10 in component environment (fmol)');
    LEGEND_STATES(:,20) = strpad('x_P11 in component environment (fmol)');
    LEGEND_STATES(:,21) = strpad('x_P12 in component environment (fmol)');
    LEGEND_STATES(:,22) = strpad('x_P13 in component environment (fmol)');
    LEGEND_STATES(:,23) = strpad('x_P8 in component environment (fmol)');
    LEGEND_STATES(:,24) = strpad('x_P9 in component environment (fmol)');
    LEGEND_CONSTANTS(:,45) = strpad('W_i in component environment (pL)');
    LEGEND_CONSTANTS(:,46) = strpad('W_e in component environment (pL)');
    LEGEND_ALGEBRAIC(:,1) = strpad('c_H in component environment (mM)');
    LEGEND_ALGEBRAIC(:,2) = strpad('c_Ke in component environment (mM)');
    LEGEND_ALGEBRAIC(:,3) = strpad('c_Ki in component environment (mM)');
    LEGEND_ALGEBRAIC(:,4) = strpad('c_MgADP in component environment (mM)');
    LEGEND_ALGEBRAIC(:,5) = strpad('c_MgATP in component environment (mM)');
    LEGEND_ALGEBRAIC(:,6) = strpad('c_Nae in component environment (mM)');
    LEGEND_ALGEBRAIC(:,7) = strpad('c_Nai in component environment (mM)');
    LEGEND_ALGEBRAIC(:,8) = strpad('c_P_i in component environment (mM)');
    LEGEND_ALGEBRAIC(:,110) = strpad('e_1 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,14) = strpad('e_2 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,57) = strpad('e_3 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,85) = strpad('e_4 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,22) = strpad('e_5 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,63) = strpad('e_6 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,58) = strpad('e_7 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,24) = strpad('e_8 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,16) = strpad('e_9 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,59) = strpad('e_10 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,12) = strpad('e_11 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,183) = strpad('e_12 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,67) = strpad('e_13 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,10) = strpad('e_14 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,26) = strpad('e_15 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,18) = strpad('e_16 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,71) = strpad('e_17 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,66) = strpad('e_18 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,20) = strpad('e_19 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,160) = strpad('e_20 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,61) = strpad('e_21 in component environment (volt)');
    LEGEND_ALGEBRAIC(:,62) = strpad('e_22 in component environment (volt)');
    LEGEND_ALGEBRAIC(:,28) = strpad('e_23 in component environment (volt)');
    LEGEND_ALGEBRAIC(:,30) = strpad('e_24 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,65) = strpad('e_25 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,74) = strpad('e_26 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,69) = strpad('e_27 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,32) = strpad('e_28 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,73) = strpad('e_29 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,34) = strpad('e_30 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,96) = strpad('e_31 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,117) = strpad('e_32 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,94) = strpad('e_33 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,38) = strpad('e_34 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,82) = strpad('e_35 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,126) = strpad('e_36 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,90) = strpad('e_37 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,98) = strpad('e_38 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,36) = strpad('e_39 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,92) = strpad('e_40 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,95) = strpad('e_41 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,125) = strpad('e_42 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,99) = strpad('e_43 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,40) = strpad('e_44 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,101) = strpad('e_45 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,144) = strpad('e_46 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,106) = strpad('e_47 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,102) = strpad('e_48 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,104) = strpad('e_49 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,103) = strpad('e_50 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,42) = strpad('e_51 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,108) = strpad('e_52 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,155) = strpad('e_53 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,105) = strpad('e_54 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,218) = strpad('e_55 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,109) = strpad('e_56 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,44) = strpad('e_57 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,216) = strpad('e_58 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,54) = strpad('e_59 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,158) = strpad('e_60 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,245) = strpad('e_61 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,240) = strpad('e_62 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,165) = strpad('e_63 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,169) = strpad('e_64 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,56) = strpad('e_65 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,170) = strpad('e_66 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,174) = strpad('e_67 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,167) = strpad('e_68 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,172) = strpad('e_69 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,46) = strpad('e_70 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,175) = strpad('e_71 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,205) = strpad('e_72 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,173) = strpad('e_73 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,177) = strpad('e_74 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,48) = strpad('e_75 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,188) = strpad('e_76 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,230) = strpad('e_77 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,181) = strpad('e_78 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,52) = strpad('e_79 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,178) = strpad('e_80 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,214) = strpad('e_81 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,192) = strpad('e_82 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,50) = strpad('e_83 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,179) = strpad('e_84 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,180) = strpad('e_85 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,198) = strpad('e_86 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,113) = strpad('f_1 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,140) = strpad('f_2 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,241) = strpad('f_3 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,88) = strpad('f_4 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,246) = strpad('f_5 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,114) = strpad('f_6 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,115) = strpad('f_7 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,166) = strpad('f_8 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,116) = strpad('f_9 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,164) = strpad('f_10 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,254) = strpad('f_11 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,186) = strpad('f_12 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,187) = strpad('f_13 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,189) = strpad('f_14 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,190) = strpad('f_15 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,81) = strpad('f_16 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,89) = strpad('f_17 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,191) = strpad('f_18 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,266) = strpad('f_19 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,163) = strpad('f_20 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,267) = strpad('f_21 in component environment (fA)');
    LEGEND_ALGEBRAIC(:,235) = strpad('f_22 in component environment (fA)');
    LEGEND_ALGEBRAIC(:,270) = strpad('f_23 in component environment (fA)');
    LEGEND_ALGEBRAIC(:,193) = strpad('f_24 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,80) = strpad('f_25 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,77) = strpad('f_26 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,78) = strpad('f_27 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,91) = strpad('f_28 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,79) = strpad('f_29 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,136) = strpad('f_30 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,121) = strpad('f_31 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,122) = strpad('f_32 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,123) = strpad('f_33 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,141) = strpad('f_34 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,133) = strpad('f_35 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,132) = strpad('f_36 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,135) = strpad('f_37 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,138) = strpad('f_38 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,142) = strpad('f_39 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,124) = strpad('f_40 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,137) = strpad('f_41 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,131) = strpad('f_42 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,134) = strpad('f_43 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,154) = strpad('f_44 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,152) = strpad('f_45 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,150) = strpad('f_46 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,151) = strpad('f_47 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,139) = strpad('f_48 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,153) = strpad('f_49 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,236) = strpad('f_50 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,237) = strpad('f_51 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,232) = strpad('f_52 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,227) = strpad('f_53 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,231) = strpad('f_54 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,225) = strpad('f_55 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,233) = strpad('f_56 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,238) = strpad('f_57 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,229) = strpad('f_58 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,264) = strpad('f_59 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,258) = strpad('f_60 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,259) = strpad('f_61 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,263) = strpad('f_62 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,262) = strpad('f_63 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,265) = strpad('f_64 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,268) = strpad('f_65 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,201) = strpad('f_66 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,203) = strpad('f_67 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,213) = strpad('f_68 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,204) = strpad('f_69 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,212) = strpad('f_70 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,209) = strpad('f_71 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,208) = strpad('f_72 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,210) = strpad('f_73 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,211) = strpad('f_74 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,228) = strpad('f_75 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,251) = strpad('f_76 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,248) = strpad('f_77 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,247) = strpad('f_78 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,250) = strpad('f_79 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,224) = strpad('f_80 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,221) = strpad('f_81 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,223) = strpad('f_82 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,255) = strpad('f_83 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,226) = strpad('f_84 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,249) = strpad('f_85 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,202) = strpad('f_86 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,9) = strpad('mu_H in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,194) = strpad('v_H in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,11) = strpad('mu_Ke in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,260) = strpad('v_Ke in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,13) = strpad('mu_Ki in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,145) = strpad('v_Ki in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,15) = strpad('mu_MgADP in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,118) = strpad('v_MgADP in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,17) = strpad('mu_MgATP in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,83) = strpad('v_MgATP in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,19) = strpad('mu_Nae in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,271) = strpad('v_Nae in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,21) = strpad('mu_Nai in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,252) = strpad('v_Nai in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,23) = strpad('mu_P7 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,168) = strpad('v_P7 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,25) = strpad('mu_P_i in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,195) = strpad('v_P_i in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,27) = strpad('V_mem in component environment (volt)');
    LEGEND_ALGEBRAIC(:,273) = strpad('I_mem in component environment (fA)');
    LEGEND_ALGEBRAIC(:,184) = strpad('Af_R13 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,68) = strpad('Ar_R13 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,185) = strpad('v_R13 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,72) = strpad('Af_R15 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,86) = strpad('Ar_R15 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,87) = strpad('v_R15 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,111) = strpad('Af_R6 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,64) = strpad('Ar_R6 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,112) = strpad('v_R6 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,60) = strpad('Af_R7 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,161) = strpad('Ar_R7 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,162) = strpad('v_R7 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,29) = strpad('mu_P14 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,197) = strpad('v_P14 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,31) = strpad('mu_P15 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,93) = strpad('v_P15 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,75) = strpad('Af_R14 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,70) = strpad('Ar_R14 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,76) = strpad('v_R14 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,33) = strpad('mu_P1 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,143) = strpad('v_P1 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,35) = strpad('mu_P2 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,146) = strpad('v_P2 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,37) = strpad('mu_P3 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,147) = strpad('v_P3 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,39) = strpad('mu_P4 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,156) = strpad('v_P4 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,41) = strpad('mu_P5 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,242) = strpad('v_P5 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,43) = strpad('mu_P6 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,243) = strpad('v_P6 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,84) = strpad('Af_R1 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,128) = strpad('Ar_R1 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,130) = strpad('v_R1 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,97) = strpad('Af_R2 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,119) = strpad('Ar_R2 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,120) = strpad('v_R2 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,127) = strpad('Af_R3 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,100) = strpad('Ar_R3 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,129) = strpad('v_R3 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,148) = strpad('Af_R4 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,107) = strpad('Ar_R4 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,149) = strpad('v_R4 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,157) = strpad('Af_R5 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,220) = strpad('Ar_R5 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,222) = strpad('v_R5 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,45) = strpad('mu_P10 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,215) = strpad('v_P10 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,47) = strpad('mu_P11 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,234) = strpad('v_P11 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,49) = strpad('mu_P12 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,261) = strpad('v_P12 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,51) = strpad('mu_P13 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,256) = strpad('v_P13 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,53) = strpad('mu_P8 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,269) = strpad('v_P8 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,55) = strpad('mu_P9 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,272) = strpad('v_P9 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,176) = strpad('Af_R10 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,206) = strpad('Ar_R10 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,207) = strpad('v_R10 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,217) = strpad('Af_R11 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,196) = strpad('Ar_R11 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,219) = strpad('v_R11 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,239) = strpad('Af_R12 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,182) = strpad('Ar_R12 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,244) = strpad('v_R12 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,159) = strpad('Af_R8 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,253) = strpad('Ar_R8 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,257) = strpad('v_R8 in component environment (fmol_per_sec)');
    LEGEND_ALGEBRAIC(:,171) = strpad('Af_R9 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,199) = strpad('Ar_R9 in component environment (J_per_mol)');
    LEGEND_ALGEBRAIC(:,200) = strpad('v_R9 in component environment (fmol_per_sec)');
    LEGEND_RATES(:,1) = strpad('d/dt x_H in component environment (fmol)');
    LEGEND_RATES(:,2) = strpad('d/dt x_Ke in component environment (fmol)');
    LEGEND_RATES(:,3) = strpad('d/dt x_Ki in component environment (fmol)');
    LEGEND_RATES(:,4) = strpad('d/dt x_MgADP in component environment (fmol)');
    LEGEND_RATES(:,5) = strpad('d/dt x_MgATP in component environment (fmol)');
    LEGEND_RATES(:,6) = strpad('d/dt x_Nae in component environment (fmol)');
    LEGEND_RATES(:,7) = strpad('d/dt x_Nai in component environment (fmol)');
    LEGEND_RATES(:,8) = strpad('d/dt x_P7 in component environment (fmol)');
    LEGEND_RATES(:,9) = strpad('d/dt x_P_i in component environment (fmol)');
    LEGEND_RATES(:,10) = strpad('d/dt q_mem in component environment (fC)');
    LEGEND_RATES(:,11) = strpad('d/dt x_P14 in component environment (fmol)');
    LEGEND_RATES(:,12) = strpad('d/dt x_P15 in component environment (fmol)');
    LEGEND_RATES(:,13) = strpad('d/dt x_P1 in component environment (fmol)');
    LEGEND_RATES(:,14) = strpad('d/dt x_P2 in component environment (fmol)');
    LEGEND_RATES(:,15) = strpad('d/dt x_P3 in component environment (fmol)');
    LEGEND_RATES(:,16) = strpad('d/dt x_P4 in component environment (fmol)');
    LEGEND_RATES(:,17) = strpad('d/dt x_P5 in component environment (fmol)');
    LEGEND_RATES(:,18) = strpad('d/dt x_P6 in component environment (fmol)');
    LEGEND_RATES(:,19) = strpad('d/dt x_P10 in component environment (fmol)');
    LEGEND_RATES(:,20) = strpad('d/dt x_P11 in component environment (fmol)');
    LEGEND_RATES(:,21) = strpad('d/dt x_P12 in component environment (fmol)');
    LEGEND_RATES(:,22) = strpad('d/dt x_P13 in component environment (fmol)');
    LEGEND_RATES(:,23) = strpad('d/dt x_P8 in component environment (fmol)');
    LEGEND_RATES(:,24) = strpad('d/dt x_P9 in component environment (fmol)');
    LEGEND_STATES  = LEGEND_STATES';
    LEGEND_ALGEBRAIC = LEGEND_ALGEBRAIC';
    LEGEND_RATES = LEGEND_RATES';
    LEGEND_CONSTANTS = LEGEND_CONSTANTS';
end

function [STATES, CONSTANTS] = initConsts()
    VOI = 0; CONSTANTS = []; STATES = []; ALGEBRAIC = [];
    CONSTANTS(:,1) = 8.314;
    CONSTANTS(:,2) = 310;
    CONSTANTS(:,3) = 96485;
    CONSTANTS(:,4) = 153400;
    CONSTANTS(:,5) = 101619537.2009;
    CONSTANTS(:,6) = 20459.5509;
    CONSTANTS(:,7) = 121.4456;
    CONSTANTS(:,8) = 3.1436;
    CONSTANTS(:,9) = 0.32549;
    CONSTANTS(:,10) = 156.3283;
    CONSTANTS(:,11) = 1977546.8577;
    CONSTANTS(:,12) = 63209.8623;
    CONSTANTS(:,13) = 157.2724;
    CONSTANTS(:,14) = 14.0748;
    CONSTANTS(:,15) = 5.0384;
    CONSTANTS(:,16) = 92.6964;
    CONSTANTS(:,17) = 4854.5924;
    CONSTANTS(:,18) = 15260.9786;
    CONSTANTS(:,19) = 13787022.8009;
    CONSTANTS(:,20) = 0.04565;
    CONSTANTS(:,21) = 0.009236;
    CONSTANTS(:,22) = 0.0012595;
    CONSTANTS(:,23) = 7.976e-05;
    CONSTANTS(:,24) = 2.3715;
    CONSTANTS(:,25) = 0.0061242;
    CONSTANTS(:,26) = 0.00083514;
    CONSTANTS(:,27) = 0.04565;
    CONSTANTS(:,28) = 330.5462;
    CONSTANTS(:,29) = 259461.6507;
    CONSTANTS(:,30) = 172042.3334;
    CONSTANTS(:,31) = 6646440.3909;
    CONSTANTS(:,32) = 597.4136;
    CONSTANTS(:,33) = 70.9823;
    CONSTANTS(:,34) = 0.015489;
    CONSTANTS(:,35) = 132850.9145;
    CONSTANTS(:,36) = 200356.0223;
    CONSTANTS(:,37) = 2238785.3951;
    CONSTANTS(:,38) = 10787.9052;
    CONSTANTS(:,39) = 15.3533;
    CONSTANTS(:,40) = 2.3822;
    CONSTANTS(:,41) = 2.2855;
    CONSTANTS(:,42) = 1540.1349;
    CONSTANTS(:,43) = -0.054951;
    CONSTANTS(:,44) = -0.94505;
    STATES(:,1) = 0.0015128;
    STATES(:,2) = 27.9828;
    STATES(:,3) = 0.038;
    STATES(:,4) = 0.038;
    STATES(:,5) = 380;
    STATES(:,6) = 777.3;
    STATES(:,7) = 1900;
    STATES(:,8) = 0.534480003169437;
    STATES(:,9) = 0.038;
    STATES(:,10) = -18408;
    STATES(:,11) = 0.00626267917464576;
    STATES(:,12) = 0.000446049400255497;
    STATES(:,13) = 4.06727417747628e-10;
    STATES(:,14) = 1.84495437200145e-09;
    STATES(:,15) = 0.00183103460985586;
    STATES(:,16) = 0.0324604777886237;
    STATES(:,17) = 0.143884405731667;
    STATES(:,18) = 0.00968432811008776;
    STATES(:,19) = 7.9739635710452e-05;
    STATES(:,20) = 0.00282186679243115;
    STATES(:,21) = 0.0281498805518011;
    STATES(:,22) = 0.0702590453291214;
    STATES(:,23) = 0.169644923453282;
    STATES(:,24) = 5.63940980127252e-07;
    CONSTANTS(:,45) = 38.0;
    CONSTANTS(:,46) = 5.182;
    CONSTANTS(:,47) =  CONSTANTS(:,43).*CONSTANTS(:,3);
    CONSTANTS(:,48) = 0.000000;
    CONSTANTS(:,49) = 0.000000;
    CONSTANTS(:,50) = 0.000000;
    CONSTANTS(:,51) = 0.000000;
    CONSTANTS(:,52) = 0.000000;
    CONSTANTS(:,53) = 0.000000;
    CONSTANTS(:,54) = 0.000000;
    CONSTANTS(:,55) = 0.000000;
    CONSTANTS(:,56) =  0.000000.*CONSTANTS(:,4);
    CONSTANTS(:,57) =  CONSTANTS(:,44).*CONSTANTS(:,3);
    if (isempty(STATES)), warning('Initial values for states not set');, end
end

function [RATES, ALGEBRAIC] = computeRates(VOI, STATES, CONSTANTS)
    global algebraicVariableCount;
    statesSize = size(STATES);
    statesColumnCount = statesSize(2);
    if ( statesColumnCount == 1)
        STATES = STATES';
        ALGEBRAIC = zeros(1, algebraicVariableCount);
        utilOnes = 1;
    else
        statesRowCount = statesSize(1);
        ALGEBRAIC = zeros(statesRowCount, algebraicVariableCount);
        RATES = zeros(statesRowCount, statesColumnCount);
        utilOnes = ones(statesRowCount, 1);
    end
    RATES(:,1) = CONSTANTS(:,48);
    RATES(:,2) = CONSTANTS(:,49);
    RATES(:,3) = CONSTANTS(:,50);
    RATES(:,4) = CONSTANTS(:,51);
    RATES(:,5) = CONSTANTS(:,52);
    RATES(:,6) = CONSTANTS(:,53);
    RATES(:,7) = CONSTANTS(:,54);
    RATES(:,9) = CONSTANTS(:,55);
    RATES(:,10) = CONSTANTS(:,56);
    ALGEBRAIC(:,31) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,11).*STATES(:,12));
    ALGEBRAIC(:,32) = ALGEBRAIC(:,31);
    ALGEBRAIC(:,71) = ALGEBRAIC(:,32);
    ALGEBRAIC(:,72) = ALGEBRAIC(:,71);
    ALGEBRAIC(:,33) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,5).*STATES(:,13));
    ALGEBRAIC(:,34) = ALGEBRAIC(:,33);
    ALGEBRAIC(:,85) = ALGEBRAIC(:,34);
    ALGEBRAIC(:,86) = ALGEBRAIC(:,85);
    ALGEBRAIC(:,87) =  CONSTANTS(:,34).*(exp(ALGEBRAIC(:,72)./( CONSTANTS(:,1).*CONSTANTS(:,2))) - exp(ALGEBRAIC(:,86)./( CONSTANTS(:,1).*CONSTANTS(:,2))));
    ALGEBRAIC(:,89) = ALGEBRAIC(:,87);
    ALGEBRAIC(:,29) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,10).*STATES(:,11));
    ALGEBRAIC(:,30) = ALGEBRAIC(:,29);
    ALGEBRAIC(:,65) = ALGEBRAIC(:,30);
    ALGEBRAIC(:,17) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,24).*STATES(:,5));
    ALGEBRAIC(:,18) = ALGEBRAIC(:,17);
    ALGEBRAIC(:,73) = ALGEBRAIC(:,18);
    ALGEBRAIC(:,74) = ALGEBRAIC(:,65)+ALGEBRAIC(:,73);
    ALGEBRAIC(:,75) = ALGEBRAIC(:,74);
    ALGEBRAIC(:,69) = ALGEBRAIC(:,32);
    ALGEBRAIC(:,70) = ALGEBRAIC(:,69);
    ALGEBRAIC(:,76) =  CONSTANTS(:,33).*(exp(ALGEBRAIC(:,75)./( CONSTANTS(:,1).*CONSTANTS(:,2))) - exp(ALGEBRAIC(:,70)./( CONSTANTS(:,1).*CONSTANTS(:,2))));
    ALGEBRAIC(:,78) = ALGEBRAIC(:,76);
    ALGEBRAIC(:,91) = ALGEBRAIC(:,78) - ALGEBRAIC(:,89);
    ALGEBRAIC(:,93) = ALGEBRAIC(:,91);
    RATES(:,12) = ALGEBRAIC(:,93);
    ALGEBRAIC(:,88) = ALGEBRAIC(:,87);
    ALGEBRAIC(:,82) = ALGEBRAIC(:,34);
    ALGEBRAIC(:,84) = ALGEBRAIC(:,82);
    ALGEBRAIC(:,13) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,22).*STATES(:,3));
    ALGEBRAIC(:,14) = ALGEBRAIC(:,13);
    ALGEBRAIC(:,90) = ALGEBRAIC(:,14);
    ALGEBRAIC(:,35) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,12).*STATES(:,14));
    ALGEBRAIC(:,36) = ALGEBRAIC(:,35);
    ALGEBRAIC(:,98) = ALGEBRAIC(:,36);
    ALGEBRAIC(:,126) = ALGEBRAIC(:,90)+ALGEBRAIC(:,98);
    ALGEBRAIC(:,128) = ALGEBRAIC(:,126);
    ALGEBRAIC(:,130) =  CONSTANTS(:,28).*(exp(ALGEBRAIC(:,84)./( CONSTANTS(:,1).*CONSTANTS(:,2))) - exp(ALGEBRAIC(:,128)./( CONSTANTS(:,1).*CONSTANTS(:,2))));
    ALGEBRAIC(:,133) = ALGEBRAIC(:,130);
    ALGEBRAIC(:,136) =  - ALGEBRAIC(:,133)+ALGEBRAIC(:,88);
    ALGEBRAIC(:,143) = ALGEBRAIC(:,136);
    RATES(:,13) = ALGEBRAIC(:,143);
    ALGEBRAIC(:,96) = ALGEBRAIC(:,36);
    ALGEBRAIC(:,97) = ALGEBRAIC(:,96);
    ALGEBRAIC(:,37) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,13).*STATES(:,15));
    ALGEBRAIC(:,38) = ALGEBRAIC(:,37);
    ALGEBRAIC(:,94) = ALGEBRAIC(:,38);
    ALGEBRAIC(:,92) = ALGEBRAIC(:,14);
    ALGEBRAIC(:,117) = ALGEBRAIC(:,94)+ALGEBRAIC(:,92);
    ALGEBRAIC(:,119) = ALGEBRAIC(:,117);
    ALGEBRAIC(:,120) =  CONSTANTS(:,35).*(exp(ALGEBRAIC(:,97)./( CONSTANTS(:,1).*CONSTANTS(:,2))) - exp(ALGEBRAIC(:,119)./( CONSTANTS(:,1).*CONSTANTS(:,2))));
    ALGEBRAIC(:,121) = ALGEBRAIC(:,120);
    ALGEBRAIC(:,132) = ALGEBRAIC(:,130);
    ALGEBRAIC(:,138) = ALGEBRAIC(:,132);
    ALGEBRAIC(:,142) =  - ALGEBRAIC(:,121)+ALGEBRAIC(:,138);
    ALGEBRAIC(:,146) = ALGEBRAIC(:,142);
    RATES(:,14) = ALGEBRAIC(:,146);
    ALGEBRAIC(:,122) = ALGEBRAIC(:,120);
    ALGEBRAIC(:,123) = ALGEBRAIC(:,122);
    ALGEBRAIC(:,95) = ALGEBRAIC(:,38);
    ALGEBRAIC(:,21) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,26).*STATES(:,7));
    ALGEBRAIC(:,22) = ALGEBRAIC(:,21);
    ALGEBRAIC(:,57) = ALGEBRAIC(:,22);
    ALGEBRAIC(:,102) = ALGEBRAIC(:,57);
    ALGEBRAIC(:,125) = ALGEBRAIC(:,95)+ALGEBRAIC(:,102);
    ALGEBRAIC(:,127) = ALGEBRAIC(:,125);
    ALGEBRAIC(:,39) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,14).*STATES(:,16));
    ALGEBRAIC(:,40) = ALGEBRAIC(:,39);
    ALGEBRAIC(:,99) = ALGEBRAIC(:,40);
    ALGEBRAIC(:,100) = ALGEBRAIC(:,99);
    ALGEBRAIC(:,129) =  CONSTANTS(:,36).*(exp(ALGEBRAIC(:,127)./( CONSTANTS(:,1).*CONSTANTS(:,2))) - exp(ALGEBRAIC(:,100)./( CONSTANTS(:,1).*CONSTANTS(:,2))));
    ALGEBRAIC(:,131) = ALGEBRAIC(:,129);
    ALGEBRAIC(:,137) = ALGEBRAIC(:,131);
    ALGEBRAIC(:,141) = ALGEBRAIC(:,123) - ALGEBRAIC(:,137);
    ALGEBRAIC(:,147) = ALGEBRAIC(:,141);
    RATES(:,15) = ALGEBRAIC(:,147);
    ALGEBRAIC(:,134) = ALGEBRAIC(:,129);
    ALGEBRAIC(:,101) = ALGEBRAIC(:,40);
    ALGEBRAIC(:,103) = ALGEBRAIC(:,57);
    ALGEBRAIC(:,104) = ALGEBRAIC(:,103);
    ALGEBRAIC(:,144) = ALGEBRAIC(:,101)+ALGEBRAIC(:,104);
    ALGEBRAIC(:,148) = ALGEBRAIC(:,144);
    ALGEBRAIC(:,41) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,15).*STATES(:,17));
    ALGEBRAIC(:,42) = ALGEBRAIC(:,41);
    ALGEBRAIC(:,106) = ALGEBRAIC(:,42);
    ALGEBRAIC(:,107) = ALGEBRAIC(:,106);
    ALGEBRAIC(:,149) =  CONSTANTS(:,37).*(exp(ALGEBRAIC(:,148)./( CONSTANTS(:,1).*CONSTANTS(:,2))) - exp(ALGEBRAIC(:,107)./( CONSTANTS(:,1).*CONSTANTS(:,2))));
    ALGEBRAIC(:,150) = ALGEBRAIC(:,149);
    ALGEBRAIC(:,152) = ALGEBRAIC(:,150);
    ALGEBRAIC(:,154) = ALGEBRAIC(:,134) - ALGEBRAIC(:,152);
    ALGEBRAIC(:,156) = ALGEBRAIC(:,154);
    RATES(:,16) = ALGEBRAIC(:,156);
    ALGEBRAIC(:,43) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,16).*STATES(:,18));
    ALGEBRAIC(:,44) = ALGEBRAIC(:,43);
    ALGEBRAIC(:,110) = ALGEBRAIC(:,44);
    ALGEBRAIC(:,111) = ALGEBRAIC(:,110);
    ALGEBRAIC(:,23) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,17).*STATES(:,8));
    ALGEBRAIC(:,24) = ALGEBRAIC(:,23);
    ALGEBRAIC(:,58) = ALGEBRAIC(:,24);
    ALGEBRAIC(:,15) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,23).*STATES(:,4));
    ALGEBRAIC(:,16) = ALGEBRAIC(:,15);
    ALGEBRAIC(:,63) = ALGEBRAIC(:,58)+ALGEBRAIC(:,16);
    ALGEBRAIC(:,64) = ALGEBRAIC(:,63);
    ALGEBRAIC(:,112) =  CONSTANTS(:,39).*(exp(ALGEBRAIC(:,111)./( CONSTANTS(:,1).*CONSTANTS(:,2))) - exp(ALGEBRAIC(:,64)./( CONSTANTS(:,1).*CONSTANTS(:,2))));
    ALGEBRAIC(:,114) = ALGEBRAIC(:,112);
    ALGEBRAIC(:,115) = ALGEBRAIC(:,114);
    ALGEBRAIC(:,59) = ALGEBRAIC(:,24);
    ALGEBRAIC(:,60) = ALGEBRAIC(:,59);
    ALGEBRAIC(:,53) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,18).*STATES(:,23));
    ALGEBRAIC(:,54) = ALGEBRAIC(:,53);
    ALGEBRAIC(:,160) = ALGEBRAIC(:,54);
    ALGEBRAIC(:,161) = ALGEBRAIC(:,160);
    ALGEBRAIC(:,162) =  CONSTANTS(:,40).*(exp(ALGEBRAIC(:,60)./( CONSTANTS(:,1).*CONSTANTS(:,2))) - exp(ALGEBRAIC(:,161)./( CONSTANTS(:,1).*CONSTANTS(:,2))));
    ALGEBRAIC(:,164) = ALGEBRAIC(:,162);
    ALGEBRAIC(:,166) = ALGEBRAIC(:,115) - ALGEBRAIC(:,164);
    ALGEBRAIC(:,168) = ALGEBRAIC(:,166);
    RATES(:,8) = ALGEBRAIC(:,168);
    ALGEBRAIC(:,51) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,9).*STATES(:,22));
    ALGEBRAIC(:,52) = ALGEBRAIC(:,51);
    ALGEBRAIC(:,183) = ALGEBRAIC(:,52);
    ALGEBRAIC(:,184) = ALGEBRAIC(:,183);
    ALGEBRAIC(:,9) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,20).*STATES(:,1));
    ALGEBRAIC(:,10) = ALGEBRAIC(:,9);
    ALGEBRAIC(:,25) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,27).*STATES(:,9));
    ALGEBRAIC(:,26) = ALGEBRAIC(:,25);
    ALGEBRAIC(:,66) = ALGEBRAIC(:,30);
    ALGEBRAIC(:,67) = ALGEBRAIC(:,10)+ALGEBRAIC(:,26)+ALGEBRAIC(:,66);
    ALGEBRAIC(:,68) = ALGEBRAIC(:,67);
    ALGEBRAIC(:,185) =  CONSTANTS(:,32).*(exp(ALGEBRAIC(:,184)./( CONSTANTS(:,1).*CONSTANTS(:,2))) - exp(ALGEBRAIC(:,68)./( CONSTANTS(:,1).*CONSTANTS(:,2))));
    ALGEBRAIC(:,187) = ALGEBRAIC(:,185);
    ALGEBRAIC(:,191) = ALGEBRAIC(:,187);
    ALGEBRAIC(:,77) = ALGEBRAIC(:,76);
    ALGEBRAIC(:,80) = ALGEBRAIC(:,77);
    ALGEBRAIC(:,193) =  - ALGEBRAIC(:,80)+ALGEBRAIC(:,191);
    ALGEBRAIC(:,197) = ALGEBRAIC(:,193);
    RATES(:,11) = ALGEBRAIC(:,197);
    ALGEBRAIC(:,55) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,19).*STATES(:,24));
    ALGEBRAIC(:,56) = ALGEBRAIC(:,55);
    ALGEBRAIC(:,170) = ALGEBRAIC(:,56);
    ALGEBRAIC(:,171) = ALGEBRAIC(:,170);
    ALGEBRAIC(:,45) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,6).*STATES(:,19));
    ALGEBRAIC(:,46) = ALGEBRAIC(:,45);
    ALGEBRAIC(:,174) = ALGEBRAIC(:,46);
    ALGEBRAIC(:,19) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,25).*STATES(:,6));
    ALGEBRAIC(:,20) = ALGEBRAIC(:,19);
    ALGEBRAIC(:,167) = ALGEBRAIC(:,20);
    ALGEBRAIC(:,172) = ALGEBRAIC(:,167);
    ALGEBRAIC(:,198) = ALGEBRAIC(:,174)+ALGEBRAIC(:,172);
    ALGEBRAIC(:,199) = ALGEBRAIC(:,198);
    ALGEBRAIC(:,200) =  CONSTANTS(:,42).*(exp(ALGEBRAIC(:,171)./( CONSTANTS(:,1).*CONSTANTS(:,2))) - exp(ALGEBRAIC(:,199)./( CONSTANTS(:,1).*CONSTANTS(:,2))));
    ALGEBRAIC(:,202) = ALGEBRAIC(:,200);
    ALGEBRAIC(:,203) = ALGEBRAIC(:,202);
    ALGEBRAIC(:,175) = ALGEBRAIC(:,46);
    ALGEBRAIC(:,176) = ALGEBRAIC(:,175);
    ALGEBRAIC(:,173) = ALGEBRAIC(:,167);
    ALGEBRAIC(:,47) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,7).*STATES(:,20));
    ALGEBRAIC(:,48) = ALGEBRAIC(:,47);
    ALGEBRAIC(:,177) = ALGEBRAIC(:,48);
    ALGEBRAIC(:,205) = ALGEBRAIC(:,173)+ALGEBRAIC(:,177);
    ALGEBRAIC(:,206) = ALGEBRAIC(:,205);
    ALGEBRAIC(:,207) =  CONSTANTS(:,29).*(exp(ALGEBRAIC(:,176)./( CONSTANTS(:,1).*CONSTANTS(:,2))) - exp(ALGEBRAIC(:,206)./( CONSTANTS(:,1).*CONSTANTS(:,2))));
    ALGEBRAIC(:,209) = ALGEBRAIC(:,207);
    ALGEBRAIC(:,212) = ALGEBRAIC(:,203) - ALGEBRAIC(:,209);
    ALGEBRAIC(:,215) = ALGEBRAIC(:,212);
    RATES(:,19) = ALGEBRAIC(:,215);
    ALGEBRAIC(:,208) = ALGEBRAIC(:,207);
    ALGEBRAIC(:,211) = ALGEBRAIC(:,208);
    ALGEBRAIC(:,178) = ALGEBRAIC(:,48);
    ALGEBRAIC(:,11) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,21).*STATES(:,2));
    ALGEBRAIC(:,12) = ALGEBRAIC(:,11);
    ALGEBRAIC(:,179) = ALGEBRAIC(:,12);
    ALGEBRAIC(:,214) = ALGEBRAIC(:,178)+ALGEBRAIC(:,179);
    ALGEBRAIC(:,217) = ALGEBRAIC(:,214);
    ALGEBRAIC(:,49) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,8).*STATES(:,21));
    ALGEBRAIC(:,50) = ALGEBRAIC(:,49);
    ALGEBRAIC(:,192) = ALGEBRAIC(:,50);
    ALGEBRAIC(:,196) = ALGEBRAIC(:,192);
    ALGEBRAIC(:,219) =  CONSTANTS(:,30).*(exp(ALGEBRAIC(:,217)./( CONSTANTS(:,1).*CONSTANTS(:,2))) - exp(ALGEBRAIC(:,196)./( CONSTANTS(:,1).*CONSTANTS(:,2))));
    ALGEBRAIC(:,221) = ALGEBRAIC(:,219);
    ALGEBRAIC(:,224) = ALGEBRAIC(:,221);
    ALGEBRAIC(:,228) = ALGEBRAIC(:,211) - ALGEBRAIC(:,224);
    ALGEBRAIC(:,234) = ALGEBRAIC(:,228);
    RATES(:,20) = ALGEBRAIC(:,234);
    ALGEBRAIC(:,151) = ALGEBRAIC(:,149);
    ALGEBRAIC(:,108) = ALGEBRAIC(:,42);
    ALGEBRAIC(:,105) = ALGEBRAIC(:,103);
    ALGEBRAIC(:,155) = ALGEBRAIC(:,108)+ALGEBRAIC(:,105);
    ALGEBRAIC(:,157) = ALGEBRAIC(:,155);
    ALGEBRAIC(:,109) = ALGEBRAIC(:,44);
    ALGEBRAIC(:,27) = STATES(:,10)./CONSTANTS(:,4);
    ALGEBRAIC(:,28) = ALGEBRAIC(:,27);
    ALGEBRAIC(:,62) = ALGEBRAIC(:,28);
    ALGEBRAIC(:,216) =  CONSTANTS(:,47).*ALGEBRAIC(:,62);
    ALGEBRAIC(:,218) = ALGEBRAIC(:,109)+ALGEBRAIC(:,216);
    ALGEBRAIC(:,220) = ALGEBRAIC(:,218);
    ALGEBRAIC(:,222) =  CONSTANTS(:,38).*(exp(ALGEBRAIC(:,157)./( CONSTANTS(:,1).*CONSTANTS(:,2))) - exp(ALGEBRAIC(:,220)./( CONSTANTS(:,1).*CONSTANTS(:,2))));
    ALGEBRAIC(:,227) = ALGEBRAIC(:,222);
    ALGEBRAIC(:,232) = ALGEBRAIC(:,227);
    ALGEBRAIC(:,237) = ALGEBRAIC(:,151) - ALGEBRAIC(:,232);
    ALGEBRAIC(:,242) = ALGEBRAIC(:,237);
    RATES(:,17) = ALGEBRAIC(:,242);
    ALGEBRAIC(:,113) = ALGEBRAIC(:,112);
    ALGEBRAIC(:,225) = ALGEBRAIC(:,222);
    ALGEBRAIC(:,233) = ALGEBRAIC(:,225);
    ALGEBRAIC(:,238) = ALGEBRAIC(:,233) - ALGEBRAIC(:,113);
    ALGEBRAIC(:,243) = ALGEBRAIC(:,238);
    RATES(:,18) = ALGEBRAIC(:,243);
    ALGEBRAIC(:,186) = ALGEBRAIC(:,185);
    ALGEBRAIC(:,188) = ALGEBRAIC(:,50);
    ALGEBRAIC(:,180) = ALGEBRAIC(:,12);
    ALGEBRAIC(:,230) = ALGEBRAIC(:,188)+ALGEBRAIC(:,180);
    ALGEBRAIC(:,239) = ALGEBRAIC(:,230);
    ALGEBRAIC(:,181) = ALGEBRAIC(:,52);
    ALGEBRAIC(:,182) = ALGEBRAIC(:,181);
    ALGEBRAIC(:,244) =  CONSTANTS(:,31).*(exp(ALGEBRAIC(:,239)./( CONSTANTS(:,1).*CONSTANTS(:,2))) - exp(ALGEBRAIC(:,182)./( CONSTANTS(:,1).*CONSTANTS(:,2))));
    ALGEBRAIC(:,247) = ALGEBRAIC(:,244);
    ALGEBRAIC(:,250) = ALGEBRAIC(:,247) - ALGEBRAIC(:,186);
    ALGEBRAIC(:,256) = ALGEBRAIC(:,250);
    RATES(:,22) = ALGEBRAIC(:,256);
    ALGEBRAIC(:,248) = ALGEBRAIC(:,244);
    ALGEBRAIC(:,251) = ALGEBRAIC(:,248);
    ALGEBRAIC(:,223) = ALGEBRAIC(:,219);
    ALGEBRAIC(:,255) =  - ALGEBRAIC(:,251)+ALGEBRAIC(:,223);
    ALGEBRAIC(:,261) = ALGEBRAIC(:,255);
    RATES(:,21) = ALGEBRAIC(:,261);
    ALGEBRAIC(:,163) = ALGEBRAIC(:,162);
    ALGEBRAIC(:,158) = ALGEBRAIC(:,54);
    ALGEBRAIC(:,159) = ALGEBRAIC(:,158);
    ALGEBRAIC(:,61) = ALGEBRAIC(:,28);
    ALGEBRAIC(:,240) =  CONSTANTS(:,57).*ALGEBRAIC(:,61);
    ALGEBRAIC(:,165) = ALGEBRAIC(:,20);
    ALGEBRAIC(:,169) = ALGEBRAIC(:,56);
    ALGEBRAIC(:,245) = ALGEBRAIC(:,240)+ALGEBRAIC(:,165)+ALGEBRAIC(:,169);
    ALGEBRAIC(:,253) = ALGEBRAIC(:,245);
    ALGEBRAIC(:,257) =  CONSTANTS(:,41).*(exp(ALGEBRAIC(:,159)./( CONSTANTS(:,1).*CONSTANTS(:,2))) - exp(ALGEBRAIC(:,253)./( CONSTANTS(:,1).*CONSTANTS(:,2))));
    ALGEBRAIC(:,258) = ALGEBRAIC(:,257);
    ALGEBRAIC(:,264) =  - ALGEBRAIC(:,258)+ALGEBRAIC(:,163);
    ALGEBRAIC(:,269) = ALGEBRAIC(:,264);
    RATES(:,23) = ALGEBRAIC(:,269);
    ALGEBRAIC(:,259) = ALGEBRAIC(:,257);
    ALGEBRAIC(:,265) = ALGEBRAIC(:,259);
    ALGEBRAIC(:,201) = ALGEBRAIC(:,200);
    ALGEBRAIC(:,268) = ALGEBRAIC(:,265) - ALGEBRAIC(:,201);
    ALGEBRAIC(:,272) = ALGEBRAIC(:,268);
    RATES(:,24) = ALGEBRAIC(:,272);
   RATES = RATES';
end

% Calculate algebraic variables
function ALGEBRAIC = computeAlgebraic(ALGEBRAIC, CONSTANTS, STATES, VOI)
    statesSize = size(STATES);
    statesColumnCount = statesSize(2);
    if ( statesColumnCount == 1)
        STATES = STATES';
        utilOnes = 1;
    else
        statesRowCount = statesSize(1);
        utilOnes = ones(statesRowCount, 1);
    end
    ALGEBRAIC(:,31) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,11).*STATES(:,12));
    ALGEBRAIC(:,32) = ALGEBRAIC(:,31);
    ALGEBRAIC(:,71) = ALGEBRAIC(:,32);
    ALGEBRAIC(:,72) = ALGEBRAIC(:,71);
    ALGEBRAIC(:,33) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,5).*STATES(:,13));
    ALGEBRAIC(:,34) = ALGEBRAIC(:,33);
    ALGEBRAIC(:,85) = ALGEBRAIC(:,34);
    ALGEBRAIC(:,86) = ALGEBRAIC(:,85);
    ALGEBRAIC(:,87) =  CONSTANTS(:,34).*(exp(ALGEBRAIC(:,72)./( CONSTANTS(:,1).*CONSTANTS(:,2))) - exp(ALGEBRAIC(:,86)./( CONSTANTS(:,1).*CONSTANTS(:,2))));
    ALGEBRAIC(:,89) = ALGEBRAIC(:,87);
    ALGEBRAIC(:,29) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,10).*STATES(:,11));
    ALGEBRAIC(:,30) = ALGEBRAIC(:,29);
    ALGEBRAIC(:,65) = ALGEBRAIC(:,30);
    ALGEBRAIC(:,17) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,24).*STATES(:,5));
    ALGEBRAIC(:,18) = ALGEBRAIC(:,17);
    ALGEBRAIC(:,73) = ALGEBRAIC(:,18);
    ALGEBRAIC(:,74) = ALGEBRAIC(:,65)+ALGEBRAIC(:,73);
    ALGEBRAIC(:,75) = ALGEBRAIC(:,74);
    ALGEBRAIC(:,69) = ALGEBRAIC(:,32);
    ALGEBRAIC(:,70) = ALGEBRAIC(:,69);
    ALGEBRAIC(:,76) =  CONSTANTS(:,33).*(exp(ALGEBRAIC(:,75)./( CONSTANTS(:,1).*CONSTANTS(:,2))) - exp(ALGEBRAIC(:,70)./( CONSTANTS(:,1).*CONSTANTS(:,2))));
    ALGEBRAIC(:,78) = ALGEBRAIC(:,76);
    ALGEBRAIC(:,91) = ALGEBRAIC(:,78) - ALGEBRAIC(:,89);
    ALGEBRAIC(:,93) = ALGEBRAIC(:,91);
    ALGEBRAIC(:,88) = ALGEBRAIC(:,87);
    ALGEBRAIC(:,82) = ALGEBRAIC(:,34);
    ALGEBRAIC(:,84) = ALGEBRAIC(:,82);
    ALGEBRAIC(:,13) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,22).*STATES(:,3));
    ALGEBRAIC(:,14) = ALGEBRAIC(:,13);
    ALGEBRAIC(:,90) = ALGEBRAIC(:,14);
    ALGEBRAIC(:,35) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,12).*STATES(:,14));
    ALGEBRAIC(:,36) = ALGEBRAIC(:,35);
    ALGEBRAIC(:,98) = ALGEBRAIC(:,36);
    ALGEBRAIC(:,126) = ALGEBRAIC(:,90)+ALGEBRAIC(:,98);
    ALGEBRAIC(:,128) = ALGEBRAIC(:,126);
    ALGEBRAIC(:,130) =  CONSTANTS(:,28).*(exp(ALGEBRAIC(:,84)./( CONSTANTS(:,1).*CONSTANTS(:,2))) - exp(ALGEBRAIC(:,128)./( CONSTANTS(:,1).*CONSTANTS(:,2))));
    ALGEBRAIC(:,133) = ALGEBRAIC(:,130);
    ALGEBRAIC(:,136) =  - ALGEBRAIC(:,133)+ALGEBRAIC(:,88);
    ALGEBRAIC(:,143) = ALGEBRAIC(:,136);
    ALGEBRAIC(:,96) = ALGEBRAIC(:,36);
    ALGEBRAIC(:,97) = ALGEBRAIC(:,96);
    ALGEBRAIC(:,37) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,13).*STATES(:,15));
    ALGEBRAIC(:,38) = ALGEBRAIC(:,37);
    ALGEBRAIC(:,94) = ALGEBRAIC(:,38);
    ALGEBRAIC(:,92) = ALGEBRAIC(:,14);
    ALGEBRAIC(:,117) = ALGEBRAIC(:,94)+ALGEBRAIC(:,92);
    ALGEBRAIC(:,119) = ALGEBRAIC(:,117);
    ALGEBRAIC(:,120) =  CONSTANTS(:,35).*(exp(ALGEBRAIC(:,97)./( CONSTANTS(:,1).*CONSTANTS(:,2))) - exp(ALGEBRAIC(:,119)./( CONSTANTS(:,1).*CONSTANTS(:,2))));
    ALGEBRAIC(:,121) = ALGEBRAIC(:,120);
    ALGEBRAIC(:,132) = ALGEBRAIC(:,130);
    ALGEBRAIC(:,138) = ALGEBRAIC(:,132);
    ALGEBRAIC(:,142) =  - ALGEBRAIC(:,121)+ALGEBRAIC(:,138);
    ALGEBRAIC(:,146) = ALGEBRAIC(:,142);
    ALGEBRAIC(:,122) = ALGEBRAIC(:,120);
    ALGEBRAIC(:,123) = ALGEBRAIC(:,122);
    ALGEBRAIC(:,95) = ALGEBRAIC(:,38);
    ALGEBRAIC(:,21) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,26).*STATES(:,7));
    ALGEBRAIC(:,22) = ALGEBRAIC(:,21);
    ALGEBRAIC(:,57) = ALGEBRAIC(:,22);
    ALGEBRAIC(:,102) = ALGEBRAIC(:,57);
    ALGEBRAIC(:,125) = ALGEBRAIC(:,95)+ALGEBRAIC(:,102);
    ALGEBRAIC(:,127) = ALGEBRAIC(:,125);
    ALGEBRAIC(:,39) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,14).*STATES(:,16));
    ALGEBRAIC(:,40) = ALGEBRAIC(:,39);
    ALGEBRAIC(:,99) = ALGEBRAIC(:,40);
    ALGEBRAIC(:,100) = ALGEBRAIC(:,99);
    ALGEBRAIC(:,129) =  CONSTANTS(:,36).*(exp(ALGEBRAIC(:,127)./( CONSTANTS(:,1).*CONSTANTS(:,2))) - exp(ALGEBRAIC(:,100)./( CONSTANTS(:,1).*CONSTANTS(:,2))));
    ALGEBRAIC(:,131) = ALGEBRAIC(:,129);
    ALGEBRAIC(:,137) = ALGEBRAIC(:,131);
    ALGEBRAIC(:,141) = ALGEBRAIC(:,123) - ALGEBRAIC(:,137);
    ALGEBRAIC(:,147) = ALGEBRAIC(:,141);
    ALGEBRAIC(:,134) = ALGEBRAIC(:,129);
    ALGEBRAIC(:,101) = ALGEBRAIC(:,40);
    ALGEBRAIC(:,103) = ALGEBRAIC(:,57);
    ALGEBRAIC(:,104) = ALGEBRAIC(:,103);
    ALGEBRAIC(:,144) = ALGEBRAIC(:,101)+ALGEBRAIC(:,104);
    ALGEBRAIC(:,148) = ALGEBRAIC(:,144);
    ALGEBRAIC(:,41) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,15).*STATES(:,17));
    ALGEBRAIC(:,42) = ALGEBRAIC(:,41);
    ALGEBRAIC(:,106) = ALGEBRAIC(:,42);
    ALGEBRAIC(:,107) = ALGEBRAIC(:,106);
    ALGEBRAIC(:,149) =  CONSTANTS(:,37).*(exp(ALGEBRAIC(:,148)./( CONSTANTS(:,1).*CONSTANTS(:,2))) - exp(ALGEBRAIC(:,107)./( CONSTANTS(:,1).*CONSTANTS(:,2))));
    ALGEBRAIC(:,150) = ALGEBRAIC(:,149);
    ALGEBRAIC(:,152) = ALGEBRAIC(:,150);
    ALGEBRAIC(:,154) = ALGEBRAIC(:,134) - ALGEBRAIC(:,152);
    ALGEBRAIC(:,156) = ALGEBRAIC(:,154);
    ALGEBRAIC(:,43) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,16).*STATES(:,18));
    ALGEBRAIC(:,44) = ALGEBRAIC(:,43);
    ALGEBRAIC(:,110) = ALGEBRAIC(:,44);
    ALGEBRAIC(:,111) = ALGEBRAIC(:,110);
    ALGEBRAIC(:,23) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,17).*STATES(:,8));
    ALGEBRAIC(:,24) = ALGEBRAIC(:,23);
    ALGEBRAIC(:,58) = ALGEBRAIC(:,24);
    ALGEBRAIC(:,15) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,23).*STATES(:,4));
    ALGEBRAIC(:,16) = ALGEBRAIC(:,15);
    ALGEBRAIC(:,63) = ALGEBRAIC(:,58)+ALGEBRAIC(:,16);
    ALGEBRAIC(:,64) = ALGEBRAIC(:,63);
    ALGEBRAIC(:,112) =  CONSTANTS(:,39).*(exp(ALGEBRAIC(:,111)./( CONSTANTS(:,1).*CONSTANTS(:,2))) - exp(ALGEBRAIC(:,64)./( CONSTANTS(:,1).*CONSTANTS(:,2))));
    ALGEBRAIC(:,114) = ALGEBRAIC(:,112);
    ALGEBRAIC(:,115) = ALGEBRAIC(:,114);
    ALGEBRAIC(:,59) = ALGEBRAIC(:,24);
    ALGEBRAIC(:,60) = ALGEBRAIC(:,59);
    ALGEBRAIC(:,53) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,18).*STATES(:,23));
    ALGEBRAIC(:,54) = ALGEBRAIC(:,53);
    ALGEBRAIC(:,160) = ALGEBRAIC(:,54);
    ALGEBRAIC(:,161) = ALGEBRAIC(:,160);
    ALGEBRAIC(:,162) =  CONSTANTS(:,40).*(exp(ALGEBRAIC(:,60)./( CONSTANTS(:,1).*CONSTANTS(:,2))) - exp(ALGEBRAIC(:,161)./( CONSTANTS(:,1).*CONSTANTS(:,2))));
    ALGEBRAIC(:,164) = ALGEBRAIC(:,162);
    ALGEBRAIC(:,166) = ALGEBRAIC(:,115) - ALGEBRAIC(:,164);
    ALGEBRAIC(:,168) = ALGEBRAIC(:,166);
    ALGEBRAIC(:,51) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,9).*STATES(:,22));
    ALGEBRAIC(:,52) = ALGEBRAIC(:,51);
    ALGEBRAIC(:,183) = ALGEBRAIC(:,52);
    ALGEBRAIC(:,184) = ALGEBRAIC(:,183);
    ALGEBRAIC(:,9) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,20).*STATES(:,1));
    ALGEBRAIC(:,10) = ALGEBRAIC(:,9);
    ALGEBRAIC(:,25) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,27).*STATES(:,9));
    ALGEBRAIC(:,26) = ALGEBRAIC(:,25);
    ALGEBRAIC(:,66) = ALGEBRAIC(:,30);
    ALGEBRAIC(:,67) = ALGEBRAIC(:,10)+ALGEBRAIC(:,26)+ALGEBRAIC(:,66);
    ALGEBRAIC(:,68) = ALGEBRAIC(:,67);
    ALGEBRAIC(:,185) =  CONSTANTS(:,32).*(exp(ALGEBRAIC(:,184)./( CONSTANTS(:,1).*CONSTANTS(:,2))) - exp(ALGEBRAIC(:,68)./( CONSTANTS(:,1).*CONSTANTS(:,2))));
    ALGEBRAIC(:,187) = ALGEBRAIC(:,185);
    ALGEBRAIC(:,191) = ALGEBRAIC(:,187);
    ALGEBRAIC(:,77) = ALGEBRAIC(:,76);
    ALGEBRAIC(:,80) = ALGEBRAIC(:,77);
    ALGEBRAIC(:,193) =  - ALGEBRAIC(:,80)+ALGEBRAIC(:,191);
    ALGEBRAIC(:,197) = ALGEBRAIC(:,193);
    ALGEBRAIC(:,55) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,19).*STATES(:,24));
    ALGEBRAIC(:,56) = ALGEBRAIC(:,55);
    ALGEBRAIC(:,170) = ALGEBRAIC(:,56);
    ALGEBRAIC(:,171) = ALGEBRAIC(:,170);
    ALGEBRAIC(:,45) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,6).*STATES(:,19));
    ALGEBRAIC(:,46) = ALGEBRAIC(:,45);
    ALGEBRAIC(:,174) = ALGEBRAIC(:,46);
    ALGEBRAIC(:,19) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,25).*STATES(:,6));
    ALGEBRAIC(:,20) = ALGEBRAIC(:,19);
    ALGEBRAIC(:,167) = ALGEBRAIC(:,20);
    ALGEBRAIC(:,172) = ALGEBRAIC(:,167);
    ALGEBRAIC(:,198) = ALGEBRAIC(:,174)+ALGEBRAIC(:,172);
    ALGEBRAIC(:,199) = ALGEBRAIC(:,198);
    ALGEBRAIC(:,200) =  CONSTANTS(:,42).*(exp(ALGEBRAIC(:,171)./( CONSTANTS(:,1).*CONSTANTS(:,2))) - exp(ALGEBRAIC(:,199)./( CONSTANTS(:,1).*CONSTANTS(:,2))));
    ALGEBRAIC(:,202) = ALGEBRAIC(:,200);
    ALGEBRAIC(:,203) = ALGEBRAIC(:,202);
    ALGEBRAIC(:,175) = ALGEBRAIC(:,46);
    ALGEBRAIC(:,176) = ALGEBRAIC(:,175);
    ALGEBRAIC(:,173) = ALGEBRAIC(:,167);
    ALGEBRAIC(:,47) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,7).*STATES(:,20));
    ALGEBRAIC(:,48) = ALGEBRAIC(:,47);
    ALGEBRAIC(:,177) = ALGEBRAIC(:,48);
    ALGEBRAIC(:,205) = ALGEBRAIC(:,173)+ALGEBRAIC(:,177);
    ALGEBRAIC(:,206) = ALGEBRAIC(:,205);
    ALGEBRAIC(:,207) =  CONSTANTS(:,29).*(exp(ALGEBRAIC(:,176)./( CONSTANTS(:,1).*CONSTANTS(:,2))) - exp(ALGEBRAIC(:,206)./( CONSTANTS(:,1).*CONSTANTS(:,2))));
    ALGEBRAIC(:,209) = ALGEBRAIC(:,207);
    ALGEBRAIC(:,212) = ALGEBRAIC(:,203) - ALGEBRAIC(:,209);
    ALGEBRAIC(:,215) = ALGEBRAIC(:,212);
    ALGEBRAIC(:,208) = ALGEBRAIC(:,207);
    ALGEBRAIC(:,211) = ALGEBRAIC(:,208);
    ALGEBRAIC(:,178) = ALGEBRAIC(:,48);
    ALGEBRAIC(:,11) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,21).*STATES(:,2));
    ALGEBRAIC(:,12) = ALGEBRAIC(:,11);
    ALGEBRAIC(:,179) = ALGEBRAIC(:,12);
    ALGEBRAIC(:,214) = ALGEBRAIC(:,178)+ALGEBRAIC(:,179);
    ALGEBRAIC(:,217) = ALGEBRAIC(:,214);
    ALGEBRAIC(:,49) =  CONSTANTS(:,1).*CONSTANTS(:,2).*log( CONSTANTS(:,8).*STATES(:,21));
    ALGEBRAIC(:,50) = ALGEBRAIC(:,49);
    ALGEBRAIC(:,192) = ALGEBRAIC(:,50);
    ALGEBRAIC(:,196) = ALGEBRAIC(:,192);
    ALGEBRAIC(:,219) =  CONSTANTS(:,30).*(exp(ALGEBRAIC(:,217)./( CONSTANTS(:,1).*CONSTANTS(:,2))) - exp(ALGEBRAIC(:,196)./( CONSTANTS(:,1).*CONSTANTS(:,2))));
    ALGEBRAIC(:,221) = ALGEBRAIC(:,219);
    ALGEBRAIC(:,224) = ALGEBRAIC(:,221);
    ALGEBRAIC(:,228) = ALGEBRAIC(:,211) - ALGEBRAIC(:,224);
    ALGEBRAIC(:,234) = ALGEBRAIC(:,228);
    ALGEBRAIC(:,151) = ALGEBRAIC(:,149);
    ALGEBRAIC(:,108) = ALGEBRAIC(:,42);
    ALGEBRAIC(:,105) = ALGEBRAIC(:,103);
    ALGEBRAIC(:,155) = ALGEBRAIC(:,108)+ALGEBRAIC(:,105);
    ALGEBRAIC(:,157) = ALGEBRAIC(:,155);
    ALGEBRAIC(:,109) = ALGEBRAIC(:,44);
    ALGEBRAIC(:,27) = STATES(:,10)./CONSTANTS(:,4);
    ALGEBRAIC(:,28) = ALGEBRAIC(:,27);
    ALGEBRAIC(:,62) = ALGEBRAIC(:,28);
    ALGEBRAIC(:,216) =  CONSTANTS(:,47).*ALGEBRAIC(:,62);
    ALGEBRAIC(:,218) = ALGEBRAIC(:,109)+ALGEBRAIC(:,216);
    ALGEBRAIC(:,220) = ALGEBRAIC(:,218);
    ALGEBRAIC(:,222) =  CONSTANTS(:,38).*(exp(ALGEBRAIC(:,157)./( CONSTANTS(:,1).*CONSTANTS(:,2))) - exp(ALGEBRAIC(:,220)./( CONSTANTS(:,1).*CONSTANTS(:,2))));
    ALGEBRAIC(:,227) = ALGEBRAIC(:,222);
    ALGEBRAIC(:,232) = ALGEBRAIC(:,227);
    ALGEBRAIC(:,237) = ALGEBRAIC(:,151) - ALGEBRAIC(:,232);
    ALGEBRAIC(:,242) = ALGEBRAIC(:,237);
    ALGEBRAIC(:,113) = ALGEBRAIC(:,112);
    ALGEBRAIC(:,225) = ALGEBRAIC(:,222);
    ALGEBRAIC(:,233) = ALGEBRAIC(:,225);
    ALGEBRAIC(:,238) = ALGEBRAIC(:,233) - ALGEBRAIC(:,113);
    ALGEBRAIC(:,243) = ALGEBRAIC(:,238);
    ALGEBRAIC(:,186) = ALGEBRAIC(:,185);
    ALGEBRAIC(:,188) = ALGEBRAIC(:,50);
    ALGEBRAIC(:,180) = ALGEBRAIC(:,12);
    ALGEBRAIC(:,230) = ALGEBRAIC(:,188)+ALGEBRAIC(:,180);
    ALGEBRAIC(:,239) = ALGEBRAIC(:,230);
    ALGEBRAIC(:,181) = ALGEBRAIC(:,52);
    ALGEBRAIC(:,182) = ALGEBRAIC(:,181);
    ALGEBRAIC(:,244) =  CONSTANTS(:,31).*(exp(ALGEBRAIC(:,239)./( CONSTANTS(:,1).*CONSTANTS(:,2))) - exp(ALGEBRAIC(:,182)./( CONSTANTS(:,1).*CONSTANTS(:,2))));
    ALGEBRAIC(:,247) = ALGEBRAIC(:,244);
    ALGEBRAIC(:,250) = ALGEBRAIC(:,247) - ALGEBRAIC(:,186);
    ALGEBRAIC(:,256) = ALGEBRAIC(:,250);
    ALGEBRAIC(:,248) = ALGEBRAIC(:,244);
    ALGEBRAIC(:,251) = ALGEBRAIC(:,248);
    ALGEBRAIC(:,223) = ALGEBRAIC(:,219);
    ALGEBRAIC(:,255) =  - ALGEBRAIC(:,251)+ALGEBRAIC(:,223);
    ALGEBRAIC(:,261) = ALGEBRAIC(:,255);
    ALGEBRAIC(:,163) = ALGEBRAIC(:,162);
    ALGEBRAIC(:,158) = ALGEBRAIC(:,54);
    ALGEBRAIC(:,159) = ALGEBRAIC(:,158);
    ALGEBRAIC(:,61) = ALGEBRAIC(:,28);
    ALGEBRAIC(:,240) =  CONSTANTS(:,57).*ALGEBRAIC(:,61);
    ALGEBRAIC(:,165) = ALGEBRAIC(:,20);
    ALGEBRAIC(:,169) = ALGEBRAIC(:,56);
    ALGEBRAIC(:,245) = ALGEBRAIC(:,240)+ALGEBRAIC(:,165)+ALGEBRAIC(:,169);
    ALGEBRAIC(:,253) = ALGEBRAIC(:,245);
    ALGEBRAIC(:,257) =  CONSTANTS(:,41).*(exp(ALGEBRAIC(:,159)./( CONSTANTS(:,1).*CONSTANTS(:,2))) - exp(ALGEBRAIC(:,253)./( CONSTANTS(:,1).*CONSTANTS(:,2))));
    ALGEBRAIC(:,258) = ALGEBRAIC(:,257);
    ALGEBRAIC(:,264) =  - ALGEBRAIC(:,258)+ALGEBRAIC(:,163);
    ALGEBRAIC(:,269) = ALGEBRAIC(:,264);
    ALGEBRAIC(:,259) = ALGEBRAIC(:,257);
    ALGEBRAIC(:,265) = ALGEBRAIC(:,259);
    ALGEBRAIC(:,201) = ALGEBRAIC(:,200);
    ALGEBRAIC(:,268) = ALGEBRAIC(:,265) - ALGEBRAIC(:,201);
    ALGEBRAIC(:,272) = ALGEBRAIC(:,268);
    ALGEBRAIC(:,1) = STATES(:,1)./CONSTANTS(:,45);
    ALGEBRAIC(:,2) = STATES(:,2)./CONSTANTS(:,46);
    ALGEBRAIC(:,3) = STATES(:,3)./CONSTANTS(:,45);
    ALGEBRAIC(:,4) = STATES(:,4)./CONSTANTS(:,45);
    ALGEBRAIC(:,5) = STATES(:,5)./CONSTANTS(:,45);
    ALGEBRAIC(:,6) = STATES(:,6)./CONSTANTS(:,46);
    ALGEBRAIC(:,7) = STATES(:,7)./CONSTANTS(:,45);
    ALGEBRAIC(:,8) = STATES(:,9)./CONSTANTS(:,45);
    ALGEBRAIC(:,79) = ALGEBRAIC(:,77);
    ALGEBRAIC(:,81) =  - ALGEBRAIC(:,79);
    ALGEBRAIC(:,83) = ALGEBRAIC(:,81);
    ALGEBRAIC(:,116) = ALGEBRAIC(:,114);
    ALGEBRAIC(:,118) = ALGEBRAIC(:,116);
    ALGEBRAIC(:,124) = ALGEBRAIC(:,122);
    ALGEBRAIC(:,135) = ALGEBRAIC(:,132);
    ALGEBRAIC(:,139) = ALGEBRAIC(:,131);
    ALGEBRAIC(:,140) = ALGEBRAIC(:,135)+ALGEBRAIC(:,124);
    ALGEBRAIC(:,145) = ALGEBRAIC(:,140);
    ALGEBRAIC(:,153) = ALGEBRAIC(:,150);
    ALGEBRAIC(:,189) = ALGEBRAIC(:,187);
    ALGEBRAIC(:,190) = ALGEBRAIC(:,187);
    ALGEBRAIC(:,194) = ALGEBRAIC(:,189);
    ALGEBRAIC(:,195) = ALGEBRAIC(:,190);
    ALGEBRAIC(:,204) = ALGEBRAIC(:,202);
    ALGEBRAIC(:,210) = ALGEBRAIC(:,208);
    ALGEBRAIC(:,213) = ALGEBRAIC(:,204)+ALGEBRAIC(:,210);
    ALGEBRAIC(:,226) = ALGEBRAIC(:,221);
    ALGEBRAIC(:,229) = ALGEBRAIC(:,225);
    ALGEBRAIC(:,231) = ALGEBRAIC(:,227);
    ALGEBRAIC(:,235) =  CONSTANTS(:,47).*ALGEBRAIC(:,229);
    ALGEBRAIC(:,236) = ALGEBRAIC(:,153)+ALGEBRAIC(:,231);
    ALGEBRAIC(:,241) =  - ALGEBRAIC(:,139) - ALGEBRAIC(:,236);
    ALGEBRAIC(:,246) = ALGEBRAIC(:,241);
    ALGEBRAIC(:,249) = ALGEBRAIC(:,248);
    ALGEBRAIC(:,252) = ALGEBRAIC(:,246);
    ALGEBRAIC(:,254) =  - ALGEBRAIC(:,226) - ALGEBRAIC(:,249);
    ALGEBRAIC(:,260) = ALGEBRAIC(:,254);
    ALGEBRAIC(:,262) = ALGEBRAIC(:,259);
    ALGEBRAIC(:,263) = ALGEBRAIC(:,259);
    ALGEBRAIC(:,266) = ALGEBRAIC(:,262)+ALGEBRAIC(:,213);
    ALGEBRAIC(:,267) =  CONSTANTS(:,57).*ALGEBRAIC(:,263);
    ALGEBRAIC(:,270) = ALGEBRAIC(:,267)+ALGEBRAIC(:,235);
    ALGEBRAIC(:,271) = ALGEBRAIC(:,266);
    ALGEBRAIC(:,273) = ALGEBRAIC(:,270);
end

% Pad out or shorten strings to a set length
function strout = strpad(strin)
    req_length = 160;
    insize = size(strin,2);
    if insize > req_length
        strout = strin(1:req_length);
    else
        strout = [strin, blanks(req_length - insize)];
    end
end

