# jsPsych.data

The jsPsych.data module contains functions for interacting with the data generated by jsPsych plugins.

---

## jsPsych.data.addProperties

```javascript
jsPsych.data.addProperties(properties)
```

### Parameters

Parameter | Type | Description
----------|------|------------
properties | object | Object of key: value pairs to add to the data.

### Return value

Returns nothing.

### Description

This method appends a set of properties to every trial in the data object, including trials that have already occurred and trials that have yet to occur. You can use this to record things like the subject ID or condition assignment.

### Examples

#### Assigning a subject ID and condition code

```javascript
jsPsych.data.addProperties({subject: 1, condition: 'control'});
```

---

## jsPsych.data.displayData

```javascript
jsPsych.data.displayData(format)
```

### Parameters

Parameter | Type | Description
----------|------|------------
format | string | Specifies whether to display the data in `'csv'` or `'json'` format.

### Return value

Returns nothing.

### Description

Outputs all of the data collected in the experiment to the screen in either JSON or CSV format. This is a useful method for quick debugging when developing an experiment.

### Examples

#### Using the on_finish callback function to show data at the end of the experiment

```javascript
var jsPsych = initJsPsych({
	on_finish: function() {
		jsPsych.data.displayData('csv');
	}
})
```

---

## jsPsych.data.get

```
jsPsych.data.get()
```

### Parameters

None.

### Return value

Returns the data collection of all data generated by the experiment.

### Description

This function is the standard starting point for accessing the data generated by the experiment. It returns a DataCollection object, which has
several methods that can be used to further filter, aggregate, and view the data. These methods are described under the DataCollection section
on this page.

### Example

```javascript
// select all trials
var all_data = jsPsych.data.get();

// get csv representation of data and log to console
console.log(all_data.csv());
```

---

## jsPsych.data.getInteractionData

```javascript
jsPsych.data.getInteractionData()
```

### Parameters

None.

### Return value

Returns a DataCollection object with all of the interaction events.

### Description

jsPsych automatically records a few different kinds of user interaction events. `blur` events occur when the user clicks on another window or tab during the experiment, indicating that they are no longer interacting with the experiment. `focus` events occur when the user clicks on the experiment window after having clicked somewhere else first (i.e., generated a `blur` event). `fullscreenenter` and `fullscreenexit` events are triggered by the browser entering and exiting fullscreen mode. However, `fullscreenenter` events only occur when the script switches the browser to fullscreen mode, e.g., with the [fullscreen plugin](../plugins/fullscreen.md). Manually entering fullscreen mode does not trigger this event. `fullscreenexit` events occur whether the user manually exits fullscreen mode or the script exits fullscreen mode. This method returns the DataCollection containing all interaction events. This is useful for tracking whether the participant completed the task without diverting attention to other windows. Events are in the form:

```javascript
{
	type: 'focus' or 'blur' or 'fullscreenenter' or 'fullscreenexit',
	trial: 10, // the trial number when the event happened
	time: 13042 // total time elapsed since the start of the experiment
}
```

### Example

```javascript
var interaction_data = jsPsych.data.getInteractionData();
// log data to console in json format
console.log(interaction_data.json());
```

---

## jsPsych.data.getLastTimelineData

```javascript
jsPsych.data.getLastTimelineData()
```

### Return value

Returns a DataCollection.

### Description

Gets all of the data generated in the same timeline as the last trial.

### Example

```javascript
var lasttimelinedata = jsPsych.data.getLastTimelineData();
```

---

## jsPsych.data.getLastTrialData

```javascript
jsPsych.data.getLastTrialData()
```

### Return value

Returns a DataCollection.

### Description

Gets the data collection containing all data generated by the last trial.

### Example

```javascript
var lasttrialdata = jsPsych.data.getLastTrialData();
```

---

## jsPsych.data.getURLVariable

```javascript
jsPsych.data.getURLVariable(var_name)
```

### Parameters

Parameter | Type | Description
----------|------|------------
var_name | string | Which variable to get the value of.

### Return value

Returns the value of a variable passed in through the query string.

### Description

For extracting a particular variable passed in through a URL query string.

### Example

```javascript
// if the URL of the page is: experiment.html?subject=1234&condition=test
console.log(jsPsych.data.getURLVariable('subject')) // logs "1234"
console.log(jsPsych.data.getURLVariable('condition')) // logs "test"
```

---

## jsPsych.data.urlVariables

```javascript
jsPsych.data.urlVariables()
```

### Return value

Returns an object (associative array) of the variables in the URL query string.

### Description

For extracting variables passed in through a URL query string.

### Example

```javascript
// if the URL of the page is: experiment.html?subject=1234&condition=test
var urlvar = jsPsych.data.urlVariables();
console.log(urlvar.subject) // logs "1234"
console.log(urlvar.condition) // logs "test"
```

---

## jsPsych.data.write

```javascript
jsPsych.data.write(data_object)
```

### Parameters

Parameter | Type | Description
----------|------|------------
data_object | object | Object of `key: value` pairs to store in jsPsych's data storage as a trial.

### Return value

Returns nothing.

### Description

This method is used by `jsPsych.finishTrial` for writing data. You should probably not use it to add data. Instead use [jsPsych.data.addProperties](#jspsychdataaddproperties).

### Example

```javascript
// don't use this! data should only be written once per trial. use jsPsych.finishTrial to save data.

var trial_data = {
	correct: true,
	rt: 487
}

jsPsych.data.write(trial_data);
```

---

## DataCollection

All data is stored in the DataCollection object. Using methods like `jsPsych.data.get()` and `jsPsych.data.getLastTrialData()` return DataCollections containing the experiment data. This is a list of all of the methods that are available to call on a DataCollection object.

#### .addToAll()

Adds a set of properties to all items in the DataCollection. Similar to `jsPsych.data.addProperties()`, except that it can be applied to a subset of the whole DataCollection by filtering down to a smaller DataCollection first.

```javascript
jsPsych.data.get().addToAll({subject_id: 123, condition: 'control'});
```

#### .addToLast()

Adds a set of properties to the last trial in the DataCollection.

```javascript
jsPsych.data.get().addToLast({success: true});
```

#### .count()

Counts the number of trials in the DataCollection.

```javascript
jsPsych.data.get().count()
```

#### .csv()

Generates a CSV string representing all of the data in the DataCollection.

```javascript
console.log(jsPsych.data.get().csv());
```

#### .filter()

Returns a subset of the DataCollection based on the filter. The filter is an object, and trials are only kept in the returned DataCollection if they contain the key: value pair(s) in the filter object. For example, the code below selects all of the trials with a correct response.

```javascript
var correct_trials = jsPsych.data.get().filter({correct: true});
```

The object can have multiple key: value pairs, and the trials must match all of them in order to be included in the returned collection.

```javascript
// keep only correct trials from the practice phase
var correct_practice_trials = jsPsych.data.get().filter({correct:true, phase: 'practice'});
```

The filter can also be an array of objects. In this case each object in the array acts as an OR filter. As long as the trial has all the key: value pairs of one of the objects in the array, it will appear in the returned collection.

```javascript
// select trials from block 1 and block 5.
var trials = jsPsych.data.get().filter([{block: 1}, {block:5}]);
```

The filter method returns a DataCollection object, so methods can be chained onto a single statement.

```javascript
// count the number of correct trials in block 1
var block_1_correct = jsPsych.data.get().filter({block:1, correct:true}).count();
```

#### .filterColumns()

Selects the set of columns listed in the array. This is the opposite of the `.ignore()` method.

```javascript
// Get only the subject, rt, and condition entries for each trial.
const subset_of_data = jsPsych.data.get().filterColumns(['subject', 'rt', 'condition'])
```

#### .filterCustom()

This method is similar to the `.filter()` method, except that it accepts a function as the filter. The function is passed a single argument, containing the data for a trial. If the function returns `true` the trial is included in the returned DataCollection.

```javascript
// count the number of trials with a response time greater than 2000ms.
var too_long = jsPsych.data.get().filterCustom(function(trial){
	return trial.rt > 2000;
}).count()
```

#### .first() / .last()

Returns a DataCollection containing the first/last *n* trials. If *n* is greater than the number of trials in the DataCollection, then these functions will return an array of length equal to the number of trials. If there are no trials in the DataCollection, then these functions will return an empty array. If the *n* argument is omitted, then the functions will use the default value of 1. If *n* is zero or a negative number, then these functions will throw an error.

```javascript
var first_trial = jsPsych.data.get().first(1);
var last_trial_with_correct_response = jsPsych.data.get().filter({correct: true}).last(1);
var last_10_trials = jsPsych.data.get().last(10);
```

#### .ignore()

Returns a DataCollection with all instances of a particular key removed from the dataset.

```javascript
// log a csv file that does not contain the trial_type values for each trial
console.log(jsPsych.data.get().ignore('trial_type').csv());
```

#### .join()

Appends one DataCollection onto another and returns the combined collection.

```javascript
// get a DataCollection with all trials that are either correct or
// have a response time greater than 200ms.
var dc1 = jsPsych.data.get().filter({correct: true});
var dc2 = jsPsych.data.get().filterCustom(function(trial){ return trial.rt > 200});
var data = dc1.join(dc2);
```

#### .json()

Generates a JSON string representing all of the data in the DataCollection.

```javascript
console.log(jsPsych.data.get().json());
```

#### .localSave()

Saves a CSV or JSON file on the computer running the experiment. If conducting an online experiment, this will download the file onto the subject's computer, and is therefore not a recommended data storage solution for online data collection.

**Warning:** This function may not behave correctly in older browsers. Upgrading to the latest version of any major web browser should solve the problem.

```javascript
// first argument is the format, second is the filename.
// the format can be either 'csv' or 'json'.
jsPsych.data.get().localSave('csv','mydata.csv');
```

#### .push()

Add a new entry to the DataCollection. This method is mostly used internally, and you shouldn't need to call it under normal circumstances.

```javascript
var data = {correct: true, rt: 500}
jsPsych.data.get().push(data);
```

#### .readOnly()

Creates a copy of the DataCollection so that any modification of the values in the DataCollection will not affect the original.

```javascript
// this line edits the rt property of the first trial
jsPsych.data.get().first(1).values()[0].rt = 100;

// readOnly creates a copy that can be modified without affecting the original

jsPsych.data.get().first(1).values()[0].rt
// outputs 100

jsPsych.data.get().readOnly().first(1).values()[0].rt = 200
jsPsych.data.get().first(1).values()[0].rt
// still outputs 100
```

#### .select()

Returns a DataColumn object (see documentation below) of a single property from a DataCollection object.

```javascript
var rt_data = jsPsych.data.get().select('rt');
rt_data.mean()
```

#### .uniqueNames()

Generates an array of all the unique key names in the set of trials contained in the DataCollection. This is especially useful when setting up a relational database (e.g., MySQL) where the column names need to be specified in advance.

```javascript
console.log(jsPsych.data.get().uniqueNames());
```

#### .values()

Returns the raw data array associated with the DataCollection. This array is modifiable, so changes to the array and values of objects in the array will change the DataCollection.

```javascript
var raw_data = jsPsych.data.get().values();

// was response in first trial correct?
if(raw_data[0].correct){
	console.log('correct!');
} else {
	console.log('incorrect.');
}
```

---

## DataColumn

DataColumn objects represent all the values of a single property in a DataCollection. They are generated by using the `.select()` method on a DataCollection. Once a DataColumn is generated, the following methods can be used.

#### .all()

Checks if all values in the DataColumn return `true` when passed to a function. The function takes a single argument, which represents one value from the DataColumn.

```javascript
// check if all the response times in the practice phase were under 1000ms
jsPsych.data.get().filter({phase: 'practice'}).select('correct').all(function(x) { return x < 1000; });
```

#### .count()

Counts the number of values in the DataColumn.

```javascript
// count how many response times there are
jsPsych.data.get().select('rt').count();
```

#### .frequencies()

Counts the number of occurrences of each unique value in the DataColumn. Returns this value as an object, where each key is a unique value and the value of each key is the number of occurrences of that key.

```javascript
// get frequencies of correct and incorrect responses
jsPsych.data.get().select('correct').frequencies();
```

#### .max() / .min()

Returns the maximum or minimum value in a DataColumn.

```javascript
jsPsych.data.get().select('rt').max();
jsPsych.data.get().select('rt').min();
```

#### .mean()

Returns the average of all the values in a DataColumn.

```javascript
jsPsych.data.get().select('rt').mean();
```

#### .median()

Returns the median of all the values in a DataColumn.

```javascript
jsPsych.data.get().select('rt').median();
```

#### .sd()

Returns the standard deviation of the values in a DataColumn.

```javascript
jsPsych.data.get().select('rt').sd();
```

#### .subset()

Filters the DataColumn to include only values that return `true` when passed through the specified function.

```javascript
// below results will be less than 200.
jsPsych.data.get().select('rt').subset(function(x){ return x < 200; }).max();
```

#### .sum()

Returns the sum of the values in a DataColumn.

```javascript
jsPsych.data.get().select('rt').sum();
```

#### .values

The raw array of values in the DataColumn.

```javascript
// note that this is not a function call.
jsPsych.data.get().select('rt').values;
```

#### .variance()

Returns the variance of the values in a DataColumn.

```javascript
jsPsych.data.get().select('rt').variance();
```
