# Download and Compile ROMS and CICE

Follow the instructions below to set up a coupled ocean/sea ice model based on the [Regional Ocean Modeling System (ROMS)](http://myroms.org/) and the [Los Alamos Sea Ice Model, CICE](https://github.com/CICE-Consortium/CICE-svn-trunk). These instructions are based on the setup on the [Compute Canada/ACENET](https://www.ace-net.ca/) high-performance computer ["Siku"](https://www.ace-net.ca/wiki/Siku) and may have to be adjusted for other systems.

## Table of Contents
* [0. Preamble](#0-preamble)  
* [1. `applications` Repository](#1-applications-repository)  
* [2. Model Coupling Toolkit (MCT)](#2-model-coupling-toolkit-mct)  
* [3. Los Alamos Sea Ice Model (CICE)](#3-los-alamos-sea-ice-model-cice)  
* [4. Regional Ocean Modeling System (ROMS)](#4-regional-ocean-modeling-system-roms)
* [5. Prepare and Run a Coupled Simulation](#5-prepare-and-run-a-coupled-simulation)
* [6. Run a Model Simulation](#6-run-a-model-simulation)

# 0. Preamble

## Prerequesites

* Compute Canada account
* GitHub account with [SSH key for password-less login]([https://docs.github.com/en/github/authenticating-to-github/connecting-to-github-with-ssh/adding-a-new-ssh-key-to-your-github-account)

## File Structure

The setup below will result in the following directory structure:

```bash
/home/<username>/
    ├── applications/    # model configurations
    |    └── NWA12/
    ├── models/          # ROMS and CICE model source code
    |    ├── CICE5.1/
    |    └── ROMS/
    └── software/        # MCT source code
         └── MCT/
```

Log into the Compute Canada cluster and change into to your `/home/` directory and create the following directories

```bash
cd ~/
mkdir models/ software/
```

## Module Versions

Add the appropriate modules

```bash
module load netcdf-fortran-mpi/4.5.2
```

# 1. `applications` Repository

The `applications` repository contains subfolders for ROMS/CICE model configurations. This is where you should keep header and input files as well as auxiliary scripts that help to set up and compile the models.

## Create a Fork of the `applications` Repository

Go to the [applications](https://github.com/NWA-BCP-Theme-III/applications) on GitHub and create a [fork](https://guides.github.com/activities/forking/) by clicking the **Fork** button in the top-right corner of the header. This should create a copy of this repository in your profile. You can access this copy by navigating through you profile or at

`https://github.com/<username>/applications`,

where `<username>` is your GitHub user name.

## Clone Your Fork

To download your fork, go to your terminal and change into your /home directory and clone your fork of the `applications` repository:

```bash
cd ~/
git clone git@github.com:<username>/applications
```

If you have set up a SSH key to your GitHub account, the download should start automatically.

Change into the directory for the NWA12 configuration and list the contents:

```bash
cd ~/applications/NWA12
ls -l
```

You should see the following files and directories:

```bash
/home/<username>/applications/NWA12/
    ├── build_cice.sh  # master shell script to compile CICE
    ├── build_roms.sh  # shell script to compile ROMS
    ├── clean_ice      # clean-up script for CICE (called by build_cice.sh)
    ├── comp_ice       # compile script for CICE (called by build_cice.sh)
    ├── include/
    |    └── nwa12.h   # header file with CPP flags for ROMS
    └── load_modules   # helper script to load the appropriate modules
    ├── NWA12_R0000/   # template directory for model runs
    └── prep_job.sh    # shell script to prepare model simulation
```

These files will be used below to compile a coupled ROMS/CICE configuration below.

If you want to run an uncoupled model configuration, you can skip the next two sections and continue with [4. Regional Ocean Modeling System (ROMS)](#4-regional-ocean-modeling-system-roms).

# 2. Model Coupling Toolkit (MCT)

The coupling between ROMS and CICE is facilitated by the [Model Coupling Toolkit (MCT)](https://www.mcs.anl.gov/research/projects/mct/), version 2.11.

Change into the `software/` directory and clone the MCT source code from GitHub:

```bash
cd ~/software
git clone git@github.com:MCSclimate/MCT.git MCT/
```

This downloads the latest version of the `master` branch into the directory `MCT/`. At the time of writing, this version is equivalent to the MCT release version 2.11, but this may change in the future. In order to make sure we always use version 2.11, we check out the commit that was tagged with `MCT_2.11.0`.

Change into the newly created directory `MCT/` and check out the version with the tag `MCT_2.11.0`:

```bash
cd MCT/
git checkout -b v2.11 MCT_2.11.0
```

The option `-b` creates a new branch `v2.11` switches to it. This allows us to keep the `master` synched with the GitHub version and will make it easy to upload to a newer version in the future, if we wish to do so. Make sure you are using the right branch:

```bash
git branch
```

The output should be

```bash
  master
* v2.11
```

where the asterisk indicates the branch you are currently on.

To install MCT, run the configuration script and compile the source code:

```bash
./configure FC=mpif90
make
```

# 3. Los Alamos Sea Ice Model (CICE)

To download and compile the Los Alamos Sea Ice Model (CICE) follow the instructions below.

## Create a Fork of the CICE5.1 Repository

Go to the [CICE5.1 Repository](https://github.com/NWA-BCP-Theme-III/CICE5.1) on GitHub and create a [fork](https://guides.github.com/activities/forking/) by clicking the **Fork** button in the top-right corner of the header. This should create a copy of this repository in your profile. You can access this copy by navigating through you profile or at

`https://github.com/<username>/CICE5.1`,

where `<username>` is your GitHub user name.

## Clone Your Fork

To download your fork, go to your terminal and change into the directory where you want to save the source code, e.g.,

```bash
cd ~/models/
```

and clone the CICE5.1 repository:

```bash
git clone git@github.com:<username>/CICE5.1.git CICE5.1
```

If you set up a SSH key to your GitHub account, the download should start automatically. By default, this creates a copy of the `dalroms` branch in the directory `CICE5.1`. Change into that directory and check that you successfully downloaded this branch:

```bash
cd ~/models/CICE5.1/
git branch
```

The output should be

```bash
* dalroms
```

The asterisk indicates the branch that is currently active.

## Compile the CICE Source Code

Change into the directory of the NWA12 configuration:

```bash
cd ~/applications/NWA12
```

Before you compile CICE, you have to specify a few settings. Open the script `comp_ice` with an editor of your choice. 

Check the following variables and change them if necessary:

| Variable |   |  
|----------|---|  
| `SITE`   | Computing environment. Make sure that a corresponding Makefile Macro exists in `~/models/CICE5.1/bld/`. The file name should be `Macros.Linux.${SITE}` and specifies the compiler flags and options for CICE. We set this to the positional input argument `"$1"` so we can specify the environment, when the script is executed. |
| `RES` | Model Configuration. This is use to specify the number of processors, block size, model layers, and ice categories further below in the script. |
| `GRID` | Model grid dimensions `<Nx>x<Ny>`, where `<Nx>` and `<Ny>` are the number of grid points in the x- and y- directions, respectively. These values have to be the same as the x- and y-dimensions in the CICE grid file. *(Note that these values are typically different from the values `Lm` and `Mm` in the ROMS namelist!)* |  

For a new model configuration, add a new case to the `if`-statement startin in line 29, e.g.,

```bash
else if ($RES == <RES>) then
   setenv NTASK     48       # total number of processors
   setenv BLCKX     84       # x-dimension of blocks ( not including )
   setenv BLCKY    127       # y-dimension of blocks (  ghost cells  )
   setenv NICELYR    7       # number of vertical layers in the ice
   setenv NSNWLYR    1       # number of vertical layers in the snow
   setenv NICECAT    5       # number of ice thickness categories
```

where `<RES>` is the name of the model configuration specified above. This segment sets the number of processors, block size, model layers, and ice categories.

| Variable  |   |  
|-----------|---|  
| `NTASK`   | Total number of CPUs to run CICE. This value has to be equal to the value of `nprocs` specified in the CICE namelist (see below).|  
| `BLCKX`   | x-dimension of blocks. |
| `BLCKY`   | y-dimension of blocks. |
| `NICELYR` | Number of vertical layers in the ice |  
| `NSNWLYR` | Number of vertical layers in the snow |  
| `NICECAT` | Number of ice thickness categories |

To run CICE in parallel, the model domain is divided into blocks (tiles). You can run multiple blocks on one processor, but we choose the dimensions such that we have one block per CPU. In the example above, we choose a total number of `NTASK` = 48 processors. If we want to run one block per CPU, we have to divide our model domain into 48 (=6x8) blocks. To specify the dimensions of the individual blocks, we divide the total number of grid points in the x-direction (`<Nx>`=504) by 6 and the total number of grid points in the y-direction (`<Ny>`=1015) by 8. This gives us `BLCKX` = 504/6 = 84 and `BLCKY` =  1015/8 &approx; 127.

Once you have set all the variables above, save the file and close your editor.

To compile the CICE source code run the build script with the name of the computing environment as input parameter:

```bash
source build_cice.sh computecanada
```

This master script

* executes `load_modules` to load the appropriate modules
* sets the path to the MCT library
* executes `clean_ice` to clean up previous builds
* executes `comp_ice` to compile the CICE source code

If the compiling was successful, you will find a subfolder `Build_cice/` in the directory of your configuration. This subfolder should contain the file `libcice.a` which is the compiled CICE library that we will link to ROMS.

# 4. Regional Ocean Modeling System (ROMS)

To download and compile the ROMS source code follow the instructions below.

## Create a Fork

Create a fork of the [ROMS repository](https://github.com/NWA-BCP-Theme-III/ROMS) by clicking the **Fork** button in the top-right corner of the header. This should create a copy of the repository in your profile. You can access this copy by navigating through you profile or at

`https://github.com/<username>/ROMS`,

where `<username>` is your GitHub user name.

## Clone Your Fork

To download your fork to your computer, open a terminal and change into the directory where you want to save the source code, e.g.,

```bash
cd ~/models/
```

and clone the ROMS repository:

```bash
git clone git@github.com:<username>/ROMS.git ROMS
```

If you set up a SSH key to your GitHub account, the download should start automatically. By default, this creates a copy of the `dalroms` branch in the directory `ROMS`. Change into that directory and check that you successfully downloaded this branch:

```bash
cd ~/models/ROMS/
git branch
```

The output should be

```bash
* dalroms
```

The asterisk indicates the branch that is currently active.

## Compile the ROMS Source Code

Change into the directory of the NWA12 configuration:

```bash
cd ~/applications/NWA12
```

Before you compile ROMS, you have to specify a few settings. Open the script `build_roms.sh` with an editor of your choice. 

Check the following variables and change them if necessary:

| Variable |   |  
|----------|---|  
| `ROMS_APPLICATION` | Model configuration. This will determine the name of the `".h"` header file in the subdirectory `include/`. |
| `FORT` | Fortran compiler. On the Compute Canada clusters, we are using `ifort`. On tikoraluk set the compiler to `gfortran`. |

Once you have set all the variables above, save the file and close your editor.

To compile the ROMS source code run the build script. Here, we use the option `-j 36` to use 36 processors to compile the code in parallel.

```bash
./build_roms.sh -j 36
```

If the compiling was successful, you will find a subfolder `Build_roms/` and the executable `romsM` in the directory of your configuration.

# 5. Prepare and Run a Coupled Simulation

To run a realistic coupled model simulation with ROMS and CICE you will need input files (e.g., model grid and bathymetry), forcing files (e.g., atmospheric, boundary, and tidal forcing), and namelist files that specify model parameters for a given simulation. We keep these files separate from the source code repositories to increase the flexibility in sharing and using the code and data.

On the Compute Canada clusters (e.g., Siku, Graham, Beluga, ...), it is recommended to use the [`SCRATCH` storage](https://docs.computecanada.ca/wiki/Storage_and_file_management#Storage_types) for intensiive read/write operations like ocean model simulations. That means it is best to keep the input and forcing files in your `~/scratch/` folder. However, this partition is not backed up and older files are subject to purging. Therefore, you should have a copy of your input and forcing files in the `~/projects/def-.../<username>/` directory.

## Input and Forcing Files

If you want to run multiple model simulations with the same model configuration, it makes sense to create one directory with all the input and forcing files. For the NWA12 configuration we keep these files in the directory `NWA12_input/` with the following structure:

```bash
~/scratch/NWA12_input/
    ├── boundary/  # boundary forcing files
    ├── clim/      # climatology files
    ├── config/    # grid/bathymetry files, nudging coefficient files
    ├── forcing/   # atmospheric forcing files
    ├── initial/   # initial condition files
    └── sponge/    # sponge layer files for boundary nudging
```

## Set up Model Experiment

For each model simulation, we create a directory that contains the namelist files with the model parameters and write the output and restart files to. These direcories should be in the `~/scratch/` partition and have the following structure:

```bash
~/scratch/NWA12_RXXX/
    ├── external/          # namelist files
    ├── output/            # model output
    |    ├── ice/          # sea ice fields
    |    └── ocean/        # ocean fields
    ├── restart/           # restart files
    ├── romsM              # ROMS executable
    └── submit_job_siku.sh # job submission script
```

The name of the directory `NWA12_RXXX/` is a combination of the name of the model configuration and a run ID with a running number that clearly defines each simulation. A template with the structure outlined above is in the `NWA12` directory of the `applications` repository. There is also a shell script `prep_job.sh` that copies the template to `~/scratch/` and sets the path according to your username and a provided runID.

To prepare a model simulation:

```bash
cd ~/applications/NWA12/
./prep_job.sh NWA12_RXXX
```
 where `NWA12_RXXX` is the runID that you should specify. Once the preparation script is finished it will prompt you the path of the run directory. Change into that directory and modify the namelists, e.g.,

 ```bash
 cd ~/scratch/NWA12_RXXX/
 ```

The three namelist files in the `external/` subdirectory are:

* `coupling_nwa12.in`
* `ice_in`
* `roms_nwa12.in`

### `coupling_nwa12.in`

This file specifies the options for the model coupling. The relevant parameters are

| Variable |   |
|----------|---|
| `Nthreads(ocean)` | Number of processors used by ROMS |
| `Nthreads(cice)`  | Number of processors used by CICE. Note that this has to be equal to the value of `NTASK` specified for the compilation of CICE and it has also be equal to the value of `nprocs` specified in the CICE namelist (see below). |
| `TimeInterval` | Coupling interval. This is the time in seconds between each call of the coupler to exchange fields between the ocean and sea ice components. |

### `ice_in`

Namelist with model parameters for CICE. Please refer to the [CICE documentation](https://ncar.github.io/CICE/users_guide/ice_nml_var.html) for information about the parameters.

### `roms_nwa12.in`

Namelist with model parameters for ROMS. Please refer to the glossary at the end of the file for information about the parameters.

# 6. Run a Model Simulation

Once you have ensured that you have all the necessary input files and set the parameters in the namelists, navigate to the run directory

```bash
cd ~/scratch/NWA12_RXXX/
```

Open the script `submit_job.sh` with and editor of your choice and set the appropriate variables for the job scheduler

| Variable      |   |
|---------------|---|
| `--account`   |   |
| `--ntasks`    | Total number of processors. This number has to be the sum of the values for `Nthreads(ocean)` and `Nthreads(cice)` in `coupling_nwa12.in` |
| `--mem`       | Memory request per node. If `--mem=0`, all memory of a node is requested. |
| `--time`      | Requested walltime. |
| `--mail-user` | Email address. |

Save the file, close your editor and submit the job

```bash
sbatch submit_job.sh
```

You can check the status of your job with the `sq` or `squeue` commands.