// ----------------------------------------------------------------------------
// Copyright (C) 2012 Louise A. Dennis, and  Michael Fisher 
//
// This file is part of Agent JPF (AJPF)
// 
// AJPF is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 3 of the License, or (at your option) any later version.
// 
// AJPF is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with AJPF; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
// 
// To contact the authors:
// http://www.csc.liv.ac.uk/~lad
//
//----------------------------------------------------------------------------

package ajpf.product;

import gov.nasa.jpf.JPF;

import java.util.Set;
import java.util.HashSet;
import java.util.Map;
import java.util.HashMap;
import java.util.List;
import java.util.ArrayList;
import java.util.logging.Level;
import java.util.logging.Logger;

import ajpf.psl.Proposition;
import ajpf.util.AJPFException;
import ajpf.util.ProbabilisticEdgeAnnotationException;
import gov.nasa.jpf.Config;


/**
 * This class will create a model of the program based on the numbered states generated by the JPF transition system.
 * 
 * These states will be labelled with atomic formulae true in that state.  
 * Edges are stored as a Map indexed by the state at the start of the edge.
 * 
 * It stores the current path being explored through the model.
 * 
 * @author louiseadennis
 *
 */
public class MCAPLmodel {
	 protected static Logger log = JPF.getLogger("ajpf.product.MCAPLmodel");
	 
	 /*
	  * States in the model indexed by their JPF state number.
	  */
	 Map<Integer, ModelState> states_by_num = new HashMap<Integer, ModelState>();

	 /* Edges of the graph indexed by incoming state number*/
	 Map<Integer, Set<Integer>> model_edges = new HashMap<Integer, Set<Integer>>();
			
	 /* Current path in the model being explored by JPF */
	 List<Integer> current_path = new ArrayList<Integer>();
	
	 /* Size of the current path */
	 int current_path_size = 0;
	
	 /*
	  * Only needed where annotations (e.g. probabilities) are being attached to edges.
	  * Often the annotation is generated before the edge is created.
	  */
	 boolean next_edge_to_be_annotated = false;
	 double next_edge_annotation = 0.0;
	
	 /*
	  * Needed when the model is being reproduced for use with a different model checker.
	  * 
	  * A list of all the atomic propositions relevant to the model.
	  */
	 Set<Proposition> props = new HashSet<Proposition>();
	
	 /*
	  * For printing.  An enuemration of the possible output formats.
	  */
	 protected static enum OutputFormat {
		 Promela, Default, Prism
	 }
	
	 /*
	  * The output format to be used.
	  */
	 OutputFormat output = OutputFormat.Default;
		
	
	 /*
	  * The configuration of this AJPF run.  Only used for generating output.
	  */
	 Config config;
			

	 /**
	  * Constructor.
	  * @param ps
	  */
	 public MCAPLmodel(Set<Proposition> ps) {
		 props = ps;
	 }
	
	 /**
	  * Is there already a state in the model with this statenum?  If so return that state.
	  * @param statenum
	  * @return
	  */
	 public ModelState containsState(int statenum) {
		 if (states_by_num.containsKey(statenum)) {
			 return states_by_num.get(statenum);
		 }
 		
		 return null;
	 }
	 
	 /**
	  * Add a new state to the model.  
	  * @param s
	  */
	 public void addState(ModelState s) {
		 states_by_num.put(s.getNum(), s);
		 log.fine("adding state");	
	 }
	 
	/**
	 * Prune the current path back to this statenum and return the index at which the statenum first occurs.
	 * We assume we are only pruning one state.
	 * @param statenum
	 * @return
	 */
	public int prune (int statenum) {
		// We assume statenum is in path
		if (lowerLogLevelThan(Level.FINE)) {
			log.fine("pruning 1 state from " + current_path);
		}
		int index = current_path_size - 1;
		// We remove the head of the path because a prune has been triggered by it.
		if (!current_path.isEmpty()) {
			current_path.remove(current_path_size - 1);
			current_path_size = current_path.size();
		}
		// We also assume the state we are looking for is closer to the end of the current path
		// than the beginning.
		for (int i = current_path_size - 1; i >= 0; i--) {
			if (current_path.get(i) != statenum) {
				current_path.remove(i);
				index = i;
			} else {
				break;
			}
		}
		current_path_size = current_path.size(); 
		log.fine("index is " + index);
		
		return index;
	}
	
	
	/**
	 * Add an edge to the model.  
	 * @param e
	 */
	public void addEdge(ModelState s) throws AJPFException {
		if (! current_path.isEmpty()) {
			Integer from = current_path.get(current_path.size() - 1);
			
			if (model_edges.containsKey(from)) {
				model_edges.get(from).add(s.getNum());
			} else {
				HashSet<Integer> edges = new HashSet<Integer>();
				edges.add(s.getNum());
				model_edges.put(from, edges);
			}
			
			// Annotate if necessary.
			if (next_edge_to_be_annotated) {
				try {
					states_by_num.get(from).edge_annotate_double(s.getNum(), next_edge_annotation);
					// s.edge_annotate_double(from, next_edge_annotation);
				} catch (AJPFException e) {
					if (e instanceof ProbabilisticEdgeAnnotationException) {
						((ProbabilisticEdgeAnnotationException) e).addFrom(s.getNum());
					}
					throw e;
				}
				next_edge_to_be_annotated = false;
				next_edge_annotation = 0.0;
			}
		}
	}
	
	public Integer getEndofPathState() {
		return current_path.get(current_path.size() - 1);
	}
	
	/**
	 * Add a state to the current path in the model.
	 * @param s
	 */
	public void addToPath(ModelState s) {
		current_path.add(s.getNum()); 
		current_path_size++;
		if (lowerLogLevelThan(Level.INFO)) {
			log.info("Current Path: " + current_path.toString());
		}
	}

	/**
	 * Mark a particular state as done - this means all outgoing edges from that state have been explored or 
	 * there is no need to explore the outgoing edges (if the product automata will always succeed beyond
	 * this point - e.g., all eventualities have been achieved).
	 * @param modelstatenum
	 */
	public void setdone(int modelstatenum) {
		states_by_num.get(modelstatenum).markdone();
	}
	
	/**
	 * Getter for the current Path
	 * @return
	 */
	public List<Integer> getCurrentPath() {
		return current_path;
	}
		
	/**
	 * Is this state already in the current path.
	 * @param s
	 * @return
	 */
	public boolean currentPathContains(ModelState s) {
		return current_path.contains(s.getNum());
	}
	
	
	/**
	 * The length of the current path.
	 * @return
	 */
	public int currentPathSize() {
		return current_path_size;
	}
	
	/**
	 * Overridable method for annotating model edges.  It assumes that the edge to be annotated is the next
	 * edge to be formed.
	 * @param a
	 */
	public void annotate_edge(double a) {
		log.fine("Annotating an edge");
		next_edge_to_be_annotated = true;
		next_edge_annotation = a;
	}
	

	/**
	 * Returns the model state with the JPF state number i.
	 * @param i
	 * @return
	 */
	public ModelState getState(Integer i) {
		return states_by_num.get(i);
	}
	
	/**
	 * Sets the output format given a string.
	 * @param format
	 */
	public void setOutputFormat(String format) {
		output = OutputFormat.valueOf(format);
	}
		
	/**
	 * Sets the configuration.
	 * @param c
	 */
	public void setConfig(Config c) {
		config = c;
	}

	/*
	 * (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	protected int higheststatenum = 0;
	public String toString() {
		 if (config.containsKey("ajpf.target_modelchecker")) {
			 if (config.getProperty("ajpf.target_modelchecker").equals("spin")) {
			 	output = OutputFormat.Promela;
			 } else if (config.getProperty("ajpf.target_modelchecker").equals("prism")) {
				 output = OutputFormat.Prism;
			 }
		 }

		String s = "";
		switch (output) {
			case Prism:	
				s += "dtmc\n\n module jpfModel\n";
				
				s += "state : [0 .." + (states_by_num.size() - 1) + "] init 0;\n";
				
				higheststatenum = states_by_num.size() - 1;

				if (!props.isEmpty()) {
					ModelState init = states_by_num.get(0);
					for (Proposition p: props) {
						String pstring = convertPropForSpin(p.toString());
						if (init.getProps().contains(p)) {
							s += pstring + ": bool init true;\n";
						} else {
							s += pstring + ": bool init false;\n";
						}
					}
				}
				
				
				for (ModelState state: states_by_num.values()) {
					int num = state.getNum();
					
					Set<Integer> edges = model_edges.get(num);
					if (edges != null) {
						s += "[] state = " + num + " -> ";
						boolean initialedge = true;
						for (int to: edges) {
							if (! initialedge ) {
								s += " + ";
							} else {
								initialedge = false;
							}
							s += printEdge(num, to, output);
							for (Proposition p: props) {
								ModelState tostate = states_by_num.get(to);
								String pstring = convertPropForSpin(p.toString());
								if (tostate.getProps().contains(p)) {
									s += " & (" + pstring + "'= true)";
								} else {
									s += " & (" + pstring + "'= false)";
								}
							}
						}
						s += ";\n";
					}
				}
				
				s += "endmodule";
				break;
			
			case Promela:
				 // We need to declare all the properties as booleans at the top of the
				 // Spin model.
				HashMap<Proposition, String> propstringhash = new HashMap<Proposition, String>();
				if (!props.isEmpty()) {
					s+="bool ";
					int i = 1;
					for (Proposition p: props) {
						String pstring = convertPropForSpin(p.toString());
						s += pstring;
						if (i < props.size()) {
							s += ", ";
							i++;
						}
						propstringhash.put(p, pstring);
					}
					s += "\n\n";
				}
								
				// Now we convert our model to a JPF proctype.
				s += "active proctype JPFModel()\n";
				s += "{\n";
				
				for (ModelState state: states_by_num.values()) {
					int num = state.getNum();
					// Creating state numbers for end states.
					if (num < 0) {
						s += "end_state" + Math.abs(num);
					} else {
						s += "state" + num;
					}
					s += ":\n";
					Set<Integer> edges = model_edges.get(num);
					for (Proposition p: props) {
						// String pstring = convertPropForSpin(p.toString());
						String pstring = propstringhash.get(p);
						s+="\t" + pstring + " = ";
						if (state.getProps().contains(p)) {
							s += "true;\n";
						} else {
							s += "false;\n";
						}
					}
					if (edges != null) {
						if (edges.size() > 1) {
							s += "\tif\n";
							for (int to: edges) {
								if (to < 0) {
									s += "\t:: goto end_state" + Math.abs(to) + ";\n";
								} else {
									s += "\t:: goto state" + to + ";\n";
								}
							}
							s += "\tfi;\n";
						} else {
							for (int to: edges) {
								if (to < 0) {
									s += "\tgoto end_state" + Math.abs(to) + ";\n";
								} else {
									s += "\tgoto state" + to + ";\n";
								}
							}
						}
					} else {
						s += "\tprintf(\"end state\\n\");\n";
					}
				}
				s += "}";
				break;
				
			case Default:
				s += "MODEL PRODUCED BY AJPF\n";
				s += "Model States:\n";
				s += "=============\n\n";
				for (ModelState state: states_by_num.values()) {
					s += state.toVerboseString();
					s += "\n";
					for (Proposition p: props) {
						String pstring = p.toString();
						s+="\t" + pstring + " = ";
						if (state.getProps().contains(p)) {
							s += "true;\n";
						} else {
							s += "false;\n";
						}
					}
				}
		
				s += "\n";
				s += "Model Edges:\n";
				s += "=============\n\n";
				for (int from : model_edges.keySet()) {
					for (int to : model_edges.get(from)) {
						s += printEdge(from, to, output);
					}
				}
		}
		return s;
		
	}
	
	/**
	 * Printing an edge in the Default format.
	 * @param from
	 * @param to
	 * @return
	 */
	protected String printEdge(int from, int to, OutputFormat f) {
		 String s = "";
		 switch (output) {
		 	case Default:
				s += from;
				s += "-->";
				s += to;
				s += "\n";
				return s;
		 	case Promela:
		 }
		 return s;
	}
	
	/**
	 * Helper function for Spin outputs.  We take the string representing a property and strip out all brackets
	 * and commas and convert to lower case.
	 * @param property
	 * @return
	 */
	protected static String convertPropForSpin(String property) {
		String pstring = property.toString();
		String pstring1 = pstring.replace("(", "");
		pstring = pstring1.replace(")", "");
		pstring1 = pstring.replace(",", "");
		pstring = pstring1.replace(".", "");
		return (pstring.toLowerCase());

	}

	/**
	 * I'm under the impression that composition of strings is quite inefficient in java.  Therefore we don't want to
	 * perform such compositions for logging messages unless absolutely necessary.  This is a "helper" function for simply
	 * determing the log level and it is wrapped around any log message that requires string composition.  I _think_ using
	 * this function doesn't introduce a competeing overhead because it is static, but I could be wrong.
	 * @param l
	 * @return
	 */
	private static boolean lowerLogLevelThan(Level l) {
		if  (log.getLevel().intValue() <= l.intValue()) {
			return true;
		}
		return false;
	}
	
}
