#include "testindividual.h"
#include "individual.h"
#include "utilities.h"
#include <iostream>

/*
void TestIndividual::testFindHomeRangeGraph()
{
    rNG.seed(31);

    // 1. build small test grid
    Grid* testGrid = new Grid(10);

    // 2. give habitat and resource values to grid
    std::vector<float> cellValues = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                                     0, 0, 0, 0, 0, 2, 0, 0, 0, 0,
                                     0, 0, 0, 0, 2, 2, 0, 0, 2, 0,
                                     0, 0, 0, 0, 0, 0, 0, 0, 2, 0,
                                     0, 2, 0, 0, 0, 2, 2, 0, 2, 0,
                                     0, 2, 0, 0, 2, 2, 0, 0, 2, 0,
                                     0, 0, 0, 0, 0, 2, 2, 0, 2, 0,
                                     0, 0, 0, 0, 0, 2, 2, 2, 2, 0,
                                     0, 0, 0, 0, 0, 2, 0, 0, 0, 0,
                                     0, 0, 0, 0, 0, 0, 0, 0, 0, 0};

    int i = 0;
    for (int y = 0; y < testGrid->size; y++) {
        for (int x = 0; x < testGrid->size; x++) {
            GridCell* cell = testGrid->GetCell(x, y);
            cell->resourceAmount = cellValues[i];
            if (cell->resourceAmount > 0) {
                cell->habitatTyp = Habitat;
            } else {
                cell->habitatTyp = Matrix;
            }
            cell->riskLevel = 0;
            i++;
        }
    }

    // 3. create one new individual
    Species* species = new Species();
    species->meanBodymass = 0.5;
    species->sdBodymass = 0.05;
    Individual* individual = new Individual(1, species, 2, false, 10);
    individual->foodShare = 0.5;
    individual->feedingRate = 16;
    individual->locomotionCost = 0;
    individual->pregnancyStatus = NotPregnant;
    individual->maxHomeRangeRadius = 70;
    individual->responsiveness.slope = 0;
    individual->responsiveness.intercept = 1;

    individual->FindHomeRangeGraph(testGrid, nullptr);

    //std::cout << "final x: " << individual->location_x << " y: " << individual->location_y << std::endl;

    // 4. sort homeRangeCells
    std::sort(individual->homeRangeCells.begin(), individual->homeRangeCells.end(), [=] (GridCell* cell_i, GridCell* cell_j) {
        if (cell_i->x == cell_j->x) {
            return cell_i->y < cell_j->y;
        }
        return cell_i->x < cell_j->x;
    });

    std::vector<std::tuple<int, int>> result{  {4,5},
                                               {5,4}, {5,5}, {5,6}, {5,7}, {5,8},
                                               {6,4}, {6,6}, {6,7},
                                               {7,7},
                                               {8,2}, {8,3}, {8,4}, {8,5}, {8,6}, {8,7}
                                             };
    // 5. verify vector with results (list of tupels given to function)
    //std::cout << "individual vector " << individual->homeRangeCells.size() << ", result vector " << result.size() << std::endl;
    QVERIFY(individual->homeRangeCells.size() == result.size());

    for (int i = 0; i < individual->homeRangeCells.size(); i++) {
        GridCell* cell = individual->homeRangeCells[i];
        QCOMPARE(cell->resourceAmount, 1);
        QVERIFY(cell->x == get<0>(result[i]) && cell->y == get<1>(result[i]));
    }

    // test foraged grid resources
    std::vector<float> newValues = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                                    0, 0, 0, 0, 0, 2, 0, 0, 0, 0,
                                    0, 0, 0, 0, 2, 2, 0, 0, 1, 0,
                                    0, 0, 0, 0, 0, 0, 0, 0, 1, 0,
                                    0, 2, 0, 0, 0, 1, 1, 0, 1, 0,
                                    0, 2, 0, 0, 1, 1, 0, 0, 1, 0,
                                    0, 0, 0, 0, 0, 1, 1, 0, 1, 0,
                                    0, 0, 0, 0, 0, 1, 1, 1, 1, 0,
                                    0, 0, 0, 0, 0, 1, 0, 0, 0, 0,
                                    0, 0, 0, 0, 0, 0, 0, 0, 0, 0};

    int j = 0;
    for (int y = 0; y < testGrid->size; y++) {
        for (int x = 0; x < testGrid->size; x++) {
            GridCell* cell = testGrid->GetCell(x, y);
            QCOMPARE(cell->resourceAmount, newValues[j]);
            j++;
        }
    }
}

// ------------------------------------------------------------------------------------------------//

void TestIndividual::testCheckHomeRangeGraph()
{
    // 1. build small test grid
    Grid* testGrid = new Grid(10);
    testGrid->landscapeResistance = 0.0;

    // 2. give habitat and resource values to grid
    std::vector<float> cellValues = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                                     0, 0, 0, 0, 0, 2, 0, 0, 0, 0,
                                     0, 0, 0, 0, 2, 2, 0, 0, 1, 0,
                                     0, 0, 0, 0, 0, 0, 0, 0, 4, 0,
                                     0, 2, 0, 0, 0, 2, 2, 0, 2, 0,
                                     0, 2, 0, 0, 4, 2, 0, 0, 4, 0,
                                     0, 0, 0, 0, 0, 2, 4, 0, 4, 0,
                                     0, 0, 0, 0, 0, 2, 4, 4, 2, 0,
                                     0, 0, 0, 0, 0, 2, 0, 0, 0, 0,
                                     0, 0, 0, 0, 0, 0, 0, 0, 0, 0};

    int i = 0;
    for (int y = 0; y < testGrid->size; y++) {
        for (int x = 0; x < testGrid->size; x++) {
            GridCell* cell = testGrid->GetCell(x, y);
            cell->resourceAmount = cellValues[i];
            if (cell->resourceAmount > 0) {
                cell->habitatTyp = Habitat;
            } else {
                cell->habitatTyp = Matrix;
            }
            cell->riskLevel = 0;
            i++;
        }
    }

    // 3. create one new individual
    Species* species = new Species();
    species->meanBodymass = 0.5;
    species->sdBodymass = 0.05;
    Individual* individual = new Individual(1, species, 2, false, 10);
    individual->foodShare = 0.5;
    individual->feedingRate = 14;
    individual->locomotionCost = 0;
    individual->pregnancyStatus = NotPregnant;
    individual->maxHomeRangeRadius = 70;
    individual->responsiveness.slope = 0;
    individual->responsiveness.intercept = 1;

    // give core cell to individual
    individual->location_x = 5;
    individual->location_y = 4;

    individual->CheckHomeRangeGraph(testGrid);

    for (int i = 0; i < individual->homeRangeCells.size(); i++) {
        std::cout << "individual vector 1 " << individual->homeRangeCells[i]->x << " ; " << individual->homeRangeCells[i]->y << std::endl;
    }

    // 4. sort homeRangeCells
    std::sort(individual->homeRangeCells.begin(), individual->homeRangeCells.end(), [=] (GridCell* cell_i, GridCell* cell_j) {
        if (cell_i->x == cell_j->x) {
            return cell_i->y < cell_j->y;
        }
        return cell_i->x < cell_j->x;
    });

    std::vector<std::tuple<int, int>> result{ {4,2}, {4,5},
                                              {5,1}, {5,2}, {5,4}, {5,5}, {5,6}, {5,7},
                                              {6,4}, {6,6}, {6,7},
                                              {8,4}
                                            };


        //std::vector<std::tuple<int, int>> result{  {4,2}, {4,5}, {5,1}, {5,2}, {5,4}, {5,5}, {5,6}, {5,7}, {6,4}, {6,6}, {6,7},
                                               //{7,7}, {8,3}, {8,4}, {8,5}, {8,6}};


    // 5. verify vector with results (list of tupels given to function)
    std::cout << "individual vector " << individual->homeRangeCells.size() << ", result vector " << result.size() << std::endl;
    QVERIFY(individual->homeRangeCells.size() == result.size());

    for (int i = 0; i < individual->homeRangeCells.size(); i++) {
        GridCell* cell = individual->homeRangeCells[i];
        QVERIFY(cell->x == get<0>(result[i]) && cell->y == get<1>(result[i]));
    }

    // test foraged grid resources
    std::vector<float> newValues = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                                    0, 0, 0, 0, 0, 1, 0, 0, 0, 0,
                                    0, 0, 0, 0, 1, 1, 0, 0, 1, 0,
                                    0, 0, 0, 0, 0, 0, 0, 0, 4, 0,
                                    0, 2, 0, 0, 0, 1, 1, 0, 1, 0,
                                    0, 2, 0, 0, 2, 1, 0, 0, 4, 0,
                                    0, 0, 0, 0, 0, 1, 2, 0, 4, 0,
                                    0, 0, 0, 0, 0, 1, 3, 4, 2, 0,
                                    0, 0, 0, 0, 0, 2, 0, 0, 0, 0,
                                    0, 0, 0, 0, 0, 0, 0, 0, 0, 0};

    int j = 0;
    for (int y = 0; y < testGrid->size; y++) {
        for (int x = 0; x < testGrid->size; x++) {
            GridCell* cell = testGrid->GetCell(x, y);
            //std::cout << "foraged cell " << cell->resourceAmount << ", result vector " << newValues[j] << std::endl;
            QCOMPARE(cell->resourceAmount, newValues[j]);
            j++;
        }
    }
}


// ------------------------------------------------------------------------------------------------//

void TestIndividual::testCheckHomeRangeGraph()
{
    // 1. build small test grid
    Grid* testGrid = new Grid(10);
    testGrid->landscapeResistance = 0.5;
    testGrid->crossMatrix = true;

    // 2. give habitat and resource values to grid
    std::vector<float> cellValues = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                                     0, 0, 0, 0, 0, 2, 0, 0, 0, 0,
                                     0, 0, 0, 0, 2, 2, 0, 0, 1, 0,
                                     0, 0, 0, 0, 0, 0, 0, 0, 4, 0,
                                     0, 2, 0, 0, 0, 2, 2, 0, 2, 0,
                                     0, 2, 0, 0, 4, 2, 0, 0, 6, 0,
                                     0, 0, 0, 0, 0, 2, 4, 0, 4, 0,
                                     0, 0, 0, 0, 0, 2, 4, 0, 2, 0,
                                     0, 0, 0, 0, 0, 2, 0, 0, 0, 0,
                                     0, 0, 0, 0, 0, 0, 0, 0, 0, 0};

    int i = 0;
    for (int y = 0; y < testGrid->size; y++) {
        for (int x = 0; x < testGrid->size; x++) {
            GridCell* cell = testGrid->GetCell(x, y);
            cell->resourceAmount = cellValues[i];
            if (cell->resourceAmount > 0) {
                cell->habitatTyp = Habitat;
            } else {
                cell->habitatTyp = Matrix;
            }
            cell->riskLevel = 0;
            i++;
        }
    }

    // 3. create one new individual
    Species* species = new Species();
    species->meanBodymass = 0.5;
    species->sdBodymass = 0.05;
    Individual* individual = new Individual(1, species, 2, false, 10);
    individual->foodShare = 0.5;
    individual->feedingRate = 21.5;
    individual->locomotionCost = 0;
    individual->pregnancyStatus = NotPregnant;
    individual->maxHomeRangeRadius = 70;
    individual->responsiveness.slope = -1;
    individual->responsiveness.intercept = 1;

    // give core cell to individual
    individual->location_x = 5;
    individual->location_y = 4;

    individual->CheckHomeRangeGraph(testGrid);

    // 4. sort homeRangeCells
    std::sort(individual->homeRangeCells.begin(), individual->homeRangeCells.end(), [=] (GridCell* cell_i, GridCell* cell_j) {
        if (cell_i->x == cell_j->x) {
            return cell_i->y < cell_j->y;
        }
        return cell_i->x < cell_j->x;
    });

    std::vector<std::tuple<int, int>> result{ {4,2}, {4,5}, {5,1}, {5,2}, {5,4}, {5,5}, {5,6}, {5,7}, {5,8}, {6,4}, {6,6}, {6,7}, {8,3}, {8,4}, {8,5}, {8,6}};

    //{4,2}  {4,5}, {5,1}, {5,2}, {5,4}, {5,5}, {5,6}, {5,7}, {6,4}, {6,6}, {6,7}, {8,4}

    // 5. verify vector with results (list of tupels given to function)
    //std::cout << "individual vector " << individual->homeRangeCells.size() << ", result vector " << result.size() << std::endl;
    QVERIFY(individual->homeRangeCells.size() == result.size());

    //for (int i = 0; i < individual->homeRangeCells.size(); i++) {
    //    std::cout << "individual vector 1 " << individual->homeRangeCells[i]->x << " ; " << individual->homeRangeCells[i]->y << std::endl;
    //}

    for (int i = 0; i < individual->homeRangeCells.size(); i++) {
        GridCell* cell = individual->homeRangeCells[i];
        QVERIFY(cell->x == get<0>(result[i]) && cell->y == get<1>(result[i]));
    }

    // test foraged grid resources
    std::vector<float> newValues =  {0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                                     0, 0, 0, 0, 0, 1, 0, 0, 0, 0,
                                     0, 0, 0, 0, 1, 1, 0, 0, 1, 0,
                                     0, 0, 0, 0, 0, 0, 0, 0, 2, 0,
                                     0, 2, 0, 0, 0, 1, 1, 0, 1, 0,
                                     0, 2, 0, 0, 2, 1, 0, 0, 3, 0,
                                     0, 0, 0, 0, 0, 1, 2, 0, 2, 0,
                                     0, 0, 0, 0, 0, 1, 2, 0, 2, 0,
                                     0, 0, 0, 0, 0, 1, 0, 0, 0, 0,
                                     0, 0, 0, 0, 0, 0, 0, 0, 0, 0};

    int j = 0;
    for (int y = 0; y < testGrid->size; y++) {
        for (int x = 0; x < testGrid->size; x++) {
            GridCell* cell = testGrid->GetCell(x, y);
            //std::cout << "foraged cell " << cell->resourceAmount << ", result vector " << newValues[j] << std::endl;
            QCOMPARE(cell->resourceAmount, newValues[j]);
            j++;
        }
    }
}


// ------------------------------------------------------------------------------------------------//

void TestIndividual::testCheckHomeRange()
{
    // 1. build small test grid
    Grid* testGrid = new Grid(10);

    // 2. give habitat and resource values to grid
    std::vector<float> cellValues = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                                     0, 0, 0, 0, 0, 2, 0, 0, 0, 0,
                                     0, 0, 0, 0, 2, 2, 0, 0, 1, 0,
                                     0, 0, 0, 0, 0, 0, 0, 0, 4, 0,
                                     0, 2, 0, 0, 0, 2, 2, 0, 2, 0,
                                     0, 2, 0, 0, 4, 2, 0, 0, 4, 0,
                                     0, 0, 0, 0, 0, 2, 4, 0, 4, 0,
                                     0, 0, 0, 0, 0, 2, 4, 4, 2, 0,
                                     0, 0, 0, 0, 0, 2, 0, 0, 0, 0,
                                     0, 0, 0, 0, 0, 0, 0, 0, 0, 0};

    int i = 0;
    for (int y = 0; y < testGrid->size; y++) {
        for (int x = 0; x < testGrid->size; x++) {
            GridCell* cell = testGrid->GetCell(x, y);
            cell->resourceAmount = cellValues[i];
            if (cell->resourceAmount > 0) {
                cell->habitatTyp = Habitat;
            } else {
                cell->habitatTyp = Matrix;
            }
            cell->riskLevel = 0;
            i++;
        }
    }

    // 3. create one new individual
    Species* species = new Species();
    species->meanBodymass = 0.5;
    species->sdBodymass = 0.05;
    Individual* individual = new Individual(1, species, 2, false, 10);
    individual->foodShare = 0.5;
    individual->feedingRate = 23;
    individual->locomotionCost = 0;
    individual->pregnancyStatus = NotPregnant;
    individual->maxHomeRangeRadius = 4;
    individual->responsiveness.slope = 0;
    //individual->responsiveness.intercept = 1;

    // give core cell to individual
    individual->location_x = 5;
    individual->location_y = 4;

    Statistics* statistics = new Statistics(1);
    individual->CheckHomeRange(testGrid, statistics, 10);

    // 4. sort homeRangeCells
    std::sort(individual->homeRangeCells.begin(), individual->homeRangeCells.end(), [=] (GridCell* cell_i, GridCell* cell_j) {
        if (cell_i->x == cell_j->x) {
            return cell_i->y < cell_j->y;
        }
        return cell_i->x < cell_j->x;
    });

    for (int i = 0; i < individual->homeRangeCells.size(); i++) {
        std::cout << "individual vector 1 " << individual->homeRangeCells[i]->x << " ; " << individual->homeRangeCells[i]->y << std::endl;
    }

    std::vector<std::tuple<int, int>> result{  {4,2}, {4,5},
                                               {5,1}, {5,2}, {5,4}, {5,5}, {5,6}, {5,7},
                                               {6,4}, {6,6}, {6,7},
                                               {7,7},
                                               {8,3}, {8,4}, {8,5}, {8,6}
                                             };

    // 5. verify vector with results (list of tupels given to function)
    // std::cout << "individual vector " << individual->homeRangeCells.size() << ", result vector " << result.size() << std::endl;
    QVERIFY(individual->homeRangeCells.size() == result.size());

    for (int i = 0; i < individual->homeRangeCells.size(); i++) {
        GridCell* cell = individual->homeRangeCells[i];
        QVERIFY(cell->x == get<0>(result[i]) && cell->y == get<1>(result[i]));
    }

    // test foraged grid resources
    std::vector<float> newValues = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                                    0, 0, 0, 0, 0, 1, 0, 0, 0, 0,
                                    0, 0, 0, 0, 1, 1, 0, 0, 1, 0,
                                    0, 0, 0, 0, 0, 0, 0, 0, 2, 0,
                                    0, 2, 0, 0, 0, 1, 1, 0, 1, 0,
                                    0, 2, 0, 0, 2, 1, 0, 0, 2, 0,
                                    0, 0, 0, 0, 0, 1, 2, 0, 2, 0,
                                    0, 0, 0, 0, 0, 1, 2, 2, 2, 0,
                                    0, 0, 0, 0, 0, 2, 0, 0, 0, 0,
                                    0, 0, 0, 0, 0, 0, 0, 0, 0, 0};

    int j = 0;
    for (int y = 0; y < testGrid->size; y++) {
        for (int x = 0; x < testGrid->size; x++) {
            GridCell* cell = testGrid->GetCell(x, y);
            //std::cout << "x: " << cell->x << ", y: " << cell->y << ", resources: " << cell->resourceAmount << std::endl;
            QCOMPARE(cell->resourceAmount, newValues[j]);
            j++;
        }
    }
}

// ------------------------------------------------------------------------------------------------//

void TestIndividual::testAssessHomeRangePersonality()
{
    // 1. build small test grid
    Grid* testGrid = new Grid(6);

    // 2. give habitat and resource values to grid
    std::vector<float> cellValues = {0, 0, 0, 0, 0, 0,
                                     0, 0, 0, 0, 0, 0,
                                     0, 0, 0, 0, 0, 0,
                                     8, 0, 8, 1, 0, 0,
                                     2, 0, 7, 0, 0, 0,
                                     0, 0, 0, 0, 0, 0};

    int i = 0;
    for (int y = 0; y < testGrid->size; y++) {
        for (int x = 0; x < testGrid->size; x++) {
            GridCell* cell = testGrid->GetCell(x, y);
            cell->resourceAmount = cellValues[i];
            if (cell->resourceAmount > 0) {
                cell->habitatTyp = Habitat;
            } else {
                cell->habitatTyp = Matrix;
            }
            cell->riskLevel = 0;
            i++;
        }
    }

    // 3. create one new individual
    Species* species = new Species();
    species->meanBodymass = 0.5;
    species->sdBodymass = 0.05;
    Individual* individual = new Individual(1, species, 2, false, 10);
    individual->foodShare = 0.5;
    individual->feedingRate = 13;
    individual->locomotionCost = 0;
    individual->pregnancyStatus = NotPregnant;
    individual->maxHomeRangeRadius = 70;
    individual->responsiveness.slope = 0;
    //individual->responsiveness.intercept = 1;

    // give core cell to individual
    individual->location_x = 3;
    individual->location_y = 3;

    GridCell* coreCell = testGrid->GetCell(individual->location_x, individual->location_y);

    individual->AssessHomeRangePersonality(testGrid, coreCell, individual->feedingRate, false);

    // 4. sort homeRangeCells
    std::sort(individual->homeRangeCells.begin(), individual->homeRangeCells.end(), [=] (GridCell* cell_i, GridCell* cell_j) {
        if (cell_i->x == cell_j->x) {
            return cell_i->y < cell_j->y;
        }
        return cell_i->x < cell_j->x;
    });

    std::vector<std::tuple<int, int>> result{  {0,3}, {0,4},
                                               {2,3}, {2,4},
                                               {3,3}
                                             };

    // 5. verify vector with results (list of tupels given to function)
    //std::cout << "individual vector " << individual->homeRangeCells.size() << ", result vector " << result.size() << std::endl;
    QVERIFY(individual->homeRangeCells.size() == result.size());

    for (int i = 0; i < individual->homeRangeCells.size(); i++) {
        GridCell* cell = individual->homeRangeCells[i];
        QVERIFY(cell->x == get<0>(result[i]) && cell->y == get<1>(result[i]));
    }

    // test foraged grid resources
    std::vector<float> newValues = {0, 0, 0, 0, 0, 0,
                                    0, 0, 0, 0, 0, 0,
                                    0, 0, 0, 0, 0, 0,
                                    4, 0, 4, 0.5, 0, 0,
                                    1, 0, 3.5, 0, 0, 0,
                                    0, 0, 0, 0, 0, 0};

    int j = 0;
    for (int y = 0; y < testGrid->size; y++) {
        for (int x = 0; x < testGrid->size; x++) {
            GridCell* cell = testGrid->GetCell(x, y);
            QCOMPARE(cell->resourceAmount, newValues[j]);
            j++;
        }
    }
}

// ------------------------------------------------------------------------------------------------//

void TestIndividual::testAssessHomeRangeNoMatrix()
{
    // 1. build small test grid
    Grid* testGrid = new Grid(6);

    // 2. give habitat and resource values to grid
    std::vector<float> cellValues = {0, 0, 0, 0, 0, 0,
                                     0, 0, 0, 0, 0, 0,
                                     0, 0, 0, 0, 0, 0,
                                     8, 0, 8, 1, 0, 0,
                                     2, 0, 7, 0, 0, 0,
                                     0, 0, 0, 0, 0, 0};

    int i = 0;
    for (int y = 0; y < testGrid->size; y++) {
        for (int x = 0; x < testGrid->size; x++) {
            GridCell* cell = testGrid->GetCell(x, y);
            cell->resourceAmount = cellValues[i];
            if (cell->resourceAmount > 0) {
                cell->habitatTyp = Habitat;
            } else {
                cell->habitatTyp = Matrix;
            }
            cell->riskLevel = 0;
            i++;
        }
    }

    // 3. create one new individual
    Species* species = new Species();
    species->meanBodymass = 0.5;
    species->sdBodymass = 0.05;
    Individual* individual = new Individual(1, species, 2, false, 10);
    individual->foodShare = 0.5;
    individual->feedingRate = 13;
    individual->locomotionCost = 0;
    individual->pregnancyStatus = NotPregnant;
    individual->maxHomeRangeRadius = 70;
    individual->responsiveness.slope = 0;
    //individual->responsiveness.intercept = 1;

    // give core cell to individual
    individual->location_x = 3;
    individual->location_y = 3;

    GridCell* coreCell = testGrid->GetCell(individual->location_x, individual->location_y);

    individual->AssessHomeRangeNoMatrix(testGrid, coreCell, individual->feedingRate, false);

    // 4. sort homeRangeCells
    std::sort(individual->homeRangeCells.begin(), individual->homeRangeCells.end(), [=] (GridCell* cell_i, GridCell* cell_j) {
        if (cell_i->x == cell_j->x) {
            return cell_i->y < cell_j->y;
        }
        return cell_i->x < cell_j->x;
    });

    std::vector<std::tuple<int, int>> result{  {2,3}, {2,4},
                                               {3,3}
                                             };

    // 5. verify vector with results (list of tupels given to function)
    //std::cout << "individual vector " << individual->homeRangeCells.size() << ", result vector " << result.size() << std::endl;
    QVERIFY(individual->homeRangeCells.size() == result.size());

    for (int i = 0; i < individual->homeRangeCells.size(); i++) {
        GridCell* cell = individual->homeRangeCells[i];
        QVERIFY(cell->x == get<0>(result[i]) && cell->y == get<1>(result[i]));
    }

    // test foraged grid resources
    std::vector<float> newValues = {0, 0, 0, 0, 0, 0,
                                    0, 0, 0, 0, 0, 0,
                                    0, 0, 0, 0, 0, 0,
                                    8, 0, 4, 0.5, 0, 0,
                                    2, 0, 3.5, 0, 0, 0,
                                    0, 0, 0, 0, 0, 0};

    int j = 0;
    for (int y = 0; y < testGrid->size; y++) {
        for (int x = 0; x < testGrid->size; x++) {
            GridCell* cell = testGrid->GetCell(x, y);
            QCOMPARE(cell->resourceAmount, newValues[j]);
            j++;
        }
    }
}

*/
// ------------------------------------------------------------------------------------------------//

void TestIndividual::testAssessHomeMatrixCrossing()
{
    // 1. build small test grid
    Grid* testGrid = new Grid(10);

    // 2. give habitat and resource values to grid
    std::vector<float> cellValues = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                                     0, 0, 0, 0, 0, 2, 0, 0, 0, 0,
                                     0, 0, 0, 0, 2, 2, 0, 0, 1, 0,
                                     0, 0, 0, 0, 0, 0, 0, 0, 4, 0,
                                     0, 2, 0, 0, 0, 2, 3, 0, 2, 0,
                                     0, 2, 0, 0, 4, 2, 0, 0, 4, 0,
                                     0, 0, 0, 0, 0, 2, 4, 0, 4, 0,
                                     0, 0, 0, 0, 0, 2, 4, 4, 2, 0,
                                     0, 0, 0, 0, 0, 2, 0, 0, 0, 0,
                                     0, 0, 0, 0, 0, 0, 0, 0, 0, 0};

    int i = 0;
    float maxResource = 0;
    for (int y = 0; y < testGrid->size; y++) {
        for (int x = 0; x < testGrid->size; x++) {
            GridCell* cell = testGrid->GetCell(x, y);
            cell->resourceAmount = cellValues[i];
            if (cell->resourceAmount > 0) {
                cell->habitatTyp = Habitat;
            } else {
                cell->habitatTyp = Matrix;
            }

            if (cell->resourceAmount > maxResource) {
                maxResource = cell->resourceAmount;
            }
            cell->riskLevel = 0;
            i++;
        }
    }
    testGrid->maxRL = maxResource;


    // 3. create one new individual
    Species* species = new Species();
    species->meanBodymass = 0.5;
    species->sdBodymass = 0.05;
    Individual* individual = new Individual(1, species, 2, false, 10);
    individual->foodShare = 0.5;
    individual->feedingRate = 13;
    individual->locomotionCost = 0;
    individual->pregnancyStatus = NotPregnant;
    individual->maxHomeRangeRadius = 70;
    individual->responsiveness.slope = -1;
    //individual->responsiveness.intercept = 1;

    // give core cell to individual
    individual->location_x = 5;
    individual->location_y = 4;

    GridCell* coreCell = testGrid->GetCell(individual->location_x, individual->location_y);

    individual->AssessHomeRangeMatrixCrossing(testGrid, coreCell, individual->feedingRate, false);

    /*
    for (int i = 0; i < individual->homeRangeCells.size(); i++) {
        std::cout << "individual vector 1 " << individual->homeRangeCells[i]->x << " ; " << individual->homeRangeCells[i]->y << std::endl;
    }
    */

    // 4. sort homeRangeCells
    std::sort(individual->homeRangeCells.begin(), individual->homeRangeCells.end(), [=] (GridCell* cell_i, GridCell* cell_j) {
        if (cell_i->x == cell_j->x) {
            return cell_i->y < cell_j->y;
        }
        return cell_i->x < cell_j->x;
    });

    std::vector<std::tuple<int, int>> result{  {4,5},
                                               {5,4}, {5,5}, {5,6}, {5,7},
                                               {6,4}, {6,6}, {6,7},
                                               {7,7}
                                             };

    // 5. verify vector with results (list of tupels given to function)
    //std::cout << "individual vector " << individual->homeRangeCells.size() << ", result vector " << result.size() << std::endl;
    QVERIFY(individual->homeRangeCells.size() == result.size());

    for (int i = 0; i < individual->homeRangeCells.size(); i++) {
        GridCell* cell = individual->homeRangeCells[i];
        QVERIFY(cell->x == get<0>(result[i]) && cell->y == get<1>(result[i]));
    }

    // test foraged grid resources
    std::vector<float> newValues =  {0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                                     0, 0, 0, 0, 0, 2, 0, 0, 0, 0,
                                     0, 0, 0, 0, 2, 2, 0, 0, 1, 0,
                                     0, 0, 0, 0, 0, 0, 0, 0, 4, 0,
                                     0, 2, 0, 0, 0, 1, 1.5, 0, 2, 0,
                                     0, 2, 0, 0, 2, 1, 0, 0, 4, 0,
                                     0, 0, 0, 0, 0, 1, 2, 0, 4, 0,
                                     0, 0, 0, 0, 0, 1, 2, 2.5, 2, 0,
                                     0, 0, 0, 0, 0, 2, 0, 0, 0, 0,
                                     0, 0, 0, 0, 0, 0, 0, 0, 0, 0};

    int j = 0;
    for (int y = 0; y < testGrid->size; y++) {
        for (int x = 0; x < testGrid->size; x++) {
            GridCell* cell = testGrid->GetCell(x, y);
            //std::cout << cell->x << ";" << cell->y << std::endl;
            QCOMPARE(cell->resourceAmount, newValues[j]);
            j++;
        }
    }
}

// ------------------------------------------------------------------------------------------------//
/*
void TestIndividual::CompareHomeRangeFunctions() {

    // test 10 times
    int testTimes = 10;
    std::vector<int> seeds{45, 31, 23, 765, 99,
                           89, 900, 7878, 43, 87};

    for (int a = 0; a < testTimes; a++) {

        rNG.seed(seeds[a]);
        //std::cout << "seed " << seeds[a] << std::endl;

        // 1. build small test grid
        Grid* testGrid = new Grid(10);
        testGrid->landscapeResistance = 0;

        // 2. give habitat and resource values to grid
        std::vector<float> cellValues = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                                         0, 0, 7, 0, 0, 1.9, 0, 4.5, 0, 0,
                                         0, 0, 6, 0, 1.7, 1.8, 0, 0, 1, 0,
                                         0, 0, 0, 0, 0, 0, 0, 0, 4.4, 0,
                                         0, 2.9, 0, 3.1, 0, 2.1, 2.6, 0, 2.7, 0,
                                         0, 2.8, 0, 0, 3.5, 2.2, 0, 0, 4.3, 0,
                                         0, 0, 0, 0, 0, 2.3, 4, 0, 4.2, 0,
                                         0, 0, 0, 5, 0, 2.4, 5.5, 4.1, 1.5, 0,
                                         0, 0, 8.1, 0, 0, 2.5, 0, 0, 0, 0,
                                         0, 0, 0, 0, 0, 0, 0, 3, 0, 0};

        int v = 0;
        for (int y = 0; y < testGrid->size; y++) {
            for (int x = 0; x < testGrid->size; x++) {
                GridCell* cell = testGrid->GetCell(x, y);
                cell->resourceAmount = cellValues[v];
                if (cell->resourceAmount > 0) {
                    cell->habitatTyp = Habitat;
                } else {
                    cell->habitatTyp = Matrix;
                }
                cell->riskLevel = 0;
                v++;
            }
        }

        // 3. create one new individual
        Species* species = new Species();
        species->meanBodymass = 0.1;
        species->sdBodymass = 0.01;
        Individual* individual = new Individual(1, species, 2, false, 10);
        //individual->locomotionCost = 0;
        //std::cout << "radius " << individual->maxHomeRangeRadius << std::endl;
        //std::cout << "foodShare " << individual->foodShare << std::endl;
        //std::cout << "feedingRate " << individual->feedingRate << std::endl;

        individual->pregnancyStatus = NotPregnant;
        individual->responsiveness.slope = 0;
        individual->responsiveness.intercept = 1;

        GridCell* coreCell = individual->FindCoreCell(testGrid, nullptr);
        individual->location_x = coreCell->x;
        individual->location_y = coreCell->y;
        //std::cout << "core cell " << individual->location_x << " ; " << individual->location_y << std::endl;

        Statistics* statistics = new Statistics(1);
        individual->CheckHomeRange(testGrid, statistics, 10);

        //for (int i = 0; i < individual->homeRangeCells.size(); i++) {
        //    std::cout << "individual vector 1 " << individual->homeRangeCells[i]->x << " ; " << individual->homeRangeCells[i]->y << std::endl;
        //}

        // 4. sort homeRangeCells
        std::sort(individual->homeRangeCells.begin(), individual->homeRangeCells.end(), [=] (GridCell* cell_i, GridCell* cell_j) {
            if (cell_i->x == cell_j->x) {
                return cell_i->y < cell_j->y;
            }
            return cell_i->x < cell_j->x;
        });

        std::vector<GridCell*> result1;
        //std::cout << "radius 1 " << individual->homerangeDiameter/2 << std::endl;

        for (int i = 0; i < individual->homeRangeCells.size(); i++) {
            result1.push_back(individual->homeRangeCells[i]);
        }

        std::vector<GridCell*> foragedGrid1;

        for (int y = 0; y < testGrid->size; y++) {
            for (int x = 0; x < testGrid->size; x++) {
                GridCell* cell = testGrid->GetCell(x, y);
                foragedGrid1.push_back(cell);
            }
        }

        bool dead1 = individual->dead;

        //------------------------------------------------//
        // start fresh with new function

        individual->homeRangeCells.clear();
        individual->dead = false;
        individual->resourceShortage = 0;

        int z = 0;
        for (int y = 0; y < testGrid->size; y++) {
            for (int x = 0; x < testGrid->size; x++) {
                GridCell* cell = testGrid->GetCell(x, y);
                cell->resourceAmount = cellValues[z];
                if (cell->resourceAmount > 0) {
                    cell->habitatTyp = Habitat;
                } else {
                    cell->habitatTyp = Matrix;
                }
                cell->riskLevel = 0;
                z++;
            }
        }

        testGrid->crossMatrix = true;
        individual->CheckHomeRangeGraph(testGrid);

        //for (int i = 0; i < individual->homeRangeCells.size(); i++) {
        //    std::cout << "individual vector 2 " << individual->homeRangeCells[i]->x << " ; " << individual->homeRangeCells[i]->y << std::endl;
        //}

        // 4. sort homeRangeCells
        std::sort(individual->homeRangeCells.begin(), individual->homeRangeCells.end(), [=] (GridCell* cell_i, GridCell* cell_j) {
            if (cell_i->x == cell_j->x) {
                return cell_i->y < cell_j->y;
            }
            return cell_i->x < cell_j->x;
        });

        std::vector<GridCell*> result2;
        //std::cout << "radius 2 " << individual->homerangeDiameter/2 << std::endl;

        for (int i = 0; i < individual->homeRangeCells.size(); i++) {
            result2.push_back(individual->homeRangeCells[i]);
        }

        // 5. verify vector with results (list of tupels given to function)
        //std::cout << "individual vector " << result1.size() << ", result vector " << result2.size() << std::endl;
        QVERIFY(result1.size() == result2.size());

        for (int i = 0; i < result1.size(); i++) {
            QVERIFY((result1[i]->x == result2[i]->x) && (result1[i]->y == result2[i]->y));
        }

        int b = 0;
        for (int y = 0; y < testGrid->size; y++) {
            for (int x = 0; x < testGrid->size; x++) {
                GridCell* cell = testGrid->GetCell(x, y);
                //std::cout << "foraged 2: " << cell->resourceAmount << " foraged 1: " << foragedGrid1[b]->resourceAmount << std::endl;
                QCOMPARE(cell->resourceAmount, foragedGrid1[b]->resourceAmount);
                b++;
            }
        }

        //std::cout << "dead1 " << dead1 << ", dead2 " << individual->dead << std::endl;
        //QVERIFY(dead1 == individual->dead);

    }
}

// ------------------------------------------------------------------------------------------------//

void TestIndividual::CompareFindHomeRangeFunctions() {

    // test 10 times
    int testTimes = 10;
    std::vector<int> seeds{45, 31, 23, 765, 99,
                           89, 900, 7878, 43, 87};

    for (int a = 0; a < testTimes; a++) {

        rNG.seed(seeds[a]);
        //std::cout << "seed " << seeds[a] << std::endl;

        // 1. build small test grid
        Grid* testGrid = new Grid(10);
        testGrid->landscapeResistance = 0;

        // 2. give habitat and resource values to grid
        std::vector<float> cellValues = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                                         0, 0, 7, 0, 0, 1.9, 0, 4.5, 0, 0,
                                         0, 0, 6, 0, 1.7, 1.8, 0, 0, 1, 0,
                                         0, 0, 0, 0, 0, 0, 0, 0, 4.4, 0,
                                         0, 2.9, 0, 3.1, 0, 2.1, 2.6, 0, 2.7, 0,
                                         0, 2.8, 0, 0, 3.5, 2.2, 0, 0, 4.3, 0,
                                         0, 0, 0, 0, 0, 2.3, 4, 0, 4.2, 0,
                                         0, 0, 0, 5, 0, 2.4, 5.5, 4.1, 1.5, 0,
                                         0, 0, 8.1, 0, 0, 2.5, 0, 0, 0, 0,
                                         0, 0, 0, 0, 0, 0, 0, 3, 0, 0};

        int v = 0;
        for (int y = 0; y < testGrid->size; y++) {
            for (int x = 0; x < testGrid->size; x++) {
                GridCell* cell = testGrid->GetCell(x, y);
                cell->resourceAmount = cellValues[v];
                if (cell->resourceAmount > 0) {
                    cell->habitatTyp = Habitat;
                } else {
                    cell->habitatTyp = Matrix;
                }
                cell->riskLevel = 0;
                v++;
            }
        }

        // 3. create one new individual
        Species* species = new Species();
        species->meanBodymass = 0.1;
        species->sdBodymass = 0.01;
        Individual* individual = new Individual(1, species, 2, false, 10);
        //individual->locomotionCost = 0;
        individual->pregnancyStatus = NotPregnant;
        individual->responsiveness.slope = 0;
        individual->responsiveness.intercept = 1;
        //std::cout << "radius " << individual->maxHomeRangeRadius << std::endl;
        //std::cout << "foodShare " << individual->foodShare << std::endl;
        //std::cout << "feedingRate " << individual->feedingRate << std::endl;

        Individual* individual2 = new Individual(2, species, 2, false, 10);
        individual2->locomotionCost = 0;
        individual2->pregnancyStatus = NotPregnant;
        individual2->responsiveness.slope = 0;
        individual2->responsiveness.intercept = 1;

        Statistics* statistics = new Statistics(1);

        rNG.seed(seeds[a]);
        individual->FindHomeRange(testGrid, nullptr, statistics, 10);
        //std::cout << "core cell " << individual->location_x << " ; " << individual->location_y << std::endl;

        rNG.seed(seeds[a] + 1);
        individual2->FindHomeRange(testGrid, nullptr, statistics, 10);
        //std::cout << "core cell " << individual2->location_x << " ; " << individual2->location_y << std::endl;

        //for (int i = 0; i < individual->homeRangeCells.size(); i++) {
        //    std::cout << "individual vector 1 " << individual->homeRangeCells[i]->x << " ; " << individual->homeRangeCells[i]->y << std::endl;
        //}

        // 4. sort homeRangeCells
        std::sort(individual->homeRangeCells.begin(), individual->homeRangeCells.end(), [=] (GridCell* cell_i, GridCell* cell_j) {
            if (cell_i->x == cell_j->x) {
                return cell_i->y < cell_j->y;
            }
            return cell_i->x < cell_j->x;
        });

        std::sort(individual2->homeRangeCells.begin(), individual2->homeRangeCells.end(), [=] (GridCell* cell_i, GridCell* cell_j) {
            if (cell_i->x == cell_j->x) {
                return cell_i->y < cell_j->y;
            }
            return cell_i->x < cell_j->x;
        });

        std::vector<GridCell*> result1a;
        std::vector<GridCell*> result1b;

        for (int i = 0; i < individual->homeRangeCells.size(); i++) {
            result1a.push_back(individual->homeRangeCells[i]);
        }
        for (int i = 0; i < individual2->homeRangeCells.size(); i++) {
            result1b.push_back(individual2->homeRangeCells[i]);
            //std::cout << "individual 2 vector 1 " << individual2->homeRangeCells[i]->x << " ; " << individual2->homeRangeCells[i]->y << std::endl;
        }

        std::vector<GridCell*> foragedGrid1;

        for (int y = 0; y < testGrid->size; y++) {
            for (int x = 0; x < testGrid->size; x++) {
                GridCell* cell = testGrid->GetCell(x, y);
                foragedGrid1.push_back(cell);
            }
        }

        bool dead1 = individual->dead;

        //------------------------------------------------//
        // start fresh with new function

        individual->homeRangeCells.clear();
        individual->dead = false;
        individual->resourceShortage = 0;

        int z = 0;
        for (int y = 0; y < testGrid->size; y++) {
            for (int x = 0; x < testGrid->size; x++) {
                GridCell* cell = testGrid->GetCell(x, y);
                cell->resourceAmount = cellValues[z];
                if (cell->resourceAmount > 0) {
                    cell->habitatTyp = Habitat;
                } else {
                    cell->habitatTyp = Matrix;
                }
                cell->riskLevel = 0;
                z++;
            }
        }

        rNG.seed(seeds[a]);
        individual->FindHomeRangeGraph(testGrid, nullptr);
        //std::cout << "core cell " << individual->location_x << " ; " << individual->location_y << std::endl;

        rNG.seed(seeds[a] + 1);
        individual2->FindHomeRangeGraph(testGrid, nullptr);
        //std::cout << "core cell " << individual2->location_x << " ; " << individual2->location_y << std::endl;

        //for (int i = 0; i < individual->homeRangeCells.size(); i++) {
        //    std::cout << "individual vector 2 " << individual->homeRangeCells[i]->x << " ; " << individual->homeRangeCells[i]->y << std::endl;
        //}

        // 4. sort homeRangeCells
        std::sort(individual->homeRangeCells.begin(), individual->homeRangeCells.end(), [=] (GridCell* cell_i, GridCell* cell_j) {
            if (cell_i->x == cell_j->x) {
                return cell_i->y < cell_j->y;
            }
            return cell_i->x < cell_j->x;
        });

        std::sort(individual2->homeRangeCells.begin(), individual2->homeRangeCells.end(), [=] (GridCell* cell_i, GridCell* cell_j) {
            if (cell_i->x == cell_j->x) {
                return cell_i->y < cell_j->y;
            }
            return cell_i->x < cell_j->x;
        });

        std::vector<GridCell*> result2a;
        std::vector<GridCell*> result2b;

        for (int i = 0; i < individual->homeRangeCells.size(); i++) {
            result2a.push_back(individual->homeRangeCells[i]);
        }
        for (int i = 0; i < individual2->homeRangeCells.size(); i++) {
            result2b.push_back(individual2->homeRangeCells[i]);
            //std::cout << "individual 2 vector 2 " << individual2->homeRangeCells[i]->x << " ; " << individual2->homeRangeCells[i]->y << std::endl;
        }

        // 5. verify vector with results (list of tupels given to function)
        //std::cout << "individual vector " << result1a.size() << ", result vector " << result2a.size() << std::endl;
        QVERIFY(result1a.size() == result2a.size());

        //std::cout << "individual vector " << result1b.size() << ", result vector " << result2b.size() << std::endl;
        QVERIFY(result1b.size() == result2b.size());

        for (int i = 0; i < result1a.size(); i++) {
            QVERIFY((result1a[i]->x == result2a[i]->x) && (result1a[i]->y == result2a[i]->y));
        }

        for (int i = 0; i < result1b.size(); i++) {
            QVERIFY((result1b[i]->x == result2b[i]->x) && (result1b[i]->y == result2b[i]->y));
        }

        int b = 0;
        for (int y = 0; y < testGrid->size; y++) {
            for (int x = 0; x < testGrid->size; x++) {
                GridCell* cell = testGrid->GetCell(x, y);
                //std::cout << "foraged 2: " << cell->resourceAmount << " foraged 1: " << foragedGrid1[b]->resourceAmount << std::endl;
                QCOMPARE(cell->resourceAmount, foragedGrid1[b]->resourceAmount);
                b++;
            }
        }

        //std::cout << "dead1 " << dead1 << ", dead2 " << individual->dead << std::endl;
        //QVERIFY(dead1 == individual->dead);
    }
}
*/

