#ifndef SIMULATION_H
#define SIMULATION_H

#include <QThread>
#include <individual.h>
#include <statistics.h>
#include <unordered_map>
#include <gridcell.h>
#include <QMutex>
#include <QObject>
#include <QRunnable>
#include <random>
#include "qtcsv/variantdata.h"
#include "qtcsv/stringdata.h"

//globals:
static const int GRID_SIZE = 7;
static const int GRID_DIMENSION = static_cast<int>(pow(2, 7) + 0.5) + 1;

enum CoreCellChoice{
    Unrestricted = 1,
    RestrictedByHabitatType = 2,
    RestrictedByCompetitors = 3,
    RestrictedByHabitatTypeAndCompetitors = 4
};

enum CommunityResponsiveness{
    Unresponsive = 0,
    Responsive = 1,
    Gradient = 2
};

struct BTdeath {
    float BTstarvation;
    float BTmatrix;
};

extern thread_local std::mt19937 rNG;

class Simulation : public QObject, public QRunnable
{
    Q_OBJECT
public:
    Simulation(QMutex* mutex, QList<QStringList>& aSpeciesList, int aCurrentRun, int aHabitatAmount, float aFragmentationLevel,
               bool aDensityControl, int aPatchLength, int aCommunityResponsiveness, bool aMatrixCosts, float aMatrixMort,
               float aGamma, float aHeritability);
    ~Simulation();

    Grid* landscape;
    std::vector<Species*> species;
    std::vector<Individual*> individuals;
    std::vector<int> behavioralTypes;
    std::vector<int> gammaTypes;
    Statistics* statistics;

    QtCSV::VariantData dataSimulation;
    QtCSV::VariantData dataSpecies;
    QtCSV::VariantData dataDiversity;
    QtCSV::VariantData dataIndividuals;
    QtCSV::VariantData dataHomeRanges;
    QtCSV::VariantData dataMap;

    // determining the scenario
    CoreCellChoice coreCellChoice;
    CommunityResponsiveness communityResponsiveness;
    float heritability;

    // landscape:
    float hurst;                 // spatial dependence of points
    int sigma;                 // variance in displacement of points
    float habitatCover;
    int habitatPatches;
    int gridSize;                // length and width of the grid
    int patchLength;             // length and width of one grid cell
    float availableFoodShare;    // muss für jedes Szanario verschieden variiert werden

    // scenario:
    bool densityControl;
    bool immigration;
    float maxMatrixMortality;
    float currentMatrixMortality;
    float searchStrategy;

    int currentRun;
    int initialNumberOfIndividuals;

    float grasslandProductivity; // overall productivity of grasslands in g/m2 * d (Buchmann et al. 2011)
    float patchProductivity;
    float averageProductivity;   // stores the daily average productivity of a landscape cell
    float maxProductivity;       // stores the resource amount of the most productive cell within the landscape
    float minProductivity;       // stores the resource amount of the least productive cell within the landscape
    float meanFileValues;        // stores the mean of all cell values from the generated landscape file

    bool finished;               // indicates if thread is finished
    bool homeRangeSampling;      // activates the use of home range sampling for all individuals to calculate MCPs
    bool print;
    int duration;
    int runtime;
    unsigned int seedMultiPurposeGenerator;

    void AddSpecies(QList<QStringList>& speciesList);
    void Setup();
    void SimulateOneDay();
    void PrepareData();
    void PrepareDiversityData();
    void PrepareSpeciesData();
    void PrepareIndividualData();
    void PrepareHomeRangeData();

signals:
    void dailyUpdate();

private:
    QMutex* mutex;
    bool needsSetup;

    std::unordered_map<int, BTdeath> mortalityPerBT;

    void HabitatLoss();
    void CreateRandomLandscape();
    void CreateLandscape();
    void CreateClumpLandscape();
    void UpdateResourceAmount();
    void CreateIndividuals();
    void SortIndividuals();
    void FindHomerangesForIndividuals();
    void CheckHomerangesForIndividuals();
    void LetIndividualsDie();
    void LifeHistory();
    void GetPregnant(Individual* individual);
    void GetOffspring(Individual* individual);
    void Senility(Individual* individual);
    void NaturalMortality(Individual* individual);
    void Density(Individual* individual);
    void CalculateCommunityIndices();
    void CalculateCellIndices();

    void run() override;
};

#endif // _H
