#include "simulation.h"
#include "utilities.h"
#include "statistics.h"
#include "neutrallandscape.h"
#include <vector>
#include <algorithm> // for std::nth_element - to calculate the median
#include <cmath>
#include <iostream>
#include <QTextStream>
#include <thread>
#include <chrono>
#include <assert.h>
#include <QCommandLineParser>

using namespace std;
thread_local std::mt19937 rNG;

Simulation::Simulation(QMutex* mutex, QList<QStringList>& aSpeciesList, int aCurrentRun, int aHabitatAmount, float aFragmentationLevel,
                       bool aDensityControl, int aPatchLength, int aCommunityResponsiveness, bool aMatrixCosts,
                       float aMatrixMort, float aGamma, float aHeritability)
{
    this->mutex = mutex;
    currentRun = aCurrentRun;
    coreCellChoice = CoreCellChoice(2);
    communityResponsiveness = CommunityResponsiveness(aCommunityResponsiveness);

    // key measures on the landscape:
    gridSize = GRID_DIMENSION;
    //std::cout << GRID_DIMENSION << std::endl;

    patchLength = aPatchLength;
    hurst = aFragmentationLevel;

    sigma = 30;
    habitatCover = static_cast<float>(aHabitatAmount / 100.0);
    habitatPatches = static_cast<int>(habitatCover * gridSize * gridSize);

    this->landscape = new Grid(gridSize);
    landscape->matrixCosts = aMatrixCosts;
    maxMatrixMortality = aMatrixMort;
    currentMatrixMortality = maxMatrixMortality;
    if (maxMatrixMortality <= 0) {
        landscape->matrixMortality = NoMortality;
    } else {
        //landscape->matrixMortality = PerFraction;
        landscape->matrixMortality = PerCell;
    }
    searchStrategy = aGamma;
    heritability = aHeritability;

    grasslandProductivity = static_cast<float>(0.685); // in g dry mass / sqm * day
    availableFoodShare = static_cast<float>(0.1);
    patchProductivity = grasslandProductivity * patchLength * patchLength * availableFoodShare;
    averageProductivity = patchProductivity;
    maxProductivity = patchProductivity;
    minProductivity = patchProductivity;
    meanFileValues = 0;

    // setup of the individuals:
    AddSpecies(aSpeciesList);
    this->statistics = new Statistics(species.size());
    initialNumberOfIndividuals = 1000;

    // build scenario:
    densityControl = aDensityControl;
    immigration = false;

//    if (currentRun % 3 == 0) {
//        homeRangeSampling = true;
//    } else {
//        homeRangeSampling = false;
//    }
    homeRangeSampling = false;

    print = false;
    needsSetup = true;
    finished = false;
}

// ------------------------------------------------------------------------------------------------//

Simulation::~Simulation()
{
    delete landscape;
    delete statistics;
    for (int i = 0; i < species.size(); i++) {
        delete species[i];
    }
    for (int i = 0; i < individuals.size(); i++) {
        delete individuals[i];
    }
}

// ------------------------------------------------------------------------------------------------//

void Simulation::Setup() {
    //generate a unique seed:
    auto now = std::chrono::steady_clock::now();
    auto now_ms = std::chrono::time_point_cast<std::chrono::milliseconds>(now);
    auto value = now_ms.time_since_epoch();
    seedMultiPurposeGenerator = static_cast<unsigned int>(value.count());
    if (print) {
        std::cout << "Init randomization with seed: " << seedMultiPurposeGenerator << std::endl;
    }
    rNG.seed(seedMultiPurposeGenerator);

    //clear all statistics:
    statistics->ClearAll();
    statistics->Tick();
    statistics->day = statistics->ticks % 365;

    // flow to setup the landscape:
    if (hurst > 1) {
        CreateRandomLandscape();
    } else {
        if (hurst > 0.9) {
            CreateClumpLandscape();
        } else {
            CreateLandscape();
        }
    }
    std::cout << "Landscape created" << std::endl;
    UpdateResourceAmount();
    std::cout << "Resources updated" << std::endl;
    // flow to setup the community:
    CreateIndividuals();
    std::cout << "Individuals created" << std::endl;
    std::shuffle(individuals.begin(), individuals.end(), rNG); // generate random order for home range search
    FindHomerangesForIndividuals();
    std::cout << "Individuals established home range" << std::endl;
    CalculateCommunityIndices();
    LetIndividualsDie();
}

// ------------------------------------------------------------------------------------------------//

void Simulation::run() {
    int started = time(nullptr);

    if (needsSetup) {
        Setup();
        needsSetup = false;
        if (mutex) emit dailyUpdate();
    }

    // define critical diversity loss for the CVA analysis
    int criticalDiversityLoss = species.size()/2;

    while (true) {
        if (mutex) mutex->lock();
        SimulateOneDay();
        if (mutex) mutex->unlock();
        if (mutex) emit dailyUpdate();
        // terminate simulation if condition is true
        if (!densityControl && (statistics->ticks > 6000 || statistics->speciesRichness < criticalDiversityLoss)) {
            PrepareSpeciesData();
            PrepareIndividualData();
            break;
        } else if (densityControl && statistics->ticks > runtime){
            break;
        }
    }

    PrepareData();
    duration = time(nullptr) - started;
    finished = true;
}

// ------------------------------------------------------------------------------------------------//

void Simulation::SimulateOneDay() {
    statistics->Tick();
    statistics->day = statistics->ticks % 365;
    currentMatrixMortality = maxMatrixMortality;

    //std::cout << "Day: " << statistics->day << " Mortality: " << currentMatrixMortality << std::endl;
    //clear daily statistics:
    statistics->Clear();

    //clear individuals of all proberties that need to change daily:
    for (uint i = 0; i < individuals.size(); i++) {
        individuals[i]->energyDeficiency = 0.0;           // all ind. start fresh in a new day

        // random home range positions of the individuals are deleted and sampled again (daily)
        for (uint j = 0; j < individuals[i]->homeRangePositionsX.size(); j++) {
            individuals[i]->homeRangePositionsX[j] = 999;
            individuals[i]->homeRangePositionsY[j] = 999;
        }
    }

    for (size_t i = 0; i < species.size(); i++) {
        species[i]->totalOffspring = 0;
        species[i]->successfulOffspring = 0;
    }

    // simulation flow:
    std::shuffle(individuals.begin(), individuals.end(), rNG);       // 1) randomize order of individuals
    UpdateResourceAmount();                                          // 2) update resources in landscape
    CheckHomerangesForIndividuals();                                 // 3) all individuals forage in their home ranges
    LifeHistory();                                                   // 4) life history contains aging, reproduction and mortality
    CalculateCommunityIndices();                                     // 5) statistical measures on the community are calculated
    LetIndividualsDie();                                             // 6) dead individuals of the day are removed from the vector
    PrepareDiversityData();

    if (densityControl && statistics->ticks >= (runtime - 300)) {
        PrepareSpeciesData();                                        // 8) community measures are calculated for the last month of a simulation
    }

    if (statistics->ticks >= (runtime - 15) && homeRangeSampling) {
        PrepareHomeRangeData();                                       // take home range samples for later MCP calculation over the last x days
    }

    if (statistics->ticks == runtime) {
        PrepareIndividualData();                                     // 9) individual statistics are reported once at the end of a simulation
    }
}

// ------------------------------------------------------------------------------------------------//
void Simulation::CreateRandomLandscape() {
    std::vector<GridCell*> gridVector;
    // prepare statistics on resource distribution
    float maxResource = 0;
    float minResource = 100;

    // store each grid cell in vector
    for (int y = 0; y < landscape->size; y++) {
        for (int x = 0; x < landscape->size; x++) {
            GridCell* cell = landscape->GetCell(x, y);
            gridVector.push_back(cell);
        }
    }
    // randomize order of grid cells
    std::shuffle(gridVector.begin(), gridVector.end(), rNG);

    for (int i = 0; i < gridVector.size(); i++) {
        GridCell* cell = gridVector[i];
        if (i < habitatPatches) {
            cell->fileValue = 1;
            cell->habitatTyp = Habitat;
            // determine random resource level in cell:
            std::normal_distribution<float> distribution(averageProductivity, 0.7);
            float resources = distribution(rNG);
            while (resources <= 0) {
                resources = distribution(rNG);
            }
            cell->scaledResourceAmount = resources;
            cell->resourceAmount = cell->scaledResourceAmount;
            assert(cell->resourceAmount >= 0);
            cell->riskLevel = 0;
        } else {
            cell->fileValue == 0;
            cell->habitatTyp = Matrix;
            cell->scaledResourceAmount = 0;
            cell->resourceAmount = cell->scaledResourceAmount;
            cell->riskLevel = 0;
        }

        if (cell->resourceAmount > maxResource) {
            maxResource = cell->resourceAmount;
        }
        if (cell->resourceAmount < minResource) {
            minResource = cell->resourceAmount;
        }
    }
    landscape->maxRL = maxResource;
}

// ------------------------------------------------------------------------------------------------//

void Simulation::CreateLandscape() {
    NeutralLandscape* n = new NeutralLandscape(hurst, sigma, GRID_SIZE);

    // reduce habitat amount to specified habitat level
    n->makeLandscape(habitatCover, 100, statistics);

    if (!n->sortedCoords) {
        NeutralLandscape::sortCoords(n);
    }

    // prepare statistics on risk distribution
    float maxRisk = 0;
    float minRisk = 1;
    float sumRisk = 0;

    // prepare statistics on resource distribution
    float maxResource = 0;
    float minResource = 100;
    float sumResource = 0;

    // parse one value to each grid cell
    int i = 0;
    for (int y = 0; y < landscape->size; y++) {
        for (int x = 0; x < landscape->size; x++) {
            GridCell* cell = landscape->GetCell(x, y);
            cell->fileValue = n->getVal(i);
            //std::cout << "fileValue: " << cell->fileValue << std::endl;

            if (cell->fileValue == 0) {
                cell->habitatTyp = Matrix;
                cell->resourceAmount = 0;
                cell->riskLevel = 0;
            } else {
                cell->habitatTyp = Habitat;
                // 1. determine risk level in cell:
                cell->riskLevel = 0; // 1 - (static_cast<float>(cell->fileValue / static_cast<float>(100.0)));
                //cell->riskLevel = (cell->riskLevel - rMin) / (rMax - rMin);
                //std::cout << "fileValue: " << cell->fileValue << " riskLevel: " << cell->riskLevel << std::endl;

                sumRisk += cell->riskLevel;
                if (cell->riskLevel > maxRisk) {
                    maxRisk = cell->riskLevel;
                }
                if (cell->riskLevel < minRisk) {
                    minRisk = cell->riskLevel;
                }

                // 2. determine random resource level in cell:
                std::normal_distribution<float> distribution(averageProductivity, 0.7);
                float resources = distribution(rNG);
                while (resources <= 0) {
                    resources = distribution(rNG);
                }
                cell->scaledResourceAmount = resources;
                cell->resourceAmount = cell->scaledResourceAmount;
                assert(cell->resourceAmount >= 0);

                sumResource += cell->resourceAmount;
                if (cell->resourceAmount > maxResource) {
                    maxResource = cell->resourceAmount;
                }
                if (cell->resourceAmount < minResource) {
                    minResource = cell->resourceAmount;
                }
            }
            //count cells for loop
            i++;
        }
    }
    landscape->maxRL = maxResource;

    if (print) {
        // output statistics on risk distribution
        // mean risk:
        float meanRisk = sumRisk / static_cast<float>(habitatPatches);
        std::cout << "meanRisk: " << meanRisk << std::endl;
        std::cout << "minRisk: " << minRisk << std::endl;
        std::cout << "maxRisk: " << maxRisk << std::endl;

        // output statistics on resource distribution
        // mean resource amount:
        float meanResource = sumResource / static_cast<float>(habitatPatches);
        std::cout << "meanResource: " << meanResource << std::endl;
        std::cout << "minResource: " << minResource << std::endl;
        std::cout << "maxResource: " << maxResource << std::endl;
    }
}

// ------------------------------------------------------------------------------------------------//

void Simulation::CreateClumpLandscape() {
    int cover = 0;

    float clump = hurst; //probability of empty patch to become shrub patch

    vector<GridCell*> gridCells = landscape->ShuffleGridCells();

    while (cover < habitatPatches) {
        for (int i = 0; i < gridCells.size(); i++) {
            GridCell* cell = gridCells[i];
            vector<GridCell*> matrixNeighbours = landscape->GetNeighboursWith(cell->x, cell->y,
                                                                              [] (const GridCell* cell) {return cell->fileValue == 0; });
            if (cell->fileValue > 0 && matrixNeighbours.size() > 0) {
               int randNeighbour = rand() % matrixNeighbours.size();
               matrixNeighbours[randNeighbour]->fileValue = 1;
               matrixNeighbours[randNeighbour]->habitatTyp = Habitat;
               cover ++;
               if (cover >= habitatPatches) {
                  return;
               }
            } else {
                float randomFloat = Utilities::RandomFloat(0,1);
                if (randomFloat < (1-clump) && cell->fileValue == 0) {
                    cell->fileValue = 1;
                    cell->habitatTyp = Habitat;
                    cover ++;
                }
                if (cover >= habitatPatches) {
                   return;
                }
            }
        }
    }
}

// ------------------------------------------------------------------------------------------------//

void Simulation::HabitatLoss() {

    std::vector<GridCell*> gridVector;
    gridVector.reserve(landscape->size * landscape->size);

    // vectorize habitat grid cells for sorting
    for (int y = 0; y < landscape->size; y++) {
        for (int x = 0; x < landscape->size; x++) {
            GridCell* cell = landscape->GetCell(x, y);
            if (cell->habitatTyp == Matrix) {
                continue;
            } else {
                gridVector.push_back(cell);
            }
        }
    }

    // habitat loss
    int lostPatches = static_cast<int>(0.1 * habitatPatches); // calculate how many patches are lost
    int newHabitatCover = statistics->actHabitatCover - lostPatches;
    if (newHabitatCover < lostPatches) {
        lostPatches = newHabitatCover;
    }

    // sort grid cells with ascending file Values (lower fileValues are lost first)
    std::sort(gridVector.begin(), gridVector.end(), [=] (GridCell* cell_i, GridCell* cell_j) {
        return (cell_i->fileValue < cell_j->fileValue);
    });

    for (int i = 0; i < lostPatches; i++) {
        gridVector[i]->fileValue = 0;
        gridVector[i]->habitatTyp = Matrix;
        gridVector[i]->riskLevel = 0;
    }
}

// ------------------------------------------------------------------------------------------------//

void Simulation::UpdateResourceAmount() {
    maxProductivity = 0;
    minProductivity = 100;

    for (int x = 0; x < landscape->size; x++) {
        for (int y = 0; y < landscape->size; y++) {
            auto cell = landscape->GetCell(x,y);
            assert(cell);

            if (cell->fileValue == 0) {
                cell->resourceAmount = 0;
                cell->scaledResourceAmount = 0;
            } else {
                std::normal_distribution<float> distribution(averageProductivity, 0.7);
                float resources = distribution(rNG);
                while (resources <= 0) {
                    resources = distribution(rNG);
                }
                cell->scaledResourceAmount = resources;
            }
            cell->resourceAmount = cell->scaledResourceAmount;
            Q_ASSERT(cell->resourceAmount >= 0);


            if (statistics->ticks == 2 && homeRangeSampling) {
                QList<QVariant> allColumnsMap;
                allColumnsMap << seedMultiPurposeGenerator << cell->x << cell->y << cell->resourceAmount;
                dataMap.addRow(allColumnsMap);
            }

            cell->speciesRichness = 0;
            cell->shannonDiversity = 0;
            cell->trueDiversity = 0;
            cell->occupancy = 0;
            for (size_t i = 0; i < species.size(); i++) {
                cell->occupantsSpecies[species[i]] = 0;
            }
            if (cell->resourceAmount > maxProductivity) {
                maxProductivity = cell->resourceAmount;
            }
            if (cell->resourceAmount < minProductivity && cell->resourceAmount > 0) {
                minProductivity = cell->resourceAmount;
            }
        }
    }
    landscape->maxRL = maxProductivity;
}

// ------------------------------------------------------------------------------------------------//

void Simulation::CreateIndividuals() {
    // prob.distribution with mass^-1.5, see Buchmann et al. 2012, Ecography
    uint speciesNumber = species.size();
    float sumBodymass = 0.0;
    float powerLawExponent = -1.5;
    std::vector<float> localBodymass = std::vector<float>(speciesNumber);

    for (uint i = 0; i < speciesNumber; i++) {
        sumBodymass += pow(species[i]->meanBodymass, powerLawExponent);
        int b = 0;
        if(i > 0) {
            b = localBodymass[i-1];
        }
        localBodymass[i] = b + pow(species[i]->meanBodymass, powerLawExponent);
    }
    this->individuals = std::vector<Individual*>(initialNumberOfIndividuals);  // initialise vector for pointers to each individual

    for (uint i = 0; i < initialNumberOfIndividuals; i++) {

        float randomFloat = Utilities::RandomFloat(0,1);
        uint initialSpecies = 0;

        for (uint i = 0; i < speciesNumber; i++) {
            if (randomFloat < (localBodymass[i] / sumBodymass)) {
                initialSpecies = i;
                break;
            }
        }
        ++statistics->individualsCounter;

        // individual is created belonging to one species, a body mass and a unique ID
        this->individuals[i] = new Individual(statistics->individualsCounter, species[initialSpecies],
                                              rNG() % 300, false, patchLength);

        // individual receives a seacrh factor (determines if the individual optimises the foraging strategy for smaller distances or higher food gain)
        this->individuals[i]->searchFactor = searchStrategy;

        // individual receives responsiveness
        float slope;
        switch (communityResponsiveness) {
        case 0:
            slope = Utilities::RandomFloat(-0.1, 0);
            break;
        case 1:
            slope = Utilities::RandomFloat(-1, -0.9);
            break;
        case 2:
            slope = Utilities::RandomFloat(-1, 0);
            break;
        }
        //this->individuals[i]->responsiveness.intercept = intercept;
        //assert(this->individuals[i]->responsiveness.intercept <= 1);
        // slope correlates with intercept
        this->individuals[i]->responsiveness.slope = slope; //1.0 + this->individuals[i]->responsiveness.intercept * -2.0;
        assert(this->individuals[i]->responsiveness.slope <= 0);
        // put individual into its behavioral type group
        this->individuals[i]->BT = floor(this->individuals[i]->responsiveness.slope * -10.0);
        if (this->individuals[i]->BT == 10) {
            this->individuals[i]->BT = 9;
        }
    }
}

// ------------------------------------------------------------------------------------------------//

void Simulation::SortIndividuals() {
    bool sortByBodymass = false;
    float sortingProbability = 0.0;
    if (sortByBodymass) {
        std::sort(individuals.begin(), individuals.end(), [=] (Individual* individual_i, Individual* individual_j) {
            return (individual_i->bodymass > individual_j->bodymass);
        });
        if (sortingProbability < 1.0) {
            auto maxIndividuals = individuals.size();
            for (uint i = 0; i < maxIndividuals - 1; i++) { // the last individual will just keep its position
                float randomFloat = Utilities::RandomFloat(0,1);
                if (randomFloat <= sortingProbability) {
                    continue;
                } else {
                    // uniform distribution of integers within specified range, guaranteed unbiased
                    int min = i + 1;
                    int max = maxIndividuals - 1;
                    assert(min <= max);
                    // get individual from a random position within specified range
                    std::uniform_int_distribution<int> uniformRange(min,max);
                    std::size_t oldPosition = uniformRange(rNG);
                    // transfer individual from current position to position i
                    // pushing all individuals behind one position further
                    std::size_t newPosition = i;
                    auto it = individuals.begin();
                    std::rotate(it + newPosition, it + oldPosition, it + oldPosition + 1);
                }
            }
        }
    } else {
        std::shuffle(individuals.begin(), individuals.end(), rNG);
    }
}

// ------------------------------------------------------------------------------------------------//

void Simulation::FindHomerangesForIndividuals() {
    for (Individual* individual: individuals) {
        individual->FindHomeRangeGraph(landscape, patchLength, nullptr, currentMatrixMortality);
    }
}

// ------------------------------------------------------------------------------------------------//

void Simulation::CheckHomerangesForIndividuals() {
    for (Individual* individual: individuals) {
        individual->CheckHomeRangeGraph(landscape, patchLength, currentMatrixMortality);
        // sample 1-2 random positions of the home range to use for later MCP calculation
        // RandomPositionSampling(individual->homeRangeCells, aPatchLength);
    }
}

// ------------------------------------------------------------------------------------------------//

void Simulation::LetIndividualsDie() {
    Grid* grid = landscape;
    individuals.erase(std::remove_if(individuals.begin(),
                                     individuals.end(),
                                     [&](Individual* individualX){
                          if (individualX->dead) {

                              // declare the former central place of a dead individual as empty again
                              GridCell* coreCell = grid->GetCell(individualX->location_x, individualX->location_y);
                              if (coreCell) {
                                  coreCell->centralPlace = Empty;
                              }

                              // delete the dead individual from vector
                              delete individualX;
                              return true;
                          }
                          return false;
                      }),
            individuals.end());
}

// ------------------------------------------------------------------------------------------------//

void Simulation::LifeHistory() {
    auto size = individuals.size();

    for (uint i = 0; i < size; i++) {
        Individual* individual = individuals[i];

        // Ignore dead individuals
        if (individual->dead) {
            continue;
        }

        // get one day older
        individual->age += 1;

        // Pregnancy and offspring:
        if (individual->sex == Female) { 
            GetPregnant(individual);
            //end of lactation period offspring is received and pregnancyStatus set to NotPregnant
            if (individual->pregCount > (individual->gestationPeriod + individual->lactationPeriod)) {
                GetOffspring(individual);
            }
        }

        // Mortality
        // 1) age-related:
        Senility(individual);
        // 2) natural mortality:
        NaturalMortality(individual);
        // 4) density-related:
        if (densityControl) {
            Density(individual);
        }
    }
}


// ------------------------------------------------------------------------------------------------//

void Simulation::GetPregnant(Individual* individual) {
    // check for non pregnant females if conditions are right for pregnancy
    if (individual->pregnancyStatus == NotPregnant) {
        if (individual->age > individual->ageFirstReproduction) {
            individual->pregCount = 1; // female gets pregnant
        }
    } else {
        individual->pregCount += 1;
    }

    // if female is in a pregnancy pregnancyStatus is updated based on the number of days
    if (individual->pregCount > 0) {
        if (individual->pregCount < individual->gestationPeriod) {
            individual->pregnancyStatus = Gestating;
        } else if (individual->pregCount < individual->gestationPeriod + individual->lactationPeriod) {
            individual->pregnancyStatus = Lactating;
        }
    }
}

// ------------------------------------------------------------------------------------------------//

void Simulation::GetOffspring(Individual* individual) {
    // days of gestation/pregnancy PLUS days of lactation
    if (!(individual->pregCount > (individual->gestationPeriod + individual->lactationPeriod))) {
        return;
    }

    // determine number of offsprings:
    int litterSize = individual->litterSize;
    assert(litterSize != 0);

    for (int i = 0; i < litterSize; i++) {
        ++statistics->individualsCounter;

        // a new individual is created
        Individual* offspring = new Individual(statistics->individualsCounter, individual->species,
                                               individual->lactationPeriod, true, patchLength);

        // inherit search factor from parent
        offspring->searchFactor = individual->searchFactor;

        // individual inherit responsiveness from parent with std dev
        // individual receives responsiveness
        float slope;
        switch (communityResponsiveness) {
        case 0:
            slope = Utilities::RandomFloat(-0.1, 0);
            break;
        case 1:
            slope = Utilities::RandomFloat(-1, -0.9);
            break;
        case 2:
            std::normal_distribution<float> distribution(individual->responsiveness.slope, heritability);
            slope = distribution(rNG);
            while (slope > 0 || slope < -1) {
                slope = distribution(rNG);
            }
            break;
        }

        offspring->responsiveness.slope = slope;
        assert(offspring->responsiveness.slope <= 0);
        // intercept correlates with slope
        // offspring->responsiveness.intercept = (slope - static_cast<float>(1.0)) / static_cast<float>(-2.0);
        // assign offspring to its behavioral type group
        offspring->BT = floor(offspring->responsiveness.slope * -10.0);
        if (offspring->BT == 10) {
            offspring->BT = 9;
        }

        // for calibration of heritability:
        //std::cout << currentRun << "," << heritability << "," << individual->ID << "," << individual->responsiveness.slope << "," << offspring->responsiveness.slope << std::endl;

        //Find home range for offspring:
        offspring->FindHomeRangeGraph(landscape, patchLength, individual, currentMatrixMortality);

        ++species[individual->species->ID]->totalOffspring;
        if (!offspring->dead) {
            ++species[individual->species->ID]->successfulOffspring;
        }
        individuals.push_back(offspring);
    } // end loop through offspring

    individual->pregnancyStatus = NotPregnant;
    individual->pregCount = 0;
}

// ------------------------------------------------------------------------------------------------//

void Simulation::Senility(Individual* individual) {
    if (individual->dead) {
        return;
    }
    std::normal_distribution<float> distribution(individual->averageLifespan, (0.1 * individual->averageLifespan));
    float lifeSpan = distribution(rNG);

    if (individual->age > lifeSpan) {
        individual->Die();
        individual->deathAge = true;
        individual->deathCause = OldAge;
    }
}

// ------------------------------------------------------------------------------------------------//

void Simulation::NaturalMortality(Individual *individual) {
    if (individual->dead) {
        return;
    }

    float randomFloat = Utilities::RandomFloat(0,1);

    if (randomFloat <= individual->naturalMortalityRate) {
        individual->Die();
        individual->deathNatural = true;
        individual->deathCause = Natural;
    }
}

// ------------------------------------------------------------------------------------------------//

void Simulation::Density(Individual* individual) {
    if (individual->dead) {
        return;
    }

    float randomFloat = Utilities::RandomFloat(0,1);

    // if individuals get closer to their capacity, probability of being dead increases:
    if (randomFloat < (static_cast<float>(individual->species->abundance) / individual->capacity)) {
        individual->Die();
        individual->deathDensity = true;
        individual->deathCause = DensityDependence;
    }
}

// ------------------------------------------------------------------------------------------------//

void Simulation::AddSpecies(QList<QStringList>& speciesList) {

    // all species included in the simulation are created with the information
    // from the speciesList generated from the user-provided csv-file
    for (int i = 1; i < speciesList.size(); i++) {
        Species* newSpecies = new Species();

        // species are identified by ID:
        newSpecies->ID = i-1;

        if (speciesList[i][0].trimmed().isEmpty()) {
           newSpecies->speciesName = nullptr;
        } else {
            newSpecies->speciesName = speciesList[i][0].toStdString();
        }

        // species order:
        if (speciesList[i][1].trimmed().isEmpty()) {
           newSpecies->speciesOrder = nullptr;
        } else {
            newSpecies->speciesOrder = speciesList[i][1].toStdString();
        }

        // bodymass values:
        if (speciesList[i][2].trimmed().isEmpty()) {
            throw 1;
        } else {
            newSpecies->meanBodymass = speciesList[i][2].toFloat();
        }

        if (speciesList[i][3].trimmed().isEmpty()) {
            throw 2;
        } else {
            newSpecies->sdBodymass = speciesList[i][3].toFloat();
        }

        // maximum homerange size
        if (speciesList[i][4].trimmed().isEmpty()) {
             newSpecies->maxHomerange = 0;
        } else {
             newSpecies->maxHomerange = speciesList[i][4].toInt();
        }

        // average lifespan
        if (speciesList[i][5].trimmed().isEmpty()) {
             newSpecies->averageLifespan = 0;
        } else {
             newSpecies->averageLifespan = speciesList[i][5].toInt();
        }

        // litter size values
        if (speciesList[i][6].trimmed().isEmpty()) {
             newSpecies->meanLitterSize = 0;
        } else {
             newSpecies->meanLitterSize = speciesList[i][6].toInt();
        }

        if (speciesList[i][7].trimmed().isEmpty()) {
            newSpecies->sdLitterSize = 0;
        } else {
            newSpecies->sdLitterSize = speciesList[i][7].toInt();
        }

        // maturity values (age of first reproduction)
        if (speciesList[i][8].trimmed().isEmpty()) {
            newSpecies->meanAFR = 0;
        } else {
            newSpecies->meanAFR = speciesList[i][8].toFloat();
        }

        if (speciesList[i][9].trimmed().isEmpty()) {
            newSpecies->sdAFR = 0;
        } else {
            newSpecies->sdAFR = speciesList[i][9].toFloat();
        }

        // gestation length
        if (speciesList[i][10].trimmed().isEmpty()) {
            newSpecies->meanGestation = 0;
        } else {
            newSpecies->meanGestation = speciesList[i][10].toFloat();
        }

        if (speciesList[i][11].trimmed().isEmpty()) {
            newSpecies->sdGestation = 0;
        } else {
            newSpecies->sdGestation = speciesList[i][11].toFloat();
        }

         // lactation length
        if (speciesList[i][12].trimmed().isEmpty()) {
            newSpecies->meanLactation = 0;
        } else {
            newSpecies->meanLactation = speciesList[i][12].toFloat();
        }

        if (speciesList[i][13].trimmed().isEmpty()) {
            newSpecies->sdLactation = 0;
        } else {
            newSpecies->sdLactation = speciesList[i][13].toFloat();
        }

        species.push_back(newSpecies);
    }
}

// ------------------------------------------------------------------------------------------------//

void Simulation::CalculateCommunityIndices() {
    // count all individuals:
    statistics->individuals = individuals.size();
    if (statistics->individuals > statistics->maxCountIndividuals) {
        statistics->maxCountIndividuals = statistics->individuals;
    }

    // count all surviving individuals for statistics:
    statistics->survival = count_if(individuals.begin(),
                                    individuals.end(),
                                    [] (Individual*  individualX) {return !individualX->dead; });

    // get population abundance for all species, beavioral types and gamms types:
    this->behavioralTypes = std::vector<int>(10, 0);
    this->gammaTypes = std::vector<int>(10, 0);

    for (auto individual: individuals) {
        if (individual->dead) {
            continue;
        }
        // increase abundance of corresponding behavioral type
        ++behavioralTypes[individual->BT];

        // increase abundance of corresponding gamma type
        int gType = floor(individual->searchFactor * 10.0);
        if (gType == 10) {
            gType = 9;
        }
        ++gammaTypes[gType];
    }

    // 1) species richness: count all populations with an abundance > 0
    int speciesRichness = 0;
    int behavioralRichness = 0;
    int gammaRichness = 0;

    // 2) shannon index for species diversity and  behavioral diversity:
    Q_ASSERT(species.size() == behavioralTypes.size() && species.size() == gammaTypes.size());

    float shannonSpecies = 0;
    float shannonBT = 0;
    float shannonGamma = 0;

    float helperVariable = 0;
    float proportionX = 0;
    for (size_t i = 0; i < species.size(); i++) {
        // 1) Abundance per species:
        if (species[i]->abundance > 0) {
            speciesRichness ++;
            proportionX = static_cast<float>(species[i]->abundance) / static_cast<float>(statistics->survival);
            helperVariable = proportionX * log(proportionX);
            shannonSpecies = shannonSpecies + helperVariable;
        }

        // 2) Abundance per BT:
        if (behavioralTypes[i] > 0) {
            behavioralRichness ++;
            proportionX = static_cast<float>(behavioralTypes[i]) / static_cast<float>(statistics->survival); //static_cast<float>(statistics->survival);
            helperVariable = proportionX * log(proportionX);
            shannonBT = shannonBT + helperVariable;
        }

        // 3) Abundance per Gamma Type:
        if (gammaTypes[i] > 0) {
            gammaRichness ++;
            proportionX = static_cast<float>(gammaTypes[i]) / static_cast<float>(statistics->survival); //static_cast<float>(statistics->survival);
            helperVariable = proportionX * log(proportionX);
            shannonGamma = shannonGamma + helperVariable;
        }

        // 4) Starvation per Species

        // count dead individuals for starvation:
        int Species_i_starvation = count_if(individuals.begin(),
                                        individuals.end(),
                                        [&] (Individual*  individualX) {return
                individualX->dead &&
                individualX->deathCause == Starvation &&
                individualX->species == species[i]; });

        species[i]->starvation = static_cast<float>(Species_i_starvation) / static_cast<float>(species[i]->abundance) * 100;

        // 5) Starvation per BT

        // count dead individuals for starvation:
        int BT_i_starvation = count_if(individuals.begin(),
                                        individuals.end(),
                                        [&] (Individual*  individualX) {return
                individualX->dead &&
                individualX->deathCause == Starvation &&
                individualX->BT == i; });

        mortalityPerBT[i].BTstarvation = static_cast<float>(BT_i_starvation) / static_cast<float>(behavioralTypes[i]) * 100;

        // 6) Matrix mortality per Species

        // count dead individuals for matrix:
        int Species_i_matrix = count_if(individuals.begin(),
                                        individuals.end(),
                                        [&] (Individual*  individualX) {return
                individualX->dead &&
                individualX->deathCause == MatrixMortality &&
                individualX->species == species[i]; });

        species[i]->predation = static_cast<float>(Species_i_matrix) / static_cast<float>(species[i]->abundance) * 100;

        // 7) Matrix mortality per BT

        // count dead individuals for matrix:
        int BT_i_matrix = count_if(individuals.begin(),
                                        individuals.end(),
                                        [&] (Individual*  individualX) {return
                individualX->dead &&
                individualX->deathCause == MatrixMortality &&
                individualX->BT == i; });

        mortalityPerBT[i].BTmatrix = static_cast<float>(BT_i_matrix) / static_cast<float>(behavioralTypes[i]) * 100;
    }

    statistics->speciesRichness = speciesRichness;
    statistics->shannonDiversity = shannonSpecies * -1;
    statistics->behavioralDiversity = shannonBT * -1;
    statistics->gammaDiversity = shannonGamma * -1;

    // 3) shannon evenness:
    statistics->shannonEvenness = statistics->shannonDiversity / log(statistics->speciesRichness);
    statistics->behavioralEvenness = statistics->behavioralDiversity / log(behavioralRichness);
    statistics->gammaEvenness = statistics->gammaDiversity / log(gammaRichness);

    // 4) effective number of species:
    statistics->trueDiversity = exp(statistics->shannonDiversity);

    // record mean homerange diameter and bodymass for the community:
    if (statistics->survival == 0) {
        statistics->meanHRDiameter = 0;  // Undefined, really.
        statistics->meanBodymass = 0;  // Undefined, really.
    } else {
        int sumDiameter = 0.0;
        float sumBodymass = 0.0;
        for (auto individual: individuals) {
            if (individual->dead) {
                continue;
            }
            sumDiameter += individual->homerangeDiameter.back();
            sumBodymass += individual->bodymass;
        }
        // Mean diameter:
        statistics->meanHRDiameter = static_cast<float>(sumDiameter) / static_cast<float>(statistics->survival);
        // Mean bodymass:
        statistics->meanBodymass = sumBodymass / static_cast<float>(statistics->survival);
    }
}

// ------------------------------------------------------------------------------------------------//

void Simulation::CalculateCellIndices() {

    int sum = 0;
    int sumCellOccupancy = 0;
    int sumCellRichness = 0;
    float sumCellDiversity = 0;

    for (int x = 0; x < landscape->size; x++) {
        for (int y = 0; y < landscape->size; y++) {
            auto cell = landscape->GetCell(x,y);
            assert(cell);

            // 1) species richness: count all populations with an abundance > 0
            int richness = 0;
            // 2) shannon diversity index:
            float shannon = 0;
            for (size_t i = 0; i < species.size(); i++) {
                if (cell->occupantsSpecies[species[i]] > 0) {
                    richness ++;
                    float proportionSpeciesX = static_cast<float>(cell->occupantsSpecies[species[i]]) / static_cast<float>(cell->occupancy);
                    float helperVariable = proportionSpeciesX * log(proportionSpeciesX);
                    shannon = shannon + helperVariable;
                }
            }
            cell->speciesRichness = richness;
            cell->shannonDiversity = shannon * -1;
            // effective number of species:
            cell->trueDiversity = exp(cell->shannonDiversity);

            if (cell->habitatTyp == Matrix) {
                continue;
            } else {
                sum ++;
                sumCellOccupancy += cell->occupancy;
                sumCellRichness += cell->speciesRichness;
                sumCellDiversity += cell->trueDiversity;
            }
        }
    }

    statistics->meanOccupancy = static_cast<float>(sumCellOccupancy) / static_cast<float>(sum);

}

// ------------------------------------------------------------------------------------------------//

void Simulation::PrepareData() {
    // prepare data that you want to save to csv-file
    QString sHabitatAmount;
    sHabitatAmount.setNum(habitatCover, 'f', 2);

    QString sFragmentationLevel;
    int H = static_cast<int>(hurst * 100);
    switch (H) {
    case 10:
        sFragmentationLevel = "high";
        break;
    case 50:
        sFragmentationLevel = "moderate";
        break;
    case 90:
        sFragmentationLevel = "weak";
        break;
    case 200:
        sFragmentationLevel = "very high";
        break;
    default:
        sFragmentationLevel.setNum(hurst * 100000, 'f', 0);
    }

    QString sScenario;
    switch (communityResponsiveness) {
    case 0:
        sScenario = "risk-seeking";
        break;
    case 1:
        sScenario = "risk-avoiding";
        break;
    case 2:
        sScenario = "risk-diverse";
        break;
    }

    QString sSearchStrategy;
    sSearchStrategy.setNum(searchStrategy, 'f', 1);

    QString sMatrixMortality;
    sMatrixMortality.setNum(maxMatrixMortality, 'f', 5);

    QString sHeritability;
    sHeritability.setNum(heritability, 'f', 2);

    // data on species level measures
    QList<QVariant> allColumnsSimulation;
    allColumnsSimulation << seedMultiPurposeGenerator << currentRun
                         << static_cast<QString>(sHabitatAmount)
                         << static_cast<QString>(sFragmentationLevel)
                         << static_cast<QString>(sScenario)
                         << static_cast<bool>(landscape->matrixCosts)
                         << static_cast<bool>(landscape->matrixMortality)
                         << static_cast<QString>(sMatrixMortality);
    dataSimulation.addRow(allColumnsSimulation);
}

// ------------------------------------------------------------------------------------------------//

void Simulation::PrepareSpeciesData() {
    // prepare data that you want to save to csv-file
    // data on species
    QString sJuvenileSuccess;
    QString sSpeciesStarvation;
    QString sSpeciesMatrix;
    QString sBTStarvation;
    QString sBTMatrix;

    for (size_t i = 0; i < species.size(); i++) {

        // calculate the mean juvenile success rate for a species
        float success = static_cast<float>(species[i]->successfulOffspring) /
                        static_cast<float>(species[i]->totalOffspring) * 100.0;

        sJuvenileSuccess.setNum(success, 'f', 2);
        sSpeciesStarvation.setNum(species[i]->starvation, 'f', 3);
        sSpeciesMatrix.setNum(species[i]->predation, 'f', 3);
        sBTStarvation.setNum(mortalityPerBT[i].BTstarvation, 'f', 3);
        sBTMatrix.setNum(mortalityPerBT[i].BTmatrix, 'f', 3);

        QList<QVariant> allColumnsSpecies;
        allColumnsSpecies << seedMultiPurposeGenerator << statistics->ticks
                             << static_cast<int>(i)
                             << static_cast<int>(species[i]->abundance)
                             << static_cast<QString>(sJuvenileSuccess)
                             << static_cast<QString>(sSpeciesStarvation)
                             << static_cast<QString>(sSpeciesMatrix)
                             << static_cast<QString>(sBTStarvation)
                             << static_cast<QString>(sBTMatrix);

        dataSpecies.addRow(allColumnsSpecies);
    }
}

// ------------------------------------------------------------------------------------------------//

void Simulation::PrepareDiversityData() {
    // prepare data that you want to save to csv-file
    // data on landscape
    QString sENS;
    sENS.setNum(exp(statistics->shannonDiversity), 'f', 2);

    CalculateCellIndices();
    QString sHabitatCompetition;
    sHabitatCompetition.setNum(statistics->meanOccupancy, 'f', 2);

    QList<QVariant> allColumnsDiversity;
    allColumnsDiversity << seedMultiPurposeGenerator << statistics->ticks
                        << static_cast<int>(statistics->speciesRichness) << static_cast<QString>(sENS)
                        << static_cast<int>(statistics->survival) << static_cast<QString>(sHabitatCompetition);
    dataDiversity.addRow(allColumnsDiversity);

}

// ------------------------------------------------------------------------------------------------//

void Simulation::PrepareIndividualData() {
    QString sResponsivness;   
    QString sBodymass;
    QString sMaxDistance;
    //QString sDailyDistance;
    QString sDiameter;
    QString sArea;
    QString sHabitat;
    QString sMatrix;

    QString sCrossing;
    float inM = static_cast<float>(patchLength);

    for (int j = 0; j < individuals.size(); j++) {
        Individual* i = individuals[j];

        sResponsivness.setNum(i->responsiveness.slope * -1.0, 'f', 3);
        sBodymass.setNum(i->bodymass * 1000, 'f', 3);
        sMaxDistance.setNum(Utilities::MeanVectorFloat(i->longestForagingBout) * inM, 'f', 2);
        //sDailyDistance.setNum(Utilities::MeanVectorFloat(i->dailyMovement) * inM, 'f', 2);
        sDiameter.setNum(Utilities::MeanVectorInt(i->homerangeDiameter) * inM, 'f', 1);
        sArea.setNum(Utilities::MeanVectorInt(i->homeRangeSize), 'f', 1);
        sHabitat.setNum(Utilities::MeanVectorInt(i->habitatCells), 'f', 1);
        sMatrix.setNum(Utilities::MeanVectorInt(i->matrixCells), 'f', 1);
        sCrossing.setNum(Utilities::MeanVectorFloat(i->firstMatrixCrossing) * 100, 'f', 1);

        QList<QVariant> allColumnsIndividuals;
        allColumnsIndividuals << seedMultiPurposeGenerator << statistics->ticks << static_cast<int>(i->ID)
                              << static_cast<QString>(sResponsivness) << static_cast<QString>(sBodymass)
                              << static_cast<int>(i->species->ID)
                              //<< i->location_x << i->location_y
                              << static_cast<QString>(sArea) << static_cast<QString>(sMatrix)
                              << static_cast<QString>(sHabitat) << static_cast<QString>(sDiameter)
                              << static_cast<QString>(sMaxDistance) //<< static_cast<QString>(sDailyDistance)
                              << static_cast<QString>(sCrossing);

        dataIndividuals.addRow(allColumnsIndividuals);
    }
}

// ------------------------------------------------------------------------------------------------//

void Simulation::PrepareHomeRangeData() {

    QString sResponsivness;
    QString sBodymass;
    QString X;
    QString Y;

    for (int j = 0; j < individuals.size(); j++) {
        Individual* i = individuals[j];

        sResponsivness.setNum(i->responsiveness.slope * -1.0, 'f', 3);
        sBodymass.setNum(i->bodymass * 1000, 'f', 3);

        for (int n = 0; n < i->nSamples; n++) {

            X.setNum(i->homeRangePositionsX[n], 'f', 4);
            Y.setNum(i->homeRangePositionsY[n], 'f', 4);

            QList<QVariant> allColumnsHomeRanges;
            allColumnsHomeRanges << seedMultiPurposeGenerator << statistics->ticks << static_cast<int>(i->ID)
                                 << static_cast<int>(i->species->ID) << static_cast<QString>(sResponsivness)
                                 << static_cast<QString>(sBodymass)
                                 << static_cast<QString>(X) << static_cast<QString>(Y);
            dataHomeRanges.addRow(allColumnsHomeRanges);
        }
    }
}


