#include "qtcsv/variantdata.h"
#include "qtcsv/stringdata.h"
#include "qtcsv/reader.h"
#include "qtcsv/writer.h"
#include "simulation.h"
#include "utilities.h"
#include <time.h>
#include <assert.h>
#include <iostream>
#include <QCommandLineParser>
#include <QThreadPool>
#include "neutrallandscape.h"
#include <QDebug>
#include <QFile>
#include <array>
#include <random>
#include <QCoreApplication>

int main(int argc, char* argv[])
{
    QCoreApplication a(argc, argv);

    QCommandLineParser parser;

    parser.addHelpOption();

    parser.addOptions({
        // An option with a value
        {{"f", "fragmentation"},
           "Level of fragmentation as Hurst factor (0.1 = high, 0.5 = intermediate, 0.9 = low and 2 = random distribution [very high fragmentation]).)", "float between 0 and 1"},
        {{"b", "boldness"},
           "Specify behavioural scenario for the community (0 = homogeneous risk-seeking, 1 = homogeneous risk-avoiding or 2 = heterogeneous risk-taking).", "integer"},
        {{"g", "deferredCosts"},
           "Specify scenario of deferred matrix costs.", "bool"},
        {{"e", "matrixMortality"},
           "Specify the mortality probability per visited matrix cell.", "float"},
        {{"i", "searchFactor"},
           "Specify the balance between maximizing food gain and minimizing distances while foraging.", "float"},
        {{"c", "cover"},
           "Specify fraction of landscape that is covered with habitat cells.", "integer between 0 and 100"},
        {{"d", "densityControl"},
           "Specify negative density dependence.", "bool"},
        {{"p", "patchLength"},
           "Specify length and width of one landscape cell", "integer"},
        {{"r", "repetitions"},
           "Number of repetitions of the same scenario in a different landscape.", "integer"},
        {{"t", "simulationTime"},
           "Temporal scale of a simulation in days.", "integer"},
        {{"s", "species"},
           "Take species input data from <file>.", "file"},
        {{"o", "filenameout"},
           "Declare <file> to save output data.", "file"},
    });

    // Process the actual command line arguments given by the user
    parser.process(a);

    // parse options from the command line
    std::cout << "parse options from the command line" << std::endl;

    // parse file path for input data:
    QString speciesFileTemplate = parser.value("species");
    std::cout << "Info for species is taken from file"  << std::endl;

    // parse file path for output data:
    QString fileNameOut = parser.value("filenameout");
    //std::cout << "filename for data: " << fileNameOut.toLatin1().constData() << std::endl;
    QString fileSpeciesOut = QString(fileNameOut);
    fileSpeciesOut.replace(".csv", "-species.csv");
    QString fileDiversityOut = QString(fileNameOut);
    fileDiversityOut.replace(".csv", "-community.csv");
    QString fileIndividualsOut = QString(fileNameOut);
    fileIndividualsOut.replace(".csv", "-individuals.csv");
    QString fileHomeRangesOut = QString(fileNameOut);
    fileHomeRangesOut.replace(".csv", "-hr.csv");
    QString fileMapOut = QString(fileNameOut);
    fileMapOut.replace(".csv", "-map.csv");

    // parse repetitions:
    int landscapeIteration  = parser.value("repetitions").toInt();

    // parse simulation time:
    int simulationTime  = parser.value("simulationTime").toInt();

    // parse scenario of density control:
    QStringList densityList = parser.value("densityControl").split(',');
    uint dimensionDensityList = static_cast<uint>(densityList.count());
    std::vector<int> densityControl;
    densityControl.reserve(dimensionDensityList);
    for (int i = 0; i < dimensionDensityList; i++) {
        densityControl.push_back(densityList[i].toInt());
    }

    // parse size of one landscape cell:
    QStringList patchLengthList = parser.value("patchLength").split(',');
    uint dimensionPatchLengthList = static_cast<uint>(patchLengthList.count());
    std::vector<int> patchLength;
    patchLength.reserve(dimensionPatchLengthList);
    for (int i = 0; i < dimensionPatchLengthList; i++) {
        patchLength.push_back(patchLengthList[i].toInt());
    }

    // parse scenario of fragmentation:
    QStringList fragmentationList = parser.value("fragmentation").split(',');
    uint dimensionfragmentationList = static_cast<uint>(fragmentationList.count());
    std::vector<float> fragmentationLevel;
    fragmentationLevel.reserve(dimensionfragmentationList);
    for (int i = 0; i < dimensionfragmentationList; i++) {
        fragmentationLevel.push_back(fragmentationList[i].toFloat());
    }

    // parse responsiveness:
    QStringList boldnessShareList = parser.value("boldness").split(',');
    uint dimensionBoldnessShare = static_cast<uint>(boldnessShareList.count());
    std::vector<int> boldnessShare;
    boldnessShare.reserve(dimensionBoldnessShare);
    for (int i = 0; i < dimensionBoldnessShare; i++) {
        boldnessShare.push_back(boldnessShareList[i].toInt());
    }

    // parse mortality probability in matrix cells:
    QStringList mortalityList = parser.value("matrixMortality").split(',');
    uint dimensionMortalityList = static_cast<uint>(mortalityList.count());
    std::vector<float> matrixMortality;
    matrixMortality.reserve(dimensionMortalityList);
    for (int i = 0; i < dimensionMortalityList; i++) {
        matrixMortality.push_back(mortalityList[i].toFloat());
    }

    // parse search factor:
    QStringList searchList = parser.value("searchFactor").split(',');
    uint dimensionSearchList = static_cast<uint>(searchList.count());
    std::vector<float> searchScenario;
    searchScenario.reserve(dimensionSearchList);
    for (int i = 0; i < dimensionSearchList; i++) {
        searchScenario.push_back(searchList[i].toFloat());
    }

    // parse habitat fraction:
    QStringList habitatList = parser.value("cover").split(',');
    uint dimensionHabitat = static_cast<uint>(habitatList.count());
    std::vector<int> habitat;
    habitat.reserve(dimensionHabitat);
    for (int i = 0; i < dimensionHabitat; i++) {
        habitat.push_back(habitatList[i].toInt());
    }

    // parse option of deferred matrix costs:
    QStringList matrixCostsList = parser.value("deferredCosts").split(',');
    uint dimensionMatrixCostsList = static_cast<uint>(matrixCostsList.count());
    std::vector<int> matrixCosts;
    matrixCosts.reserve(dimensionMatrixCostsList);
    for (int i = 0; i < dimensionMatrixCostsList; i++) {
        matrixCosts.push_back(matrixCostsList[i].toInt());
    }

    //check that species file exists
    QFile file(speciesFileTemplate);
    if (!file.open(QIODevice::ReadOnly | QIODevice::Text)) {
        qCritical() << "Could not open species file" << speciesFileTemplate;
        return 1;
    }

    // prepare input data for species:
    //qDebug() << "prepare input data";
    QList<QStringList> speciesList = QtCSV::Reader::readToList(speciesFileTemplate);
    // TODO: check that species list has at least one element

    // initialise csv-files with header:
    QtCSV::VariantData dummyData;
    dummyData.addEmptyRow();

    QStringList columnHeaderSimulation;
    columnHeaderSimulation << "Seed" << "Run" << "HabitatAmount" << "Fragmentation"
                           << "Behaviour" << "MatrixCosts" << "MatrixMortality" << "MortalityChance";
    QtCSV::Writer::write(fileNameOut, dummyData, ",", "\"", QtCSV::Writer::WriteMode::REWRITE, columnHeaderSimulation);

    QStringList columnHeaderSpecies;
    columnHeaderSpecies << "Seed" << "Day" << "Species" << "Abundance" << "JuvenileSuccess"
                        << "SpeciesStarvation" << "SpeciesMatrix" << "BTStarvation" << "BTMatrix";
    QtCSV::Writer::write(fileSpeciesOut, dummyData, ",", "\"", QtCSV::Writer::WriteMode::REWRITE, columnHeaderSpecies);

    QStringList columnHeaderDiversity;
    columnHeaderDiversity << "Seed" << "Day" << "Richness" << "ENS" << "#Individuals" << "Competition";
    QtCSV::Writer::write(fileDiversityOut, dummyData, ",", "\"", QtCSV::Writer::WriteMode::REWRITE, columnHeaderDiversity);

    QStringList columnHeaderIndividuals;
    columnHeaderIndividuals << "Seed" << "Day" << "ID" << "RiskTaking" << "Bodymass" << "Species" << "HRCells"
                            << "MatrixCells" << "HabitatCells" << "Diameter" << "MaxDist" << "firstCrossing";
    QtCSV::Writer::write(fileIndividualsOut, dummyData, ",", "\"", QtCSV::Writer::WriteMode::REWRITE, columnHeaderIndividuals);

    QStringList columnHeaderHomeRanges;
    columnHeaderHomeRanges << "Seed" << "Day" << "ID" << "Species"
                           << "RiskTaking" << "Bodymass" << "X" << "Y";
    QtCSV::Writer::write(fileHomeRangesOut, dummyData, ",", "\"", QtCSV::Writer::WriteMode::REWRITE, columnHeaderHomeRanges);

    QStringList columnHeaderMap;
    columnHeaderMap << "seed" << "x" << "y" << "value";
    QtCSV::Writer::write(fileMapOut, dummyData, ",", "\"", QtCSV::Writer::WriteMode::REWRITE, columnHeaderMap);

    std::vector<float> heritability{0.55};

    // initialise vector for all simulation threads (one thread = one repetition of one combination of parameters cover and resourceShare):
    uint numberThreads = habitat.size() * fragmentationLevel.size() * densityControl.size() *
                         patchLength.size() * boldnessShare.size() * matrixCosts.size() *
                         heritability.size() * matrixMortality.size() *
                         searchScenario.size() * landscapeIteration;
    std::vector<Simulation*> parallelSimulations;
    parallelSimulations.reserve(numberThreads);
    //disable expiring threads in ThreadPool:
    QThreadPool::globalInstance()->setExpiryTimeout(-1);
    QThreadPool::globalInstance()->setMaxThreadCount(50);
    std::cout << "ideal thread count = " << QThread::idealThreadCount() << " number of threads: " << numberThreads << std::endl;

    int start = time(nullptr);
    int run = 0;
    int n = 0;
    std::cout << "Creating the following threads: " << std::endl;
    // 1) loop through habitat amount
    for (uint a = 0; a < habitat.size(); a++) {
        // 2) loop through fragmentation level
        for (uint b = 0; b < fragmentationLevel.size(); b++) {
            // 3) loop through density control scenario
            for (uint c = 0; c < densityControl.size(); c++) {
                // 4) loop through spatial resolution
                for (uint d = 0; d < patchLength.size(); d++) {
                    // 5) loop through boldness fraction
                    for (uint e = 0; e < boldnessShare.size(); e++) {
                        for (uint h = 0; h < matrixCosts.size(); h++) {
                            for (uint j = 0; j < matrixMortality.size(); j++) {
                                for (uint k = 0; k < searchScenario.size(); k++) {
                                    for (uint m = 0; m < heritability.size(); m++) {
                                        // 6) loop through all landscape repetitions
                                        for (int g = 0; g < landscapeIteration; g++) {
                                            run = g + 1;
                                            // initialise Simulation:
                                            Simulation* simulation = new Simulation(nullptr, speciesList, run, habitat[a], fragmentationLevel[b],
                                                                                    densityControl[c], patchLength[d], boldnessShare[e],
                                                                                    matrixCosts[h], matrixMortality[j],
                                                                                    searchScenario[k], heritability[m]);
                                            simulation->runtime = simulationTime;
                                            n++;
                                            std::cout << "thread " << n << std::endl;
                                            parallelSimulations.push_back(simulation);
                                            simulation->setAutoDelete(false);
                                            QThreadPool::globalInstance()->start(simulation);
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
    }

    int done = 0;
    while (done < numberThreads) {
        // A little breathing time for the main thread.
        QThread::sleep(10);

        for (uint i = 0; i < parallelSimulations.size(); i++) {
            // run Simulation:
            Simulation* simulation = parallelSimulations[i];
            // check if thread has already been deleted:
            if (!simulation) {
                continue;
            }
            // check if thread is already finished
            if (simulation->finished) {
                done++;
                std::cout << "writing data" << std::endl;
                QtCSV::Writer::write(fileNameOut, simulation->dataSimulation, ",", "\"", QtCSV::Writer::WriteMode::APPEND);
                //std::cout << "writing data on species" << std::endl;
                QtCSV::Writer::write(fileSpeciesOut, simulation->dataSpecies, ",", "\"", QtCSV::Writer::WriteMode::APPEND);
                //std::cout << "writing data on landscape" << std::endl;
                QtCSV::Writer::write(fileDiversityOut, simulation->dataDiversity, ",", "\"", QtCSV::Writer::WriteMode::APPEND);
                //std::cout << "writing data on individuals" << std::endl;
                QtCSV::Writer::write(fileIndividualsOut, simulation->dataIndividuals, ",", "\"", QtCSV::Writer::WriteMode::APPEND);

                QtCSV::Writer::write(fileHomeRangesOut, simulation->dataHomeRanges, ",", "\"", QtCSV::Writer::WriteMode::APPEND);
                //std::cout << "writing data on map" << std::endl;
                QtCSV::Writer::write(fileMapOut, simulation->dataMap, ",", "\"", QtCSV::Writer::WriteMode::APPEND);

                //std::cout << time(nullptr) - start << " finished thread for resources: " << simulation->availableFoodShare << " time needed : " << simulation->duration << std::endl;
                delete simulation;
                parallelSimulations[i] = nullptr;
            }
        }
        std::cout << time(nullptr) - start << " finished " << done << " of " << numberThreads << " threads " << std::endl;
        //std::cout << time(nullptr) - start << " currently running " << QThreadPool::globalInstance()->activeThreadCount() << " threads " << std::endl;
    }
    return 0;
}
