#ifndef INDIVIDUAL_H
#define INDIVIDUAL_H
#include "grid.h"
#include "species.h"
#include "statistics.h"
#include <random>
#include <queue>
#include <set>

// seed distribution generator:
extern thread_local std::mt19937 rNG;

struct BodyMassValues {
    float mean;
    float sd;
};

struct Responsiveness {
    //float intercept;
    float slope;
};

enum MortalityCause{
    Starvation,         // 0
    NoCoreCell,         // 1
    FailedHR,           // 2
    MatrixMortality,    // 3
    OldAge,             // 4
    DensityDependence,  // 5
    Natural             // 6
};

enum Gender{
    Female,
    Male
};

enum PregnancyStatus{
    NotPregnant,
    Gestating,
    Lactating
};

enum FoodPreference{
    Omnivore,
    Herbivore,
    Granivore
};

enum ForagingType{
    CentralPlaceForager,
    PatrollingForager,
    NomadicForager
};

class Individual
{
public:
    Individual(int aID, Species* aSpecies, const int aAge,
               bool aOffspring, int aPatchLength);

    // Member attributes:
    int ID;
    Species* species;
    Responsiveness responsiveness;
    int BT;
    float searchFactor;
    float bodymass;
    int location_x;
    int location_y;
    int maxHomeRangeRadius;
    float feedingRate;                           // daily minimum feeding rate that needs to be fullfilled
    float energyDeficiency;                      // deviation of found resources to ectually needed resources
                                                 // if an individual encounters  a resource shortcoming in its home range
    float foodShare;                             // allometric factor of food exploitation
    float energyExpenditurePerOffspring;         // absolute increase in resource need per offspring
    float locomotionCost;
    float maxLocomotionCost;
    int maxNatalDispersal;
    int age;
    bool dead;
    MortalityCause deathCause;
    bool deathFood;
    bool deathAge;
    bool deathNatural;
    bool deathDensity;
    float capacity;
    Gender sex;
    float shelterUse;                             // index 0..1 of shelter use/need
    FoodPreference foodPreference;                // food preference; not distinguished yet
    ForagingType foragingType;                    // type of forage movement; not distinguished yet
    int averageLifespan;                          // average lifespan
    float naturalMortalityRate;                   // body mass dependent natural mortality rate
    int pregCount;                                // counts times of pregnancies
    int ageFirstReproduction;                     // age of first reproduction
    int litterSize;                               // average number of offspring produced at one birth
    int gestationPeriod;                          // gestation period
    int lactationPeriod;                          // lactation period
    float fastingEndurance;                       // time of survival during periods of resource shortage / energy crisis
    int resourceShortage;                         // maps consecutive days with a food shortage within the home range
    bool foodShortageInHomerange;                 // indicating if an individual can not find enough food in its home range
    PregnancyStatus pregnancyStatus;
    int nSamples;                                 // defines how many positions in a home range are sampled per day
    std::vector<float> homeRangePositionsX;       // store random x positions within the home range to calculate MCP later
    std::vector<float> homeRangePositionsY;       // store random y positions within the home range to calculate MCP later
    std::vector<GridCell*> homeRangeCells;
    std::vector<int> matrixCells;
    std::vector<int> habitatCells;
    std::vector<float> firstMatrixCrossing;
    std::vector<int> homerangeDiameter;
    std::vector<int> homeRangeSize;               // area of the current home range in grid cells
    std::vector<float> longestForagingBout;
    std::vector<float> dailyMovement;

    // Member methods:
    float DetermineBodymass();
    int CalculateMaxHomerange(int aPatchLength);
    float CalculateLocomotionCost(int aPatchLength);
    int CalculateMaxNatalDispersal(int aPatchLength);
    float CalculateFeedingRate();
    float CalculateFoodShare();
    int CalculateAverageLifespan();
    int CalculateAgeFirstReproduction();
    int CalculateLitterSize();
    int CalculateGestationPeriod();
    int CalculateLactationPeriod();
    float CalculateReproductiveCosts();
    float CalculateFastingEndurance();
    float CalculateCapacity(int aPatchLength);
    float CalculateNaturalMortality();
    void Die();
    float DetermineShelterUse();
    Gender DetermineSex();
    FoodPreference DetermineFoodPreference();
    ForagingType DetermineForagingType();

    void RandomPositionSampling(std::vector<GridCell*> homeRangeCells, int aPatchLength, int nSamples);      // function to randomly sample some grid cells from the home range
    GridCell* FindCoreCell(Grid* landscape, Individual* parent);
    std::tuple<bool, float> AssessHomeRangeMatrixCrossing(Grid* landscape, GridCell* coreCell, float mortalityChance, float minFeed, bool searching);
    float UpdateFeedingRate();
    void FindHomeRangeGraph(Grid* landscape, int patchLength, Individual* parent, float mortalityChance);
    void CheckHomeRangeGraph(Grid* landscape, int patchLength, float mortalityChance);
};

#endif // INDIVIDUAL_H
