#include "individual.h"
#include "utilities.h"
#include "statistics.h"
#include "simulation.h"
#include <assert.h>
#include <cmath>
#include <vector>
#include <iostream>
#include <string>
#include <map>
#include <QtAlgorithms>
#include <QtDebug>

using namespace std;

// ------------------------------------------------------------------------------------------------//

Individual::Individual(int aID, Species* aSpecies, const int aAge,
                       bool aOffspring, int aPatchLength)      // Constructor for one individual
{
    ID = aID;
    species = aSpecies;  
    bodymass = DetermineBodymass();
    sex = DetermineSex();
    shelterUse = DetermineShelterUse();
    foodPreference = DetermineFoodPreference();
    foragingType = DetermineForagingType();

    // initialise vector for pointers to random grid cells of the individual's home range
    nSamples = 3;
    this->homeRangePositionsX = std::vector<float>(nSamples);
    this->homeRangePositionsY = std::vector<float>(nSamples);

    averageLifespan = CalculateAverageLifespan();                         // average lifespan (needs to be calculated before setting the age of an individual)

    if (aOffspring) {
        // offspring gets age as the lactation time of their mother via the contructor
        age = aAge;
    } else {
        // age is randomly drawn from the lifespan
        age = rNG() % averageLifespan;
    }

    litterSize = CalculateLitterSize();                                    // litter size
    maxHomeRangeRadius = CalculateMaxHomerange(aPatchLength);              // maximum home range size
    locomotionCost = CalculateLocomotionCost(aPatchLength);                // movement costs
    maxLocomotionCost = static_cast<float>(maxHomeRangeRadius) * locomotionCost;
    ageFirstReproduction = CalculateAgeFirstReproduction();                // age of first reproduction
    gestationPeriod = CalculateGestationPeriod();                          // length of gestation period
    lactationPeriod = CalculateLactationPeriod();                          // length of lactation period
    fastingEndurance = CalculateFastingEndurance();                        // survival time under resource shortage
    maxNatalDispersal = CalculateMaxNatalDispersal(aPatchLength);          // maximum natal dispersal distance
    capacity = CalculateCapacity(aPatchLength);                                        // species landscape capacity (only relevant for NDD scenarios)
    naturalMortalityRate = CalculateNaturalMortality();                    // allometric (body mass dependent) mortality rate
    feedingRate = CalculateFeedingRate();                                  // daily feeding rate
    foodShare = CalculateFoodShare();                                      // allometric factor of food exploitation
    energyExpenditurePerOffspring = CalculateReproductiveCosts();

    dead = false;
    species->abundance++;

    deathFood = false;
    deathAge = false;
    deathNatural = false;
    deathDensity = false;
    pregCount = 0;
    resourceShortage = 0;
    energyDeficiency = 0;
    foodShortageInHomerange = false;
    pregnancyStatus = NotPregnant;

    // initial female individuals older than ageFirstReproduction are assigned a random state of pregnancy
    if (!aOffspring && sex == Female && age >= ageFirstReproduction) {
        int max = gestationPeriod + lactationPeriod;
        pregCount = rNG() % max + 1;
        if (pregCount < gestationPeriod) {
            pregnancyStatus = Gestating;
        } else if (pregCount < gestationPeriod + lactationPeriod) {
            pregnancyStatus = Lactating;
        }
    }
    BT = 0;
}

// ------------------------------------------------------------------------------------------------//

float Individual::DetermineBodymass() {
    std::normal_distribution<float> distribution(species->meanBodymass, species->sdBodymass);
    Q_ASSERT(species->meanBodymass > 0);
    Q_ASSERT(species->sdBodymass > 0);

    float bodymass = distribution(rNG);

    while (bodymass <= 0) {
        bodymass = distribution(rNG);
    }
    return bodymass;
}

// ------------------------------------------------------------------------------------------------//

int Individual::CalculateMaxHomerange(int aPatchLength) {
    float maxHomerange;
    float maxHomerange1 = 47.863 * pow(bodymass, 1.18);           // omnivores
    float maxHomerange2 = 56.23 * pow(bodymass, 0.91);            // herbivores

    if (maxHomerange1 > maxHomerange2) {
        maxHomerange = maxHomerange1;
    } else {
        maxHomerange = maxHomerange2;
    }
    // convert to m2:
    maxHomerange = maxHomerange * 10000;

    // convert to number of patches:
    // maxHomerange = maxHomerange / (aPatchLength * aPatchLength);

    //radius of maxhr in m:
    maxHomerange = sqrt(maxHomerange / M_PI);

    //radius of maxhr in gridcells:
    maxHomerange = maxHomerange / aPatchLength;

    return static_cast<int>(round(maxHomerange));
}

// ------------------------------------------------------------------------------------------------//

float Individual::CalculateLocomotionCost(int aPatchLength) {
    // in future here: adapt to food type!!!!different conversion factor

    // costs for mammals in J/m; mass in kg after Calder 1996
    locomotionCost = (10.7 * pow(bodymass, 0.68));
    // costs in g dry biomass/m; after Nagy99 p.263 Buchmann 2011
    locomotionCost = locomotionCost / 10000;
    // locomotioncost in patchlength (= PATCH_LENGTH in m)
    locomotionCost = locomotionCost * aPatchLength;
    return locomotionCost;
}

// ------------------------------------------------------------------------------------------------//

int Individual::CalculateMaxNatalDispersal(int aPatchLength) {
    // maximum natal dispersal distances for herbivorous/omnivorous mammals in km after Sutherland 2000
    float natalDispersal = 3.31 * std::pow(bodymass,  0.65);

    // dispersal distance in patchlength (= PATCH_LENGTH in m)
    return static_cast<int>(round((natalDispersal * 1000) / aPatchLength));
}

// ------------------------------------------------------------------------------------------------//

float Individual::CalculateFeedingRate() {
    // daily feeding rate of herbivorous mammals in dry g/day, mass in kg, after Nagy 2001
    return 65.765 * pow(bodymass, 0.628);
}
// ------------------------------------------------------------------------------------------------//

float Individual::CalculateFoodShare() {
    // allometric share of available food see Buchmann 2011
    return pow((bodymass / 0.001), -0.25);
}

// ------------------------------------------------------------------------------------------------//

int Individual::CalculateAverageLifespan() {
    if (Utilities::FloatsAlmostEqual(species->averageLifespan, 0)) {
        // average lifespan: allometric formula of mammals after Hamilton et al 2011 [g, days]
        return static_cast<int>(round(1766.53 * pow(bodymass,0.21)));
    } else {
        // extract average lifespan of the species
        return static_cast<int>(round(species->averageLifespan));
    }
}

// ------------------------------------------------------------------------------------------------//

int Individual::CalculateAgeFirstReproduction() {
    if (Utilities::FloatsAlmostEqual(species->meanAFR, 0)) {
        // age of first reproduction allometric after Hamilton et al 2011
        return static_cast<int>(round(293.17 * pow(bodymass,0.27)));
    } else {
        // draw from species-specific normal distribution, avoid negative values:
        std::normal_distribution<float> distribution(species->meanAFR, species->sdAFR);
        int maturity = static_cast<int>(round(distribution(rNG)));
        while (maturity <= 0) {
            maturity = static_cast<int>(round(distribution(rNG)));
        }
        return maturity;
    }
}

// ------------------------------------------------------------------------------------------------//

int Individual::CalculateLitterSize() {
    if (Utilities::FloatsAlmostEqual(species->meanLitterSize, 0)) {
        // allomteric litter size after Hamilton et al.
        return static_cast<int>(round(2.239 * pow(bodymass, -0.13)));
    } else {
        // draw from species-specific normal distribution, avoid negative values:
        std::normal_distribution<float> distribution(species->meanLitterSize, species->sdLitterSize);
        int litterSize = static_cast<int>(round(distribution(rNG)));
        while (litterSize <= 0) {
            litterSize = static_cast<int>(round(distribution(rNG)));
        }
        return litterSize;
    }
}

// ------------------------------------------------------------------------------------------------//

int Individual::CalculateGestationPeriod() {
    if (Utilities::FloatsAlmostEqual(species->meanGestation, 0)) {
        // gestation period allometric after Hamilton et al 2011
        return static_cast<int>(round(64.139 * pow(bodymass, 0.24)));
    } else {
        // draw from species-specific normal distribution, avoid negative values:
        std::normal_distribution<float> distribution(species->meanGestation, species->sdGestation);
        int gestation = static_cast<int>(round(distribution(rNG)));
        while (gestation <= 0) {
            gestation = static_cast<int>(round(distribution(rNG)));
        }
        return gestation;
    }
}

// ------------------------------------------------------------------------------------------------//

int Individual::CalculateLactationPeriod() {
    if (Utilities::FloatsAlmostEqual(species->meanLactation, 0)) {
        // lactation period allometric after Hamilton et al 2011
        return static_cast<int>(round(57.164 * pow(bodymass,0.22)));
    } else {
        // draw from species-specific normal distribution, avoid negative values:
        std::normal_distribution<float> distribution(species->meanLactation,species->sdLactation);
        int lactation = static_cast<int>(round(distribution(rNG)));
        while (lactation <= 0) {
            lactation = static_cast<int>(round(distribution(rNG)));
        }
        return lactation;
    }
}

// ------------------------------------------------------------------------------------------------//

float Individual::CalculateReproductiveCosts() {
    // energy expenditure during lactation is increased by 80% after Gittlemann 1988
    float lactationFeedingRate = feedingRate * 0.8;

    // this increase is considered linear to the number of offspring
    // energy demand per offspring:
    return lactationFeedingRate/static_cast<float>(litterSize);
}

// ------------------------------------------------------------------------------------------------//

float Individual::CalculateFastingEndurance() {
    // fasting endurance allometric after Lindstedt et al 1985
    return 9.3 * pow(bodymass,0.44);
}

// ------------------------------------------------------------------------------------------------//

float Individual::CalculateCapacity(int aPatchLength) {
    // determine intercept from landscape size:
    int intercept = aPatchLength * aPatchLength * 50;
    return intercept * pow(bodymass, -0.75);
}

// ------------------------------------------------------------------------------------------------//

float Individual::CalculateNaturalMortality() {
    // natural mortality rate for mammals after McCoy et al 2008
    float annualMortality = 0.0724 * pow(bodymass, -0.24);
    float dailyMortality = 1.0 - pow(( 1.0 - annualMortality), (1.0/365.0));
    return dailyMortality;
}

// ------------------------------------------------------------------------------------------------//

void Individual::Die() {
    Q_ASSERT(!dead);
    if (!dead) {
        species->abundance--;
    }
    dead = true;
}

// ------------------------------------------------------------------------------------------------//

Gender Individual::DetermineSex() {
    float randomFloat = Utilities::RandomFloat(0,1);
    // random assignment of sex with sexratio 1:1
    if (randomFloat < static_cast<float>(0.5)) {
        return Female;
    } else {
        return Male;
    }
}

// ------------------------------------------------------------------------------------------------//

float Individual::DetermineShelterUse() {
    // index 0..1 of shelter use/need; not distinguished yet
    return 0.5;
}

// ------------------------------------------------------------------------------------------------//

FoodPreference Individual::DetermineFoodPreference() {
    // food preference; not distinguished yet
    return Herbivore;
}

// ------------------------------------------------------------------------------------------------//

ForagingType Individual::DetermineForagingType() {
    // type of forage movement; not distinguished yet
    return CentralPlaceForager;
}

// ------------------------------------------------------------------------------------------------//

void Individual::FindHomeRangeGraph(Grid* landscape, int patchLength, Individual* parent, float mortalityChance) {

    float minFeed = UpdateFeedingRate();
    //std::cout << "FeedingRate updated" << std::endl;
    bool success = false;
    int attempt = 0;
    int maxAttempts = 100;       // number of possible search attepmts if new individuals
    if (parent) {
        maxAttempts = 10;        // number of search attempts if juvenile individual
    }

    std::tuple<bool, float> homeRangeAssessment;
    //xx attempts for each mammal to find suitable homerange:
    while (!success && attempt < maxAttempts && !dead) {
        attempt++;
        //std::cout << "attempt " << attempt << std::endl;

        //Retrieve core cell
        GridCell* coreCell = FindCoreCell(landscape, parent);
        //std::cout << "CoreCell " << coreCell->x << "," << coreCell->y << std::endl;
        // individual dies if no suitable central place can be found
        if (coreCell == nullptr) {
            deathCause = NoCoreCell;
            Die();
            return;
        } else {
            Q_ASSERT(coreCell->habitatTyp == Habitat);
            location_x = coreCell->x;
            location_y = coreCell->y;
            // std::cout << "x: " << location_x << " y: " << location_y << std::endl;
        }

        homeRangeAssessment = AssessHomeRangeMatrixCrossing(landscape, coreCell, mortalityChance, minFeed, true);

        success = get<0>(homeRangeAssessment);
        //std::cout << "attempt success " << success << std::endl;
    }

    if (success) {
        GridCell* coreCell = landscape->GetCell(location_x, location_y);
        coreCell->centralPlace = Occupied;

        //reduce remaining food in all patches of newly founded home range (only if individual was successful in establishing a home range):
        for (uint i = 0; i < homeRangeCells.size(); i++) {
            GridCell* homeRangeCell = homeRangeCells[i];
            // resource change in edge cells depends upon individual responsiveness to risk
            // calculate risk dependent foodShare reduction
            //float riskDependentFoodShareFactor = responsiveness.slope * homeRangeCell->riskLevel + responsiveness.intercept;
            homeRangeCell->resourceAmount *= (1 - foodShare);

            //only consume the needed resources to meet the energy requirements in the last cell
            if (i == homeRangeCells.size() - 1) {
                float feed = get<1>(homeRangeAssessment);
                float excess = feed - minFeed;
                homeRangeCell->resourceAmount += excess;
            }
            Q_ASSERT(homeRangeCell->resourceAmount >= 0);

            // increase occupancy of cell:
            homeRangeCell->occupancy += 1;
            homeRangeCell->occupantsSpecies[species] += 1;
        }

        // sample 1-2 random positions of the home range to use for later MCP calculation
        RandomPositionSampling(homeRangeCells, patchLength, nSamples);
    } else {
        if (!dead) {
            deathFood = true;
            deathCause = FailedHR;
            Die();
        }
    }
}

// ------------------------------------------------------------------------------------------------//

void Individual::CheckHomeRangeGraph(Grid* landscape, int patchLength, float mortalityChance) {
    float minFeed = UpdateFeedingRate();
    //std::cout << "min feed " << minFeed << std::endl;

    // retrieve core cell
    GridCell* coreCell = landscape->GetCell(location_x, location_y);
    std::tuple<bool, float> homeRangeAssessment;
    homeRangeAssessment = AssessHomeRangeMatrixCrossing(landscape, coreCell, mortalityChance, minFeed, false);

    bool success = get<0>(homeRangeAssessment);
    float feed = get<1>(homeRangeAssessment);
    //std::cout << "found feed " << get<1>(homeRangeAssessment) << std::endl;

    if (success) {
        foodShortageInHomerange = false;
        resourceShortage = 0;
    } else {
        if (dead) {
            return;
        }
        foodShortageInHomerange = true;
        resourceShortage += 1;

        // let individuals die that have not enough food in their old home range
        // mortality depends on relative food shortage
        float foodMortality = feed / minFeed;
        // actual food shortage * (days of starving/allometric fasting endurance)
        float randomFloat = Utilities::RandomFloat(0,1);
        if (randomFloat < ((1 - foodMortality) * (static_cast<float>(resourceShortage) / fastingEndurance))) {
            deathFood = true;
            deathCause = Starvation;
            Die();
        }
    }

    // sample 1-2 random positions of the home range to use for later MCP calculation
    RandomPositionSampling(homeRangeCells, patchLength, nSamples);
}

// ------------------------------------------------------------------------------------------------//

float Individual::UpdateFeedingRate() {
    float actualFeedingRate = feedingRate;
    if (pregnancyStatus == Gestating) {
        actualFeedingRate = feedingRate * static_cast<float>(1.2);
    } else if (pregnancyStatus == Lactating) {
        actualFeedingRate = feedingRate * static_cast<float>(1.8);
    }
    return(actualFeedingRate);
}

// ------------------------------------------------------------------------------------------------//

GridCell* Individual::FindCoreCell(Grid* landscape, Individual* parent) {
    GridCell* coreCell = nullptr;

    // random search for homerange core cell:
    if (parent) {
        int maxDispersalRadius = maxNatalDispersal;
        if (maxDispersalRadius > landscape->size) {
            maxDispersalRadius = landscape->size;
        }

       // random search for homerange core cell in neighbourhood within maxRadius (= natal dispersal distance)
        // individual chooses a central place within the habitat, matrix is avoided
        coreCell = landscape->GetRandomCellInRadiusWith(
                    parent->location_x,
                    parent->location_y,
                    maxDispersalRadius,
                    0,
                    [] (const GridCell* cell) {return cell->habitatTyp != Matrix; });
    } else {
        // individual chooses a central place within the habitat, matrix is avoided
        coreCell = landscape->GetRandomCellWith([] (const GridCell* cell) {return cell->habitatTyp != Matrix; });
    }
    return coreCell;
}

// ------------------------------------------------------------------------------------------------//

std::tuple<bool, float> Individual::AssessHomeRangeMatrixCrossing(Grid* landscape, GridCell* coreCell, float mortalityChance, float minFeed, bool searching) {
    bool success = false;
    float feed = 0;
    int radiusSearch = 0;
    float longestDistance = 0;
    float dailyDistance = 0;
    int countHabitatCells = 0;
    int countMatrixCells = 0;
    int firstMatrixCell = 0;

    //create new set visited cells for each search attempt
    set<GridCell*> visitedCells;

    //create new data type to store all relevant information on each cell
    class CellTree {
        public:
        GridCell* cell;
        int allAncestors = 0;
        int matrixAncestors = 0;
        float distanceToCoreCell = 0.0;
        float locomotionCostToCell = 0.0;
        float feed = 0.0;
        float visitingProbability = 0.0;
        float toleranceProbability = 0.0;

        CellTree(GridCell* cell, GridCell* coreCell, int landscapeSize, bool matrixCost, float maxRL, Individual* i, CellTree* ancestor) {
            this->cell = cell;
            if (ancestor == nullptr) {
                this->allAncestors = 1;
                this->matrixAncestors = 0;
            } else {
                this->allAncestors = ancestor->allAncestors + 1;
                if (cell->habitatTyp == Matrix) {
                   this->matrixAncestors = ancestor->matrixAncestors + 1;
                }
            }
            this->distanceToCoreCell = cell->DistanceToCoreCell(coreCell->x, coreCell->y, landscapeSize);
            this->locomotionCostToCell = i->locomotionCost * distanceToCoreCell;
            float behaviorTrait = i->responsiveness.slope * -1.0;

            /*
            std::cout << "cell x: " << cell->x << " cell y: " << cell->y
                      << " res: " << cell->resourceAmount
                      << " max res: " << maxRL
                      << " loco: " << 1.0 - (locomotionCostToCell / i->maxLocomotionCost)
                      << " behave: " << behaviorTrait
                      << " matrix: " << (behaviorTrait * static_cast<float>(matrixAncestors)) / static_cast<float>(allAncestors)
                      << std::endl;
                      */

            float relativeDistance = (1.0 - (locomotionCostToCell / i->maxLocomotionCost));
            float relativeFood = cell->resourceAmount / maxRL;

            this->visitingProbability = (((1.0 - i->searchFactor) * relativeFood) + (i->searchFactor * relativeDistance)) *
                    (1.0 - ((behaviorTrait * static_cast<float>(matrixAncestors)) / static_cast<float>(allAncestors)));

            if (matrixAncestors > 0) {
                float upperLimit = 1.0 - (static_cast<float>(matrixAncestors) / static_cast<float>(allAncestors));
                this->toleranceProbability = Utilities::RandomFloat(0.0, upperLimit);
            } else {
                this->toleranceProbability = 1.0;
            }

            if (cell->habitatTyp == Matrix) {
                if (matrixCost) { // Matrix cell w/o travelling costs
                    feed = 0 - (2 * i->locomotionCost * this->distanceToCoreCell);
                } else
                {
                    feed = 0;
                }
            } else {
                this->feed = (cell->resourceAmount * i->foodShare) -
                        (2 * i->locomotionCost * this->distanceToCoreCell);
            }
        }
    };

    //create priority queue
    // If this function returns true, left is sorted "before" right in the queue,
    // which means that left is output AFTER right.
    // [1,2,4,5] -> 5 is output next
    // insert 3, compare with 4 using std::less returns true, inserted "before" 4
    // [1,2,3,4] -> 4 is output next

    // sort by visiting probability (if probabiities are the same, sort by distance)
    auto cmp = [](CellTree left, CellTree right) {
        /*
        if (Utilities::FloatsAlmostEqual(left.visitingProbability, right.visitingProbability) &&
                Utilities::FloatsAlmostEqual(left.distanceToCoreCell, right.distanceToCoreCell)) {
            return left.feed < right.feed; // highest feeding value
        }
        if (Utilities::FloatsAlmostEqual(left.visitingProbability, right.visitingProbability)) {
            return left.distanceToCoreCell > right.distanceToCoreCell; // lowest distance
        }
        */
        return left.visitingProbability < right.visitingProbability; // highest visiting probability
    };
    priority_queue<CellTree, std::vector<CellTree>, decltype(cmp)> queue(cmp);

    CellTree firstCell(coreCell, coreCell, landscape->size, landscape->matrixCosts, landscape->maxRL, this, nullptr);
    // calculates feed and visiting probability

    //std::cout << "maxlocomotionCost " << maxLocomotionCost << std::endl;
    //Push core cell in Queue
    queue.push(firstCell);

    //store the habitat cells in homeRangeCells
    homeRangeCells.clear();

    //Loop:
    //Abruchsituation im loop: Queue leer
    while (!queue.empty()) {
        //Retrieve first cell from queue
        CellTree c = queue.top();
        assert(c.cell);

        //Delete first cell from queue
        queue.pop();

        //continue with the queue if cell has already been visited
        if (visitedCells.count(c.cell)) {
            continue;
        }

        //std::cout << " cell " << c.cell->x << ";" << c.cell->y << " Habitat = " << c.cell->habitatTyp << std::endl;

        // add food from cell to already found resources:
        feed += c.feed;
        //std::cout << "cell: " << c.cell->x << c.cell->y << " found here: " <<  c.feed << " found total: " << feed << std::endl;

        if (c.cell->habitatTyp == Habitat) {
            //homeRangeCells.push_back(c.cell);
            countHabitatCells++;
        } else {
            countMatrixCells++;
            if (countMatrixCells == 1) {
                firstMatrixCell = countHabitatCells;
            }
            if (landscape->matrixMortality == PerCell) {
                float randomFloat = Utilities::RandomFloat(0, 1);
                if (randomFloat < mortalityChance) {
                    if (searching) {
                        deathCause = FailedHR;
                    } else {
                        deathCause = MatrixMortality;
                    }
                    Die();
                    return std::make_tuple(success, feed);
                }
            }
        }

        //Push cell in set visited cells
        visitedCells.insert(c.cell);
        homeRangeCells.push_back(c.cell);

        // sum up daily movement distances
        dailyDistance += 2 * c.distanceToCoreCell;

        //record longest distance
        if (longestDistance < c.distanceToCoreCell) {
            longestDistance = c.distanceToCoreCell;
            radiusSearch = static_cast<int>(ceil(c.distanceToCoreCell));
        }

        if (!searching) {
            // individual forages independent of its final success in all habitat cells
            if (c.cell->habitatTyp != Matrix) {
                // float riskDependentFoodShareFactor = responsiveness.slope * c.cell->riskLevel + responsiveness.intercept;
                // forage with calculated risk dependent foodShare reduction
                c.cell->resourceAmount *= (1 - foodShare); // * riskDependentFoodShareFactor
                // increase occupancy of cell:
                c.cell->occupancy += 1;
                c.cell->occupantsSpecies[species] += 1;
            }
        }

        //end loop if min feed is reached
        if (feed >= minFeed) {
            success = true;
            if (!searching) {
                //only consume the needed resources to meet the energy requirements in the last cell
                float excess = feed - minFeed;
                c.cell->resourceAmount += excess;
                Q_ASSERT(c.cell->resourceAmount >= 0);
            }
            break;
        }

        //Find suitable neighbors of cell: a) not visited before, b) within maxRadius
        vector<GridCell*> suitableNeighbors;
        suitableNeighbors = landscape->GetNeighboursWith(c.cell->x, c.cell->y,
                                                         [=] (GridCell* cell) {
            // TODO: cache distanceToCoreCell calculation
            return !visitedCells.count(cell) &&
                    cell->DistanceToCoreCell(location_x, location_y, landscape->size) <= static_cast<float>(maxHomeRangeRadius);
        });

        //Push suitable neighbors in queue
        for (uint i = 0; i < suitableNeighbors.size(); i++) {
            // construct a new tree for each neighbor
            //all relevant information is calculated in the constructor
            CellTree newCell(suitableNeighbors[i], coreCell, landscape->size, landscape->matrixCosts, landscape->maxRL, this, &c);
            //std::cout << "cell x: " << newCell.cell->x << " cell y: " << newCell.cell->y
            //          << " prob: " << newCell.visitingProbability << std::endl;
            // a cell is only added to the queue if its toleranceProbability is smaller that an animals tolerance
            //std::cout << "Prob.: " << newCell.toleranceProbability << " slope " << 1 - (responsiveness.slope * -1.0) << std::endl;
            if (newCell.toleranceProbability > (responsiveness.slope * -1.0)) {
                queue.push(newCell);
            }
        }
    }

    dailyMovement.push_back(dailyDistance);
    longestForagingBout.push_back(longestDistance);
    homerangeDiameter.push_back(2 * radiusSearch);
    habitatCells.push_back(countHabitatCells);
    matrixCells.push_back(countMatrixCells);
    homeRangeSize.push_back(countMatrixCells + countHabitatCells);

    if (firstMatrixCell == 0) {
        firstMatrixCrossing.push_back(1.0);
    } else {
        firstMatrixCrossing.push_back(static_cast<float>(firstMatrixCell) / static_cast<float>(countHabitatCells));
        //std::cout << "firstMatrix: " << firstMatrixCell << " already found Habitat: " << countHabitatCells << std::endl;
    }

    if (landscape->matrixMortality == PerFraction) {
        float randomFloat = Utilities::RandomFloat(0, 1);
        float matrixShare = static_cast<float>(countMatrixCells) / (static_cast<float>(countMatrixCells) + static_cast<float>(countHabitatCells));
        float indChance = mortalityChance * matrixShare;
        if (randomFloat < indChance) {
            if (searching) {
                deathCause = FailedHR;
            } else {
                deathCause = MatrixMortality;
            }
            Die();
            return std::make_tuple(success, feed);
        }
    }

    return std::make_tuple(success, feed);
}

// ------------------------------------------------------------------------------------------------//

void Individual::RandomPositionSampling(std::vector<GridCell*> homeRangeCells, int aPatchLength, int nSamples) {

    // positions are randomly sampled from all cells belonging to the individual's home range
    // positions can be floating point numbers from any where within a cell

    int maxPositions = homeRangeCells.size();

    // assert that vector contains cells
    assert(maxPositions != 0);

    for (int i = 0; i < nSamples; i++) {

        // sample 2 random cells from the vector (can be the same cell)
        int randPosition = rNG() % maxPositions;

        // get coordinates from those random home range cells
        GridCell* randomCell = homeRangeCells[randPosition];

        // get any random x and y coordinates inside a grid cell
        float xPosition = Utilities::RandomFloat(randomCell->x * aPatchLength,
                                                  randomCell->x * aPatchLength + aPatchLength);
        float yPosition = Utilities::RandomFloat(randomCell->y * aPatchLength,
                                                  randomCell->y * aPatchLength + aPatchLength);

        // push positions in vector
        homeRangePositionsX[i] = xPosition;
        homeRangePositionsY[i] = yPosition;
    }

    /*
    // delete core cell from vector
    homeRangeCells.erase(std::remove_if(homeRangeCells.begin(),
                                        homeRangeCells.end(),
                                        [=](GridCell* cell) {
                          if (cell->x == location_x && cell->y == location_y) {
                              return true;
                          }
                          return false;
                      }),
            homeRangeCells.end());
   */
}

// ------------------------------------------------------------------------------------------------//
