#include "grid.h"
#include <set>
#include <unordered_set>
#include <algorithm>
#include <iostream>


Grid::Grid(int aSize)
{
    size = aSize;
    maxRL = 0;
    // Make an array of size pointers to GridCell arrays
    // each GridCell pointer will be the beginning of a series of GridCells
    grid = new GridCell*[size];
    // each GridCell pointer gets an array to GridCells of the length size
    for (int i = 0; i < size; ++i) {
        grid[i] = new GridCell[size];
    }

    // initialise each GridCell and add location information
    for (int x = 0; x < size; ++x) {
        for (int y = 0; y < size; ++y) {
            grid[x][y] = GridCell();
            grid[x][y].x = x;
            grid[x][y].y = y;
        }
    }
}

// ------------------------------------------------------------------------------------------------//

Grid::~Grid()
{
    for (int i = 0; i < size; ++i) {
        delete[] grid[i];
    }

    delete[] grid;
}

// ------------------------------------------------------------------------------------------------//

float Grid::CalculateDistanceBetweenCells(int cellA_x, int cellA_y, int cellB_x, int cellB_y) {

    int a1 = abs(cellA_x - cellB_x);
    int a2 = abs(cellA_x - (cellB_x - size));
    int a3 = abs(cellA_x - (cellB_x + size));
    int a = std::min(std::min(a1, a2), a3);

    int b1 = abs(cellA_y - cellB_y);
    int b2 = abs(cellA_y - (cellB_y - size));
    int b3 = abs(cellA_y - (cellB_y + size));
    int b = std::min(std::min(b1, b2), b3);

    float distance = sqrt(pow(a, 2) + pow(b, 2));
    return distance;
}

// ------------------------------------------------------------------------------------------------//

std::vector<GridCell*> Grid::ShuffleGridCells() {
    std::vector<GridCell*> gridVector;
    gridVector.reserve(size*size);

    for (int x = 0; x < size; x++) {
        for (int y = 0; y < size; y++) {
            GridCell* cell = GetCell(x, y);
            gridVector.push_back(cell);
        }
    }

    std::shuffle(gridVector.begin(), gridVector.end(), rNG);
    return gridVector;
}

// ------------------------------------------------------------------------------------------------//

GridCell* Grid::GetCell(int x, int y) {
    if (x < 0 || y < 0 || x >= size || y >= size) {
        return nullptr;
    }
    return &grid[x][y];
}

// ------------------------------------------------------------------------------------------------//

GridCell* Grid::GetCellWithBoundaryOverflow(int x, int y) {
    if (x > size - 1) {
        x = x - size;
    }

    if (y > size - 1) {
        y = y - size;
    }

    if (x < 0) {
        x = x + size;
    }

    if (y < 0) {
        y = y + size;
    }
    return &grid[x][y];
}

// ------------------------------------------------------------------------------------------------//

GridCell* Grid::GetRandomCell() {
    uint x = rNG() % size;
    uint y = rNG() % size;

    return &grid[x][y];
}

// ------------------------------------------------------------------------------------------------//

GridCell* Grid::GetRandomCellWith(std::function<bool (const GridCell*)> func) {

    GridCell* randomCell = GetRandomCell();

    while (!func(randomCell)) {
        randomCell = GetRandomCell();
    }
    return randomCell;
}

// ------------------------------------------------------------------------------------------------//

GridCell* Grid::GetRandomCellInRadius(int core_x, int core_y, int radius, int skip) {
    return GetRandomCellInRadiusWith(core_x, core_y, radius, skip, nullptr);
}

// ------------------------------------------------------------------------------------------------//

GridCell* Grid::GetRandomCellInRadiusWith(int core_x, int core_y, int radius, int skip, std::function<bool (const GridCell*)> func) {

    // for scenarios in which individuals are unselective in their choice of a core cell:
    if (!func) {
        int xDistance = 0;
        int yDistance = 0;
        int xMax = 2 * radius - (skip * 2 + 1);  // defines the range for possible cells in radius
        int xRand = rNG() % xMax+1;      // get one random integer within this range

        int yMax = 2 * radius - (skip * 2 + 1);  // defines the range for possible cells in radius
        int yRand = rNG() % yMax+1;      // get one random integer within this range

        // map this random integer to a distance from the core cell
        if (xRand < (xMax+1)/2) {
            xDistance = xRand - radius;
        } else {
            xDistance = xRand + (skip * 2 + 1) - radius;
        }

        if (yRand < (yMax+1)/2) {
            yDistance = yRand - radius;
        } else {
            yDistance = yRand + (skip * 2 + 1) - radius;
        }

        // get the x position in the coordinate system
        int x = core_x + xDistance;
        int y = core_y + yDistance;

        GridCell* randomCell = GetCellWithBoundaryOverflow(x, y);
        return randomCell;
    }

    // for scenarios in which individuals are selctive in their choice of a core cell:
    std::vector<GridCell*> cells = GetGridCellsInRadiusWith(core_x, core_y, radius, skip, func);
    if (cells.size() == 0) {
        return nullptr;
    }

    int randCell = rNG() % cells.size();
    GridCell* randomCell = cells[randCell];

    return randomCell;
}

// ------------------------------------------------------------------------------------------------//

std::vector<GridCell*> Grid::GetNeighbours(int x, int y) {
    return GetNeighboursWith(x, y, nullptr);
}

// ------------------------------------------------------------------------------------------------//

std::vector<GridCell*> Grid::GetNeighboursWith(int x, int y, std::function<bool (GridCell*)> func) {
    std::vector<GridCell*> neighbours;
    neighbours.reserve(8);

    int leftBorder = x - 1;
    int rightBorder = x + 1;
    int topBorder = y - 1;
    int bottomBorder = y + 1;

    if (x == 0) {
        leftBorder = size - 1;
    }

    if (x == size - 1) {
        rightBorder = 0;
    }

    if (y == 0) {
        topBorder = size - 1;
    }

    if (y == size - 1) {
        bottomBorder = 0;
    }

    GridCell* cell00 = GetCell(leftBorder, bottomBorder);
    if (!func || func(cell00)) {
        neighbours.push_back(cell00);
    }
    GridCell* cell01 = GetCell(leftBorder, y);
    if (!func || func(cell01)) {
        neighbours.push_back(cell01);
    }
    GridCell* cell02 = GetCell(leftBorder, topBorder);
    if (!func || func(cell02)) {
        neighbours.push_back(cell02);
    }
    GridCell* cell12 = GetCell(x, topBorder);
    if (!func || func(cell12)) {
        neighbours.push_back(cell12);
    }
    GridCell* cell10 = GetCell(x, bottomBorder);
    if (!func || func(cell10)) {
        neighbours.push_back(cell10);
    }
    GridCell* cell20 = GetCell(rightBorder, bottomBorder);
    if (!func || func(cell20)) {
        neighbours.push_back(cell20);
    }
    GridCell* cell21 = GetCell(rightBorder, y);
    if (!func || func(cell21)) {
        neighbours.push_back(cell21);
    }
    GridCell* cell22 = GetCell(rightBorder, topBorder);
    if (!func || func(cell22)) {
        neighbours.push_back(cell22);
    }

    return neighbours;
}

// ------------------------------------------------------------------------------------------------//

std::vector<GridCell*> Grid::GetGridCellsInRadius(int core_x, int core_y, int radius, int skip) {
    return GetGridCellsInRadiusWith(core_x, core_y, radius, skip, nullptr);
}

// ------------------------------------------------------------------------------------------------//

std::vector<GridCell*> Grid::GetGridCellsInRadiusWith(int core_x, int core_y, int radius, int skip, std::function<bool (GridCell*)> func) {
    std::vector<GridCell*> gridCellsInRadius;
    int maxCells = (2 * radius + 1) * (2 * radius + 1) - 1;
    gridCellsInRadius.reserve(maxCells);

    for (int x = core_x - radius; x <= core_x + radius; x++) {
        for (int y = core_y - radius; y <= core_y + radius; y++) {
            // do not push back the central cell
            if (x == core_x && y == core_y) {
                continue;
            }

            float distance = CalculateDistanceBetweenCells(core_x, core_y, x, y);
            if (distance <= radius && distance > skip) {
                GridCell* cell = GetCellWithBoundaryOverflow(x, y);
                if (!func || func(cell)) {
                    gridCellsInRadius.push_back(cell);
                }
            }
            //std::cout << "not in radius " << radius << ": x =" << x << " y = " << y << " ditance to core = " << distance << " coordinates core = x: " << core_x << " y: " << core_y << std::endl;
        }
    }
    return gridCellsInRadius;
}
