library(ggplot2)
library(ggh4x)
library(ggridges)
library(dplyr)
library(forcats)
library(viridis)
library(hrbrthemes)
library(patchwork)


theme_set(theme_ipsum() +
            theme(legend.position = "bottom",
                  text = element_text(family = "CMU Bright"),
                  axis.title.x =  element_text(size = 16, hjust = 0.5, 
                                               vjust = 0.5, family = "CMU Bright"),
                  axis.title.y = element_text(size = 16, hjust = 0.5, 
                                              vjust = 1.5, family = "CMU Bright"),
                  legend.title = element_text(size = 14, family = "CMU Bright"), 
                  legend.text = element_text(size = 12, family = "CMU Bright"),
                  plot.margin = unit(c(0.3,0.3,0.0,0.3), "cm")))

scenario <- data.table::fread("data/simulation-scenario.csv", header=TRUE, blank.lines.skip=TRUE)

scenario <- scenario %>%
  mutate(Fragmentation = factor(Fragmentation))  %>% 
  mutate(Fragmentation = fct_relevel(Fragmentation,
                                     "weak",
                                     "moderate",
                                     "high",
                                     "very high")) %>% 
  mutate(MatrixEffect = factor(MatrixEffect))  %>% 
  mutate(MatrixEffect = fct_relevel(MatrixEffect,
                                    "No costs",
                                    "Mortality risk",
                                    "Deferred costs"))

div <- data.table::fread("data/simulation-community.csv", header=TRUE, blank.lines.skip=TRUE)

div <- merge(scenario, div, by = "Seed")

ind <- data.table::fread("data/simulation-individuals.csv", header=TRUE, blank.lines.skip=TRUE)
ind <- merge(scenario, ind, by = "Seed")

species <- data.table::fread("data/simulation-species.csv", header=TRUE, blank.lines.skip=TRUE)
species <- merge(scenario, species, by = "Seed")

#### Plot 2 ####

niceColors <- c("#1FA187FF",  "#4AC16DFF", "#9FDA3AFF", "#FDE725FF")

correl_data <- subset(ind, Behaviour == "risk-diverse" & MortalityChance == 0)

correl_data2 <- correl_data %>%
  group_by(Fragmentation)  %>%
  #summarise(n = n())
  slice_sample(n = 50000)

FractionPers <- ggplot(correl_data2, aes(x = RiskTaking * - 1, y = MatrixCells / HRCells * 100,  
                                         color = Fragmentation, fill = Fragmentation)) +
  #geom_point(alpha = 0.2, shape = 4) +
  geom_smooth(span = 1, 
              alpha = 0.1, linetype = "dashed", linewidth = 1, 
              se = TRUE, level = 0.95) +  
  scale_x_continuous(breaks = c(-1, -0.75, -0.5,-0.25, 0),
                     labels = c(0, 0.25, 0.5, 0.75, 1)) +
  labs(x = "Risk-taking trait", y = "Matrix propotion in the home range [%]") +
  scale_fill_manual(values = niceColors, name = "Level of fragmentation",
                    labels = c("weak", "moderate", "high",  "very high")) +
  scale_color_manual(values = niceColors, name = "Level of fragmentation",
                     labels = c("weak", "moderate", "high",  "very high")) +
  ylim(0, 80) 

FractionPers

ggsave(filename = "plots/Plot2.pdf", plot = FractionPers,
       width = 200, height = 150, units = "mm", dpi = 300, 
       device = cairo_pdf) 


#### Plot 3 ####
##### Plot 3a species diversity ####

div$MortFactor <- div$MortalityChance
sumDiv <- div %>%
  group_by(Seed, Fragmentation, HabitatAmount, MatrixEffect, 
           MortalityChance, Behaviour, Run) %>% 
  filter(Behaviour == "risk-diverse" & Day > 7000) %>% 
  summarize(meanDiv = mean(ENS, na.rm = TRUE)) 


v_colors =  c("#6C6E72FF" , "#FCFFA4FF", "#FAC127FF", "#F57D15FF", "#D44842FF", "#9F2A63FF", 
              "#65156EFF", "#280B54FF", "#1FA187FF")

LabelsShape = c("No costs", "Mortality risk", "Deferred costs")
labels = c("", "0.001", "0.003", "0.01", 
           "0.03", "0.1",
           "0.3", "1", 
           "")
levels = c("0", "0.001", "0.003", "0.01", 
           "0.03", "0.1",
           "0.3", "1", 
           "100")

FigDiv <- ggplot(sumDiv, 
                 aes(x = as.numeric(Fragmentation), y = meanDiv, 
                     color = factor(MortalityChance * 100, levels = levels),
                     fill = factor(MortalityChance * 100, levels = levels),
                     shape = MatrixEffect)) +
  geom_line(data = subset(sumDiv, MortalityChance == 0), 
            aes(x = as.numeric(Fragmentation) - 0.4),
            stat="smooth", span = 1,
            linewidth = 1,
            linetype ="dashed",
            alpha = 1,
            show.legend = FALSE) +
  geom_line(data = subset(sumDiv, MortalityChance == 0.003), 
            aes(x = as.numeric(Fragmentation) + 0.3),
            stat="smooth", span = 1,
            linewidth = 1,
            linetype ="dashed",
            alpha = 1,
            show.legend = FALSE) +
  geom_line(data = subset(sumDiv, MortalityChance == 1), 
            aes(x = as.numeric(Fragmentation) + 0.4),
            stat="smooth", span = 1,
            linewidth = 1,
            linetype ="dashed",
            alpha = 1,
            show.legend = FALSE) +  
  geom_point(position = position_jitterdodge(seed = 3, dodge.width = 0.9, 
                                             jitter.width = 0.5),
             size = 1.5, 
             alpha = 0.5) +
  stat_summary(aes(fill = factor(MortalityChance * 100, levels = levels)),
               color = "black",
               size = 0.8,
               stroke = 0.9, 
               linewidth = 0.5,
               fun.data = "mean_sdl",  fun.args = list(mult = 1), 
               position = position_dodge(0.9),
               show.legend = FALSE) +
  scale_shape_manual(name = "Matrix effect", 
                     values = c(25, 21, 24)) +
  scale_color_manual(name = "Mortality risk\nper visited\nmatrix cell [%]",
                     breaks = levels, 
                     values = v_colors, 
                     labels = labels) +
  scale_fill_manual(name = "Mortality risk\nper visited\nmatrix cell [%]",
                    breaks = levels, 
                    values = v_colors, labels = labels) +
  scale_x_continuous(name = "Level of fragmentation", 
                     breaks = c(1,2,3,4),
                     labels = c("weak", "moderate", "high", "very high")) +
  scale_y_continuous(name = "Effective number of species", limits = c(1.2, 8.8),
                     breaks = c(0,2,4,6,8)) +
  guides(shape = guide_legend(order = 1, nrow = 3, override.aes = list(size = 3, alpha = 1,
                                                                       color = c("#1FA187FF", "#D44842FF", "#6C6E72FF"),
                                                                       fill = c("#1FA187FF", "#D44842FF", "#6C6E72FF"))),
         fill = guide_legend(order = 2, nrow = 9, override.aes = list(size = 3, alpha = 1)),
         color = guide_legend(order = 2, nrow = 9), override.aes = list(size = 3, alpha = 1)) +
  theme(legend.position = "right",
        plot.margin = unit(c(0.0,0.1,0.1,0.3), "cm"))

FigDiv

summary <- sumDiv %>%
  group_by(Fragmentation, MatrixEffect, MortalityChance) %>% 
  summarize(meanD = mean(meanDiv),
            sdD = sd(meanDiv)) 

##### Plot 3b Species distribution #####

SpeciesMean <- species %>%
  group_by(Fragmentation, Species, Behaviour, MortalityChance, MatrixEffect) %>% 
  filter(MortalityChance == 0 | MortalityChance == 0.01 | MortalityChance == 1) %>% 
  filter(Behaviour == "risk-diverse") %>% 
  summarize(meanAbundance = mean(Abundance, na.rm = TRUE)) %>% 
  group_by(Fragmentation, MortalityChance, MatrixEffect) %>%
  mutate(relativeAbundance = meanAbundance/sum(meanAbundance)) %>% 
  ungroup()

SpeciesMean <- SpeciesMean %>%
  mutate(MatrixEffect = factor(MatrixEffect))  %>% 
  mutate(MatrixEffect = fct_relevel(MatrixEffect,
                                    "No costs",
                                    "Deferred costs",
                                    "Mortality risk")) 

# New facet label names for matrix effect variable
matrix.labs <- c("No\ncosts",
                 "Deferred\ncosts",
                 "Mortality\nrisk 1%")
names(matrix.labs) <- c("No costs",
                        "Deferred costs",
                        "Mortality risk")

# facet theme
strip <- ggh4x::strip_themed(background_y = ggh4x::elem_list_rect(fill = c("#6C6E72FF", "#1FA187FF", "#280B54FF"),
                                                                  color = c("#6C6E72FF", "#1FA187FF", "#280B54FF")),
                             text_y = ggh4x::elem_list_text(color = c(rep("white", 3))),
                             background_x = ggh4x::elem_list_rect(fill = c(rep("white", 4)),
                                                                  color = c(rep("white", 4))),
                             text_x = ggh4x::elem_list_text(color = c(rep("black", 4))))

# Basic piechart
FigCommunity <- ggplot(SpeciesMean, 
                       aes(x="", y = -relativeAbundance, fill = as.factor(Species))) +
  geom_bar(stat="identity", width=1) +
  coord_polar("y", start=0) +
  scale_fill_manual(name = "Species:",
                    values = c("#004949", "#009292", "#924900", 
                               "#db6d00", "#490092", "#b66dff", 
                               "#006ddb", "#b6dbff", "#ff6db6","#ffb6db"), 
                    labels = c("1", "2", "3", "4", "5", "6", "7", "8", "9", "10")) +
  labs(y = "Level of fragmentation", x = "") +
  guides(fill = guide_legend(order = 1, nrow = 10, override.aes = list(size = 1.5))) +
  ggh4x::facet_grid2(MatrixEffect ~ Fragmentation, switch = "both",
             labeller = labeller(MatrixEffect = matrix.labs), strip = strip) +
  geom_point(aes(x = 0, y = 0), 
             color = "white", size = 5, show.legend = FALSE) + 
  theme_minimal() +
  theme(legend.position = "right", 
        text = element_text(family = "CMU Bright"),
        axis.line.y = element_blank(),
        axis.line.x = element_blank(),
        axis.ticks.y = element_blank(),
        axis.ticks.x = element_blank(),
        strip.text.x = element_text(face = 1, size = 12, family = "CMU Bright"),
        strip.text.y = element_text(face = 1, size = 12, family = "CMU Bright"),
        axis.text = element_blank(),
        axis.title.x = element_text(margin = unit(c(0, 0, 0, 0), "mm"),  size = 14, 
                                    family = "CMU Bright"),
        axis.title.y = element_text(margin = unit(c(0, -5, 0, 0), "mm"),  size = 14,
                                    family = "CMU Bright"),
        panel.grid.major.y = element_blank(),
        panel.grid.minor.y = element_blank(),
        legend.title = element_text(size = 12, family = "CMU Bright"), 
        legend.text = element_text(size = 10, family = "CMU Bright"),
        plot.margin = unit(c(0.3,0.0,0.0,-1.5), "cm"),
        panel.spacing.x = unit(2, "lines"))

FigCommunity

Plot3 <- FigDiv / FigCommunity + 
  plot_layout(heights = c(2.5, 1.5)) +
  plot_annotation(tag_levels = 'A') & 
  theme(plot.tag = element_text(family = "CMU Sans Serif Demi Condensed"))

Plot3

ggsave(
  "plots/Plot3a.pdf",
  plot = Plot3,
  dpi = 300, 
  device = cairo_pdf,
  height = 250,
  width = 210,
  units = c("mm"))


#### Plot 4 ####

##### Plot 4a competition ####

sumDiv <- div %>%
  group_by(Seed, Fragmentation, HabitatAmount, MatrixEffect, 
           MortalityChance, Behaviour) %>% 
  filter(Behaviour == "risk-diverse" & MortalityChance == 0) 

#%>% 
 # summarize(meanComp = mean(Competition, na.rm = TRUE)) 


FigCompetition <- ggplot(sumDiv, 
                         aes(x = Fragmentation, y = Competition)) +
  #geom_boxplot(notch = FALSE, fill = "#6C6E72FF") +
  stat_summary(fun.data = "mean_sdl", fun.args = list(mult = 1), position = position_dodge(0.9), color = "#6C6E72FF",
               size = 0.7, linewidth = 0.9) +
  #stat_summary(fun = mean,  geom = "pointrange", fun.min = min, fun.max = max, position = position_dodge(0.9), color = "#6C6E72FF",
  #             size = 0.75, linewidth = 1) +
  
  scale_x_discrete(labels = c("weak", "moderate", "high", "very high")) +
  scale_y_continuous(limits = c(5, 10)) +
  labs(x = "Level of fragmentation",
       y = "# Foragers per habitat cell") +
  guides(fill = guide_legend(order = 1, nrow = 3)) +
  theme(legend.position = "none",
        plot.margin = unit(c(0.3, 0.3, 0.3, 0.3), "cm"),
        axis.text.x = element_text(angle = 45, hjust = 1, family = "CMU Bright"))  

FigCompetition

summary <- sumDiv %>%
  group_by(Fragmentation, MatrixEffect, Behaviour) %>% 
  summarize(meanC = mean(Competition, na.rm = TRUE),
            sdC = sd(Competition)) 

##### Plot 4b juvenile success #####

species$JuvenileSuccess[is.nan(species$JuvenileSuccess)] <- NA

species$SpecSize = "none"
species[species$Species == 0 | species$Species == 1,  ]$SpecSize = "small"
species[species$Species == 4 | species$Species == 5,  ]$SpecSize = "intermediate"
species[species$Species == 8 | species$Species == 9,  ]$SpecSize = "large"

specJuv <- species %>%
  group_by(Seed, Species, Fragmentation, HabitatAmount,
           MatrixEffect, MortalityChance,
           Behaviour, SpecSize, Run) %>% 
  filter(MortalityChance == 0 & Behaviour == "risk-diverse" & (SpecSize == "small" | 
                                                               SpecSize == "intermediate" | 
                                                               SpecSize == "large"))  %>% 
  mutate(SpecSize = factor(SpecSize))  %>% 
  mutate(SpecSize = fct_relevel(SpecSize, 
                                "small" , 
                                "intermediate",
                                "large")) %>% 
  summarize(meanSuccess = mean(JuvenileSuccess, na.rm = TRUE)) 

mako <- c("#8AD9B1FF", "#40B7ADFF", "#348FA7FF")

FigJuvenile <- ggplot(specJuv, aes(x = Fragmentation, y = meanSuccess, 
                                   fill = as.factor(SpecSize), color = as.factor(SpecSize))) +
  geom_boxplot(position = position_dodge(0.9), notch = FALSE, color = "black") +
  #stat_summary(fun.data = "mean_sdl", fun.args = list(mult = 1), position = position_dodge(0.9)) +
  #stat_summary(fun = mean,  geom = "pointrange", fun.min = min, fun.max = max, position = position_dodge(0.9), ) +
  scale_fill_manual(values = mako,
                    name = "Species group by size") +
  scale_color_manual(values = mako,
                    name = "Species group by size") +
  labs(x = "Level of fragmentation", 
       y = "Success rate of juveniles\nto establish a home range [%]") +
  scale_x_discrete(labels = c("weak", "moderate", "high", "very high")) +
  ylim(0, NA) +
  guides(fill = guide_legend(order = 1, nrow = 3, 
                             title.position = "top", title.hjust = 0.5))

FigJuvenile

##### Plot 4c Home ranges #####

meanPerSpecies <- ind %>%
  group_by(Species, MortalityChance, Fragmentation)  %>%
  filter(MortalityChance == 0 & Behaviour == "risk-diverse")  %>%
  summarize(meanHabitat = mean(HabitatCells, na.rm = TRUE),
            meanTotal = mean(HRCells, na.rm = TRUE))  %>%  
  tidyr::pivot_longer(cols = c("meanTotal", "meanHabitat"), names_to = "HabitatType", 
                      values_to = "HomeRange") %>%
  arrange(-HomeRange)


meanPerSpecies$Factor <- paste(meanPerSpecies$Fragmentation, meanPerSpecies$HabitatType, sep = "_")

FigHR <- ggplot(meanPerSpecies,
                aes(x = as.factor(Species + 1), y = HomeRange / 100, fill = Factor)) +
  geom_col(position = "identity", alpha = 1) +
  scale_fill_manual(values = c("very high_meanHabitat" =  "#2eb62c",
                               "high_meanHabitat" = "#57c84d", 
                               "moderate_meanHabitat" = "#83d475",
                               "weak_meanHabitat" =  "#abe098",
                               "weak_meanTotal" = "#f1c27d",
                               "moderate_meanTotal" = "#e0ac69", 
                               "high_meanTotal" = "#c68642",
                               "very high_meanTotal" = "#8d5524"),
                    breaks = c("weak_meanHabitat", "weak_meanTotal",
                               "moderate_meanHabitat", "moderate_meanTotal", 
                               "high_meanHabitat", "high_meanTotal", 
                               "very high_meanHabitat",  "very high_meanTotal"),
                    labels = c("","weak", "", "moderate", 
                               "", "high", "", "very high"),
                    name = "Home range area (habitat/matrix)\nwith level of fragmentation") +
  labs(x = "Species", y = "Area [ha]") +
  ylim(0, 1) +
  theme(legend.spacing.y = unit(0.05, 'cm'),
        plot.margin = unit(c(0.1,0.1,0,0.1), "cm")) +
  guides(fill = guide_legend(byrow = TRUE, nrow = 4,
                             title.position = "top", title.hjust = 0.0))
                              

FigHR

##### Plot 4d mortality ####

mortalityChance <- species %>%
  group_by(Seed, Species, Fragmentation, HabitatAmount, 
           MatrixEffect, MortalityChance, Behaviour, Run) %>% 
  filter(Behaviour == "risk-diverse")  %>% 
  summarize(meanSpStarvation = mean(SpeciesStarvation, na.rm = TRUE),
            meanSpMatrix = mean(SpeciesMatrix, na.rm = TRUE),
            TotalSp = meanSpStarvation + meanSpMatrix) 

yel = "#fdd43e"
bl = "#739add"
gr = "#929292"

# New facet label names for matrix effect 
matrix.labs <- c("No\ncosts", "0.001", "0.003", "0.01", 
                 "0.03", "0.1", "0.3", "1", "Deferred\ncosts")
names(matrix.labs) <- c("0", "1e-05", "3e-05", "1e-04", "3e-04", 
                        "0.001", "0.003", "0.01", "1")

# facet theme
strip <- ggh4x::strip_themed(background_x = ggh4x::elem_list_rect(fill = v_colors,
                                                                  color = v_colors),
                             text_x = ggh4x::elem_list_text(color = c("white", rep("black", 3), rep("white", 4), "black")))

SpecMort <- ggplot(mortalityChance[mortalityChance$Fragmentation == "high", ]) +
  geom_smooth(aes(x = Species, y = meanSpMatrix, 
                  color = "Matrix movement", fill = "Matrix movement", linetype = "Matrix movement")) +
  geom_smooth(aes(x = Species, y = meanSpStarvation, 
                  color = "Starvation", fill = "Starvation", linetype =  "Starvation")) +
  geom_smooth(aes(x = Species, y = TotalSp, 
                  color = "Combined risk", fill = "Combined risk", linetype = "Combined risk")) +
  ggh4x::facet_wrap2(. ~ MortalityChance,
              labeller = labeller(MortalityChance = matrix.labs),
              strip = strip, nrow = 1) +
  scale_x_continuous(breaks = c(0, 4.5, 9),
                     labels = c(1, 5, 10)) +
  scale_colour_manual(name = "Mortality cause", values = c(gr, bl, yel)) +
  scale_fill_manual(name = "Mortality cause", values = c(gr, bl, yel)) +
  scale_linetype_manual(name = "Mortality cause", values = c(1, 2, 2)) +
  labs(x = "Species", y = "Daily death rate [%]") +
  theme(panel.spacing = unit(0.5, "lines"),
        strip.text = element_text(hjust = 0.5, family = "CMU Bright"),
        plot.margin = unit(c(0.1,0.1,0.0,0.1), "cm"))

SpecMort

Plot4a <- FigCompetition + FigJuvenile + FigHR +
  #plot_layout() +
  plot_layout(ncol = 3, widths = c(1, 2, 2)) +
  plot_annotation(tag_levels = 'A') & 
  theme(plot.tag = element_text(family = "CMU Sans Serif Demi Condensed"))

Plot4 <- Plot4a / SpecMort +
  plot_annotation(tag_levels = 'A') & 
  theme(plot.tag = element_text(family = "CMU Sans Serif Demi Condensed"))

Plot4

ggsave(
  "plots/Plot4a.pdf",
  plot = Plot4,
  dpi = 300, 
  device = cairo_pdf,
  height = 265,
  width = 225,
  units = c("mm"))

#### Plot 5 ####
##### Plot 5a Behavioral types ####

v_colors = c("#6C6E72FF" , "#FCFFA4FF", "#FAC127FF", "#F57D15FF", "#D44842FF", "#9F2A63FF", 
             "#65156EFF", "#280B54FF", "#1FA187FF")
labels = c("No costs", "0.001", "0.003", "0.01", 
           "0.03", "0.1",
           "0.3", "1", 
           "Deferred\ncosts")

FigDensity <- ggplot(ind[ind$Behaviour == "risk-diverse", ], 
                     aes(x = RiskTaking * -1, y = as.factor(MortalityChance * 100), 
                         fill = as.factor(MortalityChance * 100), color = as.factor(MortalityChance * 100))) + 
  geom_density_ridges(alpha = 0, quantile_lines = TRUE, quantiles = 2, linetype = 2, color = "black") + 
  geom_density_ridges(alpha = 0.7) +  
  scale_x_continuous(breaks = c(-1, -0.5, 0),
                     labels = c(0, 0.5, 1)) +
  scale_y_discrete(labels =  c("No\ncosts", "0.001", "0.003", "0.01", 
                               "0.03", "0.1",
                               "0.3", "1", 
                               "Deferred\ncosts")) +
  labs(y = "Mortality probability per matrix cell [%]", x = "Risk-taking trait") +
  scale_color_manual(name = "Mortality probability per matrix cell [%]",
                     values = v_colors, 
                     labels = labels) +
  scale_fill_manual(name = "Mortality probability per matrix cell [%]",
                    values = v_colors, 
                    labels = labels) +
  facet_grid( ~ Fragmentation) +
  theme(legend.position = "none", 
        strip.text = element_text(hjust = 0.5, 
                                  family = "CMU Bright"),
        plot.margin = unit(c(-1.5, 0.3, 0.3, -1.5), "cm")) +
  guides(fill = guide_legend(order = 1, nrow = 2),
         color = guide_legend(order = 1, nrow = 2)) 

FigDensity


##### Plot 5b Mortality for Behavioral types ####

mortalityChance <- species %>%
  group_by(Seed, Species, Fragmentation, HabitatAmount, MatrixEffect, MortalityChance,
           Behaviour, Run) %>% 
  filter(Behaviour == "risk-diverse")  %>% 
  summarize(meanBTStarvation = mean(BTStarvation, na.rm = TRUE),
            meanBTMatrix = mean(BTMatrix, na.rm = TRUE),
            TotalBT = meanBTStarvation + meanBTMatrix) 

yel = "#fdd43e"
bl = "#739add"
gr = "#929292"

# facet theme
strip <- ggh4x::strip_themed(background_x = ggh4x::elem_list_rect(fill = v_colors,
                                                                  color = v_colors),
                             text_x = ggh4x::elem_list_text(color = c("white", rep("black", 3), rep("white", 4), "black")))


# New facet label names for matrix effect 
matrix.labs <- c("No costs", "0.001", "0.003", "0.01", 
                 "0.03", "0.1", "0.3", "1", 
                 "Deferred\ncosts")
names(matrix.labs) <- c("0", "1e-05", "3e-05", "1e-04", "3e-04", 
                        "0.001", "0.003", "0.01", "1")

BTMort <- ggplot(mortalityChance[mortalityChance$Fragmentation == "high", ]) +
  geom_smooth(aes(x = Species * -1, y = meanBTMatrix,
                  color = "Matrix movement", fill = "Matrix movement", linetype = "Matrix movement")) +
  geom_smooth(aes(x = Species * -1, y = meanBTStarvation, 
                  color = "Starvation", fill = "Starvation", linetype =  "Starvation")) +
  geom_smooth(aes(x = Species * -1, y = TotalBT,
                  color = "Combined risk", fill = "Combined risk", linetype = "Combined risk")) +
  ggh4x::facet_wrap2(. ~ MortalityChance,
                     labeller = labeller(MortalityChance = matrix.labs),
                     strip = strip, nrow = 1) +
  scale_x_continuous(breaks = c(-9, -4.5, 0),
                     labels = c(0, 0.5, 1)) +
  scale_colour_manual(name = "Mortality cause", values = c(gr, bl, yel)) +
  scale_fill_manual(name = "Mortality cause", values = c(gr, bl, yel)) +
  scale_linetype_manual(name = "Mortality cause", values = c(1, 2, 2)) +
  labs(x = "Risk-taking trait", y = "Daily death rate [%]") +
  theme(panel.spacing = unit(0.5, "lines"),
        strip.text = element_text(hjust = 0.5, family = "CMU Bright"))

BTMort

Plot5 <- FigDensity / BTMort + 
  plot_layout(heights = c(2.5, 2)) +
  plot_annotation(tag_levels = 'A') & 
  theme(plot.tag = element_text(family = "CMU Sans Serif Demi Condensed"))

Plot5

ggsave(
  "plots/Plot5.pdf",
  plot = Plot5,
  dpi = 300, 
  device = cairo_pdf,
  height = 250,
  width = 210,
  units = c("mm"))


#### Plot 6 ####

### diversity differences between heterogeneous and homogeneous community

sumDiv <- div %>%
  group_by(Fragmentation, MortalityChance,  Behaviour) %>% 
  summarise(meanENS = mean(ENS, na.rm = TRUE)) 

sumDiv$MortalityChance <- replace(sumDiv$MortalityChance, sumDiv$MortalityChance == 0, 0.000003) 

Reviewer <- ggplot(sumDiv[sumDiv$MortalityChance != 1, ], 
                   aes(x = (MortalityChance) * 100, y = meanENS, color = Behaviour)) +
  #stat_summary(fun="mean", geom="line", size = 1) + 
  stat_summary(fun.data = "mean_sdl",  fun.args = list(mult = 1), 
               show.legend = FALSE) +
  geom_smooth(span = 1, 
              alpha = 0.1, linetype = "dashed", linewidth = 1, 
              se = TRUE, level = 0.95) +
  #geom_point(position = position_jitterdodge(seed = 3, dodge.width = 0.5, jitter.width = 0.2),
  #           size = 1.5, 
  #           alpha = 0.5) +
  scale_x_log10(name = "Matrix mortality [%]",
                breaks = c(0.0001, 0.001, 0.01, 0.1, 1),
                labels = c("0", "0.001", "0.01", "0.1", "1")) +
  ylab("Effective number of species (ENS)") +
  scale_color_manual(name = "Behavioural scenario",
                     values = c("#69b3a2", "#ffa600", "#404080")) +
  facet_wrap(~ Fragmentation, nrow = 1)

Reviewer

ggsave(filename = "plots/PlotReview.pdf", plot = Reviewer,
       width = 290, height = 150, units = "mm", dpi = 300, 
       device = cairo_pdf) 


sumWide <- tidyr::spread(sumDiv, key = Behaviour, value = meanENS)

sumWide$DiffShy <- (sumWide$`risk-avoiding` - sumWide$`risk-diverse`) / sumWide$`risk-diverse` * 100

summary(sumWide$DiffShy)

sumWide$DiffBold <- (sumWide$`risk-seeking` - sumWide$`risk-diverse`) / sumWide$`risk-diverse` * 100
summary(sumWide$DiffBold)

sumLong <- tidyr::pivot_longer(sumWide, cols = c("DiffBold", "DiffShy"), names_to = "Behaviour", values_to = "Diff")

sumLong <- sumLong %>%
  mutate(Fragmentation = factor(Fragmentation))  %>% 
  mutate(Fragmentation = fct_relevel(Fragmentation,
                                     "weak",
                                     "moderate",
                                     "high",
                                     "very high"))

FigCompare <- ggplot(sumLong, aes(x = as.factor(MortalityChance * 100), 
                                  y = Diff, fill = Behaviour)) +
  geom_bar(position = "dodge", stat = "identity") +
  labs(x = "Matrix mortality [%]", y = "Chnages in ENS [%]") +
  scale_y_continuous(name = "Change from ENS in\nheterogeneous risk-taking community [%]",
                     breaks = c(10, 0, -10, -20, -30, -40, -50, -60, -70),
                     labels = c("+10", "0", "-10", "-20", "-30", 
                                "-40",  "-50", "-60", "-70")) +
  scale_x_discrete(breaks = c(0, 0.001, 0.003, 0.01, 0.03, 0.1, 0.3, 1, 100),
                   labels = c("No costs", "0.001", "0.003", "0.01", 
                              "0.03", "0.1", "0.3", "1", "Deferred costs")) +
  geom_hline(yintercept = 0, color = "#ffa600", linewidth = 1, 
             show.legend = TRUE) +
  scale_fill_manual(name = "Behavioural scenario",
                    values = c("#69b3a2", "#404080"),
                    labels = c("Homogeneous\nrisk-seeking community", 
                               "Homogeneous\nrisk-averse community")) +
  facet_wrap(. ~ Fragmentation, nrow = 1) +
  theme(axis.text.x = element_text(angle = 45, hjust = 1, family = "CMU Bright"),
        strip.text = element_text(hjust = 0.5, family = "CMU Bright"))

FigCompare

ggsave(filename = "plots/Plot6a.pdf", plot = FigCompare,
       width = 210, height = 180, units = "mm", dpi = 300, 
       device = cairo_pdf) 



