# scRNAseq data analysis
# Saeki et al, 2022



########################################################################
###### Environment #####################################################
########################################################################

### Library & color palette

library(Seurat)
library(ggplot2)
library(cowplot)
library(wesanderson)
library(dplyr)
library(FactoMineR)
library(factoextra)
library(treeio)
library(ggtree)
library(ape)
library(sets)
FP <- wes_palette("Zissou1", 5, type = "discrete")

### Working directory
setwd("...zenodo/your_folder")

### Session info
# > sessionInfo()
# R version 4.1.2 (2021-11-01)
# Platform: x86_64-w64-mingw32/x64 (64-bit)
# Running under: Windows 10 x64 (build 22000)
# [1] sets_1.0-20        ape_5.5            ggtree_3.2.1       treeio_1.18.1      factoextra_1.0.7   FactoMineR_2.4     dplyr_1.0.9       
# [8] wesanderson_0.3.6  cowplot_1.1.1      ggplot2_3.3.5      SeuratObject_4.0.4 Seurat_4.0.5      



########################################################################
###### Seurat object preparation #######################################
########################################################################

# Data can be obtained from the Gene Expression Omnibus database (accession number: GSE191219)
# ("https://www.ncbi.nlm.nih.gov/geo/query/acc.cgi?acc=GSE191219")
# GSM5741807 Intact
# GSM5741808 VCD
# GSM5741809 VCD+E2
# GSM5741810 VCD+E2+P4
# GSM5741811 VCD+LP
# GSM5741812 VCD+E2+LP
# GSM5741813 VCD+E2+P4+LP
# GSM5741815 VCD+HP
# GSM5741816 VCD+E2+HP
# GSM5741817 VCD+E2+P4+HP
# GSM5741818 VCD+E2+ICI

## Intact
data_dir <- "...your_folder/GSM5741807/filtered_feature_bc_matrix"
list.files(data_dir) # Should show barcodes.tsv.gz, features.tsv.gz, and matrix.mtx.gz
expression_matrix <- Read10X(data.dir = data_dir)
Intact <- CreateSeuratObject(counts = expression_matrix, project = "Intact", min.cells = 3, min.features = 200)
Intact <- RenameCells(Intact, add.cell.id = "Intact")
Intact[["percent.mt"]] <- PercentageFeatureSet(Intact, pattern = "^mt-")
remove(expression_matrix)
gc();gc()

## VCD
data_dir <- "...your_folder/GSM5741808/filtered_feature_bc_matrix"
list.files(data_dir) # Should show barcodes.tsv.gz, features.tsv.gz, and matrix.mtx.gz
expression_matrix <- Read10X(data.dir = data_dir)
VCD_Vehicle <- CreateSeuratObject(counts = expression_matrix, project = "VCD", min.cells = 3, min.features = 200)
VCD_Vehicle <- RenameCells(VCD_Vehicle, add.cell.id = "VCD")
VCD_Vehicle[["percent.mt"]] <- PercentageFeatureSet(VCD_Vehicle, pattern = "^mt-")
remove(expression_matrix)
gc();gc()

## VCD+E2
data_dir <- "...your_folder/GSM5741809/filtered_feature_bc_matrix"
list.files(data_dir) # Should show barcodes.tsv.gz, features.tsv.gz, and matrix.mtx.gz
expression_matrix <- Read10X(data.dir = data_dir)
VCD_E2 <- CreateSeuratObject(counts = expression_matrix, project = "VCD+E2", min.cells = 3, min.features = 200)
VCD_E2 <- RenameCells(VCD_E2, add.cell.id = "VCD+E2")
VCD_E2[["percent.mt"]] <- PercentageFeatureSet(VCD_E2, pattern = "^mt-")
remove(expression_matrix)
gc();gc()

## VCD+E2+P4
data_dir <- "...your_folder/GSM5741810/filtered_feature_bc_matrix"
list.files(data_dir) # Should show barcodes.tsv.gz, features.tsv.gz, and matrix.mtx.gz
expression_matrix <- Read10X(data.dir = data_dir)
VCD_E2_P4 <- CreateSeuratObject(counts = expression_matrix, project = "VCD+E2+P4", min.cells = 3, min.features = 200)
VCD_E2_P4 <- RenameCells(VCD_E2_P4, add.cell.id = "VCD+E2+P4")
VCD_E2_P4[["percent.mt"]] <- PercentageFeatureSet(VCD_E2_P4, pattern = "^mt-")
remove(expression_matrix)
gc();gc()

## VCD+LP
data_dir <- "...your_folder/GSM5741811/filtered_feature_bc_matrix"
list.files(data_dir) # Should show barcodes.tsv.gz, features.tsv.gz, and matrix.mtx.gz
expression_matrix <- Read10X(data.dir = data_dir)
VCD_envPBDE <- CreateSeuratObject(counts = expression_matrix, project = "VCD+LP", min.cells = 3, min.features = 200)
VCD_envPBDE <- RenameCells(VCD_envPBDE, add.cell.id = "VCD+LP")
VCD_envPBDE[["percent.mt"]] <- PercentageFeatureSet(VCD_envPBDE, pattern = "^mt-")
remove(expression_matrix)
gc();gc()

## VCD+E2+LP
data_dir <- "...your_folder/GSM5741812/filtered_feature_bc_matrix"
list.files(data_dir) # Should show barcodes.tsv.gz, features.tsv.gz, and matrix.mtx.gz
expression_matrix <- Read10X(data.dir = data_dir)
VCD_envPBDE_E2 <- CreateSeuratObject(counts = expression_matrix, project = "VCD+E2+LP", min.cells = 3, min.features = 200)
VCD_envPBDE_E2 <- RenameCells(VCD_envPBDE_E2, add.cell.id = "VCD+E2+LP")
VCD_envPBDE_E2[["percent.mt"]] <- PercentageFeatureSet(VCD_envPBDE_E2, pattern = "^mt-")
remove(expression_matrix)
gc();gc()

## VCD+E2+P4+LP
data_dir <- "...your_folder/GSM5741813/filtered_feature_bc_matrix"
list.files(data_dir) # Should show barcodes.tsv.gz, features.tsv.gz, and matrix.mtx.gz
expression_matrix <- Read10X(data.dir = data_dir)
VCD_envPBDE_E2_P4 <- CreateSeuratObject(counts = expression_matrix, project = "VCD+E2+P4+LP", min.cells = 3, min.features = 200)
VCD_envPBDE_E2_P4 <- RenameCells(VCD_envPBDE_E2_P4, add.cell.id = "VCD+E2+P4+LP")
VCD_envPBDE_E2_P4[["percent.mt"]] <- PercentageFeatureSet(VCD_envPBDE_E2_P4, pattern = "^mt-")
remove(expression_matrix)
gc();gc()


## VCD+HP
data_dir <- "...your_folder/GSM5741815/filtered_feature_bc_matrix"
list.files(data_dir) # Should show barcodes.tsv.gz, features.tsv.gz, and matrix.mtx.gz
expression_matrix <- Read10X(data.dir = data_dir)
VCD_expPBDE <- CreateSeuratObject(counts = expression_matrix, project = "VCD+HP", min.cells = 3, min.features = 200)
VCD_expPBDE <- RenameCells(VCD_expPBDE, add.cell.id = "VCD+HP")
VCD_expPBDE[["percent.mt"]] <- PercentageFeatureSet(VCD_expPBDE, pattern = "^mt-")
remove(expression_matrix)
gc();gc()

## VCD+E2+HP
data_dir <- "...your_folder/GSM5741816/filtered_feature_bc_matrix"
list.files(data_dir) # Should show barcodes.tsv.gz, features.tsv.gz, and matrix.mtx.gz
expression_matrix <- Read10X(data.dir = data_dir)
VCD_expPBDE_E2 <- CreateSeuratObject(counts = expression_matrix, project = "VCD+E2+HP", min.cells = 3, min.features = 200)
VCD_expPBDE_E2 <- RenameCells(VCD_expPBDE_E2, add.cell.id = "VCD+E2+HP")
VCD_expPBDE_E2[["percent.mt"]] <- PercentageFeatureSet(VCD_expPBDE_E2, pattern = "^mt-")
remove(expression_matrix)
gc();gc()

## VCD+E2+P4+HP
data_dir <- "...your_folder/GSM5741817/filtered_feature_bc_matrix"
list.files(data_dir) # Should show barcodes.tsv.gz, features.tsv.gz, and matrix.mtx.gz
expression_matrix <- Read10X(data.dir = data_dir)
VCD_expPBDE_E2_P4 <- CreateSeuratObject(counts = expression_matrix, project = "VCD+E2+P4+HP", min.cells = 3, min.features = 200)
VCD_expPBDE_E2_P4 <- RenameCells(VCD_expPBDE_E2_P4, add.cell.id = "VCD+E2+P4+HP")
VCD_expPBDE_E2_P4[["percent.mt"]] <- PercentageFeatureSet(VCD_expPBDE_E2_P4, pattern = "^mt-")
remove(expression_matrix)
gc();gc()

## VCD+E2+ICI
data_dir <- "...your_folder/GSM5741807/filtered_feature_bc_matrix"
list.files(data_dir) # Should show barcodes.tsv.gz, features.tsv.gz, and matrix.mtx.gz
expression_matrix <- Read10X(data.dir = data_dir)
VCD_E2_ICI <- CreateSeuratObject(counts = expression_matrix, project = "VCD+E2+ICI", min.cells = 3, min.features = 200)
VCD_E2_ICI <- RenameCells(VCD_E2_ICI, add.cell.id = "VCD+E2+ICI")
VCD_E2_ICI[["percent.mt"]] <- PercentageFeatureSet(VCD_E2_ICI, pattern = "^mt-")
remove(expression_matrix)
gc();gc()


########################################################################
###### Data merging  ###################################################
########################################################################
seu <- merge(Intact, c(VCD_E2, VCD_E2_ICI, VCD_E2_P4, VCD_envPBDE, VCD_envPBDE_E2,
                       VCD_envPBDE_E2_P4, VCD_expPBDE, VCD_expPBDE_E2, VCD_expPBDE_E2_P4,
                       VCD_Vehicle), add.cell.ids = NULL, project = "VCD-PBDE")

remove(Intact, VCD_E2, VCD_E2_ICI, VCD_E2_P4, VCD_envPBDE, VCD_envPBDE_E2,
       VCD_envPBDE_E2_P4, VCD_expPBDE, VCD_expPBDE_E2, VCD_expPBDE_E2_P4,
       VCD_Vehicle)
gc();gc()

seu <- subset(seu, subset = nFeature_RNA > 500 & percent.mt < 10)
saveRDS(seu, "./Seurat_merged.rds")


########################################################################
###### Processing  #####################################################
########################################################################
seu <- readRDS("./Seurat_merged.rds")
seu$orig.ident <- as.factor(seu$orig.ident)
levels(seu$orig.ident)
seu$orig.ident <- factor(seu$orig.ident, levels = c("Intact", "VCD", "VCD+E2", "VCD+E2+P4", "VCD+HP", "VCD+E2+HP", "VCD+E2+P4+HP", "VCD+LP",
                                                    "VCD+E2+LP", "VCD+E2+P4+LP", "VCD+E2+ICI"))
Idents(seu) <- "orig.ident"

p1 <- VlnPlot(seu, "nFeature_RNA", pt.size=0)+ 
  geom_boxplot(width=0.15, fill="white")+
  NoLegend()+
  ggtitle("nFeature_RNA")+
  xlab("")+
  ylab("Gene count")+ 
  theme(plot.title = element_text(size=18, face="bold"),
        axis.text=element_text(size=10),
        axis.title=element_text(size=12,face="bold"))
p2 <- VlnPlot(seu, "nCount_RNA", pt.size=0)+ 
  geom_boxplot(width=0.15, fill="white")+
  NoLegend()+
  ggtitle("nCount_RNA")+
  xlab("")+
  ylab("UMI count")+
  scale_y_continuous(trans='log10')+ 
  theme(plot.title = element_text(size=18, face="bold"),
        axis.text=element_text(size=10),
        axis.title=element_text(size=12,face="bold"))
p3 <- VlnPlot(seu, "percent.mt", pt.size=0)+ 
  geom_boxplot(width=0.15, fill="white")+
  NoLegend()+
  ggtitle("percent.mt")+
  xlab("")+
  ylab("Mito gene %")+ 
  theme(plot.title = element_text(size=18, face="bold"),
        axis.text=element_text(size=10),
        axis.title=element_text(size=12,face="bold"))

p <- plot_grid(p1, p2, p3, ncol=3)
ggsave("./SFig5A.pdf", plot=p, width = 12, height = 5)
ggsave("./SFig5A.png", plot=p, width = 12, height = 5)


###

## Cell cycle
ConvertHumanGeneListToMM <- function(x){
  # Load human ensembl attributes
  human <- useMart("ensembl", dataset = "hsapiens_gene_ensembl", host = "https://dec2021.archive.ensembl.org/")
  # Load mouse ensembl attributes
  mouse <- useMart("ensembl", dataset = "mmusculus_gene_ensembl", host = "https://dec2021.archive.ensembl.org/")
  # Link both datasets and retrieve mouse genes from the human genes
  genes.list = biomaRt::getLDS(attributes = c("hgnc_symbol"), filters = "hgnc_symbol", values = x , mart = human, attributesL = c("mgi_symbol"), martL = mouse, uniqueRows = T)
  # Get unique names of genes (in case gene names are duplicated)
  mouse.gene.list <- unique(genes.list[, 2])
  # # Print the first 6 genes found to the screen
  # print(head(mouse.gene.list))
  return(mouse.gene.list)
}
data(cc.genes.updated.2019)
mouse.cc.genes <- lapply(X = cc.genes.updated.2019, 
                         ConvertHumanGeneListToMM)
s.genes <- mouse.cc.genes$s.genes
g2m.genes <- mouse.cc.genes$g2m.genes

seu <- NormalizeData(seu)
seu <- FindVariableFeatures(seu, selection.method = "vst", nfeatures = 2000)
seu <- ScaleData(seu)
seu <- RunPCA(seu, verbose = T)
seu <- RunUMAP(seu, dims = 1:30, verbose = T)
seu <- FindNeighbors(seu, dims = 1:30, verbose = T)
seu <- FindClusters(seu, resolution = 0.1, verbose = T)

UMAPPlot(seu, label=T, label.size =6)
UMAPPlot(seu, cells.highlight = WhichCells(seu, ident=7))

seu <- CellCycleScoring(seu, s.features = s.genes, g2m.features = g2m.genes, set.ident = F)

p1 <- UMAPPlot(seu, label=T, label.size=8)+
  NoLegend()+
  theme(plot.title = element_text(size=18, face="bold"),
        axis.text=element_text(size=10),
        axis.title=element_text(size=12,face="bold"))+
  ggtitle("Dim=30, res=0.1")
p2 <- UMAPPlot(seu, label=F, group.by="orig.ident")+
  theme(plot.title = element_text(size=18, face="bold"),
        axis.text=element_text(size=10),
        axis.title=element_text(size=12,face="bold"))+
  ggtitle("Sample")
p3 <- UMAPPlot(seu, label=F, group.by="Phase")+
  theme(plot.title = element_text(size=18, face="bold"),
        axis.text=element_text(size=10),
        axis.title=element_text(size=12,face="bold"))+
  ggtitle("Cell cycle")

p <- plot_grid(p1, p2, p3, ncol=3)
ggsave("./SFig5B.pdf", plot=p, width = 15, height = 5)
ggsave("./SFig5B.png", plot=p, width = 15, height = 5)

###
p1 <- VlnPlot(seu, "nFeature_RNA", pt.size=0, group.by = "seurat_clusters")+ 
  geom_boxplot(width=0.15, fill="white")+
  NoLegend()+
  ggtitle("nFeature_RNA")+
  xlab("")+
  ylab("Gene count")+ 
  theme(plot.title = element_text(size=18, face="bold"),
        axis.text=element_text(size=10),
        axis.title=element_text(size=12,face="bold"))
p2 <- VlnPlot(seu, "nCount_RNA", pt.size=0, group.by = "seurat_clusters")+ 
  geom_boxplot(width=0.15, fill="white")+
  NoLegend()+
  ggtitle("nCount_RNA")+
  xlab("")+
  ylab("UMI count")+
  scale_y_continuous(trans='log10')+ 
  theme(plot.title = element_text(size=18, face="bold"),
        axis.text=element_text(size=10),
        axis.title=element_text(size=12,face="bold"))
p3 <- VlnPlot(seu, "percent.mt", pt.size=0, group.by = "seurat_clusters")+ 
  geom_boxplot(width=0.15, fill="white")+
  NoLegend()+
  ggtitle("percent.mt")+
  xlab("")+
  ylab("Mito gene %")+ 
  theme(plot.title = element_text(size=18, face="bold"),
        axis.text=element_text(size=10),
        axis.title=element_text(size=12,face="bold"))

p <- plot_grid(p1, p2, p3, ncol=3)

ggsave("./SFig5C.pdf", plot=p, width = 14, height = 5)
ggsave("./SFig5C.png", plot=p, width = 14, height = 5)

###
markers <- FindAllMarkers(seu, only.pos = TRUE, min.pct = 0.25, logfc.threshold = 0.25)
write.csv(markers, "./DEG_list.csv")

markers <- read.csv("./DEG_list.csv", row.names=1)
top5 <- markers %>% group_by(cluster) %>% top_n(n = 5, wt = avg_log2FC)
palette <- wes_palette("Zissou1", 100, type = "continuous")
p <- DoHeatmap(seu, features = as.character(top5$gene), label = T) +
  scale_fill_gradientn(colors = palette) 

ggsave("./SFig5D.pdf", plot=p, width = 20, height = 10)
ggsave("./SFig5D.png", plot=p, width = 20, height = 10)


###
FP <- wes_palette("Zissou1", 5, type = "discrete")

p1 <- FeaturePlot(seu, "Col1a1", cols = FP, 
                  pt.size = 1, min.cutoff = "q10", max.cutoff = "q90")+
  NoAxes()+
  theme(plot.title = element_text(size=18, face="bold"),
        axis.text=element_text(size=10,face="bold"),
        axis.title=element_text(size=12,face="bold"))+
  ggtitle("Col1a1")

p2 <- VlnPlot(seu, "Col1a1", pt.size=0)+ 
  geom_boxplot(width=0.15, fill="white")+
  NoLegend()+
  ggtitle("percent.mt")+
  xlab("")+
  ylab("Mito gene %")+ 
  theme(plot.title = element_text(size=18, face="bold"),
        axis.text=element_text(size=10),
        axis.title=element_text(size=12,face="bold"))+
  ggtitle("Col1a1")

q1 <- plot_grid(p1, p2,ncol=2)
ggsave("./SFig5E_Col1a1.pdf", plot=q1, width = 10, height = 5)
ggsave("./SFig5E_Col1a1.png", plot=q1, width = 10, height = 5)


###
p1 <- FeaturePlot(seu, "Lyz2", cols = FP, 
                  pt.size = 1, min.cutoff = "q10", max.cutoff = "q90")+
  NoAxes()+
  theme(plot.title = element_text(size=18, face="bold"),
        axis.text=element_text(size=10,face="bold"),
        axis.title=element_text(size=12,face="bold"))+
  ggtitle("Lyz2")

p2 <- VlnPlot(seu, "Lyz2", pt.size=0)+ 
  geom_boxplot(width=0.15, fill="white")+
  NoLegend()+
  ggtitle("percent.mt")+
  xlab("")+
  ylab("Mito gene %")+ 
  theme(plot.title = element_text(size=18, face="bold"),
        axis.text=element_text(size=10),
        axis.title=element_text(size=12,face="bold"))+
  ggtitle("Lyz2")

q2 <- plot_grid(p1, p2,ncol=2)
ggsave("./SFig5E_Lyz2.pdf", plot=q2, width = 10, height = 5)
ggsave("./SFig5E_Lyz2.png", plot=q2, width = 10, height = 5)


###
p1 <- FeaturePlot(seu, "Gata3", cols = FP, 
                  pt.size = 1, min.cutoff = "q10", max.cutoff = "q90")+
  NoAxes()+
  theme(plot.title = element_text(size=18, face="bold"),
        axis.text=element_text(size=10,face="bold"),
        axis.title=element_text(size=12,face="bold"))+
  ggtitle("Gata3")

p2 <- VlnPlot(seu, "Gata3", pt.size=0)+ 
  geom_boxplot(width=0.15, fill="white")+
  NoLegend()+
  ggtitle("percent.mt")+
  xlab("")+
  ylab("Mito gene %")+ 
  theme(plot.title = element_text(size=18, face="bold"),
        axis.text=element_text(size=10),
        axis.title=element_text(size=12,face="bold"))+
  ggtitle("Gata3")

q3 <- plot_grid(p1, p2,ncol=2)
ggsave("./SFig5E_Gata3.pdf", plot=q3, width = 10, height = 5)
ggsave("./SFig5E_Gata3.png", plot=q3, width = 10, height = 5)


###
p1 <- FeaturePlot(seu, "Nkg7", cols = FP, 
                  pt.size = 1, min.cutoff = "q10", max.cutoff = "q90")+
  NoAxes()+
  theme(plot.title = element_text(size=18, face="bold"),
        axis.text=element_text(size=10,face="bold"),
        axis.title=element_text(size=12,face="bold"))+
  ggtitle("Nkg7")

p2 <- VlnPlot(seu, "Nkg7", pt.size=0)+ 
  geom_boxplot(width=0.15, fill="white")+
  NoLegend()+
  ggtitle("percent.mt")+
  xlab("")+
  ylab("Mito gene %")+ 
  theme(plot.title = element_text(size=18, face="bold"),
        axis.text=element_text(size=10),
        axis.title=element_text(size=12,face="bold"))+
  ggtitle("Nkg7")

q4 <- plot_grid(p1, p2,ncol=2)
ggsave("./SFig5E_Nkg7.pdf", plot=q4, width = 10, height = 5)
ggsave("./SFig5E_Nkg7.png", plot=q4, width = 10, height = 5)


###
p1 <- FeaturePlot(seu, "Jchain", cols = FP, 
                  pt.size = 1, min.cutoff = "q10", max.cutoff = "q90")+
  NoAxes()+
  theme(plot.title = element_text(size=18, face="bold"),
        axis.text=element_text(size=10,face="bold"),
        axis.title=element_text(size=12,face="bold"))+
  ggtitle("Jchain")

p2 <- VlnPlot(seu, "Jchain", pt.size=0)+ 
  geom_boxplot(width=0.15, fill="white")+
  NoLegend()+
  ggtitle("percent.mt")+
  xlab("")+
  ylab("Mito gene %")+ 
  theme(plot.title = element_text(size=18, face="bold"),
        axis.text=element_text(size=10),
        axis.title=element_text(size=12,face="bold"))+
  ggtitle("Jchain")

q5 <- plot_grid(p1, p2,ncol=2)
ggsave("./SFig5E_Jchain.pdf", plot=q5, width = 10, height = 5)
ggsave("./SFig5E_Jchain.png", plot=q5, width = 10, height = 5)


###
p1 <- FeaturePlot(seu, "Pecam1", cols = FP, 
                  pt.size = 1, min.cutoff = "q10", max.cutoff = "q90")+
  NoAxes()+
  theme(plot.title = element_text(size=18, face="bold"),
        axis.text=element_text(size=10,face="bold"),
        axis.title=element_text(size=12,face="bold"))+
  ggtitle("Pecam1")

p2 <- VlnPlot(seu, "Pecam1", pt.size=0)+ 
  geom_boxplot(width=0.15, fill="white")+
  NoLegend()+
  ggtitle("percent.mt")+
  xlab("")+
  ylab("Mito gene %")+ 
  theme(plot.title = element_text(size=18, face="bold"),
        axis.text=element_text(size=10),
        axis.title=element_text(size=12,face="bold"))+
  ggtitle("Pecam1")

q6 <- plot_grid(p1, p2,ncol=2)
ggsave("./SFig5E_Pecam1.pdf", plot=q6, width = 10, height = 5)
ggsave("./SFig5E_Pecam1.png", plot=q6, width = 10, height = 5)


###
p1 <- FeaturePlot(seu, "Top2a", cols = FP, 
                  pt.size = 1, min.cutoff = "q10", max.cutoff = "q90")+
  NoAxes()+
  theme(plot.title = element_text(size=18, face="bold"),
        axis.text=element_text(size=10,face="bold"),
        axis.title=element_text(size=12,face="bold"))+
  ggtitle("Top2a")

p2 <- VlnPlot(seu, "Top2a", pt.size=0)+ 
  geom_boxplot(width=0.15, fill="white")+
  NoLegend()+
  ggtitle("percent.mt")+
  xlab("")+
  ylab("Mito gene %")+ 
  theme(plot.title = element_text(size=18, face="bold"),
        axis.text=element_text(size=10),
        axis.title=element_text(size=12,face="bold"))+
  ggtitle("Top2a")

q7 <- plot_grid(p1, p2,ncol=2)
ggsave("./SFig5E_Top2a.pdf", plot=q7, width = 10, height = 5)
ggsave("./SFig5E_Top2a.png", plot=q7, width = 10, height = 5)


###
p1 <- FeaturePlot(seu, "Krt18", cols = FP, 
                  pt.size = 1, min.cutoff = "q10", max.cutoff = "q90")+
  NoAxes()+
  theme(plot.title = element_text(size=18, face="bold"),
        axis.text=element_text(size=10,face="bold"),
        axis.title=element_text(size=12,face="bold"))+
  ggtitle("Krt18")

p2 <- VlnPlot(seu, "Krt18", pt.size=0)+ 
  geom_boxplot(width=0.15, fill="white")+
  NoLegend()+
  ggtitle("percent.mt")+
  xlab("")+
  ylab("Mito gene %")+ 
  theme(plot.title = element_text(size=18, face="bold"),
        axis.text=element_text(size=10),
        axis.title=element_text(size=12,face="bold"))+
  ggtitle("Krt18")

q8 <- plot_grid(p1, p2,ncol=2)
ggsave("./SFig5E_Krt18.pdf", plot=q8, width = 10, height = 5)
ggsave("./SFig5E_Krt18.png", plot=q8, width = 10, height = 5)


###
p1 <- FeaturePlot(seu, "Mcpt4", cols = FP, 
                  pt.size = 1, min.cutoff = "q10", max.cutoff = "q90")+
  NoAxes()+
  theme(plot.title = element_text(size=18, face="bold"),
        axis.text=element_text(size=10,face="bold"),
        axis.title=element_text(size=12,face="bold"))+
  ggtitle("Mcpt4")

p2 <- VlnPlot(seu, "Mcpt4", pt.size=0)+ 
  geom_boxplot(width=0.15, fill="white")+
  NoLegend()+
  ggtitle("percent.mt")+
  xlab("")+
  ylab("Mito gene %")+ 
  theme(plot.title = element_text(size=18, face="bold"),
        axis.text=element_text(size=10),
        axis.title=element_text(size=12,face="bold"))+
  ggtitle("Mcpt4")

q9 <- plot_grid(p1, p2,ncol=2)
ggsave("./SFig5E_Mcpt4.pdf", plot=q9, width = 10, height = 5)
ggsave("./SFig5E_Mcpt4.png", plot=q9, width = 10, height = 5)


###
p1 <- FeaturePlot(seu, "Cd79a", cols = FP, 
                  pt.size = 1, min.cutoff = "q10", max.cutoff = "q90")+
  NoAxes()+
  theme(plot.title = element_text(size=18, face="bold"),
        axis.text=element_text(size=10,face="bold"),
        axis.title=element_text(size=12,face="bold"))+
  ggtitle("Cd79a")

p2 <- VlnPlot(seu, "Cd79a", pt.size=0)+ 
  geom_boxplot(width=0.15, fill="white")+
  NoLegend()+
  ggtitle("percent.mt")+
  xlab("")+
  ylab("Mito gene %")+ 
  theme(plot.title = element_text(size=18, face="bold"),
        axis.text=element_text(size=10),
        axis.title=element_text(size=12,face="bold"))+
  ggtitle("Cd79a")

q10 <- plot_grid(p1, p2,ncol=2)
ggsave("./SFig5E_Cd79a.pdf", plot=q10, width = 10, height = 5)
ggsave("./SFig5E_Cd79a.png", plot=q10, width = 10, height = 5)


###
p1 <- FeaturePlot(seu, "Retnlg", cols = FP, 
                  pt.size = 1, min.cutoff = "q10", max.cutoff = "q90")+
  NoAxes()+
  theme(plot.title = element_text(size=18, face="bold"),
        axis.text=element_text(size=10,face="bold"),
        axis.title=element_text(size=12,face="bold"))+
  ggtitle("Retnlg")

p2 <- VlnPlot(seu, "Retnlg", pt.size=0)+ 
  geom_boxplot(width=0.15, fill="white")+
  NoLegend()+
  ggtitle("percent.mt")+
  xlab("")+
  ylab("Mito gene %")+ 
  theme(plot.title = element_text(size=18, face="bold"),
        axis.text=element_text(size=10),
        axis.title=element_text(size=12,face="bold"))+
  ggtitle("Retnlg")

q11 <- plot_grid(p1, p2,ncol=2)
ggsave("./SFig5E_Retnlg.pdf", plot=q11, width = 10, height = 5)
ggsave("./SFig5E_Retnlg.png", plot=q11, width = 10, height = 5)


###
p1 <- FeaturePlot(seu, "Mpz", cols = FP, 
                  pt.size = 1, min.cutoff = "q10", max.cutoff = "q90")+
  NoAxes()+
  theme(plot.title = element_text(size=18, face="bold"),
        axis.text=element_text(size=10,face="bold"),
        axis.title=element_text(size=12,face="bold"))+
  ggtitle("Mpz")

p2 <- VlnPlot(seu, "Mpz", pt.size=0)+ 
  geom_boxplot(width=0.15, fill="white")+
  NoLegend()+
  ggtitle("percent.mt")+
  xlab("")+
  ylab("Mito gene %")+ 
  theme(plot.title = element_text(size=18, face="bold"),
        axis.text=element_text(size=10),
        axis.title=element_text(size=12,face="bold"))+
  ggtitle("Mpz")

q12 <- plot_grid(p1, p2,ncol=2)
ggsave("./SFig5E_Mpz.pdf", plot=q12, width = 10, height = 5)
ggsave("./SFig5E_Mpz.png", plot=q12, width = 10, height = 5)


###
p1 <- FeaturePlot(seu, "Acta1", cols = FP, 
                  pt.size = 1, min.cutoff = "q10", max.cutoff = "q90")+
  NoAxes()+
  theme(plot.title = element_text(size=18, face="bold"),
        axis.text=element_text(size=10,face="bold"),
        axis.title=element_text(size=12,face="bold"))+
  ggtitle("Acta1")

p2 <- VlnPlot(seu, "Acta1", pt.size=0)+ 
  geom_boxplot(width=0.15, fill="white")+
  NoLegend()+
  ggtitle("percent.mt")+
  xlab("")+
  ylab("Mito gene %")+ 
  theme(plot.title = element_text(size=18, face="bold"),
        axis.text=element_text(size=10),
        axis.title=element_text(size=12,face="bold"))+
  ggtitle("Acta1")

q13 <- plot_grid(p1, p2,ncol=2)
ggsave("./SFig5E_Acta1.pdf", plot=q13, width = 10, height = 5)
ggsave("./SFig5E_Acta1.png", plot=q13, width = 10, height = 5)


###
p1 <- FeaturePlot(seu, "Krt14", cols = FP, 
                  pt.size = 1, min.cutoff = "q10", max.cutoff = "q90")+
  NoAxes()+
  theme(plot.title = element_text(size=18, face="bold"),
        axis.text=element_text(size=10,face="bold"),
        axis.title=element_text(size=12,face="bold"))+
  ggtitle("Krt14")

p2 <- VlnPlot(seu, "Krt14", pt.size=0)+ 
  geom_boxplot(width=0.15, fill="white")+
  NoLegend()+
  ggtitle("percent.mt")+
  xlab("")+
  ylab("Mito gene %")+ 
  theme(plot.title = element_text(size=18, face="bold"),
        axis.text=element_text(size=10),
        axis.title=element_text(size=12,face="bold"))+
  ggtitle("Krt14")

q14 <- plot_grid(p1, p2,ncol=2)
ggsave("./SFig5E_Krt14.pdf", plot=q14, width = 10, height = 4)
ggsave("./SFig5E_Krt14.png", plot=q14, width = 10, height = 4)


###
q <- plot_grid(q1, q2, q3, q4, q5, q6, q7, q8, q9,
               q10, q11, q12, q13, q14, ncol=3)
ggsave("./SFig5E.pdf", plot=q, width = 20, height = 20)
ggsave("./SFig5E.png", plot=q, width = 20, height = 20)


###
Idents(seu) <- "seurat_clusters"
new.cluster.ids <- c("Fibroblast_1", "Fibroblast_2", "Macrophage", "T+NK",
                     "Fibroblast_3", "Mixed_lineage", "Plasmocyte", "Endothelial+Basal",
                     "Proliferating", "Luminal", "Mastocyte", "B", "Neutrophil",
                     "Nerve_sheath", "Striated_muscle")
names(new.cluster.ids) <- levels(seu)
seu <- RenameIdents(seu, new.cluster.ids)

UMAPPlot(seu, label=T, label.size=8)+
  NoLegend()+
  theme(plot.title = element_text(size=18, face="bold"),
        axis.text=element_text(size=10),
        axis.title=element_text(size=12,face="bold"))

seu$Annotation <- seu@active.ident
meta <- seu@meta.data

colnames(meta)
meta <- meta[,c("orig.ident" , "nCount_RNA", "nFeature_RNA", "percent.mt", "seurat_clusters",
                "S.Score", "G2M.Score", "Phase", "Annotation" )]
write.csv(meta, "./meta_data.csv")
saveRDS(seu, "./Seurat_merged_annotated.rds")



########################################################################
###### Analysis  #######################################################
########################################################################
seu <- readRDS("./Seurat_merged.rds")
meta <-read.csv("./meta_data.csv", header=T, row.names = 1)

seu$orig.ident <- as.factor(seu$orig.ident)
levels(seu$orig.ident)
seu$orig.ident <- factor(seu$orig.ident, levels = c("Intact", "VCD", "VCD+E2", "VCD+E2+P4", "VCD+HP", "VCD+E2+HP", "VCD+E2+P4+HP", "VCD+LP",
                                                    "VCD+E2+LP", "VCD+E2+P4+LP", "VCD+E2+ICI"))

seu <- NormalizeData(seu)
seu <- FindVariableFeatures(seu, selection.method = "vst", nfeatures = 2000)
seu <- ScaleData(seu)
seu <- RunPCA(seu, verbose = T)
seu <- RunUMAP(seu, dims = 1:30, verbose = T)
seu <- AddMetaData(seu, metadata = meta)

Idents(seu) <- "Annotation"

### OR ###
setwd("...your_folder")
seu <- readRDS("./Seurat_merged_annotated.rds")
meta <-read.csv("./meta_data.csv", header=T, row.names = 1)
seu <- AddMetaData(seu, metadata = meta)

table(seu@active.ident)

seu <- subset(seu, idents = c("Fibroblast_1", "Fibroblast_2", "Macrophage", "T+NK",
                              "Fibroblast_3", "Plasmocyte", "Endothelial+Basal",
                              "Proliferating", "Luminal", "Mastocyte", "B", "Neutrophil"))

p <- UMAPPlot(seu, label=T, label.size=4.5)+
  NoLegend()+
  theme(plot.title = element_text(size=18, face="bold"),
        axis.text=element_text(size=10),
        axis.title=element_text(size=12,face="bold"))+
  ggtitle("VCD_mammary_gland")+
  xlim(-15, 17)

ggsave("./Fig5B.pdf", plot=p, width = 5, height = 5)
ggsave("./Fig5B.png", plot=p, width = 5, height = 5)



##
df <- as.data.frame(table(seu$Annotation, seu$orig.ident))
colnames(df) <- c("Cluster", "Sample", "Count")
label <- c("Fibroblast_1", "Fibroblast_2", "Macrophage", "T+NK",
           "Fibroblast_3", "Plasmocyte", "Endothelial+Basal",
           "Proliferating", "Luminal", "Mastocyte", "B", "Neutrophil")
label <- factor(label, levels = c("Fibroblast_1", "Fibroblast_2", "Macrophage", "T+NK",
                                  "Fibroblast_3", "Plasmocyte", "Endothelial+Basal",
                                  "Proliferating", "Luminal", "Mastocyte", "B", "Neutrophil"))
hoge <- list()

for(i in 1:12){
  temp <- df[df$Cluster == label[i],]
  temp$fraction <- temp$Count / sum(temp$Count)
  temp$ymax <- cumsum(temp$fraction)
  temp$ymin <- c(0, head(temp$ymax, n=-1))
  L1 <- paste("n=", sum(temp$Count), sep = "")
  L2 <- paste(label[i], L1, sep = " \n")
  p <- ggplot(temp, aes(ymax=ymax, ymin=ymin, xmax=4, xmin=2.5, fill=Sample)) +
    geom_rect(colour="black") +
    scale_colour_manual(values=rep("black", 5)) +
    coord_polar(theta="y") +
    xlim(c(0, 4)) +
    theme_void() + 
    annotate("text", x = 0, y = 0, label = L2, size=6, fontface = "bold")+ 
    theme(legend.text = element_text(size=10),
          legend.key.size = unit(0.4, "cm"))
  hoge[[i]] <- p
}

p <- plot_grid(hoge[[1]], hoge[[2]], hoge[[3]], hoge[[4]], hoge[[5]], hoge[[6]],
               hoge[[7]], hoge[[8]], hoge[[9]], hoge[[10]], hoge[[11]], hoge[[12]], ncol=4)

ggsave("./SFig6A.pdf", plot=p, width = 20, height = 10)
ggsave("./SFig6A.png", plot=p, width = 20, height = 10)


##
df <- as.data.frame(table(seu$orig.ident, seu$Annotation))
colnames(df) <- c("Sample", "Cluster", "Count")
df$Cluster <- factor(df$Cluster, levels = c("Fibroblast_1", "Fibroblast_2", "Macrophage", "T+NK",
                                            "Fibroblast_3", "Plasmocyte", "Endothelial+Basal",
                                            "Proliferating", "Luminal", "Mastocyte", "B", "Neutrophil"))
label <- levels(as.factor(seu$orig.ident))
hoge <- list()

for(i in 1:11){
  temp <- df[df$Sample == label[i],]
  temp$fraction <- temp$Count / sum(temp$Count)
  temp$ymax <- cumsum(temp$fraction)
  temp$ymin <- c(0, head(temp$ymax, n=-1))
  L1 <- paste("n=", sum(temp$Count), sep = "")
  L2 <- paste(label[i], L1, sep = " \n")
  p <- ggplot(temp, aes(ymax=ymax, ymin=ymin, xmax=4, xmin=2.5, fill=Cluster)) +
    geom_rect(colour="black") +
    scale_colour_manual(values=rep("black", 5)) +
    coord_polar(theta="y") +
    xlim(c(0, 4)) +
    theme_void() + 
    annotate("text", x = 0, y = 0, label = L2, size=6, fontface = "bold")+ 
    theme(legend.text = element_text(size=10),
          legend.key.size = unit(0.4, "cm"))
  hoge[[i]] <- p
}

p <- plot_grid(hoge[[1]], hoge[[2]], hoge[[3]], hoge[[4]], hoge[[5]], hoge[[6]],
               hoge[[7]], hoge[[8]], hoge[[9]], hoge[[10]], hoge[[11]], ncol=4)

ggsave("./SFig6B.pdf", plot=p, width = 20, height = 10)
ggsave("./SFig6B.png", plot=p, width = 20, height = 10)


###
Idents(seu) <- "orig.ident"
Bulk <- AverageExpression(seu, assays = "RNA", slot = "scale.data")
Bulk <- Bulk[[1]]
Bulk <- t(Bulk)

res.pca <- PCA(Bulk, graph = FALSE)
Hor <- as.factor(c("None", "E2", "E2+ICI", "E2+P4", "None", "E2", "E2+P4", "None", "E2", "E2+P4", "None"))
Hor <- as.data.frame(Hor)

p <- fviz_pca_ind(res.pca, repel = TRUE, habillage = Hor$Hor,
                  addEllipses = F, 
                  legend.title = "Sample",
                  alpha.ind=0.7,
                  invisible=c("ind.sup", "quali"),
                  pointsize = 6,
                  labelsize = 6)+
  labs(title ="Pseudo RNAseq", x = "PC1", y = "PC2")+
  theme(axis.text=element_text(size=10),
        axis.title=element_text(size=12,face="bold"),
        plot.title = element_text(size=18, face="bold"))+
  NoLegend()

ggsave("./SFig6C-1.pdf", plot=p, width = 5, height = 5)
ggsave("./SFig6C-1.png", plot=p, width = 5, height = 5)


#
row.names(Bulk)
PBDE <- as.factor(c("None", "None", "None", "None", "Low_PBDE", "Low_PBDE", "Low_PBDE", "High_PBDE", "High_PBDE", "High_PBDE", "None"))
PBDE <- as.data.frame(PBDE)

p <- fviz_pca_ind(res.pca, repel = TRUE, habillage = PBDE$PBDE,
                  addEllipses = F, 
                  legend.title = "Sample",
                  alpha.ind=0.7,
                  invisible=c("ind.sup", "quali"),
                  pointsize = 6,
                  labelsize = 6)+
  labs(title ="Pseudo RNAseq (colored by PBDE)", x = "PC1", y = "PC2")+
  theme(axis.text=element_text(size=10),
        axis.title=element_text(size=12,face="bold"),
        plot.title = element_text(size=18, face="bold"))+
  NoLegend()

ggsave("./SFig6C-2.pdf", plot=p, width = 5, height = 5)
ggsave("./SFig6C-2.png", plot=p, width = 5, height = 5)


fviz_pca_ind(res.pca, repel = TRUE, 
             axes = c(3,4),
             habillage = label$group,
             addEllipses = F, 
             legend.title = "Sample",
             alpha.ind=0.7,
             invisible=c("ind.sup", "quali"),
             pointsize = 6,
             labelsize = 6)+
  labs(title ="Pseudo RNAseq (colored by PBDE)")+
  theme(axis.text=element_text(size=10),
        axis.title=element_text(size=12,face="bold"),
        plot.title = element_text(size=18, face="bold"))+
  NoLegend()


###
dd <- dist(Bulk)
hc <- hclust(dd, method="ward.D2")
tree <- as.treedata(as.phylo(hc))

anno <- data.frame(sample=tree@phylo$tip.label, PBDE=PBDE, Hor=Hor)

p <- ggtree(tree, alpha=0.8)
p <- p %<+% anno+geom_tiplab(aes(color=Hor), size=6, offset=0.2)+ 
  geom_tippoint(aes(shape=PBDE),size=6, alpha=0.8)+ 
  xlim(NA,7)

ggsave("./Fig5C.pdf", plot=p, width = 7, height = 4)
ggsave("./Fig5C.png", plot=p, width = 7, height = 4)



########################################################################
###### Analysis2  ######################################################
########################################################################
seu$Annotation2 <- seu$Annotation
seu$Annotation2[seu$Annotation2 == "Fibroblast_1"] <- "Fibroblast"
seu$Annotation2[seu$Annotation2 == "Fibroblast_2"] <- "Fibroblast"
seu$Annotation2[seu$Annotation2 == "Fibroblast_3"] <- "Fibroblast"
Idents(seu) <- "Annotation2"

Cluster <- levels(seu)
LP_UP <- list()
LP_DOWN <- list()
HP_UP <- list()
HP_DOWN <- list()

for(i in 1:10){
  Temp <- subset(seu, idents = Cluster[i])
  Idents(Temp) <- "orig.ident"
  ### Low_PBDE
  VCD <- FindMarkers(Temp, ident.1 = "VCD+LP", ident.2 = "VCD",  logfc.threshold = 0.25)
  VCD <- VCD[abs(VCD$avg_log2FC) > 0.25 & VCD$p_val < 0.001,]
  VCD_UP <- VCD[VCD$avg_log2FC >0, ]
  VCD_DOWN <- VCD[VCD$avg_log2FC <0, ]
  
  VCD_E2 <- FindMarkers(Temp, ident.1 = "VCD+E2+LP", ident.2 = "VCD+E2", logfc.threshold = 0.25)
  VCD_E2 <- VCD_E2[abs(VCD_E2$avg_log2FC) > 0.25 & VCD_E2$p_val < 0.001,]
  VCD_E2_UP <- VCD_E2[VCD_E2$avg_log2FC >0, ]
  VCD_E2_DOWN <- VCD_E2[VCD_E2$avg_log2FC <0, ]
  
  VCD_E2_P4 <- FindMarkers(Temp, ident.1 = "VCD+E2+P4+LP", ident.2 = "VCD+E2+P4", logfc.threshold = 0.25)
  VCD_E2_P4 <- VCD_E2_P4[abs(VCD_E2_P4$avg_log2FC) > 0.25 & VCD_E2_P4$p_val < 0.001 ,]
  VCD_E2_P4_UP <- VCD_E2_P4[VCD_E2_P4$avg_log2FC >0, ]
  VCD_E2_P4_DOWN <- VCD_E2_P4[VCD_E2_P4$avg_log2FC <0, ]
  
  UP_gene <- intersect(rownames(VCD_UP), intersect(row.names(VCD_E2_UP), row.names(VCD_E2_P4_UP)))  
  DOWN_gene <- intersect(rownames(VCD_DOWN), intersect(row.names(VCD_E2_DOWN), row.names(VCD_E2_P4_DOWN)))
  
  LP_UP <- c(LP_UP, list(UP_gene))
  LP_DOWN <- c(LP_DOWN, list(DOWN_gene))
  
  names(LP_UP)[i] <- Cluster[i]
  names(LP_DOWN)[i] <- Cluster[i]
  
  ### High_PBDE
  VCD <- FindMarkers(Temp, ident.1 = "VCD+HP", ident.2 = "VCD",  logfc.threshold = 0.25)
  VCD <- VCD[abs(VCD$avg_log2FC) > 0.25 & VCD$p_val < 0.001,]
  VCD_UP <- VCD[VCD$avg_log2FC >0, ]
  VCD_DOWN <- VCD[VCD$avg_log2FC <0, ]
  
  VCD_E2 <- FindMarkers(Temp, ident.1 = "VCD+E2+HP", ident.2 = "VCD+E2", logfc.threshold = 0.25)
  VCD_E2 <- VCD_E2[abs(VCD_E2$avg_log2FC) > 0.25 & VCD_E2$p_val < 0.001,]
  VCD_E2_UP <- VCD_E2[VCD_E2$avg_log2FC >0, ]
  VCD_E2_DOWN <- VCD_E2[VCD_E2$avg_log2FC <0, ]
  
  VCD_E2_P4 <- FindMarkers(Temp, ident.1 = "VCD+E2+P4+HP", ident.2 = "VCD+E2+P4", logfc.threshold = 0.25)
  VCD_E2_P4 <- VCD_E2_P4[abs(VCD_E2_P4$avg_log2FC) > 0.25 & VCD_E2_P4$p_val < 0.001,]
  VCD_E2_P4_UP <- VCD_E2_P4[VCD_E2_P4$avg_log2FC >0, ]
  VCD_E2_P4_DOWN <- VCD_E2_P4[VCD_E2_P4$avg_log2FC <0, ]
  
  UP_gene <- intersect(rownames(VCD_UP), intersect(row.names(VCD_E2_UP), row.names(VCD_E2_P4_UP)))  
  DOWN_gene <- intersect(rownames(VCD_DOWN), intersect(row.names(VCD_E2_DOWN), row.names(VCD_E2_P4_DOWN)))
  
  HP_UP <- c(HP_UP, list(UP_gene))
  HP_DOWN <- c(HP_DOWN, list(DOWN_gene))
  
  names(HP_UP)[i] <- Cluster[i]
  names(HP_DOWN)[i] <- Cluster[i]
}

DEG <- c(list(LP_UP), list(LP_DOWN), list(HP_UP), list(HP_DOWN))
names(DEG) <- c("LP_UP", "LP_DOWN", "HP_UP", "HP_DOWN")
saveRDS(DEG, "./DEG.rds")

DEG <- readRDS("./DEG.rds")
lp_up_gene <- unique(unlist(DEG$LP_UP))
lp_down_gene <- unique(unlist(DEG$LP_DOWN))
hp_up_gene <- unique(unlist(DEG$HP_UP))
hp_down_gene <- unique(unlist(DEG$HP_DOWN))

PBDE_up <- intersect(lp_up_gene, hp_up_gene)
PBDE_down <- intersect(lp_down_gene, hp_down_gene)

###
Idents(seu) <- "orig.ident"
levels(seu)
seu <- subset(seu, idents = c("VCD", "VCD+E2", "VCD+E2+P4", "VCD+LP", 
                              "VCD+E2+LP", "VCD+E2+P4+LP", "VCD+HP", "VCD+E2+HP", "VCD+E2+P4+HP"))
seu$orig.ident <- factor(seu$orig.ident, levels=c("VCD", "VCD+E2", "VCD+E2+P4", "VCD+LP", 
                                                  "VCD+E2+LP", "VCD+E2+P4+LP", "VCD+HP", "VCD+E2+HP", "VCD+E2+P4+HP"))
seu$PBDE <- "None"
seu$PBDE[seu$orig.ident %in% c("VCD+LP", "VCD+E2+LP", "VCD+E2+P4+LP")] <- "Low"
seu$PBDE[seu$orig.ident %in% c("VCD+HP", "VCD+E2+HP", "VCD+E2+P4+HP")] <- "High"
seu$PBDE <- factor(seu$PBDE, levels=c("None", "Low", "High"))

###
Bulk <- AverageExpression(seu, return.seurat = T)
colnames(Bulk)
Bulk$PBDE <- c(rep("None",3), rep("Low", 3), rep("High", 3))
Bulk$PBDE <- factor(Bulk$PBDE, levels=c("None", "Low", "High"))


###
Idents(seu) <- "Annotation2"

###
p <- VlnPlot(seu, "Errfi1",group.by = "PBDE", idents = "Fibroblast", pt.size=0)+ 
  geom_boxplot(width=0.15, fill="white")+
  NoLegend()+
  ggtitle("Errfi1 (Fibroblast)")+
  xlab("PBDE treatment")+ 
  theme(plot.title = element_text(size=18, face="bold"),
        axis.text=element_text(size=14),
        axis.title=element_text(size=16,face="bold"))
ggsave("./Fig5D_1.pdf", plot=p, width = 4, height = 4)
ggsave("./Fig5D_1.png", plot=p, width = 4, height = 4)

p <- FeaturePlot(seu, "Errfi1", cols = FP, 
                 pt.size = 1, min.cutoff = "q10", max.cutoff = "q90", split.by = "PBDE", ncol = 3)
ggsave("./Fig5D_2.pdf", plot=p, width = 12, height = 4)
ggsave("./Fig5D_2.png", plot=p, width = 12, height = 4)

###
p <- VlnPlot(seu, "AY036118",group.by = "PBDE", pt.size=0)+ 
  geom_boxplot(width=0.15, fill="white")+
  NoLegend()+
  xlab("PBDE treatment")+ 
  theme(plot.title = element_text(size=18, face="bold"),
        axis.text=element_text(size=14),
        axis.title=element_text(size=16,face="bold"))
ggsave("./Fig5E_1.pdf", plot=p, width = 4, height = 4)
ggsave("./Fig5E_1.png", plot=p, width = 4, height = 4)

p <- FeaturePlot(seu, "AY036118", cols = FP, 
                 pt.size = 1, min.cutoff = "q10", max.cutoff = "q90", split.by = "PBDE", ncol = 3)
ggsave("./Fig5E_2.pdf", plot=p, width = 12, height = 4)
ggsave("./Fig5E_2.png", plot=p, width = 12, height = 4)


###
up_genes <- sort(unique(c(unlist(DEG$LP_UP), unlist(DEG$HP_UP))))
down_genes <- sort(unique(c(unlist(DEG$LP_DOWN), unlist(DEG$HP_DOWN))))

hoge <- list()

for(i in 1:length(up_genes)){
  p <- VlnPlot(seu, up_genes[i], group.by = "PBDE", pt.size=0)+ 
    geom_boxplot(width=0.15, fill="white")+
    NoLegend()+
    xlab("PBDE treatment")+ 
    theme(plot.title = element_text(size=18, face="bold"),
          axis.text=element_text(size=14),
          axis.title=element_text(size=16,face="bold"))
  hoge <- c(hoge, list(p))
}

q <- plot_grid(hoge[[1]], hoge[[2]], hoge[[3]], hoge[[4]], hoge[[5]], hoge[[6]], hoge[[7]], hoge[[8]],
               hoge[[9]], hoge[[10]], hoge[[11]], hoge[[12]], hoge[[13]], hoge[[14]], hoge[[15]], hoge[[16]],
               hoge[[16]], hoge[[17]], hoge[[18]], hoge[[18]], hoge[[19]], hoge[[20]], hoge[[21]],
               hoge[[22]], hoge[[23]], hoge[[24]], hoge[[25]], hoge[[26]], hoge[[27]], hoge[[28]], hoge[[29]],
               hoge[[30]], hoge[[31]], hoge[[32]], hoge[[33]], hoge[[34]], hoge[[35]], hoge[[36]], hoge[[37]],
               hoge[[38]], hoge[[39]], hoge[[40]], hoge[[41]], hoge[[42]], ncol=6)
ggsave("./SFig7.pdf", plot=q, width = 18, height = 28)
ggsave("./SFig7.png", plot=q, width = 18, height = 28)


hoge <- list()
for(i in 1:length(down_genes)){
  p <- VlnPlot(seu, down_genes[i], group.by = "PBDE", pt.size=0)+ 
    geom_boxplot(width=0.15, fill="white")+
    NoLegend()+
    xlab("PBDE treatment")+ 
    theme(plot.title = element_text(size=18, face="bold"),
          axis.text=element_text(size=14),
          axis.title=element_text(size=16,face="bold"))
  hoge <- c(hoge, list(p))
}

q <- plot_grid(hoge[[1]], hoge[[2]], hoge[[3]], hoge[[4]], hoge[[5]], hoge[[6]], hoge[[7]], hoge[[8]],
               hoge[[9]], hoge[[10]], hoge[[11]], hoge[[12]], hoge[[13]], hoge[[14]], hoge[[15]], hoge[[16]],
               hoge[[16]], hoge[[17]], hoge[[18]], hoge[[18]], hoge[[19]], hoge[[20]], hoge[[21]],
               hoge[[22]], hoge[[23]], hoge[[24]], hoge[[25]], hoge[[26]], hoge[[27]], hoge[[28]], hoge[[29]],
               hoge[[30]], hoge[[31]], hoge[[32]], hoge[[33]], hoge[[34]], hoge[[35]], hoge[[36]], hoge[[37]],
               hoge[[38]], hoge[[39]], hoge[[40]], hoge[[41]], hoge[[42]], hoge[[43]], hoge[[44]], hoge[[45]], ncol=7)
ggsave("./SFig8.pdf", plot=q, width = 21, height = 28)
ggsave("./SFig8.png", plot=q, width = 21, height = 28)

