// __BEGIN_LICENSE__
//  Copyright (c) 2009-2013, United States Government as represented by the
//  Administrator of the National Aeronautics and Space Administration. All
//  rights reserved.
//
//  The NGT platform is licensed under the Apache License, Version 2.0 (the
//  "License"); you may not use this file except in compliance with the
//  License. You may obtain a copy of the License at
//  http://www.apache.org/licenses/LICENSE-2.0
//
//  Unless required by applicable law or agreed to in writing, software
//  distributed under the License is distributed on an "AS IS" BASIS,
//  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//  See the License for the specific language governing permissions and
//  limitations under the License.
// __END_LICENSE__

/// \file StereoSessionFactory.cc

// This include must exist for linking purposes
#include <asp/Sessions/StereoSessionFactory.h>
#include <asp/Sessions/StereoSessionMapProj.h>
#include <asp/Sessions/StereoSessionIsis.h>
#include <asp/Sessions/StereoSessionNadirPinhole.h>
#include <asp/Sessions/StereoSessionPinhole.h>
#include <asp/Sessions/StereoSessionRPC.h>
#include <asp/Sessions/StereoSessionASTER.h>

#include <asp/IsisIO/IsisInterface.h>
#include <vw/FileIO/DiskImageResourceRaw.h>
#include <vw/Camera/CameraUtilities.h>
#include <asp/Camera/SPOT_XML.h>
#include <asp/Camera/ASTER_XML.h>
#include <vw/Camera/OpticalBarModel.h>

namespace asp {

  StereoSession* StereoSessionFactory::create(std::string      & session_type, // in-out
                                              vw::GdalWriteOptions const& options,
                                              std::string const& left_image_file,
                                              std::string const& right_image_file,
                                              std::string const& left_camera_file,
                                              std::string const& right_camera_file,
                                              std::string const& out_prefix,
                                              std::string const& input_dem,
                                              bool allow_map_promote, 
                                              bool total_quiet) {

    // Known user session types are:
    // DG, RPC, ISIS, Pinhole, NadirPinhole, OpticalBar, etc.
    //
    // Hidden sessions are:
    // DGMapRPC, Blank (Guessing)
    
    // Try to guess the session if not provided
    std::string actual_session_type = session_type;
    bool quiet = true;
    boost::to_lower(actual_session_type);
    if (actual_session_type.empty()) {
      if (asp::has_pinhole_extension(left_camera_file) ||
          asp::has_pinhole_extension(right_camera_file)) {
        // There can be several types of .tsai files
        std::string error_pinhole, error_opticalbar;
        try {
          boost::shared_ptr<vw::camera::CameraModel> P = 
            vw::camera::load_pinhole_camera_model(left_camera_file);
          actual_session_type = "pinhole";
        }catch (std::exception & e) {
          error_pinhole = e.what();
          try {
            vw::camera::OpticalBarModel P;
            P.read(left_camera_file);
            actual_session_type = "opticalbar";
          }catch (std::exception & e) {
            error_opticalbar = e.what();
            vw_throw(vw::NoImplErr() << "Could not read the camera model " <<
                     left_camera_file << " as pinhole. "
                     << "The error was: " << error_pinhole << "\n"
                     << "Could not read it as opticalbar model either, the error was: "
                     << error_opticalbar);
          }
        }
      } else if (asp::has_isd_extension(left_camera_file) ||
                 asp::has_isd_extension(right_camera_file)) {
        actual_session_type = "csm";
      } else if (boost::iends_with(boost::to_lower_copy(left_image_file), ".cub") &&
                 asp::isis::IsisCubeHasCsmBlob(left_image_file)) {
        // This is a cub file that has a CSM model inside of of it
        if (!asp::isis::IsisCubeHasCsmBlob(right_image_file))
          vw::vw_throw(vw::ArgumentErr() << "Found a CSM model in " << left_image_file
                   << " but not in " << right_image_file << ".\n");
        actual_session_type = "csm";
      } else if (boost::iends_with(boost::to_lower_copy(left_image_file  ), ".cub") ||
                 boost::iends_with(boost::to_lower_copy(right_image_file ), ".cub") ||
                 boost::iends_with(boost::to_lower_copy(left_camera_file ), ".cub") ||
                 boost::iends_with(boost::to_lower_copy(right_camera_file), ".cub")) {
        actual_session_type = "isis";
      } else if (boost::iends_with(boost::to_lower_copy(left_camera_file ), ".dim") ||
                 boost::iends_with(boost::to_lower_copy(right_camera_file), ".dim")) {
        actual_session_type = "spot5";
      } else if (boost::iends_with(boost::to_lower_copy(left_camera_file ), ".xml") ||
                 boost::iends_with(boost::to_lower_copy(right_camera_file), ".xml")) {

        // Here we have several options for .xml files. Note that a
        // Digital Globe xml file has both linescan and RPC
        // models. The logic below favors the linescan sensor.
        if (actual_session_type.empty()) {
          
          // TODO(oalexan1): Try to peek in the xml file instead of doing this exhaustive
          // checking.
          
          // Try DG exact linescan model
          try {
            StereoSessionDG session;
            boost::shared_ptr<vw::camera::CameraModel>
              left_model  = session.camera_model(left_image_file,  left_camera_file, quiet),
              right_model = session.camera_model(right_image_file, right_camera_file, quiet);
            actual_session_type = "dg";
          } catch (...) {}
        }
        
        if (actual_session_type.empty()) {
          // Try PeruSat exact linescan model
          try {
            StereoSessionPeruSat session;
            boost::shared_ptr<vw::camera::CameraModel>
              left_model  = session.camera_model(left_image_file,  left_camera_file, quiet),
              right_model = session.camera_model(right_image_file, right_camera_file, quiet);
            actual_session_type = "perusat";
          } catch (...) {}
        }

        if (actual_session_type.empty()) {
          // Try Pleiades exact linescan model
          try {
            StereoSessionPleiades session;
            boost::shared_ptr<vw::camera::CameraModel>
              left_model  = session.camera_model(left_image_file,  left_camera_file, quiet),
              right_model = session.camera_model(right_image_file, right_camera_file, quiet);
            actual_session_type = "pleiades";
          } catch (...) {}
        }
        
        if (actual_session_type.empty()) {
          // Try the ASTER exact linescan model
          try {
            StereoSessionASTER session;
            boost::shared_ptr<vw::camera::CameraModel>
              left_model  = session.camera_model(left_image_file,  left_camera_file, quiet),
              right_model = session.camera_model(right_image_file, right_camera_file, quiet);
            actual_session_type = "aster";
          } catch (...) {}
        }
        
      } // end considering the xml extension case

      // Try RPC, which can either have xml cameras or no cameras at all (if embedded
      // in the tif files).
      if (actual_session_type.empty()) {
        try {
          StereoSessionRPC session;
          boost::shared_ptr<vw::camera::CameraModel>
            left_model  = session.camera_model(left_image_file,  left_camera_file, quiet),
            right_model = session.camera_model(right_image_file, right_camera_file, quiet);
          actual_session_type = "rpc";
        } catch (...) {}
      }
    }
    
    if (allow_map_promote) {
      if (!input_dem.empty() && actual_session_type == "dg") {
        // User says DG but also gives a DEM.
        actual_session_type = "dgmaprpc";
        VW_OUT(vw::DebugMessage,"asp") << "Changing session type to: dgmaprpc.\n";
      }
      if (!input_dem.empty() && actual_session_type == "rpc") {
        // User says RPC but also gives a DEM.
        actual_session_type = "rpcmaprpc";
        VW_OUT(vw::DebugMessage,"asp") << "Changing session type to: rpcmaprpc.\n";
      }
      if (!input_dem.empty() && actual_session_type == "pinhole") {
        // User says PINHOLE but also gives a DEM.
        actual_session_type = "pinholemappinhole";
        VW_OUT(vw::DebugMessage,"asp") << "Changing session type to: pinholemappinhole.\n";
      }
      if (!input_dem.empty() && actual_session_type == "opticalbar") {
        // User says OPTICAL BAR but also gives a DEM.
        actual_session_type = "opticalbarmapopticalbar";
        VW_OUT(vw::DebugMessage,"asp") << "Changing session type to: opticalbarmapopticalbar.\n";
      }
      if (!input_dem.empty() && actual_session_type == "csm") {
        // User says CSM but also gives a DEM.
        // Mapprojection can happen either with csm or RPC cameras (the latter for DG)
        std::string cam_tag = "CAMERA_MODEL_TYPE";
        std::string l_cam_type = vw::cartography::read_header_string(left_image_file, cam_tag);
        if (l_cam_type == "rpc")
          actual_session_type = "csmmaprpc";
        else
          actual_session_type = "csmmapcsm"; // used also when l_cam_type is empty
        VW_OUT(vw::DebugMessage,"asp") << "Changing session type to: " 
          << actual_session_type << ".\n";
      }
      if (!input_dem.empty() && actual_session_type == "isis") {
        // User says ISIS but also gives a DEM.
        actual_session_type = "isismapisis";
        VW_OUT(vw::DebugMessage,"asp") << "Changing session type to: isismapisis.\n";
      }
      if (!input_dem.empty() && actual_session_type == "spot5") {
        // User says SPOT5 but also gives a DEM.
        actual_session_type = "spot5maprpc";
        VW_OUT(vw::DebugMessage,"asp") << "Changing session type to: spot5maprpc.\n";
      }
      if (!input_dem.empty() && actual_session_type == "aster") {
        // User says ASTER but also gives a DEM.
        // Mapprojection can happen either with ASTER or RPC cameras 
        std::string cam_tag = "CAMERA_MODEL_TYPE";
        std::string l_cam_type 
          = vw::cartography::read_header_string(left_image_file, cam_tag);
        if (l_cam_type == "aster")
          actual_session_type = "astermapaster";
        else
          actual_session_type = "astermaprpc"; // used also when l_cam_type is empty
        VW_OUT(vw::DebugMessage,"asp") << "Changing session type to: " 
          << actual_session_type << ".\n";
      }
      if (!input_dem.empty() && actual_session_type == "pleiades") {
        // User says Pleiades but also gives a DEM.
        actual_session_type = "pleiadesmappleiades";
        VW_OUT(vw::DebugMessage,"asp") << "Changing session type to: pleiadesmappleiades.\n";
      }
      
      // Quietly switch from nadirpinhole to pinhole for mapprojected images
      if (!input_dem.empty() && actual_session_type == "nadirpinhole") {
        // User says nadirpinhole but also gives a DEM.
        actual_session_type = "pinholemappinhole";
        VW_OUT(vw::DebugMessage,"asp") << "Changing session type to: pinhole.\n";
      }
      
    } // End map promotion section

    if (!input_dem.empty() && actual_session_type == "perusat") {
      // User says PeruSat-1 or Pleiades but also gives a DEM, so the images were mapprojected.
      // If the mapprojection was done with the exact model, stereo becomes
      // painfully slow. If it was done with the RPC model, things become hard
      // to manage, and stereo needs to know both the exact and RPC model
      // and those are in different files. Hence, just don't allow mapprojected
      // images in this case.
      vw_throw(vw::NoImplErr() << "Stereo with mapprojected images and the "
               << "PeruSat-1 linescan model is not implemented. "
               << "Use instead the RPC model.");
    }
    
    // We should know the session type by now.
    VW_ASSERT(!actual_session_type.empty(),
              vw::ArgumentErr() << "Could not determine stereo session type. "
              << "Please set it explicitly using the -t switch.\n"
              << "Options include: [nadirpinhole pinhole isis dg rpc spot5 aster perusat pleiades opticalbar csm pinholemappinhole isismapisis dgmaprpc rpcmaprpc spot5maprpc astermapaster astermaprpc opticalbarmapopticalbar csmmapcsm csmmaprpc pleiadesmappleiades].\n");
    
    if (!total_quiet)
      vw::vw_out() << "Using session: " << actual_session_type << "\n";

    // Compare the current session name to all recognized types
    // - Only one of these will ever get triggered
    StereoSession* session = NULL;
    if (actual_session_type == "dg")
      session = StereoSessionDG::construct();
    else if (actual_session_type == "dgmaprpc")
        session = StereoSessionDGMapRPC::construct();
    else if (actual_session_type == "nadirpinhole")
      session = StereoSessionNadirPinhole::construct();
    else if (actual_session_type == "pinhole")
      session = StereoSessionPinhole::construct();
    else if (actual_session_type == "rpc")
      session = StereoSessionRPC::construct();
    else if (actual_session_type == "rpcmaprpc")
      session = StereoSessionRPCMapRPC::construct();
    else if (actual_session_type == "pinholemappinhole")
      session = StereoSessionPinholeMapPinhole::construct();
    else if (actual_session_type == "opticalbarmapopticalbar")
      session = StereoSessionBarMapBar::construct();
    else if (actual_session_type == "spot5maprpc")
        session = StereoSessionSpot5MapRPC::construct();
    else if (actual_session_type == "astermapaster")
        session = StereoSessionASTERMapASTER::construct();
    else if (actual_session_type == "astermaprpc")
        session = StereoSessionASTERMapRPC::construct();
    else if (actual_session_type == "pleiadesmappleiades")
        session = StereoSessionPleiadesMapPleiades::construct();
#if defined(ASP_HAVE_PKG_ISISIO) && ASP_HAVE_PKG_ISISIO == 1
    else if (actual_session_type == "isis")
      session = StereoSessionIsis::construct();
    else if (actual_session_type == "isismapisis")
      session = StereoSessionIsisMapIsis::construct();
#endif
    else if (actual_session_type == "spot5")
      session = StereoSessionSpot::construct();
    else if (actual_session_type == "perusat")
      session = StereoSessionPeruSat::construct();
    else if (actual_session_type == "pleiades")
      session = StereoSessionPleiades::construct();
    else if (actual_session_type == "aster")
      session = StereoSessionASTER::construct();
    else if (actual_session_type == "opticalbar")
      session = StereoSessionOpticalBar::construct();
    else if (actual_session_type == "csm")
      session = StereoSessionCsm::construct();
    else if (actual_session_type == "csmmapcsm")
      session = StereoSessionCsmMapCsm::construct();
    else if (actual_session_type == "csmmaprpc")
      session = StereoSessionCsmMapRpc::construct();
    if (session == 0)
      vw_throw(vw::NoImplErr() << "Unsupported stereo session type: "
               << actual_session_type);

    session->initialize(options,         // Initialize the new object
                        left_image_file,  right_image_file,
                        left_camera_file, right_camera_file,
                        out_prefix, input_dem);
    session_type = session->name(); // update the session name 

 // This is important so that the user does not load RPC cameras while thinking
 // ASTER or DG models are loaded. 
 if (session_type.find("rpc") != std::string::npos && 
     asp::stereo_settings().aster_use_csm)
      vw_throw(vw::ArgumentErr() 
               << "The --aster-use-csm option must be used only with the "
               << "ASTER session (-t aster).\n");
    
    return session;
} // End function create()

  
} // end namespace asp
