/** Author: G.P. Leemrijse
	 13-08-2023
	 TU Eindhoven
*/

module ADL[hwEvent, hwAddress, hwThread]
open util

/** Signatures and helper functions */

sig Location {
	loc_map : one hwAddress
}

some sig Instance {
	start : one Command,
	inst_map : one hwThread
}

abstract sig Command {
	nxt : lone Command,
	loc : one Location,
	cmd_map : one hwEvent
}

sig ReadCommand extends Command {
	dep : set Command
}

sig WriteCommand extends Command {
	rf : set ReadCommand,
	co : set WriteCommand
}

fun inst : Command->Instance {
	~(start.*nxt)
}

fun po : Command->Command {
	^nxt
}

fun po_loc : Command->Command {
	po & same_loc
}

fun rfi : WriteCommand->ReadCommand {
	rf & same_instance
}

fun fr : ReadCommand->WriteCommand {
	~rfi.^co
	+
	fr_init
}

fun fr_init : ReadCommand->WriteCommand {
	ident[ReadCommand - WriteCommand.rf].((same_instance & same_loc) :> WriteCommand)
}

fun same_instance : Command->Command {
	inst.~inst
}

fun same_loc : Command->Command {
	loc.~loc
}

fun other_loc : Command->Command {
	Command->Command - same_loc
}

/** Pre-execution correctness predicates */

pred at_most_one_incoming_cmd {
	all c : Command | lone c.~nxt
}

pred at_most_one_incoming_start {
	all c : Command | lone c.~start
}

pred only_start_commands_have_no_predecessor {
	Instance.start = (Command - Command.nxt)
}

pred use_all_locations {
	Command.loc = Location
}

pred dep_in_po {
	dep in po
}

pred wellformed_adl_preex {
	at_most_one_incoming_cmd
	at_most_one_incoming_start
	only_start_commands_have_no_predecessor
	acyclic[nxt]
	use_all_locations
	dep_in_po
}

/** Execution witness correctness predicates */

pred rf_and_co_of_same_location {
	rf + co in same_loc
}

/* As we only have local coherency, only writes of the
   same instance are ordered by co.
*/
pred co_of_same_instance {
	co in same_instance
}

pred reads_read_from_at_most_one_write {
	all r : ReadCommand | lone r.~rf
}

/* Per location and instance, co is total. */
pred co_total_per_loc_inst {
	all l : Location, i : Instance |
		// Intersection of all commands with l and i as location and instance, with writes
		total[co, l.~loc & i.~inst & WriteCommand]
}

/* CONSISTENCY Axiom: Semantic ordering is consistent */
pred CONSISTENCY {
	acyclic[po_loc + co + rf + fr + dep]
}

/* CONSISTENCY* Axiom: Semantic ordering is consistent, without dep.*/
pred CONSISTENCY_STAR {
	acyclic[po_loc + co + rf + fr]
}

/* NO_THIN_AIR Axiom: No reads from dependants. */
pred NO_THIN_AIR {
	acyclic[rf + dep]
}

pred legal_adl_execution {
	rf_and_co_of_same_location
	co_of_same_instance
	reads_read_from_at_most_one_write
	co_total_per_loc_inst
	CONSISTENCY
	
	//CONSISTENCY_STAR
	//NO_THIN_AIR
	
}

/* Shows only pre-execution */
run p_correct {
	wellformed_adl_preex
	no (rf + co)
} for 8

