// This file is part of BenchExec, a framework for reliable benchmarking:
// https://github.com/sosy-lab/benchexec
//
// SPDX-FileCopyrightText: 2019-2020 Dirk Beyer <https://www.sosy-lab.org>
//
// SPDX-License-Identifier: Apache-2.0

import React from "react";
import StatisticsTable from "./StatisticsTable";

const infos = [
  "displayName",
  "tool",
  "limit",
  "host",
  "os",
  "system",
  "date",
  "runset",
  "branch",
  "options",
  "property",
];

const Summary = (props) => {
  /* ++++++++++++++ Helper functions ++++++++++++++ */

  const renderOptions = (text) => {
    return text.split(/[\s]+-/).map((option, i) => (
      <li key={option}>
        <code>{i === 0 ? option : `-${option}`}</code>
      </li>
    ));
  };

  const externalLink = (url, text) => {
    if (url) {
      return (
        <a href={url} target="_blank" rel="noopener noreferrer">
          {text}
        </a>
      );
    } else {
      return text;
    }
  };

  const renderToolNameAndVersion = ({
    tool,
    version,
    project_url,
    version_url,
  }) => {
    return (
      <>
        {externalLink(project_url, tool)} {externalLink(version_url, version)}
      </>
    );
  };

  /* ++++++++++++++ Table render functions ++++++++++++++ */

  const renderRow = (row, text, colSpan, j) => {
    const isOptionRow = row === "options";
    const isToolRow = row === "tool";
    return (
      <td
        colSpan={colSpan}
        key={text + j}
        className={`header__tool-row${isOptionRow && " options"}`}
      >
        {isOptionRow ? (
          <ul>{renderOptions(text)}</ul>
        ) : isToolRow ? (
          renderToolNameAndVersion(text)
        ) : (
          text
        )}
      </td>
    );
  };

  return (
    <div id="summary">
      <div id="benchmark_setup">
        <h2>Benchmark Setup</h2>
        <table>
          <tbody>
            {infos
              .map((row) => props.tableHeader[row])
              .filter((row) => row !== null)
              .map((row) => (
                <tr key={"tr-" + row.id} className={row.id}>
                  <th key={"td-" + row.id}>{row.name}</th>
                  {row.content.map((tool, j) =>
                    renderRow(row.id, tool[0], tool[1], j),
                  )}
                </tr>
              ))}
          </tbody>
        </table>
      </div>
      <StatisticsTable
        selectColumn={props.selectColumn}
        tools={props.tools}
        switchToQuantile={props.switchToQuantile}
        hiddenCols={props.hiddenCols}
        tableData={props.tableData}
        onStatsReady={props.onStatsReady}
        stats={props.stats}
        filtered={props.filtered}
      />
      <p>
        Generated by{" "}
        <a
          className="link"
          href="https://github.com/sosy-lab/benchexec"
          target="_blank"
          rel="noopener noreferrer"
        >
          {" "}
          BenchExec {props.version}
        </a>
      </p>
    </div>
  );
};

export default Summary;
