/*
 * Copyright (c) 2012. The Genome Analysis Centre, Norwich, UK
 * MISO project contacts: Robert Davey @ TGAC
 * *********************************************************************
 *
 * This file is part of MISO.
 *
 * MISO is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * MISO is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with MISO. If not, see <http://www.gnu.org/licenses/>.
 *
 * *********************************************************************
 */

package uk.ac.bbsrc.tgac.miso.spring.ajax;

import static uk.ac.bbsrc.tgac.miso.core.util.LimsUtils.isStringEmptyOrNull;

import java.io.IOException;
import java.util.Date;

import javax.servlet.http.HttpSession;

import org.apache.commons.codec.binary.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.core.context.SecurityContextHolder;

import com.eaglegenomics.simlims.core.Note;
import com.eaglegenomics.simlims.core.User;
import com.eaglegenomics.simlims.core.manager.SecurityManager;

import net.sf.json.JSONArray;
import net.sf.json.JSONObject;
import net.sourceforge.fluxion.ajax.Ajaxified;
import net.sourceforge.fluxion.ajax.util.JSONUtils;

import uk.ac.bbsrc.tgac.miso.core.data.Sample;
import uk.ac.bbsrc.tgac.miso.core.data.impl.ProjectOverview;
import uk.ac.bbsrc.tgac.miso.core.service.naming.NamingScheme;
import uk.ac.bbsrc.tgac.miso.core.service.naming.validation.ValidationResult;
import uk.ac.bbsrc.tgac.miso.core.util.TaxonomyUtils;
import uk.ac.bbsrc.tgac.miso.service.ProjectService;
import uk.ac.bbsrc.tgac.miso.service.SampleService;

/**
 * uk.ac.bbsrc.tgac.miso.spring.ajax
 * <p/>
 * Info
 *
 * @author Rob Davey
 * @since 0.0.2
 */
@Ajaxified
public class SampleControllerHelperService {

  protected static final Logger log = LoggerFactory.getLogger(SampleControllerHelperService.class);
  @Autowired
  private SecurityManager securityManager;
  @Autowired
  private ProjectService projectService;
  @Autowired
  private SampleService sampleService;
  @Autowired
  private NamingScheme namingScheme;

  public JSONObject validateSampleAlias(HttpSession session, JSONObject json) {
    try {
      if (isStringEmptyOrNull(json.getString("alias")) && namingScheme.hasSampleAliasGenerator()) {
        // alias will be generated by DAO during save
        return JSONUtils.SimpleJSONResponse("OK");
      } else if (!json.has("alias")) {
        return JSONUtils.SimpleJSONError("No alias specified");
      } else {
        String alias = json.getString("alias");
        ValidationResult aliasValidation = namingScheme.validateSampleAlias(alias);
        if (aliasValidation.isValid()) {
          log.info("Sample alias OK!");
          return JSONUtils.SimpleJSONResponse("OK");
        } else {
          log.error("Sample alias not valid: " + alias);
          return JSONUtils.SimpleJSONError(aliasValidation.getMessage());
        }
      }
    } catch (Exception e) {
      log.error("Exception in validateSampleAlias", e);
      throw e;
    }
  }

  public JSONObject addSampleNote(HttpSession session, JSONObject json) {
    Long sampleId = json.getLong("sampleId");
    String text = json.getString("text");

    try {
      Sample sample = sampleService.get(sampleId);
      Note note = new Note();
      note.setInternalOnly(json.getString("internalOnly").equals("on"));
      note.setText(text);
      sampleService.addNote(sample, note);
    } catch (IOException e) {
      log.error("add sample note", e);
      return JSONUtils.SimpleJSONError(e.getMessage());
    }

    return JSONUtils.SimpleJSONResponse("Note saved successfully");
  }

  public JSONObject deleteSampleNote(HttpSession session, JSONObject json) {
    Long sampleId = json.getLong("sampleId");
    Long noteId = json.getLong("noteId");

    try {
      Sample sample = sampleService.get(sampleId);
      sampleService.deleteNote(sample, noteId);
      return JSONUtils.SimpleJSONResponse("OK");
    } catch (IOException e) {
      log.error("cannot remove note", e);
      return JSONUtils.SimpleJSONError("Cannot remove note: " + e.getMessage());
    }
  }

  public JSONObject getSampleByBarcode(HttpSession session, JSONObject json) {
    JSONObject response = new JSONObject();
    String barcode = json.getString("barcode");

    try {
      Sample sample = sampleService.getByBarcode(barcode);
      // Base64-encoded string, most likely a barcode image beeped in. decode and search
      if (sample == null) {
        sample = sampleService.getByBarcode(new String(Base64.decodeBase64(barcode)));
      }
      if (sample.getReceivedDate() == null) {
        response.put("name", sample.getName());
        response.put("desc", sample.getDescription());
        response.put("id", sample.getId());
        response.put("type", sample.getSampleType());
        response.put("project", sample.getProject().getName());
        return response;
      } else {
        return JSONUtils.SimpleJSONError("Sample " + sample.getName() + " has already been received");
      }
    } catch (Exception e) {
      log.error("sample not in database", e);
      return JSONUtils.SimpleJSONError(e.getMessage() + ": This sample doesn't seem to be in the database.");
    }
  }

  public JSONObject setSampleReceivedDateByBarcode(HttpSession session, JSONObject json) {
    JSONObject response = new JSONObject();
    try {
      User user = securityManager.getUserByLoginName(SecurityContextHolder.getContext().getAuthentication().getName());
      JSONArray sampleIds = JSONArray.fromObject(json.getString("samples"));
      for (int index = 0; index < sampleIds.size(); index++) {
        Sample sample = sampleService.get(sampleIds.getLong(index));
        sample.setReceivedDate(new Date());
        sample.setLastModifier(user);
        sampleService.update(sample);
      }
      response.put("result", "Samples received date saved");
      return response;
    } catch (IOException e) {
      log.error("cannot set receipt date for sample", e);
      return JSONUtils.SimpleJSONError(e.getMessage() + ": Cannot set receipt date for sample");
    }
  }

  public JSONObject lookupNCBIScientificName(HttpSession session, JSONObject json) {
    String taxon = TaxonomyUtils.checkScientificNameAtNCBI(json.getString("scientificName"));
    if (taxon != null) {
      return JSONUtils.SimpleJSONResponse("NCBI taxon is valid");
    } else {
      return JSONUtils.SimpleJSONError(
          "This scientific name is not of a known taxonomy. You may have problems when trying to submit this data to public repositories.");
    }
  }

  public JSONObject removeSampleFromOverview(HttpSession session, JSONObject json) {
    User user;
    try {
      user = securityManager.getUserByLoginName(SecurityContextHolder.getContext().getAuthentication().getName());
    } catch (IOException e) {
      log.error("remove sample from group", e);
      return JSONUtils.SimpleJSONError("Error getting currently logged in user.");
    }

    if (user != null) {
      if (json.has("sampleId") && json.has("overviewId")) {
        Long sampleId = json.getLong("sampleId");
        Long overviewId = json.getLong("overviewId");
        try {
          ProjectOverview overview = projectService.getProjectOverviewById(overviewId);
          Sample s = sampleService.get(sampleId);
          if (overview.getSamples().contains(s)) {
            if (overview.getSamples().remove(s)) {
              projectService.saveProjectOverview(overview);

              return JSONUtils.SimpleJSONResponse("Sample removed from group");
            } else {
              return JSONUtils.SimpleJSONError("Error removing sample from sample group.");
            }
          } else {
            return JSONUtils.SimpleJSONResponse("Sample not in this sample group!");
          }
        } catch (IOException e) {
          log.error("remove sample from group", e);
          return JSONUtils.SimpleJSONError("Cannot remove sample from group: " + e.getMessage());
        }
      } else {
        return JSONUtils.SimpleJSONError("No sample or sample group specified to remove.");
      }
    } else {
      return JSONUtils.SimpleJSONError("Only logged-in users can remove objects.");
    }
  }

  public void setSecurityManager(SecurityManager securityManager) {
    this.securityManager = securityManager;
  }

  public void setProjectService(ProjectService projectService) {
    this.projectService = projectService;
  }

  public void setSampleNamingScheme(NamingScheme namingScheme) {
    this.namingScheme = namingScheme;
  }
}
