#include <ctime>
#include <iomanip>
#include <iostream>
#include <sstream>
#include <string>
#include <vector>
#include <interop/interop.h>
#include <json/json.h>

const char *stupidDateFormats[] = {"%m/%d/%Y %I:%M:%S %p", "%y%m%d"};

/**
 * Compute the length of a cycle range.
 *
 * Frustratingly, sometimes, this length should be calculated on a range, while
 * in other cases, the last cycle should be selected.
 */
int length(const illumina::interop::model::run::cycle_range &range) {

  return (range.last_cycle() >= range.first_cycle())
             ? range.last_cycle() - range.first_cycle() + 1
             : 0;
}

std::string formatDate(std::tm *timeValue) {
  std::stringstream date_buffer;
  date_buffer << std::put_time(timeValue, "%Y-%m-%dT%H:%M:%S");
  return date_buffer.str();
}

std::string
getRunBasesMask(const illumina::interop::model::metrics::run_metrics &run) {
  std::stringstream basemask;
  auto first = true;

  for (const auto &read : run.run_info().reads()) {
    if (first) {
      first = false;
    } else {
      basemask << ",";
    }
    basemask << (read.is_index() ? "I" : "y") << length(read);
  }
  return basemask.str();
}

/**
 * Convert a "candlestick" or "line" point into a JSON object.
 *
 * This logic is taken from the gnuplot rendering code provided by Illumina.
 * Frustratingly, the Illumina interop library stuff all the line data into the
 * candlestick objects for no clear reason.
 */
Json::Value convert_point(
    bool is_candlestick,
    const illumina::interop::model::plot::candle_stick_point &input_point) {
  Json::Value output_point(Json::objectValue);
  output_point["x"] = input_point.x();
  if (is_candlestick) {
    output_point["low"] = input_point.lower();
    output_point["q1"] = input_point.p25();
    output_point["median"] = input_point.p50();
    output_point["q3"] = input_point.p75();
    output_point["high"] = input_point.upper();
  } else {
    output_point["y"] = input_point.y();
  }
  return output_point;
}

/**
 * Convert a "bar" point into a JSON object.
 *
 * This logic is taken from the gnuplot rendering code provided by Illumina.
 */
Json::Value
convert_point(bool is_candlestick,
              const illumina::interop::model::plot::bar_point &input_point) {
  Json::Value output_point(Json::objectValue);
  output_point["x"] = input_point.x();
  output_point["y"] = input_point.y();
  output_point["width"] = input_point.width();
  return output_point;
}

/**
 * For a plot output with points of type T, convert all the series into JSON
 * objects with the appropriate data.
 */
template <typename T>
void render(
    const std::string &type,
    const std::vector<std::pair<
        std::string, illumina::interop::model::plot::plot_data<T>>> &dataset,
    Json::Value &output) {
  if (dataset.size() == 0)
    return;

  /* Create an empty JSON array to hold all the series in this plot. */
  Json::Value series(Json::arrayValue);

  for (const auto &data : dataset) {
    for (const auto &input_series : data.second) {
      /* Illumina uses the candle stick plot for both candlestick and line data,
       * so we have to figure out which we are using. */
      bool is_candlestick =
          input_series.series_type() ==
          illumina::interop::model::plot::series<T>::Candlestick;
      /* Create an empty object for holding this series. */
      Json::Value output_series(Json::objectValue);
      /* Create a name from the name generated by Illumina plot, which comes in
       * different ragged forms. */
      std::stringstream name;
      if (input_series.title().length() == 0) {
        name << data.first;
      } else {
        name << input_series.title() << " (" << data.first << ")";
      }
      output_series["name"] = name.str();
      /* Create an empty array for holding points and fill it with the data.
       * Since C++ does not erase types, it will select the correct
       * convert_points function by type. The ::convert_points insists we want
       * it to use a convert_points function in the root namespace.*/
      Json::Value output_points(Json::arrayValue);
      for (const auto &point : input_series)
        output_points.append(::convert_point(is_candlestick, point));
      output_series["data"] = std::move(output_points);
      series.append(std::move(output_series));
    }
  }
  if (series.size() == 0) {
    return;
  }
  Json::Value result(Json::objectValue);
  result["type"] = type;
  result["series"] = std::move(series);
  output.append(std::move(result));
}

/**
* This function is here to fix some awkward type choices in the library. There
* are types which have valid subtype relationships, but the dispatch we are
* using requires exact matches. This function has the right exact match.
*/
void plot_by_lane_wrapper(
    illumina::interop::model::metrics::run_metrics &run,
    const illumina::interop::constants::metric_type metrics_name,
    const illumina::interop::model::plot::filter_options &options,
    illumina::interop::model::plot::plot_data<
        illumina::interop::model::plot::candle_stick_point> &data,
    bool skip_empty) {
  illumina::interop::logic::plot::plot_by_lane(run, metrics_name, options, data,
                                               skip_empty);
}

/**
 * For a particular Illumina metric, construct a per-cycle candlestick plot.
*
* In an awkward-to-read-way, this takes a template parameter: a function to
* translate the input interop data into a plot object, which we can then
* convert to JSON.
 */
template <void (*X)(illumina::interop::model::metrics::run_metrics &,
                    const illumina::interop::constants::metric_type,
                    const illumina::interop::model::plot::filter_options &,
                    illumina::interop::model::plot::plot_data<
                        illumina::interop::model::plot::candle_stick_point> &,
                    bool)>
void add_plot(const illumina::interop::constants::metric_type metric_name,
              const std::string &type_name, bool include_combined,
              bool include_lanes,
              illumina::interop::model::metrics::run_metrics &run,
              Json::Value &output) {
  /* Create a list of pairs between series name and the data for that series. */
  std::vector<std::pair<
      std::string, illumina::interop::model::plot::plot_data<
                       illumina::interop::model::plot::candle_stick_point>>>
      dataset;
  try {
    /* By convention, Illumina numbers lanes 1 to n and includes a zeroth lane
     * with the combined data. We iterate over the right requested subset of
     * lanes. */
    for (auto lane = include_combined ? 0 : 1;
         lane <= (include_lanes ? run.run_info().flowcell().lane_count() : 0);
         lane++) {
      /* Create an empty plot data object. */
      illumina::interop::model::plot::plot_data<
          illumina::interop::model::plot::candle_stick_point> data;
      illumina::interop::model::plot::filter_options options(
          run.run_info().flowcell().naming_method());
      options.lane(lane);
      /* Populate the data object. */
      X(run, metric_name, options, data, true);
      if (std::accumulate(
              data.begin(), data.end(), false,
              [](bool acc,
                 const illumina::interop::model::plot::series<
                     illumina::interop::model::plot::candle_stick_point> &
                     series) { return acc || series.size() > 0; })) {
        /* Set the name of the lane. If a regular lane, include `{n}` so that
         * the
         * front end can substitute the pool name. */
        std::stringstream name;
        if (lane == 0) {
          name << "Combined";
        } else {
          name << "{" << lane << "}";
        }
        dataset.push_back(std::make_pair(name.str(), std::move(data)));
      }
    }
  } catch (const std::exception &ex) {
    return;
  }
  if (!dataset.empty()) {
    render(type_name, dataset, output);
  }
}

/* Helper function to add a row to a chart. */
void add_chart_row(Json::Value &values, const std::string &name,
                   const std::string &value) {
  Json::Value result(Json::objectValue);
  result["name"] = name;
  result["value"] = value;
  values.append(std::move(result));
}
/* Create a metric that will end up as a chart. Only one of these should be
 * added to the metrics. */
void add_global_chart(
    const illumina::interop::model::metrics::run_metrics &run,
    const illumina::interop::model::summary::run_summary &run_summary,
    Json::Value &output) {
  Json::Value result(Json::objectValue);
  result["type"] = "chart";
  /* Create a JSON array of rows. */
  Json::Value values(Json::arrayValue);
  add_chart_row(values, "Ends",
                run.run_info().is_paired_end() ? "Paired" : "Single");

  std::stringstream cycles;
  cycles << run_summary.cycle_state().extracted_cycle_range().last_cycle()
         << " / " << run.run_info().total_cycles();
  add_chart_row(values, "Cycles", cycles.str());

  add_chart_row(values, "Base Mask", getRunBasesMask(run));
  std::stringstream q_30;
  q_30 << std::fixed << std::setprecision(2)
       << run_summary.total_summary().percent_gt_q30() << " %";
  add_chart_row(values, "% > Q30", q_30.str());
  if (run_summary.begin() != run_summary.end()) {
    // Read has two different meanings in this context due to Illumina
    // terminology, so we're going with clusters because that's what SAV says

    std::stringstream total_reads;
    total_reads.imbue(std::locale(""));
    total_reads << std::accumulate(
        run_summary.begin()->begin(), run_summary.begin()->end(), 0L,
        [](long acc,
           const illumina::interop::model::summary::lane_summary &summary) {
          return acc + summary.reads();
        });
    add_chart_row(values, "Clusters", total_reads.str());

    std::stringstream total_reads_pf;
    total_reads_pf.imbue(std::locale(""));
    total_reads_pf << std::accumulate(
        run_summary.begin()->begin(), run_summary.begin()->end(), 0L,
        [](long acc,
           const illumina::interop::model::summary::lane_summary &summary) {
          return acc + summary.reads_pf();
        });
    add_chart_row(values, "Clusters PF", total_reads_pf.str());
  }
  result["values"] = std::move(values);
  output.append(std::move(result));
}

/* Format a metrics value with a mean and deviation. */
std::string format(const illumina::interop::model::summary::metric_stat &stat,
                   const float scale = 1, const int precision = 2) {
  if (std::isnan(stat.mean())) {
    return "N/A";
  }
  std::stringstream output;
  output << std::setprecision(precision) << stat.mean() / scale;
  if (!std::isnan(stat.stddev())) {
    output << " ± " << stat.stddev() / scale;
  }
  return output.str();
}

/* Helper function to create a column in a table. */
void add_table_column(Json::Value &columns, const std::string &name,
                      const std::string &property) {
  Json::Value result(Json::objectValue);
  result["name"] = name;
  result["property"] = property;
  columns.append(std::move(result));
}

/* Create a chart with per-lane stats. */
void add_lane_charts(
    const illumina::interop::model::metrics::run_metrics &run,
    const illumina::interop::model::summary::run_summary &run_summary,
    Json::Value &output) {
  Json::Value result(Json::objectValue);
  result["type"] = "table";

  /* Create an empty array of columns. */
  Json::Value columns(Json::arrayValue);
  add_table_column(columns, "Lane", "lane");
  add_table_column(columns, "Pool", "pool");
  add_table_column(columns, "Density %", "densityPct");
  add_table_column(columns, "Density", "density");
  add_table_column(columns, "Density PF", "densityPf");
  add_table_column(columns, "% > Q30", "q30");
  auto index = 0;
  for (auto read = 0; read < run_summary.size(); read++) {
    auto is_index = run.run_info().read(read + 1).is_index();
    std::stringstream buffer;
    if (is_index) {
      index++;
      buffer << " (Index " << index << ")";
    } else {
      buffer << " (Read " << read + 1 - index << ")";
    }
    auto suffix = buffer.str();
    add_table_column(columns, "Errors" + suffix,
                     "errors" + std::to_string(read));
    if (!is_index) {
      add_table_column(columns, "Aligned" + suffix,
                       "aligned" + std::to_string(read));
    }
  }

  result["columns"] = std::move(columns);

  /* Create an empty array of rows; the columns will be made to match the above.
   */
  Json::Value rows(Json::arrayValue);
  for (auto lane = 0; lane < run_summary.lane_count(); lane++) {
    Json::Value row(Json::objectValue);
    row["lane"] = lane + 1;
    std::stringstream pool;
    pool << "{" << lane + 1 << "}";
    row["pool"] = pool.str();
    std::stringstream density_pct;
    density_pct << std::fixed << std::setprecision(2)
                << (run_summary[0][lane].density_pf().mean() /
                    run_summary[0][lane].density().mean()) *
                       100 << " %";
    row["densityPct"] = density_pct.str();
    row["density"] = format(run_summary[0][lane].density(), 1e3, 4);
    row["densityPf"] = format(run_summary[0][lane].density_pf(), 1e3, 4);
    row["q30"] = format(run_summary[0][lane].percent_gt_q30());
    for (auto read = 0; read < run_summary.size(); read++) {
      row["errors" + std::to_string(read)] =
          format(run_summary[read][lane].percent_gt_q30());
      if (!run.run_info().read(read + 1).is_index()) {
        row["aligned" + std::to_string(read)] =
            format(run_summary[read][lane].percent_aligned());
      }
    }
    rows.append(std::move(row));
  }
  result["rows"] = std::move(rows);

  output.append(std::move(result));
}

/* Create the Illumina yield-by-read graph. This is not provided by the Illumina
 * library, so we must fetch the yield data from the run summary. */
void add_yield_bars(
    const illumina::interop::model::metrics::run_metrics &run,
    const illumina::interop::model::summary::run_summary &run_summary,
    Json::Value &output) {
  Json::Value result(Json::objectValue);
  result["type"] = "illumina-yield-by-read";

  Json::Value categories(Json::arrayValue);
  auto index = 0;
  for (auto read = 0; read < run_summary.size(); read++) {
    auto is_index = run.run_info().read(read + 1).is_index();
    std::stringstream buffer;
    if (is_index) {
      index++;
      buffer << " Index " << index;
    } else {
      buffer << " Read " << read + 1 - index;
    }
    categories.append(buffer.str());
  }

  result["categories"] = std::move(categories);

  /* Create a series each for the yield and projected yield. */
  Json::Value series(Json::arrayValue);
  Json::Value yield_series(Json::objectValue);
  Json::Value projected_yield_series(Json::objectValue);
  yield_series["name"] = "Yield";
  projected_yield_series["name"] = "Projected Yield";

  Json::Value yield_data(Json::arrayValue);
  Json::Value projected_yield_data(Json::arrayValue);
  for (auto read = 0; read < run_summary.size(); read++) {
    yield_data.append(run_summary[read].summary().yield_g());
    projected_yield_data.append(
        run_summary[read].summary().projected_yield_g());
  }
  yield_series["data"] = std::move(yield_data);
  projected_yield_series["data"] = std::move(projected_yield_data);
  series.append(std::move(yield_series));
  series.append(std::move(projected_yield_series));
  result["series"] = std::move(series);

  output.append(std::move(result));
}

int main(int argc, const char **argv) {
  if (argc != 2) {
    return 1;
  }

  Json::Value result(Json::objectValue);

  /* Jackson expects the class to be embedded as an attribute, so we provided it
   * here. */
  result["class"] = "uk.ac.bbsrc.tgac.miso.dto.IlluminaNotificationDto";

  auto is_complete = true;

  illumina::interop::model::metrics::run_metrics run;
  try {
    run.read(argv[1]);
  } catch (illumina::interop::io::incomplete_file_exception e) {
    /* Assume that incomplete data is an active run, rather than a broken one.
     */
    is_complete = false;
  } catch (...) {
    /* We are really unable to recover from any other exceptions, so just bail
     * out. */
    return 2;
  }

  std::stringstream buffer;
  buffer << "Interop: " << illumina::interop::library_version()
         << " Instrument: " << run.run_parameters().version();
  result["software"] = buffer.str();

  /* The Illumina sequencers produce a variety of bad date formats. Reformat it
   * as "YYYY-mm-ddTHH:MM:ss". */
  for (const auto dateformat : stupidDateFormats) {
    std::tm detectedTime = {0};
    const char *output =
        strptime(run.run_info().date().c_str(), dateformat, &detectedTime);
    if (output != nullptr && *output == 0) {
      result["startDate"] = formatDate(&detectedTime);
      break;
    }
  }

  /* Copy all the trivial values from the run information.  */
  result["containerSerialNumber"] = run.run_info().flowcell_id();
  result["numCycles"] = (Json::Value::Int)run.run_info().total_cycles();
  result["pairedEndRun"] = run.run_info().is_paired_end();
  result["runAlias"] = run.run_info().name();
  result["sequencerName"] = run.run_info().instrument_name();
  result["laneCount"] =
      (Json::Value::Int)run.run_info().flowcell().lane_count();
  illumina::interop::model::summary::run_summary run_summary;
  illumina::interop::logic::summary::summarize_run_metrics(run, run_summary,
                                                           true);

  result["numReads"] = (Json::Value::Int)run.run_info().reads().size();
  result["bclCount"] =
      (Json::Value::Int)(run.run_info().flowcell().tiles_per_lane() *
                         run.run_info().flowcell().swath_count() *
                         run.run_info().flowcell().surface_count());

  result["imgCycle"] = (Json::Value::Int)run_summary.cycle_state()
                           .extracted_cycle_range()
                           .last_cycle();
  result["scoreCycle"] = (Json::Value::Int)run_summary.cycle_state()
                             .qscored_cycle_range()
                             .last_cycle();
  result["callCycle"] = (Json::Value::Int)run_summary.cycle_state()
                            .called_cycle_range()
                            .last_cycle();

  /* If there's an extraction metric with a end date, use that, reformatted as a
   * "YYYY-mm-dd" string. There can be multiple extractions, so pick the last
   * one. */
  const auto extractions =
      run.get<illumina::interop::model::metrics::extraction_metric>();
  std::time_t extraction_time(std::accumulate(
      extractions.begin(), extractions.end(), 0,
      [](illumina::interop::model::metric_base::base_metric::ulong_t a,
         const illumina::interop::model::metrics::extraction_metric &m) {
        return std::max(a, m.date_time());
      }));
  if (extraction_time == 0) {
    is_complete = false;
  } else {
    auto extraction_tm = std::localtime(&extraction_time);
    result["completionDate"] = formatDate(extraction_tm);
  }

  is_complete &= run_summary.cycle_state().called_cycle_range().last_cycle() ==
                 run.run_info().total_cycles();

  int readLength = 0;
  Json::Value indexLengths(Json::arrayValue);

  for (const auto &read : run.run_info().reads()) {
    if (read.is_index()) {
      indexLengths.append(length(read));
    } else {
      readLength = std::max(readLength, length(read));
    }
  }

  result["readLength"] = readLength;
  result["runBasesMask"] = getRunBasesMask(run);
  result["indexLengths"] = std::move(indexLengths);

  /* We can't tell the difference between the stopped or running states, so we
   * just assume running if it isn't finished. */
  result["healthType"] = is_complete ? "Completed" : "Running";

  /* Collect all the metrics. */
  Json::Value metrics_results(Json::arrayValue);
  add_plot<illumina::interop::logic::plot::plot_by_cycle>(
      illumina::interop::constants::Q30Percent, "illumina-q30-by-cycle", true,
      true, run, metrics_results);
  add_plot<illumina::interop::logic::plot::plot_by_cycle>(
      illumina::interop::constants::CalledIntensity,
      "illumina-called-intensity-by-cycle", true, true, run, metrics_results);
  add_plot<illumina::interop::logic::plot::plot_by_cycle>(
      illumina::interop::constants::BasePercent,
      "illumina-base-percent-by-cycle", true, true, run, metrics_results);
  add_plot<plot_by_lane_wrapper>(illumina::interop::constants::Clusters,
                                 "illumina-cluster-density-by-lane", true,
                                 false, run, metrics_results);
  add_global_chart(run, run_summary, metrics_results);
  add_lane_charts(run, run_summary, metrics_results);
  add_yield_bars(run, run_summary, metrics_results);
  /* The FastWriter will convert the metrics into a compact JSON string. */
  Json::FastWriter fastWriter;
  result["metrics"] = fastWriter.write(metrics_results);

  /* Write everything to standard output from consumption by Java. */
  std::cout << result << std::endl;
  return 0;
}
