package uk.ac.bbsrc.tgac.miso.webapp.controller.rest;

import java.io.IOException;
import java.net.URI;
import java.util.Collections;
import java.util.Set;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.ws.rs.core.Response.Status;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.SessionAttributes;
import org.springframework.web.util.UriComponents;
import org.springframework.web.util.UriComponentsBuilder;

import uk.ac.bbsrc.tgac.miso.core.data.PoolOrder;
import uk.ac.bbsrc.tgac.miso.core.data.PoolOrderCompletion;
import uk.ac.bbsrc.tgac.miso.core.data.SequencingParameters;
import uk.ac.bbsrc.tgac.miso.core.data.type.PlatformType;
import uk.ac.bbsrc.tgac.miso.core.util.PaginatedDataSource;
import uk.ac.bbsrc.tgac.miso.core.util.PaginationFilter;
import uk.ac.bbsrc.tgac.miso.dto.DataTablesResponseDto;
import uk.ac.bbsrc.tgac.miso.dto.Dtos;
import uk.ac.bbsrc.tgac.miso.dto.PoolOrderCompletionDto;
import uk.ac.bbsrc.tgac.miso.dto.PoolOrderDto;
import uk.ac.bbsrc.tgac.miso.service.PoolOrderCompletionService;
import uk.ac.bbsrc.tgac.miso.service.PoolOrderService;
import uk.ac.bbsrc.tgac.miso.service.SequencingParametersService;
import uk.ac.bbsrc.tgac.miso.webapp.util.PoolPickerResponse;
import uk.ac.bbsrc.tgac.miso.webapp.util.PoolPickerResponse.PoolPickerEntry;

@Controller
@RequestMapping("/rest")
@SessionAttributes("poolorder")
public class PoolOrderRestController extends RestController {

  private final JQueryDataTableBackend<PoolOrderCompletion, PoolOrderCompletionDto> jQueryBackend = new JQueryDataTableBackend<PoolOrderCompletion, PoolOrderCompletionDto>() {

    @Override
    protected PaginatedDataSource<PoolOrderCompletion> getSource() throws IOException {
      return poolOrderCompletionService;
    }

    @Override
    protected PoolOrderCompletionDto asDto(PoolOrderCompletion model) {
      return Dtos.asDto(model);
    }
  };

  protected static final Logger log = LoggerFactory.getLogger(PoolOrderRestController.class);

  @Autowired
  private PoolOrderService poolOrderService;
  @Autowired
  private SequencingParametersService sequencingParametersService;
  @Autowired
  private PoolOrderCompletionService poolOrderCompletionService;

  @RequestMapping(value = "/pool/{id}/orders", method = RequestMethod.GET, produces = { "application/json" })
  @ResponseBody
  public Set<PoolOrderDto> getOrdersByPool(@PathVariable("id") Long id, UriComponentsBuilder uriBuilder, HttpServletResponse response)
      throws IOException {
    Set<PoolOrderDto> dtos = Dtos.asPoolOrderDtos(poolOrderService.getByPool(id));
    for (PoolOrderDto dto : dtos) {
      writeUrls(dto, uriBuilder);
    }
    return dtos;
  }

  @RequestMapping(value = "/pool/{id}/dt/completions", method = RequestMethod.GET, produces = { "application/json" })
  @ResponseBody
  public DataTablesResponseDto<PoolOrderCompletionDto> getCompletionsByPool(@PathVariable("id") Long id, UriComponentsBuilder uriBuilder, HttpServletRequest request, HttpServletResponse response)
      throws IOException {
    return jQueryBackend.get(request, response, uriBuilder, PaginationFilter.pool(id));
  }

  @RequestMapping(value = "/poolorder/{id}", method = RequestMethod.GET, produces = { "application/json" })
  @ResponseBody
  public PoolOrderDto getPoolOrder(@PathVariable("id") Long id, UriComponentsBuilder uriBuilder, HttpServletResponse response)
      throws IOException {
    PoolOrder result = poolOrderService.get(id);
    if (result == null) {
      throw new RestException("No pool order found with ID: " + id, Status.NOT_FOUND);
    } else {
      return writeUrls(Dtos.asDto(result), uriBuilder);
    }
  }

  @RequestMapping(value = "/poolorder", method = RequestMethod.POST, headers = { "Content-type=application/json" })
  @ResponseBody
  public ResponseEntity<?> createPoolOrder(@RequestBody PoolOrderDto poolOrderDto, UriComponentsBuilder b, HttpServletResponse response)
      throws IOException {
    PoolOrder poolOrder = Dtos.to(poolOrderDto);
    Long id = poolOrderService.create(poolOrder);
    UriComponents uriComponents = b.path("/poolorder/{id}").buildAndExpand(id);
    HttpHeaders headers = new HttpHeaders();
    headers.setLocation(uriComponents.toUri());
    return new ResponseEntity<>(headers, HttpStatus.CREATED);
  }
  
  @RequestMapping(value = "/poolorder/dt/completions/all/{platform}", method = RequestMethod.GET, produces = { "application/json" })
  @ResponseBody
  public DataTablesResponseDto<PoolOrderCompletionDto> getDtCompletions(@PathVariable String platform, UriComponentsBuilder uriBuilder,
      HttpServletRequest request, HttpServletResponse response)
      throws IOException {
    return jQueryBackend.get(request, response, uriBuilder, PaginationFilter.platformType(PlatformType.valueOf(platform)));
  }

  @RequestMapping(value = "/poolorder/dt/completions/active/{platform}", method = RequestMethod.GET, produces = { "application/json" })
  @ResponseBody
  public DataTablesResponseDto<PoolOrderCompletionDto> getDtCompletionsUnfulfilled(@PathVariable String platform,
      UriComponentsBuilder uriBuilder, HttpServletRequest request,
      HttpServletResponse response)
      throws IOException {
    return jQueryBackend.get(request, response, uriBuilder, PaginationFilter.fulfilled(false),
        PaginationFilter.platformType(PlatformType.valueOf(platform)));
  }

  @RequestMapping(value = "/poolorder/dt/completions/pending/{platform}", method = RequestMethod.GET, produces = { "application/json" })
  @ResponseBody
  public DataTablesResponseDto<PoolOrderCompletionDto> getDtCompletionsPending(@PathVariable String platform,
      UriComponentsBuilder uriBuilder, HttpServletRequest request,
      HttpServletResponse response)
      throws IOException {
    return jQueryBackend.get(request, response, uriBuilder, PaginationFilter.pending(),
        PaginationFilter.platformType(PlatformType.valueOf(platform)));
  }

  @RequestMapping(value = "/poolorder/{id}", method = RequestMethod.PUT, headers = { "Content-type=application/json" })
  @ResponseBody
  public ResponseEntity<?> updatePoolOrder(@PathVariable("id") Long id, @RequestBody PoolOrderDto poolOrderDto,
      HttpServletResponse response) throws IOException {
    PoolOrder poolOrder = poolOrderService.get(id);
    if (poolOrder == null) {
      throw new RestException("No pool order found with ID: " + id, Status.NOT_FOUND);
    }
    poolOrder.setPartitions(poolOrderDto.getPartitions());
    SequencingParameters parameters = sequencingParametersService.get(poolOrderDto.getParameters().getId());
    if (parameters == null) {
      throw new RestException("No sequencing parameters found with ID: " + poolOrderDto.getParameters(), Status.BAD_REQUEST);
    }
    poolOrder.setSequencingParameter(parameters);
    poolOrderService.update(poolOrder);
    return new ResponseEntity<>(HttpStatus.OK);
  }

  @RequestMapping(value = "/poolorder/{id}", method = RequestMethod.DELETE)
  @ResponseBody
  @ResponseStatus(HttpStatus.NO_CONTENT)
  public void deletePoolOrder(@PathVariable(name = "id", required = true) long id, HttpServletResponse response) throws IOException {
    PoolOrder order = poolOrderService.get(id);
    if (order == null) {
      throw new RestException("Pool Order " + id + " not found", Status.NOT_FOUND);
    }
    poolOrderService.delete(order);
  }

  private static PoolOrderDto writeUrls(PoolOrderDto poolOrderDto, UriComponentsBuilder uriBuilder) {
    URI baseUri = uriBuilder.build().toUri();
    poolOrderDto
        .setUrl(UriComponentsBuilder.fromUri(baseUri).path("/rest/poolorder/{id}").buildAndExpand(poolOrderDto.getId()).toUriString());
    poolOrderDto.setCreatedByUrl(
        UriComponentsBuilder.fromUri(baseUri).path("/rest/user/{id}").buildAndExpand(poolOrderDto.getCreatedById()).toUriString());
    poolOrderDto.setUpdatedByUrl(
        UriComponentsBuilder.fromUri(baseUri).path("/rest/user/{id}").buildAndExpand(poolOrderDto.getUpdatedById()).toUriString());
    return poolOrderDto;
  }

  @RequestMapping(value = "/poolorder/picker/active", method = RequestMethod.GET, produces = { "application/json" })
  @ResponseBody
  public PoolPickerResponse getPickersByUnfulfilled(@RequestParam("platform") String platform) throws IOException {
    return getPoolPickerWithFilters(100,
        PaginationFilter.platformType(PlatformType.valueOf(platform)),
        PaginationFilter.fulfilled(false));
  }

  @RequestMapping(value = "/poolorder/picker/chemistry", method = RequestMethod.GET, produces = { "application/json" })
  @ResponseBody
  public PoolPickerResponse getPickersByChemistry(@RequestParam("platform") String platform,
      @RequestParam("seqParamsId") Long paramsId,
      @RequestParam("fulfilled") boolean fulfilled) throws IOException {
    return getPoolPickerWithFilters(100,
        PaginationFilter.platformType(PlatformType.valueOf(platform)),
        PaginationFilter.fulfilled(fulfilled),
        PaginationFilter.sequencingParameters(paramsId));
  }

  private PoolPickerResponse getPoolPickerWithFilters(Integer limit, PaginationFilter... filters) throws IOException {
    PoolPickerResponse ppr = new PoolPickerResponse();
    ppr.populate(poolOrderCompletionService, true, "lastUpdated", limit, PoolOrderRestController::orderTransform, filters);
    return ppr;
  }

  private static PoolPickerEntry orderTransform(PoolOrderCompletion order) {
    return new PoolPickerEntry(Dtos.asDto(order.getPool(), true), Collections.singletonList(Dtos.asDto(order)));
  }
}
