/*
 * Copyright (c) 2012. The Genome Analysis Centre, Norwich, UK
 * MISO project contacts: Robert Davey @ TGAC
 * *********************************************************************
 *
 * This file is part of MISO.
 *
 * MISO is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * MISO is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with MISO.  If not, see <http://www.gnu.org/licenses/>.
 *
 * *********************************************************************
 */

package uk.ac.bbsrc.tgac.miso.core.data;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.persistence.CascadeType;
import javax.persistence.Column;
import javax.persistence.Entity;
import javax.persistence.EnumType;
import javax.persistence.Enumerated;
import javax.persistence.GeneratedValue;
import javax.persistence.GenerationType;
import javax.persistence.Id;
import javax.persistence.Inheritance;
import javax.persistence.InheritanceType;
import javax.persistence.JoinColumn;
import javax.persistence.JoinTable;
import javax.persistence.ManyToMany;
import javax.persistence.ManyToOne;
import javax.persistence.OneToMany;
import javax.persistence.Table;
import javax.persistence.Temporal;
import javax.persistence.TemporalType;
import javax.persistence.Transient;

import com.eaglegenomics.simlims.core.Group;
import com.eaglegenomics.simlims.core.Note;
import com.eaglegenomics.simlims.core.SecurityProfile;
import com.eaglegenomics.simlims.core.User;

import uk.ac.bbsrc.tgac.miso.core.data.impl.InstrumentImpl;
import uk.ac.bbsrc.tgac.miso.core.data.impl.SequencerPartitionContainerImpl;
import uk.ac.bbsrc.tgac.miso.core.data.impl.UserImpl;
import uk.ac.bbsrc.tgac.miso.core.data.impl.changelog.RunChangeLog;
import uk.ac.bbsrc.tgac.miso.core.data.type.HealthType;
import uk.ac.bbsrc.tgac.miso.core.data.type.PlatformType;
import uk.ac.bbsrc.tgac.miso.core.security.SecurableByProfile;

/**
 * A Run represents a sequencing run on a single sequencing instrument, referenced by a {@link Instrument}, comprising one or more
 * {@link SequencerPartitionContainer} objects in which {@link Pool}s are placed on {@link SequencerPoolPartition}s.
 * <p/>
 * Runs can be QCed via {@link RunQC} objects, and are always associated with a given {@link PlatformType}
 * 
 * @author Rob Davey
 * @since 0.0.2
 */
@Entity
@Table(name = "Run")
@Inheritance(strategy = InheritanceType.JOINED)
public abstract class Run
    implements SecurableByProfile, Comparable<Run>, Watchable, Nameable, ChangeLoggable, Aliasable,
    Serializable {
  private static final long serialVersionUID = 1L;

  /** Field PREFIX */
  public static final String PREFIX = "RUN";

  public static final Long UNSAVED_ID = 0L;

  private String accession;

  @Column(nullable = false)
  private String alias;

  @OneToMany(targetEntity = RunChangeLog.class, mappedBy = "run", cascade = CascadeType.REMOVE)
  private final Collection<ChangeLog> changeLogs = new ArrayList<>();
  @Temporal(TemporalType.DATE)
  private Date completionDate;

  @ManyToMany(targetEntity = SequencerPartitionContainerImpl.class)
  @JoinTable(name = "Run_SequencerPartitionContainer", joinColumns = {
      @JoinColumn(name = "Run_runId") }, inverseJoinColumns = {
          @JoinColumn(name = "containers_containerId") })
  private List<SequencerPartitionContainer> containers = new ArrayList<>();

  private String description;
  private String filePath;

  @Enumerated(EnumType.STRING)
  @Column(nullable = false)
  private HealthType health = HealthType.Unknown;

  @ManyToOne(targetEntity = UserImpl.class)
  @JoinColumn(name = "creator", nullable = false, updatable = false)
  private User creator;

  @Column(name = "created", nullable = false, updatable = false)
  @Temporal(TemporalType.TIMESTAMP)
  private Date creationTime;

  @ManyToOne(targetEntity = UserImpl.class)
  @JoinColumn(name = "lastModifier", nullable = false)
  private User lastModifier;

  @Column(nullable = false)
  @Temporal(TemporalType.TIMESTAMP)
  private Date lastModified;

  private String metrics;

  private String name;

  @ManyToMany(targetEntity = Note.class, cascade = CascadeType.ALL)
  @JoinTable(name = "Run_Note", joinColumns = {
      @JoinColumn(name = "run_runId") }, inverseJoinColumns = {
          @JoinColumn(name = "notes_noteId") })
  private Collection<Note> notes = new HashSet<>();

  @Id
  @GeneratedValue(strategy = GenerationType.AUTO)
  private long runId = UNSAVED_ID;

  @ManyToOne(targetEntity = SecurityProfile.class, cascade = CascadeType.PERSIST)
  @JoinColumn(name = "securityProfile_profileId")
  private SecurityProfile securityProfile = new SecurityProfile();

  @ManyToOne(targetEntity = InstrumentImpl.class)
  @JoinColumn(name = "instrumentId", nullable = false)
  private Instrument sequencer;

  @ManyToOne
  @JoinColumn(name = "sequencingParameters_parametersId")
  private SequencingParameters sequencingParameters;
  @Temporal(TemporalType.DATE)
  private Date startDate;

  @Transient
  // not Hibernate-managed
  private Group watchGroup;

  @ManyToMany(targetEntity = UserImpl.class)
  @JoinTable(name = "Run_Watcher", joinColumns = { @JoinColumn(name = "runId") }, inverseJoinColumns = { @JoinColumn(name = "userId") })
  private Set<User> watchUsers = new HashSet<>();

  /**
   * Construct a new Run with a default empty SecurityProfile
   */
  public Run() {
    setSecurityProfile(new SecurityProfile());
  }

  /**
   * Construct a new Run with a SecurityProfile owned by the given User
   * 
   * @param user
   *          of type User
   * 
   */
  public Run(User user) {
    setSecurityProfile(new SecurityProfile(user));
  }

  public void addNote(Note note) {
    this.notes.add(note);
  }

  public void addSequencerPartitionContainer(SequencerPartitionContainer f) {
    f.setSecurityProfile(getSecurityProfile());
    if (f.getId() == 0L && f.getIdentificationBarcode() == null) {
      // can't validate it so add it anyway. this will only usually be the case for new run population.
      this.containers.add(f);
    } else {
      if (!this.containers.contains(f)) {
        this.containers.add(f);
      }
    }
  }

  @Override
  public void addWatcher(User user) {
    watchUsers.add(user);
  }

  @Override
  public int compareTo(Run t) {
    if (getId() < t.getId()) return -1;
    if (getId() > t.getId()) return 1;
    return 0;
  }

  @Override
  public ChangeLog createChangeLog(String summary, String columnsChanged, User user) {
    RunChangeLog changeLog = new RunChangeLog();
    changeLog.setRun(this);
    changeLog.setSummary(summary);
    changeLog.setColumnsChanged(columnsChanged);
    changeLog.setUser(user);
    return changeLog;
  }

  @Override
  public boolean equals(Object obj) {
    if (obj == null) return false;
    if (obj == this) return true;
    if (!(obj instanceof Run)) return false;
    Run them = (Run) obj;
    // If not saved, then compare resolved actual objects. Otherwise
    // just compare IDs.
    if (getId() == UNSAVED_ID || them.getId() == UNSAVED_ID) {
      return getAlias().equals(them.getAlias());
    } else {
      return getId() == them.getId();
    }
  }

  public String getAccession() {
    return accession;
  }

  @Override
  public String getAlias() {
    return alias;
  }

  public Date getCompletionDate() {
    return completionDate;
  }

  public String getDescription() {
    return description;
  }

  public String getFilePath() {
    return filePath;
  }

  public HealthType getHealth() {
    return health;
  }

  @Override
  public long getId() {
    return runId;
  }

  @Override
  public User getLastModifier() {
    return lastModifier;
  }

  @Override
  public void setLastModifier(User lastModifier) {
    this.lastModifier = lastModifier;
  }

  @Override
  public Date getLastModified() {
    return lastModified;
  }

  @Override
  public void setLastModified(Date lastModified) {
    this.lastModified = lastModified;
  }

  @Override
  public User getCreator() {
    return creator;
  }

  @Override
  public void setCreator(User creator) {
    this.creator = creator;
  }

  @Override
  public Date getCreationTime() {
    return creationTime;
  }

  @Override
  public void setCreationTime(Date created) {
    this.creationTime = created;
  }

  public String getMetrics() {
    return metrics;
  }

  @Override
  public String getName() {
    return name;
  }

  public Collection<Note> getNotes() {
    return notes;
  }

  @Override
  public SecurityProfile getSecurityProfile() {
    return securityProfile;
  }

  public List<SequencerPartitionContainer> getSequencerPartitionContainers() {
    if (this.containers != null) Collections.sort(this.containers);
    return containers;
  }

  public Instrument getSequencer() {
    return sequencer;
  }

  public SequencingParameters getSequencingParameters() {
    return sequencingParameters;
  }

  public Date getStartDate() {
    return startDate;
  }

  @Override
  public String getWatchableIdentifier() {
    return getName();
  }

  @Override
  public Set<User> getWatchers() {
    Set<User> allWatchers = new HashSet<>();
    if (watchGroup != null) allWatchers.addAll(watchGroup.getUsers());
    if (watchUsers != null) allWatchers.addAll(watchUsers);
    return allWatchers;
  }

  public Group getWatchGroup() {
    return watchGroup;
  }

  public Set<User> getWatchUsers() {
    return watchUsers;
  }

  @Override
  public int hashCode() {
    if (getId() != UNSAVED_ID) {
      return (int) getId();
    } else {
      final int PRIME = 37;
      int hashcode = 1;
      if (getAlias() != null) hashcode = PRIME * hashcode + getAlias().hashCode();
      return hashcode;
    }
  }

  public boolean isFull() {
    return containers.size() >= sequencer.getPlatform().getNumContainers();
  }

  @Override
  public void removeWatcher(User user) {
    watchUsers.remove(user);
  }

  public void setAccession(String accession) {
    this.accession = accession;
  }

  public void setAlias(String alias) {
    this.alias = alias;
  }

  public void setCompletionDate(Date completionDate) {
    this.completionDate = completionDate;
  }

  public void setDescription(String description) {
    this.description = description;
  }

  public void setFilePath(String filePath) {
    this.filePath = filePath;
  }

  public void setHealth(HealthType health) {
    if (health.isAllowedFromSequencer()) {
      this.health = health;
    } else {
      throw new IllegalArgumentException("Cannot set a status to " + health.getKey());
    }
  }

  public void setId(long id) {
    this.runId = id;
  }

  public void setMetrics(String metrics) {
    this.metrics = metrics;
  }

  public void setName(String name) {
    this.name = name;
  }

  public void setNotes(Collection<Note> notes) {
    this.notes = notes;
  }

  @Override
  public void setSecurityProfile(SecurityProfile securityProfile) {
    this.securityProfile = securityProfile;
  }

  public void setSequencerPartitionContainers(List<SequencerPartitionContainer> containers) {
    this.containers = containers;
  }

  public void setSequencer(Instrument sequencer) {
    this.sequencer = sequencer;
  }

  public void setSequencingParameters(SequencingParameters parameters) {
    this.sequencingParameters = parameters;
  }

  public void setStartDate(Date startDate) {
    this.startDate = startDate;
  }

  @Override
  public void setWatchGroup(Group watchGroup) {
    this.watchGroup = watchGroup;
  }

  public void setWatchUsers(Set<User> watchUsers) {
    this.watchUsers = watchUsers;
  }

  @Override
  public Collection<ChangeLog> getChangeLog() {
    return changeLogs;
  }

  public Boolean getPairedEnd() {
    return null;
  }

  public void setPairedEnd(boolean pairedEnd) {
    throw new UnsupportedOperationException("Cannot set paired end on runs from this platform.");
  }
  
  public abstract PlatformType getPlatformType();

  public String getProgress() {
    return getHealth() == HealthType.Running ? "Running" : "";
  }

  @Override
  public boolean isSaved() {
    return getId() != UNSAVED_ID;
  }

}
