#R Scripts for the analyses and figures of Mellado et al.2024  "Developmental instability, body mass, and reproduction predict immunological response in short-tailed bats"

require(ggplot2)
require(gridExtra)
require(MuMIn)
require(car)
require(effects)
require(visreg)
require(effectsize)
require(lme4)

#Read data file combining experiments performed in Nov 2018, May 2019, Oct 2020, and May2021
pha<-read.table("Carollia_PHA.txt",header=T, row.names = 1, stringsAsFactors = T)
summary(pha)
mean(pha$IPHA)
sd(pha$IPHA)


boxplot(BMass~Sex:BSeason, data = pha)
boxplot(IPHA~Sex:BSeason, data = pha)

#Sexual dimorphism in variables
t.test(IPHA~Sex,data=pha)
t.test(ForA~Sex,data=pha)
t.test(BMass~Sex,data=pha)

#Body Mass change according to season and sex
lmBM1<-lm(BMass~BSeason*Sex, data = pha)
anova(lmBM1)
plot(predictorEffects(lmBM1, "BSeason", partial.residuals=T))
summary(lmBM1)
confint(lmBM1)

#Full model, including Body Mass, Forearm Asymmetry, Sex, and Breeding Season, as well as interactions
lmpha1<-lm(IPHA~(BMass+ForA)*BSeason*Sex,data=pha,na.action = "na.fail")
#Parameter estimates
summary(lmpha1)
#Checking for multicollinearity
vif(lmpha1, type = "predictor")
#Model selection
modpha1<-dredge(lmpha1,extra = "R^2")
#Save model selection table
write.table(as.data.frame(modpha1),"ModSel01.txt")

#Reduced model, selection shows one best model
lmpha2<-lm(IPHA~BMass*BSeason+ForA*Sex,data=pha,na.action = "na.fail")

#Parameter estimates
summary(lmpha2)
#Confidence intervals
confint(lmpha2)
#Standardized parameters. These are more informative to compare effects of predictors at different scales
effectsize(lmpha2)
#Checking for multicollinearity
vif(lmpha2, type = "predictor")
#Diagnostic plots OK
plot(lmpha2)

#Effect plots:
#Define plotting variables to group observations by sex using pch and colour
Sexpch<-c(16,17)[as.numeric(pha$Sex)]
SexCol<-c("#E69F00", "#009E73")[as.numeric(pha$Sex)]

#Use visreg to save partial residuals
vis.lmBM<-visreg(lmpha2, xvar = "BMass", by = "BSeason")
vis.lmForA<-visreg(lmpha2, xvar = "ForA", by = "Sex")

#add partial residuals to original data frame for plotting. Made sure that object order is the same in both sets.
pha$Part.Res.BM<-vis.lmBM$res$visregRes
pha$Part.Res.ForA<-vis.lmForA$res$visregRes[order(as.numeric(row.names(vis.lmForA$res)))]

#Plot with visreg partial residuals: BMass by BSeason (Figure 1)
plotBM<-ggplot(data = pha, aes(x = BMass, y = Part.Res.BM)) +
  geom_point(size = 3, shape = Sexpch, colour = SexCol) +
  facet_wrap(BSeason ~ ., nrow = 1) +
  geom_smooth(method = "lm")+
  labs(title = "", x = "Body Mass (g)", y = "IPHA Partial Residuals") +
  theme(text = element_text(size=rel(4.5)),panel.border=element_rect(linetype="solid",fill=NA), strip.text.x = element_text(size = 12))

plotForA<-ggplot(data = pha, aes(x = ForA, y = Part.Res.ForA)) +
  geom_point(size = 3, shape = Sexpch) +
  facet_wrap(Sex ~ ., nrow = 1) +
  geom_smooth(method = "lm")+
  labs(title = "", x = "Forearm Asymmetry (mm)", y = "IPHA Partial Residuals") +
  theme(text = element_text(size=rel(4.5)),panel.border=element_rect(linetype="solid",fill=NA), strip.text.x = element_text(size = 12))

Figure1<-grid.arrange(plotBM, plotForA,nrow=2)
ggsave("Figure1.svg", plot = Figure1, device = "svg", width = 10, height = 10)

#Models including only reproductively active males (TL > 6 mm)
phaTL<-read.table("Carollia_phaTL.txt",header=T, row.names=1)
phaTL$BSeason<-factor(phaTL$BSeason)
summary(phaTL)

#Full linear model including forearm asymmetry, body mass, testicle length, and breeding season
lmTL1<-lm(IPHA~(ForA+BMass+TL)*BSeason,data=phaTL,na.action = "na.fail")
#Parameter estimates
summary(lmTL1)
#Model selection
modTL1<-dredge(lmTL1,extra = "R^2")
#Save model selection table
write.table(as.data.frame(modTL1),"ModSel-TL01.txt")

#Best fitting model
lmTL2<-lm(IPHA~ForA+TL,data=phaTL,na.action = "na.fail")
#Parameter estimates
summary(lmTL2)
#Confidence intervals
confint(lmTL2)
#Standardized parameters. These are more informative to compare effects of predictors at different scales
effectsize(lmTL2)
#Checking for multicollinearity
vif(lmTL2)
#Diagnostic plots OK
plot(lmTL2)

#Effect plots
#save partial residuals
lmTLForA.vis<-visreg(lmTL2, "ForA")
lmTL.vis<-visreg(lmTL2, "TL")

#plot effect of forearm asymmetry
plotTLForA<-ggplot(data = lmTLForA.vis$res, aes(x = ForA, y = visregRes)) +
  geom_point(size = 3, shape = 17) +
  ylim(0.1,1)+
  geom_smooth(method = "lm")+
  labs(title = "", x = "Forearm Asymmetry (mm)", y = "IPHA Partial Residuals") +
  theme(text = element_text(size=rel(4.5)),panel.border=element_rect(linetype="solid",fill=NA))
#plot effect of testicle length
plotForATL<-ggplot(data = lmTL.vis$res, aes(x = TL, y = visregRes)) +
  geom_point(size = 3, shape = 17) +
  ylim(0.1,1)+
  geom_smooth(method = "lm")+
  labs(title = "", x = "Testicle Length (mm)", y = "IPHA Partial Residuals") +
  theme(text = element_text(size=rel(4.5)),panel.border=element_rect(linetype="solid",fill=NA))

Figure2<-grid.arrange(plotTLForA,plotForATL,nrow=1)
ggsave("Figure2.svg", plot = Figure2, device = "svg", width = 12, height = 6)



#Supplementary figure 2
#Read data with 24 individual measurements over 36 hours.

pha36<-read.table("pha36hours.txt", header = T)
pha36$Ind<-factor(pha36$Ind)

#Fit quadratic model
mlm1 <- lmer(IPHA ~ Hour + I(Hour^2) + (1 | Ind), data = pha36)
summary(mlm1)

#plot effect
plot(predictorEffects(mlm1, partial.residuals = T),
partial.residual=list(pch=1,col="black",cex=0.7,smooth=F,fig.show="hide"),
main="",
axes=list(x=list(Hour = list(lab="Hours after injection")),
y=list(lab="IPHA")))


