import json
from os import listdir
import sys
import geojson
import geojson_validation
import openpyxl 

def writeResultsToExcel(model, results, learning_type):
    sheet = sheets[model]
    column = columns[learning_type]
    wb = openpyxl.load_workbook('evaluation.xlsx')
    ws = wb[sheet]
    for test in results:
        cell = int(test["test"])+1
        ws.cell(row=cell, column=column, value = test['evaluation'])
    wb.save('evaluation.xlsx')

def ordered(obj):
    if isinstance(obj, dict):
        return sorted((k, ordered(v)) for k, v in obj.items())
    if isinstance(obj, list):
        return sorted((ordered(x) for x in obj), key=str)
    else:
        return obj

def is_same_geojson(geo_1, geo_2):
    return ordered(geo_1) == ordered(geo_2)

def evaluate_responses(path):
    evaluations = []
    for response in listdir(path):
        with open(f'{path}/{response}', "r") as f:
            response_json = f.read()
            test_number = response.partition('-')[2].rstrip('-response.txt')
            try:
                response_geojson = geojson.loads(response_json)
                if response_geojson.get("type") == "Feature" and "geometry" in response_geojson:
                    response_geometry =  response_geojson.get['geometry']
                    result = geojson_validation.is_valid(response_geometry)
                    if (result['valid'] == 'yes'):
                        with open(f'./boundaries-expected-results/expected-result-{test_number}.txt', "r") as f:
                            expected_result = geojson.loads(f.read())
                            expected_geometry = expected_result.features[0].geometry
                            if (is_same_geojson(response_geometry, expected_geometry)):
                                print(f'test {test_number}: 1 semantically correct \n')
                                evaluations.append({"test": test_number, "evaluation": 1})
                            else: 
                                print(f'test {test_number}: 0 only syntactically valid \n')
                                evaluations.append({"test": test_number, "evaluation": 0})
                    else: 
                        print(f'test {test_number}: -1 is NOT a valid GEOJSON \n')
                        evaluations.append({"test": test_number, "evaluation": -1})
                else: 
                    print(f'test {test_number}: -1 is NOT a valid GEOJSON \n')
                    evaluations.append({"test": test_number, "evaluation": -1})
                    
            except ValueError as e:
                print(f'test {test_number}: -1 is NOT a valid JSON \n')
                evaluations.append({"test": test_number, "evaluation": -1})
    return evaluations

""" Config parameters """
""" Model folder name respective to model name"""
models = {
    "llama3:instruct": "llama3-it",
    "codegemma:instruct": "codegemma-it",
    "mistral:instruct": "mistral-it"
}
""" Excell sheet name respective to model """
sheets = {
    "llama3:instruct": "Llama3",
    "codegemma:instruct": "CodeGemma",
    "mistral:instruct": "Mistral"
}
""" Sheet column respective to learning type """
columns = {
    "zero-shot": 2,
    "few-shot-1": 3,
    "few-shot-2": 4,
    "few-shot-3": 5,
}

""" 
    Program to run evaluation on all model responses of the given learning type
    Args:
            model(string), learning type(string)
    Example call:
        >> python evaluate_responses.py "codegemma:instruct" "zero-shot"
"""
def main():
    model = sys.argv[1]
    learning_type = sys.argv[2]
    path = f'./{models[model]}/{learning_type}'

    evaluations = evaluate_responses(path)
    writeResultsToExcel(model, evaluations, learning_type)

    print(f'\n//////// Model {model} {learning_type} evaluation complete')

if __name__ == "__main__":
    main()