#### R Script to produce the analyses and graphs (figures 3-5) for the following paper:

# Can starlings use a reliable cue of future food deprivation to adaptively 
# modify foraging and fat reserves?

# Authors: Menno van Berkel1, Melissa Bateson1, Daniel Nettle1 and Jonathon Dunn1
# 1Centre for Behaviour and Evolution & Institute of Neuroscience, Newcastle University, Newcastle, UK

# Script written by Jonathon Dunn
# March 2018: first version of script for submission 

# Notes on the variable names:
# When the data frames have been manipulated to give summaries, the naming convention is: 
# <name data frame data based on> . <response variable(s)> . <variable(s) data split by>

# Remove any existing data/values in the global environment
rm(list=ls())

packages<-c("ggplot2",
            "plyr",
            "dplyr",
            "lme4",
            "MuMIn",
            "arm")

lapply(packages, require, character.only = TRUE)

# Set the ggplot2 theme to be theme_bw() globally and specify accessory functions
# needed for analysis and plotting
theme_set(theme_bw())

# Create length function that can handle NAs
length2 <- function (x, na.rm=FALSE) {
  if (na.rm) sum(!is.na(x))
  else       length(x)
}

#### Read in the data ####

# Setwd


# Read in .csv files
ft<-read.csv("foraging_time.csv")
ce<-read.csv("crumb_eaten.csv")
w<-read.csv("weights.csv")

# Examine data frames
head(ft)
head(ce)
head(w)

############################# Time spent foraging ##################################

# Use data frame ft 

# Model 1: predictors of time spent foraging post cue and pre deprivation
m1<-lmer(Time_spent_foraging_post_cue_and_pre_deprivation~
                Day*Cue*Treatment+Foraging_pre_cue+(1|Natal_nest/Bird),data=ft,REML=F,na.action=na.fail)
summary(m1)
m1.s<-standardize(m1,standardize.y=F)
m1.d <- dredge(m1.s)
(m1.ds<-subset(m1.d, delta < 2))
nrow(m1.d) 
nrow(m1.ds) 
m1.a<-model.avg(m1.ds)
summary(m1.a) 
confint(m1.a) 

# Create Figure 3
(ft.foraging.day.treatment.cue<-ft%>%
    group_by(Treatment,Day,Cue)%>%
    summarise(mean=mean(Time_spent_foraging_post_cue_and_pre_deprivation,na.rm=T),
              se = sd(Time_spent_foraging_post_cue_and_pre_deprivation,na.rm=T)/sqrt(length2(Time_spent_foraging_post_cue_and_pre_deprivation,na.rm=T))))

(g.foraging.day.treatment.cue<-ggplot(ft.foraging.day.treatment.cue,aes(x=Day,y=mean,colour=Cue))+
    geom_point()+
    stat_smooth(method="lm",se=F)+
    geom_errorbar(aes(ymin=mean-se,ymax=mean+se),position=position_dodge(0.1),width=0.1,linetype="solid")+
    facet_wrap(~Treatment)+
    xlab("Day")+
    ylab("Time spent foraging post-cue (s)")+
    scale_colour_manual(values=c("darkred","darkblue"),name="Cue",breaks=c("Bright", "Dark"),
                        labels=c("Not present", "Present"))+
    theme(legend.position="bottom"))


# Export graph
ggsave("Fig3.jpg",g.foraging.day.treatment.cue,height=6,width=6,dpi=300)


################## Food intake ####################

# Use data frame ce

# Model 2: predictors of food intake pre deprivation
m2<-lmer(Food_intake_pre_deprivation~
                 Day*Cue*Treatment+
                 Food_eaten_before+
                 (1|Natal_nest/Bird),data=ce,REML=F,na.action=na.fail)
summary(m2)
m2.s<-standardize(m2,standardize.y=F)
m2.d <- dredge(m2.s)
(m2.ds<-subset(m2.d, delta < 2))
nrow(m2.d) 
nrow(m2.ds) 
m2.a<-model.avg(m2.ds)
summary(m2.a) 
confint(m2.a) 

# Create Figure 4
(ce.food.day.treatment.cue<-ce%>%
    group_by(Treatment,Day,Cue)%>%
    summarise(mean=mean(Food_intake_pre_deprivation,na.rm=T),
              se = sd(Food_intake_pre_deprivation,na.rm=T)/sqrt(length2(Food_intake_pre_deprivation,na.rm=T))))

(g.food.day.treatment.cue<-ggplot(ce.food.day.treatment.cue,aes(x=Day,y=mean,colour=Cue))+
    geom_point()+
    stat_smooth(method="lm",se=F)+
    geom_errorbar(aes(ymin=mean-se,ymax=mean+se),position=position_dodge(0.1),width=0.1,linetype="solid")+
    facet_wrap(~Treatment)+
    xlab("Day")+
    ylab("Food intake pre-deprivation (g)")+
    scale_colour_manual(values=c("darkred","darkblue"),name="Cue",breaks=c("Bright", "Dark"),
                        labels=c("Not present", "Present"))+
    theme(legend.position="bottom"))

# Export graph
ggsave("Fig4.jpg",g.food.day.treatment.cue,height=6,width=6,dpi=300)

################## Body mass at dawn ####################

# Use data frame w

# Model 3: predictors of body mass at dawn
m3<-lmer(Dawn_body_mass~Tarsus+Sex+Day*Treatment+Days_since_last_deprivation+(1|Natal_nest/Bird),na.action=na.fail,REML=F,data=w)
summary(m3)
m3.s<-standardize(m3,standardize.y=F)
m3.d <- dredge(m3.s)
(m3.ds<-subset(m3.d, delta < 2))
nrow(m3.d) 
nrow(m3.ds) 
m3.a<-model.avg(m3.ds)
summary(m3.a) 
confint(m3.a) 

# Create Figure 5
(w.bodycondition.day.treatment<-w%>%
    group_by(Treatment,Day)%>%
    summarise(mean=mean(Body_condition,na.rm=T),
              se = sd(Body_condition,na.rm=T)/sqrt(length2(Body_condition,na.rm=T))))

(g.bodycondition.day.treatment<-ggplot(w.bodycondition.day.treatment,aes(x=Day,y=mean,colour=Treatment))+
    geom_point(position=position_dodge(0.9))+
    stat_smooth(method="lm",se=F)+
    geom_errorbar(aes(ymin=mean-se,ymax=mean+se),position=position_dodge(0.9),width=0.1,linetype="solid")+
    xlab("Day")+
    ylab("Body condition at dawn")+
    scale_colour_manual(values=c("darkgreen","purple"),name="Treatment group")+
    theme(legend.position="bottom"))

# Export graph
ggsave("Fig5.jpg",g.bodycondition.day.treatment,height=6,width=6,dpi=300)

