
# load and install binary packages
ipak <- function(pkg){
    new.pkg <- pkg[!(pkg %in% installed.packages()[, "Package"])]
    if(length(new.pkg)) install.packages(new.pkg, dependencies = TRUE)
    sapply(pkg, require, character.only = TRUE)
}
packages <- c("ape", "reshape", "geiger", "data.table", "coda")
ipak(packages)

# load functions #
pder <- function (tree, tips) {
  tree <- tree
  treeabsent <- tree$tip.label[which(!(tree$tip.label %in% tips))]
  treeX <- drop.tip(tree, tip=treeabsent) #drop taxa not in C1  
  PD <- round(sum(treeX$edge.length), 4)
  return(PD)
}

# Calculates MPD, MNND for entrire trees ONLY! #
  mpnd.1 <- function (branch){
  branch[which(branch==0)] = NA_character_
  treemelt <- na.omit(melt(as.matrix(branch)), stringsAsFactors=F)
  colnames(treemelt) <- c("sp1", "sp2", "dist")

  xx <- aggregate(as.numeric(as.character(dist))~sp1, data=treemelt, FUN=function(x) c(mean=mean(x)))
  MPD <- as.numeric(as.character(xx[,2]))
  MNND <- data.table(treemelt)[ ,list(dist = min(as.numeric(as.character(dist)))), 'sp1']
  list(
    MPD = round(MPD, 4),    
    MNND = round(MNND$dist, 4)
  )
}

mpnd <- function (branch, tree.com){
  branch[which(branch==0)] = NA_character_
  treemelt <- na.omit(melt(as.matrix(branch)), stringsAsFactors=F)
  colnames(treemelt) <- c("sp1", "sp2", "dist")
  
  #select sub community pairs based on focal sub community and calculate pairwise distances
  # community tree
  c1.nat <- treemelt[treemelt$sp2 %in% tree.com[[2]], ]
  c2.nat <- treemelt[treemelt$sp2 %in% tree.com[[3]], ]
  c3.nat <- treemelt[treemelt$sp2 %in% tree.com[[4]], ]
  
  #Calculate MPD and MNND for each Community
  c1 <- c1.nat[c1.nat$sp1 %in% tree.com[[2]],]
  xx <- aggregate(as.numeric(as.character(dist))~sp1, data=c1, FUN=function(x) c(mean=mean(x)))
  mp.c1 <- as.numeric(as.character(xx[,2])) 
  nn.c1 <- data.table(c1)[,list(dist = min(as.numeric(as.character(dist)))), 'sp1']
    
  c2 <- c2.nat[c2.nat$sp1 %in% tree.com[[3]],]
  xx <- aggregate(as.numeric(as.character(dist))~sp1, data=c2, FUN=function(x) c(mean=mean(x)))
  mp.c2 <- as.numeric(as.character(xx[,2])) 
  nn.c2 <- data.table(c2)[,list(dist = min(as.numeric(as.character(dist)))), 'sp1']

  c3 <- c3.nat[c3.nat$sp1 %in% tree.com[[4]],]
  xx <- aggregate(as.numeric(as.character(dist))~sp1, data=c3, FUN=function(x) c(mean=mean(x)))
  mp.c3 <- as.numeric(as.character(xx[,2])) 
  nn.c3 <- data.table(c3)[,list(dist = min(as.numeric(as.character(dist)))), 'sp1']
  
  #Calculate MPD and MNND BETWEEN each Community
  c12 <- c1.nat[c1.nat$sp1 %in% tree.com[[3]],]
  xx <- aggregate(as.numeric(as.character(dist))~sp2, data=c12, FUN=function(x) c(mean=mean(x)))
  mp.c12 <- as.numeric(as.character(xx[,2]))
  nn.c12 <- data.table(c12)[,list(dist = min(as.numeric(as.character(dist)))), 'sp1']
  
  c13 <- c1.nat[c1.nat$sp1 %in% tree.com[[4]],]
  xx <- aggregate(as.numeric(as.character(dist))~sp2, data=c13, FUN=function(x) c(mean=mean(x)))
  mp.c31 <- as.numeric(as.character(xx[,2]))
  nn.c13 <- data.table(c13)[,list(dist = min(as.numeric(as.character(dist)))), 'sp1']
  
  c21 <- c2.nat[c2.nat$sp1 %in% tree.com[[2]],]
  nn.c21 <- data.table(c21)[,list(dist = min(as.numeric(as.character(dist)))), 'sp1']
  #mp.c21 <- as.numeric(as.character(c21$dist))
  c23 <- c2.nat[c2.nat$sp1 %in% tree.com[[4]],]
  xx <- aggregate(as.numeric(as.character(dist))~sp2, data=c23, FUN=function(x) c(mean=mean(x)))
  mp.c23 <- as.numeric(as.character(xx[,2]))
  nn.c23 <- data.table(c23)[,list(dist = min(as.numeric(as.character(dist)))), 'sp1']
  
  c31 <- c3.nat[c3.nat$sp1 %in% tree.com[[2]],]
  nn.c31 <- data.table(c31)[,list(dist = min(as.numeric(as.character(dist)))), 'sp1']
  #mp.c31 <- as.numeric(as.character(c31$dist))
  c32 <- c3.nat[c3.nat$sp1 %in% tree.com[[3]],]
  nn.c32 <- data.table(c32)[,list(dist = min(as.numeric(as.character(dist)))), 'sp1']
  #mp.c32 <- as.numeric(as.character(c32$dist))
  
  list(
    c12.MNND = round(c(nn.c12$dist, nn.c21$dist), 4),
    c23.MNND = round(c(nn.c23$dist, nn.c32$dist), 4),
    c31.MNND = round(c(nn.c31$dist, nn.c13$dist), 4),

    c12.MPD = round(mp.c12, 4),
    c23.MPD = round(mp.c23, 4),
    c31.MPD = round(mp.c31, 4),
    
    c1.MNND = round(nn.c1$dist, 4),
    c2.MNND = round(nn.c2$dist, 4),
    c3.MNND = round(nn.c3$dist, 4),
    
    c1.MPD = round(mp.c1, 4),
    c2.MPD = round(mp.c2, 4),
    c3.MPD = round(mp.c3, 4)
  )         
}

options(digits=15) # set number of digits to use in calculations (NOT rounding!)

# Set Working Directory #
wd <- "/wd"

# Choose Dataset #
ds <- "Poales"

# Set Outgroup #
#outgroup <- "Angiopteris_evecta"
outgroup <- "Elaeis_guineensis"

# dating options #
datings <- c("r8s", "PATHd8", "RAxML", "BLADJ")

for(z in 1: length(datings)){
  dating <- datings[[z]]

# Load Master Tree (Boot) and Process #
fd <- list.dirs(wd)
#fd <- fd[grepl(ds, fd)]
fd <- fd[grepl("master", fd)]
fd <- fd[grepl(paste(dating), fd)]
fd <- fd[!grepl("MACOSX", fd)]

setwd(fd)
bt <- list.files(fd, pattern = paste(ds))
master.ML <-read.tree(bt)
if(dating == "RAxML"){master.ML$edge.length <- 1000*master.ML$edge.length}

# Load TRUE Tree and Process #
fd <- list.dirs(wd)
fd <- fd[grepl("TRUE", fd)]
fd <- fd[grepl(paste(dating), fd)]
fd <- fd[!grepl("MACOSX", fd)]

t <- list.files(fd, pattern = paste(ds))
setwd(fd)
master.best <- read.tree(t[[1]])
if(dating == "RAxML"){master.best$edge.length <- 1000*master.best$edge.length}
best <- drop.tip(master.best, tip = outgroup)
PD.best <- sum(best$edge.length) #PD without outgroup

# Load Master ML Tree and Process #
master.ML.drop <- drop.tip(master.ML, tip = outgroup)
PD.master.ML <- sum(master.ML.drop$edge.length) #PD without outgroup

#set working directory for community trees
cfolders <- list.dirs(path=wd, full.names = TRUE, recursive = TRUE)
cfolders <- cfolders[grepl(ds, cfolders)]
cfolders <- cfolders[grepl("community", cfolders)]
cfolders <- cfolders[grepl(dating, cfolders)] #select folders pertaining to selected dating method

for (k in 1: length(cfolders)){
  
  #get sub community lists
  setwd(cfolders[[k]])
  setwd('..')
  dr <- getwd()
  t <- unzip(list.files(dr, pattern = "community"))
  t <- t[!grepl("MACOSX", t)]
  TXT <- Sys.glob(paste(dr, "/*.community.txt", sep=""))
  
  if (length(TXT) == 0) stop("Incorrect COMM!")
  COMM <- list()
  
  for (txt in TXT) {
    
    basename <- sub("^(\\S+?)\\..+", "\\1", txt, perl=T)
    
    FH1 <- file(txt)
    open(FH1, open="r")
    
    while (length(line <- readLines(FH1, n=1)) > 0) {
      
      if (grepl("^>", line, perl=T)) {
        
        id <- sub("^>(\\S+?)\\s.+", "\\1", line, perl=T)
        mpd <- sub("^>\\S+?\\s+?(\\S+).*", "\\1", line, perl=T)
        c1.m <- sub("^>\\S+?\\s+?\\S+?\\s+?\\S+?=(\\S+?),.*", "\\1", line, perl=T)
        c2.m <- sub("^>\\S+?\\s+?\\S+?\\s+?\\S+?=\\S+?=(\\S+?),.*", "\\1", line, perl=T)
        c3.m <- sub("^>\\S+?\\s+?\\S+?\\s+?\\S+?=\\S+?=\\S+?=(\\S+?),.*", "\\1", line, perl=T)
        c1c2.m <- sub("^>\\S+?\\s+?\\S+?\\s+?\\S+?=\\S+?=\\S+?=\\S+?=(\\S+?),.*", "\\1", line, perl=T)
        c2c3.m <- sub("^>\\S+?\\s+?\\S+?\\s+?\\S+?=\\S+?=\\S+?=\\S+?=\\S+?=(\\S+?),.*", "\\1", line, perl=T)
        c3c1.m <- sub("^>\\S+?\\s+?\\S+?\\s+?\\S+?=\\S+?=\\S+?=\\S+?=\\S+?=\\S+?=(.*)", "\\1", line, perl=T)
        }
      else if (grepl("^\\s*$", line, perl=T)) {
        next
      }
      else {
        line <- gsub("\\s+", "", line, perl=T)
        
        
        c1 <- sub("^.*c1=(\\S+?),c2.*", "\\1", line, perl=T)
        c2 <- sub("^.*c2=(\\S+?),c3.*", "\\1", line, perl=T)
        c3 <- sub("^.*c3=(\\S+)", "\\1", line, perl=T)
        
        COMM[[id]] <- list(name=paste(id, ".tre", sep=''), mpd=mpd, c1=c1, c2=c2, c3=c3, c1.m=c1.m, c2.m=c2.m, c3.m=c3.m, 
                           c1c2.m=c1c2.m, c2c3.m=c2c3.m, c3c1.m=c3c1.m)
      }
    }
    close(FH1)
  }
  
  file.remove(t)
  subc <- rbindlist(COMM)
  
  if(dating == "RAxML"){
    subc$name <- gsub(paste(ds), paste("RAxML_best.", ds, sep = ""), subc$name)
    #RAxML trees are named differently
  }else{
  subc$name <- gsub("tre", "best.tre", subc$name)
  }
  
  setwd(cfolders[[k]])
  trees <- list.files(pattern = "best.zip")
  trees <- unzip(trees)
  trees <- gsub("\\./", "", trees)
  
  #combine trees and their sub-communities (only when 1 tree = 1 set of sub-communities); also calculate community PD
  tree.com <- list()
  for (i in 1: length(trees)){
    c <- name.check(best, data.names=c(strsplit(as.character(subc$c1[i]), ',')[[1]], 
                                       strsplit(as.character(subc$c2[i]), ',')[[1]], strsplit(as.character(subc$c3[i]), ',')[[1]]))
    xx <- drop.tip(best, tip=c$tree_not_data)
    PD.c <- sum(xx$edge.length)
    tree.com[[i]] <- c(trees[which(trees == subc$name[i])], strsplit(as.character(subc$c1[i]), ','), 
                       strsplit(as.character(subc$c2[i]), ','), strsplit(as.character(subc$c3[i]), ','), as.numeric(subc$mpd[i]), as.numeric(PD.c/PD.best))
  
  }
  
  tree.com0 <- tree.com
  
  out <-NULL
  for(i in 1: length(tree.com0)){
      tree.com <- tree.com0[[i]]

  community.ML <- read.tree(tree.com[[1]])
  if(dating == "RAxML"){community.ML$edge.length <- 1000*community.ML$edge.length}
  
  # Process TRUE Tree #
  x <- name.check(master.best, data.names=c(tree.com[[2]], tree.com[[3]], tree.com[[4]]))
  best.phy <- drop.tip(master.best, tip=x$tree_not_data) #prune tree to include only taxa present in communities
  branch.best <- cophenetic.phylo(best.phy)
  mpnd.ML.c <- mpnd(branch.best, tree.com=tree.com) #MPD, MNND
  #mpnd.ML.c <- data.frame(mpnd.ML.c) #MPD, MNND
  mpnd.MLall <- mpnd.1(branch.best) #Total MPD/MNND of Species Pool on TRUE tree
  
  ML.c1.PD <- pder(best.phy, tips=tree.com[[2]]) #PD of c1 on TRUE tree  
  ML.c2.PD <- pder(best.phy, tips=tree.com[[3]]) #PD of c2 on TRUE tree  
  ML.c3.PD <- pder(best.phy, tips=tree.com[[4]]) #PD of c3 on TRUE tree
  
  # Process Master ML tree #
  x <- name.check(master.ML, data.names=c(tree.com[[2]], tree.com[[3]], tree.com[[4]]))
  master.ML.phy <- drop.tip(master.ML, tip=x$tree_not_data) #prune tree to include only taxa present in communities
  branch.master.ML <- cophenetic.phylo(master.ML.phy)
  mpnd.master.ML.c <- mpnd(branch.master.ML, tree.com=tree.com) #MPD, MNND
  #mpnd.master.ML.c <- data.frame(mpnd.master.ML.c) #MPD, MNND
  mpnd.master.MLall <- mpnd.1(branch.master.ML) #Total MPD/MNND of Species Pool on master ML tree
  
  master.ML.c1.PD <- pder(master.ML.phy, tips=tree.com[[2]]) #PD of c1 on master tree  
  master.ML.c2.PD <- pder(master.ML.phy, tips=tree.com[[3]]) #PD of c2 on master tree  
  master.ML.c3.PD <- pder(master.ML.phy, tips=tree.com[[4]]) #PD of c3 on master tree
  
  # Process Community ML tree #
  x <- name.check(community.ML, data.names=c(tree.com[[2]], tree.com[[3]], tree.com[[4]]))
  community.ML.phy <- drop.tip(community.ML, tip=x$tree_not_data) #prune tree to include only taxa present in communities
  branch.community.ML <- cophenetic.phylo(community.ML.phy)
  mpnd.community.ML.c <- mpnd(branch.community.ML, tree.com=tree.com) #MPD, MNND
  #mpnd.community.ML.c <- data.frame(mpnd.community.ML.c) #MPD, MNND
  mpnd.community.MLall <- mpnd.1(branch.community.ML) #Total MPD/MNND of Species Pool on community ML tree
  
  community.ML.c1.PD <- pder(community.ML.phy, tips=tree.com[[2]]) #PD of c1 on TRUE tree  
  community.ML.c2.PD <- pder(community.ML.phy, tips=tree.com[[3]]) #PD of c2 on TRUE tree  
  community.ML.c3.PD <- pder(community.ML.phy, tips=tree.com[[4]]) #PD of c3 on TRUE tree
  
  # Get TRUE Relationships from the TRUE Tree # (Account for the possibility of EXACT matches) 
  #BETA MNND 
  a <- HPDinterval(as.mcmc(as.numeric(mpnd.ML.c$c12.MNND)), prob=0.950000)
  b <- HPDinterval(as.mcmc(as.numeric(mpnd.ML.c$c31.MNND)), prob=0.950000)
  if(max(a[[1]],b[[1]]) <= min(a[[2]], b[[2]])){ML.c1.MNNDcloser <- "same"} else if(a[[1]] > b[[1]]){ML.c1.MNNDcloser <- "c3"} else if(a[[1]] < b[[1]]){ML.c1.MNNDcloser <- "c2"}
  a <- HPDinterval(as.mcmc(as.numeric(mpnd.ML.c$c23.MNND)), prob=0.950000)
  b <- HPDinterval(as.mcmc(as.numeric(mpnd.ML.c$c12.MNND)), prob=0.950000)
  if(max(a[[1]],b[[1]]) <= min(a[[2]], b[[2]])){ML.c2.MNNDcloser <- "same"} else if(a[[1]] > b[[1]]){ML.c2.MNNDcloser <- "c1"} else if(a[[1]] < b[[1]]){ML.c2.MNNDcloser <- "c3"}
  a <- HPDinterval(as.mcmc(as.numeric(mpnd.ML.c$c31.MNND)), prob=0.950000)
  b <- HPDinterval(as.mcmc(as.numeric(mpnd.ML.c$c23.MNND)), prob=0.950000)
  if(max(a[[1]],b[[1]]) <= min(a[[2]], b[[2]])){ML.c3.MNNDcloser <- "same"} else if(a[[1]] > b[[1]]){ML.c3.MNNDcloser <- "c2"} else if(a[[1]] < b[[1]]){ML.c3.MNNDcloser <- "c1"}
  #BETA MNND 
  a <- HPDinterval(as.mcmc(as.numeric(mpnd.ML.c$c12.MPD)), prob=0.950000)
  b <- HPDinterval(as.mcmc(as.numeric(mpnd.ML.c$c31.MPD)), prob=0.950000)
  if(max(a[[1]],b[[1]]) <= min(a[[2]], b[[2]])){ML.c1.MPDcloser <- "same"} else if(a[[1]] > b[[1]]){ML.c1.MPDcloser <- "c3"} else if(a[[1]] < b[[1]]){ML.c1.MPDcloser <- "c2"}
  a <- HPDinterval(as.mcmc(as.numeric(mpnd.ML.c$c23.MPD)), prob=0.950000)
  b <- HPDinterval(as.mcmc(as.numeric(mpnd.ML.c$c12.MPD)), prob=0.950000)
  if(max(a[[1]],b[[1]]) <= min(a[[2]], b[[2]])){ML.c2.MPDcloser <- "same"} else if(a[[1]] > b[[1]]){ML.c2.MPDcloser <- "c1"} else if(a[[1]] < b[[1]]){ML.c2.MPDcloser <- "c3"}
  a <- HPDinterval(as.mcmc(as.numeric(mpnd.ML.c$c31.MPD)), prob=0.950000)
  b <- HPDinterval(as.mcmc(as.numeric(mpnd.ML.c$c23.MPD)), prob=0.950000)
  if(max(a[[1]],b[[1]]) <= min(a[[2]], b[[2]])){ML.c3.MPDcloser <- "same"} else if(a[[1]] > b[[1]]){ML.c3.MPDcloser <- "c2"} else if(a[[1]] < b[[1]]){ML.c3.MPDcloser <- "c1"}
  
  #ALPHA PD
  ML.c12.PDgreater <- if(ML.c1.PD != ML.c2.PD){as.character(c("c1", "c2")[which(min(ML.c1.PD, ML.c2.PD) != c(ML.c1.PD, ML.c2.PD))])} else{"same"}
  ML.c23.PDgreater <- if(ML.c2.PD != ML.c3.PD){as.character(c("c2", "c3")[which(min(ML.c2.PD, ML.c3.PD) != c(ML.c2.PD, ML.c3.PD))])} else{"same"}
  ML.c31.PDgreater <- if(ML.c3.PD != ML.c1.PD){as.character(c("c3", "c1")[which(min(ML.c3.PD, ML.c1.PD) != c(ML.c3.PD, ML.c1.PD))])} else{"same"}
  #ALPHA MNND  
  a <- HPDinterval(as.mcmc(as.numeric(mpnd.ML.c$c1.MNND)), prob=0.950000)
  b <- HPDinterval(as.mcmc(as.numeric(mpnd.ML.c$c2.MNND)), prob=0.950000)
  if(max(a[[1]],b[[1]]) <= min(a[[2]], b[[2]])){ML.c12.MNNDgreater <- "same"} else if(a[[1]] > b[[1]]){ML.c12.MNNDgreater <- "c1"} else if(a[[1]] < b[[1]]){ML.c12.MNNDgreater <- "c2"}
  a <- HPDinterval(as.mcmc(as.numeric(mpnd.ML.c$c2.MNND)), prob=0.950000)
  b <- HPDinterval(as.mcmc(as.numeric(mpnd.ML.c$c3.MNND)), prob=0.950000)
  if(max(a[[1]],b[[1]]) <= min(a[[2]], b[[2]])){ML.c23.MNNDgreater <- "same"} else if(a[[1]] > b[[1]]){ML.c23.MNNDgreater <- "c2"} else if(a[[1]] < b[[1]]){ML.c23.MNNDgreater <- "c3"}
  a <- HPDinterval(as.mcmc(as.numeric(mpnd.ML.c$c3.MNND)), prob=0.950000)
  b <- HPDinterval(as.mcmc(as.numeric(mpnd.ML.c$c1.MNND)), prob=0.950000)
  if(max(a[[1]],b[[1]]) <= min(a[[2]], b[[2]])){ML.c31.MNNDgreater <- "same"} else if(a[[1]] > b[[1]]){ML.c31.MNNDgreater <- "c3"} else if(a[[1]] < b[[1]]){ML.c31.MNNDgreater <- "c1"}
  #ALPHA MPD  
  a <- HPDinterval(as.mcmc(as.numeric(mpnd.ML.c$c1.MPD)), prob=0.950000)
  b <- HPDinterval(as.mcmc(as.numeric(mpnd.ML.c$c2.MPD)), prob=0.950000)
  if(max(a[[1]],b[[1]]) <= min(a[[2]], b[[2]])){ML.c12.MPDgreater <- "same"} else if(a[[1]] > b[[1]]){ML.c12.MPDgreater <- "c1"} else if(a[[1]] < b[[1]]){ML.c12.MPDgreater <- "c2"}
  a <- HPDinterval(as.mcmc(as.numeric(mpnd.ML.c$c2.MPD)), prob=0.950000)
  b <- HPDinterval(as.mcmc(as.numeric(mpnd.ML.c$c3.MPD)), prob=0.950000)
  if(max(a[[1]],b[[1]]) <= min(a[[2]], b[[2]])){ML.c23.MPDgreater <- "same"} else if(a[[1]] > b[[1]]){ML.c23.MPDgreater <- "c2"} else if(a[[1]] < b[[1]]){ML.c23.MPDgreater <- "c3"}
  a <- HPDinterval(as.mcmc(as.numeric(mpnd.ML.c$c3.MPD)), prob=0.950000)
  b <- HPDinterval(as.mcmc(as.numeric(mpnd.ML.c$c1.MPD)), prob=0.950000)
  if(max(a[[1]],b[[1]]) <= min(a[[2]], b[[2]])){ML.c31.MPDgreater <- "same"} else if(a[[1]] > b[[1]]){ML.c31.MPDgreater <- "c3"} else if(a[[1]] < b[[1]]){ML.c31.MPDgreater <- "c1"}

  # Get relationships from Master ML trees #
  #BETA MNND 
  a <- HPDinterval(as.mcmc(as.numeric(mpnd.master.ML.c$c12.MNND)), prob=0.950000)
  b <- HPDinterval(as.mcmc(as.numeric(mpnd.master.ML.c$c31.MNND)), prob=0.950000)
  if(max(a[[1]],b[[1]]) <= min(a[[2]], b[[2]])){master.ML.c1.MNNDcloser <- "same"} else if(a[[1]] > b[[1]]){master.ML.c1.MNNDcloser <- "c3"} else if(a[[1]] < b[[1]]){master.ML.c1.MNNDcloser <- "c2"}
  a <- HPDinterval(as.mcmc(as.numeric(mpnd.master.ML.c$c23.MNND)), prob=0.950000)
  b <- HPDinterval(as.mcmc(as.numeric(mpnd.master.ML.c$c12.MNND)), prob=0.950000)
  if(max(a[[1]],b[[1]]) <= min(a[[2]], b[[2]])){master.ML.c2.MNNDcloser <- "same"} else if(a[[1]] > b[[1]]){master.ML.c2.MNNDcloser <- "c1"} else if(a[[1]] < b[[1]]){master.ML.c2.MNNDcloser <- "c3"}
  a <- HPDinterval(as.mcmc(as.numeric(mpnd.master.ML.c$c31.MNND)), prob=0.950000)
  b <- HPDinterval(as.mcmc(as.numeric(mpnd.master.ML.c$c23.MNND)), prob=0.950000)
  if(max(a[[1]],b[[1]]) <= min(a[[2]], b[[2]])){master.ML.c3.MNNDcloser <- "same"} else if(a[[1]] > b[[1]]){master.ML.c3.MNNDcloser <- "c2"} else if(a[[1]] < b[[1]]){master.ML.c3.MNNDcloser <- "c1"}
  #BETA MNND 
  a <- HPDinterval(as.mcmc(as.numeric(mpnd.master.ML.c$c12.MPD)), prob=0.950000)
  b <- HPDinterval(as.mcmc(as.numeric(mpnd.master.ML.c$c31.MPD)), prob=0.950000)
  if(max(a[[1]],b[[1]]) <= min(a[[2]], b[[2]])){master.ML.c1.MPDcloser <- "same"} else if(a[[1]] > b[[1]]){master.ML.c1.MPDcloser <- "c3"} else if(a[[1]] < b[[1]]){master.ML.c1.MPDcloser <- "c2"}
  a <- HPDinterval(as.mcmc(as.numeric(mpnd.master.ML.c$c23.MPD)), prob=0.950000)
  b <- HPDinterval(as.mcmc(as.numeric(mpnd.master.ML.c$c12.MPD)), prob=0.950000)
  if(max(a[[1]],b[[1]]) <= min(a[[2]], b[[2]])){master.ML.c2.MPDcloser <- "same"} else if(a[[1]] > b[[1]]){master.ML.c2.MPDcloser <- "c1"} else if(a[[1]] < b[[1]]){master.ML.c2.MPDcloser <- "c3"}
  a <- HPDinterval(as.mcmc(as.numeric(mpnd.master.ML.c$c31.MPD)), prob=0.950000)
  b <- HPDinterval(as.mcmc(as.numeric(mpnd.master.ML.c$c23.MPD)), prob=0.950000)
  if(max(a[[1]],b[[1]]) <= min(a[[2]], b[[2]])){master.ML.c3.MPDcloser <- "same"} else if(a[[1]] > b[[1]]){master.ML.c3.MPDcloser <- "c2"} else if(a[[1]] < b[[1]]){master.ML.c3.MPDcloser <- "c1"}
  
  #ALPHA PD
  master.ML.c12.PDgreater <- if(master.ML.c1.PD != master.ML.c2.PD){as.character(c("c1", "c2")[which(min(master.ML.c1.PD, master.ML.c2.PD) != c(master.ML.c1.PD, master.ML.c2.PD))])} else{"same"}
  master.ML.c23.PDgreater <- if(master.ML.c2.PD != master.ML.c3.PD){as.character(c("c2", "c3")[which(min(master.ML.c2.PD, master.ML.c3.PD) != c(master.ML.c2.PD, master.ML.c3.PD))])} else{"same"}
  master.ML.c31.PDgreater <- if(master.ML.c3.PD != master.ML.c1.PD){as.character(c("c3", "c1")[which(min(master.ML.c3.PD, master.ML.c1.PD) != c(master.ML.c3.PD, master.ML.c1.PD))])} else{"same"}
  #ALPHA MNND  
  a <- HPDinterval(as.mcmc(as.numeric(mpnd.master.ML.c$c1.MNND)), prob=0.950000)
  b <- HPDinterval(as.mcmc(as.numeric(mpnd.master.ML.c$c2.MNND)), prob=0.950000)
  if(max(a[[1]],b[[1]]) <= min(a[[2]], b[[2]])){master.ML.c12.MNNDgreater <- "same"} else if(a[[1]] > b[[1]]){master.ML.c12.MNNDgreater <- "c1"} else if(a[[1]] < b[[1]]){master.ML.c12.MNNDgreater <- "c2"}
  a <- HPDinterval(as.mcmc(as.numeric(mpnd.master.ML.c$c2.MNND)), prob=0.950000)
  b <- HPDinterval(as.mcmc(as.numeric(mpnd.master.ML.c$c3.MNND)), prob=0.950000)
  if(max(a[[1]],b[[1]]) <= min(a[[2]], b[[2]])){master.ML.c23.MNNDgreater <- "same"} else if(a[[1]] > b[[1]]){master.ML.c23.MNNDgreater <- "c2"} else if(a[[1]] < b[[1]]){master.ML.c23.MNNDgreater <- "c3"}
  a <- HPDinterval(as.mcmc(as.numeric(mpnd.master.ML.c$c3.MNND)), prob=0.950000)
  b <- HPDinterval(as.mcmc(as.numeric(mpnd.master.ML.c$c1.MNND)), prob=0.950000)
  if(max(a[[1]],b[[1]]) <= min(a[[2]], b[[2]])){master.ML.c31.MNNDgreater <- "same"} else if(a[[1]] > b[[1]]){master.ML.c31.MNNDgreater <- "c3"} else if(a[[1]] < b[[1]]){master.ML.c31.MNNDgreater <- "c1"}
  #ALPHA MPD  
  a <- HPDinterval(as.mcmc(as.numeric(mpnd.master.ML.c$c1.MPD)), prob=0.950000)
  b <- HPDinterval(as.mcmc(as.numeric(mpnd.master.ML.c$c2.MPD)), prob=0.950000)
  if(max(a[[1]],b[[1]]) <= min(a[[2]], b[[2]])){master.ML.c12.MPDgreater <- "same"} else if(a[[1]] > b[[1]]){master.ML.c12.MPDgreater <- "c1"} else if(a[[1]] < b[[1]]){master.ML.c12.MPDgreater <- "c2"}
  a <- HPDinterval(as.mcmc(as.numeric(mpnd.master.ML.c$c2.MPD)), prob=0.950000)
  b <- HPDinterval(as.mcmc(as.numeric(mpnd.master.ML.c$c3.MPD)), prob=0.950000)
  if(max(a[[1]],b[[1]]) <= min(a[[2]], b[[2]])){master.ML.c23.MPDgreater <- "same"} else if(a[[1]] > b[[1]]){master.ML.c23.MPDgreater <- "c2"} else if(a[[1]] < b[[1]]){master.ML.c23.MPDgreater <- "c3"}
  a <- HPDinterval(as.mcmc(as.numeric(mpnd.master.ML.c$c3.MPD)), prob=0.950000)
  b <- HPDinterval(as.mcmc(as.numeric(mpnd.master.ML.c$c1.MPD)), prob=0.950000)
  if(max(a[[1]],b[[1]]) <= min(a[[2]], b[[2]])){master.ML.c31.MPDgreater <- "same"} else if(a[[1]] > b[[1]]){master.ML.c31.MPDgreater <- "c3"} else if(a[[1]] < b[[1]]){master.ML.c31.MPDgreater <- "c1"}

  # Get relationships from Community ML trees #
  #BETA MNND 
  a <- HPDinterval(as.mcmc(as.numeric(mpnd.community.ML.c$c12.MNND)), prob=0.950000)
  b <- HPDinterval(as.mcmc(as.numeric(mpnd.community.ML.c$c31.MNND)), prob=0.950000)
  if(max(a[[1]],b[[1]]) <= min(a[[2]], b[[2]])){community.ML.c1.MNNDcloser <- "same"} else if(a[[1]] > b[[1]]){community.ML.c1.MNNDcloser <- "c3"} else if(a[[1]] < b[[1]]){community.ML.c1.MNNDcloser <- "c2"}
  a <- HPDinterval(as.mcmc(as.numeric(mpnd.community.ML.c$c23.MNND)), prob=0.950000)
  b <- HPDinterval(as.mcmc(as.numeric(mpnd.community.ML.c$c12.MNND)), prob=0.950000)
  if(max(a[[1]],b[[1]]) <= min(a[[2]], b[[2]])){community.ML.c2.MNNDcloser <- "same"} else if(a[[1]] > b[[1]]){community.ML.c2.MNNDcloser <- "c1"} else if(a[[1]] < b[[1]]){community.ML.c2.MNNDcloser <- "c3"}
  a <- HPDinterval(as.mcmc(as.numeric(mpnd.community.ML.c$c31.MNND)), prob=0.950000)
  b <- HPDinterval(as.mcmc(as.numeric(mpnd.community.ML.c$c23.MNND)), prob=0.950000)
  if(max(a[[1]],b[[1]]) <= min(a[[2]], b[[2]])){community.ML.c3.MNNDcloser <- "same"} else if(a[[1]] > b[[1]]){community.ML.c3.MNNDcloser <- "c2"} else if(a[[1]] < b[[1]]){community.ML.c3.MNNDcloser <- "c1"}
  #BETA MPD 
  a <- HPDinterval(as.mcmc(as.numeric(mpnd.community.ML.c$c12.MPD)), prob=0.950000)
  b <- HPDinterval(as.mcmc(as.numeric(mpnd.community.ML.c$c31.MPD)), prob=0.950000)
  if(max(a[[1]],b[[1]]) <= min(a[[2]], b[[2]])){community.ML.c1.MPDcloser <- "same"} else if(a[[1]] > b[[1]]){community.ML.c1.MPDcloser <- "c3"} else if(a[[1]] < b[[1]]){community.ML.c1.MPDcloser <- "c2"}
  a <- HPDinterval(as.mcmc(as.numeric(mpnd.community.ML.c$c23.MPD)), prob=0.950000)
  b <- HPDinterval(as.mcmc(as.numeric(mpnd.community.ML.c$c12.MPD)), prob=0.950000)
  if(max(a[[1]],b[[1]]) <= min(a[[2]], b[[2]])){community.ML.c2.MPDcloser <- "same"} else if(a[[1]] > b[[1]]){community.ML.c2.MPDcloser <- "c1"} else if(a[[1]] < b[[1]]){community.ML.c2.MPDcloser <- "c3"}
  a <- HPDinterval(as.mcmc(as.numeric(mpnd.community.ML.c$c31.MPD)), prob=0.950000)
  b <- HPDinterval(as.mcmc(as.numeric(mpnd.community.ML.c$c23.MPD)), prob=0.950000)
  if(max(a[[1]],b[[1]]) <= min(a[[2]], b[[2]])){community.ML.c3.MPDcloser <- "same"} else if(a[[1]] > b[[1]]){community.ML.c3.MPDcloser <- "c2"} else if(a[[1]] < b[[1]]){community.ML.c3.MPDcloser <- "c1"}
  
  #ALPHA PD
  community.ML.c12.PDgreater <- if(community.ML.c1.PD != community.ML.c2.PD){as.character(c("c1", "c2")[which(min(community.ML.c1.PD, community.ML.c2.PD) != c(community.ML.c1.PD, community.ML.c2.PD))])} else{"same"}
  community.ML.c23.PDgreater <- if(community.ML.c2.PD != community.ML.c3.PD){as.character(c("c2", "c3")[which(min(community.ML.c2.PD, community.ML.c3.PD) != c(community.ML.c2.PD, community.ML.c3.PD))])} else{"same"}
  community.ML.c31.PDgreater <- if(community.ML.c3.PD != community.ML.c1.PD){as.character(c("c3", "c1")[which(min(community.ML.c3.PD, community.ML.c1.PD) != c(community.ML.c3.PD, community.ML.c1.PD))])} else{"same"}
  #ALPHA MNND  
  a <- HPDinterval(as.mcmc(as.numeric(mpnd.community.ML.c$c1.MNND)), prob=0.950000)
  b <- HPDinterval(as.mcmc(as.numeric(mpnd.community.ML.c$c2.MNND)), prob=0.950000)
  if(max(a[[1]],b[[1]]) <= min(a[[2]], b[[2]])){community.ML.c12.MNNDgreater <- "same"} else if(a[[1]] > b[[1]]){community.ML.c12.MNNDgreater <- "c1"} else if(a[[1]] < b[[1]]){community.ML.c12.MNNDgreater <- "c2"}
  a <- HPDinterval(as.mcmc(as.numeric(mpnd.community.ML.c$c2.MNND)), prob=0.950000)
  b <- HPDinterval(as.mcmc(as.numeric(mpnd.community.ML.c$c3.MNND)), prob=0.950000)
  if(max(a[[1]],b[[1]]) <= min(a[[2]], b[[2]])){community.ML.c23.MNNDgreater <- "same"} else if(a[[1]] > b[[1]]){community.ML.c23.MNNDgreater <- "c2"} else if(a[[1]] < b[[1]]){community.ML.c23.MNNDgreater <- "c3"}
  a <- HPDinterval(as.mcmc(as.numeric(mpnd.community.ML.c$c3.MNND)), prob=0.950000)
  b <- HPDinterval(as.mcmc(as.numeric(mpnd.community.ML.c$c1.MNND)), prob=0.950000)
  if(max(a[[1]],b[[1]]) <= min(a[[2]], b[[2]])){community.ML.c31.MNNDgreater <- "same"} else if(a[[1]] > b[[1]]){community.ML.c31.MNNDgreater <- "c3"} else if(a[[1]] < b[[1]]){community.ML.c31.MNNDgreater <- "c1"}
  #ALPHA MPD  
  a <- HPDinterval(as.mcmc(as.numeric(mpnd.community.ML.c$c1.MPD)), prob=0.950000)
  b <- HPDinterval(as.mcmc(as.numeric(mpnd.community.ML.c$c2.MPD)), prob=0.950000)
  if(max(a[[1]],b[[1]]) <= min(a[[2]], b[[2]])){community.ML.c12.MPDgreater <- "same"} else if(a[[1]] > b[[1]]){community.ML.c12.MPDgreater <- "c1"} else if(a[[1]] < b[[1]]){community.ML.c12.MPDgreater <- "c2"}
  a <- HPDinterval(as.mcmc(as.numeric(mpnd.community.ML.c$c2.MPD)), prob=0.950000)
  b <- HPDinterval(as.mcmc(as.numeric(mpnd.community.ML.c$c3.MPD)), prob=0.950000)
  if(max(a[[1]],b[[1]]) <= min(a[[2]], b[[2]])){community.ML.c23.MPDgreater <- "same"} else if(a[[1]] > b[[1]]){community.ML.c23.MPDgreater <- "c2"} else if(a[[1]] < b[[1]]){community.ML.c23.MPDgreater <- "c3"}
  a <- HPDinterval(as.mcmc(as.numeric(mpnd.community.ML.c$c3.MPD)), prob=0.950000)
  b <- HPDinterval(as.mcmc(as.numeric(mpnd.community.ML.c$c1.MPD)), prob=0.950000)
  if(max(a[[1]],b[[1]]) <= min(a[[2]], b[[2]])){community.ML.c31.MPDgreater <- "same"} else if(a[[1]] > b[[1]]){community.ML.c31.MPDgreater <- "c3"} else if(a[[1]] < b[[1]]){community.ML.c31.MPDgreater <- "c1"}

  #TRUE PD of entire species pool 
  c <- name.check(master.best, data.names = c(tree.com[[2]], tree.com[[3]], tree.com[[4]]))
  xx <- drop.tip(master.best, tip = c$tree_not_data[c$tree_not_data != outgroup])
  xx2 <- drop.tip(xx, tip = outgroup)
  Mbest_PD.c <- sum(xx2$edge.length) #PD of species pool on TRUE tree
  #Master PD of entire species pool
  c <- name.check(master.ML, data.names = c(tree.com[[2]], tree.com[[3]], tree.com[[4]]))
  xx <- drop.tip(master.ML, tip = c$tree_not_data[c$tree_not_data != outgroup])
  xx2 <- drop.tip(xx, tip = outgroup)
  master.ML_PD.c <- sum(xx2$edge.length) #PD of species pool on Master tree
  #Community PD of entire species pool 
  c <- name.check(community.ML, data.names = c(tree.com[[2]], tree.com[[3]], tree.com[[4]]))
  xx <- drop.tip(community.ML, tip = c$tree_not_data[c$tree_not_data != outgroup])
  xx2 <- drop.tip(xx, tip = outgroup)
  community.ML_PD.c <- sum(xx2$edge.length) #PD of species pool on TRUE tree

    out1 <- list(
    # Overall PD of Entire Species Pool #
    MPD.comm = tree.com[[5]],
    PD.comm = tree.com[6][[1]], # Ratio of Faith's PD of entire community on Master tree
    TRUE_PD.c = Mbest_PD.c,
    master.ML_PD.c =  master.ML_PD.c,     ## PERCENTAGE of Correct ML Trees Analysis (master ML) ##
    community.ML_PD.c = community.ML_PD.c,     ## PERCENTAGE of Correct ML Trees Analysis (community ML) ##
    
    # Overall MPD / MNND of Entire Species Pool #
    TRUE_MPD.c = round(mean(mpnd.MLall[[1]]), 4),
    master.ML_MPD.c = round(mean(mpnd.master.MLall[[1]]), 4),  ## PERCENTAGE of Correct ML Trees Analysis (master ML) ##
    community.ML_MPD.c = round(mean(mpnd.community.MLall[[1]]), 4),  ## PERCENTAGE of Correct ML Trees Analysis (community ML) ##
    TRUE_MNND.c = round(mean(mpnd.MLall[[2]]), 4),
    master.ML_MNND.c = round(mean(mpnd.master.MLall[[2]]), 4),  ## PERCENTAGE of Correct ML Trees Analysis (master ML) ##
    community.ML_MNND.c = round(mean(mpnd.community.MLall[[2]]), 4),  ## PERCENTAGE of Correct ML Trees Analysis (community ML) ##
   
    #TRUE tree [MNND]
    ML.c12.MNND = round(mean(as.numeric(as.character(mpnd.ML.c$c12.MNND))), 4),
    ML.c23.MNND = round(mean(as.numeric(as.character(mpnd.ML.c$c23.MNND))), 4),
    ML.c31.MNND = round(mean(as.numeric(as.character(mpnd.ML.c$c31.MNND))), 4),
    #Community ML tree [MNND]
    community.ML.c12.MNND = round(mean(as.numeric(as.character(mpnd.community.ML.c$c12.MNND))), 4),
    community.ML.c23.MNND = round(mean(as.numeric(as.character(mpnd.community.ML.c$c23.MNND))), 4),
    community.ML.c31.MNND = round(mean(as.numeric(as.character(mpnd.community.ML.c$c31.MNND))), 4),
    #Master ML tree [MNND]
    master.ML.c12.MNND = round(mean(as.numeric(as.character(mpnd.master.ML.c$c12.MNND))), 4),
    master.ML.c23.MNND = round(mean(as.numeric(as.character(mpnd.master.ML.c$c23.MNND))), 4),
    master.ML.c31.MNND = round(mean(as.numeric(as.character(mpnd.master.ML.c$c31.MNND))), 4),
    
    #TRUE tree [MPD]
    ML.c12.MPD = round(mean(as.numeric(as.character(mpnd.ML.c$c12.MPD))), 4),
    ML.c23.MPD = round(mean(as.numeric(as.character(mpnd.ML.c$c23.MPD))), 4),
    ML.c31.MPD = round(mean(as.numeric(as.character(mpnd.ML.c$c31.MPD))), 4),
    #Community ML tree [MPD]
    community.ML.c12.MPD = round(mean(as.numeric(as.character(mpnd.community.ML.c$c12.MPD))), 4),
    community.ML.c23.MPD = round(mean(as.numeric(as.character(mpnd.community.ML.c$c23.MPD))), 4),
    community.ML.c31.MPD = round(mean(as.numeric(as.character(mpnd.community.ML.c$c31.MPD))), 4),
    #Master ML tree [MPD]
    master.ML.c12.MPD = round(mean(as.numeric(as.character(mpnd.master.ML.c$c12.MPD))), 4),
    master.ML.c23.MPD = round(mean(as.numeric(as.character(mpnd.master.ML.c$c23.MPD))), 4),
    master.ML.c31.MPD = round(mean(as.numeric(as.character(mpnd.master.ML.c$c31.MPD))), 4),

    #TRUE tree [MNND]
    ML.c1.MNNDcloser = ML.c1.MNNDcloser,
    ML.c2.MNNDcloser = ML.c2.MNNDcloser,
    ML.c3.MNNDcloser = ML.c3.MNNDcloser,
    #master ML tree [MNND]
    master.ML.c1.MNNDcloser = master.ML.c1.MNNDcloser,
    master.ML.c2.MNNDcloser = master.ML.c2.MNNDcloser,
    master.ML.c3.MNNDcloser = master.ML.c3.MNNDcloser,
    #community ML tree [MNND]
    community.ML.c1.MNNDcloser = community.ML.c1.MNNDcloser,
    community.ML.c2.MNNDcloser = community.ML.c2.MNNDcloser,
    community.ML.c3.MNNDcloser = community.ML.c3.MNNDcloser,

    #TRUE tree [MPD]
    ML.c1.MPDcloser = ML.c1.MPDcloser,
    ML.c2.MPDcloser = ML.c2.MPDcloser,
    ML.c3.MPDcloser = ML.c3.MPDcloser,
    #master ML tree [MPD]
    master.ML.c1.MPDcloser = master.ML.c1.MPDcloser,
    master.ML.c2.MPDcloser = master.ML.c2.MPDcloser,
    master.ML.c3.MPDcloser = master.ML.c3.MPDcloser,
    #community ML tree [MPD]
    community.ML.c1.MPDcloser = community.ML.c1.MPDcloser,
    community.ML.c2.MPDcloser = community.ML.c2.MPDcloser,
    community.ML.c3.MPDcloser = community.ML.c3.MPDcloser,
    
    ##############################################
    ## TRUE vs Master ML & TRUE vc Community ML ##
    ##############################################
    
    # Summing up the Discrepancies by c1 c2 c3 [MNND]
    #Community ML Trees
    community.ML.closer.c1_beta_error.MNND = length(which(ML.c1.MNNDcloser != community.ML.c1.MNNDcloser)),
    community.ML.closer.c2_beta_error.MNND = length(which(ML.c2.MNNDcloser != community.ML.c2.MNNDcloser)),
    community.ML.closer.c3_beta_error.MNND = length(which(ML.c3.MNNDcloser != community.ML.c3.MNNDcloser)),
    #Master ML Trees
    master.ML.closer.Mc1_beta_error.MNND = length(which(ML.c1.MNNDcloser != master.ML.c1.MNNDcloser)),
    master.ML.closer.Mc2_beta_error.MNND = length(which(ML.c2.MNNDcloser != master.ML.c2.MNNDcloser)),
    master.ML.closer.Mc3_beta_error.MNND = length(which(ML.c3.MNNDcloser != master.ML.c3.MNNDcloser)),
    
    # Summing up the Discrepancies by c1 c2 c3 [MPD]
    #Community ML Trees
    community.ML.closer.c1_beta_error.MPD = length(which(ML.c1.MPDcloser != community.ML.c1.MPDcloser)),
    community.ML.closer.c2_beta_error.MPD = length(which(ML.c2.MPDcloser != community.ML.c2.MPDcloser)),
    community.ML.closer.c3_beta_error.MPD = length(which(ML.c3.MPDcloser != community.ML.c3.MPDcloser)),
    #Master ML Trees
    master.ML.closer.Mc1_beta_error.MPD = length(which(ML.c1.MPDcloser != master.ML.c1.MPDcloser)),
    master.ML.closer.Mc2_beta_error.MPD = length(which(ML.c2.MPDcloser != master.ML.c2.MPDcloser)),
    master.ML.closer.Mc3_beta_error.MPD = length(which(ML.c3.MPDcloser != master.ML.c3.MPDcloser)),
    
    # Summing up the Discrepancies by Error Type [MNND] #
    #Community ML Trees
    community.ML.beta_error_typeS.MNND.c = length(which(community.ML.c1.MNNDcloser != "same" & ML.c1.MNNDcloser != "same" & community.ML.c1.MNNDcloser != ML.c1.MNNDcloser)) +
      length(which(community.ML.c2.MNNDcloser != "same" & ML.c2.MNNDcloser != "same" & community.ML.c2.MNNDcloser != ML.c2.MNNDcloser)) +
      length(which(community.ML.c3.MNNDcloser != "same" & ML.c3.MNNDcloser != "same" & community.ML.c3.MNNDcloser != ML.c3.MNNDcloser)),
    community.ML.beta_error_type1.MNND.c = length(which(ML.c1.MNNDcloser == "same" & community.ML.c1.MNNDcloser != "same")) +
      length(which(ML.c2.MNNDcloser == "same" & community.ML.c2.MNNDcloser != "same")) +
      length(which(ML.c3.MNNDcloser == "same" & community.ML.c3.MNNDcloser != "same")),
    community.ML.beta_error_type2.MNND.c = length(which(ML.c1.MNNDcloser != "same" & community.ML.c1.MNNDcloser == "same")) +
      length(which(ML.c2.MNNDcloser != "same" & community.ML.c2.MNNDcloser == "same")) +
      length(which(ML.c3.MNNDcloser != "same" & community.ML.c3.MNNDcloser == "same")),
    community.ML.beta_error_total_error.MNND.c = length(which(community.ML.c1.MNNDcloser != ML.c1.MNNDcloser)) + 
      length(which(community.ML.c2.MNNDcloser != ML.c2.MNNDcloser)) + 
      length(which(community.ML.c3.MNNDcloser != ML.c3.MNNDcloser)),
    
    #Master ML Trees
    master.ML.beta_error_typeS.MNND.m = length(which(master.ML.c1.MNNDcloser != "same" & ML.c1.MNNDcloser != "same" & master.ML.c1.MNNDcloser != ML.c1.MNNDcloser)) +
      length(which(master.ML.c2.MNNDcloser != "same" & ML.c2.MNNDcloser != "same" & master.ML.c2.MNNDcloser != ML.c2.MNNDcloser)) +
      length(which(master.ML.c3.MNNDcloser != "same" & ML.c3.MNNDcloser != "same" & master.ML.c3.MNNDcloser != ML.c3.MNNDcloser)),
    master.ML.beta_error_type1.MNND.m = length(which(ML.c1.MNNDcloser == "same" & master.ML.c1.MNNDcloser != "same")) +
      length(which(ML.c2.MNNDcloser == "same" & master.ML.c2.MNNDcloser != "same")) +
      length(which(ML.c3.MNNDcloser == "same" & master.ML.c3.MNNDcloser != "same")),
    master.ML.beta_error_type2.MNND.m = length(which(ML.c1.MNNDcloser != "same" & master.ML.c1.MNNDcloser == "same")) +
      length(which(ML.c2.MNNDcloser != "same" & master.ML.c2.MNNDcloser == "same")) +
      length(which(ML.c3.MNNDcloser != "same" & master.ML.c3.MNNDcloser == "same")),
    master.ML.beta_error_total_error.MNND.m = length(which(master.ML.c1.MNNDcloser != ML.c1.MNNDcloser)) + 
      length(which(master.ML.c2.MNNDcloser != ML.c2.MNNDcloser)) + 
      length(which(master.ML.c3.MNNDcloser != ML.c3.MNNDcloser)),
    
    # Summing up the Discrepancies by Error Type [MPD] #
    #Community Bootstrap Trees
    community.ML.beta_error_typeS.MPD.c = length(which(community.ML.c1.MPDcloser != "same" & ML.c1.MPDcloser != "same" & community.ML.c1.MPDcloser != ML.c1.MPDcloser)) +
      length(which(community.ML.c2.MPDcloser != "same" & ML.c2.MPDcloser != "same" & community.ML.c2.MPDcloser != ML.c2.MPDcloser)) +
      length(which(community.ML.c3.MPDcloser != "same" & ML.c3.MPDcloser != "same" & community.ML.c3.MPDcloser != ML.c3.MPDcloser)),
    community.ML.beta_error_type1.MPD.c = length(which(ML.c1.MPDcloser == "same" & community.ML.c1.MPDcloser != "same")) +
      length(which(ML.c2.MPDcloser == "same" & community.ML.c2.MPDcloser != "same")) +
      length(which(ML.c3.MPDcloser == "same" & community.ML.c3.MPDcloser != "same")),
    community.ML.beta_error_type2.MPD.c = length(which(ML.c1.MPDcloser != "same" & community.ML.c1.MPDcloser == "same")) +
      length(which(ML.c2.MPDcloser != "same" & community.ML.c2.MPDcloser == "same")) +
      length(which(ML.c3.MPDcloser != "same" & community.ML.c3.MPDcloser == "same")),
    community.ML.beta_error_total_error.MPD.c = length(which(community.ML.c1.MPDcloser != ML.c1.MPDcloser)) + 
      length(which(community.ML.c2.MPDcloser != ML.c2.MPDcloser)) + 
      length(which(community.ML.c3.MPDcloser != ML.c3.MPDcloser)),
    
    #Master Bootstrap Trees
    master.ML.beta_error_typeS.MPD.m = length(which(master.ML.c1.MPDcloser != "same" & ML.c1.MPDcloser != "same" & master.ML.c1.MPDcloser != ML.c1.MPDcloser)) +
      length(which(master.ML.c2.MPDcloser != "same" & ML.c2.MPDcloser != "same" & master.ML.c2.MPDcloser != ML.c2.MPDcloser)) +
      length(which(master.ML.c3.MPDcloser != "same" & ML.c3.MPDcloser != "same" & master.ML.c3.MPDcloser != ML.c3.MPDcloser)),
    master.ML.beta_error_type1.MPD.m = length(which(ML.c1.MPDcloser == "same" & master.ML.c1.MPDcloser != "same")) +
      length(which(ML.c2.MPDcloser == "same" & master.ML.c2.MPDcloser != "same")) +
      length(which(ML.c3.MPDcloser == "same" & master.ML.c3.MPDcloser != "same")),
    master.ML.beta_error_type2.MPD.m = length(which(ML.c1.MPDcloser != "same" & master.ML.c1.MPDcloser == "same")) +
      length(which(ML.c2.MPDcloser != "same" & master.ML.c2.MPDcloser == "same")) +
      length(which(ML.c3.MPDcloser != "same" & master.ML.c3.MPDcloser == "same")),
    master.ML.beta_error_total_error.MPD.m = length(which(master.ML.c1.MPDcloser != ML.c1.MPDcloser)) + 
      length(which(master.ML.c2.MPDcloser != ML.c2.MPDcloser)) + 
      length(which(master.ML.c3.MPDcloser != ML.c3.MPDcloser)),
    
    
    ###########################
    # ALPHA Diversity Metrics #
    ###########################
    
    # TRUE tree [PD]
    ML.c1.PD = round(ML.c1.PD, 4),
    ML.c2.PD = round(ML.c2.PD, 4),
    ML.c3.PD = round(ML.c3.PD, 4),
    # Community ML Tree [PD]
    community.ML.c1.PD = round(community.ML.c1.PD, 4),
    community.ML.c2.PD = round(community.ML.c2.PD, 4),
    community.ML.c3.PD = round(community.ML.c3.PD, 4),
    # Master ML Tree [PD]
    master.ML.c1.PD = round(master.ML.c1.PD, 4),
    master.ML.c2.PD = round(master.ML.c2.PD, 4),
    master.ML.c3.PD = round(master.ML.c3.PD, 4),
    
    #TRUE tree [MNND]
    ML.c1.MNND = round(mean(as.numeric(as.character(mpnd.ML.c$c1.MNND))), 4),
    ML.c2.MNND = round(mean(as.numeric(as.character(mpnd.ML.c$c2.MNND))), 4),
    ML.c3.MNND = round(mean(as.numeric(as.character(mpnd.ML.c$c3.MNND))), 4),
    #Community ML tree [MNND]
    community.ML.c1.MNND = round(mean(as.numeric(as.character(mpnd.community.ML.c$c1.MNND))), 4),
    community.ML.c2.MNND = round(mean(as.numeric(as.character(mpnd.community.ML.c$c2.MNND))), 4),
    community.ML.c3.MNND = round(mean(as.numeric(as.character(mpnd.community.ML.c$c3.MNND))), 4),
    #Master ML tree [MNND]
    master.ML.c1.MNND = round(mean(as.numeric(as.character(mpnd.master.ML.c$c1.MNND))), 4),
    master.ML.c2.MNND = round(mean(as.numeric(as.character(mpnd.master.ML.c$c2.MNND))), 4),
    master.ML.c3.MNND = round(mean(as.numeric(as.character(mpnd.master.ML.c$c3.MNND))), 4),

    #TRUE tree [MPD]
    ML.c1.MPD = round(mean(as.numeric(as.character(mpnd.ML.c$c1.MPD))), 4),
    ML.c2.MPD = round(mean(as.numeric(as.character(mpnd.ML.c$c2.MPD))), 4),
    ML.c3.MPD = round(mean(as.numeric(as.character(mpnd.ML.c$c3.MPD))), 4),
    #Community ML tree [MPD]
    community.ML.c1.MPD = round(mean(as.numeric(as.character(mpnd.community.ML.c$c1.MPD))), 4),
    community.ML.c2.MPD = round(mean(as.numeric(as.character(mpnd.community.ML.c$c2.MPD))), 4),
    community.ML.c3.MPD = round(mean(as.numeric(as.character(mpnd.community.ML.c$c3.MPD))), 4),
    #Master ML tree [MPD]
    master.ML.c1.MPD = round(mean(as.numeric(as.character(mpnd.master.ML.c$c1.MPD))), 4),
    master.ML.c2.MPD = round(mean(as.numeric(as.character(mpnd.master.ML.c$c2.MPD))), 4),
    master.ML.c3.MPD = round(mean(as.numeric(as.character(mpnd.master.ML.c$c3.MPD))), 4),
 
    ###############################
    # ALPHA Diversity Comparisons #
    ###############################
    
    # TRUE tree [MNND]
    ML.c12.MNNDgreater = ML.c12.MNNDgreater,
    ML.c23.MNNDgreater = ML.c23.MNNDgreater,
    ML.c31.MNNDgreater = ML.c31.MNNDgreater,
    #master ML tree [MNND]
    master.ML.c12.MNNDgreater = master.ML.c12.MNNDgreater,
    master.ML.c23.MNNDgreater = master.ML.c23.MNNDgreater,
    master.ML.c31.MNNDgreater = master.ML.c31.MNNDgreater,
    #community ML tree [MNND]
    community.ML.c12.MNNDgreater = community.ML.c12.MNNDgreater,
    community.ML.c23.MNNDgreater = community.ML.c23.MNNDgreater,
    community.ML.c31.MNNDgreater = community.ML.c31.MNNDgreater,
   
    # TRUE tree [MPD]
    ML.c12.MPDgreater = ML.c12.MPDgreater,
    ML.c23.MPDgreater = ML.c23.MPDgreater,
    ML.c31.MPDgreater = ML.c31.MPDgreater,
    #master ML tree [MPD]
    master.ML.c12.MPDgreater = master.ML.c12.MPDgreater,
    master.ML.c23.MPDgreater = master.ML.c23.MPDgreater,
    master.ML.c31.MPDgreater = master.ML.c31.MPDgreater,
    #community ML tree [MPD]
    community.ML.c12.MPDgreater = community.ML.c12.MPDgreater,
    community.ML.c23.MPDgreater = community.ML.c23.MPDgreater,
    community.ML.c31.MPDgreater = community.ML.c31.MPDgreater,
   
    # TRUE tree [PD]
    ML.c12.PDgreater = ML.c12.PDgreater,
    ML.c23.PDgreater = ML.c23.PDgreater,
    ML.c31.PDgreater = ML.c31.PDgreater,
    #master ML tree [PD]
    master.ML.c12.PDgreater = master.ML.c12.PDgreater,
    master.ML.c23.PDgreater = master.ML.c23.PDgreater,
    master.ML.c31.PDgreater = master.ML.c31.PDgreater,
    #community ML tree [PD]
    community.ML.c12.PDgreater = community.ML.c12.PDgreater,
    community.ML.c23.PDgreater = community.ML.c23.PDgreater,
    community.ML.c31.PDgreater = community.ML.c31.PDgreater,
   
    ##############################################
    ## TRUE vs Master ML & TRUE vc Community ML ##
    ##############################################
    
    # Summing up the Discrepancies by c12 c23 c31 [MNND] #
    #Community ML Trees
    community.ML.greater.c12_alpha_error.MNND = length(which(ML.c12.MNNDgreater != community.ML.c12.MNNDgreater)),
    community.ML.greater.c23_alpha_error.MNND = length(which(ML.c23.MNNDgreater != community.ML.c23.MNNDgreater)),
    community.ML.greater.c31_alpha_error.MNND = length(which(ML.c31.MNNDgreater != community.ML.c31.MNNDgreater)),
    #Master ML Trees
    master.ML.greater.Mc12_alpha_error.MNND = length(which(ML.c12.MNNDgreater != master.ML.c12.MNNDgreater)),
    master.ML.greater.Mc23_alpha_error.MNND = length(which(ML.c23.MNNDgreater != master.ML.c23.MNNDgreater)),
    master.ML.greater.Mc31_alpha_error.MNND = length(which(ML.c31.MNNDgreater != master.ML.c31.MNNDgreater)),

    # Summing up the Discrepancies by c12 c23 c31 [MPD] #
    #Community ML Trees
    community.ML.greater.c12_alpha_error.MPD = length(which(ML.c12.MPDgreater != community.ML.c12.MPDgreater)),
    community.ML.greater.c23_alpha_error.MPD = length(which(ML.c23.MPDgreater != community.ML.c23.MPDgreater)),
    community.ML.greater.c31_alpha_error.MPD = length(which(ML.c31.MPDgreater != community.ML.c31.MPDgreater)),
    #Master Bootstrap Trees
    master.ML.greater.Mc12_alpha_error.MPD = length(which(ML.c12.MPDgreater != master.ML.c12.MPDgreater)),
    master.ML.greater.Mc23_alpha_error.MPD = length(which(ML.c23.MPDgreater != master.ML.c23.MPDgreater)),
    master.ML.greater.Mc31_alpha_error.MPD = length(which(ML.c31.MPDgreater != master.ML.c31.MPDgreater)),
    
    # Summing up the Discrepancies by c12 c23 c31 [PD] #
    #Community ML Trees
    community.ML.greater.c12_alpha_error.PD = length(which(ML.c12.PDgreater != community.ML.c12.PDgreater)),
    community.ML.greater.c23_alpha_error.PD = length(which(ML.c23.PDgreater != community.ML.c23.PDgreater)),
    community.ML.greater.c31_alpha_error.PD = length(which(ML.c31.PDgreater != community.ML.c31.PDgreater)),
    #Master Bootstrap Trees
    master.ML.greater.Mc12_alpha_error.PD = length(which(ML.c12.PDgreater != master.ML.c12.PDgreater)),
    master.ML.greater.Mc23_alpha_error.PD = length(which(ML.c23.PDgreater != master.ML.c23.PDgreater)),
    master.ML.greater.Mc31_alpha_error.PD = length(which(ML.c31.PDgreater != master.ML.c31.PDgreater)),
    
    # Summing up the Discrepancies by Error Type [MNND] #
    #Community ML Trees
    community.ML.alpha_error_typeS.MNND.c = length(which(community.ML.c12.MNNDgreater != "same" & ML.c12.MNNDgreater != "same" & community.ML.c12.MNNDgreater != ML.c12.MNNDgreater)) +
      length(which(community.ML.c23.MNNDgreater != "same" & ML.c23.MNNDgreater != "same" & community.ML.c23.MNNDgreater != ML.c23.MNNDgreater)) +
      length(which(community.ML.c31.MNNDgreater != "same" & ML.c31.MNNDgreater != "same" & community.ML.c31.MNNDgreater != ML.c31.MNNDgreater)),
    community.ML.alpha_error_type1.MNND.c = length(which(ML.c12.MNNDgreater == "same" & community.ML.c12.MNNDgreater != "same")) +
      length(which(ML.c23.MNNDgreater == "same" & community.ML.c23.MNNDgreater != "same")) +
      length(which(ML.c31.MNNDgreater == "same" & community.ML.c31.MNNDgreater != "same")),
    community.ML.alpha_error_type2.MNND.c = length(which(ML.c12.MNNDgreater != "same" & community.ML.c12.MNNDgreater == "same")) +
      length(which(ML.c23.MNNDgreater != "same" & community.ML.c23.MNNDgreater == "same")) +
      length(which(ML.c31.MNNDgreater != "same" & community.ML.c31.MNNDgreater == "same")),
    community.ML.alpha_error_total_error.MNND.c = length(which(community.ML.c12.MNNDgreater != ML.c12.MNNDgreater)) + 
      length(which(community.ML.c23.MNNDgreater != ML.c23.MNNDgreater)) + 
      length(which(community.ML.c31.MNNDgreater != ML.c31.MNNDgreater)),
    #Master ML Trees
    master.ML.alpha_error_typeS.MNND.m = length(which(master.ML.c12.MNNDgreater != "same" & ML.c12.MNNDgreater != "same" & master.ML.c12.MNNDgreater != ML.c12.MNNDgreater)) +
      length(which(master.ML.c23.MNNDgreater != "same" & ML.c23.MNNDgreater != "same" & master.ML.c23.MNNDgreater != ML.c23.MNNDgreater)) +
      length(which(master.ML.c31.MNNDgreater != "same" & ML.c31.MNNDgreater != "same" & master.ML.c31.MNNDgreater != ML.c31.MNNDgreater)),
    master.ML.alpha_error_type1.MNND.m = length(which(ML.c12.MNNDgreater == "same" & master.ML.c12.MNNDgreater != "same")) +
      length(which(ML.c23.MNNDgreater == "same" & master.ML.c23.MNNDgreater != "same")) +
      length(which(ML.c31.MNNDgreater == "same" & master.ML.c31.MNNDgreater != "same")),
    master.ML.alpha_error_type2.MNND.m = length(which(ML.c12.MNNDgreater != "same" & master.ML.c12.MNNDgreater == "same")) +
      length(which(ML.c23.MNNDgreater != "same" & master.ML.c23.MNNDgreater == "same")) +
      length(which(ML.c31.MNNDgreater != "same" & master.ML.c31.MNNDgreater == "same")),
    master.ML.alpha_error_total_error.MNND.m = length(which(master.ML.c12.MNNDgreater != ML.c12.MNNDgreater)) + 
      length(which(master.ML.c23.MNNDgreater != ML.c23.MNNDgreater)) + 
      length(which(master.ML.c31.MNNDgreater != ML.c31.MNNDgreater)),
    
    # Summing up the Discrepancies by Error Type [MPD] #
    #Community ML Trees
    community.ML.alpha_error_typeS.MPD.c = length(which(community.ML.c12.MPDgreater != "same" & ML.c12.MPDgreater != "same" & community.ML.c12.MPDgreater != ML.c12.MPDgreater)) +
      length(which(community.ML.c23.MPDgreater != "same" & ML.c23.MPDgreater != "same" & community.ML.c23.MPDgreater != ML.c23.MPDgreater)) +
      length(which(community.ML.c31.MPDgreater != "same" & ML.c31.MPDgreater != "same" & community.ML.c31.MPDgreater != ML.c31.MPDgreater)),
    community.ML.alpha_error_type1.MPD.c = length(which(ML.c12.MPDgreater == "same" & community.ML.c12.MPDgreater != "same")) +
      length(which(ML.c23.MPDgreater == "same" & community.ML.c23.MPDgreater != "same")) +
      length(which(ML.c31.MPDgreater == "same" & community.ML.c31.MPDgreater != "same")),
    community.ML.alpha_error_type2.MPD.c = length(which(ML.c12.MPDgreater != "same" & community.ML.c12.MPDgreater == "same")) +
      length(which(ML.c23.MPDgreater != "same" & community.ML.c23.MPDgreater == "same")) +
      length(which(ML.c31.MPDgreater != "same" & community.ML.c31.MPDgreater == "same")),
    community.ML.alpha_error_total_error.MPD.c = length(which(community.ML.c12.MPDgreater != ML.c12.MPDgreater)) + 
      length(which(community.ML.c23.MPDgreater != ML.c23.MPDgreater)) + 
      length(which(community.ML.c31.MPDgreater != ML.c31.MPDgreater)),
    #Master ML Trees
    master.ML.alpha_error_typeS.MPD.m = length(which(master.ML.c12.MPDgreater != "same" & ML.c12.MPDgreater != "same" & master.ML.c12.MPDgreater != ML.c12.MPDgreater)) +
      length(which(master.ML.c23.MPDgreater != "same" & ML.c23.MPDgreater != "same" & master.ML.c23.MPDgreater != ML.c23.MPDgreater)) +
      length(which(master.ML.c31.MPDgreater != "same" & ML.c31.MPDgreater != "same" & master.ML.c31.MPDgreater != ML.c31.MPDgreater)),
    master.ML.alpha_error_type1.MPD.m = length(which(ML.c12.MPDgreater == "same" & master.ML.c12.MPDgreater != "same")) +
      length(which(ML.c23.MPDgreater == "same" & master.ML.c23.MPDgreater != "same")) +
      length(which(ML.c31.MPDgreater == "same" & master.ML.c31.MPDgreater != "same")),
    master.ML.alpha_error_type2.MPD.m = length(which(ML.c12.MPDgreater != "same" & master.ML.c12.MPDgreater == "same")) +
      length(which(ML.c23.MPDgreater != "same" & master.ML.c23.MPDgreater == "same")) +
      length(which(ML.c31.MPDgreater != "same" & master.ML.c31.MPDgreater == "same")),
    master.ML.alpha_error_total_error.MPD.m = length(which(master.ML.c12.MPDgreater != ML.c12.MPDgreater)) + 
      length(which(master.ML.c23.MPDgreater != ML.c23.MPDgreater)) + 
      length(which(master.ML.c31.MPDgreater != ML.c31.MPDgreater)),
    
    # Summing up the Discrepancies by Error Type [PD] #
    #Community ML Trees
    community.ML.alpha_error_typeS.PD.c = length(which(community.ML.c12.PDgreater != "same" & ML.c12.PDgreater != "same" & community.ML.c12.PDgreater != ML.c12.PDgreater)) +
      length(which(community.ML.c23.PDgreater != "same" & ML.c23.PDgreater != "same" & community.ML.c23.PDgreater != ML.c23.PDgreater)) +
      length(which(community.ML.c31.PDgreater != "same" & ML.c31.PDgreater != "same" & community.ML.c31.PDgreater != ML.c31.PDgreater)),
    community.ML.alpha_error_type1.PD.c = length(which(ML.c12.PDgreater == "same" & community.ML.c12.PDgreater != "same")) +
      length(which(ML.c23.PDgreater == "same" & community.ML.c23.PDgreater != "same")) +
      length(which(ML.c31.PDgreater == "same" & community.ML.c31.PDgreater != "same")),
    community.ML.alpha_error_type2.PD.c = length(which(ML.c12.PDgreater != "same" & community.ML.c12.PDgreater == "same")) +
      length(which(ML.c23.PDgreater != "same" & community.ML.c23.PDgreater == "same")) +
      length(which(ML.c31.PDgreater != "same" & community.ML.c31.PDgreater == "same")),
    community.ML.alpha_error_total_error.PD.c = length(which(community.ML.c12.PDgreater != ML.c12.PDgreater)) + 
      length(which(community.ML.c23.PDgreater != ML.c23.PDgreater)) + 
      length(which(community.ML.c31.PDgreater != ML.c31.PDgreater)),
    #Master ML Trees
    master.ML.alpha_error_typeS.PD.m = length(which(master.ML.c12.PDgreater != "same" & ML.c12.PDgreater != "same" & master.ML.c12.PDgreater != ML.c12.PDgreater)) +
      length(which(master.ML.c23.PDgreater != "same" & ML.c23.PDgreater != "same" & master.ML.c23.PDgreater != ML.c23.PDgreater)) +
      length(which(master.ML.c31.PDgreater != "same" & ML.c31.PDgreater != "same" & master.ML.c31.PDgreater != ML.c31.PDgreater)),
    master.ML.alpha_error_type1.PD.m = length(which(ML.c12.PDgreater == "same" & master.ML.c12.PDgreater != "same")) +
      length(which(ML.c23.PDgreater == "same" & master.ML.c23.PDgreater != "same")) +
      length(which(ML.c31.PDgreater == "same" & master.ML.c31.PDgreater != "same")),
    master.ML.alpha_error_type2.PD.m = length(which(ML.c12.PDgreater != "same" & master.ML.c12.PDgreater == "same")) +
      length(which(ML.c23.PDgreater != "same" & master.ML.c23.PDgreater == "same")) +
      length(which(ML.c31.PDgreater != "same" & master.ML.c31.PDgreater == "same")),
    master.ML.alpha_error_total_error.PD.m = length(which(master.ML.c12.PDgreater != ML.c12.PDgreater)) + 
      length(which(master.ML.c23.PDgreater != ML.c23.PDgreater)) + 
      length(which(master.ML.c31.PDgreater != ML.c31.PDgreater))
)
  out <- rbind(out, out1)  
  #print(i)
  }
setwd(cfolders[[k]])
del <- list.files(path=cfolders[[k]], pattern='tre') #list community trees
file.remove(del)
write.csv(out, paste(ds, dating, "output_interval.csv", sep = "_"))
print(Sys.time())
print(k)
}
print(dating)  
}  

## Summarize ##  
setwd("..")
setwd("..")
wd2 <- getwd()
cfolders <- list.dirs(path = wd2, full.names = TRUE, recursive = TRUE)
cfolders <- cfolders[!grepl("MACOSX", cfolders)]

cfoldersR <- cfolders[grepl("r8s", cfolders)] #select r8s folders
cfoldersP <- cfolders[grepl("PATHd8", cfolders)] #select PATHd8s folders
cfoldersX <- cfolders[grepl("RAxML", cfolders)] #select RAxML folders
cfoldersB <- cfolders[grepl("BLADJ", cfolders)] #select BLADJ folders
cfolders <- c(cfoldersR, cfoldersP, cfoldersB, cfoldersX)

summ.summ <- NULL
col.name <- c()
for (i in 1:length(cfolders)) {
  setwd(cfolders[[i]])
  
  cnames <- as.data.frame(cfolders[[i]])
  cnames <- data.frame(do.call('rbind', strsplit(as.character(cnames$cfolders), '/', fixed=TRUE)))
  col.name[i] <- as.character(paste(cnames[,ncol(cnames)-1], cnames[,ncol(cnames)], sep = '_'))

  s <- list.files(path=cfolders[[i]], pattern="output_interval.csv")
  summ <- read.delim(s, sep=',')
  x <- summ[,c(grepl("total", colnames(summ)))]
  x2 <- data.frame(colSums(x))
  
  if(i==1){summ.summ <- rbind(summ.summ, x2)}else{summ.summ <- cbind(summ.summ, x2)}
  
  cnames <- as.data.frame(cfolders[[i]])
  cnames <- data.frame(do.call('rbind', strsplit(as.character(cnames$cfolders), '/', fixed=TRUE)))
  col.name[i] <- as.character(paste(cnames[,ncol(cnames)-1], cnames[,ncol(cnames)], sep = '_'))
  print(i)
}

colnames(summ.summ) <- col.name
setwd(cfolders[[1]])
setwd('..')
setwd('..')
setwd('..')
write.csv(summ.summ, paste(ds, "_summary_output_interval.csv", sep = ""))
