from __future__ import division, print_function, absolute_import

__docformat__ = "restructuredtext en"

__all__ = ['eigs', 'eigsh', 'svds', 'ArpackError', 'ArpackNoConvergence']

import numpy as np
from scipy.sparse.linalg.interface import aslinearoperator, LinearOperator
from scipy.sparse.linalg import eigs
from scipy.sparse import eye, isspmatrix, isspmatrix_csr
from scipy.linalg import lu_factor, lu_solve
from scipy.sparse.sputils import isdense
from scipy.sparse.linalg import gmres, splu
from scipy._lib._util import _aligned_zeros
from scipy._lib._threadsafety import ReentrancyLock

def _herm(x):
    return x.T.conj()

def svds(A, k=6, ncv=None, tol=0, which='LM', v0=None,
         maxiter=None, return_singular_vectors=True):
    """Compute the largest k singular values/vectors for a sparse matrix.

    Parameters
    ----------
    A : {sparse matrix, LinearOperator}
        Array to compute the SVD on, of shape (M, N)
    k : int, optional
        Number of singular values and vectors to compute.
        Must be 1 <= k < min(A.shape).
    ncv : int, optional
        The number of Lanczos vectors generated
        ncv must be greater than k+1 and smaller than n;
        it is recommended that ncv > 2*k
        Default: ``min(n, max(2*k + 1, 20))``
    tol : float, optional
        Tolerance for singular values. Zero (default) means machine precision.
    which : str, ['LM' | 'SM'], optional
        Which `k` singular values to find:

            - 'LM' : largest singular values
            - 'SM' : smallest singular values

        .. versionadded:: 0.12.0
    v0 : ndarray, optional
        Starting vector for iteration, of length min(A.shape). Should be an
        (approximate) left singular vector if N > M and a right singular
        vector otherwise.
        Default: random

        .. versionadded:: 0.12.0
    maxiter : int, optional
        Maximum number of iterations.

        .. versionadded:: 0.12.0
    return_singular_vectors : bool or str, optional
        - True: return singular vectors (True) in addition to singular values.

        .. versionadded:: 0.12.0

        - "u": only return the u matrix, without computing vh (if N > M).
        - "vh": only return the vh matrix, without computing u (if N <= M).

        .. versionadded:: 0.16.0

    Returns
    -------
    u : ndarray, shape=(M, k)
        Unitary matrix having left singular vectors as columns.
        If `return_singular_vectors` is "vh", this variable is not computed,
        and None is returned instead.
    s : ndarray, shape=(k,)
        The singular values.
    vt : ndarray, shape=(k, N)
        Unitary matrix having right singular vectors as rows.
        If `return_singular_vectors` is "u", this variable is not computed,
        and None is returned instead.


    Notes
    -----
    This is a naive implementation using ARPACK as an eigensolver
    on A.H * A or A * A.H, depending on which one is more efficient.

    Examples
    --------
    >>> from scipy.sparse import csc_matrix
    >>> from scipy.sparse.linalg import svds, eigs
    >>> A = csc_matrix([[1, 0, 0], [5, 0, 2], [0, -1, 0], [0, 0, 3]], dtype=float)
    >>> u, s, vt = svds(A, k=2)
    >>> s
    array([ 2.75193379,  5.6059665 ])
    >>> np.sqrt(eigs(A.dot(A.T), k=2)[0]).real
    array([ 5.6059665 ,  2.75193379])
    """

    print("calling modified version of svds")

    if not (isinstance(A, LinearOperator) or isspmatrix(A)):
        A = np.asarray(A)

    n, m = A.shape

    if k <= 0 or k >= min(n, m):
        raise ValueError("k must be between 1 and min(A.shape), k=%d" % k)

    if isinstance(A, LinearOperator):
        if n > m:
            X_dot = A.matvec
            X_matmat = A.matmat
            XH_dot = A.rmatvec
        else:
            X_dot = A.rmatvec
            XH_dot = A.matvec

            dtype = getattr(A, 'dtype', None)
            if dtype is None:
                dtype = A.dot(np.zeros([m,1])).dtype

            # A^H * V; works around lack of LinearOperator.adjoint.
            # XXX This can be slow!
            def X_matmat(V):
                out = np.empty((V.shape[1], m), dtype=dtype)
                for i, col in enumerate(V.T):
                    out[i, :] = A.rmatvec(col.reshape(-1, 1)).T
                return out.T

    else:
        if n > m:
            X_dot = X_matmat = A.dot
            XH_dot = _herm(A).dot
        else:
            XH_dot = A.dot
            X_dot = X_matmat = _herm(A).dot

    def matvec_XH_X(x):
        return XH_dot(X_dot(x))

    XH_X = LinearOperator(matvec=matvec_XH_X, dtype=A.dtype,
                          shape=(min(A.shape), min(A.shape)))

    # Get a low rank approximation of the implicitly defined gramian matrix.
    # This is not a stable way to approach the problem.
    eigvals, eigvec = eigs(XH_X, k=k, tol=tol ** 2, maxiter=maxiter,
                                  ncv=ncv, which=which, v0=v0)

    # In 'LM' mode try to be clever about small eigenvalues.
    # Otherwise in 'SM' mode do not try to be clever.
    if which == 'SM' or which == 'LM':
        eigvals = np.maximum(eigvals.real, 0)

        s = np.sqrt(eigvals)
        if not return_singular_vectors:
            return s

        if n > m:
            v = eigvec
            u = X_matmat(v) / s if return_singular_vectors != 'vh' else None
            vh = _herm(v)
        else:
            u = eigvec
            vh = _herm(X_matmat(u) / s) if return_singular_vectors != 'u' else None

    else:

        raise ValueError("which must be either 'LM' or 'SM'.")

    return u, s, vh
