import scipy
import time

from pymor.operators.constructions import induced_norm
from pymor.operators.numpy import NumpyMatrixOperator
from pymor.vectorarrays.numpy import NumpyVectorArray

from problems import *
from algorithms import orthogonal_part, testlimit
from localize_problem import localize_problem

from calculate_csi import calculate_csi
from calculate_lambda_min import calculate_lambda_min
from basis_generation import get_random_bases

import simdb.run as sdb

def calculate_local_approximations(gq,lq, bases, maxnorms, tolerances, num_testvecs):
    u = gq["d"].solve()
    l = gq["localizer"]
    coarse_grid_resolution = gq["coarse_grid_resolution"]
    failure_tolerance = 1e-15

    for ypos, xpos in np.ndindex((coarse_grid_resolution-1,coarse_grid_resolution-1)):
        ldict = lq[xpos, ypos]

        omstarnorm = induced_norm(ldict["omega_star_energy_0_product"])(l.localize_vector_array(u, ldict["omega_star_space"]))

        lsol = l.localize_vector_array(u, ldict["range_space"])
        lenergy_product = ldict["range_energy_0_product"]

        space = ldict["range_space"]
        quotients = {}
        local_failure_tolerance = failure_tolerance / ( (coarse_grid_resolution -1)**2 )
        for tolerance in tolerances:
            max_op_norm = tolerance / ldict["c_si"]
            testlimit_zeta = testlimit(
                failure_tolerance=local_failure_tolerance,
                dim_S=ldict["transfer_operator"].source.dim,
                dim_R=ldict["transfer_operator"].range.dim,
                num_testvecs=num_testvecs,
                target_error=max_op_norm,
                lambda_min=ldict["lambda_min"]
                )

            num_vecs = np.count_nonzero(maxnorms[space] > testlimit_zeta)
            num_vecs = min(num_vecs, len(bases[space]))
            basis = bases[space].copy(ind=range(num_vecs))

            # remove u_f:
            lsol_minus_u_f = lsol - ldict["local_solution"]
            # remove constant part:
            if not ldict["omega_has_dirichlet"]:
                constant_one = NumpyVectorArray(np.ones(lsol_minus_u_f.space.dim))
                constant_one_normed = constant_one * (1./ induced_norm(ldict["range_l2_product"])(constant_one))
                lsol_minus_u_f = orthogonal_part(constant_one_normed, ldict["range_l2_product"], lsol_minus_u_f)

            lsolorth = orthogonal_part(basis, lenergy_product, lsol_minus_u_f)
            lsolorth_norm = induced_norm(lenergy_product)(lsolorth)

            quotient = lsolorth_norm / omstarnorm
            if xpos == 0 and ypos == 0 and tolerance < 1e-7 and False:
                import pdb
                pdb.set_trace()
            quotients[tolerance] = quotient[0]

        sdb.append_values(quotients=[quotients[key] for key in reversed(sorted(quotients.keys()))])
        sdb.flush()

        ldict["quotients"] = quotients


def plot(gq, lq, xpos, ypos):
    import matplotlib.pyplot as plt
    plt.xscale("log")
    plt.yscale("log")
    plt.gca().invert_xaxis()
    xvals = sorted(lq[xpos,ypos]["quotients"].keys())
    if xpos == -1:
        coarse_grid_resolution = gq["coarse_grid_resolution"]
        for ypos2, xpos2 in np.ndindex((coarse_grid_resolution-1,coarse_grid_resolution-1)):
            yvals = [lq[xpos2,ypos2]["quotients"][xval] for xval in xvals]
            plt.plot(xvals, yvals, "x-")
    else:
        yvals = [lq[xpos,ypos]["quotients"][xval] for xval in xvals]
        plt.plot(xvals, yvals, "x-")
    plt.plot(xvals, xvals, "o-", label="1")
    plt.plot(xvals, np.array(xvals)/1e2, "o-", label="-2")
    plt.plot(xvals, np.array(xvals)/1e4, "o-", label="-4")
    plt.plot(xvals, np.array(xvals)/1e6, "o-", label="-6")
    plt.legend()
    plt.show()

for problem in ["h", "poisson"]:
    experimentname = "local_approximations"
    
    resolution=200
    coarse_grid_resolution = 10
    num_testvecs = 20
    max_basis_size = 80

    sdb.new_dataset(experimentname,
                    problem=problem,
                    coarse_grid_resolution=coarse_grid_resolution,
                    resolution=resolution,
                    num_testvecs=num_testvecs,
                    max_basis_size=max_basis_size)

    if problem == "h":
        p = h_problem()
    elif problem == "poisson":
        p = poisson_problem()
    else:
        raise "Jo!"

    gq, lq = localize_problem(p, coarse_grid_resolution, resolution)
    calculate_csi(gq, lq)
    calculate_lambda_min(gq, lq)
    tols = np.logspace(4, -10, 80)
    sdb.add_values(tolerances=tols)

    iterations = 1000

    lasttime = time.time()
    for _ in range(iterations):
        thistime = time.time()
        print("duration for last iteration {}".format(thistime - lasttime))
        lasttime = thistime

        bases, maxnorms = get_random_bases(gq, lq, max_basis_size, num_testvecs)
        calculate_local_approximations(gq,lq, bases, maxnorms, tols, num_testvecs=num_testvecs)


