import numpy as np

from pymor.vectorarrays.numpy import NumpyVectorArray
from pymor.operators.numpy import NumpyMatrixOperator
from operator_svd import operator_svd4
from pymor.operators.constructions import induced_norm
from pymor.algorithms.gram_schmidt import gram_schmidt

from algorithms import orthogonal_part

def get_optimal_basis(transop, source_product, range_product):
    transopmat = transop.apply(NumpyVectorArray(np.identity(transop.source.dim)))
    transop_assembled = NumpyMatrixOperator(transopmat.data.T)
    tU, ts, tV = operator_svd4(transop_assembled, source_product, range_product)
    return tU, ts, tV

def get_optimal_bases(gq, lq, size):
    coarse_grid_resolution = gq["coarse_grid_resolution"]
    returnvalue = {}
    for ypos, xpos in np.ndindex((coarse_grid_resolution-1,coarse_grid_resolution-1)):
        ldict = lq[xpos, ypos]
        tU, ts, tV = get_optimal_basis(ldict["transfer_operator"], ldict["source_l2_product"], ldict["range_energy_0_product"])
        pou = ldict["pou_operator"]
        # put local solution into basis
        # put constant function into basis
        lsol = ldict["local_solution"]
        basis = NumpyVectorArray(np.ones((1,lsol.space.dim)))
        basis.append(lsol)

        oprange = tU.copy(ind=range(min(size, len(tU))))
        constant_function = NumpyVectorArray(np.ones((1,oprange.space.dim)))
        constant_function *= 1./induced_norm(ldict["range_l2_product"])(constant_function)[0]
        oprange = orthogonal_part(constant_function, ldict["range_l2_product"], oprange)
        basis.append(oprange)
        
        # now apply partition of untiy
        basis = pou.apply(basis)

        gram_schmidt(basis, product=ldict["pou_range_h1_product"], copy=False)
        returnvalue[ldict["pou_range_space"]] = basis

    return returnvalue


def get_random_basis(transop, size):
    basis = transop.apply(NumpyVectorArray(np.random.normal(size=(size,transop.source.dim))))
    return basis

def get_maxnorms(transop, basis, product, num_testvecs):
    assert np.linalg.norm(product.apply2(basis, basis) - np.identity(len(basis))) < 1e-6
    testvecs = transop.apply(NumpyVectorArray(np.random.normal(size=(num_testvecs, transop.source.dim))))
    norm = induced_norm(product)
    maxnorms = []
    for i in range(len(basis)+1):
        subbasis = basis.copy(ind=range(i))
        import pdb
        #pdb.set_trace()
        testvecs -= subbasis.lincomb(product.apply2(testvecs, subbasis))
        maxnorms.append(np.max(norm(testvecs)))

    return np.array(maxnorms)

def get_random_bases(gq, lq, size, num_testvecs):
    coarse_grid_resolution = gq["coarse_grid_resolution"]
    returnvalue = {}
    allmaxnorms = {}
    for ypos, xpos in np.ndindex((coarse_grid_resolution-1,coarse_grid_resolution-1)):
        ldict = lq[xpos, ypos]
        pou = ldict["pou_operator"]
        # put local solution into basis
        # put constant function into basis
        lsol = ldict["local_solution"]
        if not ldict["omega_has_dirichlet"]:
            # constant function and local solution
            basis = NumpyVectorArray(np.ones((1,lsol.space.dim)))
            basis.append(lsol)
        else:
            # no constant function, but local solution
            basis = lsol.copy()

        # get op * random
        oprange = get_random_basis(ldict["transfer_operator"], size)
        # remove constant part
        if not ldict["omega_has_dirichlet"]:
            constant_function = NumpyVectorArray(np.ones((1,oprange.space.dim)))
            constant_function *= 1./induced_norm(ldict["range_l2_product"])(constant_function)[0]
            oprange = orthogonal_part(constant_function, ldict["range_l2_product"], oprange)

        # orthogonalize to apply test
        gram_schmidt(oprange, product=ldict["range_energy_0_product"], copy=False)
        maxnorms = get_maxnorms(ldict["transfer_operator"], oprange, ldict["range_energy_0_product"], num_testvecs)
        returnvalue[ldict["range_space"]] = oprange
        allmaxnorms[ldict["range_space"]] = maxnorms

        # use it
        basis.append(oprange)
        
        # now apply partition of untiy
        basis = pou.apply(basis)

        gram_schmidt(basis, product=ldict["pou_range_h1_product"], copy=False)
        returnvalue[ldict["pou_range_space"]] = basis
        allmaxnorms[ldict["pou_range_space"]] = maxnorms

    return returnvalue, allmaxnorms
