#-------------------------------------------------------------------------------
# Copyright (c) 2012 University of Illinois, NCSA.
# All rights reserved. This program and the accompanying materials
# are made available under the terms of the 
# University of Illinois/NCSA Open Source License
# which accompanies this distribution, and is available at
# http://opensource.ncsa.illinois.edu/license.html
#-------------------------------------------------------------------------------
#--------------------------------------------------------------------------------------------------#
##' Runs heirarchical meta-analysis of plant trait data 
##'
##' \code{pecan.ma} runs a heirarchical Bayes meta-analytical model.
##' This model combines prior information with data from studies on the particular species or group of interest.
##' Data that is incorporated into the meta-analysis include the mean (Y), sample size (n),
##' and precision (obs.prec). 
##' Where a set of data includes more than one level of treatment, comes from more than one site,
##' or comes from both field and greenhouse studies, these variables are included as random (treatment, site)
##' or fixed (greenhouse) effects. 
##' The pecan.ma function writes a model for each specific data set and prior using the write.ma.model()
##' function to modify the ma.model.template.bug generic model. 
##' 
##' 
##' @title Trait Meta-analysis 
##' @param trait.data list of dataframes, one per trait for which data is available, generated by call to query.traits()
##' @param prior.distns dataframe of prior distributions generated by call to query.priors()
##' @param taupriors priors on variance parameters, can be scaled as needed with data mean
##' @param j.iter number of mcmc samples
##' @param settings list of settings from settings file
##' @param outdir output directory
##' @param overdispersed TRUE by default, if set to FALSE, data mean will be used as starting point for MCMC chains (use with caution)
##' @return four chains with 5000 total samples from posterior 
##' @author David LeBauer, Michael C. Dietze
##' @export
##' @examples
##' \dontrun{
##' prior.distns <- query.priors('ebifarm.c4crop', c('SLA', 'c2n_leaf'))
##' trait.data <- query.traits('938', c('SLA', 'c2n_leaf'))
##' pecan.ma(prior.distns, trait.data, 25000)
##' }
#--------------------------------------------------------------------------------------------------#
pecan.ma <- function(trait.data, prior.distns, taupriors,
                     j.iter, settings, outdir,
                     overdispersed = TRUE){
  
  madata <- list()
  ## Meta-analysis for each trait
  mcmc.object <- list() #  initialize output list of mcmc objects for each trait
  mcmc.mat    <- list()
  
  ## Set inputs for jags.model()
  j.chains <- 4
  j.iter <- as.numeric(j.iter)  # Added by SPS 08.27.2013. issue #1803
  ## log the mcmc chain parameters
  sink(file = file.path(outdir,'meta-analysis.log'), split = TRUE)
  cat(paste( 'Each meta-analysis will be run with: \n',
            j.iter, ' total iterations,\n',
            j.chains, ' chains, \n',
            'a burnin of ', j.iter/2, ' samples,\n',
            ', \nthus the total number of samples will be ', j.chains*(j.iter/2),
             '\n', sep = '')
      )

  for(trait.name in names(trait.data)) {
    
    prior <- prior.distns[trait.name, c('distn', 'parama', 'paramb', 'n')]
    
    writeLines(paste('################################################'))
    writeLines(paste('------------------------------------------------'))
    writeLines(paste('starting meta-analysis for:\n\n', trait.name,'\n'))
    writeLines(paste('------------------------------------------------'))
    data <- trait.data[[trait.name]]
    data <- data[, which(!colnames(data) %in% c("cite","trait_id","se"))] ## remove citation and other unneeded co lumns
    data <- data[order(data$site,data$trt),]#not sure why, but required for JAGS model

    ##check for excess missing data
    if(all(is.na(data$obs.prec))){
      writeLines("NO ERROR STATS PROVIDED, DROPPING RANDOM EFFECTS")
      data$site = rep(1,nrow(data))
      data$trt  = rep(0,nrow(data))
    }

    if(!is.null(settings$meta.analysis$random.effects)){
      if(!as.logical(settings$meta.analysis$random.effects)){
        data$site = rep(1,nrow(data))
        data$trt  = rep(0,nrow(data))
      }
    }
    
    #print out some data summaries to check
    writeLines(paste('prior for ', trait.name, ' (using R parameterization):\n',
                prior$distn, '(',prior$parama, ', ', prior$paramb, ')', sep = ''))
    writeLines(paste('data max:', max(data$Y,na.rm=TRUE), '\ndata min:', min(data$Y,na.rm=TRUE), '\nmean:',
                     signif(mean(data$Y,na.rm=TRUE),3), '\nn:', length(data$Y)))
    writeLines('stem plot of data points')
    writeLines(paste(stem(data$Y)))
    if(any(!is.na(data$obs.prec)) && all(!is.infinite(data$obs.prec))){
      writeLines('stem plot of obs.prec:')
      writeLines(paste(stem(data$obs.prec^2)))
    } else {
      writeLines(paste('no estimates of SD for', trait.name))
    }
        

    madata[[trait.name]] <- data
    jag.model.file <-  file.path(outdir, paste(trait.name, ".model.bug", sep=""))  # file to store model

    ## run the meta-analysis in JAGS
    jags.out <- single.MA(data, j.chains, j.iter,
                          taupriors$tauA, taupriors$tauB[trait.name],
                          prior, jag.model.file, overdispersed)
    
    print(summary(jags.out))
    
    jags.out.trunc <- window(jags.out, start = j.iter/2)
 
    mcmc.object[[trait.name]] <- jags.out.trunc
  }
  save(madata, file = file.path(outdir,'madata.Rdata'))
  sink()
  return(mcmc.object)
}
#==================================================================================================#


####################################################################################################
### EOF.  End of R script file.    					
####################################################################################################
