## ensures data.table objects treated as such
## http://stackoverflow.com/q/24501245/513006
.datatable.aware=TRUE

##' Load met data from PEcAn formatted met driver
##'
##' subsets a PEcAn formatted met driver file and converts to a data.table / data.frame object
##' @title load CF met
##' @param met.nc object of class ncdf4 representing an open CF compliant, PEcAn standard netcdf file with met data
##' @param lat numeric value of latutude
##' @param lon numeric value of longitude
##' @param start.date format is "YYYY-MM-DD"
##' @param end.date format is "YYYY-MM-DD"
##' @return data.table of met data
##' @export
##' @author David LeBauer
load.cfmet <- cruncep_nc2dt <- function(met.nc, lat, lon, start.date, end.date){
  lat <<- lat # send lat to global environment if daily
  ## Lat and Lon
  Lat <- ncvar_get(met.nc, "lat")
  Lon <- ncvar_get(met.nc, "lon")
  
  if(min(abs(Lat-lat)) > 2.5 | min(abs(Lon-lon)) > 2.5) logger.error("lat / lon (", lat, ",", lon, ") outside range of met file (", range(Lat), ",", range(Lon))
  
  lati <- which.min(abs(Lat - lat))
  loni <- which.min(abs(Lon - lon))

  time.idx <- ncvar_get(met.nc, "time")

  ## confirm that time units are PEcAn standard
  time.units <- unlist(strsplit(met.nc$dim$time$units, " since "))
  if(!grepl("days", time.units[1])) {
      logger.error("time dimension does not have units of days")
  }
  if(!ymd(time.units[2]) == ymd("1700-01-01")){
      logger.error("time dimension of met input does not start at 1700-01-01")
  }
  all.dates <- data.table(index = seq(time.idx),
                          date = ymd("1700-01-01") +
                          days(floor(time.idx)) +
                          minutes(as.integer(ud.convert(time.idx - floor(time.idx), "days", "minutes"))))
  
 
  if(ymd(start.date) + days(1) < min(all.dates$date)) logger.error("run start date", ymd(start.date), "before met data starts", min(all.dates$date))
  if(ymd(end.date) > max(all.dates$date)) logger.error("run end date",   ymd(start.date), "after met data ends", min(all.dates$date))

  run.dates <- all.dates[date > ymd(start.date) & date < ymd(end.date),
                           list(index, date, doy = yday(date),
                                year = year(date), month = month(date),
                                day  = day(date), hour = hour(date))]
  
  results <- list()

  data(mstmip_vars, package = "PEcAn.utils")

  variables <- as.character(mstmip_vars$standard_name[mstmip_vars$standard_name %in% attributes(met.nc$var)$names])
  
  vars <- lapply(variables, function(x) get.ncvector(x, lati = lati, loni = loni, run.dates = run.dates, met.nc = met.nc))
  
  names(vars) <- variables
  
  result <- cbind(run.dates, as.data.table(vars[!sapply(vars, is.null)]))

  return(result)
}


##' Temporal downscaling of daily or subdaily met data
##'
##' @title Downscale CF met data
##' @param cfmet data.table with CF variables generated by \code{\link{load.cfmet}}  
##' @param output.dt time step for output
##' @return downscaled result
##' @author David LeBauer
cfmet.downscale.time <- cruncep_hourly <- function(cfmet, output.dt = 1, ...){

  ## time step
  dt_hr <- as.numeric(round(difftime(cfmet$date[2], cfmet$date[1],  units = "hours")))
  
  if(dt_hr > output.dt & dt_hr < 6) {
    downscaled.result <- cfmet.downscale.subdaily(subdailymet = cfmet, output.dt = output.dt)
  } else if(dt_hr >= 6 & dt_hr < 24){
    cfmet <- cbind(cfmet, cfmet[,list(air_temperature_max = max(air_temperature),
                                      air_temperature_min = min(air_temperature)), by = 'year,doy'])
    logger.warn("timestep of input met data is between 6 and 24 hours.\n",
                "PEcAn will automatically convert this to daily data\n",
                "you should confirm validity of downscaling, in particular that min / max temperatures are realistic")
  } else if (dt_hr == 24) {
    downscaled.result <- cfmet.downscale.daily(dailymet = cfmet, output.dt = output.dt, ...)
  } else if(dt_hr > 24){
    logger.error("only daily and sub-daily downscaling supported")
  } else if (dt_hr == output.dt) {
    downscaled.result <- cfmet
  }
  return(downscaled.result)
}


##' Subdaily to hourly (or less) downscaling
##'
##' Uses simple spline to interpolate variables with diurnal variability, otherwise uses averaging or repeating
##' for variables with no clear diurnal pattern. For all variables except temperature, negative values are set to zero.
##' @title subdaily downscaling
##' @param subdailymet data table with climate variables queried from \code{\link{load.cfmet}}
##' @param output.dt output timestep. default is one hour
##' @export
##' @return weather file for input to BioGro and related crop growth functions
##' @author David LeBauer
cfmet.downscale.subdaily <- function(subdailymet, output.dt = 1){
  ## converting surface_downwelling_shortwave_flux_in_air from W/m2 avg to PPFD
  new.date <- subdailymet[,list(hour = 0:(23 / output.dt) / output.dt),
                    by = c("year", "month", "day", "doy")]
  
  new.date$date <- new.date[,list(date = ymd(paste(year, month, day)) + hours(hour))]
  
  solarMJ <- ud.convert(subdailymet$surface_downwelling_shortwave_flux_in_air, paste0("W ", output.dt, "h"), "MJ")
  PAR <- 0.486 * solarMJ ## Cambell and Norman 1998 p 151, ch 10
  subdailymet$ppfd <- ud.convert(PAR, "mol s", "micromol h")
  
  downscaled.result <- list()
  downscaled.result[["surface_downwelling_shortwave_flux_in_air"]] <- subdailymet$surface_downwelling_shortwave_flux_in_air 
  downscaled.result[["ppfd"]] <- subdailymet$ppfd
  
  for(var in c("air_pressure", "specific_humidity",
               "precipitation_flux", "air_temperature", "northward_wind", "eastward_wind", "surface_downwelling_shortwave_flux_in_air", "ppfd")){
    if(var %in% colnames(subdailymet)){
      ## convert units from subdaily to hourly
      hrscale <- ifelse(var %in%
                          c("surface_downwelling_shortwave_flux_in_air",
                            "precipitation_flux"),
                        output.dt, 1)
      
      f <- splinefun(as.numeric(subdailymet$date), (subdailymet[[var]] / hrscale), method = "monoH.FC")
      downscaled.result[[var]] <- f(as.numeric(new.date$date))
      if(!var == "air_temperature"){
        downscaled.result[[var]][downscaled.result[[var]] < 0] <- 0
      }
    }
    
  }
  
  downscaled.result <- cbind(new.date, as.data.table(downscaled.result))#[date <= max(result$date),]
  
}

##' Simple, Fast Daily to Hourly Climate Downscaling
##'
##' Based on weach family of functions but 5x faster than weachNEW,
##' and requiring metric units (temperature in celsius, windspeed in kph,
##' precip in mm, relative humidity as fraction). 
##' Derived from the weachDT function in the BioCro package.
##' @title daily to subdaily downscaling
##' @param dailymet data table with climate variables
##' @param lat latitude (for calculating solar radiation)
##' @param output.dt output timestep
##' @export
##' @return weather file for input to BioGro and related crop / ecosystem models
##' @author David LeBauer
cfmet.downscale.daily <- weachDT <- function(dailymet, output.dt = 1, lat = lat){
  
  tint <- 24 / output.dt
  tseq <- 0:(23 * output.dt) / output.dt

  setkeyv(dailymet, c("year", "doy"))
  
  setnames(dailymet, c("air_temperature_max", "air_temperature_min"), c("tmax", "tmin"))

  light <- dailymet[,lightME(DOY = doy, t.d = tseq, lat = lat),
                    by = c("year", "doy")]
  
  light$Itot <- light[,list(I.dir + I.diff)]
  resC2 <- light[, list(resC2 = (Itot - min(Itot)) / max(Itot)), by = c("year", "doy")]$resC2
  solarR <- dailymet[,list(year, doy, 
                           solarR = rep(surface_downwelling_shortwave_flux_in_air * 2.07 * 10^5 /36000, each = tint) * resC2)]
  
  SolarR <- cbind(resC2, solarR)[,list(SolarR = solarR * resC2)]$SolarR
  
  ## Temperature
  Temp <- dailymet[,list(Temp = tmin + (sin(2*pi*(tseq-10)/tint) + 1)/2 * (tmax - tmin), hour = tseq),
                   by = 'year,doy']$Temp
  
  ## Relative Humidity
  RH <-   dailymet[,list(RH = rep(relative_humidity, each = tint), hour = tseq), by = 'year,doy']
  setkeyv(RH, c('year','doy','hour'))  
  
 # if(!"air_pressure" %in% colnames(dailymet)) air_pressure <- 
  qair <- dailymet[,list(year, doy, tmin, tmax, air_pressure,
                         air_temperature,
                         qmin = rh2qair(rh = relative_humidity/100, T = tmin),
                         qmax = rh2qair(rh = relative_humidity/100, T =tmax))]
  
  a <- qair[,list(year, doy, tmin, tmax, air_temperature, qmin, qmax, pressure = ud.convert(air_pressure, "Pa", "millibar"))][ ,list(year, doy, rhmin = qair2rh(qmin, air_temperature, pressure),       rhmax = qair2rh(qmax, air_temperature, pressure))]
  rhscale <- (cos(2 * pi * (tseq - 10)/tint) + 1)/2
  RH <- a[, list(RH = rhmin + rhscale * (rhmax - rhmin)), by = c("year", "doy")]$RH
  ## Wind Speed
  if('wind_speed' %in% colnames(dailymet)){
      wind_speed <- rep(dailymet$wind_speed, each = tint)
  } else {
      northward_wind <- rep(dailymet$northward_wind, each = tint)
      eastward_wind <- rep(dailymet$eastward_wind, each = tint)
      wind_speed <- sqrt(northward_wind^2 + eastward_wind^2)
  }
  ## Precipitation
  precip <- rep(dailymet$precipitation_flux / tint, each = tint)
  
  ## Hour
  time <- dailymet[,list(hour = tseq), by = c("year", "doy")]
  
  ans <- data.table(time,
                    downwelling_photosynthetic_photon_flux = SolarR,
                    air_temperature = ud.convert(Temp, "kelvin", "celsius"), 
                    relative_humidity = RH,
                    wind = wind_speed,
                    precipitation_flux = precip)
  return(ans)
}

met2model.BIOCRO <- function(met){
    met[ , `:=` (wind =  sqrt(northward_wind^2 + eastward_wind^2),
        air_temperature = ud.convert(air_temperature, "kelvin", "celsius"))]
    return(met)
}

##' Get time series vector from netCDF file
##'
##' internal convenience function for
##' streamlining extraction of data from netCDF files
##' with CF-compliant variable names
##' 
##' @title Get time series vector from netCDF file
##' @param var 
##' @param lati 
##' @param loni 
##' @param run.dates 
##' @param met.nc netcdf file with CF variable names
##' @return numeric vector
##' @export
##' @author David Shaner LeBauer
get.ncvector <- function(var, lati = lati, loni = loni,
                         run.dates = run.dates, met.nc){
  
  start.idx = c(lat = lati, lon = loni, time = run.dates$index[1])
  count.idx = c(lat = 1, lon = 1, time = nrow(run.dates))
  dim.order <- sapply(met.nc$var$air_temperature$dim, function(x) x$name)
  ncvar_get2 <- function(var){
    ans <-  ncvar_get(nc = met.nc, varid = var,
                      start = start.idx[dim.order],
                      count = count.idx[dim.order])
    return(as.numeric(ans))
  }
  
  if(var %in% attributes(met.nc$var)$names){
    ans <- ncvar_get2(var)
  } else if (var == "air_pressure"){
    ans <- 1013.25
  } else if (var == "wind"){
    ans <- sqrt(ncvar_get2("northward_wind")^2 + ncvar_get2("eastward_wind")^2)
  } else {
    ans <- NULL
  } 
  
  if (var == "precipitation_flux"){
    precip_units <- met.nc$var[["precipitation_flux"]]$units
    if(grepl("kg m-2", precip_units)){
      precip_units <- gsub("kg m-2", "mm", precip_units)
    }
    ans <- ud.convert(ans, precip_units, "mm day-1")
    
  }
  return(ans)
}
