#-------------------------------------------------------------------------------
# Copyright (c) 2012 University of Illinois, NCSA.
# All rights reserved. This program and the accompanying materials
# are made available under the terms of the 
# University of Illinois/NCSA Open Source License
# which accompanies this distribution, and is available at
# http://opensource.ncsa.illinois.edu/license.html
#-------------------------------------------------------------------------------
#' @title allom.BayesFit
#' @name  allom.BayesFit
#' @aliases allom.BayesFit
#' 
#' @description Module to fit a common power-law allometric model
#' to a mixture of raw data and allometric equations
#' in a Heirarchical Bayes framework with multiple imputation
#' of the allometric data
#' 
#' @param allom - object (usually generated by query.allom.data) which
#'                  needs to be a list with two entries:
#'            'field' - contains a list, each entry for which is
#'                      a data frame with 'x' and 'y'. Can be NULL
#'            'parm' -  a single data frame with the following components:
#'            \itemize{         
#'                   \item{n}     {sample size}
#'                   \item{a}     {eqn coefficient}
#'                   \item{b}     {eqn coefficient}
#'                   \item{c}     {eqn coefficient}
#'                   \item{d}     {eqn coefficient}
#'                   \item{e}     {eqn coefficient}
#'                   \item{se}    {standard error}
#'                   \item{eqn}   {sample size}
#'                   \item{Xmin}  {smallest tree sampled (cm)}
#'                   \item{Xmax}  {largest tree sampled (cm)}
#'                   \item{Xcor}  {units correction on X}
#'                   \item{Ycor}  {units correction on Y}
#'                   \item{Xtype} {type of measurement on the X}
#'                   \item{spp}   { - USFS species code}
#'          }
#' @param nrep - number of MCMC replicates
#'
#' @param form   functional form of the allometry: "power" vs "exp"
#'
#' @details  dependencies: requires MCMCpack and mvtnorm
#'
#' note: runs 1 chain, but multiple chains can be simulated by
#'       multiple function calls
#'       
#' @return returns MCMC chain and ONE instance of 'data'
#' note: in many cases the estimates are multiply imputed 
#'
#' @author Michael Dietze
#' 
#' 
allom.BayesFit <- function(allom,nrep=10000,form="power") {

  ## check for valid inputs
  if(!(form %in% ('power'))){
    print(c("allom.BayesFit: Requested functional form",form,"not currently supported"))
    return(NULL)
  }
  if(is.null(allom)){
    print("allom.BayesFit: no data recieved, allom is NULL")
    return(NULL)
  }
  if(!is.list(allom)){
    print("allom.BayesFit: arguement allom must be a list")
    return(NULL)
  }
  if(!is.numeric(nrep) | nrep <= 0){
    print(c("allom.BayesFit: invalid nrep",nrep))
    return(NULL)
  }
  
  
  require(mvtnorm)
  require(MCMCpack)
  
  ##grab required variables from allom$parm
  n    <- nu(allom[['parm']]$n)
  a    <- nu(allom[['parm']]$a)
  b    <- nu(allom[['parm']]$b)
  c    <- nu(allom[['parm']]$c)
  d    <- nu(allom[['parm']]$d)
  e    <- nu(allom[['parm']]$e)
  se   <- nu(allom[['parm']]$se)
  Xcor <- nu(allom[['parm']]$Xcor)
  Ycor <- nu(allom[['parm']]$Ycor)
  Xtype<- as.character(allom[['parm']]$Xtype)
  eqn  <- nu(allom[['parm']]$eqn)
  rng  <- cbind(nu(allom$parm$Xmin),nu(allom$parm$Xmax))
  spp  <- nu(allom[['parm']]$spp)

  ## declare constants
  ntally = nrow(allom[['parm']]); if(is.null(ntally)) ntally = 0;
  nfield = length(allom[['field']])
  nsite  = ntally + nfield
  my.spp = unique(spp)
  nspp = length(my.spp)
  
  if(nsite == 0){
    print(c("allomBayesFit no data"))
    return(NULL)
  }
  
  ## define priors
  s1  = s2 = 0.1 # IG prior on the within-study variance
  mu0 = c(0.2,8/3) # normal prior mean on global mean
  V0  = matrix(c(100,0,0,100),2,2) # normal prior variance on global mean
  V0I = solve(V0)
  m0V0 = t(mu0)%*%V0I %*% mu0
  V0Imu0 = V0I %*% mu0
  v = 0.1         ## wishart prior on across-study variance
  S = diag(0.1,2) 

  ## declare storage
  b0GIBBS  = matrix(0,nrep,nsite)
  b1GIBBS  = matrix(0,nrep,nsite)
  muGIBBS  = matrix(0,nrep,2)
  sigGIBBS = matrix(0,nrep,nsite)
  tauGIBBS = matrix(0,nrep,3)
  DGIBBS   = rep(NA,nrep)
  BgGIBBS  = matrix(0,nrep,2)
  SgGIBBS  = rep(NA,nrep)
  DgGIBBS  = rep(NA,nrep)
  
  ## initialization
  mu  = mu0
  b0  = rep(mu[1],nsite)
  b1  = rep(mu[2],nsite)
  tau = diag(c(1,1))
  tauI= solve(tau)
  sigma = rep(0.3,nsite)
  sinv  = 1/sigma
  data  = allom[['field']]
  if(ntally > 0){
    for(i in 1:ntally){
      data[[i+nfield]] = list(x=rep(0,n[i]),y=rep(0,n[i]))
    }
  }
  x=y<-NULL
  Sg = 1
  Bg = mu0
  SgI = 1/Sg
  D = Dg = 0

  ## MCMC LOOP
  pb <- txtProgressBar(min = 0, max = nrep, style = 3)
  for(g in 1:nrep){
    
    ## For tabulated equations, impute X,Y data --------------------------------------
    if(ntally > 0){
    for(j in 1:ntally){
      x0 <- runif(n[j],rng[j,1],rng[j,2])
      if(!is.na(Xcor[j])){
        x <- Xcor[j]*x0
      }else{
        if(Xtype[i] == "d.b.h.^2"){
          ## convert to sq inches
          x = x0*x0/(2.54*2.54)
        } else {
          x = x0*x0*pi/4 ## convert to cm Basal Area
        }
      }      
      y <- NA      
      if(eqn[j] == 1){        
        y = a[j] + b[j]*c[j]*log10(x)
        y = 10^rnorm(n[j],y,se[j])
      } else if(eqn[j] == 2){
        y = a[j] + b[j]*x + c[j]*d[j]*log(x)
        y = exp(rnorm(n[j],y,se[j]))
      } else if(eqn[j] == 3){
        y = a[j] + b[j]*log(x) + c[j]*(d[j]+(e[j]*log(x)))
        y = exp(rnorm(n[j],y,se[j]))
      } else if(eqn[j] == 4){
        y = a[j] + b[j]*x + c[j]*x^d[j]
        y = rnorm(n[j],y,se[j])
      } else if(eqn[j] == 5){
        y = a[j] + b[j]*x + c[j]*x^2 + d[j]*x^3      
        y = rnorm(n[j],y,se[j])
      } else if(eqn[j] == 6){
        y = a[j] *(exp( b[j] + (c[j]*log(x)) + d[j]*x))
        y = rnorm(n[j],y,se[j])
      } else if(eqn[j] == 7){
        y = a[j] + ((b[j]*(x^c[j]))/((x^c[j])+ d[j]))
        y = rnorm(n[j],y,se[j])
      } else if(eqn[j] == 8){
        y = a[j] + b[j]*log10(x)
        y = 100^rnorm(n[j],y,se[j])
      }else if(eqn[j] == 9){
        y = log(a[j]) + b[j]*log(x)
        y = exp(rnorm(n[j],y,se[j]))
      }
      y[y<=0] <- NA
      y = y*Ycor[j]
      s2 = which(!is.na(y))
      data[[nfield+j]]$x = x0[s2] ## store the std units, not the transformed
      data[[nfield+j]]$y = y[s2]  ## store y transformed to std units
    } ## end loop over tally entries
    } ## end check for ntally > 0
    
  if(FALSE){
    #diagnostics
    pdf("DvBscatter.pdf")
    plot(1,1,type='n',log='xy',xlim=c(0.1,1000),ylim=c(0.0001,100000))
    BETA <- matrix(NA,nsite,2)
    for(i in 1:nsite){
      points(data[[i]]$x,data[[i]]$y,col=i)
      BETA[i,] <- coef(lm(log10(data[[i]]$y) ~ log10(data[[i]]$x)))
    }
    hist(BETA[,1],breaks=20)
    hist(BETA[,2],breaks=20)
    plot(BETA)
    dev.off()
  }

    ## Fit "random site" hierarchical allometry -----------------------------
    tauImu = tauI %*% mu
    for(j in 1:nsite){
      
      ## Update study-level regression parameters
      X = cbind(rep(1,length(data[[j]]$x)),log(data[[j]]$x))
      Y = log(data[[j]]$y)
      bigV <- solve( sinv[j]*t(X)%*%X + tauI)
      littlev <- sinv[j]*t(X) %*% Y + tauImu
      beta <- t(rmvnorm(1,bigV %*% littlev,bigV))
      b0[j] <- beta[1]
      b1[j] <- beta[2]
      
      ## Update study-level error
      u1 <- s1 + nrow(X)/2
      u2 <- s2 + 0.5*crossprod(Y-X %*% beta)
      sinv[j] <- rgamma(1,u1,u2)    ## precision
      
      ## Calculate Deviance
      D[j] <- -2*sum(dnorm(Y,X%*%beta,sigma,log=TRUE))
    }
    sigma <- 1/sinv  ## variance
    
    ## Update across-study means
    B = cbind(b0,b1)
    bigV <- solve( nrow(B)*tauI + V0I)
    littlev <- V0Imu0
    for(i in 1:nrow(B)){
      littlev <- littlev + tauI %*% B[i,]
    }
    mu <- t(rmvnorm(1,bigV %*% littlev,bigV))
    
    ## Update across-study variance
    u1 <- v + nrow(B)
    u2 <- S + crossprod(B - t(matrix(mu,nrow=2,ncol=nrow(B))))
    tau <- riwish(u1,u2)
    tauI  <- solve(tau)
    
    ## Fit "random species" hierarchical model -------------------------------------
    if(nspp > 1){
      
      
      
    }
    
    
    ## Fit alternative non-heirarchical model --------------------------------------
    X <- Y <- NULL
    for(i in 1:nsite){
      X <- c(X,data[[i]]$x)
      Y <- c(Y,data[[i]]$y)
    }
    Y = log(Y)
    X = cbind(rep(1,length(X)),log(X))
    bigV <- solve(SgI*t(X)%*%X + V0I )
    littlev <- SgI*t(X) %*% Y + V0Imu0
    Bg  <- t(rmvnorm(1,bigV %*% littlev,bigV))
    u1  <- s1 + nrow(X)/2
    u2  <- s2 + 0.5*crossprod(Y-X %*% Bg)
    SgI <- rgamma(1,u1,u2)    ## precision
    Sg  <- 1/SgI  ## variance
    Dg <- -2*sum(dnorm(Y,X%*%Bg,sqrt(Sg),log=TRUE))
    
    ## Store Parameter estimates
    b0GIBBS[g,] <- b0  
    b1GIBBS[g,] <- b1
    muGIBBS[g,] <- mu
    sigGIBBS[g,] <- sigma
    tauGIBBS[g,] <- vech(tau)
    DGIBBS[g]  <- sum(D)
    
    BgGIBBS[g,] <- Bg
    SgGIBBS[g]  <- Sg
    DgGIBBS[g]  <- Dg  

    setTxtProgressBar(pb,g)
  }## END MCMC LOOP
  close(pb)
  
  out <- cbind(b0GIBBS,b1GIBBS,muGIBBS,sigGIBBS,tauGIBBS,DGIBBS,BgGIBBS,SgGIBBS,DgGIBBS)
  colnames(out) <- c(paste("b0",1:nsite,sep="."),
                  paste("b1",1:nsite,sep="."),
                  "mu0","mu1",
                  paste("sig",1:nsite,sep="."),
                  "tau11","tau12","tau22","D",
                  "Bg0","Bg1","Sg","Dg")
  return(list(mc=as.mcmc(out),obs=data))
  
} ## END allom.BayesFit
