
##-------------------------------------------------------------------------------
## Copyright (c) 2012 University of Illinois, NCSA.
## All rights reserved. This program and the accompanying materials
## are made available under the terms of the 
## University of Illinois/NCSA Open Source License
## which accompanies this distribution, and is available at
## http://opensource.ncsa.illinois.edu/license.html
##-------------------------------------------------------------------------------
##--------------------------------------------------------------------------------------------------#
##' Convert BioCro output to netCDF
##'
##' Converts BioCro output to netCDF.
##' Modified from on model2netcdf.sipnet and model2netcdf.ED2 by
##' Shawn Serbin and Mike Dietze
##' @name model2netcdf.BIOCRO
##' @title Function to convert biocro model output to standard netCDF format
##' @param outdir Location of model output
##' @param lat Latitude of the site
##' @param lon Longitude of the site
##' @export
##'
##' @author David LeBauer, Deepak Jaiswal
model2netcdf.BIOCRO <- function(resultDT, genus = genus, outdir, lat = -9999, lon = -9999) {
  ## longname prefix station_* used for a point
  ## http://cf-pcmdi.llnl.gov/documents/cf-conventions/1.4/cf-conventions.html#scalar-coordinate-variables
  x <- ncdim_def("lat", "degrees_east",
                 vals =  as.numeric(lat),
                 longname = "station_latitude") 
  y <- ncdim_def("lon", "degrees_north",
                 vals = as.numeric(lon),
                 longname = "station_longitude")
  
  dates <- ymd(paste0(resultDT$Year, "-01-01")) + days(as.numeric(resultDT$DayofYear - 1)) + hours(resultDT$Hour)
  days_since_origin <- ymd_hms(dates) - ymd_hms("1700-01-01 00:00:00")
  if(!units(days_since_origin) == 'days') stop('check time units')
  t <- ncdim_def("time", "days since 1700-01-01", 
                 as.numeric(days_since_origin)) #define netCDF dimensions for variables
  if(exists('genus') & (genus == "Saccharum")){
    for(variable in c("Leaf", "Root", "Stem", "LAI", "DayofYear")) {
      x <- result[[variable]]
      result[[variable]] <- c(x[1], rep(x[-1], 24, each = TRUE))
    }
  }
  
  vars <- list()
  
  c2biomass <- 0.4
  vars <- list(
    TotLivBiom  = mstmipvar("TotLivBiom", x, y, t),
    RootBiom    = mstmipvar("RootBiom", x, y, t),
    StemBiom    = mstmipvar("StemBiom", x, y, t),
    Evap        = mstmipvar("Evap", x, y, t),
    TVeg        = mstmipvar("TVeg", x, y, t),
    LAI         = mstmipvar("LAI", x, y, t))
  
  k <- ud.convert(1, "Mg/ha", "kg/m2") / c2biomass
  
  results <- with(resultDT, list(
    TotLivBiom = k * (Leaf + Root + Stem + Rhizome + Grain),
    RootBiom = k * Root,
    StemBiom = k * Stem,
    Evap = ud.convert(SoilEvaporation + CanopyTrans, "Mg/ha/h", "kg/m2/s"),
    TVeg = ud.convert(CanopyTrans, "Mg/ha/h", "kg/m2/s"),
    LAI =  LAI))
  
  nc <- nc_create(filename = file.path(outdir, "result.nc"),
                  vars = vars)
  varfile <- file(file.path(outdir, paste("result", "nc", "var", sep=".")), "w")
  
  ## Output netCDF data
  for(.vname in names(vars)) {
    # make sure only floats are in array
    x <- which(results[[.vname]]< -1e100)
    if (length(x) > 0) {
      logger.debug(.vname, "found", length(x), "values < -1e100")
      results[[.vname]][[x]] <- NA  
    }
    x <- which(results[[.vname]] > 1e100)
    if (length(x) > 0) {
      logger.debug(.vname, "found", length(x), "values > 1e100")
      results[[.vname]][[x]] <- NA  
    }
    # write results
    ncvar_put(nc, varid = vars[[.vname]], vals = results[[.vname]])
    cat(paste(vars[[.vname]]$name, vars[[.vname]]$longname), file=varfile, sep="\n")
  }
  
  ncatt_put(nc, 0, "description",
            "This is an output from the BioCro Crop model generated by the model2netcdf.BIOCRO.R function in the PEcAn.BIOCRO package; see github.com:PecanProject/pecan/wiki for more information")
  close(varfile)
  nc_close(nc)
  
}

####################################################################################################
### EOF.  End of R script file.              
####################################################################################################
