function [confint,themean,confint2] = ConfInt_TDist(data,interval,distribution)
%CONFINT Calculate confidence interval based on the 
% t-distribution (for small samples sizes)
%
%   [confint,themean,confint2] = ConfInt_TDist(data)
% 
% Returns half of the (symmetric) 95% confidence interval
% about the mean of each row of data. 
%
% The function also allows specification of the confidence 
% level and distribution, and returns the means and absolute 
% values of the upper and lower confidence limits
%
%   [confint,themean,confint2] = ConfInt(data,interval)
%
% INPUTS:
%   -data: the data values (stimuli in rows, observations in columns)
%   -interval: calculate the [interval] confidence interval (between 0
%              and 1), default: 0.95
%   -distribution: allows specification of the distribution used to 
%                  calculate the CI: 'normal' or 'tdist'. Default
%                  depends on the number of samples.
%
% OUTPUTS:
%   -confint: the size of the upper and lower confidence intervals
%   (symmetric)
%   -themean: the mean values
%   -confint2: the upper and lower confidence values of themean
%
%
% Jon Francombe, 22/10/2012
% j.francombe@surrey.ac.uk
%
% SEE ALSO tinv, mean, std
%

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


%%%%%%%%%%%%%%%% ERROR CHECKS AND DEFAULTS %%%%%%%%%%%%%%%%%

% Set default confidence interval + distribution
if nargin < 3
    if size(n,2) < 30
        distribution = 'tdist';
    else 
        distribution = 'normal';
    end
end

if nargin < 2
    interval = 0.95;
end

% Only allow 'normal' and 'tdist'
if ~strcmp(distribution,'normal') && ~strcmp(distribution,'tdist')
    error('Specified distribution must be ''normal'' or ''tdist''')
end

% Make sure confidence interval is > 0 and < 100
if interval <= 0 || interval >= 100
    error('Specified confidence interval must fall between 0 and 100')
end

% Warn if using the wrong distribution
if strcmp(distribution,'tdist') && size(data,2) > 30
    warning('t-distribution used to calculate confidence intervals, but n>30 so could use normal distribution instead')
elseif strcmp(distribution,'normal') && size(data,2) <= 30
    warning('30 or fewer data points; t-distribution should be used for calculating confidence interval')
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Number of data points
n = size(data,2);

% Mean
themean = mean(data,2);

% Standard deviation
stdev = std(data,0,2);          % flag = 0 (use n-1), dimension = 2

% Standard error
sterr = stdev/sqrt(n);

% One-sided interval
oneside = interval + ((1-interval)/2);

switch distribution
    
    case 'tdist'
        % Use the t-distribution        
        stat = tinv(oneside,n-1);      % Upper
        % upper = tinv(((1-interval)/2),n-1);       % Lower: but it's symmetric
        
    
    case 'normal'
        % Use the standard normal distribution
        stat = norminv(oneside,0,1);
        
end
        
% Symetric lower & upper CIs
confint = (sterr * stat);

% Lower confidence interval
confint2(:,1) = themean - confint;
% Upper confidence interval
confint2(:,2) = themean + confint;


