!-------------------------------------------------------------------------
!         NASA/GSFC, Data Assimilation Office, Code 910.3, GEOS/DAS      !
!-------------------------------------------------------------------------
!BOP
!
! !MODULE:  m_jacobian --- Tests agreement of tlm with adm using Jacobian
!
! !INTERFACE:
!
module m_jacobian

! !USES:

   use precision
   use prognostics, only : imr
   use prognostics, only : jnp
   use prognostics, only : nl
   use prognostics, only : prognostics_initial
   use prognostics, only : prognostics_final
   use prognostics, only : dyn_prog

   use stepon,  only : stepon_set
   use stepon,  only : stepon_finalize
   use control, only : mod2cont
   
   use m_ioutil, only : luavail
   use m_stdio,  only : stdout
   use m_stdio,  only : stderr

   use m_inpak90
   use m_die
#if defined( SPMD )
#define CPP_PRT_PREFIX  if(gid == 0)
      use mod_comm,  only : gid
#else
#define CPP_PRT_PREFIX
#endif
   
  implicit none

!
! !PUBLIC MEMBER FUNCTIONS:
!
  PRIVATE 

  PUBLIC jacobian_test_main

!
! !DESCRIPTION: This module computes tests tlm versus adm code by  
!               computing colums of the Jacobian matrices corresponding
!               to each operator. Since a column of the adjoint Jacobian
!               should be identical to a row of the tlm Jacobian, since 
!               the interstion of any row with any column is a single 
!               value, that value should be the same from each output.
!
!               The column outputs are generated by inputing unit vectors. 
!
!
! !REVISION HISTORY:
!
!  20Mar2003 Errico   Initial design/interfaces.
!  27Mar2003 Todling  Redefined myname & defined myname_
!  20Oct2005 Errico   Set proper accounting for pole values
!  22Nov2006 Oloso    Protect prints with CPP directive
!  13May2007 Todling  Introduced dyn_prog
!
!EOP
!-------------------------------------------------------------------------

  character(len=*), parameter :: myname   = 'm_jacobian'
  character(len=*), parameter :: filetemp = 'jac_test_temp'
  character(len=*), parameter :: JACRC    = 'jactest.rc'

  logical, parameter  :: verbose = .true.

  integer             :: unit0          ! unit for temp file
  integer, parameter  :: nfields_def=5  ! default number of fields in control vector   
  integer, parameter  :: npoints_def=6  ! default number of points to test
  integer, parameter  :: ntests_def=nfields_def*npoints_def 
  character(len=nfields_def), parameter :: field_names_def='UVTQP'
  integer, allocatable, save :: pertindex(:,:)
  integer,              save :: ntests
  integer,              save :: nfields


  CONTAINS

! 
!-------------------------------------------------------------------------
!         NASA/GSFC, Data Assimilation Office, Code 910.3, GEOS/DAS      !
!-------------------------------------------------------------------------
!BOP
!
! !ROUTINE: jacobian_test_main --main subroutine that calls testing routines 
!
! !INTERFACE:
   
  subroutine jacobian_test_main ( n, m )  

! !USES:

! !INPUT PARAMETERS:
!  none   

! !INPUT/OUTPUT PARAMETERS:

   integer, intent(in) :: n
   integer, intent(in) :: m

! !OUTPUT PARAMETERS:
!  none

! !DESCRIPTION: This routine calls the initial setup, partial Jacobian
!               generator, and post processor for comparing TLM and ADM 
!               codes using the Jacobian method. 
!
! !REMARKS
!
! !REVISION HISTORY:
!
!  20Mar2003  Errico    Initial code 
!
!EOP
!-------------------------------------------------------------------------


  call jacobian_init
  call jacobian_tlmadm ( n, m )
  CPP_PRT_PREFIX call jacobian_processor
  call jacobian_clean

  end subroutine jacobian_test_main





!-------------------------------------------------------------------------
!         NASA/GSFC, Data Assimilation Office, Code 910.3, GEOS/DAS      !
!-------------------------------------------------------------------------
!BOP
!
! !ROUTINE: jacobian_init --- specify points and fields to test
!
! !INTERFACE:
  subroutine jacobian_init

! !USES:

! !INPUT PARAMETERS:
!  none   

! !INPUT/OUTPUT PARAMETERS:
!  none

! !OUTPUT PARAMETERS:
!  none

!
! !DESCRIPTION:  specify the specific set of grid points and fields 
!                to be tested by the Jacbian technique.  The specified set
!                will be tested for both input and output to both TLM and
!                ADM codes, resulting in ntests**2 unique pairs of 
!                variables to be compared.
!
!                The four values of pertindex (for each test point) are:
!                (1) longitudinal index
!                (2) latitudinal  index
!                (3) vertical level index
!                (4) field number index (see character field_names)
!
! !REMARKS       It is important to check any unique points, such as the poles,
!                next to the poles, and maybe the equator.  The matching will
!                only work if the poles are treated in particular ways, 
!                although there are more than ony way to treat them.  
!                See the routine jacobian_pole for an explanation of what
!                is required for the current (Oct 2005) versions of the TLM
!                and ADM codes.
!
!
! !REVISION HISTORY:
!
!  20Mar2003  Errico    Initial code 
!
!EOP
!-------------------------------------------------------------------------
  integer             :: np, nf, nt
  integer             :: jnph, jnpm1
  integer	      :: ierr

  call RC_jac_points_ (ntests,nfields,ierr)
     if(ierr==0) return

  ntests  = ntests_def
  nfields = nfields_def
  if ( .not. allocated(pertindex) ) then
     allocate ( pertindex(4,ntests) )
  else
     deallocate ( pertindex )
     allocate   ( pertindex(4,ntests) )
  endif

  jnph=jnp/2
  jnpm1=jnp-1
 
  nt=0
  do np=1,npoints_def
    do nf=1,nfields
      nt=nt+1
      pertindex(4,nt)=nf
      if (np==1) pertindex(1:3,nt)= (/1,jnph,10/) 
      if (np==2) pertindex(1:3,nt)= (/imr,jnph,nl/) 
      if (np==3) pertindex(1:3,nt)= (/1,1,nl/) 
      if (np==4) pertindex(1:3,nt)= (/1,2,10/) 
      if (np==5) pertindex(1:3,nt)= (/1,jnp,nl/) 
      if (np==6) pertindex(1:3,nt)= (/1,jnpm1,10/) 
    enddo
  enddo
  if (nt /= ntests) stop 'nt /= ntests_def in jacobian_init'

  end subroutine jacobian_init

!-------------------------------------------------------------------------
!         NASA/GSFC, Data Assimilation Office, Code 910.3, GEOS/DAS      !
!-------------------------------------------------------------------------
!BOP
!
! !IROUTINE: RC_jac_points_  --- Read 'jac_points_to_test::' table from the
!                                resource file
!
! !INTERFACE:
!
      subroutine RC_jac_points_ (nt,nflds,stat)

      integer, intent(out)  :: nt 
      integer, intent(out)  :: nflds 
      integer, intent(out)  :: stat 

!
! !REVISION HISTORY:
!
!  17Feb04 Todling  Converted from pre-existing code (m_sqc)
!
!EOP
!-------------------------------------------------------------------------

      character(len=*), parameter :: myname    = 'RC_ktinfo_'
      character(len=*), parameter :: tablename = 'jac_points_to_test::'

      integer, allocatable :: mypoints(:,:)

      character(len=255) token
      integer iret, nf, np

!     Initialize table entries
!     ------------------------
      stat  = 0
      np    = 0
      nt    = 0
      iret  = 0
      nflds = nfields_def  ! for now, no. of fields to test is default

!     Load resources
!     --------------
      call i90_loadf (JACRC, iret)
      if (iret/=0) then
         write(stderr,'(2a,i5,2a)') myname, ': I90_loadf error, iret =', iret, &
                                            ': trying to load ', JACRC
         stat = 99
         return
      end if

!     Read table with lat/lon/lev indeces for test
!     First read to determine total number of pnts
!     --------------------------------------------
      call I90_label(tablename, iret)
      if (iret/=0) then
         write(stderr,'(2a,i5,2a)') myname, ': I90_label error, iret=', iret, &
                                         ': trying to read ', tablename
         call I90_release()
         stat = 98
         return
      end if
      do while (iret==0)     ! read 1st entry of each column to get total entries
         call I90_GLine ( iret ) ! iret=-1: end of file; +1: end of table
             if (iret==0) np = np + 1
      end do
      allocate ( mypoints(4,np), pertindex(4,np*nflds), stat=iret )   ! allocate index array
	if(iret/=0) call die(myname,'alloc(mypoints)',iret)

!     Read table again, now to get lat/lon/lev indeces
!     -------------------------------------------------
      call I90_label(tablename, iret)
      if (iret/=0) then
         write(stderr,'(2a,i5,2a)') myname, ': I90_label error, iret=', iret, &
                                         ': trying to read ', tablename
         call I90_release()
         stat = 98
         return
      end if
      np   = 0
      iret = 0
      do while (iret==0)     ! read table entries

         call I90_GLine ( iret ) ! iret=-1: end of file; +1: end of table
         if (iret==0) then       ! OK, we have next row of table
             np = np + 1

             mypoints(1,np) = I90_GInt( iret )    ! longitude
             if (iret/=0) then
                 write(stderr,'(2a,i5)') myname, ': I90_GInt error (lon), iret=', iret
                 call die(myname)
             end if
             mypoints(2,np) = I90_GInt( iret )    ! latitude
             if (iret/=0) then
                 write(stderr,'(2a,i5)') myname, ': I90_GInt error (lon), iret=', iret
                 call die(myname)
             end if
             mypoints(3,np) = I90_GInt( iret )    ! level
             if (iret/=0) then
                 write(stderr,'(2a,i5)') myname, ': I90_GInt error (lon), iret=', iret
                 call die(myname)
             end if

             CPP_PRT_PREFIX write(stdout,'(a,3(a,i3))') myname, &
                         ' lon_indx = ', mypoints(1,np),      &
                         ' lat_indx = ', mypoints(2,np),      &
                         ' lev_indx = ', mypoints(3,np)
             call flush(6)
         end if

      end do
      nt = 0
      do np=1,np
        do nf=1,nflds
          nt=nt+1
          pertindex(1:3,nt) = mypoints(1:3,np)
          pertindex(4,nt)   = nf
        end do
      end do

      deallocate ( mypoints )

!     release resource file:
!     ---------------------
      call I90_release()

      end subroutine RC_jac_points_


!-------------------------------------------------------------------------
!         NASA/GSFC, Data Assimilation Office, Code 910.3, GEOS/DAS      !
!-------------------------------------------------------------------------
!BOP
!
! !ROUTINE: jacobian_tlmadm --- run tlm and adm to make partial jacobian
!
! !INTERFACE:
  subroutine jacobian_tlmadm ( n, m ) 

! !USES:

! !INPUT PARAMETERS:
!  none   

! !INPUT/OUTPUT PARAMETERS:

   integer, intent(in) :: n
   integer, intent(in) :: m

! !OUTPUT PARAMETERS:
!  none

!
! !DESCRIPTION: Unit vectors of tlm and adm input are created based
!               based on a selected set of points and fields to be tested.
!               Each in turn is input into the tlm and adj executions, 
!               after that input is modified to ensure that any pole
!               values are specified properly (all equal). The output of
!               both the tlm and adm is also modified to ensure proper
!               pole specification of values, since this is not assumed to
!               necessarily occur in the tlm, adm, or adiabatic FVDAS. 
!               Output of the tlm and adm at the same selected points
!               are written to a sequential file, to be read and 
!               reorganized by the jacobian post processor. 
!                
!
! !REMARKS
!
! !REVISION HISTORY:
!
!  20Mar2003  Errico    Initial code 
!  27Mar2003  Todling   Turn (n,m) in to input arguments (def. earlier)
!
!EOP
!-------------------------------------------------------------------------

  character(len=*), parameter :: myname_ = myname//'m_jacobian'
  integer  :: nloc
  type(dyn_prog) :: prog
  type(dyn_prog) :: prog_tl
  type(dyn_prog) :: prog_ad

  real(r8) x(n)      ! independent variables (initial reference state)
  real(r8) y(m)      ! dependent variables (final reference state)
  real(r8) xpert(n)  ! in adm, this is grad(J) with respect to x
  real(r8) ypert(m)  ! in adm, this is grad(J) with respect to y
  real(r8) f(imr,jnp,nl,nfields) ! re-shapped xpert or ypert
 
  integer    nt


  nloc=imr*jnp*nl*nfields ! expected size of state vector
  if ( n/=nloc ) call MP_die(myname_, 'unexpected state vector dim',99)
  call stepon_set ( prog )        ! read in (initial nonlinear) state
  call mod2cont ( n, x, prog )    ! copy model arrays to long vec x

  call prognostics_initial ( prog_tl )
  call prognostics_initial ( prog_ad )

  unit0= luavail()

  CPP_PRT_PREFIX open (unit=unit0, file= filetemp, form='unformatted')

  do nt=1,ntests  ! loop over selected grid points and fields to test
!
! first run TLM for one selcted perturbation
    f=0.d0
    f(pertindex(1,nt),pertindex(2,nt),pertindex(3,nt),pertindex(4,nt))=1.d0
    CPP_PRT_PREFIX print ('(a,5i5)'), 'pert TLM begins for ', pertindex(1:4,nt), nt 
    call jacobian_pole (f,3,5,'sum')
    xpert=reshape(f(:,:,:,:),(/n/))
    ypert=0.d0
!
!_RTcall func_tl( n, x, xpert, m, y, ypert, prog, prog_tl )   
!
    f=reshape(ypert(:),(/imr, jnp, nl, nfields/))
    call jacobian_pole (f,3,5,'averg')
    CPP_PRT_PREFIX call jacobian_wtemp (f,nt,'TOUT')
!
! fnext run ADM for one selcted perturbation
    f=0.d0
    f(pertindex(1,nt),pertindex(2,nt),pertindex(3,nt),pertindex(4,nt))=1.d0
    CPP_PRT_PREFIX print ('(a,5i5)'), 'pert ADM begins for ', pertindex(1:4,nt), nt 
    call jacobian_pole (f,3,5,'averg')
    ypert=reshape(f(:,:,:,:),(/n/))
    xpert=0.d0
!
!_RTcall func_ad( n, x, xpert, m, y, ypert ) 
!
    f=reshape(xpert(:),(/imr, jnp, nl, nfields/))
    call jacobian_pole (f,3,5,'sum')
    CPP_PRT_PREFIX call jacobian_wtemp (f,nt,'AOUT')

  enddo ! loop over nt
!
  CPP_PRT_PREFIX close (unit=unit0)
  CPP_PRT_PREFIX print *,' '
  nt=nt-1  ! remove additional increment produced at end of do loop
  CPP_PRT_PREFIX print *,'temporary Jacobian file created for',nt,' test vectors'

  call prognostics_final ( prog_ad )
  call prognostics_final ( prog_tl )
  call stepon_finalize   ( prog )

  end subroutine jacobian_tlmadm 



!-------------------------------------------------------------------------
!         NASA/GSFC, Data Assimilation Office, Code 910.3, GEOS/DAS      !
!-------------------------------------------------------------------------
!BOP
!
! !ROUTINE: jacobian_wtemp -- write 1 record of temporary file
!
! !INTERFACE:
  subroutine jacobian_wtemp (f,nt,type)

! !USES:

! !INPUT PARAMETERS:
  integer, intent(in)          :: nt
  character(len=*), intent(in) :: type
  real(r8), intent(in)         :: f(imr,jnp,nl,nfields)

! !INPUT/OUTPUT PARAMETERS:
!  none

! !OUTPUT PARAMETERS:
!  none

!
! !DESCRIPTION: write output for selected points, in the order they 
!               are generated by tlm or adm executions, to a file
!
! !REMARKS
!
! !REVISION HISTORY:
!
!  20Mar2003  Errico    Initial code 
!
!EOP
!-------------------------------------------------------------------------

  integer n1

  write (unit0) type, nt,                                               &
    (f(pertindex(1,n1),pertindex(2,n1),pertindex(3,n1),pertindex(4,n1)) &
     ,n1=1,ntests)

  end subroutine jacobian_wtemp



!-------------------------------------------------------------------------
!         NASA/GSFC, Data Assimilation Office, Code 910.3, GEOS/DAS      !
!-------------------------------------------------------------------------
!BOP
!
! !ROUTINE: jacobian_pole --- correct input/output pole points
!
! !INTERFACE:
  subroutine jacobian_pole (f,nfirst,nlast,type)

! !USES:

! !INPUT PARAMETERS:
  integer, intent(in)          :: nfirst
  integer, intent(in)          :: nlast
  character(len=*), intent(in) :: type

! !INPUT/OUTPUT PARAMETERS:
  real(r8), intent(inout)      :: f(imr,jnp,nl,nfields)

! !OUTPUT PARAMETERS:
!  none

!
! !DESCRIPTION: depending on the value of argument type:
!               'copy1' : copy pole values from first (i=1) value
!               'averg' : set all pole values equal to averge of pole values
!               'none'  : do nothing 
!               'sum'   : set all pole values equal to sum of pole values
!
! !REMARKS:
!     1) 'copy1' only copies the pole value for the field at i=1 into those 
!        at other values of i. The former settings of those other values are 
!        therefore ignored. If an initial perturbation is introduced at one 
!        of those other values but not for i=1, the result will be 0 for all
!        values. This option can confuse the test's interpretation.
!     2) If the pert test is conducted by setting a value=1 at the pole for
!        only one value of i, the option 'sum' will set all values to 1
!        when any pole point is specified. The option 'avg' on the other hand
!        will set all values equal to 1/imr.
!     3) With respect to the TLM output, the options 'none' and 'averg'
!        will result in identcal results (except for round-off issues).
!        The TLM output as of Oct 2005 already has the condition that
!        pole values for all index i are identical.
!     4) As of Oct 2005, the ADM output does not satisfy the condition that
!        pole values for all index i are identical.  Two possible options
!        are that all should equal the averaged (over i=1,...,imr) value
!        of the output, or the sum of the output. These options differ by
!        a factor of imr. 
!     5) If the pole is considered as a single grid 
!        box, having the area of the pole cap, the proper value is the sum
!        of the ADM output pole values. In this case, any use of the ADM 
!        output as a sensitivity gradient for computing a J'=dJ/dT T' 
!        contribution by the pole should not sum contributions from all
!        index i, since they are all redundant. The pole value considered 
!        this way will tend to create a discontinuity at the pole when
!        displayed on the model grid, since values at points adjacent to 
!        the pole will tend to diminish proportionally to 1/imr, as each
!        non-pole grid box represents a smaller area. 
!     6) If the pole is considered to be imr independent sectors, then 
!        setting the pole values equal to the average of imr the pole 
!        values is appropriate. This is the same as if the pole were
!        considered as a cap defined as one point, except the values 
!        would be imr smaller in the case of the adjoint output. For the 
!        ADM, by treating the pole as imr independent sectors, the values 
!        at the pole scale the same as non pole values as imr is varied.
!        So, using 'averg' for the output appears most appropriate. Only
!        using 'sum', however, will result in a match with the TLM output
!        to pass the Jacobian test. 
!     7) The TLM apparently creates output where the imr pole values
!        are all identical.  There is nothing internal to the TLM that
!        ensures it begins with identical values.  So, the ADM has no
!        assurance that its output has such a property, although each time
!        step begins with its enforecement. 
!     8) The current  version (Oct 2005) of the SV-norms software for the
!        total energy norm treats the pole as a single point, as it must 
!        because all components of the control vector must be independent.
!        It the uses 'copy1' (equivalent to 'sum' if all other values are
!        zero) to initialize the TLM, with 'averg' applied to the output.
!        It then uses 'averg' to initialize the ADM, with 'sum' applied to
!        the output.  This is one of the same combinations required here 
!        to get the TLM and ADM to match.   
!                
!
! !REVISION HISTORY:
!
!  20Mar2003  Errico    Initial code 
!
!EOP
!-------------------------------------------------------------------------

  integer   :: i,k,nf
  real(r8)  :: sumsp, sumnp      

!   
  if (type == 'copy1') then

    do nf=nfirst,nlast ! operate only on scalar fields
      do k=1,nl
        f(2:imr,  1,k,nf)=f(1,  1,k,nf)
        f(2:imr,jnp,k,nf)=f(1,jnp,k,nf)
      enddo
    enddo

  elseif (type == 'averg') then

    do nf=nfirst,nlast  ! operate only on scalar fields
      do k=1,nl
        sumsp=sum(f(1:imr,  1,k,nf))/imr
        sumnp=sum(f(1:imr,jnp,k,nf))/imr
        f(1:imr,  1,k,nf)=sumsp
        f(1:imr,jnp,k,nf)=sumnp
      enddo  ! loop over k
    enddo    ! loop over nf

  elseif (type == 'sum') then

    do nf = nfirst, nlast
      do k = 1, nl
        sumsp=sum(f(1:imr,  1,k,nf))
        sumnp=sum(f(1:imr,jnp,k,nf))
        f(1:imr,  1,k,nf)=sumsp
        f(1:imr,jnp,k,nf)=sumnp
     end do
   end do

  endif
          
  end subroutine jacobian_pole  



!-------------------------------------------------------------------------
!         NASA/GSFC, Data Assimilation Office, Code 910.3, GEOS/DAS      !
!-------------------------------------------------------------------------
!BOP
!
! !ROUTINE: jacobian_processor --- write processed output to file
!
! !INTERFACE:
  subroutine jacobian_processor

! !USES:

! !INPUT PARAMETERS:
!  none   

! !INPUT/OUTPUT PARAMETERS:
!  none

! !OUTPUT PARAMETERS:
!  none

!
! !DESCRIPTION: 
!
! !REMARKS
!
! !REVISION HISTORY:
!
!  20Mar2003  Errico    Initial code 
!
!EOP
!-------------------------------------------------------------------------
!    

   character(len=4)  :: type           ! indicates either adm or tlm output
   character(len=2)  :: field_name_1   ! field name (TIN or AOUT)
   character(len=2)  :: field_name_2   ! field name (AIN or TOUT)
   character(len=14) :: field_point_1  ! indeces of grid point (TIN or AOUT)
   character(len=14) :: field_point_2  ! indeces of grid point (AIN or TOUT)
   character(len=1)  :: flag           ! blank if no error detected
   
   integer :: digits    
   integer :: n1        ! tlm input or adm output index
   integer :: n2        ! adm input or tlm output index
   integer :: nread     ! counter for reading temp file records
   integer :: numerrs   ! counter for number of possible errors indentified 
   integer :: maxdigits ! maximum number of digits agreeing for diff /= 0
   integer :: mindigits ! minimum number of digits agreeing for diff /= 0
   integer :: ntests2   ! number of  pairs tested

   real(r8) :: tlmout(ntests,ntests)
   real(r8) :: admout(ntests,ntests)
   real(r8) :: diff
!
! read tlm and adm output from file
      unit0= luavail()
      open (unit=unit0,file=filetemp,form='unformatted')
      do nread=1,ntests
        read (unit0) type,n1,tlmout(1:ntests,n1)
        read (unit0) type,n2,admout(1:ntests,n2)
      enddo
      close (unit0)
!
! initialize some output values
      numerrs=0
      maxdigits=0
      mindigits=100
      field_name_1(1:1)=' '
      field_name_2(1:1)=' '
      
!
! write out file explanation
      ntests2=ntests*ntests 
      write (stdout,('(a,i6,a)')) ' Testing of adm tlm for ',ntests2, &
                          ' pairs. Explanation of output follows:'
      write (stdout,('(a)')) ' 1. input tlm, output adm field name' 
      write (stdout,('(a)')) ' 2. input adm, output tlm field name' 
      write (stdout,('(a)')) ' 3. input tlm, output adm grid-point'
      write (stdout,('(a)')) ' 4. input adm, output tlm grid-point' 
      write (stdout,('(a)')) ' 5. tlm output value'
      write (stdout,('(a)')) ' 6. adm output value'
      write (stdout,('(a,a)')) ' 7. number of identical digits',    &
                              ' (-1 means one zero but not other)'
      write (stdout,('(a)')) ' 8. flag: blank ok; 1, 2, or * suspect'
      write (stdout,('(a)')) ' '

!  n1 is the index for the tlm input field perturbed or adm output field
!  n2 is the index for the adm input field perturbed or tlm ouput field
!  write results for all pair of test values

      do n1=1,ntests
        field_name_1(2:2) = field_names_def(pertindex(4,n1):pertindex(4,n1))
        write (field_point_1,('(a2,i3,a1,i3,a1,i3,a1)') ) &
    ' (', pertindex(1,n1), ',', pertindex(2,n1), ',', pertindex(3,n1), ')'
 
        do n2=1,ntests 
          field_name_2(2:2) = field_names_def(pertindex(4,n2):pertindex(4,n2))
          write (field_point_2,('(a2,i3,a1,i3,a1,i3,a1)') ) &
    ' (', pertindex(1,n2), ',', pertindex(2,n2), ',', pertindex(3,n2), ')'

          call jacobian_flag (flag,digits,diff,tlmout(n2,n1),admout(n1,n2))
          if (digits < 99 ) maxdigits = max(maxdigits,digits)
          if (digits >= 0 ) mindigits = min(mindigits,digits)
          if (flag /= ' ') numerrs=numerrs+1
          write (stdout,'(2a2,2a14,1p2e20.12,i3,2x,a1)')               & 
            field_name_1, field_name_2, field_point_1, field_point_2, &
            tlmout(n2,n1), admout(n1,n2), digits, flag 
        enddo
      enddo

      write (stdout,('(a,i5)'))                                     &
        ' Maximum number of digits found for non-zero diffs is ',  &
                maxdigits
      write (stdout,('(a,i5)'))                                     &
        ' Minimum number of digits found for non-zero values is ', &
                mindigits
      write (stdout,('(a,i5)'))                                     &
        ' Numbers of possible errors flagged is ',numerrs   

    end  subroutine jacobian_processor


!-------------------------------------------------------------------------
!         NASA/GSFC, Data Assimilation Office, Code 910.3, GEOS/DAS      !
!-------------------------------------------------------------------------
!BOP
!
! !ROUTINE: jacobian_flag --- routine to compare 2 jacobian values 
!
! !INTERFACE:
      subroutine jacobian_flag (flag,digits,diff,f1,f2)

! !USES:

! !INPUT PARAMETERS:
      real(r8), intent(in)  :: f1
      real(r8), intent(in)  :: f2

! !INPUT/OUTPUT PARAMETERS:
!  none

! !OUTPUT PARAMETERS:
      character(len=1), intent(out) :: flag      
      integer,          intent(out) :: digits
      real(r8),         intent(out) :: diff

!
! !DESCRIPTION: 
!
! !REMARKS
!
! !REVISION HISTORY:
!
!  20Mar2003  Errico    Initial code 
!
!EOP
!-------------------------------------------------------------------------
 
      real(r8), parameter   :: dzero=0.d0
      real(r8), parameter   :: mdigits=9
!
! Pairs of values with less than mdigits+1 agreement are flagged as
! suspected errors. The value 9 has been chosen from experience.
! It appears that differences can occur in the 10th digit for some pairs
! of values when the code is actually correct, presumably due to 
! differences in round-off.
!
      real(r8)              :: asum
      real(r8)              :: ratio

      diff=f1-f2  
       
      if (diff == dzero) then
        digits=99
        flag=' ' 
      elseif ( (f1 /= dzero) .and. (f2 == dzero) )then
        digits=-1  
        flag='1'
      elseif ( (f1 == dzero) .and. (f2 /= dzero) )then
        digits=-2  
        flag='2'
      else
        asum=0.5*(abs(f1)+abs(f2)) 
        ratio= abs(diff)/asum
        digits=-log10(ratio)
        if (digits > mdigits) then
          flag=' '
        else
          flag='*'
        endif
      endif  

end subroutine jacobian_flag
subroutine jacobian_clean
  deallocate (pertindex)
end subroutine jacobian_clean

end module m_jacobian


subroutine jacobian_test( n, m )
  use m_jacobian
  implicit none
  integer n, m
  call jacobian_test_main ( n, m )  
end subroutine jacobian_test
