!  $Id$ 

#include "MAPL_Generic.h"

!=============================================================================
module GEOS_CatchCNGridCompMod

!BOP
! !MODULE: GEOS_CatchCN --- ESMF gridded component implementing CatchmentCN LSM

! !DESCRIPTION:
!
!   {\tt Catch} is a gridded component to compute the energy and water
!   fluxes due to land-surface processes, using the Catchment LSM
!   of Koster et al. (2014). 
!   Koster, R. D., G. Walker, G. J. Collatz, and P. E. Thornton, 2014. 
!   Hydroclimatic controls on the means and variability of vegetation 
!   phenology and carbon uptake. J. Climate, 27, 5632-5652. doi: 
!   10.1175/JCLI-D-13-00477.1. 
!   All of its calculations are done
!   in a tile space defined by the inherited location stream.
!   It has a two-stage run method. The first stage obtains
!   drag coefficients at all the subtiles and defines
!   effective tile-mean surface quantities. The second
!   stage calls the Catchment-CN LSM. {\tt CatchCN} has no children.

!
! !USES:

  use sfclayer  ! using module that contains sfc layer code
  use ESMF
  use GEOS_Mod
  use GEOS_UtilsMod
  use DragCoefficientsMod
  use CATCHMENT_CN_MODEL
  use compute_rc_mod
  use CN_DriverMod
  USE STIEGLITZSNOW,   ONLY :                 &
       snow_albedo, StieglitzSnow_calc_tpsnow, N_CONSTIT,   &
       NUM_DUDP, NUM_DUSV, NUM_DUWT, NUM_DUSD, &
       NUM_BCDP, NUM_BCSV, NUM_BCWT, NUM_BCSD, &
       NUM_OCDP, NUM_OCSV, NUM_OCWT, NUM_OCSD, &
       NUM_SUDP, NUM_SUSV, NUM_SUWT, NUM_SUSD, &
       NUM_SSDP, NUM_SSSV, NUM_SSWT, NUM_SSSD, &
       StieglitzSnow_calc_asnow

  USE CATCH_CONSTANTS, ONLY :                 &
       N_GT           => CATCH_N_GT,          &
       N_SNOW         => CATCH_N_SNOW,        &
       RHOFS          => CATCH_SNWALB_RHOFS,  &
       SNWALB_VISMAX  => CATCH_SNWALB_VISMAX, &
       SNWALB_NIRMAX  => CATCH_SNWALB_NIRMAX, &
       SLOPE          => CATCH_SNWALB_SLOPE

  USE  clm_varpar, ONLY :                     &
       NUM_ZON, NUM_VEG, VAR_COL, VAR_PFT,    &
       CN_zone_weight, map_cat, firefac, numpft
 
  USE MAPL
  use MAPL_ConstantsMod,only: Tzero => MAPL_TICE, pi => MAPL_PI 
  use clm_time_manager, only: get_days_per_year, get_step_size
  use pftvarcon,        only: noveg
  USE lsm_routines,     ONLY : sibalb, catch_calc_soil_moist, irrigation_rate

implicit none
private

  include "netcdf.inc"

! !PUBLIC MEMBER FUNCTIONS:

public SetServices

!
!EOP

integer,parameter :: FSAT=1  !  Saturated subtile
integer,parameter :: FTRN=2  !  Transition subtile
integer,parameter :: FWLT=3  !  Wilting subtile
integer,parameter :: FSNW=4  !  Snowcover subtile

integer,parameter :: NUM_SUBTILES=4

! Vegetation type as follows:
!                  1:  BROADLEAF EVERGREEN TREES
!                  2:  BROADLEAF DECIDUOUS TREES
!                  3:  NEEDLELEAF TREES
!                  4:  GROUND COVER
!                  5:  BROADLEAF SHRUBS
!                  6:  DWARF TREES (TUNDRA)
!===================================================
!ALT: we currently use only 6 types (see above)
!     in the legacy code we used to have 8 
!     (or 10 with the sea and land ice) with
!     these additional entries
!                  7:  BARE SOIL
!                  8:  DESERT

integer           :: NUM_ENSEMBLE
integer,parameter :: NTYPS = MAPL_NUMVEGTYPES

real,   parameter :: HPBL           = 1000.
real,   parameter :: MIN_VEG_HEIGHT = 0.01
real,   parameter :: Z0_BY_ZVEG     = 0.13
real,   parameter :: D0_BY_ZVEG     = 0.66

! Emissivity values from Wilber et al (1999, NATA-TP-1999-209362)
! Fu-Liou bands have been combined to Chou bands (though these are broadband only)
! IGBP veg types have been mapped to Sib-Mosaic types 
! Details in ~suarez/Emiss on cerebus

real,   parameter :: EMSVEG(NTYPS) = (/ 0.99560, 0.99000, 0.99560, 0.99320, &
                                        0.99280, 0.99180 /)
real,   parameter :: EMSBARESOIL   =    0.94120
real,   parameter :: EMSSNO        =    0.99999

! moved SURFLAY from catchment.F90 to enable run-time changes for off-line system
! - reichle, 29 Oct 2010


! ROOTL import from GEOS_VegdynGridComp was disabled and brought the look up table 
! in order to obtain ROOTL for primary and secondary types.

!  map catchment type into PFT
!  ---------------------------
!PFT 	Description
! 0 	bare
! 1 	needleleaf evergreen temperate tree
! 2 	needleleaf evergreen boreal tree
! 3 	needleleaf deciduous boreal tree
! 4 	broadleaf evergreen tropical tree
! 5 	broadleaf evergreen temperate tree
! 6 	broadleaf deciduous tropical tree
! 7 	broadleaf deciduous temperate tree
! 8 	broadleaf deciduous boreal tree
! 9 	broadleaf evergreen temperate shrub
! 10 	broadleaf deciduous temperate shrub [moisture + deciduous]
! 11 	broadleaf deciduous temperate shrub [moisture stress only]
! 12 	broadleaf deciduous boreal shrub
! 13 	arctic c3 grass
! 14 	cool c3 grass [moisture + deciduous]
! 15 	cool c3 grass [moisture stress only]
! 16 	warm c4 grass [moisture + deciduous]
! 17 	warm c4 grass [moisture stress only]
! 18 	crop          [moisture + deciduous]
! 19 	crop          [moisture stress only]

! Catchment types and PFT mapping:
! 
! 1:  BROADLEAF EVERGREEN TREES  => 4,5
! 2:  BROADLEAF DECIDUOUS TREES  => 6,7,8
! 3:  NEEDLELEAF TREES           => 1,2,3
! 4:  GROUND COVER               => 13-19
! 5:  BROADLEAF SHRUBS           => 9,10,11
! 6:  DWARF TREES (TUNDRA)       => 12
! 7:  BARE SOIL                  => 0
! 8:  DESERT                     => 0
! 9:  ICE                        => n/a

! index map for CLM PFTs --> catchment veg types

! pchakrab: save the logical variable OFFLINE
! Internal state and its wrapper
type T_OFFLINE_MODE
   private
   integer :: CATCH_OFFLINE
end type T_OFFLINE_MODE
type OFFLINE_WRAP
   type(T_OFFLINE_MODE), pointer :: ptr=>null()
end type OFFLINE_WRAP

integer :: RUN_IRRIG, USE_ASCATZ0, Z0_FORMULATION, IRRIG_METHOD, AEROSOL_DEPOSITION, N_CONST_LAND4SNWALB
integer :: ATM_CO2, PRESCRIBE_DVG, SCALE_ALBFPAR,CHOOSEMOSFC
real    :: SURFLAY              ! Default (Ganymed-3 and earlier) SURFLAY=20.0 for Old Soil Params
                                !         (Ganymed-4 and later  ) SURFLAY=50.0 for New Soil Params
real    :: CO2
integer :: CO2_YEAR_IN          ! years when atmospheric carbon dioxide concentration increases, starting from 1850
real    :: DTCN                 ! Time step for carbon/nitrogen routines in CatchmentCN model (default 5400)
real    :: FWETC, FWETL
logical :: USE_FWET_FOR_RUNOFF

contains

!BOP

! !IROUTINE: SetServices -- Sets ESMF services for component
! !INTERFACE:

subroutine SetServices ( GC, RC )

! !ARGUMENTS:

    type(ESMF_GridComp),intent(INOUT) :: GC
    integer, optional,  intent(  OUT) :: RC

! !DESCRIPTION:
! This version uses GEOS\_GenericSetServices, overriding
! only the run method. It also relies on MAPL\_Generic to
! handle data services. 

!EOP
!
! ErrLog Variables

    character(len=ESMF_MAXSTR) :: Iam
    character(len=ESMF_MAXSTR) :: COMP_NAME
    integer                    :: STATUS

! Local Variables

    type(MAPL_MetaComp), pointer :: MAPL=>null()
    type(T_OFFLINE_MODE), pointer :: internal=>null()
    type(OFFLINE_WRAP) :: wrap
    integer :: OFFLINE_MODE
    integer :: RESTART
    character(len=ESMF_MAXSTR)              :: SURFRC
    type(ESMF_Config)                       :: SCF 

! Begin...
! --------

! Get my name and set-up traceback handle
! ------------------------------------------------------------------------------

    Iam='SetServices'
    call ESMF_GridCompGet ( GC, NAME=COMP_NAME, RC=STATUS )
    VERIFY_(STATUS)
    Iam=trim(COMP_NAME)//trim(Iam)

! pchakrab: Read CATCHMENT_OFFLINE from resource file and save
! it in the private internal state of the GridComp. It is a little
! unusual to read resource file in SetServices, but we need to know
! at this stage where we are running Catch in the offline mode or not

    allocate(internal, stat=status)
    VERIFY_(status)
    wrap%ptr => internal
    call ESMF_UserCompSetInternalState(gc, 'OfflineMode', wrap, status)

    call MAPL_GetObjectFromGC(gc, MAPL, rc=status)
    VERIFY_(status)
    call MAPL_GetResource ( MAPL, OFFLINE_MODE, Label="CATCHMENT_OFFLINE:", DEFAULT=0, RC=STATUS)
    VERIFY_(STATUS)
    wrap%ptr%CATCH_OFFLINE = OFFLINE_MODE

    call MAPL_GetResource ( MAPL, NUM_ENSEMBLE, Label="NUM_LDAS_ENSEMBLE:", DEFAULT=1, RC=STATUS)
    VERIFY_(STATUS)

    call MAPL_GetResource (MAPL, SURFRC, label = 'SURFRC:', default = 'GEOS_SurfaceGridComp.rc', RC=STATUS) ; VERIFY_(STATUS)   
    SCF = ESMF_ConfigCreate(rc=status) ; VERIFY_(STATUS)
    call ESMF_ConfigLoadFile(SCF,SURFRC,rc=status) ; VERIFY_(STATUS)

    call MAPL_GetResource (SCF, SURFLAY,             label='SURFLAY:',             DEFAULT=50.,     __RC__ )
    call MAPL_GetResource (SCF, Z0_FORMULATION,      label='Z0_FORMULATION:',      DEFAULT=4,       __RC__ )
    call MAPL_GetResource (SCF, USE_ASCATZ0,         label='USE_ASCATZ0:',         DEFAULT=0,       __RC__ )
    call MAPL_GetResource (SCF, RUN_IRRIG,           label='RUN_IRRIG:',           DEFAULT=0,       __RC__ )
    call MAPL_GetResource (SCF, IRRIG_METHOD,        label='IRRIG_METHOD:',        DEFAULT=0,       __RC__ )
    call MAPL_GetResource (SCF, CHOOSEMOSFC,         label='CHOOSEMOSFC:',         DEFAULT=1,       __RC__ )
    call MAPL_GetResource (SCF, USE_FWET_FOR_RUNOFF, label='USE_FWET_FOR_RUNOFF:', DEFAULT=.FALSE., __RC__ )
    
    if (.NOT. USE_FWET_FOR_RUNOFF) then
       call MAPL_GetResource (SCF, FWETC, label='FWETC:', DEFAULT= 0.02, __RC__ )
       call MAPL_GetResource (SCF, FWETL, label='FWETL:', DEFAULT= 0.02, __RC__ )
    else
       call MAPL_GetResource (SCF, FWETC, label='FWETC:', DEFAULT=0.005, __RC__ )
       call MAPL_GetResource (SCF, FWETL, label='FWETL:', DEFAULT=0.025, __RC__ )
    endif

    ! GOSWIM ANOW_ALBEDO 
    ! 0 : GOSWIM snow albedo scheme is turned off
    ! 9 : i.e. N_CONSTIT in Stieglitz to turn on GOSWIM snow albedo scheme 
    call MAPL_GetResource (SCF, N_CONST_LAND4SNWALB, label='N_CONST_LAND4SNWALB:', DEFAULT=0  , __RC__ )

    ! Get parameters to zero the deposition rate 
    ! 1: Use all GOCART aerosol values, 0: turn OFF everythying, 
    ! 2: turn off dust ONLY,3: turn off Black Carbon ONLY,4: turn off Organic Carbon ONLY
    ! __________________________________________
    call MAPL_GetResource (SCF, AEROSOL_DEPOSITION, label='AEROSOL_DEPOSITION:' , DEFAULT=0  , __RC__ )

    ! CATCHCN
    call MAPL_GetResource (SCF, DTCN, label='DTCN:', DEFAULT=5400. , __RC__ )
    ! ATM_CO2
    ! 0: uses a fix value defined by CO2
    ! 1: CT tracker monthly mean diurnal cycle
    ! 2: CT tracker monthly mean diurnal cycle scaled to match EEA global average CO2
    ! 3: spatially fixed interannually varyiing CMIP from getco2.F90 look up table (AGCM only)
    ! 4: import AGCM model CO2 (AGCM only)
    call MAPL_GetResource (SCF, ATM_CO2, label='ATM_CO2:', DEFAULT=2  , __RC__ )

    ! PRESCRIBE_DVG: Prescribe daily LAI and SAI data from an archived CATCHCN simulation 
    ! 0--NO Run CN Model interactively
    ! 1--YES Prescribe interannually varying LAI and SAI
    ! 2--YES Prescribe climatological LAI and SAI
    ! 3--Estimated LAI/SAI using anomalies at the beginning of the foeecast and climatological LAI/SAI
    call MAPL_GetResource (SCF, PRESCRIBE_DVG, label='PRESCRIBE_DVG:', DEFAULT=0  , __RC__ )

    ! SCALE_ALBFPAR: Scale CATCHCN ALBEDO and FPAR
    ! 0-- NO scaling is performed
    ! 1-- Scale albedo to match interannually varying MODIS NIRDF and VISDF anomaly
    ! 2-- Scale albedo to match CDFs of model fPAR to MODIS CDFs of fPAR 
    ! 3-- Pefform above both 1 and 2 scalings 
    call MAPL_GetResource (SCF, SCALE_ALBFPAR, label='SCALE_ALBFPAR:', DEFAULT=0  , __RC__ )

    ! Global mean CO2 
    call MAPL_GetResource (SCF, CO2,         label='CO2:',      DEFAULT=350.e-6, __RC__ )
    call MAPL_GetResource (SCF, CO2_YEAR_IN, label='CO2_YEAR:', DEFAULT=  -9999, __RC__ )
    call ESMF_ConfigDestroy(SCF, __RC__)

! Set the Run entry points
! ------------------------

    call MAPL_GridCompSetEntryPoint ( GC, ESMF_METHOD_RUN, RUN1, RC=STATUS )
    VERIFY_(STATUS)
    call MAPL_GridCompSetEntryPoint ( GC, ESMF_METHOD_RUN, RUN2, RC=STATUS )
    VERIFY_(STATUS)


! Set the state variable specs.
! -----------------------------

!BOS

!  !IMPORT STATE:

    call MAPL_AddImportSpec(GC                         ,&
         LONG_NAME          = 'surface_pressure'            ,&
         UNITS              = 'Pa'                          ,&
         SHORT_NAME         = 'PS'                          ,&
         DIMS               = MAPL_DimsTileOnly             ,&
         VLOCATION          = MAPL_VLocationNone            ,&
                                                  RC=STATUS  ) 

    VERIFY_(STATUS)

    call MAPL_AddImportSpec(GC                         ,&
         LONG_NAME          = 'surface_air_temperature'     ,&
         UNITS              = 'K'                           ,&
         SHORT_NAME         = 'TA'                          ,&
         DIMS               = MAPL_DimsTileOnly             ,&
         VLOCATION          = MAPL_VLocationNone            ,&
                                                  RC=STATUS  ) 

    VERIFY_(STATUS)

    call MAPL_AddImportSpec(GC                         ,&
         LONG_NAME          = 'surface_air_specific_humidity',&
         UNITS              = 'kg kg-1'                     ,&
         SHORT_NAME         = 'QA'                          ,&
         DIMS               = MAPL_DimsTileOnly             ,&
         VLOCATION          = MAPL_VLocationNone            ,&
                                                  RC=STATUS  ) 
    VERIFY_(STATUS)

    call MAPL_AddImportSpec(GC                         ,&
         LONG_NAME          = 'surface_wind_speed'          ,&
         UNITS              = 'm s-1'                       ,&
         SHORT_NAME         = 'UU'                          ,&
         DIMS               = MAPL_DimsTileOnly             ,&
         VLOCATION          = MAPL_VLocationNone            ,&
                                                  RC=STATUS  ) 
    VERIFY_(STATUS)

     call MAPL_AddImportSpec(GC,                             &
        LONG_NAME          = 'levellm_uwind',                     &
        UNITS              = 'm s-1',                             &
        SHORT_NAME         = 'UWINDLMTILE',                       &
        DIMS               = MAPL_DimsTileOnly,                   &
        VLOCATION          = MAPL_VLocationNone,                  &
                                                       RC=STATUS  )
     VERIFY_(STATUS)

     call MAPL_AddImportSpec(GC,                             &
        LONG_NAME          = 'levellm_vwind',                     &
        UNITS              = 'm s-1',                             &
        SHORT_NAME         = 'VWINDLMTILE',                       &
        DIMS               = MAPL_DimsTileOnly,                   &
        VLOCATION          = MAPL_VLocationNone,                  &
                                                       RC=STATUS  )
     VERIFY_(STATUS)

    call MAPL_AddImportSpec(GC                         ,&
         LONG_NAME          = 'liquid_water_convective_precipitation',&
         UNITS              = 'kg m-2 s-1'                  ,&
         SHORT_NAME         = 'PCU'                         ,&
         DIMS               = MAPL_DimsTileOnly             ,&
         VLOCATION          = MAPL_VLocationNone            ,&
                                                  RC=STATUS  ) 
    VERIFY_(STATUS)

    call MAPL_AddImportSpec(GC                         ,&
         LONG_NAME          = 'liquid_water_large_scale_precipitation',&
         UNITS              = 'kg m-2 s-1'                  ,&
         SHORT_NAME         = 'PLS'                         ,&
         DIMS               = MAPL_DimsTileOnly             ,&
         VLOCATION          = MAPL_VLocationNone            ,&
                                                  RC=STATUS  ) 
    VERIFY_(STATUS)

    call MAPL_AddImportSpec(GC                         ,&
         LONG_NAME          = 'snowfall'                    ,&
         UNITS              = 'kg m-2 s-1'                  ,&
         SHORT_NAME         = 'SNO'                         ,&
         DIMS               = MAPL_DimsTileOnly             ,&
         VLOCATION          = MAPL_VLocationNone            ,&
                                                  RC=STATUS  ) 
    VERIFY_(STATUS)

    call MAPL_AddImportSpec(GC                         ,&
         LONG_NAME          = 'icefall'                     ,&
         UNITS              = 'kg m-2 s-1'                  ,&
         SHORT_NAME         = 'ICE'                         ,&
         DIMS               = MAPL_DimsTileOnly             ,&
         VLOCATION          = MAPL_VLocationNone            ,&
         RC=STATUS  )
    
    VERIFY_(STATUS)
    
    call MAPL_AddImportSpec(GC                         ,&
         LONG_NAME          = 'freezing_rain_fall'          ,&
         UNITS              = 'kg m-2 s-1'                  ,&
         SHORT_NAME         = 'FRZR'                        ,&
         DIMS               = MAPL_DimsTileOnly             ,&
         VLOCATION          = MAPL_VLocationNone            ,&
         RC=STATUS  )
    
    VERIFY_(STATUS)

    call MAPL_AddImportSpec(GC                         ,&
         LONG_NAME          = 'surface_downwelling_par_beam_flux',&
         UNITS              = 'W m-2'                       ,&
         SHORT_NAME         = 'DRPAR'                       ,&
         DIMS               = MAPL_DimsTileOnly             ,&
         VLOCATION          = MAPL_VLocationNone            ,&
         RC=STATUS  ) 
    VERIFY_(STATUS)
    
    call MAPL_AddImportSpec(GC                         ,&
         LONG_NAME          = 'surface_downwelling_par_diffuse_flux',&
         UNITS              = 'W m-2'                       ,&
         SHORT_NAME         = 'DFPAR'                       ,&
         DIMS               = MAPL_DimsTileOnly             ,&
         VLOCATION          = MAPL_VLocationNone            ,&
                                                  RC=STATUS  ) 
    VERIFY_(STATUS)

    call MAPL_AddImportSpec(GC                         ,&
         LONG_NAME          = 'surface_downwelling_nir_beam_flux',&
         UNITS              = 'W m-2'                       ,&
         SHORT_NAME         = 'DRNIR'                       ,&
         DIMS               = MAPL_DimsTileOnly             ,&
         VLOCATION          = MAPL_VLocationNone            ,&
                                                  RC=STATUS  ) 
    VERIFY_(STATUS)

    call MAPL_AddImportSpec(GC                         ,&
         LONG_NAME          = 'surface_downwelling_nir_diffuse_flux',&
         UNITS              = 'W m-2'                       ,&
         SHORT_NAME         = 'DFNIR'                       ,&
         DIMS               = MAPL_DimsTileOnly             ,&
         VLOCATION          = MAPL_VLocationNone            ,&
                                                  RC=STATUS  ) 
    VERIFY_(STATUS)

    call MAPL_AddImportSpec(GC                         ,&
         LONG_NAME          = 'surface_downwelling_uvr_beam_flux',&
         UNITS              = 'W m-2'                       ,&
         SHORT_NAME         = 'DRUVR'                       ,&
         DIMS               = MAPL_DimsTileOnly             ,&
         VLOCATION          = MAPL_VLocationNone            ,&
                                                  RC=STATUS  ) 
    VERIFY_(STATUS)

    call MAPL_AddImportSpec(GC                         ,&
         LONG_NAME          = 'surface_downwelling_uvr_diffuse_flux',&
         UNITS              = 'W m-2'                       ,&
         SHORT_NAME         = 'DFUVR'                       ,&
         DIMS               = MAPL_DimsTileOnly             ,&
         VLOCATION          = MAPL_VLocationNone            ,&
                                                  RC=STATUS  ) 
    VERIFY_(STATUS)

    call MAPL_AddImportSpec(GC                         ,&
         LONG_NAME          = 'surface_downwelling_longwave_flux',&
         UNITS              = 'W m-2'                       ,&
         SHORT_NAME         = 'LWDNSRF'                     ,&
         DIMS               = MAPL_DimsTileOnly             ,&
         VLOCATION          = MAPL_VLocationNone            ,&
                                                  RC=STATUS  ) 
    VERIFY_(STATUS)

    call MAPL_AddImportSpec(GC                         ,&
         LONG_NAME          = 'linearization_of_surface_upwelling_longwave_flux',&
         UNITS              = 'W m-2'                       ,&
         SHORT_NAME         = 'ALW'                         ,&
         DIMS               = MAPL_DimsTileOnly             ,&
         VLOCATION          = MAPL_VLocationNone            ,&
                                                  RC=STATUS  ) 
    VERIFY_(STATUS)

    call MAPL_AddImportSpec(GC                         ,&
         LONG_NAME          = 'linearization_of_surface_upwelling_longwave_flux',&
         UNITS              = 'W_m-2 K-1'                   ,&
         SHORT_NAME         = 'BLW'                         ,&
         DIMS               = MAPL_DimsTileOnly             ,&
         VLOCATION          = MAPL_VLocationNone            ,&
                                                  RC=STATUS  ) 
    VERIFY_(STATUS)

    IF (ATM_CO2 == 4) THEN
       call MAPL_AddImportSpec(GC,                              &
            SHORT_NAME         = 'CO2SC',                             &
            LONG_NAME          = 'CO2 Surface Concentration Bin 001', &
            UNITS              = '1e-6',                              &
            DIMS               = MAPL_DimsTileOnly,                   &
            VLOCATION          = MAPL_VLocationNone,                  &
            RC=STATUS  )
       VERIFY_(STATUS)
    ENDIF

    call MAPL_AddImportSpec(GC                         ,&
         LONG_NAME          = 'leaf_area_index'             ,&
         UNITS              = '1'                           ,&
         SHORT_NAME         = 'LAI'                         ,&
         DIMS               = MAPL_DimsTileOnly             ,&
         VLOCATION          = MAPL_VLocationNone            ,&
                                                  RC=STATUS  ) 
    VERIFY_(STATUS)

    call MAPL_AddImportSpec(GC                         ,&
         LONG_NAME          = 'greeness_fraction'           ,&
         UNITS              = '1'                           ,&
         SHORT_NAME         = 'GRN'                         ,&
         DIMS               = MAPL_DimsTileOnly             ,&
         VLOCATION          = MAPL_VLocationNone            ,&
                                                  RC=STATUS  ) 
    VERIFY_(STATUS)

    call MAPL_AddImportSpec(GC                         ,&
         LONG_NAME          = 'evaporation'                 ,&
         UNITS              = 'kg m-2 s-1'                  ,&
         SHORT_NAME         = 'EVAP'                        ,&
         DIMS               = MAPL_DimsTileOnly             ,&
         VLOCATION          = MAPL_VLocationNone            ,&
                                                  RC=STATUS  ) 
    VERIFY_(STATUS)

    call MAPL_AddImportSpec(GC                         ,&
         LONG_NAME          = 'derivative_of_evaporation_wrt_QS',&
         UNITS              = 'kg m-2 s-1'                  ,&
         SHORT_NAME         = 'DEVAP'                       ,&
         DIMS               = MAPL_DimsTileOnly             ,&
         VLOCATION          = MAPL_VLocationNone            ,&
                                                  RC=STATUS  ) 
    VERIFY_(STATUS)

    call MAPL_AddImportSpec(GC                         ,&
         LONG_NAME          = 'upward_sensible_heat_flux'   ,&
         UNITS              = 'W m-2'                       ,&
         SHORT_NAME         = 'SH'                          ,&
         DIMS               = MAPL_DimsTileOnly             ,&
         VLOCATION          = MAPL_VLocationNone            ,&
                                                  RC=STATUS  ) 
    VERIFY_(STATUS)

    call MAPL_AddImportSpec(GC                         ,&
         LONG_NAME          = 'derivative_of_sensible_heat_wrt_Ts',&
         UNITS              = 'W m-2 K-1'                   ,&
         SHORT_NAME         = 'DSH'                         ,&
         DIMS               = MAPL_DimsTileOnly             ,&
         VLOCATION          = MAPL_VLocationNone            ,&
                                                  RC=STATUS  ) 
    VERIFY_(STATUS)

    call MAPL_AddImportSpec(GC                         ,&
         LONG_NAME          = 'surface_layer_height'        ,&
         UNITS              = 'm'                           ,&
         SHORT_NAME         = 'DZ'                          ,&
         DIMS               = MAPL_DimsTileOnly             ,&
         VLOCATION          = MAPL_VLocationNone            ,&
                                                  RC=STATUS  ) 
    VERIFY_(STATUS)

   call MAPL_AddImportSpec(GC                         ,&
        LONG_NAME          = 'vegetation_root_length'      ,&
        UNITS              = 'm'                           ,&
        SHORT_NAME         = 'ROOTL'                       ,&
        DIMS               = MAPL_DimsTileOnly             ,&
        VLOCATION          = MAPL_VLocationNone            ,&
                                                 RC=STATUS  ) 
   VERIFY_(STATUS)

   call MAPL_AddImportSpec(GC                         ,&
        LONG_NAME          = 'canopy_height'               ,&
        UNITS              = 'm'                           ,&
        SHORT_NAME         = 'Z2CH'                        ,&
        DIMS               = MAPL_DimsTileOnly             ,&
        VLOCATION          = MAPL_VLocationNone            ,&
                                                 RC=STATUS  ) 
   VERIFY_(STATUS)
    
    call MAPL_AddImportSpec(GC,                         &
         SHORT_NAME         = 'THATM',                       &
         LONG_NAME          = 'effective_surface_skin_temperature',&
         UNITS              = 'K',                           &
         DIMS               = MAPL_DimsTileOnly,             &
         VLOCATION          = MAPL_VLocationNone,            &
                                                  RC=STATUS  )
    VERIFY_(STATUS)

    call MAPL_AddImportSpec(GC,                         &
         SHORT_NAME         = 'QHATM',                       &
         LONG_NAME          = 'effective_surface_specific_humidity',&
         UNITS              = 'kg kg-1',                     &
         DIMS               = MAPL_DimsTileOnly,             &
         VLOCATION          = MAPL_VLocationNone,            &
                                                  RC=STATUS  )
    VERIFY_(STATUS)
 
    call MAPL_AddImportSpec(GC,                         &
         SHORT_NAME         = 'CTATM',                       &
         LONG_NAME          = 'surface_exchange_coefficient_for_heat', &
         UNITS              = 'kg m-2 s-1',                  &
         DIMS               = MAPL_DimsTileOnly,             &
         VLOCATION          = MAPL_VLocationNone,            &
                                                  RC=STATUS  )
    VERIFY_(STATUS)

    call MAPL_AddImportSpec(GC,                         &
         SHORT_NAME         = 'CQATM',                       &
         LONG_NAME          = 'surface_exchange_coefficient_for_moisture', &
         UNITS              = 'kg m-2 s-1',                  &
         DIMS               = MAPL_DimsTileOnly,             &
         VLOCATION          = MAPL_VLocationNone,            &
                                                  RC=STATUS  )
    VERIFY_(STATUS)

    call MAPL_AddImportSpec(GC                                ,&
       SHORT_NAME = 'ASCATZ0'                                 ,&
       LONG_NAME  = 'ASCAT_roughness_length'		      ,&
       UNITS      = 'm'                                       ,&
       DIMS       = MAPL_DimsTileOnly                         ,&
       VLOCATION  = MAPL_VLocationNone                        ,&
       RC=STATUS  )
    VERIFY_(STATUS)  

    call MAPL_AddImportSpec(GC                             ,&
        SHORT_NAME         = 'NDVI'                        ,& 
        LONG_NAME          = 'normalized_difference_vegetation_index' ,&
        UNITS              = '1'                           ,&        
        DIMS               = MAPL_DimsTileOnly             ,&
        VLOCATION          = MAPL_VLocationNone            ,&
                                                  RC=STATUS  ) 
    VERIFY_(STATUS)

    call MAPL_AddImportSpec(GC,                         &
         LONG_NAME          = 'dust_dry_depos_all_bins', &
         UNITS              = 'kg m-2 s-1',                  &
         SHORT_NAME         = 'DUDP',                        &
         DIMS               = MAPL_DimsTileOnly,             &
         UNGRIDDED_DIMS     = (/NUM_DUDP/),                  &
         VLOCATION          = MAPL_VLocationNone,            &
         RC=STATUS  ) 
    VERIFY_(STATUS)
    
    call MAPL_AddImportSpec(GC,                         &
         LONG_NAME          = 'dust_wet_depos_conv_scav_all_bins', &
         UNITS              = 'kg m-2 s-1',                  &
         SHORT_NAME         = 'DUSV',                        &
         DIMS               = MAPL_DimsTileOnly,             &
         UNGRIDDED_DIMS     = (/NUM_DUSV/),                  &
         VLOCATION          = MAPL_VLocationNone,            &
         RC=STATUS  ) 
    VERIFY_(STATUS)
    
    call MAPL_AddImportSpec(GC,                         &
         LONG_NAME          = 'dust_wet_depos_ls_scav_all_bins', &
         UNITS              = 'kg m-2 s-1',                  &
         SHORT_NAME         = 'DUWT',                        &
         DIMS               = MAPL_DimsTileOnly,             &
         UNGRIDDED_DIMS     = (/NUM_DUWT/),                  &
         VLOCATION          = MAPL_VLocationNone,            &
         RC=STATUS  ) 
    VERIFY_(STATUS)
    
    call MAPL_AddImportSpec(GC,                         &
         LONG_NAME          = 'dust_gravity_sett_all_bins', &
         UNITS              = 'kg m-2 s-1',                  &
         SHORT_NAME         = 'DUSD',                        &
         DIMS               = MAPL_DimsTileOnly,             &
         UNGRIDDED_DIMS     = (/NUM_DUSD/),                  &
         VLOCATION          = MAPL_VLocationNone,            &
         RC=STATUS  ) 
    VERIFY_(STATUS)
    
    call MAPL_AddImportSpec(GC,                         &
         LONG_NAME          = 'black_carbon_dry_depos_all_bins', &
         UNITS              = 'kg m-2 s-1',                  &
         SHORT_NAME         = 'BCDP',                        &
         DIMS               = MAPL_DimsTileOnly,             &
         UNGRIDDED_DIMS     = (/NUM_BCDP/),                  &
         VLOCATION          = MAPL_VLocationNone,            &
         RC=STATUS  ) 
    VERIFY_(STATUS)
    
    call MAPL_AddImportSpec(GC,                         &
         LONG_NAME          = 'black_carbon_wet_depos_conv_scav_all_bins', &
         UNITS              = 'kg m-2 s-1',                  &
         SHORT_NAME         = 'BCSV',                        &
         DIMS               = MAPL_DimsTileOnly,             &
         UNGRIDDED_DIMS     = (/NUM_BCSV/),                  &
         VLOCATION          = MAPL_VLocationNone,            &
         RC=STATUS  ) 
    VERIFY_(STATUS)
    
    call MAPL_AddImportSpec(GC,                         &
         LONG_NAME          = 'black_carbon_wet_depos_ls_scav_all_bins', &
         UNITS              = 'kg m-2 s-1',                  &
         SHORT_NAME         = 'BCWT',                        &
         DIMS               = MAPL_DimsTileOnly,             &
         UNGRIDDED_DIMS     = (/NUM_BCWT/),                  &
         VLOCATION          = MAPL_VLocationNone,            &
         RC=STATUS  ) 
    VERIFY_(STATUS)
 
    call MAPL_AddImportSpec(GC,                         &
         LONG_NAME          = 'black_carbon_gravity_sett_all_bins', &
         UNITS              = 'kg m-2 s-1',                  &
         SHORT_NAME         = 'BCSD',                        &
         DIMS               = MAPL_DimsTileOnly,             &
         UNGRIDDED_DIMS     = (/NUM_BCSD/),                  &
         VLOCATION          = MAPL_VLocationNone,            &
         RC=STATUS  ) 
    VERIFY_(STATUS)
    
    call MAPL_AddImportSpec(GC,                         &
         LONG_NAME          = 'organic_carbon_dry_depos_all_bins', &
         UNITS              = 'kg m-2 s-1',                  &
         SHORT_NAME         = 'OCDP',                        &
         DIMS               = MAPL_DimsTileOnly,             &
         UNGRIDDED_DIMS     = (/NUM_OCDP/),                  &
         VLOCATION          = MAPL_VLocationNone,            &
         RC=STATUS  ) 
    VERIFY_(STATUS)
    
    call MAPL_AddImportSpec(GC,                         &
         LONG_NAME          = 'organic_carbon_wet_depos_conv_scav_all_bins', &
         UNITS              = 'kg m-2 s-1',                  &
         SHORT_NAME         = 'OCSV',                        &
         DIMS               = MAPL_DimsTileOnly,             &
         UNGRIDDED_DIMS     = (/NUM_OCSV/),                  &
         VLOCATION          = MAPL_VLocationNone,            &
         RC=STATUS  ) 
    VERIFY_(STATUS)
    
    call MAPL_AddImportSpec(GC,                         &
         LONG_NAME          = 'organic_carbon_wet_depos_ls_scav_all_bins', &
         UNITS              = 'kg m-2 s-1',                  &
         SHORT_NAME         = 'OCWT',                        &
         DIMS               = MAPL_DimsTileOnly,             &
         UNGRIDDED_DIMS     = (/NUM_OCWT/),                  &
         VLOCATION          = MAPL_VLocationNone,            &
         RC=STATUS  ) 
    VERIFY_(STATUS)
    
    call MAPL_AddImportSpec(GC,                         &
         LONG_NAME          = 'organic_carbon_gravity_sett_all_bins', &
         UNITS              = 'kg m-2 s-1',                  &
         SHORT_NAME         = 'OCSD',                        &
         DIMS               = MAPL_DimsTileOnly,             &
         UNGRIDDED_DIMS     = (/NUM_OCSD/),                  &
         VLOCATION          = MAPL_VLocationNone,            &
         RC=STATUS  ) 
    VERIFY_(STATUS)
    
    call MAPL_AddImportSpec(GC,                         &
         LONG_NAME          = 'sulfate_dry_depos_all_bins', &
         UNITS              = 'kg m-2 s-1',                  &
         SHORT_NAME         = 'SUDP',                        &
         DIMS               = MAPL_DimsTileOnly,             &
         UNGRIDDED_DIMS     = (/NUM_SUDP/),                  &
         VLOCATION          = MAPL_VLocationNone,            &
         RC=STATUS  ) 
    VERIFY_(STATUS)
    
    call MAPL_AddImportSpec(GC,                         &
         LONG_NAME          = 'sulfate_wet_depos_conv_scav_all_bins', &
         UNITS              = 'kg m-2 s-1',                  &
         SHORT_NAME         = 'SUSV',                        &
         DIMS               = MAPL_DimsTileOnly,             &
         UNGRIDDED_DIMS     = (/NUM_SUSV/),                  &
         VLOCATION          = MAPL_VLocationNone,            &
         RC=STATUS  ) 
    VERIFY_(STATUS)
    
    call MAPL_AddImportSpec(GC,                         &
         LONG_NAME          = 'sulfate_wet_depos_ls_scav_all_bins', &
         UNITS              = 'kg m-2 s-1',                  &
         SHORT_NAME         = 'SUWT',                        &
         DIMS               = MAPL_DimsTileOnly,             &
         UNGRIDDED_DIMS     = (/NUM_SUWT/),                  &
         VLOCATION          = MAPL_VLocationNone,            &
         RC=STATUS  ) 
    VERIFY_(STATUS)
    
    call MAPL_AddImportSpec(GC,                         &
         LONG_NAME          = 'sulfate_gravity_sett_all_bins', &
         UNITS              = 'kg m-2 s-1',                  &
         SHORT_NAME         = 'SUSD',                        &
         DIMS               = MAPL_DimsTileOnly,             &
         UNGRIDDED_DIMS     = (/NUM_SUSD/),                  &
         VLOCATION          = MAPL_VLocationNone,            &
         RC=STATUS  ) 
    VERIFY_(STATUS)
    
    call MAPL_AddImportSpec(GC,                         &
         LONG_NAME          = 'sea_salt_dry_depos_all_bins', &
         UNITS              = 'kg m-2 s-1',                  &
         SHORT_NAME         = 'SSDP',                        &
         DIMS               = MAPL_DimsTileOnly,             &
         UNGRIDDED_DIMS     = (/NUM_SSDP/),                  &
         VLOCATION          = MAPL_VLocationNone,            &
         RC=STATUS  ) 
    VERIFY_(STATUS)
    
    call MAPL_AddImportSpec(GC,                         &
         LONG_NAME          = 'sea_salt_wet_depos_conv_scav_all_bins', &
         UNITS              = 'kg m-2 s-1',                  &
         SHORT_NAME         = 'SSSV',                        &
         DIMS               = MAPL_DimsTileOnly,             &
         UNGRIDDED_DIMS     = (/NUM_SSSV/),                  &
         VLOCATION          = MAPL_VLocationNone,            &
         RC=STATUS  ) 
    VERIFY_(STATUS)
    
    call MAPL_AddImportSpec(GC,                         &
         LONG_NAME          = 'sea_salt_wet_depos_ls_scav_all_bins', &
         UNITS              = 'kg m-2 s-1',                  &
         SHORT_NAME         = 'SSWT',                        &
         DIMS               = MAPL_DimsTileOnly,             &
         UNGRIDDED_DIMS     = (/NUM_SSWT/),                  &
         VLOCATION          = MAPL_VLocationNone,            &
         RC=STATUS  ) 
    VERIFY_(STATUS)
    
    call MAPL_AddImportSpec(GC,                         &
         LONG_NAME          = 'sea_salt_gravity_sett_all_bins', &
         UNITS              = 'kg m-2 s-1',                  &
         SHORT_NAME         = 'SSSD',                        &
         DIMS               = MAPL_DimsTileOnly,             &
         UNGRIDDED_DIMS     = (/NUM_SSSD/),                  &
         VLOCATION          = MAPL_VLocationNone,            &
         RC=STATUS  ) 
    VERIFY_(STATUS)

!  !INTERNAL STATE:

! if is_offline, some variables ( in the last) are not required
  if      ( OFFLINE_MODE == 1 ) then
     RESTART = MAPL_RestartSkip
  elseif  ( OFFLINE_MODE == 2 ) then
     RESTART = MAPL_RestartOptional
  elseif  ( OFFLINE_MODE == 0 ) then
     RESTART = MAPL_RestartRequired
  else
     ASSERT_(.FALSE.)
  endif

  call MAPL_AddInternalSpec(GC                  ,&
    LONG_NAME          = 'topo_baseflow_param_1'     ,&
    UNITS              = 'kg m-4'                    ,&
    SHORT_NAME         = 'BF1'                       ,&
    FRIENDLYTO         = trim(COMP_NAME)             ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = MAPL_RestartRequired        ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                  ,&
    LONG_NAME          = 'topo_baseflow_param_2'     ,&
    UNITS              = 'm'                         ,&
    SHORT_NAME         = 'BF2'                       ,&
    FRIENDLYTO         = trim(COMP_NAME)             ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = MAPL_RestartRequired        ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                  ,&
    LONG_NAME          = 'topo_baseflow_param_3'     ,&
    UNITS              = 'log(m)'                    ,&
    SHORT_NAME         = 'BF3'                       ,&
    FRIENDLYTO         = trim(COMP_NAME)             ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = MAPL_RestartRequired        ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                  ,&
    LONG_NAME          = 'max_rootzone_water_content',&
    UNITS              = 'kg m-2'                    ,&
    SHORT_NAME         = 'VGWMAX'                    ,&
    FRIENDLYTO         = trim(COMP_NAME)             ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = MAPL_RestartRequired        ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                  ,&
    LONG_NAME          = 'moisture_threshold'        ,&
    UNITS              = 'kg m-2'                    ,&
    SHORT_NAME         = 'CDCR1'                     ,&
    FRIENDLYTO         = trim(COMP_NAME)             ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = MAPL_RestartRequired        ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                  ,&
    LONG_NAME          = 'max_water_content'         ,&
    UNITS              = 'kg m-2'                    ,&
    SHORT_NAME         = 'CDCR2'                     ,&
    FRIENDLYTO         = trim(COMP_NAME)             ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = MAPL_RestartRequired        ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                  ,&
    LONG_NAME          = 'saturated_matric_potential',&
    UNITS              = 'm'                         ,&
    SHORT_NAME         = 'PSIS'                      ,&
    FRIENDLYTO         = trim(COMP_NAME)             ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = MAPL_RestartRequired        ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                  ,&
    LONG_NAME          = 'clapp_hornberger_b'        ,&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'BEE'                       ,&
    FRIENDLYTO         = trim(COMP_NAME)             ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = MAPL_RestartRequired        ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                  ,&
    LONG_NAME          = 'soil_porosity'             ,&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'POROS'                     ,&
    FRIENDLYTO         = trim(COMP_NAME)             ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = MAPL_RestartRequired        ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                  ,&
    LONG_NAME          = 'wetness_at_wilting_point'  ,&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'WPWET'                     ,&
    FRIENDLYTO         = trim(COMP_NAME)             ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = MAPL_RestartRequired        ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                  ,&
    LONG_NAME          = 'sfc_sat_hydraulic_conduct' ,&
    UNITS              = 'm s-1'                     ,&
    SHORT_NAME         = 'COND'                      ,&
    FRIENDLYTO         = trim(COMP_NAME)             ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = MAPL_RestartRequired        ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                  ,&
    LONG_NAME          = 'vertical_transmissivity'   ,&
    UNITS              = 'm-1'                       ,&
    SHORT_NAME         = 'GNU'                       ,&
    FRIENDLYTO         = trim(COMP_NAME)             ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = MAPL_RestartRequired        ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                  ,&
    LONG_NAME          = 'wetness_param_1'           ,&
    UNITS              = 'm+2 kg-1'                  ,&
    SHORT_NAME         = 'ARS1'                      ,&
    FRIENDLYTO         = trim(COMP_NAME)             ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = MAPL_RestartRequired        ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                  ,&
    LONG_NAME          = 'wetness_param_2'           ,&
    UNITS              = 'm+2 kg-1'                  ,&
    SHORT_NAME         = 'ARS2'                      ,&
    FRIENDLYTO         = trim(COMP_NAME)             ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = MAPL_RestartRequired        ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                  ,&
    LONG_NAME          = 'wetness_param_3'           ,&
    UNITS              = 'm+4 kg-2'                  ,&
    SHORT_NAME         = 'ARS3'                      ,&
    FRIENDLYTO         = trim(COMP_NAME)             ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = MAPL_RestartRequired        ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                  ,&
    LONG_NAME          = 'shape_param_1'             ,&
    UNITS              = 'm+2 kg-1'                  ,&
    SHORT_NAME         = 'ARA1'                      ,&
    FRIENDLYTO         = trim(COMP_NAME)             ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = MAPL_RestartRequired        ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                  ,&
    LONG_NAME          = 'shape_param_2'             ,&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'ARA2'                      ,&
    FRIENDLYTO         = trim(COMP_NAME)             ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = MAPL_RestartRequired        ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                  ,&
    LONG_NAME          = 'shape_param_3'             ,&
    UNITS              = 'm+2 kg-1'                  ,&
    SHORT_NAME         = 'ARA3'                      ,&
    FRIENDLYTO         = trim(COMP_NAME)             ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = MAPL_RestartRequired        ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                  ,&
    LONG_NAME          = 'shape_param_4'             ,&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'ARA4'                      ,&
    FRIENDLYTO         = trim(COMP_NAME)             ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = MAPL_RestartRequired        ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                  ,&
    LONG_NAME          = 'min_theta_param_1'         ,&
    UNITS              = 'm+2 kg-1'                  ,&
    SHORT_NAME         = 'ARW1'                      ,&
    FRIENDLYTO         = trim(COMP_NAME)             ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = MAPL_RestartRequired        ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                  ,&
    LONG_NAME          = 'min_theta_param_2'         ,&
    UNITS              = 'm+2 kg-1'                  ,&
    SHORT_NAME         = 'ARW2'                      ,&
    FRIENDLYTO         = trim(COMP_NAME)             ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = MAPL_RestartRequired        ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                  ,&
    LONG_NAME          = 'min_theta_param_3'         ,&
    UNITS              = 'm+4 kg-2'                  ,&
    SHORT_NAME         = 'ARW3'                      ,&
    FRIENDLYTO         = trim(COMP_NAME)             ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = MAPL_RestartRequired        ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                  ,&
    LONG_NAME          = 'min_theta_param_4'         ,&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'ARW4'                      ,&
    FRIENDLYTO         = trim(COMP_NAME)             ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = MAPL_RestartRequired        ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                  ,&
    LONG_NAME          = 'water_transfer_param_1'    ,&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'TSA1'                      ,&
    FRIENDLYTO         = trim(COMP_NAME)             ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = MAPL_RestartRequired        ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                  ,&
    LONG_NAME          = 'water_transfer_param_2'    ,&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'TSA2'                      ,&
    FRIENDLYTO         = trim(COMP_NAME)             ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = MAPL_RestartRequired        ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                  ,&
    LONG_NAME          = 'water_transfer_param_3'    ,&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'TSB1'                      ,&
    FRIENDLYTO         = trim(COMP_NAME)             ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = MAPL_RestartRequired        ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                  ,&
    LONG_NAME          = 'water_transfer_param_4'    ,&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'TSB2'                      ,&
    FRIENDLYTO         = trim(COMP_NAME)             ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = MAPL_RestartRequired        ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                  ,&
    LONG_NAME          = 'water_transfer_param_5'    ,&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'ATAU'                      ,&
    FRIENDLYTO         = trim(COMP_NAME)             ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = MAPL_RestartRequired        ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                  ,&
    LONG_NAME          = 'water_transfer_param_6'    ,&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'BTAU'                      ,&
    FRIENDLYTO         = trim(COMP_NAME)             ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = MAPL_RestartRequired        ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                  ,&
    LONG_NAME          = 'vegetation_type'           ,&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'ITY'                   ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    UNGRIDDED_DIMS     = (/NUM_VEG/)                 ,&
    RESTART            = MAPL_RestartRequired        ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                  ,&
    LONG_NAME          = 'vegetation_fraction'       ,&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'FVG'                       ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    UNGRIDDED_DIMS     = (/NUM_VEG/)                 ,&
    RESTART            = MAPL_RestartRequired        ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                  ,&
    LONG_NAME          = 'canopy_temperature'        ,&
    UNITS              = 'K'                         ,&
    SHORT_NAME         = 'TC'                        ,&
    FRIENDLYTO         = trim(COMP_NAME)             ,&
    DIMS               = MAPL_DimsTileTile           ,&
    NUM_SUBTILES       = NUM_SUBTILES                ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = MAPL_RestartRequired        ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                  ,&
    LONG_NAME          = 'canopy_specific_humidity'  ,&
    UNITS              = 'kg kg-1'                   ,&
    SHORT_NAME         = 'QC'                        ,&
    FRIENDLYTO         = trim(COMP_NAME)             ,&
    DIMS               = MAPL_DimsTileTile           ,&
    NUM_SUBTILES       = NUM_SUBTILES                ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = MAPL_RestartRequired        ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                  ,&
    LONG_NAME          = 'ground_temperature'        ,&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'TG'                        ,&
    FRIENDLYTO         = trim(COMP_NAME)             ,&
    DIMS               = MAPL_DimsTileTile           ,&
    NUM_SUBTILES       = NUM_SUBTILES                ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = MAPL_RestartRequired        ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                  ,&
    LONG_NAME          = 'interception_reservoir_capac',&
    UNITS              = 'kg m-2'                    ,&
    SHORT_NAME         = 'CAPAC'                     ,&
    FRIENDLYTO         = trim(COMP_NAME)             ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = MAPL_RestartRequired        ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                  ,&
    LONG_NAME          = 'catchment_deficit'         ,&
    UNITS              = 'kg m-2'                    ,&
    SHORT_NAME         = 'CATDEF'                    ,&
    FRIENDLYTO         = trim(COMP_NAME)             ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = MAPL_RestartRequired        ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                  ,&
    LONG_NAME          = 'root_zone_excess'          ,&
    UNITS              = 'kg m-2'                    ,&
    SHORT_NAME         = 'RZEXC'                     ,&
    FRIENDLYTO         = trim(COMP_NAME)             ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = MAPL_RestartRequired        ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                  ,&
    LONG_NAME          = 'surface_excess'            ,&
    UNITS              = 'kg m-2'                    ,&
    SHORT_NAME         = 'SRFEXC'                    ,&
    FRIENDLYTO         = trim(COMP_NAME)             ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = MAPL_RestartRequired        ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                  ,&
    LONG_NAME          = 'soil_heat_content_layer_1' ,&
    UNITS              = 'J m-2'                     ,&
    SHORT_NAME         = 'GHTCNT1'                   ,&
    FRIENDLYTO         = trim(COMP_NAME)             ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = MAPL_RestartRequired        ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                  ,&
    LONG_NAME          = 'soil_heat_content_layer_2' ,&
    UNITS              = 'J_m-2'                     ,&
    SHORT_NAME         = 'GHTCNT2'                   ,&
    FRIENDLYTO         = trim(COMP_NAME)             ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = MAPL_RestartRequired        ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                  ,&
    LONG_NAME          = 'soil_heat_content_layer_3' ,&
    UNITS              = 'J m-2'                     ,&
    SHORT_NAME         = 'GHTCNT3'                   ,&
    FRIENDLYTO         = trim(COMP_NAME)             ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = MAPL_RestartRequired        ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                  ,&
    LONG_NAME          = 'soil_heat_content_layer_4' ,&
    UNITS              = 'J m-2'                     ,&
    SHORT_NAME         = 'GHTCNT4'                   ,&
    FRIENDLYTO         = trim(COMP_NAME)             ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = MAPL_RestartRequired        ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                  ,&
    LONG_NAME          = 'soil_heat_content_layer_5' ,&
    UNITS              = 'J m-2'                     ,&
    SHORT_NAME         = 'GHTCNT5'                   ,&
    FRIENDLYTO         = trim(COMP_NAME)             ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = MAPL_RestartRequired        ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                  ,&
    LONG_NAME          = 'soil_heat_content_layer_6' ,&
    UNITS              = 'J m-2'                     ,&
    SHORT_NAME         = 'GHTCNT6'                   ,&
    FRIENDLYTO         = trim(COMP_NAME)             ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = MAPL_RestartRequired        ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                  ,&
    LONG_NAME          = 'mean_catchment_temp_incl_snw',&
    UNITS              = 'K'                         ,&
    SHORT_NAME         = 'TSURF'                     ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = RESTART                     ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                  ,&
    LONG_NAME          = 'snow_mass_layer_1'         ,&
    UNITS              = 'kg m-2'                    ,&
    SHORT_NAME         = 'WESNN1'                    ,&
    FRIENDLYTO         = trim(COMP_NAME)             ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = MAPL_RestartRequired        ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                  ,&
    LONG_NAME          = 'snow_mass_layer_2'         ,&
    UNITS              = 'kg m-2'                    ,&
    SHORT_NAME         = 'WESNN2'                    ,&
    FRIENDLYTO         = trim(COMP_NAME)             ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = MAPL_RestartRequired        ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                  ,&
    LONG_NAME          = 'snow_mass_layer_3'         ,&
    UNITS              = 'kg m-2'                    ,&
    SHORT_NAME         = 'WESNN3'                    ,&
    FRIENDLYTO         = trim(COMP_NAME)             ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = MAPL_RestartRequired        ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                  ,&
    LONG_NAME          = 'heat_content_snow_layer_1' ,&
    UNITS              = 'J m-2'                     ,&
    SHORT_NAME         = 'HTSNNN1'                   ,&
    FRIENDLYTO         = trim(COMP_NAME)             ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = MAPL_RestartRequired        ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                  ,&
    LONG_NAME          = 'heat_content_snow_layer_2' ,&
    UNITS              = 'J m-2'                     ,&
    SHORT_NAME         = 'HTSNNN2'                   ,&
    FRIENDLYTO         = trim(COMP_NAME)             ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = MAPL_RestartRequired        ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                  ,&
    LONG_NAME          = 'heat_content_snow_layer_3' ,&
    UNITS              = 'J m-2'                     ,&
    SHORT_NAME         = 'HTSNNN3'                   ,&
    FRIENDLYTO         = trim(COMP_NAME)             ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = MAPL_RestartRequired        ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                  ,&
    LONG_NAME          = 'snow_depth_layer_1'        ,&
    UNITS              = 'm'                         ,&
    SHORT_NAME         = 'SNDZN1'                    ,&
    FRIENDLYTO         = trim(COMP_NAME)             ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = MAPL_RestartRequired        ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                  ,&
    LONG_NAME          = 'snow_depth_layer_2'        ,&
    UNITS              = 'm'                         ,&
    SHORT_NAME         = 'SNDZN2'                    ,&
    FRIENDLYTO         = trim(COMP_NAME)             ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = MAPL_RestartRequired        ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                  ,&
    LONG_NAME          = 'snow_depth_layer_3'        ,&
    UNITS              = 'm'                         ,&
    SHORT_NAME         = 'SNDZN3'                    ,&
    FRIENDLYTO         = trim(COMP_NAME)             ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = MAPL_RestartRequired        ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                  ,&
    LONG_NAME          = 'surface_heat_exchange_coefficient',&
    UNITS              = 'kg m-2 s-1'                ,&
    SHORT_NAME         = 'CH'                        ,&
    DIMS               = MAPL_DimsTileTile           ,&
    NUM_SUBTILES       = NUM_SUBTILES                ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = RESTART                     ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                  ,&
    LONG_NAME          = 'surface_momentum_exchange_coefficient',&
    UNITS              = 'kg m-2 s-1'                ,&
    SHORT_NAME         = 'CM'                        ,&
    DIMS               = MAPL_DimsTileTile           ,&
    NUM_SUBTILES       = NUM_SUBTILES                ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = RESTART                     ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                  ,&
    LONG_NAME          = 'surface_moisture_exchange_coffiecient',&
    UNITS              = 'kg m-2 s-1'                ,&
    SHORT_NAME         = 'CQ'                        ,&
    DIMS               = MAPL_DimsTileTile           ,&
    NUM_SUBTILES       = NUM_SUBTILES                ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = RESTART                     ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                  ,&
    LONG_NAME          = 'subtile_fractions'         ,&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'FR'                        ,&
    DIMS               = MAPL_DimsTileTile           ,&
    NUM_SUBTILES       = NUM_SUBTILES                ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = RESTART                     ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC,                  &
    SHORT_NAME         = 'WW',                        &
    LONG_NAME          = 'vertical_velocity_scale_squared', &
    UNITS              = 'm+2 s-2',                   &
    DIMS               = MAPL_DimsTileTile,           &
    NUM_SUBTILES       = NUM_SUBTILES                ,&
    VLOCATION          = MAPL_VLocationNone,          &
    RESTART            = RESTART                     ,&
                                           RC=STATUS  )
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC,                  &
    SHORT_NAME         = 'DCH',                        &
    LONG_NAME          = 'ch difference, optional in louissurface', &
    UNITS              = '1',                   &
    DIMS               = MAPL_DimsTileTile,           &
    NUM_SUBTILES       = NUM_SUBTILES                ,&
    VLOCATION          = MAPL_VLocationNone,          &
    RESTART            = MAPL_RestartSkip            ,&
                                           RC=STATUS  )
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC,                  &
    SHORT_NAME         = 'DCQ',                        &
    LONG_NAME          = 'cq difference, optional in louissurface', &
    UNITS              = '1',                   &
    DIMS               = MAPL_DimsTileTile,           &
    NUM_SUBTILES       = NUM_SUBTILES                ,&
    VLOCATION          = MAPL_VLocationNone,          &
    RESTART            = MAPL_RestartSkip            ,&
                                           RC=STATUS  )
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                       ,&
    LONG_NAME          = 'catchment_tile_id'         ,&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'TILE_ID'                   ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = MAPL_RestartRequired        ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                       ,&
    LONG_NAME          = 'CLM_nitrogen_deposition'   ,&
    UNITS              = 'g m-2 s-1'                 ,&
    SHORT_NAME         = 'NDEP'                      ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = MAPL_RestartRequired        ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                       ,&
    LONG_NAME          = 'cli_2m_T_(MERRA2)'      ,&
    UNITS              = 'K'                         ,&
    SHORT_NAME         = 'CLI_T2M'                   ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = MAPL_RestartRequired        ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                       ,&
    LONG_NAME          = 'MODIS soil albedo vis dir' ,&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'BGALBVR'                   ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = MAPL_RestartRequired        ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                       ,&
    LONG_NAME          = 'MODIS soil albedo vis dif' ,&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'BGALBVF'                   ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = MAPL_RestartRequired        ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                       ,&
    LONG_NAME          = 'MODIS soil albedo nir dir' ,&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'BGALBNR'                   ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = MAPL_RestartRequired        ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddInternalSpec(GC                       ,&
    LONG_NAME          = 'MODIS soil albedo nir dif' ,&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'BGALBNF'                   ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
    RESTART            = MAPL_RestartRequired        ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  ! Interactive CN model or write out anomalies
  
  call MAPL_AddInternalSpec(GC                       ,&
       LONG_NAME          = 'column_rst_vars'           ,&
       UNITS              = '1'                         ,&
       SHORT_NAME         = 'CNCOL'                     ,&
       DIMS               = MAPL_DimsTileOnly           ,&
       VLOCATION          = MAPL_VLocationNone          ,&
       UNGRIDDED_DIMS     = (/NUM_ZON*VAR_COL/)         ,&
       RESTART            = MAPL_RestartRequired        ,&
       RC=STATUS  ) 
  VERIFY_(STATUS)
  
  call MAPL_AddInternalSpec(GC                       ,&
       LONG_NAME          = 'PFT_rst_vars'              ,&
       UNITS              = '1'                         ,&
       SHORT_NAME         = 'CNPFT'                     ,&
       DIMS               = MAPL_DimsTileOnly           ,&
       VLOCATION          = MAPL_VLocationNone          ,&
       UNGRIDDED_DIMS     = (/NUM_ZON*NUM_VEG*VAR_PFT/) ,&
       RESTART            = MAPL_RestartRequired        ,&
       RC=STATUS  ) 
  VERIFY_(STATUS)
  
  call MAPL_AddInternalSpec(GC                       ,&
       LONG_NAME          = 'CN sum for ground temp'    ,&
       UNITS              = 'K'                         ,&
       SHORT_NAME         = 'TGWM'                      ,&
       DIMS               = MAPL_DimsTileOnly           ,&
       VLOCATION          = MAPL_VLocationNone          ,&
       UNGRIDDED_DIMS     = (/NUM_ZON/)                 ,&
       RESTART            = MAPL_RestartOptional        ,&
       RC=STATUS  ) 
  VERIFY_(STATUS)
  
  call MAPL_AddInternalSpec(GC                       ,&
       LONG_NAME          = 'CN sum for soil moisture'  ,&
       UNITS              = '1'                         ,&
       SHORT_NAME         = 'RZMM'                      ,&
       DIMS               = MAPL_DimsTileOnly           ,&
       VLOCATION          = MAPL_VLocationNone          ,&
       UNGRIDDED_DIMS     = (/NUM_ZON/)                 ,&
       RESTART            = MAPL_RestartOptional        ,&
       RC=STATUS  ) 
  VERIFY_(STATUS)
  
  call MAPL_AddInternalSpec(GC                       ,&
       LONG_NAME          = 'CN sum for sfc soil moist' ,&
       UNITS              = '1'                         ,&
       SHORT_NAME         = 'SFMCM'                     ,&
       DIMS               = MAPL_DimsTileOnly           ,&
       VLOCATION          = MAPL_VLocationNone          ,&
       RESTART            = MAPL_RestartOptional        ,&
       RC=STATUS  ) 
  VERIFY_(STATUS)
  
  call MAPL_AddInternalSpec(GC                       ,&
       LONG_NAME          = 'CN sum for baseflow'       ,&
       UNITS              = 'kg m-2 s-1'                ,&
       SHORT_NAME         = 'BFLOWM'                    ,&
       DIMS               = MAPL_DimsTileOnly           ,&
       VLOCATION          = MAPL_VLocationNone          ,&
       RESTART            = MAPL_RestartOptional        ,&
       RC=STATUS  ) 
  VERIFY_(STATUS)
  
  call MAPL_AddInternalSpec(GC                       ,&
       LONG_NAME          = 'CN sum for total water'    ,&
       UNITS              = 'kg m-2'                    ,&
       SHORT_NAME         = 'TOTWATM'                   ,&
       DIMS               = MAPL_DimsTileOnly           ,&
       VLOCATION          = MAPL_VLocationNone          ,&
       RESTART            = MAPL_RestartOptional        ,&
       RC=STATUS  ) 
  VERIFY_(STATUS)
  
  call MAPL_AddInternalSpec(GC                       ,&
       LONG_NAME          = 'CN sum for air temperature',&
       UNITS              = 'K'                         ,&
       SHORT_NAME         = 'TAIRM'                     ,&
       DIMS               = MAPL_DimsTileOnly           ,&
       VLOCATION          = MAPL_VLocationNone          ,&
       RESTART            = MAPL_RestartOptional        ,&
       RC=STATUS  ) 
  VERIFY_(STATUS)
  
  call MAPL_AddInternalSpec(GC                       ,&
       LONG_NAME          = 'CN sum for soil temp'      ,&
       UNITS              = 'K'                         ,&
       SHORT_NAME         = 'TPM'                       ,&
       DIMS               = MAPL_DimsTileOnly           ,&
       VLOCATION          = MAPL_VLocationNone          ,&
       RESTART            = MAPL_RestartOptional        ,&
       RC=STATUS  ) 
  VERIFY_(STATUS)
  
  call MAPL_AddInternalSpec(GC                       ,&
       LONG_NAME          = 'CN summing counter'        ,&
       UNITS              = '1'                         ,&
       SHORT_NAME         = 'CNSUM'                     ,&
       DIMS               = MAPL_DimsTileOnly           ,&
       VLOCATION          = MAPL_VLocationNone          ,&
       RESTART            = MAPL_RestartOptional        ,&
       RC=STATUS  ) 
  VERIFY_(STATUS)
  
  call MAPL_AddInternalSpec(GC                       ,&
       LONG_NAME          = 'CN sum for sunlit photosyn',&
       UNITS              = 'umol m-2 s-1'              ,&
       SHORT_NAME         = 'PSNSUNM'                   ,&
       DIMS               = MAPL_DimsTileOnly           ,&
       VLOCATION          = MAPL_VLocationNone          ,&
       UNGRIDDED_DIMS     = (/NUM_VEG,NUM_ZON/)         ,&
       RESTART            = MAPL_RestartOptional        ,&
       RC=STATUS  ) 
  VERIFY_(STATUS)
  
  call MAPL_AddInternalSpec(GC                       ,&
       LONG_NAME          = 'CN sum for shaded photosyn',&
       UNITS              = 'umol m-2 s-1'              ,&
       SHORT_NAME         = 'PSNSHAM'                   ,&
       DIMS               = MAPL_DimsTileOnly           ,&
       VLOCATION          = MAPL_VLocationNone          ,&
       UNGRIDDED_DIMS     = (/NUM_VEG,NUM_ZON/)         ,&
       RESTART            = MAPL_RestartOptional        ,&
       RC=STATUS  ) 
  VERIFY_(STATUS)
  
  call MAPL_AddInternalSpec(GC                       ,&
       LONG_NAME          = 'CN sum for snow depth'     ,&
       UNITS              = 'm'                         ,&
       SHORT_NAME         = 'SNDZM'                     ,&
       DIMS               = MAPL_DimsTileOnly           ,&
       VLOCATION          = MAPL_VLocationNone          ,&
       RESTART            = MAPL_RestartOptional        ,&
       RC=STATUS  ) 
  VERIFY_(STATUS)
  
  call MAPL_AddInternalSpec(GC                       ,&
       LONG_NAME          = 'CN sum for area snow cover',&
       UNITS              = '1'                         ,&
       SHORT_NAME         = 'ASNOWM'                    ,&
       DIMS               = MAPL_DimsTileOnly           ,&
       VLOCATION          = MAPL_VLocationNone          ,&
       RESTART            = MAPL_RestartOptional        ,&
       RC=STATUS  ) 
  VERIFY_(STATUS)
  
  IF (PRESCRIBE_DVG == 3) THEN
     ! Add ESAI (NTILES,NV,NZ)
     ! LAI/SAI in forecast system: 3 S2S reading ; 4 GEOSldas writing 

     call MAPL_AddInternalSpec(GC                         ,&
          LONG_NAME          = 'Stem Area Index V1 Z1 anomaly'     ,&
          UNITS              = '1'                         ,&
          SHORT_NAME         = 'CNSAI11A'                  ,&
          DIMS               = MAPL_DimsTileOnly           ,&
          VLOCATION          = MAPL_VLocationNone          ,&
          RESTART            = MAPL_RestartOptional        ,&
          RC=STATUS  ) 
     VERIFY_(STATUS)
     call MAPL_AddInternalSpec(GC                         ,&
          LONG_NAME          = 'Stem Area Index V1 Z2 anomaly'     ,&
          UNITS              = '1'                         ,&
          SHORT_NAME         = 'CNSAI12A'                  ,&
          DIMS               = MAPL_DimsTileOnly           ,&
          VLOCATION          = MAPL_VLocationNone          ,&
          RESTART            = MAPL_RestartOptional        ,&
          RC=STATUS  ) 
     VERIFY_(STATUS)
     call MAPL_AddInternalSpec(GC                         ,&
          LONG_NAME          = 'Stem Area Index V1 Z3 anomaly'     ,&
          UNITS              = '1'                         ,&
          SHORT_NAME         = 'CNSAI13A'                  ,&
          DIMS               = MAPL_DimsTileOnly           ,&
          VLOCATION          = MAPL_VLocationNone          ,&
          RESTART            = MAPL_RestartOptional        ,&
          RC=STATUS  ) 
     VERIFY_(STATUS)
     
     call MAPL_AddInternalSpec(GC                         ,&
          LONG_NAME          = 'Stem Area Index V2 Z1 anomaly'     ,&
          UNITS              = '1'                         ,&
          SHORT_NAME         = 'CNSAI21A'                  ,&
          DIMS               = MAPL_DimsTileOnly           ,&
          VLOCATION          = MAPL_VLocationNone          ,&
          RESTART            = MAPL_RestartOptional        ,&
          RC=STATUS  ) 
     VERIFY_(STATUS)
     call MAPL_AddInternalSpec(GC                         ,&
          LONG_NAME          = 'Stem Area Index V2 Z2 anomaly'     ,&
          UNITS              = '1'                         ,&
          SHORT_NAME         = 'CNSAI22A'                  ,&
          DIMS               = MAPL_DimsTileOnly           ,&
          VLOCATION          = MAPL_VLocationNone          ,&
          RESTART            = MAPL_RestartOptional        ,&
          RC=STATUS  ) 
     VERIFY_(STATUS)
     call MAPL_AddInternalSpec(GC                         ,&
          LONG_NAME          = 'Stem Area Index V2 Z3 anomaly'     ,&
          UNITS              = '1'                         ,&
          SHORT_NAME         = 'CNSAI23A'                  ,&
          DIMS               = MAPL_DimsTileOnly           ,&
          VLOCATION          = MAPL_VLocationNone          ,&
          RESTART            = MAPL_RestartOptional        ,&
          RC=STATUS  ) 
     VERIFY_(STATUS)
     
     call MAPL_AddInternalSpec(GC                         ,&
          LONG_NAME          = 'Stem Area Index V3 Z1 anomaly'     ,&
          UNITS              = '1'                         ,&
          SHORT_NAME         = 'CNSAI31A'                  ,&
          DIMS               = MAPL_DimsTileOnly           ,&
          VLOCATION          = MAPL_VLocationNone          ,&
          RESTART            = MAPL_RestartOptional        ,&
          RC=STATUS  ) 
     VERIFY_(STATUS)
     call MAPL_AddInternalSpec(GC                         ,&
          LONG_NAME          = 'Stem Area Index V3 Z2 anomaly'     ,&
          UNITS              = '1'                         ,&
          SHORT_NAME         = 'CNSAI32A'                  ,&
          DIMS               = MAPL_DimsTileOnly           ,&
          VLOCATION          = MAPL_VLocationNone          ,&
          RESTART            = MAPL_RestartOptional        ,&
          RC=STATUS  ) 
     VERIFY_(STATUS)
     call MAPL_AddInternalSpec(GC                         ,&
          LONG_NAME          = 'Stem Area Index V3 Z3 anomaly'     ,&
          UNITS              = '1'                         ,&
          SHORT_NAME         = 'CNSAI33A'                  ,&
          DIMS               = MAPL_DimsTileOnly           ,&
          VLOCATION          = MAPL_VLocationNone          ,&
          RESTART            = MAPL_RestartOptional        ,&
          RC=STATUS  ) 
     VERIFY_(STATUS)
     
     call MAPL_AddInternalSpec(GC                         ,&
          LONG_NAME          = 'Stem Area Index V4 Z1 anomaly'     ,&
          UNITS              = '1'                         ,&
          SHORT_NAME         = 'CNSAI41A'                  ,&
          DIMS               = MAPL_DimsTileOnly           ,&
          VLOCATION          = MAPL_VLocationNone          ,&
          RESTART            = MAPL_RestartOptional        ,&
          RC=STATUS  ) 
     VERIFY_(STATUS)
     call MAPL_AddInternalSpec(GC                         ,&
          LONG_NAME          = 'Stem Area Index V4 Z2 anomaly'     ,&
          UNITS              = '1'                         ,&
          SHORT_NAME         = 'CNSAI42A'                  ,&
          DIMS               = MAPL_DimsTileOnly           ,&
          VLOCATION          = MAPL_VLocationNone          ,&
          RESTART            = MAPL_RestartOptional        ,&
          RC=STATUS  ) 
     VERIFY_(STATUS)
     call MAPL_AddInternalSpec(GC                         ,&
          LONG_NAME          = 'Stem Area Index V4 Z3 anomaly'     ,&
          UNITS              = '1'                         ,&
          SHORT_NAME         = 'CNSAI43A'                  ,&
          DIMS               = MAPL_DimsTileOnly           ,&
          VLOCATION          = MAPL_VLocationNone          ,&
          RESTART            = MAPL_RestartOptional        ,&
          RC=STATUS  ) 
     VERIFY_(STATUS)
     
     ! Add ELAI (NTILES,NV,NZ)
     
     call MAPL_AddInternalSpec(GC                         ,&
          LONG_NAME          = 'Leaf Area Index V1 Z1 anomaly'     ,&
          UNITS              = '1'                         ,&
          SHORT_NAME         = 'CNLAI11A'                  ,&
          DIMS               = MAPL_DimsTileOnly           ,&
          VLOCATION          = MAPL_VLocationNone          ,&
          RESTART            = MAPL_RestartOptional        ,&
          RC=STATUS  ) 
     VERIFY_(STATUS)
     call MAPL_AddInternalSpec(GC                         ,&
          LONG_NAME          = 'Leaf Area Index V1 Z2 anomaly'     ,&
          UNITS              = '1'                         ,&
          SHORT_NAME         = 'CNLAI12A'                  ,&
          DIMS               = MAPL_DimsTileOnly           ,&
          VLOCATION          = MAPL_VLocationNone          ,&
          RESTART            = MAPL_RestartOptional        ,&
          RC=STATUS  ) 
     VERIFY_(STATUS)
     call MAPL_AddInternalSpec(GC                         ,&
          LONG_NAME          = 'Leaf Area Index V1 Z3 anomaly'     ,&
          UNITS              = '1'                         ,&
          SHORT_NAME         = 'CNLAI13A'                  ,&
          DIMS               = MAPL_DimsTileOnly           ,&
          VLOCATION          = MAPL_VLocationNone          ,&
          RESTART            = MAPL_RestartOptional        ,&
          RC=STATUS  ) 
     VERIFY_(STATUS)
     
     call MAPL_AddInternalSpec(GC                         ,&
          LONG_NAME          = 'Leaf Area Index V2 Z1 anomaly'     ,&
          UNITS              = '1'                         ,&
          SHORT_NAME         = 'CNLAI21A'                  ,&
          DIMS               = MAPL_DimsTileOnly           ,&
          VLOCATION          = MAPL_VLocationNone          ,&
          RESTART            = MAPL_RestartOptional        ,&
          RC=STATUS  ) 
     VERIFY_(STATUS)
     call MAPL_AddInternalSpec(GC                         ,&
          LONG_NAME          = 'Leaf Area Index V2 Z2 anomaly'     ,&
          UNITS              = '1'                         ,&
          SHORT_NAME         = 'CNLAI22A'                  ,&
          DIMS               = MAPL_DimsTileOnly           ,&
          VLOCATION          = MAPL_VLocationNone          ,&
          RESTART            = MAPL_RestartOptional        ,&
          RC=STATUS  ) 
     VERIFY_(STATUS)
     call MAPL_AddInternalSpec(GC                         ,&
          LONG_NAME          = 'Leaf Area Index V2 Z3 anomaly'     ,&
          UNITS              = '1'                         ,&
          SHORT_NAME         = 'CNLAI23A'                  ,&
          DIMS               = MAPL_DimsTileOnly           ,&
          VLOCATION          = MAPL_VLocationNone          ,&
          RESTART            = MAPL_RestartOptional        ,&
          RC=STATUS  ) 
     VERIFY_(STATUS)
     
     call MAPL_AddInternalSpec(GC                         ,&
          LONG_NAME          = 'Leaf Area Index V3 Z1 anomaly'     ,&
          UNITS              = '1'                         ,&
          SHORT_NAME         = 'CNLAI31A'                  ,&
          DIMS               = MAPL_DimsTileOnly           ,&
          VLOCATION          = MAPL_VLocationNone          ,&
          RESTART            = MAPL_RestartOptional        ,&
          RC=STATUS  ) 
     VERIFY_(STATUS)
     call MAPL_AddInternalSpec(GC                         ,&
          LONG_NAME          = 'Leaf Area Index V3 Z2 anomaly'     ,&
          UNITS              = '1'                         ,&
          SHORT_NAME         = 'CNLAI32A'                  ,&
          DIMS               = MAPL_DimsTileOnly           ,&
          VLOCATION          = MAPL_VLocationNone          ,&
          RESTART            = MAPL_RestartOptional        ,&
          RC=STATUS  ) 
     VERIFY_(STATUS)
     call MAPL_AddInternalSpec(GC                         ,&
          LONG_NAME          = 'Leaf Area Index V3 Z3 anomaly'     ,&
          UNITS              = '1'                         ,&
          SHORT_NAME         = 'CNLAI33A'                  ,&
          DIMS               = MAPL_DimsTileOnly           ,&
          VLOCATION          = MAPL_VLocationNone          ,&
          RESTART            = MAPL_RestartOptional        ,&
          RC=STATUS  ) 
     VERIFY_(STATUS)
     
     call MAPL_AddInternalSpec(GC                         ,&
          LONG_NAME          = 'Leaf Area Index V4 Z1 anomaly'     ,&
          UNITS              = '1'                         ,&
          SHORT_NAME         = 'CNLAI41A'                  ,&
          DIMS               = MAPL_DimsTileOnly           ,&
          VLOCATION          = MAPL_VLocationNone          ,&
          RESTART            = MAPL_RestartOptional        ,&
          RC=STATUS  ) 
     VERIFY_(STATUS)
     call MAPL_AddInternalSpec(GC                         ,&
          LONG_NAME          = 'Leaf Area Index V4 Z2 anomaly'     ,&
          UNITS              = '1'                         ,&
          SHORT_NAME         = 'CNLAI42A'                  ,&
          DIMS               = MAPL_DimsTileOnly           ,&
          VLOCATION          = MAPL_VLocationNone          ,&
          RESTART            = MAPL_RestartOptional        ,&
          RC=STATUS  ) 
     VERIFY_(STATUS)
     call MAPL_AddInternalSpec(GC                         ,&
          LONG_NAME          = 'Leaf Area Index V4 Z3 anomaly'     ,&
          UNITS              = '1'                         ,&
          SHORT_NAME         = 'CNLAI43A'                  ,&
          DIMS               = MAPL_DimsTileOnly           ,&
          VLOCATION          = MAPL_VLocationNone          ,&
          RESTART            = MAPL_RestartOptional        ,&
          RC=STATUS  ) 
     VERIFY_(STATUS)
     
  endif

  !---------- GOSWIM snow impurity related variables ----------

  if (N_CONST_LAND4SNWALB /= 0) then 
  
     call MAPL_AddInternalSpec(GC                  ,&
          LONG_NAME          = 'dust_mass_in_snow_bin_1'   ,&
          UNITS              = 'kg m-2'                    ,&
          SHORT_NAME         = 'RDU001'                    ,&
          FRIENDLYTO         = trim(COMP_NAME)             ,&
          DIMS               = MAPL_DimsTileOnly           ,&
          UNGRIDDED_DIMS     = (/N_SNOW/)                  ,&
          VLOCATION          = MAPL_VLocationNone          ,&
          RESTART            = MAPL_RestartOptional        ,&
          RC=STATUS  ) 
     VERIFY_(STATUS)
     
     call MAPL_AddInternalSpec(GC                  ,&
          LONG_NAME          = 'dust_mass_in_snow_bin_2'   ,&
          UNITS              = 'kg m-2'                    ,&
          SHORT_NAME         = 'RDU002'                    ,&
          FRIENDLYTO         = trim(COMP_NAME)             ,&
          DIMS               = MAPL_DimsTileOnly           ,&
          UNGRIDDED_DIMS     = (/N_SNOW/)                  ,&
          VLOCATION          = MAPL_VLocationNone          ,&
          RESTART            = MAPL_RestartOptional        ,&
          RC=STATUS  ) 
     VERIFY_(STATUS)
     
     call MAPL_AddInternalSpec(GC                  ,&
          LONG_NAME          = 'dust_mass_in_snow_bin_3'   ,&
          UNITS              = 'kg m-2'                    ,&
          SHORT_NAME         = 'RDU003'                    ,&
          FRIENDLYTO         = trim(COMP_NAME)             ,&
          DIMS               = MAPL_DimsTileOnly           ,&
          UNGRIDDED_DIMS     = (/N_SNOW/)                  ,&
          VLOCATION          = MAPL_VLocationNone          ,&
          RESTART            = MAPL_RestartOptional        ,&
          RC=STATUS  ) 
     VERIFY_(STATUS)
     
     call MAPL_AddInternalSpec(GC                  ,&
          LONG_NAME          = 'dust_mass_in_snow_bin_4'   ,&
          UNITS              = 'kg m-2'                    ,&
          SHORT_NAME         = 'RDU004'                    ,&
          FRIENDLYTO         = trim(COMP_NAME)             ,&
          DIMS               = MAPL_DimsTileOnly           ,&
          UNGRIDDED_DIMS     = (/N_SNOW/)                  ,&
          VLOCATION          = MAPL_VLocationNone          ,&
          RESTART            = MAPL_RestartOptional        ,&
          RC=STATUS  ) 
     VERIFY_(STATUS)
     
     call MAPL_AddInternalSpec(GC                  ,&
          LONG_NAME          = 'dust_mass_in_snow_bin_5'   ,&
          UNITS              = 'kg m-2'                    ,&
          SHORT_NAME         = 'RDU005'                    ,&
          FRIENDLYTO         = trim(COMP_NAME)             ,&
          DIMS               = MAPL_DimsTileOnly           ,&
          UNGRIDDED_DIMS     = (/N_SNOW/)                  ,&
          VLOCATION          = MAPL_VLocationNone          ,&
          RESTART            = MAPL_RestartOptional        ,&
          RC=STATUS  ) 
     VERIFY_(STATUS)
     
     call MAPL_AddInternalSpec(GC                  ,&
          LONG_NAME          = 'hydrophobic_black_carbon_mass_in_snow_bin_1',&
          UNITS              = 'kg m-2'                    ,&
          SHORT_NAME         = 'RBC001'                    ,&
          FRIENDLYTO         = trim(COMP_NAME)             ,&
          DIMS               = MAPL_DimsTileOnly           ,&
          UNGRIDDED_DIMS     = (/N_SNOW/)                  ,&
          VLOCATION          = MAPL_VLocationNone          ,&
          RESTART            = MAPL_RestartOptional        ,&
          RC=STATUS  ) 
     VERIFY_(STATUS)
     
     call MAPL_AddInternalSpec(GC                  ,&
          LONG_NAME          = 'hydrophilic_black_carbon_mass_in_snow_bin_2',&
          UNITS              = 'kg m-2'                    ,&
          SHORT_NAME         = 'RBC002'                    ,&
          FRIENDLYTO         = trim(COMP_NAME)             ,&
          DIMS               = MAPL_DimsTileOnly           ,&
          UNGRIDDED_DIMS     = (/N_SNOW/)                  ,&
          VLOCATION          = MAPL_VLocationNone          ,&
          RESTART            = MAPL_RestartOptional        ,&
          RC=STATUS  ) 
     VERIFY_(STATUS)
     
     call MAPL_AddInternalSpec(GC                  ,&
          LONG_NAME          = 'hydrophobic_organic_carbon_mass_in_snow_bin_1',&
          UNITS              = 'kg m-2'                    ,&
          SHORT_NAME         = 'ROC001'                    ,&
          FRIENDLYTO         = trim(COMP_NAME)             ,&
          DIMS               = MAPL_DimsTileOnly           ,&
          UNGRIDDED_DIMS     = (/N_SNOW/)                  ,&
          VLOCATION          = MAPL_VLocationNone          ,&
          RESTART            = MAPL_RestartOptional        ,&
          RC=STATUS  ) 
     VERIFY_(STATUS)
     
     call MAPL_AddInternalSpec(GC                  ,&
          LONG_NAME          = 'hydrophilic_organic_carbon_mass_in_snow_bin_2',&
          UNITS              = 'kg m-2'                    ,&
          SHORT_NAME         = 'ROC002'                    ,&
          FRIENDLYTO         = trim(COMP_NAME)             ,&
          DIMS               = MAPL_DimsTileOnly           ,&
          UNGRIDDED_DIMS     = (/N_SNOW/)                  ,&
          VLOCATION          = MAPL_VLocationNone          ,&
          RESTART            = MAPL_RestartOptional        ,&
          RC=STATUS  ) 
     VERIFY_(STATUS)
     
  endif

! IRRIGATION MODEL INTERNAL

  IF (RUN_IRRIG /= 0) THEN

     call MAPL_AddInternalSpec(GC                  ,&
          LONG_NAME          = 'fraction_of_irrigated_cropland',&
          UNITS              = '1'                         ,&
          SHORT_NAME         = 'IRRIGFRAC'                 ,&
          FRIENDLYTO         = trim(COMP_NAME)             ,&
          DIMS               = MAPL_DimsTileOnly           ,&
          VLOCATION          = MAPL_VLocationNone          ,&
          RESTART            = MAPL_RestartRequired        ,&
          RC=STATUS  ) 
     VERIFY_(STATUS)
     
     call MAPL_AddInternalSpec(GC                  ,&
          LONG_NAME          = 'fraction_of_paddy_cropland',&
          UNITS              = '1'                         ,&
          SHORT_NAME         = 'PADDYFRAC'                 ,&
          FRIENDLYTO         = trim(COMP_NAME)             ,&
          DIMS               = MAPL_DimsTileOnly           ,&
          VLOCATION          = MAPL_VLocationNone          ,&
          RESTART            = MAPL_RestartRequired        ,&
          RC=STATUS  ) 
     VERIFY_(STATUS)
     
     
     call MAPL_AddInternalSpec(GC                          ,&
          LONG_NAME          = 'Maximum_LAI'               ,&
          UNITS              = '1'                         ,&
          SHORT_NAME         = 'LAIMAX'                    ,&
          FRIENDLYTO         = trim(COMP_NAME)             ,&
          DIMS               = MAPL_DimsTileOnly           ,&
          VLOCATION          = MAPL_VLocationNone          ,&
          RESTART            = MAPL_RestartRequired        ,&
          RC=STATUS  ) 
     VERIFY_(STATUS)
     
     call MAPL_AddInternalSpec(GC                          ,&
          LONG_NAME          = 'Minimum_LAI'               ,&
          UNITS              = '1'                         ,&
          SHORT_NAME         = 'LAIMIN'                    ,&
          FRIENDLYTO         = trim(COMP_NAME)             ,&
          DIMS               = MAPL_DimsTileOnly           ,&
          VLOCATION          = MAPL_VLocationNone          ,&
          RESTART            = MAPL_RestartRequired        ,&
          RC=STATUS  ) 
     VERIFY_(STATUS)
     
     call MAPL_AddInternalSpec(GC                          ,&
          LONG_NAME          = 'CLM_primary_type'          ,&
          UNITS              = '1'                         ,&
          SHORT_NAME         = 'CLMPT'                     ,&
          FRIENDLYTO         = trim(COMP_NAME)             ,&
          DIMS               = MAPL_DimsTileOnly           ,&
          VLOCATION          = MAPL_VLocationNone          ,&
          RESTART            = MAPL_RestartRequired        ,&
          RC=STATUS  ) 
     VERIFY_(STATUS)
     
     call MAPL_AddInternalSpec(GC                          ,&
          LONG_NAME          = 'CLM_secondary_type'        ,&
          UNITS              = '1'                         ,&
          SHORT_NAME         = 'CLMST'                     ,&
          FRIENDLYTO         = trim(COMP_NAME)             ,&
          DIMS               = MAPL_DimsTileOnly           ,&
          VLOCATION          = MAPL_VLocationNone          ,&
          RESTART            = MAPL_RestartRequired        ,&
          RC=STATUS  ) 
     VERIFY_(STATUS)
     
     call MAPL_AddInternalSpec(GC                          ,&
          LONG_NAME          = 'CLM_primary_fraction'      ,&
          UNITS              = '1'                         ,&
          SHORT_NAME         = 'CLMPF'                     ,&
          FRIENDLYTO         = trim(COMP_NAME)             ,&
          DIMS               = MAPL_DimsTileOnly           ,&
          VLOCATION          = MAPL_VLocationNone          ,&
          RESTART            = MAPL_RestartRequired        ,&
          RC=STATUS  ) 
     VERIFY_(STATUS)
     
     call MAPL_AddInternalSpec(GC                          ,&
          LONG_NAME          = 'CLM_secondary_fraction'    ,&
          UNITS              = '1'                         ,&
          SHORT_NAME         = 'CLMSF'                     ,&
          FRIENDLYTO         = trim(COMP_NAME)             ,&
          DIMS               = MAPL_DimsTileOnly           ,&
          VLOCATION          = MAPL_VLocationNone          ,&
          RESTART            = MAPL_RestartRequired        ,&
          RC=STATUS  ) 
     VERIFY_(STATUS)
     
  ENDIF


!EOS

  !  EXPORT STATE:

  IF (RUN_IRRIG /= 0) THEN
     call MAPL_AddExportSpec(GC                            ,&
          LONG_NAME          = 'irrigation_rate'           ,&
          UNITS              = 'kg m-2 s-1'                ,&
          SHORT_NAME         = 'IRRIGRATE'                 ,&
          DIMS               = MAPL_DimsTileOnly           ,&
          VLOCATION          = MAPL_VLocationNone          ,&
          RC=STATUS  ) 
     VERIFY_(STATUS)
  ENDIF
  
  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'evaporation'               ,&
    UNITS              = 'kg m-2 s-1'                ,&
    SHORT_NAME         = 'EVAPOUT'                   ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

     call MAPL_AddExportSpec(GC,                     &
     LONG_NAME          = 'sublimation'               ,&
     UNITS              = 'kg m-2 s-1'                ,&
     SHORT_NAME         = 'SUBLIM'                    ,&
     DIMS               = MAPL_DimsTileOnly           ,&
     VLOCATION          = MAPL_VLocationNone          ,&
                                               RC=STATUS  ) 
     VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'upward_sensible_heat_flux' ,&
    UNITS              = 'W m-2'                     ,&
    SHORT_NAME         = 'SHOUT'                     ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'runoff_flux'               ,&
    UNITS              = 'kg m-2 s-1'                ,&
    SHORT_NAME         = 'RUNOFF'                    ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'interception_loss_energy_flux',&
    UNITS              = 'W m-2'                     ,&
    SHORT_NAME         = 'EVPINT'                    ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'baresoil_evap_energy_flux' ,&
    UNITS              = 'W m-2'                     ,&
    SHORT_NAME         = 'EVPSOI'                    ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'transpiration_energy_flux' ,&
    UNITS              = 'W m-2'                     ,&
    SHORT_NAME         = 'EVPVEG'                    ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'snow_ice_evaporation_energy_flux',&
    UNITS              = 'W m-2'                     ,&
    SHORT_NAME         = 'EVPICE'                    ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)


  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'soil moisture in Upper 10cm'     ,&
    UNITS              = 'kg m-2'                    ,&
    SHORT_NAME         = 'WAT10CM'                   ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  )
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'totoal soil moisture'      ,&
    UNITS              = 'kg m-2'                    ,&
    SHORT_NAME         = 'WATSOI'                    ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  )
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'soil frozen water content' ,&
    UNITS              = 'kg m-2'                    ,&
    SHORT_NAME         = 'ICESOI'                    ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  )
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'snowpack_evaporation_energy_flux',&
    UNITS              = 'W m-2'                     ,&
    SHORT_NAME         = 'EVPSNO'                    ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  )
  VERIFY_(STATUS)
  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'baseflow_flux'             ,&
    UNITS              = 'kg m-2 s-1'                ,&
    SHORT_NAME         = 'BASEFLOW'                  ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'overland_runoff_including_throughflow'  ,&
    UNITS              = 'kg m-2 s-1'                ,&
    SHORT_NAME         = 'RUNSURF'                   ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'snowmelt_flux'             ,&
    UNITS              = 'kg m-2 s-1'                ,&
    SHORT_NAME         = 'SMELT'                     ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'surface_outgoing_longwave_flux',&
    UNITS              = 'W m-2'                     ,&
    SHORT_NAME         = 'HLWUP'                     ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC                    ,&
    LONG_NAME          = 'surface_net_downward_longwave_flux',&
    UNITS              = 'W m-2'                     ,&
    SHORT_NAME         = 'LWNDSRF'                   ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
    VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC                    ,&
    LONG_NAME          = 'surface_net_downward_shortwave_flux',&
    UNITS              = 'W m-2'                     ,&
    SHORT_NAME         = 'SWNDSRF'                   ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
    VERIFY_(STATUS)


  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'total_latent_energy_flux'  ,&
    UNITS              = 'W m-2'                     ,&
    SHORT_NAME         = 'HLATN'                     ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'rainwater_infiltration_flux',&
    UNITS              = 'kg m-2 s-1'                ,&
    SHORT_NAME         = 'QINFIL'                    ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'areal_fraction_saturated_zone',&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'AR1'                       ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'areal_fraction_transpiration_zone',&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'AR2'                       ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'root_zone_equilibrium_moisture',&
    UNITS              = 'kg m-2'                    ,&
    SHORT_NAME         = 'RZEQ'                      ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'ground_energy_flux'        ,&
    UNITS              = 'W m-2'                     ,&
    SHORT_NAME         = 'GHFLX'                     ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'ave_catchment_temp_incl_snw',& 
    UNITS              = 'K'                         ,&
    SHORT_NAME         = 'TPSURF'                    ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'temperature_top_snow_layer',&
    UNITS              = 'K'                         ,&
    SHORT_NAME         = 'TPSNOW'                    ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'temperature_unsaturated_zone',&
    UNITS              = 'K'                         ,&
    SHORT_NAME         = 'TPUNST'                    ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'temperature_saturated_zone',&
    UNITS              = 'K'                         ,&
    SHORT_NAME         = 'TPSAT'                     ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'temperature_wilted_zone'   ,&
    UNITS              = 'K'                         ,&
    SHORT_NAME         = 'TPWLT'                     ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'fractional_area_of_land_snowcover',&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'ASNOW'                     ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'downward_heat_flux_into_snow',&
    UNITS              = 'W m-2'                     ,&
    SHORT_NAME         = 'SHSNOW'                    ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  )
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'averaged_snow_temperature' ,&
    UNITS              = 'K'                         ,&
    SHORT_NAME         = 'AVETSNOW'                  ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  )
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'fractional_area_of_saturated_zone',&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'FRSAT'                     ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'fractional_area_of_unsaturated_zone',&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'FRUST'                     ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'fractional_area_of_wilting_zone',&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'FRWLT'                     ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'snow_mass'                 ,&
    UNITS              = 'kg m-2'                    ,&
    SHORT_NAME         = 'SNOWMASS'                  ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'snow_depth_within_snow_covered_area_fraction' ,&
    UNITS              = 'm'                         ,&
    SHORT_NAME         = 'SNOWDP'                    ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'surface_soil_wetness'      ,&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'WET1'                      ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'root_zone_soil_wetness'    ,&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'WET2'                      ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'ave_prof_soil__moisture'   ,&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'WET3'                      ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'water_surface_layer'       ,&
    UNITS              = 'm3 m-3'                    ,&
    SHORT_NAME         = 'WCSF'                      ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  )
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'water_root_zone'           ,&
    UNITS              = 'm3 m-3'                    ,&
    SHORT_NAME         = 'WCRZ'                      ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  )
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'water_ave_prof'            ,&
    UNITS              = 'm3 m-3'                   ,&
    SHORT_NAME         = 'WCPR'                      ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  )
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'soil_temperatures_layer_1' ,&
    UNITS              = 'K'                         ,&  ! units now K, rreichle & borescan, 6 Nov 2020
    SHORT_NAME         = 'TP1'                       ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'soil_temperatures_layer_2' ,&
    UNITS              = 'K'                         ,&  ! units now K, rreichle & borescan, 6 Nov 2020
    SHORT_NAME         = 'TP2'                       ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'soil_temperatures_layer_3' ,&
    UNITS              = 'K'                         ,&  ! units now K, rreichle & borescan, 6 Nov 2020
    SHORT_NAME         = 'TP3'                       ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'soil_temperatures_layer_4' ,&
    UNITS              = 'K'                         ,&  ! units now K, rreichle & borescan, 6 Nov 2020
    SHORT_NAME         = 'TP4'                       ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'soil_temperatures_layer_5' ,&
    UNITS              = 'K'                         ,&  ! units now K, rreichle & borescan, 6 Nov 2020
    SHORT_NAME         = 'TP5'                       ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'soil_temperatures_layer_6' ,&
    UNITS              = 'K'                         ,&  ! units now K, rreichle & borescan, 6 Nov 2020
    SHORT_NAME         = 'TP6'                       ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'surface_emissivity'        ,&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'EMIS'                      ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'surface_albedo_visible_beam',&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'ALBVR'                     ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'surface_albedo_visible_diffuse',&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'ALBVF'                     ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'surface_albedo_near_infrared_beam',&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'ALBNR'                     ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'surface_albedo_near_infrared_diffuse',&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'ALBNF'                     ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'change_surface_skin_temperature',&
    UNITS              = 'K'                         ,&
    SHORT_NAME         = 'DELTS'                     ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'change_surface_specific_humidity',&
    UNITS              = 'kg kg-1'                   ,&
    SHORT_NAME         = 'DELQS'                     ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'change_evaporation'        ,&
    UNITS              = 'kg m-2 s-1'                ,&
    SHORT_NAME         = 'DELEVAP'                   ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'change_upward_sensible_energy_flux',&
    UNITS              = 'W m-2'                     ,&
    SHORT_NAME         = 'DELSH'                     ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'surface_skin_temperature'  ,&
    UNITS              = 'K'                         ,&
    SHORT_NAME         = 'TST'                       ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'land_surface_skin_temperature'  ,&
    UNITS              = 'K'                         ,&
    SHORT_NAME         = 'LST'                       ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'surface_specific_humidity' ,&
    UNITS              = 'kg kg-1'                   ,&
    SHORT_NAME         = 'QST'                       ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'turbulence_surface_skin_temperature',&
    UNITS              = 'K'                         ,&
    SHORT_NAME         = 'TH'                        ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'turbulence_surface_skin_specific_hum',&
    UNITS              = 'kg kg-1'                   ,&
    SHORT_NAME         = 'QH'                        ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'surface_heat_exchange_coefficient',&
    UNITS              = 'kg m-2 s-1'                ,&
    SHORT_NAME         = 'CHT'                       ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'surface_momentum_exchange_coefficient',&
    UNITS              = 'kg m-2 s-1'                ,&
    SHORT_NAME         = 'CMT'                       ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'surface_moisture_exchange_coefficient',&
    UNITS              = 'kg m-2 s-1'                ,&
    SHORT_NAME         = 'CQT'                       ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'neutral_drag_coefficient'  ,&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'CNT'                       ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'surface_bulk_richardson_number',&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'RIT'                       ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'surface_roughness'         ,&
    UNITS              = 'm'                         ,&
    SHORT_NAME         = 'Z0'                        ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

     call MAPL_AddExportSpec(GC,                    &
        SHORT_NAME         = 'MOT2M',                     &
        LONG_NAME          = 'temperature 2m wind from MO sfc', &
        UNITS              = 'K',                         &
        DIMS               = MAPL_DimsTileOnly           ,&
        VLOCATION          = MAPL_VLocationNone,          &
                                               RC=STATUS  )
     VERIFY_(STATUS)

     call MAPL_AddExportSpec(GC,                    &
        SHORT_NAME         = 'MOQ2M',                     &
        LONG_NAME          = 'humidity 2m wind from MO sfc',    &
        UNITS              = 'kg kg-1',                   &
        DIMS               = MAPL_DimsTileOnly           ,&
        VLOCATION          = MAPL_VLocationNone,          &
                                               RC=STATUS  )
     VERIFY_(STATUS)

     call MAPL_AddExportSpec(GC,                    &
        SHORT_NAME         = 'MOU2M',                    &
        LONG_NAME          = 'zonal 2m wind from MO sfc',&
        UNITS              = 'm s-1',                     &
        DIMS               = MAPL_DimsTileOnly           ,&
        VLOCATION          = MAPL_VLocationNone,          &
                                               RC=STATUS  )
     VERIFY_(STATUS)

     call MAPL_AddExportSpec(GC,                    &
        SHORT_NAME         = 'MOV2M',                    &
        LONG_NAME          = 'meridional 2m wind from MO sfc', &
        UNITS              = 'm s-1',                     &
        DIMS               = MAPL_DimsTileOnly           ,&
        VLOCATION          = MAPL_VLocationNone,          &
                                               RC=STATUS  )
     VERIFY_(STATUS)

     call MAPL_AddExportSpec(GC,                    &
        SHORT_NAME         = 'MOT10M',                     &
        LONG_NAME          = 'temperature 10m wind from MO sfc', &
        UNITS              = 'K',                         &
        DIMS               = MAPL_DimsTileOnly           ,&
        VLOCATION          = MAPL_VLocationNone,          &
                                               RC=STATUS  )
     VERIFY_(STATUS)

     call MAPL_AddExportSpec(GC,                    &
        SHORT_NAME         = 'MOQ10M',                     &
        LONG_NAME          = 'humidity 10m wind from MO sfc',    &
        UNITS              = 'kg kg-1',                   &
        DIMS               = MAPL_DimsTileOnly           ,&
        VLOCATION          = MAPL_VLocationNone,          &
                                               RC=STATUS  )
     VERIFY_(STATUS)

     call MAPL_AddExportSpec(GC,                    &
        SHORT_NAME         = 'MOU10M',                    &
        LONG_NAME          = 'zonal 10m wind from MO sfc',&
        UNITS              = 'm s-1',                     &
        DIMS               = MAPL_DimsTileOnly           ,&
        VLOCATION          = MAPL_VLocationNone,          &
                                               RC=STATUS  )
     VERIFY_(STATUS)

     call MAPL_AddExportSpec(GC,                    &
        SHORT_NAME         = 'MOV10M',                    &
        LONG_NAME          = 'meridional 10m wind from MO sfc', &
        UNITS              = 'm s-1',                     &
        DIMS               = MAPL_DimsTileOnly           ,&
        VLOCATION          = MAPL_VLocationNone,          &
                                               RC=STATUS  )
     VERIFY_(STATUS)

     call MAPL_AddExportSpec(GC,                    &
        SHORT_NAME         = 'MOU50M',                    &
        LONG_NAME          = 'zonal 50m wind from MO sfc',&
        UNITS              = 'm s-1',                     &
        DIMS               = MAPL_DimsTileOnly           ,&
        VLOCATION          = MAPL_VLocationNone,          &
                                               RC=STATUS  )
     VERIFY_(STATUS)

     call MAPL_AddExportSpec(GC,                    &
        SHORT_NAME         = 'MOV50M',                    &
        LONG_NAME          = 'meridional 50m wind from MO sfc', &
        UNITS              = 'm s-1',                     &
        DIMS               = MAPL_DimsTileOnly           ,&
        VLOCATION          = MAPL_VLocationNone,          &
                                               RC=STATUS  )
     VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'surface_roughness_for_heat',&
    UNITS              = 'm'                         ,&
    SHORT_NAME         = 'Z0H'                       ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    LONG_NAME          = 'zero_plane_displacement_height',&
    UNITS              = 'm'                         ,&
    SHORT_NAME         = 'D0'                        ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    SHORT_NAME         = 'GUST',                      &
    LONG_NAME          = 'gustiness',                 &
    UNITS              = 'm s-1',                     &
    DIMS               = MAPL_DimsTileOnly,           &
    VLOCATION          = MAPL_VLocationNone,          &
                                           RC=STATUS  )
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    SHORT_NAME         = 'VENT',                      &
    LONG_NAME          = 'surface_ventilation_velocity',&
    UNITS              = 'm s-1',                     &
    DIMS               = MAPL_DimsTileOnly,           &
    VLOCATION          = MAPL_VLocationNone,          &
                                           RC=STATUS  )
  VERIFY_(STATUS)


     call MAPL_AddExportSpec(GC,                             &
        SHORT_NAME         = 'ACCUM',                             &
        LONG_NAME          = 'net_ice_accumulation_rate',         &
        UNITS              = 'kg m-2 s-1',                        &
        DIMS               = MAPL_DimsTileOnly,                   &
        VLOCATION          = MAPL_VLocationNone,                  &
                                                       RC=STATUS  )
     VERIFY_(STATUS)


  call MAPL_AddExportSpec(GC,                    &
    SHORT_NAME         = 'EVLAND',                    &
    LONG_NAME          = 'Evaporation_land',          &
    UNITS              = 'kg m-2 s-1',                &
    DIMS               = MAPL_DimsTileOnly,           &
    VLOCATION          = MAPL_VLocationNone,          &
                                           RC=STATUS  )
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    SHORT_NAME         = 'PRLAND',                    &
    LONG_NAME          = 'Total_precipitation_land',  &
    UNITS              = 'kg m-2 s-1',                &
    DIMS               = MAPL_DimsTileOnly,           &
    VLOCATION          = MAPL_VLocationNone,          &
                                           RC=STATUS  )
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    SHORT_NAME         = 'SNOLAND',                   &
    LONG_NAME          = 'snowfall_land',             &
    UNITS              = 'kg m-2 s-1',                &
    DIMS               = MAPL_DimsTileOnly,           &
    VLOCATION          = MAPL_VLocationNone,          &
                                           RC=STATUS  )
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    SHORT_NAME         = 'DRPARLAND',                 &
    LONG_NAME          = 'surface_downwelling_par_beam_flux', &
    UNITS              = 'W m-2',                     &
    DIMS               = MAPL_DimsTileOnly,           &
    VLOCATION          = MAPL_VLocationNone,          &
                                           RC=STATUS  )
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    SHORT_NAME         = 'DFPARLAND',                 &
    LONG_NAME          = 'surface_downwelling_par_diffuse_flux', &
    UNITS              = 'W m-2',                     &
    DIMS               = MAPL_DimsTileOnly,           &
    VLOCATION          = MAPL_VLocationNone,          &
                                           RC=STATUS  )
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    SHORT_NAME         = 'LHSNOW',                    &
    LONG_NAME          = 'Latent_heat_flux_snow',     &
    UNITS              = 'W m-2',                     &
    DIMS               = MAPL_DimsTileOnly,           &
    VLOCATION          = MAPL_VLocationNone,          &
                                           RC=STATUS  )
  VERIFY_(STATUS)


  call MAPL_AddExportSpec(GC,                    &
    SHORT_NAME         = 'SWNETSNOW',                    &
    LONG_NAME          = 'Net_shortwave_snow',        &
    UNITS              = 'W m-2',                     &
    DIMS               = MAPL_DimsTileOnly,           &
    VLOCATION          = MAPL_VLocationNone,          &
                                           RC=STATUS  )
  VERIFY_(STATUS)


  call MAPL_AddExportSpec(GC,                    &
    SHORT_NAME         = 'LWUPSNOW',                    &
    LONG_NAME          = 'Net_longwave_snow',         &
    UNITS              = 'W m-2',                     &
    DIMS               = MAPL_DimsTileOnly,           &
    VLOCATION          = MAPL_VLocationNone,          &
                                           RC=STATUS  )
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    SHORT_NAME         = 'LWDNSNOW',                    &
    LONG_NAME          = 'Net_longwave_snow',         &
    UNITS              = 'W m-2',                     &
    DIMS               = MAPL_DimsTileOnly,           &
    VLOCATION          = MAPL_VLocationNone,          &
                                           RC=STATUS  )
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    SHORT_NAME         = 'TCSORIG',                   &
    LONG_NAME          = 'Input_tc_for_snow',         &
    UNITS              = 'K',                         &
    DIMS               = MAPL_DimsTileOnly,           &
    VLOCATION          = MAPL_VLocationNone,          &
                                           RC=STATUS  )
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    SHORT_NAME         = 'TPSN1IN',                   &
    LONG_NAME          = 'Input_temp_of_top_snow_lev',&
    UNITS              = 'K',                         &
    DIMS               = MAPL_DimsTileOnly,           &
    VLOCATION          = MAPL_VLocationNone,          &
                                           RC=STATUS  )
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    SHORT_NAME         = 'TPSN1OUT',                  &
    LONG_NAME          = 'Output_temp_of_top_snow_lev',&
    UNITS              = 'K',                         &
    DIMS               = MAPL_DimsTileOnly,           &
    VLOCATION          = MAPL_VLocationNone,          &
                                           RC=STATUS  )
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    SHORT_NAME         = 'GHSNOW',                    &
    LONG_NAME          = 'Ground_heating_snow',       &
    UNITS              = 'W m-2',                     &
    DIMS               = MAPL_DimsTileOnly,           &
    VLOCATION          = MAPL_VLocationNone,          &
                                           RC=STATUS  )
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    SHORT_NAME         = 'LHLAND',                    &
    LONG_NAME          = 'Latent_heat_flux_land',     &
    UNITS              = 'W m-2',                     &
    DIMS               = MAPL_DimsTileOnly,           &
    VLOCATION          = MAPL_VLocationNone,          &
                                           RC=STATUS  )
  VERIFY_(STATUS)


  call MAPL_AddExportSpec(GC,                    &
    SHORT_NAME         = 'SHLAND',                    &
    LONG_NAME          = 'Sensible_heat_flux_land',   &
    UNITS              = 'W m-2',                     &
    DIMS               = MAPL_DimsTileOnly,           &
    VLOCATION          = MAPL_VLocationNone,          &
                                           RC=STATUS  )
  VERIFY_(STATUS)


  call MAPL_AddExportSpec(GC,                    &
    SHORT_NAME         = 'SWLAND',                    &
    LONG_NAME          = 'Net_shortwave_land',        &
    UNITS              = 'W m-2',                     &
    DIMS               = MAPL_DimsTileOnly,           &
    VLOCATION          = MAPL_VLocationNone,          &
                                           RC=STATUS  )
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    SHORT_NAME         = 'SWDOWNLAND',                &
    LONG_NAME          = 'Incident_shortwave_land',   &
    UNITS              = 'W m-2',                     &
    DIMS               = MAPL_DimsTileOnly,           &
    VLOCATION          = MAPL_VLocationNone,          &
                                           RC=STATUS  )
  VERIFY_(STATUS)


  call MAPL_AddExportSpec(GC,                    &
    SHORT_NAME         = 'LWLAND',                    &
    LONG_NAME          = 'Net_longwave_land',         &
    UNITS              = 'W m-2',                     &
    DIMS               = MAPL_DimsTileOnly,           &
    VLOCATION          = MAPL_VLocationNone,          &
                                           RC=STATUS  )
  VERIFY_(STATUS)


  call MAPL_AddExportSpec(GC,                    &
    SHORT_NAME         = 'GHLAND',                    &
    LONG_NAME          = 'Ground_heating_land',       &
    UNITS              = 'W m-2',                     &
    DIMS               = MAPL_DimsTileOnly,           &
    VLOCATION          = MAPL_VLocationNone,          &
                                           RC=STATUS  )
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    SHORT_NAME         = 'GHTSKIN',                   &
    LONG_NAME          = 'Ground_heating_skin_temp',  &
    UNITS              = 'W m-2',                     &
    DIMS               = MAPL_DimsTileOnly,           &
    VLOCATION          = MAPL_VLocationNone,          &
                                           RC=STATUS  )
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    SHORT_NAME         = 'SMLAND',                    &
    LONG_NAME          = 'Snowmelt_flux_land',        &
    UNITS              = 'kg m-2 s-1',                &
    DIMS               = MAPL_DimsTileOnly,           &
    VLOCATION          = MAPL_VLocationNone,          &
                                           RC=STATUS  )
  VERIFY_(STATUS)


  call MAPL_AddExportSpec(GC,                    &
    SHORT_NAME         = 'TWLAND',                    &
    LONG_NAME          = 'Avail_water_storage_land',  &
    UNITS              = 'kg m-2',                    &
    DIMS               = MAPL_DimsTileOnly,           &
    VLOCATION          = MAPL_VLocationNone,          &
                                           RC=STATUS  )
  VERIFY_(STATUS)


  call MAPL_AddExportSpec(GC,                    &
    SHORT_NAME         = 'TELAND',                    &
    LONG_NAME          = 'Total_energy_storage_land', &
    UNITS              = 'J m-2',                     &
    DIMS               = MAPL_DimsTileOnly,           &
    VLOCATION          = MAPL_VLocationNone,          &
                                           RC=STATUS  )
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    SHORT_NAME         = 'TSLAND',                    &
    LONG_NAME          = 'Total_snow_storage_land',   &
    UNITS              = 'kg m-2',                    &
    DIMS               = MAPL_DimsTileOnly,           &
    VLOCATION          = MAPL_VLocationNone,          &
                                           RC=STATUS  )
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    SHORT_NAME         = 'DWLAND',                    &
    LONG_NAME          = 'rate_of_change_of_total_land_water',&
    UNITS              = 'kg m-2 s-1',                &
    DIMS               = MAPL_DimsTileOnly,           &
    VLOCATION          = MAPL_VLocationNone,          &
                                           RC=STATUS  )
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    SHORT_NAME         = 'DHLAND',                    &
    LONG_NAME          = 'rate_of_change_of_total_land_energy',&
    UNITS              = 'W m-2',                     &
    DIMS               = MAPL_DimsTileOnly,           &
    VLOCATION          = MAPL_VLocationNone,          &
                                           RC=STATUS  )
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    SHORT_NAME         = 'SPLAND',                    &
    LONG_NAME          = 'rate_of_spurious_land_energy_source',&
    UNITS              = 'W m-2',                     &
    DIMS               = MAPL_DimsTileOnly,           &
    VLOCATION          = MAPL_VLocationNone,          &
                                           RC=STATUS  )
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    SHORT_NAME         = 'SPWATR',                    &
    LONG_NAME          = 'rate_of_spurious_land_water_source',&
    UNITS              = 'kg m-2 s-1',                &
    DIMS               = MAPL_DimsTileOnly,           &
    VLOCATION          = MAPL_VLocationNone,          &
                                           RC=STATUS  )
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC,                    &
    SHORT_NAME         = 'SPSNOW',                    &
    LONG_NAME          = 'rate_of_spurious_snow_energy',&
    UNITS              = 'W m-2',                     &
    DIMS               = MAPL_DimsTileOnly,           &
    VLOCATION          = MAPL_VLocationNone,          &
                                           RC=STATUS  )
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC                         ,&
    LONG_NAME          = 'vegetation_type'           ,&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'ITY'                       ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC                         ,&
    LONG_NAME          = 'CN_exposed_leaf-area_index',&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'CNLAI'                     ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC                         ,&
    LONG_NAME          = 'CN_total_leaf-area_index'  ,&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'CNTLAI'                    ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC                         ,&
    LONG_NAME          = 'CN_exposed_stem-area_index',&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'CNSAI'                     ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC                         ,&
    LONG_NAME          = 'CN_total_carbon'           ,&
    UNITS              = 'kg m-2'                    ,&
    SHORT_NAME         = 'CNTOTC'                    ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC                         ,&
    LONG_NAME          = 'CN_total_vegetation_carbon',&
    UNITS              = 'kg m-2'                    ,&
    SHORT_NAME         = 'CNVEGC'                    ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC                         ,&
    LONG_NAME          = 'CN_total_root_carbon'      ,&
    UNITS              = 'kg m-2'                    ,&
    SHORT_NAME         = 'CNROOT'                    ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC                         ,&
    LONG_NAME          = 'CN_net_primary_production' ,&
    UNITS              = 'kg m-2 s-1'                ,&
    SHORT_NAME         = 'CNNPP'                     ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC                         ,&
    LONG_NAME          = 'CN_gross_primary_production',&
    UNITS              = 'kg m-2 s-1'                ,&
    SHORT_NAME         = 'CNGPP'                     ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC                         ,&
    LONG_NAME          = 'CN_total_soil_respiration' ,&
    UNITS              = 'kg m-2 s-1'                ,&
    SHORT_NAME         = 'CNSR'                      ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC                         ,&
    LONG_NAME          = 'CN_net_ecosystem_exchange' ,&
    UNITS              = 'kg m-2 s-1'                ,&
    SHORT_NAME         = 'CNNEE'                     ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC                         ,&
    LONG_NAME          = 'abstract_C_pool_to_meet_excess_MR_demand' ,&
    UNITS              = 'kg m-2'                    ,&
    SHORT_NAME         = 'CNXSMR'                    ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC                         ,&
    LONG_NAME          = 'CN_added_to_maintain_positive_C' ,&
    UNITS              = 'kg m-2 s-1'                ,&
    SHORT_NAME         = 'CNADD'                     ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC                         ,&
    LONG_NAME          = 'CN_carbon_loss_to_fire'    ,&
    UNITS              = 'kg m-2 s-1'                ,&
    SHORT_NAME         = 'CNLOSS'                    ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC                         ,&
    LONG_NAME          = 'CN_fractional_area_burn_rate' ,&
    UNITS              = 's-1'                       ,&
    SHORT_NAME         = 'CNBURN'                    ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC                         ,&
    LONG_NAME          = 'absorbed_PAR'              ,&
    UNITS              = 'W m-2'                     ,&
    SHORT_NAME         = 'PARABS'                    ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC                         ,&
    LONG_NAME          = 'incident_PAR'              ,&
    UNITS              = 'W m-2'                     ,&
    SHORT_NAME         = 'PARINC'                    ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC                         ,&
    LONG_NAME          = 'saturated_stomatal_conductance' ,&
    UNITS              = 'm s-1'                     ,&
    SHORT_NAME         = 'SCSAT'                     ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC                         ,&
    LONG_NAME          = 'unstressed_stomatal_conductance' ,&
    UNITS              = 'm s-1'                     ,&
    SHORT_NAME         = 'SCUNS'                     ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC                         ,&
    LONG_NAME          = 'transpiration coefficient' ,&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'BTRANT'                    ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC                         ,&
    LONG_NAME          = 'solar induced fluorescence',&
    UNITS              = 'umol m-2 sm s-1'           ,&
    SHORT_NAME         = 'SIF'                       ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC                         ,&
    LONG_NAME          = 'fire season length'        ,&
    UNITS              = 'days'                      ,&
    SHORT_NAME         = 'CNFSEL'                    ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC                         ,&
    LONG_NAME          = 'CO2 Surface Concentration used'  ,&
    UNITS              = '1e-6'                      ,&
    SHORT_NAME         = 'CNCO2'                     ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

! Add ESAI (NTILES,NV,NZ)

  call MAPL_AddExportSpec(GC                         ,&
    LONG_NAME          = 'Stem Area Index V1 Z1'     ,&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'CNSAI11'                   ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)
  call MAPL_AddExportSpec(GC                         ,&
    LONG_NAME          = 'Stem Area Index V1 Z2'     ,&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'CNSAI12'                   ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)
  call MAPL_AddExportSpec(GC                         ,&
    LONG_NAME          = 'Stem Area Index V1 Z3'     ,&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'CNSAI13'                   ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)
  
  call MAPL_AddExportSpec(GC                         ,&
    LONG_NAME          = 'Stem Area Index V2 Z1'     ,&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'CNSAI21'                   ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)
  call MAPL_AddExportSpec(GC                         ,&
    LONG_NAME          = 'Stem Area Index V2 Z2'     ,&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'CNSAI22'                   ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)
  call MAPL_AddExportSpec(GC                         ,&
    LONG_NAME          = 'Stem Area Index V2 Z3'     ,&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'CNSAI23'                   ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC                         ,&
    LONG_NAME          = 'Stem Area Index V3 Z1'     ,&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'CNSAI31'                   ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)
  call MAPL_AddExportSpec(GC                         ,&
    LONG_NAME          = 'Stem Area Index V3 Z2'     ,&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'CNSAI32'                   ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)
  call MAPL_AddExportSpec(GC                         ,&
    LONG_NAME          = 'Stem Area Index V3 Z3'     ,&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'CNSAI33'                   ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC                         ,&
    LONG_NAME          = 'Stem Area Index V4 Z1'     ,&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'CNSAI41'                   ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)
  call MAPL_AddExportSpec(GC                         ,&
    LONG_NAME          = 'Stem Area Index V4 Z2'     ,&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'CNSAI42'                   ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)
  call MAPL_AddExportSpec(GC                         ,&
    LONG_NAME          = 'Stem Area Index V4 Z3'     ,&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'CNSAI43'                   ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

! Add ELAI (NTILES,NV,NZ)

  call MAPL_AddExportSpec(GC                         ,&
    LONG_NAME          = 'Leaf Area Index V1 Z1'     ,&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'CNLAI11'                   ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)
  call MAPL_AddExportSpec(GC                         ,&
    LONG_NAME          = 'Leaf Area Index V1 Z2'     ,&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'CNLAI12'                   ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)
  call MAPL_AddExportSpec(GC                         ,&
    LONG_NAME          = 'Leaf Area Index V1 Z3'     ,&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'CNLAI13'                   ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)
  
  call MAPL_AddExportSpec(GC                         ,&
    LONG_NAME          = 'Leaf Area Index V2 Z1'     ,&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'CNLAI21'                   ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)
  call MAPL_AddExportSpec(GC                         ,&
    LONG_NAME          = 'Leaf Area Index V2 Z2'     ,&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'CNLAI22'                   ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)
  call MAPL_AddExportSpec(GC                         ,&
    LONG_NAME          = 'Leaf Area Index V2 Z3'     ,&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'CNLAI23'                   ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC                         ,&
    LONG_NAME          = 'Leaf Area Index V3 Z1'     ,&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'CNLAI31'                   ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)
  call MAPL_AddExportSpec(GC                         ,&
    LONG_NAME          = 'Leaf Area Index V3 Z2'     ,&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'CNLAI32'                   ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)
  call MAPL_AddExportSpec(GC                         ,&
    LONG_NAME          = 'Leaf Area Index V3 Z3'     ,&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'CNLAI33'                   ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC                         ,&
    LONG_NAME          = 'Leaf Area Index V4 Z1'     ,&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'CNLAI41'                   ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)
  call MAPL_AddExportSpec(GC                         ,&
    LONG_NAME          = 'Leaf Area Index V4 Z2'     ,&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'CNLAI42'                   ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)
  call MAPL_AddExportSpec(GC                         ,&
    LONG_NAME          = 'Leaf Area Index V4 Z3'     ,&
    UNITS              = '1'                         ,&
    SHORT_NAME         = 'CNLAI43'                   ,&
    DIMS               = MAPL_DimsTileOnly           ,&
    VLOCATION          = MAPL_VLocationNone          ,&
                                           RC=STATUS  ) 
  VERIFY_(STATUS)

  call MAPL_AddExportSpec(GC                  ,&
       LONG_NAME          = 'flushed_out_dust_mass_flux_from_the_bottom_layer_bin_1',&
       UNITS              = 'kg m-2 s-1'                ,&
       SHORT_NAME         = 'RMELTDU001'                ,&
       DIMS               = MAPL_DimsTileOnly           ,&
       VLOCATION          = MAPL_VLocationNone          ,&
       RC=STATUS  ) 
  VERIFY_(STATUS)
  
  call MAPL_AddExportSpec(GC                  ,&
       LONG_NAME          = 'flushed_out_dust_mass_flux_from_the_bottom_layer_bin_2',&
       UNITS              = 'kg m-2 s-1'                ,&
       SHORT_NAME         = 'RMELTDU002'                ,&
       DIMS               = MAPL_DimsTileOnly           ,&
       VLOCATION          = MAPL_VLocationNone          ,&
       RC=STATUS  ) 
  VERIFY_(STATUS)
  
  call MAPL_AddExportSpec(GC                  ,&
       LONG_NAME          = 'flushed_out_dust_mass_flux_from_the_bottom_layer_bin_3',&
       UNITS              = 'kg m-2 s-1'                ,&
       SHORT_NAME         = 'RMELTDU003'                ,&
       DIMS               = MAPL_DimsTileOnly           ,&
       VLOCATION          = MAPL_VLocationNone          ,&
       RC=STATUS  ) 
  VERIFY_(STATUS)
  
  call MAPL_AddExportSpec(GC                  ,&
       LONG_NAME          = 'flushed_out_dust_mass_flux_from_the_bottom_layer_bin_4',&
       UNITS              = 'kg m-2 s-1'                ,&
       SHORT_NAME         = 'RMELTDU004'                ,&
       DIMS               = MAPL_DimsTileOnly           ,&
       VLOCATION          = MAPL_VLocationNone          ,&
       RC=STATUS  ) 
  VERIFY_(STATUS)
  
  call MAPL_AddExportSpec(GC                  ,&
       LONG_NAME          = 'flushed_out_dust_mass_flux_from_the_bottom_layer_bin_5',&
       UNITS              = 'kg m-2 s-1'                ,&
       SHORT_NAME         = 'RMELTDU005'                ,&
       DIMS               = MAPL_DimsTileOnly           ,&
       VLOCATION          = MAPL_VLocationNone          ,&
       RC=STATUS  ) 
  VERIFY_(STATUS)
  
  call MAPL_AddExportSpec(GC                  ,&
       LONG_NAME          = 'flushed_out_black_carbon_mass_flux_from_the_bottom_layer_bin_1',&
       UNITS              = 'kg m-2 s-1'                ,&
       SHORT_NAME         = 'RMELTBC001'                ,&
       DIMS               = MAPL_DimsTileOnly           ,&
       VLOCATION          = MAPL_VLocationNone          ,&
       RC=STATUS  ) 
  VERIFY_(STATUS)
  
  call MAPL_AddExportSpec(GC                  ,&
       LONG_NAME          = 'flushed_out_black_carbon_mass_flux_from_the_bottom_layer_bin_2',&
       UNITS              = 'kg m-2 s-1'                ,&
       SHORT_NAME         = 'RMELTBC002'                ,&
       DIMS               = MAPL_DimsTileOnly           ,&
       VLOCATION          = MAPL_VLocationNone          ,&
       RC=STATUS  ) 
  VERIFY_(STATUS)
  
  call MAPL_AddExportSpec(GC                  ,&
       LONG_NAME          = 'flushed_out_organic_carbon_mass_flux_from_the_bottom_layer_bin_1',&
       UNITS              = 'kg m-2 s-1'                ,&
       SHORT_NAME         = 'RMELTOC001'                ,&
       DIMS               = MAPL_DimsTileOnly           ,&
       VLOCATION          = MAPL_VLocationNone          ,&
       RC=STATUS  ) 
  VERIFY_(STATUS)
  
  call MAPL_AddExportSpec(GC                  ,&
       LONG_NAME          = 'flushed_out_organic_carbon_mass_flux_from_the_bottom_layer_bin_2',&
       UNITS              = 'kg m-2 s-1'                ,&
       SHORT_NAME         = 'RMELTOC002'                ,&
       DIMS               = MAPL_DimsTileOnly           ,&
       VLOCATION          = MAPL_VLocationNone          ,&
                                               RC=STATUS  ) 
  VERIFY_(STATUS)

!EOS

    call MAPL_TimerAdd(GC,    name="RUN1"  ,RC=STATUS)
    VERIFY_(STATUS)
    if (OFFLINE_MODE /=0) then
       call MAPL_TimerAdd(GC,    name="-RUN0"  ,RC=STATUS)
       VERIFY_(status)
    end if
    call MAPL_TimerAdd(GC,    name="-SURF" ,RC=STATUS)
    VERIFY_(STATUS)
    call MAPL_TimerAdd(GC,    name="RUN2"  ,RC=STATUS)
    VERIFY_(STATUS)
    call MAPL_TimerAdd(GC,    name="-CATCH" ,RC=STATUS)
    VERIFY_(STATUS)
    call MAPL_TimerAdd(GC,    name="-ALBEDO" ,RC=STATUS)
    VERIFY_(STATUS)

! Set generic init and final method
! ---------------------------------

    call MAPL_GenericSetServices ( GC, RC=STATUS )
    VERIFY_(STATUS)

    RETURN_(ESMF_SUCCESS)

end subroutine SetServices

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!BOP
! !IROUTINE: RUN1 -- First Run stage for the catchment component
! !INTERFACE:

subroutine RUN1 ( GC, IMPORT, EXPORT, CLOCK, RC )

! !ARGUMENTS:

    type(ESMF_GridComp),intent(inout) :: GC     !Gridded component
    type(ESMF_State),   intent(inout) :: IMPORT !Import state
    type(ESMF_State),   intent(inout) :: EXPORT !Export state
    type(ESMF_Clock),   intent(inout) :: CLOCK  !The clock
    integer,optional,   intent(out  ) :: RC     !Error code:

! !DESCRIPTION: Does the cds computation and roughness length
!EOP
! ErrLog Variables

    character(len=ESMF_MAXSTR) :: IAm
    integer :: STATUS

    character(len=ESMF_MAXSTR) :: COMP_NAME

! Locals

    type(MAPL_MetaComp),pointer :: MAPL
    type(ESMF_State)                :: INTERNAL
    type(ESMF_Alarm)                :: ALARM
    type(ESMF_Config)               :: CF
    type(ESMF_VM)                   :: VM

! -----------------------------------------------------
! IMPORT Pointers
! ----------------------------------------------------  -

    real, dimension(:),     pointer :: PS
    real, dimension(:),     pointer :: TA
    real, dimension(:),     pointer :: QA
    real, dimension(:),     pointer :: UU
    real, pointer, dimension(:)    :: UWINDLMTILE
    real, pointer, dimension(:)    :: VWINDLMTILE
    real, dimension(:),     pointer :: DZ
    real, dimension(:),     pointer :: LAI
    real, dimension(:),     pointer :: Z2CH
    real, dimension(:),     pointer :: PCU
    real, dimension(:),     pointer :: ASCATZ0
    real, dimension(:),     pointer :: NDVI

! -----------------------------------------------------
! INTERNAL Pointers
! -----------------------------------------------------

    real, dimension(:,:), pointer :: ITY
    real, dimension(:,:), pointer :: FVG
    real, dimension(:,:), pointer :: TC
    real, dimension(:,:), pointer :: QC
    real, dimension(:,:), pointer :: CH
    real, dimension(:,:), pointer :: CM
    real, dimension(:,:), pointer :: CQ
    real, dimension(:,:), pointer :: FR
    real, dimension(:,:), pointer :: WW
    real, dimension(:,:), pointer :: cncol
    real, dimension(:,:), pointer :: cnpft
    real, dimension(:,:), pointer :: DCH
    real, dimension(:,:), pointer :: DCQ

! -----------------------------------------------------
! EXPORT Pointers
! -----------------------------------------------------

    real, dimension(:),   pointer :: TH
    real, dimension(:),   pointer :: QH
    real, dimension(:),   pointer :: CHT
    real, dimension(:),   pointer :: CMT
    real, dimension(:),   pointer :: CQT
    real, dimension(:),   pointer :: CNT
    real, dimension(:),   pointer :: RIT
    real, dimension(:),   pointer :: Z0
    real, dimension(:),   pointer :: Z0H
    real, dimension(:),   pointer :: D0
    real, dimension(:),   pointer :: GST
    real, dimension(:),   pointer :: VNT
   real, pointer, dimension(:  )  :: MOT2M
   real, pointer, dimension(:  )  :: MOQ2M
   real, pointer, dimension(:  )  :: MOU2M
   real, pointer, dimension(:  )  :: MOV2M
   real, pointer, dimension(:  )  :: MOT10M
   real, pointer, dimension(:  )  :: MOQ10M
   real, pointer, dimension(:  )  :: MOU10M
   real, pointer, dimension(:  )  :: MOV10M
   real, pointer, dimension(:  )  :: MOU50M
   real, pointer, dimension(:  )  :: MOV50M
    real, dimension(:),   pointer :: ITYO


! From old bucket version of CDS calculation
! ------------------------------------------

    integer             :: N
    integer             :: NT
    real,   allocatable :: UCN(:)
    real,   allocatable :: TVA(:)
    real,   allocatable :: TVS(:)
    real,   allocatable :: URA(:)
    real,   allocatable :: UUU(:)
    real,   allocatable :: ZVG(:)
    real,   allocatable :: DZE(:)
    real,   allocatable :: D0T(:)
    real,   allocatable :: CHX(:)
    real,   allocatable :: CQX(:)
    real,   allocatable :: CN(:)
    real,   allocatable :: RE(:)
    real,   allocatable :: ZT(:)
    real,   allocatable :: ZQ(:)
    integer,allocatable :: VEG1(:)
    integer,allocatable :: VEG2(:)
    real,   allocatable :: FVG1(:)
    real,   allocatable :: FVG2(:)
    real,   allocatable :: Z0T(:,:)
   real, allocatable              :: U50M (:)
   real, allocatable              :: V50M (:)
   real, allocatable              :: T10M (:)
   real, allocatable              :: Q10M (:)
   real, allocatable              :: U10M (:)
   real, allocatable              :: V10M (:)
   real, allocatable              :: T2M (:)
   real, allocatable              :: Q2M (:)
   real, allocatable              :: U2M (:)
   real, allocatable              :: V2M (:)
   real, allocatable              :: RHOH(:)
   real, allocatable              :: VKH(:)
   real, allocatable              :: VKM(:)
   real, allocatable              :: USTAR(:)
   real, allocatable              :: XX(:)
   real, allocatable              :: YY(:)
   real, allocatable              :: CU(:)
   real, allocatable              :: CT(:)
   real, allocatable              :: RIB(:)
   real, allocatable              :: ZETA(:)
   real, allocatable              :: WS(:)
   integer, allocatable           :: IWATER(:)
   real, allocatable              :: PSMB(:)
   real, allocatable              :: PSL(:)
   integer                        :: niter

   real                           :: SCALE4Z0

! gkw: for CN model
! -----------------
    integer, parameter :: nveg  = num_veg ! number of vegetation types
    integer, parameter :: nzone = num_zon ! number of stress zones

    integer, allocatable :: ityp(:,:,:)
    real,    allocatable :: fveg(:,:,:), elai(:,:,:), esai(:,:,:), wtzone(:,:), lai1(:), lai2(:), wght(:)

    integer :: nv, nz, ib
    real    :: bare
    logical, save :: first = .true.
    integer*8, save :: istep = 1 ! gkw: legacy variable from offline

  ! Offline mode

   type(OFFLINE_WRAP)             :: wrap
   integer                        :: OFFLINE_MODE, CHOOSEZ0

!=============================================================================
! Begin...
! ------------------------------------------------------------------------------


! ------------------------------------------------------------------------------
! Get the target component's name and set-up traceback handle.
! ------------------------------------------------------------------------------

    call ESMF_GridCompGet ( GC, name=COMP_NAME, RC=STATUS )
    VERIFY_(STATUS)
    Iam=trim(COMP_NAME)//"::RUN1"

    ! Get component's offline mode from its pvt internal state
    call ESMF_UserCompGetInternalState(gc, 'OfflineMode', wrap, status)
    VERIFY_(status)
    OFFLINE_MODE = wrap%ptr%CATCH_OFFLINE

    call ESMF_VMGetCurrent ( VM, RC=STATUS ) 
    ! if (MAPL_AM_I_Root(VM)) print *, trim(Iam)//'::OFFLINE mode: ', is_OFFLINE
    
! Get my internal MAPL_Generic state
! ----------------------------------

    call MAPL_GetObjectFromGC ( GC, MAPL, RC=STATUS )
    VERIFY_(STATUS)

! Start timers
! ------------

    call MAPL_TimerOn(MAPL,"TOTAL")
    call MAPL_TimerOn(MAPL,"RUN1")

! Get parameters from generic state
! ---------------------------------

    call MAPL_Get ( MAPL                          ,&
                                INTERNAL_ESMF_STATE=INTERNAL   ,&
                                                      RC=STATUS )
    VERIFY_(STATUS)

    call MAPL_GetResource ( MAPL, CHOOSEZ0, Label="CHOOSEZ0:", DEFAULT=3, RC=STATUS)
    VERIFY_(STATUS)

    call MAPL_GetResource ( MAPL, SCALE4Z0, Label="SCALE4Z0:", DEFAULT=0.5, RC=STATUS)
    VERIFY_(STATUS)

! Pointers to inputs
!-------------------

   call MAPL_GetPointer(IMPORT,UU     , 'UU'     ,    RC=STATUS)
   VERIFY_(STATUS)
   call MAPL_GetPointer(IMPORT,UWINDLMTILE     , 'UWINDLMTILE'     ,    RC=STATUS)
   VERIFY_(STATUS)
   call MAPL_GetPointer(IMPORT,VWINDLMTILE     , 'VWINDLMTILE'     ,    RC=STATUS)
   VERIFY_(STATUS)
   call MAPL_GetPointer(IMPORT,DZ     , 'DZ'     ,    RC=STATUS)
   VERIFY_(STATUS)
   call MAPL_GetPointer(IMPORT,TA     , 'TA'     ,    RC=STATUS)
   VERIFY_(STATUS)
   call MAPL_GetPointer(IMPORT,QA     , 'QA'     ,    RC=STATUS)
   VERIFY_(STATUS)
   call MAPL_GetPointer(IMPORT,PS     , 'PS'     ,    RC=STATUS)
   VERIFY_(STATUS)
   call MAPL_GetPointer(IMPORT,LAI    , 'LAI'    ,    RC=STATUS)
   VERIFY_(STATUS)
   call MAPL_GetPointer(IMPORT,Z2CH   , 'Z2CH'   ,    RC=STATUS)
   VERIFY_(STATUS)
   call MAPL_GetPointer(IMPORT,PCU    , 'PCU'    ,    RC=STATUS)
   VERIFY_(STATUS)
   call MAPL_GetPointer(IMPORT,ASCATZ0, 'ASCATZ0',    RC=STATUS)
   VERIFY_(STATUS)
   call MAPL_GetPointer(IMPORT,NDVI   , 'NDVI'   ,    RC=STATUS)
   VERIFY_(STATUS)

! Pointers to internals
!----------------------
 
   call MAPL_GetPointer(INTERNAL,ITY  , 'ITY'    ,    RC=STATUS)
   VERIFY_(STATUS)
   call MAPL_GetPointer(INTERNAL,FVG  , 'FVG'    ,    RC=STATUS)
   VERIFY_(STATUS)
   call MAPL_GetPointer(INTERNAL,TC   , 'TC'     ,    RC=STATUS)
   VERIFY_(STATUS)
   call MAPL_GetPointer(INTERNAL,QC   , 'QC'     ,    RC=STATUS)
   VERIFY_(STATUS)
   call MAPL_GetPointer(INTERNAL,FR   , 'FR'     ,    RC=STATUS)
   VERIFY_(STATUS)
   call MAPL_GetPointer(INTERNAL,CH   , 'CH'     ,    RC=STATUS)
   VERIFY_(STATUS)
   call MAPL_GetPointer(INTERNAL,CM   , 'CM'     ,    RC=STATUS)
   VERIFY_(STATUS)
   call MAPL_GetPointer(INTERNAL,CQ   , 'CQ'     ,    RC=STATUS)
   VERIFY_(STATUS)
   call MAPL_GetPointer(INTERNAL,WW   , 'WW'     ,    RC=STATUS)
   VERIFY_(STATUS)
   call MAPL_GetPointer(INTERNAL,CNCOL  ,'CNCOL'  ,   RC=STATUS)
   VERIFY_(STATUS)
   call MAPL_GetPointer(INTERNAL,CNPFT  ,'CNPFT'  ,   RC=STATUS)
   VERIFY_(STATUS)
   call MAPL_GetPointer(INTERNAL,DCH  , 'DCH'     ,    RC=STATUS)
   VERIFY_(STATUS)
   call MAPL_GetPointer(INTERNAL,DCQ  , 'DCQ'     ,    RC=STATUS)
   VERIFY_(STATUS)

! Pointers to outputs
!--------------------

   call MAPL_GetPointer(EXPORT,QH    , 'QH'      ,    RC=STATUS)
   VERIFY_(STATUS)
   call MAPL_GetPointer(EXPORT,TH    , 'TH'      ,    RC=STATUS)
   VERIFY_(STATUS)
   call MAPL_GetPointer(EXPORT,CHT   , 'CHT'     ,    RC=STATUS)
   VERIFY_(STATUS)
   call MAPL_GetPointer(EXPORT,CMT   , 'CMT'     ,    RC=STATUS)
   VERIFY_(STATUS)
   call MAPL_GetPointer(EXPORT,CQT   , 'CQT'     ,    RC=STATUS)
   VERIFY_(STATUS)
   call MAPL_GetPointer(EXPORT,CNT   , 'CNT'     ,    RC=STATUS)
   VERIFY_(STATUS)
   call MAPL_GetPointer(EXPORT,RIT   , 'RIT'     ,    RC=STATUS)
   VERIFY_(STATUS)
   call MAPL_GetPointer(EXPORT,Z0    , 'Z0'      ,    RC=STATUS)
   VERIFY_(STATUS)
   call MAPL_GetPointer(EXPORT,Z0H   , 'Z0H'     ,    RC=STATUS)
   VERIFY_(STATUS)
   call MAPL_GetPointer(EXPORT,D0    , 'D0'      ,    RC=STATUS)
   VERIFY_(STATUS)
   call MAPL_GetPointer(EXPORT,GST   , 'GUST'    ,    RC=STATUS)
   VERIFY_(STATUS)
   call MAPL_GetPointer(EXPORT,VNT   , 'VENT'    ,    RC=STATUS)
   VERIFY_(STATUS)
   call MAPL_GetPointer(EXPORT,MOT2M, 'MOT2M'   ,    RC=STATUS)
   VERIFY_(STATUS)
   call MAPL_GetPointer(EXPORT,MOQ2M, 'MOQ2M'   ,    RC=STATUS)
   VERIFY_(STATUS)
   call MAPL_GetPointer(EXPORT,MOU2M, 'MOU2M'  ,    RC=STATUS)
   VERIFY_(STATUS)
   call MAPL_GetPointer(EXPORT,MOV2M, 'MOV2M'  ,    RC=STATUS)
   VERIFY_(STATUS)
   call MAPL_GetPointer(EXPORT,MOT10M, 'MOT10M'   ,    RC=STATUS)
   VERIFY_(STATUS)
   call MAPL_GetPointer(EXPORT,MOQ10M, 'MOQ10M'   ,    RC=STATUS)
   VERIFY_(STATUS)
   call MAPL_GetPointer(EXPORT,MOU10M, 'MOU10M'  ,    RC=STATUS)
   VERIFY_(STATUS)
   call MAPL_GetPointer(EXPORT,MOV10M, 'MOV10M'  ,    RC=STATUS)
   VERIFY_(STATUS)
   call MAPL_GetPointer(EXPORT,MOU50M, 'MOU50M'  ,    RC=STATUS)
   VERIFY_(STATUS)
   call MAPL_GetPointer(EXPORT,MOV50M, 'MOV50M'  ,    RC=STATUS)
   VERIFY_(STATUS)
   call MAPL_GetPointer(EXPORT,ITYO  , 'ITY'     ,    RC=STATUS)
   VERIFY_(STATUS)

   NT = size(TA)

   allocate(TVA(NT),STAT=STATUS)
   VERIFY_(STATUS)
   allocate(TVS(NT),STAT=STATUS)
   VERIFY_(STATUS)
   allocate(URA(NT),STAT=STATUS)
   VERIFY_(STATUS)
   allocate(UUU(NT),STAT=STATUS)
   VERIFY_(STATUS)
   allocate(VEG1(NT),STAT=STATUS)
   VERIFY_(STATUS)
   allocate(VEG2(NT),STAT=STATUS)
   VERIFY_(STATUS)
   allocate(FVG1(NT),STAT=STATUS)
   VERIFY_(STATUS)
   allocate(FVG2(NT),STAT=STATUS)
   VERIFY_(STATUS)
   allocate(DZE(NT),STAT=STATUS)
   VERIFY_(STATUS)
   allocate(ZVG(NT),STAT=STATUS)
   VERIFY_(STATUS)
   allocate(Z0T(NT,NUM_SUBTILES),STAT=STATUS)
   VERIFY_(STATUS)
   allocate(D0T(NT),STAT=STATUS)
   VERIFY_(STATUS)
   allocate(CHX(NT),STAT=STATUS)
   VERIFY_(STATUS)
   allocate(CQX(NT),STAT=STATUS)
   VERIFY_(STATUS)
   allocate(RE (NT),STAT=STATUS)
   VERIFY_(STATUS)
   allocate(CN (NT),STAT=STATUS)
   VERIFY_(STATUS)
   allocate(ZT (NT),STAT=STATUS)
   VERIFY_(STATUS)
   allocate(ZQ (NT),STAT=STATUS)
   VERIFY_(STATUS)
   allocate(UCN(NT),STAT=STATUS)
   VERIFY_(STATUS)
   allocate(T2M (NT)  ,STAT=STATUS)
   VERIFY_(STATUS)
   allocate(Q2M (NT)  ,STAT=STATUS)
   VERIFY_(STATUS)
   allocate(U2M (NT)  ,STAT=STATUS)
   VERIFY_(STATUS)
   allocate(v2M (NT)  ,STAT=STATUS)
   VERIFY_(STATUS)
   allocate(T10M (NT)  ,STAT=STATUS)
   VERIFY_(STATUS)
   allocate(Q10M (NT)  ,STAT=STATUS)
   VERIFY_(STATUS)
   allocate(U10M (NT)  ,STAT=STATUS)
   VERIFY_(STATUS)
   allocate(v10M (NT)  ,STAT=STATUS)
   VERIFY_(STATUS)
   allocate(U50M (NT)  ,STAT=STATUS)
   VERIFY_(STATUS)
   allocate(v50M (NT)  ,STAT=STATUS)
   VERIFY_(STATUS)
   allocate(RHOH(NT) ,STAT=STATUS)
   VERIFY_(STATUS)
   allocate(PSMB(NT) ,STAT=STATUS)
   VERIFY_(STATUS)
   allocate(PSL(NT) ,STAT=STATUS)
   VERIFY_(STATUS)
   allocate(VKH(NT) ,STAT=STATUS)
   VERIFY_(STATUS)
   allocate(VKM(NT) ,STAT=STATUS)
   VERIFY_(STATUS)
   allocate(USTAR(NT) ,STAT=STATUS)
   VERIFY_(STATUS)
   allocate(XX(NT)   ,STAT=STATUS)
   VERIFY_(STATUS)
   allocate(YY(NT)   ,STAT=STATUS)
   VERIFY_(STATUS)
   allocate(CU(NT)   ,STAT=STATUS)
   VERIFY_(STATUS)
   allocate(CT(NT)   ,STAT=STATUS)
   VERIFY_(STATUS)
   allocate(RIB(NT)  ,STAT=STATUS)
   VERIFY_(STATUS)
   allocate(ZETA(NT) ,STAT=STATUS)
   VERIFY_(STATUS)
   allocate(WS(NT)   ,STAT=STATUS)
   VERIFY_(STATUS)
   allocate(IWATER(NT),STAT=STATUS)
   VERIFY_(STATUS)

   allocate(   ityp(nt,nveg,nzone) )
   allocate(   fveg(nt,nveg,nzone) )
   allocate( wtzone(nt,nzone) )
   allocate(   elai(nt,nveg,nzone) )
   allocate(   esai(nt,nveg,nzone) )

   allocate ( lai1(nt) )
   allocate ( lai2(nt) )
   allocate ( wght(nt) )

!  Vegetation types used to index into tables
!--------------------------------------------

   where(ITY(:,1) > 0.)
     VEG1 = map_cat(nint(ITY(:,1)))  ! gkw: primary   CN PFT type mapped to catchment type; ITY should be > 0 even if FVEG=0
    elsewhere
     VEG1 = map_cat(nint(ITY(:,2)))  ! gkw: primary   CN PFT type mapped to catchment type; ITY should be > 0 even if FVEG=0
   endwhere
   where(ITY(:,3) > 0.)
     VEG2 = map_cat(nint(ITY(:,3)))  ! gkw: secondary CN PFT type mapped to catchment type; ITY should be > 0 even if FVEG=0
    elsewhere
     VEG2 = map_cat(nint(ITY(:,4)))  ! gkw: secondary CN PFT type mapped to catchment type; ITY should be > 0 even if FVEG=0
   endwhere
   _ASSERT((count(VEG1>NTYPS.or.VEG1<1)==0),'needs informative message')
   _ASSERT((count(VEG2>NTYPS.or.VEG2<1)==0),'needs informative message')

   ! At this point, bare soil is not allowed in CatchCN. FVEG in BCs 
   ! files do not have bare soil either. However, at times, tiny bare 
   ! fractions appear due to truncation. We add that tiny fraction to the
   ! largest of the 4 fractions and ensure bare is zero. (Sarith 3/3/16)

   DO N = 1, NT
      BARE = 1.      

      DO NV = 1, NVEG
         BARE = BARE - FVG(N,NV)! subtract vegetated fractions 
      END DO

      if (BARE /= 0.) THEN
         IB = MAXLOC(FVG(N,:),1)
         FVG (N,IB) = FVG(N,IB) + BARE ! This also corrects cases sum gt 1.
      ENDIF
      
   END DO
   
   FVG1 = fvg(:,1) + fvg(:,2)  ! gkw: primary   vegetation fraction
   FVG2 = fvg(:,3) + fvg(:,4)  ! gkw: secondary vegetation fraction

! set CLM CN PFT & fraction, set carbon zone weights
! --------------------------------------------------
   do nz = 1,nzone
     ityp(:,:,nz) = nint(ity(:,:))
     fveg(:,:,nz) = fvg(:,:)
     wtzone(:,nz) = CN_zone_weight(nz)
   end do

! initialize CN model and transfer restart variables on startup
! -------------------------------------------------------------
   if(first) then
     if ((PRESCRIBE_DVG == 0) .OR.(PRESCRIBE_DVG == 3)) then 
        call CN_init(istep,nt,nveg,nzone,ityp,fveg,var_col,var_pft,cncol=cncol,cnpft=cnpft)  
        call get_CN_LAI(nt,nveg,nzone,ityp,fveg,elai,esai=esai)  
        if(PRESCRIBE_DVG == 3) call  read_prescribed_LAI (INTERNAL, CLOCK, GC, NT, elai,esai)
     else
        call CN_init(istep,nt,nveg,nzone,ityp,fveg,var_col,var_pft)  
     endif
     first = .false.
   endif

   ! For the OFFLINE case, first update some diagnostic vars
   if (OFFLINE_MODE /=0) then
      call MAPL_TimerOn(MAPL, "-RUN0")
      call RUN0(gc, import, export, clock, rc)
      call MAPL_TimerOff(MAPL, "-RUN0")
   end if

! obtain LAI from previous time step (from CN model)
! --------------------------------------------------

   IF(PRESCRIBE_DVG == 0) THEN
      call get_CN_LAI(nt,nveg,nzone,ityp,fveg,elai,esai=esai)
   ELSE
      call  read_prescribed_LAI (INTERNAL, CLOCK, GC, NT, elai,esai)
   ENDIF
      
   lai1 = 0.
   wght = 0.
   do nz = 1,nzone
     do nv = 1,2
       lai1(:) = lai1(:) + max(elai(:,nv,nz),0.)*fveg(:,nv,nz)*wtzone(:,nz)
       wght(:) = wght(:) +                       fveg(:,nv,nz)*wtzone(:,nz)
     end do
   end do
   lai1 = lai1 / max(wght,1.e-8) ! LAI for primary vegetation type

   lai2 = 0.
   wght = 0.
   do nz = 1,nzone
     do nv = 3,4
       lai2(:) = lai2(:) + max(elai(:,nv,nz),0.)*fveg(:,nv,nz)*wtzone(:,nz)
       wght(:) = wght(:) +                       fveg(:,nv,nz)*wtzone(:,nz)
     end do
   end do
   lai2 = lai2 / max(wght,1.e-8) ! LAI for secondary vegetation type

   lai = fvg1*lai1 + fvg2*lai2   ! gkw: this is a VEGDYN import

   deallocate ( ityp )
   deallocate ( fveg )
   deallocate ( elai )
   deallocate ( esai )
   deallocate ( wtzone )

!  Clear the output tile accumulators
!------------------------------------
   
   CHX = 0.0
   CQX = 0.0

   if(associated(TH )) TH  = 0.0
   if(associated(QH )) QH  = 0.0
   if(associated(CMT)) CMT = 0.0
   if(associated(CNT)) CNT = 0.0
   if(associated(RIT)) RIT = 0.0
   if(associated(Z0H)) Z0H = 0.0
   if(associated(GST)) GST = 0.0
   if(associated(VNT)) VNT = 0.0
   if(associated(MOU50M)) MOU50M = 0.0
   if(associated(MOV50M)) MOV50M = 0.0
   if(associated(MOT10M)) MOT10M = 0.0
   if(associated(MOQ10M)) MOQ10M = 0.0
   if(associated(MOU10M)) MOU10M = 0.0
   if(associated(MOV10M)) MOV10M = 0.0
   if(associated( MOT2M))  MOT2M = 0.0
   if(associated( MOQ2M))  MOQ2M = 0.0
   if(associated( MOU2M))  MOU2M = 0.0
   if(associated( MOV2M))  MOV2M = 0.0

   SUBTILES: do N=1,NUM_SUBTILES

!  Effective vegetation height. In catchment, LAI dependence 
!   includes the effect of partially vegetated areas,
!   as well as the phenology of the deciduous types. These
!   effects will be separated in future formulations.

   ZVG  = fvg1*(Z2CH - SCALE4Z0*(Z2CH - MIN_VEG_HEIGHT)*exp(-LAI1)) + &
          fvg2*(Z2CH - SCALE4Z0*(Z2CH - MIN_VEG_HEIGHT)*exp(-LAI2)) 


!  For now roughnesses and displacement heights
!   are the same for all subtiles.

   Z0T(:,N)  = Z0_BY_ZVEG*ZVG
   IF (USE_ASCATZ0 == 1) THEN
      WHERE (NDVI <= 0.2)
         Z0T(:,N)  = ASCATZ0
      END WHERE
   ENDIF
   D0T  = D0_BY_ZVEG*ZVG

   DZE  = max(DZ - D0T, 10.)

   if(associated(Z0 )) Z0  = Z0T(:,N)
   if(associated(D0 )) D0  = D0T

!  Compute the three surface exchange coefficients
!-------------------------------------------------

   call MAPL_TimerOn(MAPL,"-SURF")
   if(CHOOSEMOSFC.eq.0) then
   WW(:,N) = 0.
   CM(:,N) = 0.

    call louissurface(3,N,UU,WW,PS,TA,TC,QA,QC,PCU,LAI,Z0T,DZE,CM,CN,RIB,ZT,ZQ,CH,CQ,UUU,UCN,RE,DCH,DCQ)

   elseif (CHOOSEMOSFC.eq.1)then
  
    niter = 6   ! number of internal iterations in the helfand MO surface layer routine
    IWATER = 3
  
    PSMB = PS * 0.01            ! convert to MB
! Approximate pressure at top of surface layer: hydrostatic, eqn of state using avg temp and press
    PSL = PSMB * (1. - (DZE*MAPL_GRAV)/(MAPL_RGAS*(TA+TC(:,N)) ) ) /   &
               (1. + (DZE*MAPL_GRAV)/(MAPL_RGAS*(TA+TC(:,N)) ) )
  
    CALL helfsurface( UWINDLMTILE,VWINDLMTILE,TA,TC(:,N),QA,QC(:,N),PSL,PSMB,Z0T(:,N),lai,  &
                      IWATER,DZE,niter,nt,RHOH,VKH,VKM,USTAR,XX,YY,CU,CT,RIB,ZETA,WS,  &
                      t2m,q2m,u2m,v2m,t10m,q10m,u10m,v10m,u50m,v50m,CHOOSEZ0)
  
    CM(:,N)  = VKM
    CH(:,N)  = VKH
    CQ(:,N)  = VKH
  
    CN = (MAPL_KARMAN/ALOG(DZE/Z0T(:,N) + 1.0)) * (MAPL_KARMAN/ALOG(DZE/Z0T(:,N) + 1.0))
    ZT = Z0T(:,N)
    ZQ = Z0T(:,N)
    RE = 0.
    UUU = UU  
    UCN = 0.
  
!  Aggregate to tiles for MO only diagnostics
!--------------------------------------------
      if(associated(MOU50M))MOU50M = MOU50M + U50M(:)*FR(:,N)
      if(associated(MOV50M))MOV50M = MOV50M + V50M(:)*FR(:,N)
      if(associated(MOT10M))MOT10M = MOT10M + T10M(:)*FR(:,N)
      if(associated(MOQ10M))MOQ10M = MOQ10M + Q10M(:)*FR(:,N)
      if(associated(MOU10M))MOU10M = MOU10M + U10M(:)*FR(:,N)
      if(associated(MOV10M))MOV10M = MOV10M + V10M(:)*FR(:,N)
      if(associated(MOT2M))MOT2M = MOT2M + T2M(:)*FR(:,N)
      if(associated(MOQ2M))MOQ2M = MOQ2M + Q2M(:)*FR(:,N)
      if(associated(MOU2M))MOU2M = MOU2M + U2M(:)*FR(:,N)
      if(associated(MOV2M))MOV2M = MOV2M + V2M(:)*FR(:,N)

    endif
    call MAPL_TimerOff(MAPL,"-SURF")

!  Aggregate to tile
!-------------------

      CHX     = CHX + CH(:,N)*FR(:,N)
      CQX     = CQX + CQ(:,N)*FR(:,N)

      if(associated(CMT)) CMT     = CMT + CM(:,N)        *FR(:,N)
      if(associated(CNT)) CNT     = CNT + CN(:  )        *FR(:,N)
      if(associated(RIT)) RIT     = RIT + RIB(:  )       *FR(:,N)
      if(associated( TH)) TH      = TH  + CH(:,N)*TC(:,N)*FR(:,N)
      if(associated( QH)) QH      = QH  + CQ(:,N)*QC(:,N)*FR(:,N)
      if(associated(Z0H)) Z0H     = Z0H + ZT             *FR(:,N)
      if(associated(VNT)) VNT     = VNT + UUU            *FR(:,N)

      WW(:,N) = max(CH(:,N)*(TC(:,N)-TA-(MAPL_GRAV/MAPL_CP)*DZE)/TA + MAPL_VIREPS*CQ(:,N)*(QC(:,N)-QA),0.0)
      WW(:,N) = (HPBL*MAPL_GRAV*WW(:,N))**(2./3.)
      if(associated(GST)) GST     = GST + WW(:,N)        *FR(:,N)

   end do SUBTILES

   if(associated( TH)) TH  = TH /CHX
   if(associated( QH)) QH  = QH /CQX
   if(associated(CHT)) CHT = CHX
   if(associated(CQT)) CQT = CQX
   if(associated(GST)) GST = sqrt(max(GST+UCN,0.0))
   if(associated(ITYO)) ITYO = real(VEG1)   ! gkw: primary type exported... where it is used?

   deallocate ( lai1 )
   deallocate ( lai2 )
   deallocate ( wght )

   deallocate(TVA)
   deallocate(TVS)
   deallocate(URA)
   deallocate(UUU)
   deallocate(ZVG)
   deallocate(DZE)
   deallocate(Z0T)
   deallocate(D0T)
   deallocate(CHX)
   deallocate(CQX)
   deallocate(VEG1)
   deallocate(VEG2)
   deallocate(FVG1)
   deallocate(FVG2)
   deallocate(RE )
   deallocate(CN )
   deallocate(ZT )
   deallocate(ZQ )
   deallocate(UCN)
   deallocate(U50M )
   deallocate(V50M )
   deallocate(T10M )
   deallocate(Q10M )
   deallocate(U10M )
   deallocate(V10M )
   deallocate(T2M )
   deallocate(Q2M )
   deallocate(U2M )
   deallocate(V2M )
   deallocate(RHOH)
   deallocate(VKH)
   deallocate(VKM)
   deallocate(USTAR)
   deallocate(XX)
   deallocate(YY)
   deallocate(CU)
   deallocate(CT)
   deallocate(RIB)
   deallocate(ZETA)
   deallocate(WS)
   deallocate(IWATER)
   deallocate(PSMB)
   deallocate(PSL)

!  All done
! ------------------------------------------------------------------------------

    call MAPL_TimerOff ( MAPL, "RUN1"  )
    call MAPL_TimerOff ( MAPL, "TOTAL" )

    RETURN_(ESMF_SUCCESS)

end subroutine RUN1

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
! ------------------------------------------------------------------------------
! ------------------------------------------------------------------------------

subroutine RUN2 ( GC, IMPORT, EXPORT, CLOCK, RC )

! ------------------------------------------------------------------------------
! !ARGUMENTS:
! ------------------------------------------------------------------------------

    type(ESMF_GridComp),intent(inout) :: GC
    type(ESMF_State),   intent(inout) :: IMPORT
    type(ESMF_State),   intent(inout) :: EXPORT
    type(ESMF_Clock),   intent(inout) :: CLOCK
    integer,optional,   intent(out  ) :: RC

! ------------------------------------------------------------------------------
! ErrLog Variables
! ------------------------------------------------------------------------------

    character(len=ESMF_MAXSTR) :: Iam="RUN2"
    integer :: STATUS
    character(len=ESMF_MAXSTR) :: COMP_NAME

! ------------------------------------------------------------------------------
! Local derived type aliases
! ------------------------------------------------------------------------------

    type(MAPL_MetaComp),pointer     :: MAPL
    type(ESMF_Alarm)                :: ALARM

    integer :: IM,JM
    integer :: incl_Louis_extra_derivs

    real    :: SCALE4Z0

! ------------------------------------------------------------------------------
! Begin: Get the target components name and
! set-up traceback handle.
! ------------------------------------------------------------------------------

    call ESMF_GridCompGet ( GC, name=COMP_NAME, RC=STATUS )
    VERIFY_(STATUS)
    Iam=trim(COMP_NAME)//trim(Iam)

! Get my internal MAPL_Generic state
!-----------------------------------

    call MAPL_GetObjectFromGC ( GC, MAPL, RC=STATUS)
    VERIFY_(STATUS)

! Get parameters from generic state.
!-----------------------------------

    call MAPL_Get(MAPL, RUNALARM=ALARM, RC=STATUS)
    VERIFY_(STATUS)

    call MAPL_GetResource ( MAPL, incl_Louis_extra_derivs, Label="INCL_LOUIS_EXTRA_DERIVS:", DEFAULT=1, RC=STATUS)
    VERIFY_(STATUS)
    call MAPL_GetResource ( MAPL, SCALE4Z0, Label="SCALE4Z0:", DEFAULT=0.5, RC=STATUS)
    VERIFY_(STATUS)

! ------------------------------------------------------------------------------
! If its time, recalculate the LSM tile routine
! ------------------------------------------------------------------------------

    call MAPL_TimerOn ( MAPL,"TOTAL" )
    call MAPL_TimerOn ( MAPL,"RUN2"  )

    if(ESMF_AlarmIsRinging(ALARM, RC=STATUS))then
       call ESMF_AlarmRingerOff(ALARM, RC=STATUS)
       VERIFY_(STATUS)
       call Driver ( RC=STATUS )
       VERIFY_(STATUS)
    endif

    call MAPL_TimerOff ( MAPL, "RUN2"  )
    call MAPL_TimerOff ( MAPL, "TOTAL" )

    RETURN_(ESMF_SUCCESS)

    contains

! ------------------------------------------------------------------------------
! ------------------------------------------------------------------------------

      subroutine Driver ( RC )
        integer,optional,intent(OUT) :: RC

        character(len=ESMF_MAXSTR) :: IAm
        integer :: STATUS

        ! --------------------------------------------------------------------------
        ! Local derived type aliases
        ! --------------------------------------------------------------------------

        type(ESMF_STATE) :: INTERNAL

        ! -----------------------------------------------------
        ! IMPORT Pointers
        ! -----------------------------------------------------

        real, dimension(:),   pointer :: PS
        real, dimension(:),   pointer :: TA
        real, dimension(:),   pointer :: QA
        real, dimension(:),   pointer :: UU
        real, dimension(:),   pointer :: DZ
        real, dimension(:),   pointer :: PCU
        real, dimension(:),   pointer :: PLS
        real, dimension(:),   pointer :: SNO

        real, dimension(:),   pointer :: THATM
        real, dimension(:),   pointer :: QHATM
        real, dimension(:),   pointer :: CTATM
        real, dimension(:),   pointer :: CQATM
        real, dimension(:),   pointer :: ICE
        real, dimension(:),   pointer :: FRZR
        real, dimension(:),   pointer :: drpar
        real, dimension(:),   pointer :: dfpar
        real, dimension(:),   pointer :: drnir
        real, dimension(:),   pointer :: dfnir
        real, dimension(:),   pointer :: druvr
        real, dimension(:),   pointer :: dfuvr
        real, dimension(:),   pointer :: lwdnsrf
        real, dimension(:),   pointer :: alw
        real, dimension(:),   pointer :: blw
        real, dimension(:),   pointer :: CO2SC

        real, dimension(:),   pointer :: evap
        real, dimension(:),   pointer :: devap
        real, dimension(:),   pointer :: sh
        real, dimension(:),   pointer :: dsh

        real, dimension(:),   pointer :: ROOTL
        real, dimension(:),   pointer :: Z2CH
        real, dimension(:),   pointer :: LAI
        real, dimension(:),   pointer :: GRN
        real, dimension(:),   pointer :: ASCATZ0
        real, dimension(:),   pointer :: NDVI

        real, dimension(:,:), pointer :: DUDP
        real, dimension(:,:), pointer :: DUSV
        real, dimension(:,:), pointer :: DUWT
        real, dimension(:,:), pointer :: DUSD
        real, dimension(:,:), pointer :: BCDP
        real, dimension(:,:), pointer :: BCSV
        real, dimension(:,:), pointer :: BCWT
        real, dimension(:,:), pointer :: BCSD
        real, dimension(:,:), pointer :: OCDP
        real, dimension(:,:), pointer :: OCSV
        real, dimension(:,:), pointer :: OCWT
        real, dimension(:,:), pointer :: OCSD
        real, dimension(:,:), pointer :: SUDP
        real, dimension(:,:), pointer :: SUSV
        real, dimension(:,:), pointer :: SUWT
        real, dimension(:,:), pointer :: SUSD
        real, dimension(:,:), pointer :: SSDP
        real, dimension(:,:), pointer :: SSSV
        real, dimension(:,:), pointer :: SSWT
        real, dimension(:,:), pointer :: SSSD

        ! -----------------------------------------------------
        ! INTERNAL Pointers
        ! -----------------------------------------------------

        real, dimension(:),   pointer :: bf1
        real, dimension(:),   pointer :: bf2
        real, dimension(:),   pointer :: bf3
        real, dimension(:),   pointer :: vgwmax
        real, dimension(:),   pointer :: cdcr1
        real, dimension(:),   pointer :: cdcr2
        real, dimension(:),   pointer :: psis
        real, dimension(:),   pointer :: bee
        real, dimension(:),   pointer :: poros
        real, dimension(:),   pointer :: wpwet
        real, dimension(:),   pointer :: cond
        real, dimension(:),   pointer :: gnu
        real, dimension(:),   pointer :: ars1
        real, dimension(:),   pointer :: ars2
        real, dimension(:),   pointer :: ars3
        real, dimension(:),   pointer :: ara1
        real, dimension(:),   pointer :: ara2
        real, dimension(:),   pointer :: ara3
        real, dimension(:),   pointer :: ara4
        real, dimension(:),   pointer :: arw1
        real, dimension(:),   pointer :: arw2
        real, dimension(:),   pointer :: arw3
        real, dimension(:),   pointer :: arw4
        real, dimension(:),   pointer :: tsa1
        real, dimension(:),   pointer :: tsa2
        real, dimension(:),   pointer :: tsb1
        real, dimension(:),   pointer :: tsb2
        real, dimension(:),   pointer :: atau
        real, dimension(:),   pointer :: btau
        real, dimension(:,:), pointer :: ity
        real, dimension(:,:), pointer :: fvg
        real, dimension(:),   pointer :: capac
        real, dimension(:),   pointer :: catdef
        real, dimension(:),   pointer :: rzexc
        real, dimension(:),   pointer :: srfexc
        real, dimension(:),   pointer :: ghtcnt1
        real, dimension(:),   pointer :: ghtcnt2
        real, dimension(:),   pointer :: ghtcnt3
        real, dimension(:),   pointer :: ghtcnt4
        real, dimension(:),   pointer :: ghtcnt5
        real, dimension(:),   pointer :: ghtcnt6
        real, dimension(:),   pointer :: tsurf
        real, dimension(:),   pointer :: wesnn1
        real, dimension(:),   pointer :: wesnn2
        real, dimension(:),   pointer :: wesnn3
        real, dimension(:),   pointer :: htsnnn1
        real, dimension(:),   pointer :: htsnnn2
        real, dimension(:),   pointer :: htsnnn3
        real, dimension(:),   pointer :: sndzn1
        real, dimension(:),   pointer :: sndzn2
        real, dimension(:),   pointer :: sndzn3
        real, dimension(:,:), pointer :: tc
        real, dimension(:,:), pointer :: tg
        real, dimension(:,:), pointer :: qc
        real, dimension(:,:), pointer :: ch
        real, dimension(:,:), pointer :: cm
        real, dimension(:,:), pointer :: cq
        real, dimension(:,:), pointer :: fr
        real, dimension(:,:), pointer :: dcq
        real, dimension(:,:), pointer :: dch
        real, dimension(:),   pointer :: tile_id
        real, dimension(:),   pointer :: ndep
        real, dimension(:),   pointer :: cli_t2m
        real, dimension(:),   pointer :: bgalbvr
        real, dimension(:),   pointer :: bgalbvf
        real, dimension(:),   pointer :: bgalbnr
        real, dimension(:),   pointer :: bgalbnf
        real, dimension(:,:), pointer :: cncol
        real, dimension(:,:), pointer :: cnpft
        real, dimension(:,:), pointer :: tgwm
        real, dimension(:,:), pointer :: rzmm
        real, dimension(:),   pointer :: sfmcm
        real, dimension(:),   pointer :: bflowm
        real, dimension(:),   pointer :: totwatm
        real, dimension(:),   pointer :: tairm
        real, dimension(:),   pointer :: tpm
        real, dimension(:),   pointer :: cnsum
        real, dimension(:,:,:), pointer :: psnsunm
        real, dimension(:,:,:), pointer :: psnsham
        real, dimension(:),   pointer :: sndzm
        real, dimension(:),   pointer :: asnowm
        real, dimension(:,:), pointer :: RDU001
        real, dimension(:,:), pointer :: RDU002
        real, dimension(:,:), pointer :: RDU003
        real, dimension(:,:), pointer :: RDU004
        real, dimension(:,:), pointer :: RDU005
        real, dimension(:,:), pointer :: RBC001
        real, dimension(:,:), pointer :: RBC002
        real, dimension(:,:), pointer :: ROC001
        real, dimension(:,:), pointer :: ROC002
        real, dimension(:),   pointer :: IRRIGFRAC
        real, dimension(:),   pointer :: PADDYFRAC
        real, dimension(:),   pointer :: LAIMAX
        real, dimension(:),   pointer :: LAIMIN
        real, dimension(:),   pointer :: CLMPT
        real, dimension(:),   pointer :: CLMST
        real, dimension(:),   pointer :: CLMPF
        real, dimension(:),   pointer :: CLMSF

        ! -----------------------------------------------------
        ! EXPORT Pointers
        ! -----------------------------------------------------

        real, dimension(:),   pointer :: evapout
        real, dimension(:),   pointer :: sublim
        real, dimension(:),   pointer :: shout
        real, dimension(:),   pointer :: runoff
        real, dimension(:),   pointer :: evpint
        real, dimension(:),   pointer :: evpsoi
        real, dimension(:),   pointer :: evpveg
        real, dimension(:),   pointer :: evpice
        real, dimension(:),   pointer :: evpsno
        real, dimension(:),   pointer :: bflow
        real, dimension(:),   pointer :: runsurf
        real, dimension(:),   pointer :: smelt
        real, dimension(:),   pointer :: accum
        real, dimension(:),   pointer :: hlwup
        real, dimension(:),   pointer :: swndsrf
        real, dimension(:),   pointer :: lwndsrf
        real, dimension(:),   pointer :: hlatn
        real, dimension(:),   pointer :: qinfil
        real, dimension(:),   pointer :: ar1
        real, dimension(:),   pointer :: ar2
        real, dimension(:),   pointer :: rzeq
        real, dimension(:),   pointer :: ghflx
        real, dimension(:),   pointer :: tpsurf
        real, dimension(:),   pointer :: tpsn1
        real, dimension(:),   pointer :: tpust
        real, dimension(:),   pointer :: tpsat
        real, dimension(:),   pointer :: tpwlt
        real, dimension(:),   pointer :: asnow
        real, dimension(:),   pointer :: frsat
        real, dimension(:),   pointer :: frust
        real, dimension(:),   pointer :: frwlt
        real, dimension(:),   pointer :: tp1
        real, dimension(:),   pointer :: tp2
        real, dimension(:),   pointer :: tp3
        real, dimension(:),   pointer :: tp4
        real, dimension(:),   pointer :: tp5
        real, dimension(:),   pointer :: tp6
        real, dimension(:),   pointer :: emis
        real, dimension(:),   pointer :: albvr
        real, dimension(:),   pointer :: albvf
        real, dimension(:),   pointer :: albnr
        real, dimension(:),   pointer :: albnf
        real, dimension(:),   pointer :: delts
        real, dimension(:),   pointer :: delqs
        real, dimension(:),   pointer :: delevap
        real, dimension(:),   pointer :: delsh
        real, dimension(:),   pointer :: tst
        real, dimension(:),   pointer :: lst
        real, dimension(:),   pointer :: qst

        real, dimension(:),   pointer :: WET1
        real, dimension(:),   pointer :: WET2
        real, dimension(:),   pointer :: WET3
        real, dimension(:),   pointer :: WCSF
        real, dimension(:),   pointer :: WCRZ
        real, dimension(:),   pointer :: WCPR
        real, dimension(:),   pointer :: SNOMAS
        real, dimension(:),   pointer :: SNOWDP

        real, dimension(:),   pointer :: EVLAND
        real, dimension(:),   pointer :: PRLAND
        real, dimension(:),   pointer :: SNOLAND
        real, dimension(:),   pointer :: DRPARLAND
        real, dimension(:),   pointer :: DFPARLAND
        real, dimension(:),   pointer :: LHSNOW
        real, dimension(:),   pointer :: SWNETSNOW1
        real, dimension(:),   pointer :: LWUPSNOW
        real, dimension(:),   pointer :: LWDNSNOW
        real, dimension(:),   pointer :: TCSORIG
        real, dimension(:),   pointer :: TPSN1IN
        real, dimension(:),   pointer :: TPSN1OUT
        real, dimension(:),   pointer :: GHSNOW
        real, dimension(:),   pointer :: LHLAND
        real, dimension(:),   pointer :: SHLAND
        real, dimension(:),   pointer :: SWLAND
        real, dimension(:),   pointer :: SWDOWNLAND
        real, dimension(:),   pointer :: LWLAND
        real, dimension(:),   pointer :: GHLAND
        real, dimension(:),   pointer :: GHTSKIN
        real, dimension(:),   pointer :: SMLAND
        real, dimension(:),   pointer :: TWLAND
        real, dimension(:),   pointer :: TELAND
        real, dimension(:),   pointer :: TSLAND
        real, dimension(:),   pointer :: DWLAND
        real, dimension(:),   pointer :: DHLAND
        real, dimension(:),   pointer :: SPLAND
        real, dimension(:),   pointer :: SPWATR
        real, dimension(:),   pointer :: SPSNOW

        real, dimension(:),   pointer :: CNLAI
        real, dimension(:),   pointer :: CNTLAI
        real, dimension(:),   pointer :: CNSAI
        real, dimension(:),   pointer :: CNTOTC
        real, dimension(:),   pointer :: CNVEGC
        real, dimension(:),   pointer :: CNROOT
        real, dimension(:),   pointer :: CNNPP
        real, dimension(:),   pointer :: CNGPP
        real, dimension(:),   pointer :: CNSR
        real, dimension(:),   pointer :: CNNEE
        real, dimension(:),   pointer :: CNXSMR
        real, dimension(:),   pointer :: CNADD
        real, dimension(:),   pointer :: CNLOSS
        real, dimension(:),   pointer :: CNBURN
        real, dimension(:),   pointer :: PARABS
        real, dimension(:),   pointer :: PARINC
        real, dimension(:),   pointer :: SCSAT
        real, dimension(:),   pointer :: SCUNS
        real, dimension(:),   pointer :: BTRANT
        real, dimension(:),   pointer :: SIF
        real, dimension(:),   pointer :: CNFSEL
        real, dimension(:),   pointer :: CNCO2 
        real, dimension(:),   pointer :: CNSAI11
        real, dimension(:),   pointer :: CNSAI12 
        real, dimension(:),   pointer :: CNSAI13 
        real, dimension(:),   pointer :: CNSAI21 
        real, dimension(:),   pointer :: CNSAI22
        real, dimension(:),   pointer :: CNSAI23 
        real, dimension(:),   pointer :: CNSAI31 
        real, dimension(:),   pointer :: CNSAI32
        real, dimension(:),   pointer :: CNSAI33
        real, dimension(:),   pointer :: CNSAI41
        real, dimension(:),   pointer :: CNSAI42
        real, dimension(:),   pointer :: CNSAI43
        real, dimension(:),   pointer :: CNLAI11
        real, dimension(:),   pointer :: CNLAI12 
        real, dimension(:),   pointer :: CNLAI13 
        real, dimension(:),   pointer :: CNLAI21 
        real, dimension(:),   pointer :: CNLAI22
        real, dimension(:),   pointer :: CNLAI23 
        real, dimension(:),   pointer :: CNLAI31 
        real, dimension(:),   pointer :: CNLAI32
        real, dimension(:),   pointer :: CNLAI33
        real, dimension(:),   pointer :: CNLAI41
        real, dimension(:),   pointer :: CNLAI42
        real, dimension(:),   pointer :: CNLAI43

        real, dimension(:),   pointer :: WAT10CM
        real, dimension(:),   pointer :: WATSOI
        real, dimension(:),   pointer :: ICESOI
        real, dimension(:),   pointer :: SHSNOW
        real, dimension(:),   pointer :: AVETSNOW
        real, pointer, dimension(:)   :: RMELTDU001
        real, pointer, dimension(:)   :: RMELTDU002
        real, pointer, dimension(:)   :: RMELTDU003
        real, pointer, dimension(:)   :: RMELTDU004
        real, pointer, dimension(:)   :: RMELTDU005
        real, pointer, dimension(:)   :: RMELTBC001
        real, pointer, dimension(:)   :: RMELTBC002
        real, pointer, dimension(:)   :: RMELTOC001
        real, pointer, dimension(:)   :: RMELTOC002
        real, pointer, dimension(:)   :: IRRIGRATE

        ! --------------------------------------------------------------------------
        ! Local pointers for tile variables
        ! --------------------------------------------------------------------------

        INTEGER,pointer,dimension(:) :: CAT_ID
        real,pointer,dimension(:) :: dzsf
        real,pointer,dimension(:) :: swnetfree
        real,pointer,dimension(:) :: swnetsnow
        real,pointer,dimension(:) :: qa1
        real,pointer,dimension(:) :: qa2
        real,pointer,dimension(:) :: qa4
        real,pointer,dimension(:) :: tilezero
        real,pointer,dimension(:) :: zth
        real,pointer,dimension(:) :: lats
        real,pointer,dimension(:) :: lons
        real,pointer,dimension(:) :: slr
        real,pointer,dimension(:) :: rdc
	real,pointer,dimension(:) :: PRECU
	real,pointer,dimension(:) :: PRELS
	real,pointer,dimension(:) :: SNOW
	real,pointer,dimension(:) :: UUU, RHO
	real,pointer,dimension(:) :: LAI0,GRN0,ZVG
	real,pointer,dimension(:) :: Z0, D0
	real,pointer,dimension(:) :: sfmc, rzmc, prmc, entot, wtot
	real,pointer,dimension(:) :: ghflxsno, ghflxtskin
        real,pointer,dimension(:) :: SHSNOW1, AVETSNOW1, WAT10CM1, WATSOI1, ICESOI1
        real,pointer,dimension(:) :: LHSNOW1, LWUPSNOW1, LWDNSNOW1, NETSWSNOW
        real,pointer,dimension(:) :: TCSORIG1, TPSN1IN1, TPSN1OUT1
	real,pointer,dimension(:) :: WCHANGE, ECHANGE, HSNACC, EVACC, SHACC
	real,pointer,dimension(:) :: SNOVR, SNOVF, SNONR, SNONF
	real,pointer,dimension(:) :: VSUVR, VSUVF
	real,pointer,dimension(:) :: ALWX, BLWX
        real,pointer,dimension(:) :: LHACC, SUMEV
	real,pointer,dimension(:) :: fveg1, fveg2
        real,pointer,dimension(:) :: FICE1
        real,pointer,dimension(:) :: SLDTOT

!       real*8,pointer,dimension(:) :: fsum

        real,pointer,dimension(:,:) :: ghtcnt
        real,pointer,dimension(:,:) :: wesnn
        real,pointer,dimension(:,:) :: htsnnn
        real,pointer,dimension(:,:) :: sndzn
        real,pointer,dimension(:,:) :: shsbt
        real,pointer,dimension(:,:) :: dshsbt
        real,pointer,dimension(:,:) :: evsbt
        real,pointer,dimension(:,:) :: devsbt
        real,pointer,dimension(:,:) :: DEDTC 
        real,pointer,dimension(:,:) :: DHSDQA
        real,pointer,dimension(:,:) :: CFT
        real,pointer,dimension(:,:) :: RA
        real,pointer,dimension(:,:) :: CFQ
        real,pointer,dimension(:,:) :: TCO
        real,pointer,dimension(:,:) :: QCO
        real,pointer,dimension(:,:) :: DQS
        real,pointer,dimension(:,:) :: QSAT

        integer,dimension(:),pointer :: veg1
        integer,dimension(:),pointer :: veg2

        real,pointer,dimension(:) :: RCSAT 
	real,pointer,dimension(:) :: DRCSDT
	real,pointer,dimension(:) :: DRCSDQ
	real,pointer,dimension(:) :: RCUNS 
	real,pointer,dimension(:) :: DRCUDT
	real,pointer,dimension(:) :: DRCUDQ

        real,pointer,dimension(:,:,:) :: RCONSTIT
        real,pointer,dimension(:,:)   :: TOTDEPOS
        real,pointer,dimension(:,:)   :: RMELT

        ! --------------------------------------------------------------------------
        ! Locals for parameter lookup
        ! --------------------------------------------------------------------------

        ! vegetation calculations

        real,dimension(NTYPS) :: VGRF11
        real,dimension(NTYPS) :: VGRF12
        real,dimension(NTYPS) :: VGTR11
        real,dimension(NTYPS) :: VGTR12
        real,dimension(NTYPS) :: VGROCA
        real,dimension(NTYPS) :: VGROTD
        real,dimension(NTYPS) :: VGRDRS
        real,dimension(NTYPS) :: VGDDA, VGDDB, VGDDC
        real,dimension(NTYPS) :: VGRDA, VGRDB

        real,dimension(:),allocatable :: RSL1, RSL2
        real,dimension(:),allocatable :: SQSCAT
        real,allocatable,dimension(:) :: rdc_tmp_1, rdc_tmp_2

        ! albedo calculation stuff

        type(ESMF_Config)           :: CF
        type(MAPL_SunOrbit)         :: ORBIT
        type(ESMF_Time)             :: CURRENT_TIME, StopTime, NextTime
        type(ESMF_Time)             :: BEFORE
        type(ESMF_Time)             :: NOW
        type(ESMF_Time)             :: MODELSTART
        type(ESMF_Time)             :: AFTER
        type(ESMF_TimeInterval)     :: DELT
        type(ESMF_TimeInterval)     :: TINT
        real                        :: DT_SOLAR
        type(ESMF_Alarm)            :: SOLALARM
        logical                     :: solalarmison
        logical                     :: debugzth
        real                        :: FAC

        real                        :: DT
        integer                     :: NTILES
        integer                     :: I, J, K, N

	! dummy variables for call to get snow temp

        real    :: FICE
        logical :: DUMFLAG1,DUMFLAG2
        integer                         :: nmax
        type(ESMF_VM)                   :: VM

#ifdef DBG_CNLSM_INPUTS
        ! vars for debugging purposes
        type(ESMF_Grid)                 :: TILEGRID
        type (MAPL_LocStream)           :: LOCSTREAM
        integer, pointer                :: mask(:)
        integer                         :: nt
        integer, save                   :: unit_i=0
        logical, save                   :: firsttime=.true.
        integer                         :: unit
	integer 			:: NT_GLOBAL

#endif

       ! Offline case

        type(OFFLINE_WRAP)          :: wrap
        integer                     :: OFFLINE_MODE
        real,dimension(:,:),allocatable :: ALWN, BLWN
        ! un-adelterated TC's and QC's
        real, pointer               :: TC1_0(:), TC2_0(:),  TC4_0(:)
        real, pointer               :: QA1_0(:), QA2_0(:),  QA4_0(:)
        real, pointer               :: PLSIN(:)

        ! --------------------------------------------------------------------------
        ! Lookup tables
        ! --------------------------------------------------------------------------

        data VGRF11 / 0.100, 0.100, 0.070, 0.105, 0.100, 0.100 /
        data VGRF12 / 0.160, 0.160, 0.160, 0.360, 0.160, 0.160 /
        data VGTR11 / 0.050, 0.050, 0.050, 0.070, 0.050, 0.050 /
        data VGTR12 / 0.001, 0.001, 0.001, 0.220, 0.001, 0.001 /
        data VGROTD / 1.000, 1.000, 0.500, 0.500, 0.500, 0.200 / 

        data VGROCA / 0.384E-6, 0.384E-6, 0.384E-6, 0.384E-6, 0.384E-6, 0.384E-6/
        data VGRDRS / 0.750E13, 0.750E13, 0.750E13, 0.400E13, 0.750E13, 0.750E13/

! Correction to RDC formulation -Randy Koster, 4/1/2011
!        data VGRDA / 285.9, 294.9, 652.9,  25.8,  100.7,  22.9,  23.8, 23.8/
!        data VGRDB / 5.1 ,  7.2, 10.8,  4.8,  1.8,  5.1,  .000, .000/

        data VGRDA / 285.9, 355.18, 660.24,  30.06,  100.7,  24.36/
        data VGRDB / 5.1 ,  7.2, 10.5,  4.8,  1.8,  5.1/

! gkw: following is for CN model
! ------------------------------
    integer, parameter :: nveg  = num_veg ! number of vegetation types
    integer, parameter :: nzone = num_zon ! number of stress zones

    real, allocatable, dimension(:) :: btran, wgt, bt1, bt2, bt4, wpp, fwet
    real, allocatable, dimension(:) :: sm1, sm2, sm4
    real, allocatable, dimension(:) :: btran1, btran2, btran3, tcx, qax
    real, allocatable, dimension(:) :: rcx, rcxdt, rcxdq, tx1, tx2, tx3, qx1, qx2, qx3
    real, allocatable, dimension(:,:) :: tgw, rzm, rc00, rcdt,rcdq, totcolc, wtzone
    real, allocatable, dimension(:,:,:) :: elai,esai,fveg,tlai,psnsun,psnsha,laisun,laisha
    integer, allocatable, dimension(:,:,:) :: ityp
    real, allocatable, dimension(:) :: car1, car2, car4
    real, allocatable, dimension(:) :: para
    real, allocatable, dimension(:) :: dayl, dayl_fac
    real, allocatable, dimension(:), save :: nee, npp, gpp, sr, padd, root, vegc, xsmr,burn, closs , fsel

    ! ***************************************************************************************************************************************************************
    ! Begin Carbon Tracker variables
    !
    ! use EEA global average CO2 to scale 2001-2014 CarbonTracker CO2 monthly mean diurnal cycle to obtain CO2 for 1850-2000.
    ! extended from the last cycle when carbon reaches equilibrium with the 2001-2014 CarbonTracker CO2 monthly mean diurnal 
    ! cycle * 280ppm/389.8899ppm, fzeng, Apr 2017.
    ! EEA global average CO2 is from http://www.eea.europa.eu/data-and-maps/figures/atmospheric-concentration-of-co2-ppm-1  
    ! --------------------------------------------------------------------------------------------------------------------  

    real               :: co2g                 ! global average atmospheric carbon dioxide concentration, varies after 1850
    integer, parameter :: byr_co2g  = 1851     ! year global average atmospheric CO2 concentration began to increase from 280.e-6 
    integer, parameter :: myr_co2g  = 1950     ! year global average atmospheric CO2 concentration reached 311.e-6 
    integer, parameter :: eyr_co2g  = 2012     ! year global average atmospheric CO2 concentration reached 391.e-6 
    real, parameter    :: co2g_byr  = 280.e-6  ! pre-industrial global average atmospheric carbon dioxide concentration (i.e. before byr_co2g)
    real, parameter    :: co2g_myr  = 311.e-6  ! global average atmospheric CO2 concentration in myr_co2g
    real, parameter    :: co2g_eyr  = 391.e-6  ! global average atmospheric CO2 concentration in eyr_co2g
    real, parameter    :: dco2g_1   = (co2g_myr-co2g_byr)/(myr_co2g-byr_co2g) ! yearly atmospheric CO2 concentration increment for period 1 (byr_co2g to myr_co2g)
    real, parameter    :: dco2g_2   = (co2g_eyr-co2g_myr)/(eyr_co2g-myr_co2g) ! yearly atmospheric CO2 concentration increment for period 2 (myr_co2g to eyr_co2g)
    real, parameter    :: CTco2g = 389.8899e-6     ! Spatial (tile area weighted) and temporal average of 2001-2014 CarbonTracker CO2
    real, allocatable, dimension(:) :: co2v  ! spatial varying atmospheric carbon dioxide concentration

    ! parameters for calculating CT indices for tiles
    ! -----------------------------------------------
    integer, parameter :: CT_grid_N_lon  = 120  ! lon dimension CarbonTracker CO2 data
    integer, parameter :: CT_grid_N_lat  =  90  ! lat dimension CarbonTracker CO2 data
    real, parameter    :: CT_grid_dlon = 360./real(CT_grid_N_lon), CT_grid_dlat = 180./real(CT_grid_N_lat)
    INTEGER            ::  info, comm, CTfile, Y1, M1, This3H, ThisCO2_Year, NUNQ, CO2_YEAR
    logical, allocatable, dimension (:)        :: unq_mask
    integer, allocatable, dimension (:,:)      :: CT_index
    integer, allocatable, dimension (:)        :: ct2cat, ThisIndex, loc_int
    integer, allocatable, dimension (:), save  :: ct_tid
    real, dimension (:,:,:,:),     allocatable :: CTCO2_TMP
    real, dimension (:,:,:), save, allocatable :: CT_CO2V
    logical, save      :: first_ct = .true.
    integer, save      :: FIRST_YY
        
    ! End Carbon Tracker variables
    ! *************************************************************************************************************************************************************

    ! prescribe DYNVEG parameters
    ! ---------------------------

    real, parameter :: dtc = 0.03 ! canopy temperature perturbation (K) [approx 1:10000]
    real, parameter :: dea = 0.10 ! vapor pressure perturbation (Pa) [approx 1:10000]

    real, allocatable, dimension(:) :: totwat ! total soil liquid water (kg/m2)
    real, save :: ashift = 0. ! for baseflow. gkw: this should match value in routine "base" in catchment

    integer :: nz, iv
    real :: cn1, cn2, cn3, cn12, cn23, ar, f1, f2, f3, f4, ax1, ax2, ax4

    real, allocatable, dimension(:,:) :: psnsunx,psnshax,sifsunx,sifshax,laisunx,laishax,elaz,esaz,fvez
    integer, allocatable, dimension(:,:) :: ityz

    real, allocatable, dimension(:) :: ht, tp, soilice
    real :: zbar, frice

    real, allocatable, dimension(:,:)  :: col
    real, allocatable, dimension(:,:,:) :: pft

    integer :: ntile, nv, dpy, ierr, iok, ndt
    integer, save :: year_prev = -9999
 

    integer :: AGCM_YY, AGCM_MM, AGCM_DD, AGCM_MI, AGCM_S, AGCM_HH, dofyr, sofmin
    logical, save :: first = .true.
    integer*8, save :: istep = 1 ! gkw: legacy variable from offline

   ! real :: co2
    real, external :: getco2

    ! temporaries for call to SIBALB for each type
    ! --------------------------------------------
    real, allocatable, dimension(:) :: lai1, lai2, wght
    real, allocatable, dimension(:) :: ALBVR_tmp, ALBNR_tmp, ALBVF_tmp, ALBNF_tmp
    real, allocatable, dimension(:) :: SNOVR_tmp, SNONR_tmp, SNOVF_tmp, SNONF_tmp

    ! Variables for FPAR scaling
    ! --------------------------
    
    real, save,allocatable,dimension (:,:,:,:) :: Kappa, Lambda, Mu
    real, save,allocatable,dimension (:,:,:)   :: MnVal, MxVal
    integer, save, allocatable, dimension (:)  :: modis_tid, ThisMIndex
    integer                                    :: n_modis, NTCurrent, CDFfile, infos, comms
    integer, allocatable, dimension (:,:)      :: modis_index
    integer, allocatable, dimension (:)        :: modis2cat
    real   , allocatable, dimension (:)        :: m_lons, m_lats
    real   , allocatable, dimension (:,:)      :: scaled_fpar, parav, parzone, unscaled_fpar
    REAL   , PARAMETER                         :: TILEINT = 2.
    integer, PARAMETER                         :: NOCTAD = 46, NSETS = 2
    real                                       :: CLM4_fpar, CLM4_cdf, MODIS_fpar, tmparr(1,1,1,2), &
         ThisK, ThisL, ThisM, ThisMin, ThisMax, tmparr2(1,1,1), ThisFPAR, ZFPAR  
    character (len=ESMF_MAXSTR)   :: VISMEANFILE, VISSTDFILE, NIRMEANFILE, NIRSTDFILE, FPARMEANFILE, FPARSTDFILE
    real, allocatable, dimension (:)           :: MODISVISmean, MODISVISstd, MODISNIRmean, MODISNIRstd, MODELFPARmean, MODELFPARstd
    logical, save :: first_fpar = .true.

        IAm=trim(COMP_NAME)//"::RUN2::Driver"

        ! Begin

        IAm=trim(COMP_NAME)//"Driver"

        ! --------------------------------------------------------------------------
        ! Get time step from configuration
        ! --------------------------------------------------------------------------

        call ESMF_GridCompGet  ( GC, CONFIG=CF, RC=STATUS )
        VERIFY_(STATUS)

        ! --------------------------------------------------------------------------
        ! Get my internal MAPL_Generic state
        ! --------------------------------------------------------------------------

        call MAPL_GetObjectFromGC ( GC, MAPL, RC=STATUS )
        VERIFY_(STATUS)

        call MAPL_Get(MAPL, HEARTBEAT = DT, RC=STATUS)
        VERIFY_(STATUS)

        call ESMF_ConfigGetAttribute ( CF, DT                  ,&
             Label   = trim(COMP_NAME)//"_DT:"     ,&
             Default = DT                          ,&
             RC=STATUS )
        VERIFY_(STATUS)

        ! Get component's private internal state
        call ESMF_UserCompGetInternalState(gc, 'OfflineMode', wrap, status)
        VERIFY_(status)

        call ESMF_VMGetCurrent ( VM, RC=STATUS )
        ! Component's offline mode
        OFFLINE_MODE = wrap%ptr%CATCH_OFFLINE
        ! if (MAPL_AM_I_Root(VM)) print *, trim(Iam)//'::OFFLINE mode: ', is_OFFLINE

        ! --------------------------------------------------------------------------
        ! Get parameters from generic state.
        ! --------------------------------------------------------------------------

        call MAPL_Get ( MAPL                 ,&
             RUNALARM  = ALARM                            ,&
             ORBIT     = ORBIT                            ,&
             TILELATS  = LATS                             ,&
             TILELONS  = LONS                             ,&
             INTERNAL_ESMF_STATE = INTERNAL               ,&
             RC=STATUS )
        VERIFY_(STATUS)

        ! -----------------------------------------------------
        ! IMPORT Pointers
        ! -----------------------------------------------------

        call MAPL_GetPointer(IMPORT,PS     ,'PS'     ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(IMPORT,TA     ,'TA'     ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(IMPORT,QA     ,'QA'     ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(IMPORT,UU     ,'UU'     ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(IMPORT,DZ     ,'DZ'     ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(IMPORT,PCU    ,'PCU'    ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(IMPORT,PLS    ,'PLS'    ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(IMPORT,SNO    ,'SNO'    ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(IMPORT,ICE    ,'ICE'    ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(IMPORT,FRZR   ,'FRZR'   ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(IMPORT,DRPAR  ,'DRPAR'  ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(IMPORT,DFPAR  ,'DFPAR'  ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(IMPORT,DRNIR  ,'DRNIR'  ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(IMPORT,DFNIR  ,'DFNIR'  ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(IMPORT,DRUVR  ,'DRUVR'  ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(IMPORT,DFUVR  ,'DFUVR'  ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(IMPORT,LWDNSRF,'LWDNSRF',RC=STATUS); VERIFY_(STATUS)

        call MAPL_GetPointer(IMPORT,ALW    ,'ALW'    ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(IMPORT,BLW    ,'BLW'    ,RC=STATUS); VERIFY_(STATUS)

        call MAPL_GetPointer(IMPORT,EVAP   ,'EVAP'   ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(IMPORT,DEVAP  ,'DEVAP'  ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(IMPORT,SH     ,'SH'     ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(IMPORT,DSH    ,'DSH'    ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(IMPORT,THATM  ,'THATM'  ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(IMPORT,QHATM  ,'QHATM'  ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(IMPORT,CTATM  ,'CTATM'  ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(IMPORT,CQATM  ,'CQATM'  ,RC=STATUS); VERIFY_(STATUS)        
        IF (ATM_CO2 == 4) call MAPL_GetPointer(IMPORT,CO2SC  ,'CO2SC'  ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(IMPORT,LAI    ,'LAI'    ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(IMPORT,GRN    ,'GRN'    ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(IMPORT,ROOTL  ,'ROOTL'  ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(IMPORT,Z2CH   ,'Z2CH'   ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(IMPORT,ASCATZ0,'ASCATZ0',RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(IMPORT,NDVI   ,'NDVI'   ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(IMPORT,DUDP   ,'DUDP'   ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(IMPORT,DUSV   ,'DUSV'   ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(IMPORT,DUWT   ,'DUWT'   ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(IMPORT,DUSD   ,'DUSD'   ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(IMPORT,BCDP   ,'BCDP'   ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(IMPORT,BCSV   ,'BCSV'   ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(IMPORT,BCWT   ,'BCWT'   ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(IMPORT,BCSD   ,'BCSD'   ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(IMPORT,OCDP   ,'OCDP'   ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(IMPORT,OCSV   ,'OCSV'   ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(IMPORT,OCWT   ,'OCWT'   ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(IMPORT,OCSD   ,'OCSD'   ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(IMPORT,SUDP   ,'SUDP'   ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(IMPORT,SUSV   ,'SUSV'   ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(IMPORT,SUWT   ,'SUWT'   ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(IMPORT,SUSD   ,'SUSD'   ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(IMPORT,SSDP   ,'SSDP'   ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(IMPORT,SSSV   ,'SSSV'   ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(IMPORT,SSWT   ,'SSWT'   ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(IMPORT,SSSD   ,'SSSD'   ,RC=STATUS); VERIFY_(STATUS)

        ! -----------------------------------------------------
        ! INTERNAL Pointers
        ! -----------------------------------------------------

        call MAPL_GetPointer(INTERNAL,BF1        ,'BF1'        ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,BF2        ,'BF2'        ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,BF3        ,'BF3'        ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,VGWMAX     ,'VGWMAX'     ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,CDCR1      ,'CDCR1'      ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,CDCR2      ,'CDCR2'      ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,PSIS       ,'PSIS'       ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,BEE        ,'BEE'        ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,POROS      ,'POROS'      ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,WPWET      ,'WPWET'      ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,COND       ,'COND'       ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,GNU        ,'GNU'        ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,ARS1       ,'ARS1'       ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,ARS2       ,'ARS2'       ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,ARS3       ,'ARS3'       ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,ARA1       ,'ARA1'       ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,ARA2       ,'ARA2'       ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,ARA3       ,'ARA3'       ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,ARA4       ,'ARA4'       ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,ARW1       ,'ARW1'       ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,ARW2       ,'ARW2'       ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,ARW3       ,'ARW3'       ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,ARW4       ,'ARW4'       ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,TSA1       ,'TSA1'       ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,TSA2       ,'TSA2'       ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,TSB1       ,'TSB1'       ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,TSB2       ,'TSB2'       ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,ATAU       ,'ATAU'       ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,BTAU       ,'BTAU'       ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,ITY        ,'ITY'        ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,FVG        ,'FVG'        ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,TC         ,'TC'         ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,QC         ,'QC'         ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,TG         ,'TG'         ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,CAPAC      ,'CAPAC'      ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,CATDEF     ,'CATDEF'     ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,RZEXC      ,'RZEXC'      ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,SRFEXC     ,'SRFEXC'     ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,GHTCNT1    ,'GHTCNT1'    ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,GHTCNT2    ,'GHTCNT2'    ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,GHTCNT3    ,'GHTCNT3'    ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,GHTCNT4    ,'GHTCNT4'    ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,GHTCNT5    ,'GHTCNT5'    ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,GHTCNT6    ,'GHTCNT6'    ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,TSURF      ,'TSURF'      ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,WESNN1     ,'WESNN1'     ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,WESNN2     ,'WESNN2'     ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,WESNN3     ,'WESNN3'     ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,HTSNNN1    ,'HTSNNN1'    ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,HTSNNN2    ,'HTSNNN2'    ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,HTSNNN3    ,'HTSNNN3'    ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,SNDZN1     ,'SNDZN1'     ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,SNDZN2     ,'SNDZN2'     ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,SNDZN3     ,'SNDZN3'     ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,CH         ,'CH'         ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,CM         ,'CM'         ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,CQ         ,'CQ'         ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,FR         ,'FR'         ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,DCQ        ,'DCQ'        ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,DCH        ,'DCH'        ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,TILE_ID    ,'TILE_ID'    ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,NDEP       ,'NDEP'       ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,CLI_T2M    ,'CLI_T2M'    ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,BGALBVR    ,'BGALBVR'    ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,BGALBVF    ,'BGALBVF'    ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,BGALBNR    ,'BGALBNR'    ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,BGALBNF    ,'BGALBNF'    ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,CNCOL      ,'CNCOL'      ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,CNPFT      ,'CNPFT'      ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,TGWM       ,'TGWM'       ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,RZMM       ,'RZMM'       ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,SFMCM      ,'SFMCM'      ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,BFLOWM     ,'BFLOWM'     ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,TOTWATM    ,'TOTWATM'    ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,TAIRM      ,'TAIRM'      ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,TPM        ,'TPM'        ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,CNSUM      ,'CNSUM'      ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,PSNSUNM    ,'PSNSUNM'    ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,PSNSHAM    ,'PSNSHAM'    ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,SNDZM      ,'SNDZM'      ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(INTERNAL,ASNOWM     ,'ASNOWM'     ,RC=STATUS); VERIFY_(STATUS)
 
        if (N_CONST_LAND4SNWALB /= 0) then
           call MAPL_GetPointer(INTERNAL,RDU001     ,'RDU001'     , RC=STATUS); VERIFY_(STATUS)
           call MAPL_GetPointer(INTERNAL,RDU002     ,'RDU002'     , RC=STATUS); VERIFY_(STATUS)
           call MAPL_GetPointer(INTERNAL,RDU003     ,'RDU003'     , RC=STATUS); VERIFY_(STATUS)
           call MAPL_GetPointer(INTERNAL,RDU004     ,'RDU004'     , RC=STATUS); VERIFY_(STATUS)
           call MAPL_GetPointer(INTERNAL,RDU005     ,'RDU005'     , RC=STATUS); VERIFY_(STATUS)
           call MAPL_GetPointer(INTERNAL,RBC001     ,'RBC001'     , RC=STATUS); VERIFY_(STATUS)
           call MAPL_GetPointer(INTERNAL,RBC002     ,'RBC002'     , RC=STATUS); VERIFY_(STATUS)
           call MAPL_GetPointer(INTERNAL,ROC001     ,'ROC001'     , RC=STATUS); VERIFY_(STATUS)
           call MAPL_GetPointer(INTERNAL,ROC002     ,'ROC002'     , RC=STATUS); VERIFY_(STATUS)
        endif

        IF (RUN_IRRIG /= 0) THEN
           call MAPL_GetPointer(INTERNAL,IRRIGFRAC ,'IRRIGFRAC' , RC=STATUS); VERIFY_(STATUS)
           call MAPL_GetPointer(INTERNAL,PADDYFRAC ,'PADDYFRAC' , RC=STATUS); VERIFY_(STATUS)
           call MAPL_GetPointer(INTERNAL,LAIMAX    ,'LAIMAX'    , RC=STATUS); VERIFY_(STATUS)
           call MAPL_GetPointer(INTERNAL,LAIMIN    ,'LAIMIN'    , RC=STATUS); VERIFY_(STATUS)
           call MAPL_GetPointer(INTERNAL,CLMPT     ,'CLMPT'     , RC=STATUS); VERIFY_(STATUS)
           call MAPL_GetPointer(INTERNAL,CLMST     ,'CLMST'     , RC=STATUS); VERIFY_(STATUS)
           call MAPL_GetPointer(INTERNAL,CLMPF     ,'CLMPF'     , RC=STATUS); VERIFY_(STATUS)
           call MAPL_GetPointer(INTERNAL,CLMSF     ,'CLMSF'     , RC=STATUS); VERIFY_(STATUS)
        ENDIF

        ! -----------------------------------------------------
        ! EXPORT POINTERS
        ! -----------------------------------------------------

        call MAPL_GetPointer(EXPORT,EVAPOUT,'EVAPOUT',ALLOC=.true.,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,SUBLIM,'SUBLIM',ALLOC=.true.,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,SHOUT,  'SHOUT'  ,ALLOC=.true.,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,RUNOFF, 'RUNOFF' ,ALLOC=.true.,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,EVPINT, 'EVPINT' ,ALLOC=.true.,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,EVPSOI, 'EVPSOI' ,ALLOC=.true.,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,EVPVEG, 'EVPVEG' ,ALLOC=.true.,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,EVPICE, 'EVPICE' ,ALLOC=.true.,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,WAT10CM,'WAT10CM',ALLOC=.true.,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,WATSOI, 'WATSOI' ,ALLOC=.true.,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,ICESOI, 'ICESOI' ,ALLOC=.true.,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,EVPSNO, 'EVPSNO'              ,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,BFLOW,  'BASEFLOW',ALLOC=.true.,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,RUNSURF,'RUNSURF',ALLOC=.true.,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,SMELT,  'SMELT'  ,ALLOC=.true.,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,HLWUP,  'HLWUP'  ,ALLOC=.true.,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,SWNDSRF,'SWNDSRF',ALLOC=.true.,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,LWNDSRF,'LWNDSRF',ALLOC=.true.,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,HLATN,  'HLATN'  ,ALLOC=.true.,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,QINFIL, 'QINFIL' ,ALLOC=.true.,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,AR1,    'AR1'    ,ALLOC=.true.,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,AR2,    'AR2'    ,ALLOC=.true.,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,RZEQ,   'RZEQ'   ,ALLOC=.true.,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,GHFLX,  'GHFLX'  ,ALLOC=.true.,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,TPSURF, 'TPSURF' ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,TPSN1,  'TPSNOW' ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,TPUST,  'TPUNST' ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,TPSAT,  'TPSAT'  ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,TPWLT,  'TPWLT'  ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,ASNOW,  'ASNOW'  ,ALLOC=.true.,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,SHSNOW, 'SHSNOW' ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,AVETSNOW,'AVETSNOW',           RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,FRSAT,  'FRSAT'  ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,FRUST,  'FRUST'  ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,FRWLT,  'FRWLT'  ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,TP1,    'TP1'    ,ALLOC=.true.,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,TP2,    'TP2'    ,ALLOC=.true.,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,TP3,    'TP3'    ,ALLOC=.true.,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,TP4,    'TP4'    ,ALLOC=.true.,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,TP5,    'TP5'    ,ALLOC=.true.,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,TP6,    'TP6'    ,ALLOC=.true.,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,EMIS,   'EMIS'   ,ALLOC=.true.,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,ALBVR,  'ALBVR'  ,ALLOC=.true.,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,ALBVF,  'ALBVF'  ,ALLOC=.true.,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,ALBNR,  'ALBNR'  ,ALLOC=.true.,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,ALBNF,  'ALBNF'  ,ALLOC=.true.,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,DELTS,  'DELTS'  ,ALLOC=.true.,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,DELQS,  'DELQS'  ,ALLOC=.true.,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,TST  ,  'TST'    ,ALLOC=.true.,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,QST  ,  'QST'    ,ALLOC=.true.,RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,LST  ,  'LST'    ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,WET1 ,  'WET1'   ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,WET2 ,  'WET2'   ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,WET3 ,  'WET3'   ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,WCSF ,  'WCSF'   ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,WCRZ ,  'WCRZ'   ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,WCPR ,  'WCPR'   ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,ACCUM,  'ACCUM'  ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,SNOMAS,'SNOWMASS',             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,SNOWDP, 'SNOWDP' ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,EVLAND, 'EVLAND' ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,PRLAND, 'PRLAND' ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,SNOLAND, 'SNOLAND' ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,DRPARLAND, 'DRPARLAND' ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,DFPARLAND, 'DFPARLAND' ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,LHSNOW, 'LHSNOW' ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,SWNETSNOW1, 'SWNETSNOW' ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,LWUPSNOW, 'LWUPSNOW' ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,LWDNSNOW, 'LWDNSNOW' ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,TCSORIG, 'TCSORIG' ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,TPSN1IN, 'TPSN1IN' ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,TPSN1OUT, 'TPSN1OUT' ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,LHLAND, 'LHLAND' ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,SHLAND, 'SHLAND' ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,SWLAND, 'SWLAND' ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,SWDOWNLAND, 'SWDOWNLAND' ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,LWLAND, 'LWLAND' ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,GHLAND, 'GHLAND' ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,GHSNOW, 'GHSNOW' ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,GHTSKIN,'GHTSKIN',             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,SMLAND, 'SMLAND' ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,TWLAND, 'TWLAND' ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,TELAND, 'TELAND' ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,TSLAND, 'TSLAND' ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,DWLAND, 'DWLAND' ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,DHLAND, 'DHLAND' ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,SPLAND, 'SPLAND' ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,SPWATR, 'SPWATR' ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,SPSNOW, 'SPSNOW' ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,CNLAI,  'CNLAI'  ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,CNTLAI, 'CNTLAI' ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,CNSAI,  'CNSAI'  ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,CNTOTC, 'CNTOTC' ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,CNVEGC, 'CNVEGC' ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,CNROOT, 'CNROOT' ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,CNNPP,  'CNNPP'  ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,CNGPP,  'CNGPP'  ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,CNSR,   'CNSR'   ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,CNNEE,  'CNNEE'  ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,CNXSMR, 'CNXSMR' ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,CNADD,  'CNADD'  ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,CNLOSS, 'CNLOSS' ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,CNBURN, 'CNBURN' ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,PARABS, 'PARABS' ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,PARINC, 'PARINC' ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,SCSAT,  'SCSAT'  ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,SCUNS,  'SCUNS'  ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,BTRANT, 'BTRANT' ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,SIF,    'SIF'    ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,CNFSEL, 'CNFSEL' ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,CNCO2,  'CNCO2'  ,             RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,CNSAI11, 'CNSAI11' ,           RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,CNSAI12, 'CNSAI12' ,           RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,CNSAI13, 'CNSAI13' ,           RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,CNSAI21, 'CNSAI21' ,           RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,CNSAI22, 'CNSAI22' ,           RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,CNSAI23, 'CNSAI23' ,           RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,CNSAI31, 'CNSAI31' ,           RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,CNSAI32, 'CNSAI32' ,           RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,CNSAI33, 'CNSAI33' ,           RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,CNSAI41, 'CNSAI41' ,           RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,CNSAI42, 'CNSAI42' ,           RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,CNSAI43, 'CNSAI43' ,           RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,CNLAI11, 'CNLAI11' ,           RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,CNLAI12, 'CNLAI12' ,           RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,CNLAI13, 'CNLAI13' ,           RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,CNLAI21, 'CNLAI21' ,           RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,CNLAI22, 'CNLAI22' ,           RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,CNLAI23, 'CNLAI23' ,           RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,CNLAI31, 'CNLAI31' ,           RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,CNLAI32, 'CNLAI32' ,           RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,CNLAI33, 'CNLAI33' ,           RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,CNLAI41, 'CNLAI41' ,           RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,CNLAI42, 'CNLAI42' ,           RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,CNLAI43, 'CNLAI43' ,           RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,RMELTDU001,'RMELTDU001',  RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,RMELTDU002,'RMELTDU002',  RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,RMELTDU003,'RMELTDU003',  RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,RMELTDU004,'RMELTDU004',  RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,RMELTDU005,'RMELTDU005',  RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,RMELTBC001,'RMELTBC001',  RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,RMELTBC002,'RMELTBC002',  RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,RMELTOC001,'RMELTOC001',  RC=STATUS); VERIFY_(STATUS)
        call MAPL_GetPointer(EXPORT,RMELTOC002,'RMELTOC002',  RC=STATUS); VERIFY_(STATUS)
        IF (RUN_IRRIG /= 0) call MAPL_GetPointer(EXPORT,IRRIGRATE ,'IRRIGRATE' ,  RC=STATUS); VERIFY_(STATUS)

        NTILES = size(PS)

    allocate(   ityp(ntiles,nveg,nzone) )
    allocate(   fveg(ntiles,nveg,nzone) )
    allocate( wtzone(ntiles,nzone) )
    allocate(   elai(ntiles,nveg,nzone) )
    allocate(   esai(ntiles,nveg,nzone) )
    allocate(   tlai(ntiles,nveg,nzone) )

! set CLM CN PFT & fraction, set carbon zone weights
! --------------------------------------------------
    do nz = 1,nzone
      ityp(:,:,nz) = nint(ity(:,:))
      fveg(:,:,nz) = fvg(:,:)
      wtzone(:,nz) = CN_zone_weight(nz)
    end do

    IF(PRESCRIBE_DVG == 0) THEN

       ! obtain LAI from previous time step (from CN model)
       ! --------------------------------------------------
       call get_CN_LAI(ntiles,nveg,nzone,ityp,fveg,elai,esai=esai)

    else

       ! read from daily files
       ! ---------------------

       call  read_prescribed_LAI (INTERNAL, CLOCK, GC, NTILES, elai,esai)

    endif

    if(associated(CNSAI11)) CNSAI11 = esai(:,1,1)
    if(associated(CNSAI12)) CNSAI12 = esai(:,1,2)
    if(associated(CNSAI13)) CNSAI13 = esai(:,1,3)
    if(associated(CNSAI21)) CNSAI21 = esai(:,2,1)
    if(associated(CNSAI22)) CNSAI22 = esai(:,2,2)
    if(associated(CNSAI23)) CNSAI23 = esai(:,2,3)
    if(associated(CNSAI31)) CNSAI31 = esai(:,3,1)
    if(associated(CNSAI32)) CNSAI32 = esai(:,3,2)
    if(associated(CNSAI33)) CNSAI33 = esai(:,3,3)
    if(associated(CNSAI41)) CNSAI41 = esai(:,4,1)
    if(associated(CNSAI42)) CNSAI42 = esai(:,4,2)
    if(associated(CNSAI43)) CNSAI43 = esai(:,4,3)
    if(associated(CNLAI11)) CNLAI11 = elai(:,1,1)
    if(associated(CNLAI12)) CNLAI12 = elai(:,1,2)
    if(associated(CNLAI13)) CNLAI13 = elai(:,1,3)
    if(associated(CNLAI21)) CNLAI21 = elai(:,2,1)
    if(associated(CNLAI22)) CNLAI22 = elai(:,2,2)
    if(associated(CNLAI23)) CNLAI23 = elai(:,2,3)
    if(associated(CNLAI31)) CNLAI31 = elai(:,3,1)
    if(associated(CNLAI32)) CNLAI32 = elai(:,3,2)
    if(associated(CNLAI33)) CNLAI33 = elai(:,3,3)
    if(associated(CNLAI41)) CNLAI41 = elai(:,4,1)
    if(associated(CNLAI42)) CNLAI42 = elai(:,4,2)
    if(associated(CNLAI43)) CNLAI43 = elai(:,4,3)

! OPTIONAL IMPOSE MONTHLY MEAN DIURNAL CYCLE FROM NOAA CARBON TRACKER
! -------------------------------------------------------------------

    IF ((ATM_CO2 == 1).OR.(ATM_CO2 == 2)) THEN
       READ_CT_CO2: IF(first_ct) THEN

          ! Carbon Tracker grid tiles mapping

          allocate (CT_INDEX  (1:CT_grid_N_lon, 1:CT_grid_N_lat))
          do j = 1, CT_grid_N_lat
             do i = 1, CT_grid_N_lon
                CT_INDEX (i,j) = (j - 1) * CT_grid_N_lon + i
             end do
          end do

          allocate (ct2cat (1:  NTILES))
          allocate (ct_tid (1:  NTILES))
          
          ct_tid = -9999
          ct2cat = 0

          do N = 1, NTILES             
             I =  NINT ((CEILING (lons(n)*90./MAPL_PI)*2 + 180.) / CT_grid_dlon)
             J =  NINT ((CEILING (lats(n)*90./MAPL_PI)*2 +  90.) / CT_grid_dlat)
             CT2CAT  (N) = ct_index (i,j)
          end do

          N = count(ct2cat > 0)
          
          allocate (unq_mask(1:N ))
          allocate (loc_int (1:N ))
          
          loc_int = pack(ct2cat ,mask = (ct2cat > 0))
          call MAPL_Sort (loc_int)

          unq_mask = .true.
          
          do i = 2,N 
             unq_mask(i) = .not.(loc_int(i) == loc_int(i-1)) 
          end do
          
          NUNQ = count(unq_mask)  
          
          allocate (ThisIndex (1:NUNQ))
          ThisIndex =  pack(loc_int, mask = unq_mask )

          do i = 1, NUNQ             
             where (ct2cat ==  ThisIndex(i)) ct_tid = i                
          end do
          
          ! Reading Carbon Tracker CO2_MonthlyMean_DiurnalCycle

          call ESMF_ClockGet( CLOCK, startTime=MODELSTART, RC=STATUS ); VERIFY_(STATUS)
          call ESMF_TimeGet ( MODELSTART, YY = FIRST_YY,  rc=status )  ; VERIFY_(STATUS)
          CALL ESMF_VMGet(vm, MPICOMMUNICATOR=comm, rc=status); VERIFY_(status)
          call MPI_Info_create(info, STATUS); VERIFY_(status)
          call MPI_Info_set(info, "romio_cb_read", "automatic", STATUS); VERIFY_(status)

          STATUS = NF_OPEN ('CO2_MonthlyMean_DiurnalCycle.nc4', NF_NOWRITE, CTfile); VERIFY_(status) 

          allocate (CT_CO2V (1: NUNQ, 1:12, 1:8))
          allocate (CTCO2_TMP (1:CT_grid_N_lon, 1:CT_grid_N_lat, 1:12, 1:8))

          STATUS = NF_GET_VARA_REAL (CTfile, VarID(CTfile,'CO2'),  (/1,1,1,1/), &
               (/CT_grid_N_lon, CT_grid_N_lat, 12, 8/), CTCO2_TMP);VERIFY_(STATUS) 

          do N = 1, NUNQ
             I = MOD (ThisIndex(N), CT_grid_N_lon)
             IF(I == 0) I = CT_grid_N_lon
             J = (ThisIndex(N) -I) / CT_grid_N_lon + 1 

             CT_CO2V (N,:,:) = CTCO2_TMP (I,J,:,:)
             
          end do

          status = NF_CLOSE (CTFile); VERIFY_(status) 
          first_ct = .false.

          deallocate (CTCO2_TMP,ct2cat, unq_mask, loc_int, ct_index, ThisIndex)

       ENDIF READ_CT_CO2       
    ENDIF
    
 ! OPTIONAL FPAR SCALING
! ---------------------
 
    if  (SCALE_ALBFPAR >= 2) then 
       IF (ntiles > 0) THEN
          INTILALIZE_FPAR_PARAM : if(first_fpar) then
             
             ! Initialize FPAR MODIS scale parameters
             ! --------------------------------------
             
!             CALL ESMF_VMGet(vm, MPICOMMUNICATOR=comms, rc=status)
!             VERIFY_(status)
!             call MPI_Info_create(infos, STATUS)
!             call MPI_Info_set(infos, "romio_cb_read", "automatic", STATUS)
             
             STATUS = NF_OPEN ('FPAR_CDF_Params-M09.nc4', NF_NOWRITE, CDFfile)
             STATUS = NF_INQ_DIMID  (CDFfile, 'tile10D', k); VERIFY_(STATUS)
             STATUS = NF_INQ_DIMLEN (CDFfile, K, n_modis)  ; VERIFY_(STATUS)
             
             allocate (m_lons (1 : n_modis))
             allocate (m_lats (1 : n_modis))  
             
             STATUS = NF_GET_VARA_REAL (CDFfile, VarID(CDFfile,'lon'),  (/1/), (/n_modis/), m_lons);VERIFY_(STATUS)
             STATUS = NF_GET_VARA_REAL (CDFfile, VarID(CDFfile,'lat'),  (/1/), (/n_modis/), m_lats);VERIFY_(STATUS) 
             
             allocate (modis_index (1: 360/nint(TILEINT), 1: 180/nint(TILEINT)))
             modis_index = -9999    
             
             ! vector to grid 10x10 MODIS tiles
             
             do i = 1, n_modis
                
                k = NINT (((m_lons(i) + TILEINT/2.) + 180.) / TILEINT)
                n = NINT (((m_lats(i) + TILEINT/2.) +  90.) / TILEINT)
                modis_index (k, n) = i
                
             end do
             
             ! for each catchment-tile overlying MODIS 10x10 tile 
             
             allocate (modis2cat (1:  NTILES))
             allocate (modis_tid (1:  NTILES))
             
             modis_tid = -9999
             modis2cat = 0
             
             do i = 1, NTILES
                
                k =  NINT ((CEILING (lons(i)*90./MAPL_PI)*2 + 180.) / TILEINT)
                n =  NINT ((CEILING (lats(i)*90./MAPL_PI)*2 +  90.) / TILEINT)
                if(k <=   3) k =   3
                if(k >= 178) k = 178
                modis2cat (i) = modis_index (k,n)

             end do

             K = count(modis2cat > 0)
             
             allocate (unq_mask(1:K ))
             allocate (loc_int (1:K ))
             
             loc_int = pack(modis2cat ,mask = (modis2cat > 0))
             call MAPL_Sort (loc_int)
             unq_mask = .true.
             
             do i = 2,K 
                unq_mask(i) = .not.(loc_int(i) == loc_int(i-1)) 
             end do
             
             NUNQ = count(unq_mask)  
             
             allocate (ThisIndex (1:NUNQ))
             ThisIndex =  pack(loc_int, mask = unq_mask )
             
             allocate (Kappa (1: NUNQ, 1: NUMPFT, 1 : NOCTAD, 1 : 2))
             allocate (Lambda(1: NUNQ, 1: NUMPFT, 1 : NOCTAD, 1 : 2))
             allocate (Mu    (1: NUNQ, 1: NUMPFT, 1 : NOCTAD, 1 : 2))
             allocate (MnVal (1: NUNQ, 1: NUMPFT, 1 : NOCTAD))
             allocate (MxVal (1: NUNQ, 1: NUMPFT, 1 : NOCTAD))

             Kappa    = -9999.
             Lambda   = -9999.
             Mu       = -9999.

             do i = 1, NUNQ
                
                where (modis2cat ==  ThisIndex(i)) modis_tid = i
                
             end do
             
             do i = 1, NUNQ
                do K = 1,NOCTAD
                   do n = 1, NUMPFT
                      IF (ThisIndex(i) >= 1) THEN
                         STATUS = NF_GET_VARA_REAL(CDFFile, VARID(CDFFile,'Kappa' ),(/ThisIndex(i),N,K,1/), (/1,1,1,2/), tmparr);VERIFY_(STATUS)
                         Kappa (i,N,K,:) = tmparr (1,1,1,:) 
                         STATUS = NF_GET_VARA_REAL(CDFFile, VARID(CDFFile,'Lambda'),(/ThisIndex(i),N,K,1/), (/1,1,1,2/), tmparr);VERIFY_(STATUS)
                         Lambda(i,N,K,:) = tmparr (1,1,1,:) 
                         STATUS = NF_GET_VARA_REAL(CDFFile, VARID(CDFFile,'Mu'    ),(/ThisIndex(i),N,K,1/), (/1,1,1,2/), tmparr);VERIFY_(STATUS)
                         Mu    (i,N,K,:) = tmparr (1,1,1,:)               
                         STATUS = NF_GET_VARA_REAL(CDFFile, VARID(CDFFile,'MinVal'),(/ThisIndex(i),N,K/), (/1,1,1/), tmparr2);VERIFY_(STATUS)
                         MnVal(i,N,K)  = tmparr2 (1,1,1)               
                         STATUS = NF_GET_VARA_REAL(CDFFile, VARID(CDFFile,'MaxVal'),(/ThisIndex(i),N,K/), (/1,1,1/), tmparr2);VERIFY_(STATUS)
                         MxVal(i,N,K) = tmparr2 (1,1,1)             
                      ENDIF
                   end do
                end do
             end do
             status = NF_CLOSE (CDFFile)

             deallocate ( modis2cat, unq_mask, loc_int, modis_index, m_lons, m_lats)
             
             first_fpar = .false.

          endif INTILALIZE_FPAR_PARAM
       endif
    end if

        ! --------------------------------------------------------------------------
        ! ALLOCATE LOCAL POINTERS
        ! --------------------------------------------------------------------------

	allocate(GHTCNT (6,NTILES))
	allocate(WESNN  (3,NTILES))
	allocate(HTSNNN (3,NTILES))
	allocate(SNDZN  (3,NTILES))
	allocate(TILEZERO (NTILES))
	allocate(DZSF     (NTILES))
	allocate(SWNETFREE(NTILES))
	allocate(SWNETSNOW(NTILES))
	allocate(VEG1     (NTILES))
	allocate(VEG2     (NTILES))
	allocate(RCSAT    (NTILES))
	allocate(DRCSDT   (NTILES))
	allocate(DRCSDQ   (NTILES))
	allocate(RCUNS    (NTILES))
	allocate(DRCUDT   (NTILES))
	allocate(DRCUDQ   (NTILES))
	allocate(ZTH      (NTILES))  
	allocate(SLR      (NTILES))  
	allocate(RSL1     (NTILES)) 
	allocate(RSL2     (NTILES)) 
	allocate(SQSCAT   (NTILES))
	allocate(RDC      (NTILES))  
	allocate(RDC_TMP_1(NTILES))
	allocate(RDC_TMP_2(NTILES))
	allocate(UUU      (NTILES))
	allocate(RHO      (NTILES))
	allocate(ZVG      (NTILES))
	allocate(LAI0     (NTILES))
	allocate(GRN0     (NTILES))
	allocate(Z0       (NTILES))
	allocate(D0       (NTILES))
	allocate(SFMC     (NTILES))
	allocate(RZMC     (NTILES))
	allocate(PRMC     (NTILES))
	allocate(ENTOT    (NTILES))
	allocate(ghflxsno (NTILES))
	allocate(ghflxtskin(NTILES))
	allocate(WTOT     (NTILES))
	allocate(WCHANGE  (NTILES))
	allocate(ECHANGE  (NTILES))
	allocate(HSNACC   (NTILES))
	allocate(EVACC    (NTILES))
	allocate(SHACC    (NTILES))
	allocate(VSUVR    (NTILES))
	allocate(VSUVF    (NTILES))
	allocate(SNOVR    (NTILES))
	allocate(SNOVF    (NTILES))
	allocate(SNONR    (NTILES))
	allocate(SNONF    (NTILES))
	allocate(CAT_ID   (NTILES))
	allocate(ALWX     (NTILES))
	allocate(BLWX     (NTILES))
        allocate(SHSNOW1   (NTILES))
        allocate(AVETSNOW1 (NTILES))
        allocate(WAT10CM1  (NTILES))
        allocate(WATSOI1   (NTILES))
        allocate(ICESOI1   (NTILES))
        allocate(LHSNOW1   (NTILES))
        allocate(LWUPSNOW1 (NTILES))
        allocate(LWDNSNOW1 (NTILES))
        allocate(NETSWSNOW (NTILES))
        allocate(TCSORIG1  (NTILES))
        allocate(TPSN1IN1  (NTILES))
        allocate(TPSN1OUT1 (NTILES))
        allocate(LHACC     (NTILES))
        allocate(SUMEV     (NTILES))
	allocate(fveg1     (NTILES))
	allocate(fveg2     (NTILES))
        allocate(FICE1     (NTILES)) 
        allocate(SLDTOT    (NTILES)) 

        allocate(SHSBT    (NTILES,NUM_SUBTILES))
        allocate(DSHSBT   (NTILES,NUM_SUBTILES))
        allocate(EVSBT    (NTILES,NUM_SUBTILES))
        allocate(DEVSBT   (NTILES,NUM_SUBTILES))
        allocate(DEDTC    (NTILES,NUM_SUBTILES))
        allocate(DHSDQA   (NTILES,NUM_SUBTILES))
        allocate(CFT      (NTILES,NUM_SUBTILES))
        allocate(CFQ      (NTILES,NUM_SUBTILES))
        allocate(TCO      (NTILES,NUM_SUBTILES))
        allocate(QCO      (NTILES,NUM_SUBTILES))
        allocate(DQS      (NTILES,NUM_SUBTILES))
        allocate(QSAT     (NTILES,NUM_SUBTILES))
        allocate(RA       (NTILES,NUM_SUBTILES))
        allocate(RCONSTIT (NTILES,N_SNOW,N_constit))
        allocate(TOTDEPOS (NTILES,N_constit))
        allocate(RMELT    (NTILES,N_constit))
        allocate(ALWN     (NTILES,NUM_SUBTILES))
        allocate(BLWN     (NTILES,NUM_SUBTILES))
        allocate(TC1_0    (NTILES))
        allocate(TC2_0    (NTILES))
        allocate(TC4_0    (NTILES))
        allocate(QA1_0    (NTILES))
        allocate(QA2_0    (NTILES))
        allocate(QA4_0    (NTILES))
        allocate(PLSIN    (NTILES))

        call ESMF_VMGetCurrent ( VM, RC=STATUS )
        ! --------------------------------------------------------------------------
        ! Catchment Id and vegetation types used to index into tables
        ! --------------------------------------------------------------------------

        CAT_ID = nint(tile_id)

        where(ITY(:,1) > 0.)             ! gkw: account for "split" types
          VEG1 = map_cat(nint(ITY(:,1))) ! map  primary  CN PFT to catchment type
         elsewhere
          VEG1 = map_cat(nint(ITY(:,2))) ! map  primary  CN PFT to catchment type
        endwhere
        where(ITY(:,3) > 0.)
          VEG2 = map_cat(nint(ITY(:,3))) ! map secondary CN PFT to catchment type
         elsewhere
          VEG2 = map_cat(nint(ITY(:,4))) ! map secondary CN PFT to catchment type
        endwhere

        fveg1(:) = fvg(:,1) + fvg(:,2) ! sum veg fractions (primary)   gkw: NVEG specific
        fveg2(:) = fvg(:,3) + fvg(:,4) ! sum veg fractions (secondary) gkw: fveg1+fveg2=1

        allocate ( lai1(ntiles) )
        allocate ( lai2(ntiles) )
        allocate ( wght(ntiles) )

        lai1 = 0.
        wght = 0.
        do nz = 1,nzone
          do nv = 1,2
            lai1(:) = lai1(:) + max(elai(:,nv,nz),0.)*fveg(:,nv,nz)*wtzone(:,nz)
            wght(:) = wght(:) +                       fveg(:,nv,nz)*wtzone(:,nz)
          end do
        end do
        lai1 = lai1 / max(wght,1.e-8) ! LAI for primary vegetation type

        lai2 = 0.
        wght = 0.
        do nz = 1,nzone
          do nv = 3,4
            lai2(:) = lai2(:) + max(elai(:,nv,nz),0.)*fveg(:,nv,nz)*wtzone(:,nz)
            wght(:) = wght(:) +                       fveg(:,nv,nz)*wtzone(:,nz)
          end do
        end do
        lai2 = lai2 / max(wght,1.e-8) ! LAI for secondary vegetation type

! LAI seen by the land model
! --------------------------
        lai = fveg1*lai1 + fveg2*lai2 ! gkw: prognostic LAI on catch_internal_rst (overwrite VEGDYN import)

        ! --------------------------------------------------------------------------
        ! surface layer depth for soil moisture
        ! --------------------------------------------------------------------------
        
        DZSF(    :) = SURFLAY

        ! --------------------------------------------------------------------------
        ! build arrays from internal state
        ! --------------------------------------------------------------------------

        GHTCNT(1,:) = GHTCNT1
        GHTCNT(2,:) = GHTCNT2
        GHTCNT(3,:) = GHTCNT3
        GHTCNT(4,:) = GHTCNT4
        GHTCNT(5,:) = GHTCNT5
        GHTCNT(6,:) = GHTCNT6

        WESNN (1,:) = WESNN1
        WESNN (2,:) = WESNN2
        WESNN (3,:) = WESNN3

        HTSNNN(1,:) = HTSNNN1
        HTSNNN(2,:) = HTSNNN2
        HTSNNN(3,:) = HTSNNN3

        SNDZN (1,:) = SNDZN1
        SNDZN (2,:) = SNDZN2
        SNDZN (3,:) = SNDZN3

        debugzth = .false.

        ! --------------------------------------------------------------------------
        ! Get the current time. 
        ! --------------------------------------------------------------------------

        call ESMF_ClockGet( CLOCK, currTime=CURRENT_TIME, startTime=MODELSTART, TIMESTEP=DELT,  RC=STATUS )
        VERIFY_(STATUS)
        if (MAPL_AM_I_Root(VM).and.debugzth) then
         print *,' start time of clock '
         CALL ESMF_TimePrint ( MODELSTART, OPTIONS="string", RC=STATUS )
        endif

        ! --------------------------------------------------------------------------
        ! retrieve the zenith angle
        ! --------------------------------------------------------------------------

!! The next sequence is to make sure that the albedo here and in solar are in sync
!!
! Need to know when Solar was called last, so first get the solar alarm
        call ESMF_ClockGetAlarm ( CLOCK, alarmname="SOLAR_Alarm", ALARM=SOLALARM, RC=STATUS )
!        VERIFY_(STATUS)
      if(status==0) then 
! Get the interval of the solar alarm - first get it in seconds
        call ESMF_ConfigGetAttribute ( CF, DT_SOLAR, Label="SOLAR_DT:", DEFAULT=DT, RC=STATUS )
        VERIFY_(STATUS)
! Now make an ESMF interval from the increment in seconds
        CALL ESMF_TimeIntervalSet ( TINT, S=NINT(DT_SOLAR), RC=STATUS )
        VERIFY_(STATUS)
! Now print out the solar alarm interval
        if (MAPL_AM_I_Root(VM).and.debugzth) CALL ESMF_TimeIntervalPrint ( TINT, OPTIONS="string", RC=STATUS )
! Now find out if it is ringing now: if so, set "BEFORE" to last time it rang before now
         solalarmison = ESMF_AlarmIsRinging(SOLALARM,RC=STATUS)
         VERIFY_(STATUS)
         if (MAPL_AM_I_Root(VM).and.debugzth)print *,' logical for solar alarm ',solalarmison
!     if so, set "BEFORE" to last time it rang before now
        if(solalarmison) then
         if (MAPL_AM_I_Root(VM).and.debugzth)print *,' In catch, solar alarm is ringing '
         NOW = CURRENT_TIME
         BEFORE = NOW - TINT
! Now print out the last time solar alarm rang
         if (MAPL_AM_I_Root(VM).and.debugzth)CALL ESMF_TimePrint ( BEFORE, OPTIONS="string", RC=STATUS )
!     If alarm is not ringing now, find out when it rang last
        else
         if (MAPL_AM_I_Root(VM).and.debugzth)print *,' In catch, solar alarm is not ringing '
         call ESMF_AlarmGet ( SOLALARM, prevRingTime=BEFORE, RC=STATUS )
         VERIFY_(STATUS)
! PrevRingTime can lie: if alarm never went off yet it gives next alarm time, not prev.
         if(BEFORE > CURRENT_TIME) then
          BEFORE = BEFORE-TINT
          if (MAPL_AM_I_Root(VM).and.debugzth)print *,' In catch, solar alarm not ringing, prev time lied '
          if (MAPL_AM_I_Root(VM).and.debugzth)CALL ESMF_TimePrint ( BEFORE, OPTIONS="string", RC=STATUS )
         else
          if (MAPL_AM_I_Root(VM).and.debugzth)print *,' In catch, solar alarm not ringing, prev time okay '
          if (MAPL_AM_I_Root(VM).and.debugzth)CALL ESMF_TimePrint ( BEFORE, OPTIONS="string", RC=STATUS )
         endif
! Now print out the last time solar alarm rang
        endif
     else
        BEFORE = CURRENT_TIME
        TINT = DELT
     end if
! Get the zenith angle at the center of the time between the last solar call and the next one
        call MAPL_SunGetInsolation(LONS, LATS,      &
            ORBIT, ZTH, SLR, &
            INTV = TINT,     &
            currTime=BEFORE+DELT,  &
            RC=STATUS )
        VERIFY_(STATUS)

        ZTH = max(0.0,ZTH)
        ZVG  = fveg1*(Z2CH - SCALE4Z0*(Z2CH - MIN_VEG_HEIGHT)*exp(-LAI1)) + &
             fveg2*(Z2CH - SCALE4Z0*(Z2CH - MIN_VEG_HEIGHT)*exp(-LAI2)) 


        !  For now roughnesses and displacement heights
        !   are the same for all subtiles.
        !---------------------------------------------------

        Z0   = Z0_BY_ZVEG*ZVG
        IF (USE_ASCATZ0 == 1) WHERE (NDVI <= 0.2) Z0 = ASCATZ0        
        D0   = D0_BY_ZVEG*ZVG

        UUU = max(UU,MAPL_USMIN) * (log((ZVG-D0+Z0)/Z0) &
             / log((max(DZ-D0,10.)+Z0)/Z0))

        !--------------- GOSWIM IMPORTS FROM GOCART ---------------
        ! Initialization 
        RCONSTIT(:,:,:)  = 0.0
        TOTDEPOS(:,:) = 0.0
        RMELT(:,:)  = 0.0
        !------------------------------------------------------------------

        ! Zero the light-absorbing aerosol (LAA) deposition rates from  GOCART:

        select case (AEROSOL_DEPOSITION)
        case (0)
           DUDP(:,:)=0.
           DUSV(:,:)=0.
           DUWT(:,:)=0.
           DUSD(:,:)=0.
           BCDP(:,:)=0.
           BCSV(:,:)=0.
           BCWT(:,:)=0.
           BCSD(:,:)=0.
           OCDP(:,:)=0.
           OCSV(:,:)=0.
           OCWT(:,:)=0.
           OCSD(:,:)=0.
           
        case (2)
           DUDP(:,:)=0.
           DUSV(:,:)=0.
           DUWT(:,:)=0.
           DUSD(:,:)=0.
           
        case (3)
           BCDP(:,:)=0.
           BCSV(:,:)=0.
           BCWT(:,:)=0.
           BCSD(:,:)=0.
           
        case (4)
           OCDP(:,:)=0.
           OCSV(:,:)=0.
           OCWT(:,:)=0.
           OCSD(:,:)=0.
           
        end select

! Convert the dimentions for LAAs from GEOS_SurfGridComp.F90 to GEOS_LandIceGridComp.F90
! Note: Explanations of each variable
! TOTDEPOS(:,1): Combined dust deposition from size bin 1 (dry, conv-scav, ls-scav, sed)
! TOTDEPOS(:,2): Combined dust deposition from size bin 2 (dry, conv-scav, ls-scav, sed)
! TOTDEPOS(:,3): Combined dust deposition from size bin 3 (dry, conv-scav, ls-scav, sed)
! TOTDEPOS(:,4): Combined dust deposition from size bin 4 (dry, conv-scav, ls-scav, sed)
! TOTDEPOS(:,5): Combined dust deposition from size bin 5 (dry, conv-scav, ls-scav, sed)
! TOTDEPOS(:,6): Combined hydrophobic BC deposition from size bin 1 (dry, conv-scav, ls-scav, sed)
! TOTDEPOS(:,7): Combined hydrophilic BC deposition from size bin 2 (dry, conv-scav, ls-scav, sed)
! TOTDEPOS(:,8): Combined hydrophobic OC deposition from size bin 1 (dry, conv-scav, ls-scav, sed)
! TOTDEPOS(:,9): Combined hydrophilic OC deposition from size bin 2 (dry, conv-scav, ls-scav, sed)
!============================= Possible future applications ====================================
! TOTDEPOS(:,10): Combined sulfate deposition from size bin 3 (dry, conv-scav, ls-scav, sed)
! TOTDEPOS(:,11): Combined sea salt deposition from size bin 1 (dry, conv-scav, ls-scav, sed)
! TOTDEPOS(:,12): Combined sea salt deposition from size bin 2 (dry, conv-scav, ls-scav, sed)
! TOTDEPOS(:,13): Combined sea salt deposition from size bin 3 (dry, conv-scav, ls-scav, sed)
! TOTDEPOS(:,14): Combined sea salt deposition from size bin 4 (dry, conv-scav, ls-scav, sed)
! TOTDEPOS(:,15): Combined sea salt deposition from size bin 5 (dry, conv-scav, ls-scav, sed)

        TOTDEPOS(:,1) = DUDP(:,1) + DUSV(:,1) + DUWT(:,1) + DUSD(:,1)
        TOTDEPOS(:,2) = DUDP(:,2) + DUSV(:,2) + DUWT(:,2) + DUSD(:,2)
        TOTDEPOS(:,3) = DUDP(:,3) + DUSV(:,3) + DUWT(:,3) + DUSD(:,3)
        TOTDEPOS(:,4) = DUDP(:,4) + DUSV(:,4) + DUWT(:,4) + DUSD(:,4)
        TOTDEPOS(:,5) = DUDP(:,5) + DUSV(:,5) + DUWT(:,5) + DUSD(:,5)
        TOTDEPOS(:,6) = BCDP(:,1) + BCSV(:,1) + BCWT(:,1) + BCSD(:,1)
        TOTDEPOS(:,7) = BCDP(:,2) + BCSV(:,2) + BCWT(:,2) + BCSD(:,2)
        TOTDEPOS(:,8) = OCDP(:,1) + OCSV(:,1) + OCWT(:,1) + OCSD(:,1)
        TOTDEPOS(:,9) = OCDP(:,2) + OCSV(:,2) + OCWT(:,2) + OCSD(:,2)
!============================= Possible future applications ====================================
!        TOTDEPOS(:,10) = SUDP(:,1) + SUSV(:,1) + SUWT(:,1) + SUSD(:,1)
!        TOTDEPOS(:,11) = SSDP(:,1) + SSSV(:,1) + SSWT(:,1) + SSSD(:,1)
!        TOTDEPOS(:,12) = SSDP(:,2) + SSSV(:,2) + SSWT(:,2) + SSSD(:,2)
!        TOTDEPOS(:,13) = SSDP(:,3) + SSSV(:,3) + SSWT(:,3) + SSSD(:,3)
!        TOTDEPOS(:,14) = SSDP(:,4) + SSSV(:,4) + SSWT(:,4) + SSSD(:,4)
!        TOTDEPOS(:,15) = SSDP(:,5) + SSSV(:,5) + SSWT(:,5) + SSSD(:,5)

! --------------- GOSWIM PROGRNOSTICS ---------------------------

        if (N_CONST_LAND4SNWALB /= 0) then

           ! Conversion of the masses of the snow impurities
           ! Note: Explanations of each variable
           ! Number of snow layer is 15: N = 1-15
           ! RCONSTIT(NTILES,N,1): Dust mass from bin 1 in layer N
           ! RCONSTIT(NTILES,N,2): Dust mass from bin 2 in layer N
           ! RCONSTIT(NTILES,N,3): Dust mass from bin 3 in layer N
           ! RCONSTIT(NTILES,N,4): Dust mass from bin 4 in layer N
           ! RCONSTIT(NTILES,N,5): Dust mass from bin 5 in layer N
           ! RCONSTIT(NTILES,N,6): Hydrophobic BC mass from bin 1 in layer N
           ! RCONSTIT(NTILES,N,7): Hydrophilic BC mass from bin 2 in layer N
           ! RCONSTIT(NTILES,N,8): Hydrophobic OC mass from bin 1 in layer N
           ! RCONSTIT(NTILES,N,9): Hydrophilic OC mass from bin 2 in layer N
           !============================= Possible future applications ====================================
           ! RCONSTIT(NTILES,N,10): Sulfate mass from size bin 3 in layer N
           ! RCONSTIT(NTILES,N,11): Sea salt mass from size bin 1 in layer N
           ! RCONSTIT(NTILES,N,12): Sea salt mass from size bin 2 in layer N
           ! RCONSTIT(NTILES,N,13): Sea salt mass from size bin 3 in layer N
           ! RCONSTIT(NTILES,N,14): Sea salt mass from size bin 4 in layer N
           ! RCONSTIT(NTILES,N,15): Sea salt mass from size bin 5 in layer N
           
           RCONSTIT(:,:,1) = RDU001(:,:)
           RCONSTIT(:,:,2) = RDU002(:,:)
           RCONSTIT(:,:,3) = RDU003(:,:)
           RCONSTIT(:,:,4) = RDU004(:,:)
           RCONSTIT(:,:,5) = RDU005(:,:)
           RCONSTIT(:,:,6) = RBC001(:,:)
           RCONSTIT(:,:,7) = RBC002(:,:)
           RCONSTIT(:,:,8) = ROC001(:,:)
           RCONSTIT(:,:,9) = ROC002(:,:)

!============================= Possible future applications ====================================
!        RCONSTIT(:,:,10) = RSU003(:,:)
!        RCONSTIT(:,:,11) = RSS001(:,:)
!        RCONSTIT(:,:,12) = RSS002(:,:)
!        RCONSTIT(:,:,13) = RSS003(:,:)
!        RCONSTIT(:,:,14) = RSS004(:,:)
!        RCONSTIT(:,:,15) = RSS005(:,:)
        endif

        ! --------------------------------------------------------------------------
        ! Parameters that depend on vegetation type only                             gkw: these are not used in unified
        ! --------------------------------------------------------------------------

        RSL1   = VGRDRS(VEG1)/(ROOTL*VGROTD(VEG1)) 

        RSL2   = ROOTL*VGROCA(VEG1)
        RSL2   = (RSL2 - 3.0 - 2.*alog(RSL2/(1.-RSL2)))/(8.*MAPL_PI*ROOTL*VGROTD(VEG1))

        ! --------------------------------------------------------------------------
        ! Greenness and type dependent parameters
        ! --------------------------------------------------------------------------

        SQSCAT = fveg1*((VGTR11(VEG1)+VGRF11(VEG1)) * GRN  + (VGTR12(VEG1)+VGRF12(VEG1)) * (1.-GRN)) + & 
                 fveg2*((VGTR11(VEG2)+VGRF11(VEG2)) * GRN  + (VGTR12(VEG2)+VGRF12(VEG2)) * (1.-GRN))
        SQSCAT = sqrt(1.0 - SQSCAT)

        ! --------------------------------------------------------------------------
        ! LAI and type dependent parameters; RDC formulation now uses veg fractions gkw: 2013-11-25, see note from Randy
        ! --------------------------------------------------------------------------
        ! jkolassa Oct 2020: RDC formulation previously implemented in GEOSldas Catchment-CN
        ! RDC = max(VGRDA(VEG1),VGRDA(VEG2))*min(1.,lai/2.)
        
        ! jkolassa Oct 2020: updated RDC formulation to the one used in F. Zeng's science-validated, published Catchment-CN simulations
        rdc_tmp_1 = max( VGRDA(VEG1)*min( 1., LAI1/VGRDB(VEG1) ), 0.001)
        rdc_tmp_2 = max( VGRDA(VEG2)*min( 1., LAI2/VGRDB(VEG2) ), 0.001)
        RDC = max(rdc_tmp_1,rdc_tmp_2)*min(1.,lai/2.)
        RDC = max(RDC,0.001)

        RHO = PS/(MAPL_RGAS*(TA*(1+MAPL_VIREPS*QA)))

        DEDTC=0.0
        DHSDQA=0.0

        if(OFFLINE_MODE /=0) then
           do N=1,NUM_SUBTILES
              CFT   (:,N) = 1.0
              CFQ   (:,N) = 1.0
              SHSBT (:,N) = MAPL_CP*CH(:,N)*(TC(:,N)-TA)
              EVSBT (:,N) = CQ(:,N)*(QC(:,N)-QA)
              DSHSBT(:,N) = MAPL_CP*CH(:,N)
              DEVSBT(:,N) = CQ(:,N)
              BLWN(:,N) = EMIS*MAPL_STFBOL*TC(:,N)*TC(:,N)*TC(:,N)
              ALWN(:,N) = -3.0*BLWN(:,N)*TC(:,N)
              BLWN(:,N) =  4.0*BLWN(:,N)
           end do
           if(CHOOSEMOSFC==0 .and. incl_Louis_extra_derivs ==1) then
              do N=1,NUM_SUBTILES
                 DEVSBT(:,N)=CQ(:,N)+max(0.0,-DCQ(:,N)*MAPL_VIREPS*TC(:,N)*(QC(:,N)-QA))
                 DEDTC(:,N) =max(0.0,-DCQ(:,N)*(1.+MAPL_VIREPS*QC(:,N))*(QC(:,N)-QA))
                 DSHSBT(:,N)=MAPL_CP*(CH(:,N)+max(0.0,-DCH(:,N)*(1.+MAPL_VIREPS*QC(:,N))*(TC(:,N)-TA)))
                 DHSDQA(:,N)=max(0.0,-MAPL_CP*DCH(:,N)*MAPL_VIREPS*TC(:,N)*(TC(:,N)-TA))
              enddo
           endif
        else
           do N=1,NUM_SUBTILES
              CFT   (:,N) = (CH(:,N)/CTATM)
              CFQ   (:,N) = (CQ(:,N)/CQATM)
              SHSBT (:,N) = (SH  + DSH  *(TC(:,N)-THATM))*CFT(:,N)
              EVSBT (:,N) = (EVAP+ DEVAP*(QC(:,N)-QHATM))*CFQ(:,N)
              DSHSBT(:,N) =  DSH  *CFT(:,N)
              DEVSBT(:,N) =  DEVAP*CFQ(:,N)
              ALWN(:,N)=ALW
              BLWN(:,N)=BLW
           end do
        end if

        do N=1,NUM_SUBTILES
           DQS(:,N) = GEOS_DQSAT ( TC(:,N), PS, QSAT=QSAT(:,N), PASCALS=.true., RAMP=0.0 )
           QC (:,N) = min(max(QA(:),QSAT(:,N)),QC(:,N))
           QC (:,N) = max(min(QA(:),QSAT(:,N)),QC(:,N))
           RA (:,N) = RHO/CH(:,N)
        end do


        QC(:,FSNW) = QSAT(:,FSNW)

	! --------------------------------------------------------------------------
        ! get total solid precip
        ! --------------------------------------------------------------------------

        SLDTOT = SNO+ICE+FRZR

	! --------------------------------------------------------------------------
	! protect the forcing from unsavory values, as per practice in offline
	! driver
	! --------------------------------------------------------------------------

        ASSERT_(count(PLS<0.)==0)
        ASSERT_(count(PCU<0.)==0)
        ASSERT_(count(SLDTOT<0.)==0)

        LAI0  = max(0.0001     , LAI)
        GRN0  = max(0.0001     , GRN)		
        ZTH   = max(0.0001     , ZTH)

        TCO   = TC
        QCO   = QC

        ! --------------------------------------------------------------------------
        ! actual CATCHMENT call
        ! --------------------------------------------------------------------------

        TILEZERO = 0.0

        call MAPL_TimerOn  ( MAPL, "-CATCH" )


! ----------------------------------------------------------------------------------------

! gkw: start on main CN block

    allocate(   btran(ntiles) )
    allocate(     wgt(ntiles) )
    allocate(     bt1(ntiles) )
    allocate(     bt2(ntiles) )
    allocate(     bt4(ntiles) )
    allocate(     wpp(ntiles) )
    allocate(    fwet(ntiles) )
    allocate(     sm1(ntiles) )
    allocate(     sm2(ntiles) )
    allocate(     sm4(ntiles) )
    allocate(  btran1(ntiles) )
    allocate(  btran2(ntiles) )
    allocate(  btran3(ntiles) )
    allocate(     tcx(ntiles) )
    allocate(     qax(ntiles) )
    allocate(     rcx(ntiles) )
    allocate(   rcxdt(ntiles) )
    allocate(   rcxdq(ntiles) )
    allocate(     tx1(ntiles) )
    allocate(     tx2(ntiles) )
    allocate(     tx3(ntiles) )
    allocate(     qx1(ntiles) )
    allocate(     qx2(ntiles) )
    allocate(     qx3(ntiles) )
    allocate(    car1(ntiles) )
    allocate(    car2(ntiles) )
    allocate(    car4(ntiles) )
    allocate( parzone(ntiles,nveg) )
    allocate(    para(ntiles) )
    allocate(   parav(ntiles,nveg) )
    allocate (scaled_fpar  (NTILES,NVEG))
    allocate (unscaled_fpar(NTILES,NVEG))
    allocate (  totwat(ntiles) )
    if(.not. allocated(npp )) allocate(     npp(ntiles) )
    if(.not. allocated(gpp )) allocate(     gpp(ntiles) )
    if(.not. allocated(sr  )) allocate(      sr(ntiles) )
    if(.not. allocated(nee )) allocate(     nee(ntiles) )
    if(.not. allocated(padd)) allocate(    padd(ntiles) )
    if(.not. allocated(root)) allocate(    root(ntiles) )
    if(.not. allocated(vegc)) allocate(    vegc(ntiles) )
    if(.not. allocated(xsmr)) allocate(    xsmr(ntiles) )
    if(.not. allocated(burn)) allocate(    burn(ntiles) )
    if(.not. allocated(fsel)) allocate(    fsel(ntiles) )
    if(.not. allocated(closs))allocate(   closs(ntiles) )
    allocate(    dayl(ntiles) )
    allocate(dayl_fac(ntiles) )
    allocate(CO2V    (ntiles) )

    allocate(     tgw(ntiles,nzone) )
    allocate(     rzm(ntiles,nzone) )
    allocate(    rc00(ntiles,nzone) )
    allocate(    rcdt(ntiles,nzone) )
    allocate(    rcdq(ntiles,nzone) )
    allocate( totcolc(ntiles,nzone) )

    allocate( psnsunx(ntiles,nveg) )
    allocate( psnshax(ntiles,nveg) )
    allocate( sifsunx(ntiles,nveg) )
    allocate( sifshax(ntiles,nveg) )
    allocate( laisunx(ntiles,nveg) )
    allocate( laishax(ntiles,nveg) )
    allocate(    elaz(ntiles,nveg) )
    allocate(    esaz(ntiles,nveg) )
    allocate(    fvez(ntiles,nveg) )
    allocate(    ityz(ntiles,nveg) )

    allocate( psnsun(ntiles,nveg,nzone) )
    allocate( psnsha(ntiles,nveg,nzone) )
    allocate( laisun(ntiles,nveg,nzone) )
    allocate( laisha(ntiles,nveg,nzone) )

    allocate(      ht(N_gt) )
    allocate(      tp(N_gt) )
    allocate( soilice(N_gt) )

! get current date & time gkw: this is used to transfer CN restart vars & set declination
! -----------------------
    call ESMF_TimeGet  ( CURRENT_TIME, YY = AGCM_YY,       &
				       MM = AGCM_MM,       &
				       DD = AGCM_DD,       &
                                       H  = AGCM_HH,       &
                                       M  = AGCM_MI,       &
				       S  = AGCM_S ,       &  
				       dayOfYear = dofyr , &
				       rc=status )
    VERIFY_(STATUS)

    AGCM_S = AGCM_S + 60 * AGCM_MI + 3600 * AGCM_HH
    sofmin = AGCM_S

! get ending time; determine if this is last call before ending time
! ------------------------------------------------------------------
    call ESMF_ClockGet ( clock,  StopTime=StopTime ,rc=STATUS )
    VERIFY_(STATUS)

    NextTime = CURRENT_TIME + DELT

    if(ntiles > 0) then ! gkw: skip threads with no land tiles

! gkw: assign new vegetation types and fractions
! ----------------------------------------------
    cat_id = nint(tile_id) ! gkw: temporary for debugging

! compute daylength (and daylength factor)
! ----------------------------------------

    ! current daylight duration
    call MAPL_SunGetDaylightDuration(ORBIT,lats,dayl,currTime=CURRENT_TIME,RC=STATUS)
    VERIFY_(STATUS)
    ! maximum daylight duration (at solstice)
    call MAPL_SunGetDaylightDurationMax(ORBIT,lats,dayl_fac,currTime=CURRENT_TIME,RC=STATUS)
    VERIFY_(STATUS)
    ! dayl_fac is ratio current:maximum dayl squared (min 0.01 [gkw: from CLM4])
    dayl_fac = min(1.,max(0.01,(dayl/dayl_fac)**2))

! gkw: obtain catchment area fractions and soil moisture
! ------------------------------------------------------
call catch_calc_soil_moist( ntiles, veg1, dzsf, vgwmax, cdcr1, cdcr2, psis, bee, poros, wpwet,           &
                              ars1, ars2, ars3, ara1, ara2, ara3, ara4, arw1, arw2, arw3, arw4,              &
                              srfexc, rzexc, catdef, car1, car2, car4, sfmc, rzmc, prmc )
                              
! obtain saturated canopy resistance following Farquhar, CLM4 implementation    

! compute RC & PSN in each of the CN zones
! ----------------------------------------

! "btran" in the catchment zones; map into CN zones
! -------------------------------------------------
       sm1 = 1.0
       bt1 = 1.0          ! saturated area

! gkw: bt2 is unstressed region only (subtract saturated and wilting areas)
       do n = 1,ntiles
         if(car2(n) > 0.) then
           bt2(n)=(rzmc(n)/poros(n) - car1(n) - car4(n)*wpwet(n))/car2(n)
          else
           bt2(n)= rzmc(n)/poros(n)
         endif
         bt2(n) = max(bt2(n),wpwet(n))
         bt2(n) = min(bt2(n),1.)
       end do

       sm2 = bt2
       bt2 = bt2**(-bee)

       wpp = wpwet ** (-bee)    ! soil water potential at wilting point
       bt2 = (bt2-wpp)/(1.-wpp) ! unstressed area (using soil water potential; PSIS common to all terms)

!      sm4 = wpwet - 1.e-7      ! wilting point minus epsilon (to account for truncation)

       do n = 1,ntiles
         if(car4(n) > 0.) then
           sm4(n)=(rzmc(n)/poros(n) - car1(n) - car2(n)*sm2(n))/car4(n)
          else
           sm4(n)= wpwet(n)
         endif
         sm4(n) = max(sm4(n),1.e-3)
         sm4(n) = min(sm4(n),wpwet(n)-1.e-7)
       end do

       bt4 = 0.0          ! stressed area

       do n = 1,ntiles

         ax1 = car1(n)
	 ax2 = car2(n)
         ax4 = 1. - ax1 - ax2

	 cn1 = wtzone(n,1)
	 cn2 = wtzone(n,2)
	 cn3 = wtzone(n,3)

! CN zone 1
         if(ax1 .gt. cn1) then
           f1 = cn1 ; f2 = 0. ; f4 = 0.
          else
           if((ax1+ax2) .gt. cn1) then
             f1 = ax1 ; f2 = cn1-ax1 ; f4 = 0.
            else
             f1 = ax1 ; f2 = ax2 ; f4 = cn1-ax1-ax2
           endif
         endif

         btran1(n) = (f1*bt1(n)     + f2*bt2(n)     + f4*bt4(n)    )/cn1
	 tgw(n,1) =  (f1*tg(n,fsat) + f2*tg(n,ftrn) + f4*tg(n,fwlt))/cn1
	 tx1(n)   =  (f1*tc(n,fsat) + f2*tc(n,ftrn) + f4*tc(n,fwlt))/cn1
	 qx1(n)   =  (f1*qc(n,fsat) + f2*qc(n,ftrn) + f4*qc(n,fwlt))/cn1
         rzm(n,1) =  (f1*sm1(n)     + f2*sm2(n)     + f4*sm4(n)    )/cn1

! CN zone 2
         if(ax1 .gt. cn1) then
           cn12 = cn1 + cn2
           if(car1(n) .gt. cn12) then
             f1 = cn2 ; f2 = 0. ; f4 = 0.
            else
             if((ax1+ax2) .lt. cn12) then
               f1 = ax1-cn1 ; f2 = ax2 ; f4 = cn12-ax1-ax2
              else
               f1 = ax1-cn1 ; f2 = cn12-ax1 ; f4 = 0.
             endif
           endif
          else
           cn23 = cn2 + cn3
           if(ax4 .gt. cn23) then
             f1 = 0. ; f2 = 0. ; f4 = cn2
            else
             if(ax4 .lt. cn3) then
               f1 = 0. ; f2 = cn2 ; f4 = 0.
              else
               f1 = 0. ; f2 = cn23-ax4 ; f4 = ax4-cn3
             endif
           endif
         endif

         btran2(n) = (f1*bt1(n)     + f2*bt2(n)     + f4*bt4(n)    )/cn2
	 tgw(n,2) =  (f1*tg(n,fsat) + f2*tg(n,ftrn) + f4*tg(n,fwlt))/cn2
	 tx2(n)   =  (f1*tc(n,fsat) + f2*tc(n,ftrn) + f4*tc(n,fwlt))/cn2
	 qx2(n)   =  (f1*qc(n,fsat) + f2*qc(n,ftrn) + f4*qc(n,fwlt))/cn2
         rzm(n,2) =  (f1*sm1(n)     + f2*sm2(n)     + f4*sm4(n)    )/cn2

! CN zone 3
         if(ax4 .gt. cn3) then
           f1 = 0. ; f2 = 0. ; f4 = cn3
          else
           if((ax4+ax2) .gt. cn3) then
             f1 = 0. ; f2 = cn3-ax4 ; f4 = ax4
            else
             f1 = cn3-ax4-ax2 ; f2 = ax2 ; f4 = ax4
           endif
         endif

         btran3(n) = (f1*bt1(n)     + f2*bt2(n)     + f4*bt4(n)    )/cn3
	 tgw(n,3) =  (f1*tg(n,fsat) + f2*tg(n,ftrn) + f4*tg(n,fwlt))/cn3
	 tx3(n)   =  (f1*tc(n,fsat) + f2*tc(n,ftrn) + f4*tc(n,fwlt))/cn3
	 qx3(n)   =  (f1*qc(n,fsat) + f2*qc(n,ftrn) + f4*qc(n,fwlt))/cn3
         rzm(n,3) =  (f1*sm1(n)     + f2*sm2(n)     + f4*sm4(n)    )/cn3

       end do

! soil temperature and hydrologic state
! -------------------------------------
    DO N=1,ntiles

! soil temperatures
! -----------------
      zbar = -sqrt(1.e-20+catdef(n)/bf1(n))+bf2(n)
      HT(:)=GHTCNT(:,N)
      CALL GNDTMP_CN(poros(n),zbar,ht,frice,tp,soilice)

      ! At the CatchCNGridComp level, tp1, tp2, .., tp6 are export variables in units of Kelvin,
      ! - rreichle & borescan, 6 Nov 2020

      tp1(n) = tp(1) + Tzero  
      tp2(n) = tp(2) + Tzero
      tp3(n) = tp(3) + Tzero
      tp4(n) = tp(4) + Tzero
      tp5(n) = tp(5) + Tzero
      tp6(n) = tp(6) + Tzero

! total soil liquid water
! -----------------------
      totwat(n) = cdcr2(n) - catdef(n) + rzexc(n) + srfexc(n)
      totwat(n) = totwat(n)*(1. - frice)

! baseflow
! --------
      zbar = sqrt(1.e-20+catdef(n)/bf1(n))-bf2(n)
      bflow(n) = (1.-frice)*1000.* &
    	    cond(n)*exp(-(bf3(n)-ashift)-gnu(n)*zbar)/gnu(n)
      IF(catdef(n) >= cdcr1(n)) bflow(n) = 0.
      bflow(n) = min(cond(n),bflow(n))
    end do

! get CO2
! -------


    if(ATM_CO2 == 3) CO2 = GETCO2(AGCM_YY,dofyr)

    CO2V (:) = CO2
    
! use CO2SC from GOCART/CO2
! -------------------------
    
    IF (ATM_CO2 == 4) THEN
       
       where ((CO2SC >= 0.) .and. (CO2SC <= 1000.))
          CO2V = CO2SC * 1e-6
       end where
       
    endif

    IF(ATM_CO2 == 1) co2g = 1.    ! DO NOT SCALE USE CT CLIMATOLOGY

    CALC_CTCO2_SF: IF(ATM_CO2 == 2)  THEN

       ! Compute scale factor to scale CarbonTracker CO2 monthly mean diurnal cycle (3-hourly)
       CO2_YEAR = AGCM_YY
       IF(CO2_YEAR_IN > 0) CO2_YEAR = CO2_YEAR_IN

       ! update EEA global average CO2 and co2 scalar at the beginning of each year, fz, 26 Sep 2016
       ! -------------------------------------------------------------------------------------------

       IF (AGCM_YY /= CO2_YEAR) CO2_YEAR = CO2_YEAR + AGCM_YY - FIRST_YY
          
       if (CO2_YEAR  < byr_co2g) then
          co2g = co2g_byr
       elseif ((CO2_YEAR >= byr_co2g).AND.(CO2_YEAR <= myr_co2g)) then
          co2g = co2g_byr + dco2g_1 * (CO2_YEAR - byr_co2g)
       else
          co2g = co2g_myr + dco2g_2 * (CO2_YEAR - myr_co2g)
       endif
     
       co2g = co2g / CTco2g  ! = co2g/CTco2g, is used to scale CarbonTracker CO2 monthly mean diurnal cycle (3-hourly)
     
    ENDIF CALC_CTCO2_SF

    if(associated(BTRANT)) btrant = 0. 
    if(associated(SIF))    sif    = 0. 

! fraction of foliage that is wet  gkw 20140327
! -------------------------------
    do n = 1,ntiles
      if(lai(n) > 1.e-4) then
        fwet(n) = min(1.,max(0.,capac(n)/(0.2*lai(n))))
       else
        fwet(n) = 0.
      endif
    end do

    para(:) = 0. ! zero out absorbed PAR summing array
    parav(:, :) = 0. !
    scaled_fpar = 1.

    do nz = 1,nzone

      if(nz == 1) then
        btran = btran1
        tcx = tx1
        qax = qx1
      endif

      if(nz == 2) then
        btran = btran2
        tcx = tx2
        qax = qx2
      endif

      if(nz == 3) then
        btran = btran3
        tcx = tx3
        qax = qx3
      endif

      do nv = 1,nveg
        elaz(:,nv) = elai(:,nv,nz)
        esaz(:,nv) = esai(:,nv,nz)
        ityz(:,nv) = ityp(:,nv,nz)
        fvez(:,nv) = fveg(:,nv,nz)
      end do

      do n = 1,ntiles
        if(tp1(n) < (Tzero-0.01)) btran(n) = 0. ! no photosynthesis if ground fully frozen
      end do

      USE_CT_CO2: IF((ATM_CO2 == 1).OR.(ATM_CO2 == 2)) THEN

         IF(AGCM_DD < 16) THEN

            ! interpolate between AGCM_MM - 1 and  AGCM_MM

            M1 = AGCM_MM -1
            Y1 = AGCM_YY
            if(M1 == 0) then ; M1 = 12 ; Y1 = AGCM_YY -1 ; endif

            call ESMF_TimeSet(BEFORE, YY = Y1, MM = M1, DD = 16, &
                 H =  0, M =  0, S = 0, rc = STATUS) ; VERIFY_(STATUS)
            call ESMF_TimeSet(AFTER , YY = AGCM_YY, MM = AGCM_MM,   DD = 15, &
                 H = 23, M = 59, S = 59, rc = STATUS); VERIFY_(STATUS)
         
            call MAPL_Interp_Fac (CURRENT_TIME,BEFORE,AFTER,FAC,RC=STATUS ) ; VERIFY_(STATUS)
            ASSERT_(FAC >= 0.0)
            ASSERT_(FAC <= 1.0)        

            DO N = 1,NTILES
               CO2V (N) = (FAC * CT_CO2V (CT_TID (N),M1, AGCM_HH/3+1)  +  (1.0-FAC) * CT_CO2V (CT_TID (N),AGCM_MM, AGCM_HH/3+1)) * &
                 CO2G * 1.e-6 ! scale by EEA global average CO2 * convert from ppm
            END DO
         ELSE
            
            ! interpolate between AGCM_MM and  AGCM_MM + 1

            M1 = AGCM_MM +1
            Y1 = AGCM_YY
            if(M1 == 13) then ; M1 = 1 ; Y1 = AGCM_YY +1 ; endif

            call ESMF_TimeSet(BEFORE , YY = AGCM_YY, MM = AGCM_MM,   DD = 16, &
                 H =  0, M =  0, S =  0, rc = STATUS) ; VERIFY_(STATUS)           
            call ESMF_TimeSet(AFTER, YY = Y1, MM = M1, DD = 15, &
                 H = 23, M = 59, S = 59, rc = STATUS) ; VERIFY_(STATUS)
         
            call MAPL_Interp_Fac (CURRENT_TIME,BEFORE,AFTER,FAC,RC=STATUS ) ; VERIFY_(STATUS)
            ASSERT_(FAC >= 0.0)
            ASSERT_(FAC <= 1.0)        
            DO N = 1,NTILES
               CO2V (N) = (FAC * CT_CO2V (CT_TID (N),AGCM_MM, AGCM_HH/3+1)  +  (1.0-FAC) * CT_CO2V (CT_TID (N),M1 , AGCM_HH/3+1)) * &
                 CO2G * 1.e-6 ! scale by EEA global average CO2 * convert from ppm
            END DO
         ENDIF
          
      ENDIF USE_CT_CO2

      call compute_rc(ntiles,nveg,TCx,QAx,TA,PS,ZTH,DRPAR,DFPAR,            &
                      elaz,esaz,ityz,fvez,btran,fwet,                       &
                      RCx,RCxDT,RCxDQ,psnsunx,psnshax,laisunx,laishax,      &
                      dayl_fac,co2v,dtc,dea,parzone,sifsunx,sifshax)

      rc00(:,nz) = rcx(:)
      rcdt(:,nz) = rcxdt(:)
      rcdq(:,nz) = rcxdq(:)

      psnsun(:,:,nz) = psnsunx(:,:)
      psnsha(:,:,nz) = psnshax(:,:)
      laisun(:,:,nz) = laisunx(:,:)
      laisha(:,:,nz) = laishax(:,:)

      do nv = 1,nveg
         para(:)     = para(:) + parzone(:,nv)*wtzone(:,nz)*fvez(:,nv)
         parav(:,nv) = parav (:,nv) + parzone(:,nv)*wtzone(:,nz)
      end do
      if(associated(BTRANT)) btrant(:) = btrant(:) + btran(:)*wtzone(:,nz)
      if(associated(SIF)) then
        do nv = 1,nveg
          sif(:) = sif(:) + wtzone(:,nz)*fvez(:,nv)*(sifsunx(:,nv)*laisunx(:,nv) + sifshax(:,nv)*laishax(:,nv))
        end do
      endif

   end do

   do nv = 1,nveg
      unscaled_fpar (:,nv) = parav (:,nv)/ (DRPAR(:) + DFPAR(:) + 1.e-20)
   end do

   NTCurrent = CEILING (real (dofyr) / 8.)
    
   ! FPAR scaling to match MODIS CDF
   ! -------------------------------
   
    DO_FS1 : if  (SCALE_ALBFPAR >= 2) then

        IF (ntiles > 0) THEN
           
           NT_LOOP1 : do n = 1,NTILES

              NV_LOOP1 : do nv = 1,nveg

                 CLM4_fpar = parav (n,nv) / (DRPAR (n) + DFPAR (n) + 1.e-20)
                 K = -1

                 if(CLM4_fpar > 0.) then
                    
                    k = NINT(ITY(N,nv)) 
                    if(minval(Kappa  (modis_tid (n), k, NTCurrent, :)) < 0.) then
                       k = -1
                       if(nv  == 1) k = NINT(ITY(N,2)) 
                       if(nv  == 2) k = NINT(ITY(N,1))
                       if(nv  == 3) k = NINT(ITY(N,4)) 
                       if(nv  == 4) k = NINT(ITY(N,3))
                       if(minval(Kappa  (modis_tid (n), k, NTCurrent, :)) < 0.) k = -1
                       if((K == -1).and.(nv > 2)) then
                          if(minval(Kappa  (modis_tid (n), NINT(ITY(N,2)), NTCurrent, :)) > 0.) k = NINT(ITY(N,2))
                          if(minval(Kappa  (modis_tid (n), NINT(ITY(N,1)), NTCurrent, :)) > 0.) k = NINT(ITY(N,1))
                       endif
                    endif
                    
                 endif

                 if((K > 0).and.(CLM4_fpar > 0)) then
                    
                    ! Computing probability of CLM4 FPAR
                    
                    ThisK   = Kappa  (modis_tid (n), k, NTCurrent, 2)
                    ThisL   = Lambda (modis_tid (n), k, NTCurrent, 2)
                    ThisM   = Mu     (modis_tid (n), k, NTCurrent, 2)
                    ThisMin = MnVal  (modis_tid (n), k, NTCurrent)
                    ThisMax = MxVal  (modis_tid (n), k, NTCurrent)
                    
                    if (CLM4_fpar < ThisMin) CLM4_fpar = ThisMin
                    if (CLM4_fpar > ThisMax) CLM4_fpar = ThisMax
                    if((ThisL == 0.).or.(ThisM == 0.)) print *,thisK,ThisL, ThisM, CLM4_fpar, ThisMin, ThisMax
                    if((ThisL == 0.).or.(ThisM == 0.)) print *,n,k,NTCurrent,modis_tid (n)
                    CLM4_cdf = ThisK * betai (ThisL, ThisM, (CLM4_fpar - ThisMin)/ThisMax)
                    
                    ! Computing corresponding MODIS FPAR for the same probability
                    
                    ThisK   = Kappa  (modis_tid (n), k, NTCurrent, 1)
                    ThisL   = Lambda (modis_tid (n), k, NTCurrent, 1)
                    ThisM   = Mu     (modis_tid (n), k, NTCurrent, 1)
                    ThisMin = MnVal  (modis_tid (n), k, NTCurrent)
                    ThisMax = MxVal  (modis_tid (n), k, NTCurrent)
                    
                    scaled_fpar (n,nv) = cdf2fpar (CLM4_cdf, ThisK, ThisL, ThisM, ThisMin, ThisMax)                                 
                    if((scaled_fpar (n,nv) > 1.).or.(scaled_fpar (n,nv) < 0.)) then
                       print *, 'PROB 1', CLM4_cdf, ThisK, ThisL, ThisM, ThisMin, ThisMax, scaled_fpar (n,nv)
                    endif
                    
                    scaled_fpar (n,nv) = scaled_fpar (n,nv) / (CLM4_fpar + 1.e-20)
                    
                 endif
              end do NV_LOOP1
              
           end do NT_LOOP1
           
           para  (:) = 0. ! zero out absorbed PAR summing array
           parav = 0.

           if(associated(BTRANT)) btrant = 0. 
           if(associated(SIF))    sif    = 0. 
           
           do nz = 1,num_zon
              
              if(nz == 1) then
                 btran = btran1
                 tcx = tx1
                 qax = qx1
              endif
              
              if(nz == 2) then
                 btran = btran2
                 tcx = tx2
                 qax = qx2
              endif
              
              if(nz == 3) then
                 btran = btran3
                 tcx = tx3
                 qax = qx3
              endif
              
              do nv = 1,num_veg
                 elaz(:,nv) = elai(:,nv,nz)
                 esaz(:,nv) = esai(:,nv,nz)
                 ityz(:,nv) = ityp(:,nv,nz)
                 fvez(:,nv) = fveg(:,nv,nz)
              end do
              
              do n = 1,NTILES
                 if(tp1(n) < (Tzero-0.01)) btran(n) = 0. ! no photosynthesis if ground fully frozen
              end do

              call compute_rc(NTILES,nveg,TCx,QAx,                       &
                   TA, PS, ZTH,DRPAR,DFPAR,                              &
                   elaz,esaz,ityz,fvez,btran,fwet,                       &
                   RCx,RCxDT,RCxDQ,psnsunx,psnshax,laisunx,laishax,      &
                   dayl_fac,co2v,dtc,dea,parzone,sifsunx,sifshax,        &
                   fpar_sf = scaled_fpar )
              
              rc00(:,nz) = rcx(:)
              rcdt(:,nz) = rcxdt(:)
              rcdq(:,nz) = rcxdq(:)
              
              psnsun(:,:,nz) = psnsunx(:,:)
              psnsha(:,:,nz) = psnshax(:,:)
              laisun(:,:,nz) = laisunx(:,:)
              laisha(:,:,nz) = laishax(:,:)
  
              do nv = 1,nveg
                 para(:)     = para(:) + parzone(:,nv)*wtzone(:,nz)*fvez(:,nv)
                 parav(:,nv) = parav (:,nv) + parzone(:,nv)*wtzone(:,nz)
              end do

              if(associated(BTRANT)) btrant(:) = btrant(:) + btran(:)*wtzone(:,nz)
              if(associated(SIF)) then
                 do nv = 1,nveg
                    sif(:) = sif(:) + wtzone(:,nz)*fvez(:,nv)*(sifsunx(:,nv)*laisunx(:,nv) + sifshax(:,nv)*laishax(:,nv))
                 end do
              endif
              
           end do
           
        endif

     endif DO_FS1

     ! Below we are recycling the scaled_fpar array - from this point, it contains fpar scaled or otherwise
     ! ----------------------------------------------------------------------------------------------------

     do nv = 1,nveg
        scaled_fpar (:,nv) = parav (:,nv)/ (DRPAR(:) + DFPAR(:) + 1.e-20)
     end do

    if(associated(CNCO2)) CNCO2 = CO2V * 1e6
    deallocate (co2v)

    if(associated(PARABS)) parabs = para
    if(associated(PARINC)) parinc = drpar + dfpar

    ! --------------------------------------------------------------------------
    ! Update raditation exports
    ! --------------------------------------------------------------------------
    
    allocate ( ALBVR_tmp(ntiles) )
    allocate ( ALBNR_tmp(ntiles) )
    allocate ( ALBVF_tmp(ntiles) )
    allocate ( ALBNF_tmp(ntiles) )
    allocate ( SNOVR_tmp(ntiles) )
    allocate ( SNONR_tmp(ntiles) )
    allocate ( SNOVF_tmp(ntiles) )
    allocate ( SNONF_tmp(ntiles) )
    
    call    SIBALB(NTILES, VEG1,LAI1,GRN, ZTH,         & 
         BGALBVR, BGALBVF, BGALBNR, BGALBNF, & ! gkw: MODIS soil background albedo
         ALBVR, ALBNR, ALBVF, ALBNF, MODIS_SCALE=.TRUE.  )         ! instantaneous snow-free albedos on tiles

    if  ((SCALE_ALBFPAR == 1).OR.(SCALE_ALBFPAR == 3)) then 

       if(.not.allocated (MODISVISmean )) allocate (MODISVISmean  (1:NTILES))
       if(.not.allocated (MODISVISstd  )) allocate (MODISVISstd   (1:NTILES))
       if(.not.allocated (MODISNIRmean )) allocate (MODISNIRmean  (1:NTILES))
       if(.not.allocated (MODISNIRstd  )) allocate (MODISNIRstd   (1:NTILES))
       if(.not.allocated (MODELFPARmean)) allocate (MODELFPARmean (1:NTILES))
       if(.not.allocated (MODELFPARstd )) allocate (MODELFPARstd  (1:NTILES))

       if(ntiles > 0) then
                    
          call MAPL_GetResource(MAPL,VISMEANFILE  , label = 'VISMEAN_FILE:' , default = 'MODISVISmean.dat'  , RC=STATUS )  ; VERIFY_(STATUS)    
          call MAPL_GetResource(MAPL,VISSTDFILE   , label = 'VISSTD_FILE:'  , default = 'MODISVISstd.dat'   , RC=STATUS )  ; VERIFY_(STATUS)      
          call MAPL_GetResource(MAPL,NIRMEANFILE  , label = 'NIRMEAN_FILE:' , default = 'MODISNIRmean.dat'  , RC=STATUS )  ; VERIFY_(STATUS)     
          call MAPL_GetResource(MAPL,NIRSTDFILE   , label = 'NIRSTD_FILE:'  , default = 'MODISNIRstd.dat'   , RC=STATUS )  ; VERIFY_(STATUS) 

          call MAPL_GetResource(MAPL,FPARMEANFILE , label = 'MODELFPARMEAN_FILE:', default = 'MODELFPARmean.dat' , RC=STATUS )  ; VERIFY_(STATUS)     
          call MAPL_GetResource(MAPL,FPARSTDFILE  , label = 'MODELFPARSTD_FILE:' , default = 'MODELFPARstd.dat'  , RC=STATUS )  ; VERIFY_(STATUS)      
           
          call MAPL_ReadForcing(MAPL,'MODISVISmean' ,VISMEANFILE ,CURRENT_TIME,MODISVISmean  ,ON_TILES=.true.,RC=STATUS) ; VERIFY_(STATUS)
          call MAPL_ReadForcing(MAPL,'MODISVISstd'  ,VISSTDFILE  ,CURRENT_TIME,MODISVISstd   ,ON_TILES=.true.,RC=STATUS) ; VERIFY_(STATUS)
          call MAPL_ReadForcing(MAPL,'MODISNIRmean' ,NIRMEANFILE ,CURRENT_TIME,MODISNIRmean  ,ON_TILES=.true.,RC=STATUS) ; VERIFY_(STATUS)
          call MAPL_ReadForcing(MAPL,'MODISNIRstd'  ,NIRSTDFILE  ,CURRENT_TIME,MODISNIRstd   ,ON_TILES=.true.,RC=STATUS) ; VERIFY_(STATUS)
          call MAPL_ReadForcing(MAPL,'MODELFPARmean',FPARMEANFILE,CURRENT_TIME,MODELFPARmean ,ON_TILES=.true.,RC=STATUS) ; VERIFY_(STATUS)
          call MAPL_ReadForcing(MAPL,'MODELFPARstd' ,FPARSTDFILE ,CURRENT_TIME,MODELFPARstd  ,ON_TILES=.true.,RC=STATUS) ; VERIFY_(STATUS)

         do n = 1,NTILES
            ThisFPAR = (unscaled_fpar(n,1)*FVG(N,1) + unscaled_fpar(n,2)*FVG(N,2))/(FVG(N,1) + FVG(N,2) + 1.e-20)
            ZFPAR    = (ThisFPAR - MODELFPARmean (n)) / MODELFPARstd (n)
            ALBVF(n) = AMIN1 (1., AMAX1(0.001,ZFPAR * MODISVISstd(n) + MODISVISmean (n)))
            ALBNF(n) = AMIN1 (1., AMAX1(0.001,ZFPAR * MODISNIRstd(n) + MODISNIRmean (n)))                
          end do
       endif
    endif

    call STIEGLITZSNOW_CALC_TPSNOW(NTILES, HTSNNN(1,:), WESNN(1,:), TPSN1OUT1, FICE1)
    TPSN1OUT1 =  TPSN1OUT1 + Tzero
    
    call   SNOW_ALBEDO(NTILES,N_snow, N_CONST_LAND4SNWALB, VEG1, LAI1, ZTH,        &
         RHOFS,                                              &   
         SNWALB_VISMAX, SNWALB_NIRMAX, SLOPE,                & 
         WESNN, HTSNNN, SNDZN,                               &
         ALBVR, ALBNR, ALBVF, ALBNF,                         & ! instantaneous snow-free albedos on tiles
         SNOVR, SNONR, SNOVF, SNONF,                         & ! instantaneous snow albedos on tiles
         RCONSTIT, UUU, TPSN1OUT1, DRPAR, DFPAR)           
    
    call    SIBALB(NTILES, VEG2,LAI2,GRN, ZTH,         & 
         BGALBVR, BGALBVF, BGALBNR, BGALBNF, & ! gkw: MODIS soil background albedo
         ALBVR_tmp, ALBNR_tmp, ALBVF_tmp, ALBNF_tmp, MODIS_SCALE=.TRUE. ) ! instantaneous snow-free albedos on tiles
  
    if ((SCALE_ALBFPAR == 1).OR.(SCALE_ALBFPAR == 3)) then 
       if(ntiles > 0) then
          do n = 1,NTILES
             ThisFPAR  = (unscaled_fpar(n,3)*FVG(N,3) + unscaled_fpar(n,4)*FVG(N,4))/(FVG(N,3) + FVG(N,4) + 1.e-20)
             ZFPAR    = (ThisFPAR - MODELFPARmean (n)) / MODELFPARstd (n)
             ALBVF_tmp(n) = AMIN1 (1., AMAX1(0.001,ZFPAR * MODISVISstd(n) + MODISVISmean (n)))
             ALBNF_tmp(n) = AMIN1 (1., AMAX1(0.001,ZFPAR * MODISNIRstd(n) + MODISNIRmean (n)))          
          end do
       endif
    endif

    call   SNOW_ALBEDO(NTILES,N_snow, N_CONST_LAND4SNWALB, VEG2, LAI2, ZTH,        &
         RHOFS,                                              &   
         SNWALB_VISMAX, SNWALB_NIRMAX, SLOPE,                & 
         WESNN, HTSNNN, SNDZN,                               &
         ALBVR_tmp, ALBNR_tmp, ALBVF_tmp, ALBNF_tmp, & ! instantaneous snow-free albedos on tiles
         SNOVR_tmp, SNONR_tmp, SNOVF_tmp, SNONF_tmp, & ! instantaneous snow albedos on tiles
         RCONSTIT, UUU, TPSN1OUT1, DRPAR, DFPAR )  
    
    ALBVR(:) = ALBVR(:)*fveg1(:) + ALBVR_tmp(:)*fveg2(:)
    ALBNR(:) = ALBNR(:)*fveg1(:) + ALBNR_tmp(:)*fveg2(:)
    ALBVF(:) = ALBVF(:)*fveg1(:) + ALBVF_tmp(:)*fveg2(:)
    ALBNF(:) = ALBNF(:)*fveg1(:) + ALBNF_tmp(:)*fveg2(:)
    
    SNOVR(:) = SNOVR(:)*fveg1(:) + SNOVR_tmp(:)*fveg2(:)
    SNONR(:) = SNONR(:)*fveg1(:) + SNONR_tmp(:)*fveg2(:)
    SNOVF(:) = SNOVF(:)*fveg1(:) + SNOVF_tmp(:)*fveg2(:)
    SNONF(:) = SNONF(:)*fveg1(:) + SNONF_tmp(:)*fveg2(:)
    
    ! --------------------------------------------------------------------------
    ! albedo/swnet partitioning
    ! --------------------------------------------------------------------------
    
    VSUVR = DRPAR + DRUVR
    VSUVF = DFPAR + DFUVR
    
    if(associated(SWDOWNLAND)) SWDOWNLAND = DRPAR + DFPAR + DRUVR + DFUVR + DRNIR + DFNIR
    
    SWNETFREE = (1.-ALBVR)*VSUVR + (1.-ALBVF)*VSUVF + (1.-ALBNR)*DRNIR + (1.-ALBNF)*DFNIR 
    SWNETSNOW = (1.-SNOVR)*VSUVR + (1.-SNOVF)*VSUVF + (1.-SNONR)*DRNIR + (1.-SNONF)*DFNIR 

! set the number of days per year when crossing year boundary or on restart gkw: use GEOS5/MAPL value
! -------------------------------------------------------------------------
    if(AGCM_YY .ne. year_prev) then
      dpy = get_days_per_year(AGCM_YY) ! set the number of days for current year
      year_prev = AGCM_YY
    endif

    RUN_CLM : IF(PRESCRIBE_DVG == 0) THEN
       
       ! CN time step over 4 hours may fail; limit to 4 hours; verify that DTCN is a multiple of DT
       ! ------------------------------------------------------------------------------------------
       dtcn = min(dtcn,14400.)
       if(mod(dtcn,dt) /= 0) stop 'dtcn'
       
       ndt = get_step_size( nint(dtcn) ) ! gkw: get_step_size must be called here to set CN model time step
       
       ! sum over interval for CN
       ! ------------------------
       tgwm    = tgwm    + tgw
       tpm     = tpm     + tp1
       sfmcm   = sfmcm   + sfmc
       rzmm    = rzmm    + rzm
       bflowm  = bflowm  + bflow
       totwatm = totwatm + totwat
       tairm   = tairm   + TA
       psnsunm = psnsunm + psnsun*laisun
       psnsham = psnsham + psnsha*laisha
       do n = 1,N_snow
          sndzm(:) = sndzm(:) + sndzn(n,:)
       end do
       asnowm = asnowm + asnow
       cnsum   = cnsum   + 1.
       
       ! call CN model every DTCN seconds
       ! --------------------------------

       if(mod(AGCM_S,nint(dtcn)) == 0) then

          ! compute mean state over interval
          ! --------------------------------
          do nz = 1,nzone
             tgwm(:,nz) = tgwm(:,nz) / cnsum(:)
             rzmm(:,nz) = rzmm(:,nz) / cnsum(:)
             do nv = 1,nveg
                psnsunm(:,nv,nz) = psnsunm(:,nv,nz) / cnsum(:)
                psnsham(:,nv,nz) = psnsham(:,nv,nz) / cnsum(:)
             end do
          end do
          tpm     = tpm     / cnsum
          sfmcm   = sfmcm   / cnsum
          bflowm  = bflowm  / cnsum
          totwatm = totwatm / cnsum
          tairm   = tairm   / cnsum
          sndzm   = sndzm   / cnsum
          asnowm  = asnowm  / cnsum
          
          laisun = 1.
          laisha = 1.
          
          call CN_Driver(istep,ntiles,nveg,nzone,dayl, &
               tgwm,tpm,tp2,tp3,tp4,tp5,tp6, &
               sfmcm,rzmm,wpwet, &
               psis,bee,poros,vgwmax,bflowm,totwatm, &
               tairm,psnsunm,psnsham,laisun,laisha, &
               ityp,fveg,wtzone,sndzm,asnowm,ndep,elai,esai,tlai,totcolc,cat_id,cli_t2m, &
               npp,gpp,sr,nee,root,padd,vegc,xsmr,burn,fsel,closs,firefac)
          
          ! save scaled CN diagnostics
          ! --------------------------
          if(associated(CNLAI)) then
             cnlai(:) = 0.
             do nz = 1,nzone
                do nv = 1,nveg
                   cnlai(:) = cnlai(:) + elai(:,nv,nz)*fveg(:,nv,nz)*wtzone(:,nz)
                end do
             end do
             cnlai(:) = cnlai(:) * cnsum
          endif
          
          if(associated(CNTLAI)) then
             cntlai(:) = 0.
             do nz = 1,nzone
                do nv = 1,nveg
                   cntlai(:) = cntlai(:) + tlai(:,nv,nz)*fveg(:,nv,nz)*wtzone(:,nz)
                end do
             end do
             cntlai(:) = cntlai(:) * cnsum
          endif
          
          if(associated(CNSAI)) then
             cnsai(:) = 0.
             do nz = 1,nzone
                do nv = 1,nveg
                   cnsai(:) = cnsai(:) + esai(:,nv,nz)*fveg(:,nv,nz)*wtzone(:,nz)
                end do
             end do
             cnsai(:) = cnsai(:) * cnsum
          endif
          
          if(associated(CNTOTC)) then
             cntotc(:) = 0.
             do nz = 1,nzone
                cntotc(:) = cntotc(:) + 1.e-3*totcolc(:,nz)*wtzone(:,nz)
             end do
             cntotc(:) = cntotc(:) * cnsum
          endif
          
          ! reset summing arrays
          ! --------------------
          tgwm    = 0.
          tpm     = 0.
          sfmcm   = 0.
          rzmm    = 0.
          bflowm  = 0.
          totwatm = 0.
          tairm   = 0.
          psnsunm = 0.
          psnsham = 0.
          sndzm   = 0.
          asnowm  = 0.
          cnsum   = 0.
          
       else ! CN diags set to zero
          
          if(associated(CNLAI )) cnlai  = 0.
          if(associated(CNTLAI)) cntlai = 0.
          if(associated(CNSAI )) cnsai  = 0.
          if(associated(CNTOTC)) cntotc = 0.
          
       endif

       ! CN_Driver outputs at DTCN are saved and used to populate below exports
       !                        uniformly outside DTCN.  
       ! -----------------------------------------------------------------------

       if(associated(CNVEGC)) cnvegc = 1.e-3*vegc  ! * cnsum
       if(associated(CNROOT)) cnroot = 1.e-3*root  ! * cnsum
       if(associated(CNNPP )) cnnpp  = 1.e-3*npp   ! * cnsum
       if(associated(CNGPP )) cngpp  = 1.e-3*gpp   ! * cnsum
       if(associated(CNSR  )) cnsr   = 1.e-3*sr    ! * cnsum
       if(associated(CNNEE )) cnnee  = 1.e-3*nee   ! * cnsum
       if(associated(CNXSMR)) cnxsmr = 1.e-3*xsmr  ! * cnsum
       if(associated(CNADD )) cnadd  = 1.e-3*padd  ! * cnsum
       if(associated(CNLOSS)) cnloss = 1.e-3*closs ! * cnsum ! total fire C loss (kg/m2/s)
       if(associated(CNBURN)) cnburn = burn        ! * cnsum ! area fractional fire burn rate (s-1)
       if(associated(CNFSEL)) cnfsel = fsel        ! * cnsum ! fire season length (days)
       
       ! copy CN_restart vars to catch_internal_rst gkw: only do if stopping
       ! ------------------------------------------
       if(NextTime == StopTime) then
          
          call CN_exit(ntiles,nveg,nzone,ityp,fveg,cncol,var_col,cnpft,var_pft)     
          i = 1
          do iv = 1,VAR_PFT
             do nv = 1,NUM_VEG
                do nz = 1, NUM_ZON
                   do  n = 1,ntiles
                      ! to ensure unused array elements don't have crazy numbers in restart files.
                      if(fveg (n,nv,nz) == 0.) cnpft (n,i) = 0.
                   end do
                   i = i + 1
                end do
             end do
          end do
       endif

    endif RUN_CLM

! update LAI for primary & secondary vegetation types
! ---------------------------------------------------
    lai1 = 0.
    wght = 0.
    do nz = 1,nzone
      do nv = 1,2
        lai1(:) = lai1(:) + max(elai(:,nv,nz),0.)*fveg(:,nv,nz)*wtzone(:,nz)
        wght(:) = wght(:) +                       fveg(:,nv,nz)*wtzone(:,nz)
      end do
    end do
    lai1 = lai1 / max(wght,1.e-8) ! LAI for primary vegetation type

    lai2 = 0.
    wght = 0.
    do nz = 1,nzone
      do nv = 3,4
        lai2(:) = lai2(:) + max(elai(:,nv,nz),0.)*fveg(:,nv,nz)*wtzone(:,nz)
        wght(:) = wght(:) +                       fveg(:,nv,nz)*wtzone(:,nz)
      end do
    end do
    lai2 = lai2 / max(wght,1.e-8) ! LAI for secondary vegetation type

    lai = fveg1*lai1 + fveg2*lai2 ! gkw: prognostic LAI on catch_internal_rst (overwrite VEGDYN import)
    LAI0  = max(0.0001     , LAI)

! have stomatal resistance in the CN zones; map as conductance into catchment zones
! ---------------------------------------------------------------------------------
    do n = 1,ntiles

      ax1 = car1(n)
      ax2 = car2(n)
      ax4 = 1. - ax1 - ax2

      cn1 = wtzone(n,1)
      cn2 = wtzone(n,2)
      cn3 = wtzone(n,3)

! catchment: saturated area

      if(ax1 .lt. cn1) then
        f1 = ax1 ; f2 = 0. ; f3 = 0.
       else
        if(ax1 .lt. (cn1+cn2)) then
          f1 = cn1 ; f2 = ax1-cn1 ; f3 = 0.
         else
          f1 = cn1 ; f2 = cn2 ; f3 = ax1-cn1-cn2
        endif
      endif

      if(ax1 .gt. 0.) then
        rcsat(n) = ax1/(f1/rc00(n,1)+f2/rc00(n,2)+f3/rc00(n,3))
        rcxdt(n) = ax1/(f1/rcdt(n,1)+f2/rcdt(n,2)+f3/rcdt(n,3))
        rcxdq(n) = ax1/(f1/rcdq(n,1)+f2/rcdq(n,2)+f3/rcdq(n,3))
       else
        rcsat(n) = 1.e3
        rcxdt(n) = 1.e3
        rcxdq(n) = 1.e3
      endif

! compute deriviatives
      drcsdt(n) = (rcxdt(n) - rcsat(n)) / dtc
      drcsdq(n) = (rcxdq(n) - rcsat(n)) / (0.622*dea/PS(n))

! catchment: unstressed area

      if(ax1 .lt. cn1) then
        ar = ax1 + ax2
        if(ar .lt. cn1) then
          f1 = ax2 ; f2 = 0. ; f3 = 0.
         else
          if(ar .lt. (cn1+cn2)) then
            f1 = cn1-ax1 ; f2 = ar-cn1 ; f3 = 0.
           else
            f1 = cn1-ax1 ; f2 = cn2 ; f3 = ar-cn1-cn2
          endif
        endif
       else
        ar = ax2 + ax4
        if(ar .lt. cn3) then
          f1 = 0. ; f2 = 0. ; f3 = ax2
         else
          if(ax4 .gt. cn3) then
            f1 = 0. ; f2 = ax2 ; f3 = 0.
           else
            f1 = 0. ; f2 = ar-cn3 ; f3 = cn3-ax4
          endif
        endif
      endif

      if(ax2 .gt. 0.) then
        rcuns(n) = ax2/(f1/rc00(n,1)+f2/rc00(n,2)+f3/rc00(n,3))
        rcxdt(n) = ax2/(f1/rcdt(n,1)+f2/rcdt(n,2)+f3/rcdt(n,3))
        rcxdq(n) = ax2/(f1/rcdq(n,1)+f2/rcdq(n,2)+f3/rcdq(n,3))
       else
        rcuns(n) = 1.e3
        rcxdt(n) = 1.e3
        rcxdq(n) = 1.e3
      endif

! compute deriviatives
      drcudt(n) = (rcxdt(n) - rcuns(n)) / dtc
      drcudq(n) = (rcxdq(n) - rcuns(n)) / (0.622*dea/PS(n))

    end do

    if(associated(SCSAT )) scsat  = 1. / rcsat
    if(associated(SCUNS )) scuns  = 1. / rcuns

    endif ! end of check for zero tiles

! gkw: end of main CN block

    PLSIN = PLS

    ! --------------------------------------------------------------------------
    ! Call Irrigation Model 
    ! --------------------------------------------------------------------------
    
    IF ((RUN_IRRIG /= 0).AND.(ntiles >0))  THEN  
       
       CALL CATCH_CALC_SOIL_MOIST (                                     &
            NTILES,VEG1,dzsf,vgwmax,cdcr1,cdcr2,psis,bee,poros,wpwet,   &
            ars1,ars2,ars3,ara1,ara2,ara3,ara4,arw1,arw2,arw3,arw4,     &
            srfexc,rzexc,catdef, CAR1, CAR2, CAR4, sfmc, rzmc, prmc)
	    
       call irrigation_rate (IRRIG_METHOD,                                 & 
            NTILES, AGCM_HH, AGCM_MI, sofmin, lons, IRRIGFRAC, PADDYFRAC,  &
            CLMPT,CLMST, CLMPF, CLMSF, LAIMAX, LAIMIN, LAI0,               &
            POROS, WPWET, VGWMAX, RZMC, IRRIGRATE)
       
       PLSIN = PLS + IRRIGRATE
       
    ENDIF


! Andrea Molod (Oct 21, 2016):
 
        do N=1,NUM_SUBTILES
           DQS(:,N) = GEOS_DQSAT ( TC(:,N), PS, QSAT=QSAT(:,N),PASCALS=.true., RAMP=0.0 )
           QC (:,N) = min(max(QA(:),QSAT(:,N)),QC(:,N))
           QC (:,N) = max(min(QA(:),QSAT(:,N)),QC(:,N))
           RA (:,N) = RHO/CH(:,N)
        end do

#ifdef DBG_CNLSM_INPUTS
        call MAPL_Get(MAPL, LocStream=LOCSTREAM, RC=STATUS)
        VERIFY_(STATUS)
        call MAPL_LocStreamGet(LOCSTREAM, TILEGRID=TILEGRID, RC=STATUS)
        VERIFY_(STATUS)

        call MAPL_TileMaskGet(tilegrid,  mask, rc=status)
        VERIFY_(STATUS)

         if (UNIT_i == 0) then
           unit_i = GETFILE( "catchcn_inputs.data", form="unformatted", RC=STATUS )
           VERIFY_(STATUS)
        endif
        unit = unit_i

! Inputs

        call MAPL_VarWrite(unit, tilegrid, PCU, mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, PLS, mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, SNO, mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, ICE,  mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, FRZR, mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, UUU, mask=mask, rc=status); VERIFY_(STATUS)

        call MAPL_VarWrite(unit, tilegrid, EVSBT (:,FSAT), mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, DEVSBT(:,FSAT), mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, DEDTC (:,FSAT), mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, SHSBT (:,FSAT), mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, DHSDQA(:,FSAT), mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, DSHSBT(:,FSAT), mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, EVSBT (:,FTRN), mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, DEVSBT(:,FTRN), mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, DEDTC (:,FTRN), mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, SHSBT (:,FTRN), mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, DHSDQA(:,FTRN), mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, DSHSBT(:,FTRN), mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, EVSBT (:,FWLT), mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, DEVSBT(:,FWLT), mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, DEDTC (:,FWLT), mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, SHSBT (:,FWLT), mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, DHSDQA(:,FWLT), mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, DSHSBT(:,FWLT), mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, EVSBT (:,FSNW), mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, DEVSBT(:,FSNW), mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, DEDTC (:,FSNW), mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, SHSBT (:,FSNW), mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, DHSDQA(:,FSNW), mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, DSHSBT(:,FSNW), mask=mask, rc=status); VERIFY_(STATUS)

        call MAPL_VarWrite(unit, tilegrid, TA, mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, QA, mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, RA(:,FSAT),  mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, RA(:,FTRN),  mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, RA(:,FWLT),  mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, RA(:,FSNW), mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, ZTH,  mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, SWNETFREE,  mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, SWNETSNOW,  mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, LWDNSRF, mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, PS*.01, mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, LAI0,  mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, GRN0,  mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, SQSCAT,  mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, RSL1,  mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, RSL2,  mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, RDC, mask=mask, rc=status); VERIFY_(STATUS)

        call MAPL_VarWrite(unit, tilegrid, QSAT(:,FSAT), mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, DQS(:,FSAT) , mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, ALWN(:,1)   , mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, BLWN(:,1)   , mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, QSAT(:,FTRN), mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, DQS(:,FTRN) , mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, ALWN(:,2)   , mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, BLWN(:,2)   , mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, QSAT(:,FWLT), mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, DQS(:,FWLT) , mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, ALWN(:,3)   , mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, BLWN(:,3)   , mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, QSAT(:,FSNW), mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, DQS(:,FSNW) , mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, ALWN(:,4)   , mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, BLWN(:,4)   , mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, RCSAT       , mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, DRCSDT      , mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, DRCSDQ      , mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, RCUNS       , mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, DRCUDT      , mask=mask, rc=status); VERIFY_(STATUS)
        call MAPL_VarWrite(unit, tilegrid, DRCUDQ      , mask=mask, rc=status); VERIFY_(STATUS)

! params
        if (firsttime) then
            firsttime = .false.
           unit = GETFILE( "catchcn_params.data", form="unformatted", RC=STATUS )
           VERIFY_(STATUS)

           NT_GLOBAL = size(mask)

           call WRITE_PARALLEL(NT_GLOBAL, UNIT)
           call WRITE_PARALLEL(DT, UNIT)
           call WRITE_PARALLEL(USE_FWET_FOR_RUNOFF, UNIT)
           call MAPL_VarWrite(unit, tilegrid, LONS,  mask=mask, rc=status); VERIFY_(STATUS)
           call MAPL_VarWrite(unit, tilegrid, LATS,  mask=mask, rc=status); VERIFY_(STATUS)
           call MAPL_VarWrite(unit, tilegrid, VEG1,  mask=mask, rc=status); VERIFY_(STATUS)
           call MAPL_VarWrite(unit, tilegrid, VEG2,  mask=mask, rc=status); VERIFY_(STATUS)
           call MAPL_VarWrite(unit, tilegrid, FVEG1, mask=mask, rc=status); VERIFY_(STATUS)
           call MAPL_VarWrite(unit, tilegrid, FVEG2, mask=mask, rc=status); VERIFY_(STATUS)
           call MAPL_VarWrite(unit, tilegrid, DZSF,  mask=mask, rc=status); VERIFY_(STATUS)
           call MAPL_VarWrite(unit, tilegrid, BF1,   mask=mask, rc=status); VERIFY_(STATUS)
           call MAPL_VarWrite(unit, tilegrid, BF2,   mask=mask, rc=status); VERIFY_(STATUS)
           call MAPL_VarWrite(unit, tilegrid, BF3,   mask=mask, rc=status); VERIFY_(STATUS)
           call MAPL_VarWrite(unit, tilegrid, VGWMAX,mask=mask, rc=status); VERIFY_(STATUS)
           call MAPL_VarWrite(unit, tilegrid, CDCR1, mask=mask, rc=status); VERIFY_(STATUS)
           call MAPL_VarWrite(unit, tilegrid, CDCR2, mask=mask, rc=status); VERIFY_(STATUS)
           call MAPL_VarWrite(unit, tilegrid, PSIS,  mask=mask, rc=status); VERIFY_(STATUS)
           call MAPL_VarWrite(unit, tilegrid, BEE,   mask=mask, rc=status); VERIFY_(STATUS)
           call MAPL_VarWrite(unit, tilegrid, POROS, mask=mask, rc=status); VERIFY_(STATUS)
           call MAPL_VarWrite(unit, tilegrid, WPWET, mask=mask, rc=status); VERIFY_(STATUS)
           call MAPL_VarWrite(unit, tilegrid, COND,  mask=mask, rc=status); VERIFY_(STATUS)
           call MAPL_VarWrite(unit, tilegrid, GNU,   mask=mask, rc=status); VERIFY_(STATUS)
           call MAPL_VarWrite(unit, tilegrid, ARS1,  mask=mask, rc=status); VERIFY_(STATUS)
           call MAPL_VarWrite(unit, tilegrid, ARS2,  mask=mask, rc=status); VERIFY_(STATUS)
           call MAPL_VarWrite(unit, tilegrid, ARS3,  mask=mask, rc=status); VERIFY_(STATUS)
           call MAPL_VarWrite(unit, tilegrid, ARA1,  mask=mask, rc=status); VERIFY_(STATUS)
           call MAPL_VarWrite(unit, tilegrid, ARA2,  mask=mask, rc=status); VERIFY_(STATUS)
           call MAPL_VarWrite(unit, tilegrid, ARA3,  mask=mask, rc=status); VERIFY_(STATUS)
           call MAPL_VarWrite(unit, tilegrid, ARA4,  mask=mask, rc=status); VERIFY_(STATUS)
           call MAPL_VarWrite(unit, tilegrid, ARW1,  mask=mask, rc=status); VERIFY_(STATUS)
           call MAPL_VarWrite(unit, tilegrid, ARW2,  mask=mask, rc=status); VERIFY_(STATUS)
           call MAPL_VarWrite(unit, tilegrid, ARW3,  mask=mask, rc=status); VERIFY_(STATUS)
           call MAPL_VarWrite(unit, tilegrid, ARW4,  mask=mask, rc=status); VERIFY_(STATUS)
           call MAPL_VarWrite(unit, tilegrid, TSA1,  mask=mask, rc=status); VERIFY_(STATUS)
           call MAPL_VarWrite(unit, tilegrid, TSA2,  mask=mask, rc=status); VERIFY_(STATUS)
           call MAPL_VarWrite(unit, tilegrid, TSB1,  mask=mask, rc=status); VERIFY_(STATUS)
           call MAPL_VarWrite(unit, tilegrid, TSB2,  mask=mask, rc=status); VERIFY_(STATUS)
           call MAPL_VarWrite(unit, tilegrid, ATAU,  mask=mask, rc=status); VERIFY_(STATUS)
           call MAPL_VarWrite(unit, tilegrid, BTAU,  mask=mask, rc=status); VERIFY_(STATUS)

           call FREE_FILE(unit, RC=STATUS)
           VERIFY_(STATUS)

! Updates
           unit = GETFILE( "catchcn_updates.data", form="unformatted", RC=STATUS )
           VERIFY_(STATUS)

           call MAPL_VarWrite(unit, tilegrid, TG(:,FSAT), mask=mask, rc=status); VERIFY_(STATUS)
           call MAPL_VarWrite(unit, tilegrid, TG(:,FTRN), mask=mask, rc=status); VERIFY_(STATUS)
           call MAPL_VarWrite(unit, tilegrid, TG(:,FWLT), mask=mask, rc=status); VERIFY_(STATUS)
           call MAPL_VarWrite(unit, tilegrid, TC(:,FSAT), mask=mask, rc=status); VERIFY_(STATUS)
           call MAPL_VarWrite(unit, tilegrid, TC(:,FTRN), mask=mask, rc=status); VERIFY_(STATUS)
           call MAPL_VarWrite(unit, tilegrid, TC(:,FWLT), mask=mask, rc=status); VERIFY_(STATUS)
           call MAPL_VarWrite(unit, tilegrid, QC(:,FSAT), mask=mask, rc=status); VERIFY_(STATUS)
           call MAPL_VarWrite(unit, tilegrid, QC(:,FTRN), mask=mask, rc=status); VERIFY_(STATUS)
           call MAPL_VarWrite(unit, tilegrid, QC(:,FWLT), mask=mask, rc=status); VERIFY_(STATUS)
           call MAPL_VarWrite(unit, tilegrid, CAPAC,      mask=mask, rc=status); VERIFY_(STATUS)
           call MAPL_VarWrite(unit, tilegrid, CATDEF,     mask=mask, rc=status); VERIFY_(STATUS)
           call MAPL_VarWrite(unit, tilegrid, RZEXC,      mask=mask, rc=status); VERIFY_(STATUS)
           call MAPL_VarWrite(unit, tilegrid, SRFEXC,     mask=mask, rc=status); VERIFY_(STATUS)
           call MAPL_VarWrite(unit, tilegrid, GHTCNT(1,:),mask=mask, rc=status); VERIFY_(STATUS)
           call MAPL_VarWrite(unit, tilegrid, GHTCNT(2,:),mask=mask, rc=status); VERIFY_(STATUS)
           call MAPL_VarWrite(unit, tilegrid, GHTCNT(3,:),mask=mask, rc=status); VERIFY_(STATUS)
           call MAPL_VarWrite(unit, tilegrid, GHTCNT(4,:),mask=mask, rc=status); VERIFY_(STATUS)
           call MAPL_VarWrite(unit, tilegrid, GHTCNT(5,:),mask=mask, rc=status); VERIFY_(STATUS)
           call MAPL_VarWrite(unit, tilegrid, GHTCNT(6,:),mask=mask, rc=status); VERIFY_(STATUS)
           call MAPL_VarWrite(unit, tilegrid, WESNN(1,:), mask=mask, rc=status); VERIFY_(STATUS)
           call MAPL_VarWrite(unit, tilegrid, WESNN(2,:), mask=mask, rc=status); VERIFY_(STATUS)
           call MAPL_VarWrite(unit, tilegrid, WESNN(3,:), mask=mask, rc=status); VERIFY_(STATUS)
           call MAPL_VarWrite(unit, tilegrid, HTSNNN(1,:),mask=mask, rc=status); VERIFY_(STATUS)
           call MAPL_VarWrite(unit, tilegrid, HTSNNN(2,:),mask=mask, rc=status); VERIFY_(STATUS)
           call MAPL_VarWrite(unit, tilegrid, HTSNNN(3,:),mask=mask, rc=status); VERIFY_(STATUS)
           call MAPL_VarWrite(unit, tilegrid, SNDZN(1,:), mask=mask, rc=status); VERIFY_(STATUS)
           call MAPL_VarWrite(unit, tilegrid, SNDZN(2,:), mask=mask, rc=status); VERIFY_(STATUS)
           call MAPL_VarWrite(unit, tilegrid, SNDZN(3,:), mask=mask, rc=status); VERIFY_(STATUS)
           
           call FREE_FILE(unit, RC=STATUS)
           VERIFY_(STATUS)

        end if
        deallocate(mask)
#endif

! call unified land model
! -----------------------

        if (ntiles > 0) then

           call CATCHCN ( NTILES, LONS, LATS, DT,USE_FWET_FOR_RUNOFF, &
                FWETC, FWETL, cat_id, VEG1,VEG2,FVEG1,FVEG2,DZSF     ,&
                PCU      ,     PLSIN ,     SNO, ICE, FRZR            ,&
                UUU                                                  ,&

                EVSBT(:,FSAT),     DEVSBT(:,FSAT),     DEDTC(:,FSAT) ,&
                SHSBT(:,FSAT),     DHSDQA(:,FSAT),     DSHSBT(:,FSAT),&
                EVSBT(:,FTRN),     DEVSBT(:,FTRN),     DEDTC(:,FTRN) ,&
                SHSBT(:,FTRN),     DHSDQA(:,FTRN),     DSHSBT(:,FTRN),&
                EVSBT(:,FWLT),     DEVSBT(:,FWLT),     DEDTC(:,FWLT) ,&
                SHSBT(:,FWLT),     DHSDQA(:,FWLT),     DSHSBT(:,FWLT),&
                EVSBT(:,FSNW),     DEVSBT(:,FSNW),     DEDTC(:,FSNW) ,&
                SHSBT(:,FSNW),     DHSDQA(:,FSNW),     DSHSBT(:,FSNW),&

                TA           ,     QA                                ,&

                RA(:,FSAT), RA(:,FTRN), RA(:,FWLT), RA(:,FSNW)       ,&

                ZTH,  SWNETFREE, SWNETSNOW, LWDNSRF                  ,&

                PS*.01                                               ,&

                LAI0, GRN0,       SQSCAT, RSL1, RSL2, RDC            ,&

                QSAT(:,FSAT) ,    DQS(:,FSAT) ,   ALWN(:,1),    BLWN(:,1) ,&
                QSAT(:,FTRN) ,    DQS(:,FTRN) ,   ALWN(:,2),    BLWN(:,2) ,&
                QSAT(:,FWLT) ,    DQS(:,FWLT) ,   ALWN(:,3),    BLWN(:,3) ,&
                QSAT(:,FSNW) ,    DQS(:,FSNW) ,   ALWN(:,4),    BLWN(:,4) ,&

           RCSAT,DRCSDT,DRCSDQ,  RCUNS,DRCUDT,DRCUDQ,              &
                BF1, BF2, BF3, VGWMAX, CDCR1, CDCR2, PSIS	          ,&
                BEE, POROS, WPWET, COND, GNU	                  ,&
                ARS1, ARS2, ARS3, ARA1, ARA2, ARA3, ARA4	          ,&
                ARW1, ARW2, ARW3, ARW4, TSA1, TSA2, TSB1, TSB2	  ,&
                ATAU, BTAU, .false.			          ,&

                TG(:,FSAT), TG(:,FTRN), TG(:,FWLT)		          ,& 
                TC(:,FSAT), TC(:,FTRN), TC(:,FWLT)		          ,& 
                QC(:,FSAT), QC(:,FTRN), QC(:,FWLT)		          ,&

                CAPAC, CATDEF, RZEXC, SRFEXC, GHTCNT                 ,&
                WESNN, HTSNNN, SNDZN                                 ,&

                EVAPOUT, SHOUT, RUNOFF, EVPINT, EVPSOI, EVPVEG       ,&
                EVPICE                                               ,&
                BFLOW                                                ,&
                RUNSURF                                              ,&
                SMELT                                                ,&
                HLWUP                                                ,&
                SWNDSRF                                              ,&
                HLATN                                                ,&
                QINFIL                                               ,&
                AR1                                                  ,&
                AR2                                                  ,&
                RZEQ                                                 ,&
                GHFLX                                                ,&
                GHFLXSNO                                             ,&
                GHFLXTSKIN                                           ,&
                TC(:,FSNW)                                           ,&
                ASNOW                                                ,&
                TP1, TP2, TP3, TP4, TP5, TP6,  SFMC, RZMC, PRMC      ,&
                ENTOT,WTOT, WCHANGE, ECHANGE, HSNACC, EVACC, SHACC   ,&
                TSURF                                                ,&
                SHSNOW1, AVETSNOW1, WAT10CM1, WATSOI1, ICESOI1       ,&
                LHSNOW1, LWUPSNOW1, LWDNSNOW1, NETSWSNOW             ,&
                TCSORIG1, TPSN1IN1, TPSN1OUT1                        ,&
                TC1_0=TC1_0, TC2_0=TC2_0, TC4_0=TC4_0                ,&
                QA1_0=QA1_0, QA2_0=QA2_0, QA4_0=QA4_0                ,&
                RCONSTIT=RCONSTIT, RMELT=RMELT, TOTDEPOS=TOTDEPOS, LHACC=LHACC)
           
           
           ! Change units of TP1, TP2, .., TP6 export variables from Celsius to Kelvin.
           ! This used to be done at the level the Surface GridComp.
           ! With this change, gridded TSOIL[n] exports from Surface and tile-space TP[n] exports
           ! from Catch are now consistently in units of Kelvin.
           ! - rreichle, borescan, 6 Nov 2020
           
           TP1 = TP1 + MAPL_TICE
           TP2 = TP2 + MAPL_TICE
           TP3 = TP3 + MAPL_TICE
           TP4 = TP4 + MAPL_TICE
           TP5 = TP5 + MAPL_TICE
           TP6 = TP6 + MAPL_TICE
            
        end if

        if (OFFLINE_MODE /=0) then
           TC(:,FSAT) = TC1_0
           TC(:,FTRN) = TC2_0
           TC(:,FWLT) = TC4_0
           QC(:,FSAT) = QA1_0
           QC(:,FTRN) = QA2_0
           QC(:,FWLT) = QA4_0
           EVACC = 0.0
           SHACC = 0.0
        endif

        QC(:,FSNW) =  GEOS_QSAT ( TC(:,FSNW), PS, PASCALS=.true., RAMP=0.0 )

        ! --------------------------------------------------------------------------
        ! update subtile fractions
        ! --------------------------------------------------------------------------

        EMIS    = fveg1*(EMSVEG(VEG1) + (EMSBARESOIL - EMSVEG(VEG1))*exp(-LAI1)) + &
                  fveg2*(EMSVEG(VEG2) + (EMSBARESOIL - EMSVEG(VEG2))*exp(-LAI2))

        EMIS    = EMIS     *(1.-ASNOW) + EMSSNO   *ASNOW

        call MAPL_SunGetInsolation(LONS, LATS,      &
            ORBIT, ZTH, SLR, &
            INTV = TINT,     &
            currTime=CURRENT_TIME+DELT,  &
            RC=STATUS )
        VERIFY_(STATUS)

        ZTH = max(0.0,ZTH)

        ! --------------------------------------------------------------------------
        ! Update raditation exports
        ! --------------------------------------------------------------------------


        call MAPL_TimerOn(MAPL,"-ALBEDO")
        call    SIBALB(NTILES, VEG1,LAI1,GRN, ZTH,         & 
                       BGALBVR, BGALBVF, BGALBNR, BGALBNF, & ! gkw: MODIS soil background albedo
                       ALBVR, ALBNR, ALBVF, ALBNF, MODIS_SCALE=.TRUE.  )         ! instantaneous snow-free albedos on tiles

        if ((SCALE_ALBFPAR == 1).OR.(SCALE_ALBFPAR == 3)) then
           do n = 1,NTILES
              ThisFPAR  = (unscaled_fpar(n,1)*FVG(N,1) + unscaled_fpar(n,2)*FVG(N,2))/(FVG(N,1) + FVG(N,2) + 1.e-20)
              ZFPAR     = (ThisFPAR - MODELFPARmean (n)) / MODELFPARstd (n)
              ALBVF(n)  = AMIN1 (1., AMAX1(0.001,ZFPAR * MODISVISstd(n) + MODISVISmean (n)))                      
           end do
        endif

        call STIEGLITZSNOW_CALC_TPSNOW(NTILES, HTSNNN(1,:), WESNN(1,:), TPSN1OUT1, FICE1)
        TPSN1OUT1 =  TPSN1OUT1 + Tzero        

        call   SNOW_ALBEDO(NTILES,N_snow, N_CONST_LAND4SNWALB, VEG1, LAI1, ZTH,        &
                 RHOFS,                                              &   
                 SNWALB_VISMAX, SNWALB_NIRMAX, SLOPE,                & 
                 WESNN, HTSNNN, SNDZN,                               &
                 ALBVR, ALBNR, ALBVF, ALBNF,                         & ! instantaneous snow-free albedos on tiles
                 SNOVR, SNONR, SNOVF, SNONF,                         & ! instantaneous snow albedos on tiles
                 RCONSTIT, UUU, TPSN1OUT1,DRPAR, DFPAR)

        call    SIBALB(NTILES, VEG2,LAI2,GRN, ZTH,         & 
                       BGALBVR, BGALBVF, BGALBNR, BGALBNF, & ! gkw: MODIS soil background albedo
                       ALBVR_tmp, ALBNR_tmp, ALBVF_tmp, ALBNF_tmp, MODIS_SCALE=.TRUE.  ) ! instantaneous snow-free albedos on tiles

        if ((SCALE_ALBFPAR == 1).OR.(SCALE_ALBFPAR == 3)) then
           if(ntiles > 0) then
              do n = 1,NTILES
                 ThisFPAR  = (unscaled_fpar(n,3)*FVG(N,3) + unscaled_fpar(n,4)*FVG(N,4))/(FVG(N,3) + FVG(N,4) + 1.e-20)
                 ZFPAR    = (ThisFPAR - MODELFPARmean (n)) / MODELFPARstd (n)
                 ALBVF_tmp(n) = AMIN1 (1., AMAX1(0.001,ZFPAR * MODISVISstd(n) + MODISVISmean (n)))
                 ALBNF_tmp(n) = AMIN1 (1., AMAX1(0.001,ZFPAR * MODISNIRstd(n) + MODISNIRmean (n)))          
              end do
           endif
           if(allocated (MODISVISmean)) deallocate (MODISVISmean, MODISVISstd, MODISNIRmean, MODISNIRstd, MODELFPARmean, MODELFPARstd)
        endif

        call   SNOW_ALBEDO(NTILES,N_snow, N_CONST_LAND4SNWALB, VEG2, LAI2, ZTH,        &
                 RHOFS,                                              &   
                 SNWALB_VISMAX, SNWALB_NIRMAX, SLOPE,                & 
                 WESNN, HTSNNN, SNDZN,                               &
                 ALBVR_tmp, ALBNR_tmp, ALBVF_tmp, ALBNF_tmp, & ! instantaneous snow-free albedos on tiles
                 SNOVR_tmp, SNONR_tmp, SNOVF_tmp, SNONF_tmp, & ! instantaneous snow albedos on tiles
                 RCONSTIT, UUU, TPSN1OUT1,DRPAR, DFPAR ) 

        ALBVR(:) = ALBVR(:)*fveg1(:) + ALBVR_tmp(:)*fveg2(:)
        ALBNR(:) = ALBNR(:)*fveg1(:) + ALBNR_tmp(:)*fveg2(:)
        ALBVF(:) = ALBVF(:)*fveg1(:) + ALBVF_tmp(:)*fveg2(:)
        ALBNF(:) = ALBNF(:)*fveg1(:) + ALBNF_tmp(:)*fveg2(:)

        SNOVR(:) = SNOVR(:)*fveg1(:) + SNOVR_tmp(:)*fveg2(:)
        SNONR(:) = SNONR(:)*fveg1(:) + SNONR_tmp(:)*fveg2(:)
        SNOVF(:) = SNOVF(:)*fveg1(:) + SNOVF_tmp(:)*fveg2(:)
        SNONF(:) = SNONF(:)*fveg1(:) + SNONF_tmp(:)*fveg2(:)

        ALBVR   = ALBVR    *(1.-ASNOW) + SNOVR    *ASNOW
        ALBVF   = ALBVF    *(1.-ASNOW) + SNOVF    *ASNOW
        ALBNR   = ALBNR    *(1.-ASNOW) + SNONR    *ASNOW
        ALBNF   = ALBNF    *(1.-ASNOW) + SNONF    *ASNOW
        call MAPL_TimerOff(MAPL,"-ALBEDO")

        LWNDSRF = LWDNSRF - HLWUP

        ! --------------------------------------------------------------------------
        ! update outputs
        ! --------------------------------------------------------------------------

        DELTS = 0.0
        DELQS = 0.0

        do N=1,NUM_SUBTILES
           DELTS   = DELTS + CFT(:,N)*(TC(:,N)-TCO(:,N))*FR(:,N)
           DELQS   = DELQS + CFQ(:,N)*(QC(:,N)-QCO(:,N))*FR(:,N)
        end do

        FR(:,FSAT) =           AR1  * (1-ASNOW)
        FR(:,FTRN) =           AR2  * (1-ASNOW)
        FR(:,FWLT) = (1.0-(AR1+AR2))* (1-ASNOW)
        FR(:,FSNW) =                     ASNOW

        FR = min( max( fr,0.0 ), 1.0 )

        TST   = 0.0
        QST   = 0.0
        do N=1,NUM_SUBTILES
           TST     = TST   +           TC(:,N)          *FR(:,N)
           QST     = QST   +           QC(:,N)          *FR(:,N)
        end do

        if ( OFFLINE_MODE ==0 ) then
!amm add correction term to latent heat diagnostics (HLATN is always allocated)
!    this will impact the export LHLAND
            HLATN = HLATN - LHACC
! also add some portion of the correction term to evap from soil, int, veg and snow
           SUMEV = EVPICE+EVPSOI+EVPVEG+EVPINT
           where(SUMEV>0.)
           EVPICE = EVPICE - EVACC*EVPICE/SUMEV
           EVPSOI = EVPSOI - EVACC*EVPSOI/SUMEV
           EVPINT = EVPINT - EVACC*EVPINT/SUMEV
           EVPVEG = EVPVEG - EVACC*EVPVEG/SUMEV
           endwhere
        endif

        if(associated( LST  )) LST    = TST
        if(associated( TPSURF))TPSURF = TSURF
        if(associated( WET1 )) WET1   = max(min(SFMC / POROS,1.0),0.0)
        if(associated( WET2 )) WET2   = max(min(RZMC / POROS,1.0),0.0)
        if(associated( WET3 )) WET3   = max(min(PRMC / POROS,1.0),0.0)
        if(associated( WCSF )) WCSF   = SFMC
        if(associated( WCRZ )) WCRZ   = RZMC
        if(associated( WCPR )) WCPR   = PRMC
        if(associated( ACCUM)) ACCUM  = SLDTOT - EVPICE*(1./MAPL_ALHS) - SMELT
        if(associated(PRLAND)) PRLAND = PCU+PLS+SLDTOT
        if(associated(SNOLAND)) SNOLAND = SLDTOT
        if(associated(EVPSNO)) EVPSNO = EVPICE
        if(associated(SUBLIM)) SUBLIM = EVPICE*(1./MAPL_ALHS)*FR(:,FSNW)
        if(associated(EVLAND)) EVLAND = EVAPOUT-EVACC
        if(associated(DRPARLAND)) DRPARLAND = DRPAR
        if(associated(DFPARLAND)) DFPARLAND = DFPAR
        if(associated(LHLAND)) LHLAND = HLATN
        if(associated(SHLAND)) SHLAND = SHOUT-SHACC
        if(associated(SWLAND)) SWLAND = SWNDSRF
        if(associated(LWLAND)) LWLAND = LWNDSRF
        if(associated(GHLAND)) GHLAND = GHFLX
        if(associated(GHSNOW)) GHSNOW = GHFLXSNO
        if(associated(SHSNOW)) SHSNOW = SHSNOW1                   
        if(associated(AVETSNOW)) AVETSNOW = AVETSNOW1             
        if(associated(WAT10CM)) WAT10CM = WAT10CM1                
        if(associated(WATSOI)) WATSOI = WATSOI1                   
        if(associated(ICESOI)) ICESOI = ICESOI1                   
        if(associated(LHSNOW)) LHSNOW = LHSNOW1                   
        if(associated(LWUPSNOW)) LWUPSNOW = LWUPSNOW1             
        if(associated(LWDNSNOW)) LWDNSNOW = LWDNSNOW1             
        if(associated(SWNETSNOW1)) SWNETSNOW1 = NETSWSNOW         
        if(associated(TCSORIG)) TCSORIG = TCSORIG1                
        if(associated(TPSN1IN)) TPSN1IN = TPSN1IN1                
        if(associated(TPSN1OUT)) TPSN1OUT = TPSN1OUT1
        if(associated(GHTSKIN))GHTSKIN = GHFLXTSKIN
        if(associated(SMLAND)) SMLAND = SMELT
        if(associated(TWLAND)) TWLAND = WTOT
        if(associated(TELAND)) TELAND = ENTOT
        if(associated(TSLAND)) TSLAND = WESNN (1,:) + WESNN (2,:) + WESNN (3,:)
        if(associated(DWLAND)) DWLAND = WCHANGE
        if(associated(DHLAND)) DHLAND = ECHANGE
        if(associated(SPLAND)) SPLAND = SHACC
        if(associated(SPWATR)) SPWATR = EVACC
        if(associated(SPSNOW)) SPSNOW = HSNACC

        if(associated(FRSAT )) FRSAT  = max( min( FR(:,FSAT),1.0 ), 0.0 )
        if(associated(FRUST )) FRUST  = max( min( FR(:,FTRN),1.0 ), 0.0 )
        if(associated(FRWLT )) FRWLT  = max( min( FR(:,FWLT),1.0 ), 0.0 )

        if(associated(SNOMAS)) SNOMAS = WESNN (1,:) + WESNN (2,:) + WESNN (3,:)
        if(associated(SNOWDP)) SNOWDP = SNDZN (1,:) + SNDZN (2,:) + SNDZN (3,:)

        if(associated(RMELTDU001)) RMELTDU001 = RMELT(:,1) 
        if(associated(RMELTDU002)) RMELTDU002 = RMELT(:,2) 
        if(associated(RMELTDU003)) RMELTDU003 = RMELT(:,3) 
        if(associated(RMELTDU004)) RMELTDU004 = RMELT(:,4) 
        if(associated(RMELTDU005)) RMELTDU005 = RMELT(:,5) 
        if(associated(RMELTBC001)) RMELTBC001 = RMELT(:,6) 
        if(associated(RMELTBC002)) RMELTBC002 = RMELT(:,7) 
        if(associated(RMELTOC001)) RMELTOC001 = RMELT(:,8) 
        if(associated(RMELTOC002)) RMELTOC002 = RMELT(:,9) 

        if(associated(TPSN1OUT)) then
           where(WESNN(1,:)>0.)
            TPSN1OUT = TPSN1OUT1                
           elsewhere
            TPSN1OUT = MAPL_UNDEF
           end where              
        end if

        if(associated(TPSN1)) then
           where(WESNN(1,:)>0.)
              TPSN1  = TC(:,FSNW)
           elsewhere
              TPSN1  = MAPL_UNDEF
           end where
        end if

        if(associated(TPSAT)) then
           where(FR(:,FSAT)>0.)
              TPSAT  = TC(:,FSAT)
           elsewhere
              TPSAT  = MAPL_UNDEF
           end where
        end if

        if(associated(TPWLT)) then
           where(FR(:,FWLT)>0.)
              TPWLT  = TC(:,FWLT)
           elsewhere
              TPWLT  = MAPL_UNDEF
           end where
        end if

        if(associated(TPUST)) then
           where(FR(:,FTRN)>0.)
              TPUST  = TC(:,FTRN)
           elsewhere
              TPUST  = MAPL_UNDEF
           end where
        end if


        ! --------------------------------------------------------------------------
        ! update internal state arrays
        ! --------------------------------------------------------------------------

        GHTCNT1 = GHTCNT(1,:)
        GHTCNT2 = GHTCNT(2,:)
        GHTCNT3 = GHTCNT(3,:)
        GHTCNT4 = GHTCNT(4,:)
        GHTCNT5 = GHTCNT(5,:)
        GHTCNT6 = GHTCNT(6,:)

        WESNN1  = WESNN (1,:)
        WESNN2  = WESNN (2,:)
        WESNN3  = WESNN (3,:)

        HTSNNN1 = HTSNNN(1,:)
        HTSNNN2 = HTSNNN(2,:)
        HTSNNN3 = HTSNNN(3,:)

        SNDZN1  = SNDZN (1,:)
        SNDZN2  = SNDZN (2,:)
        SNDZN3  = SNDZN (3,:)

        if (N_CONST_LAND4SNWALB /= 0) then
           RDU001(:,:) = RCONSTIT(:,:,1) 
           RDU002(:,:) = RCONSTIT(:,:,2) 
           RDU003(:,:) = RCONSTIT(:,:,3) 
           RDU004(:,:) = RCONSTIT(:,:,4) 
           RDU005(:,:) = RCONSTIT(:,:,5) 
           RBC001(:,:) = RCONSTIT(:,:,6) 
           RBC002(:,:) = RCONSTIT(:,:,7) 
           ROC001(:,:) = RCONSTIT(:,:,8) 
           ROC002(:,:) = RCONSTIT(:,:,9) 
        end if

        ! --------------------------------------------------------------------------

        deallocate ( wght )
        deallocate ( lai1 )
        deallocate ( lai2 )
        if (allocated (ALBVR_tmp)) deallocate ( ALBVR_tmp )
        if (allocated (ALBNR_tmp)) deallocate ( ALBNR_tmp )
        if (allocated (ALBVF_tmp)) deallocate ( ALBVF_tmp )
        if (allocated (ALBNF_tmp)) deallocate ( ALBNF_tmp )
        if (allocated (SNOVR_tmp)) deallocate ( SNOVR_tmp )
        if (allocated (SNONR_tmp)) deallocate ( SNONR_tmp )
        if (allocated (SNOVF_tmp)) deallocate ( SNOVF_tmp )
        if (allocated (SNONF_tmp)) deallocate ( SNONF_tmp )

	deallocate(GHTCNT   )
	deallocate(WESNN    )
	deallocate(HTSNNN   )
	deallocate(SNDZN    )
	deallocate(TILEZERO )
	deallocate(DZSF     )
	deallocate(SWNETFREE)
	deallocate(SWNETSNOW)
	deallocate(VEG1     )
	deallocate(VEG2     )
	deallocate(RCSAT    )
	deallocate(DRCSDT   )
	deallocate(DRCSDQ   )
	deallocate(RCUNS    )
	deallocate(DRCUDT   )
	deallocate(DRCUDQ   )
	deallocate(ZTH      )
	deallocate(SLR      )
	deallocate(RSL1     )
	deallocate(RSL2     )
	deallocate(SQSCAT   )
	deallocate(RDC      )
	deallocate(RDC_TMP_1)
	deallocate(RDC_TMP_2)
	deallocate(UUU      )
	deallocate(RHO      )
	deallocate(ZVG      )
	deallocate(LAI0     )
	deallocate(GRN0     )
	deallocate(Z0       )
	deallocate(D0       )
	deallocate(SFMC     )
	deallocate(RZMC     )
	deallocate(PRMC     )
	deallocate(ENTOT    )
	deallocate(WTOT     )
	deallocate(GHFLXSNO )
	deallocate(SHSNOW1  )
	deallocate(AVETSNOW1)
	deallocate(WAT10CM1 )
	deallocate(WATSOI1  )
	deallocate(ICESOI1  )
	deallocate(LHSNOW1  )
	deallocate(LWUPSNOW1)
	deallocate(LWDNSNOW1)
	deallocate(NETSWSNOW)
	deallocate(TCSORIG1 )
	deallocate(LHACC )
	deallocate(SUMEV )
	deallocate(TPSN1IN1 )
	deallocate(TPSN1OUT1)
	deallocate(GHFLXTSKIN)
	deallocate(WCHANGE  )
	deallocate(ECHANGE  )
	deallocate(HSNACC   )
	deallocate(EVACC    )
	deallocate(SHACC    )
	deallocate(VSUVR    )
	deallocate(VSUVF    )
	deallocate(SNOVR    )
	deallocate(SNOVF    )
	deallocate(SNONR    )
	deallocate(SNONF    )
	deallocate(SHSBT    )
	deallocate(DSHSBT   )
	deallocate(EVSBT    )
	deallocate(DEVSBT   )
	deallocate(DEDTC    )
	deallocate(DHSDQA   )
	deallocate(CFT      )
	deallocate(CFQ      )
	deallocate(TCO      )
	deallocate(QCO      )
	deallocate(DQS      )
	deallocate(QSAT     )
	deallocate(RA       )
	deallocate(CAT_ID   )
	deallocate(ALWX     )
	deallocate(BLWX     )
	deallocate(ALWN     )
	deallocate(BLWN     )
	deallocate(TC1_0    )
	deallocate(TC2_0    )
	deallocate(TC4_0    )
	deallocate(QA1_0    )
	deallocate(QA2_0    )
	deallocate(QA4_0    )
	deallocate(fveg1    )
	deallocate(fveg2    )
	deallocate(RCONSTIT )
	deallocate(TOTDEPOS )
	deallocate(RMELT    )
	deallocate(FICE1    )
	deallocate(SLDTOT )
	deallocate(   btran )
	deallocate(     wgt )
	deallocate(     bt1 )
	deallocate(     bt2 )
	deallocate(     bt4 )
	deallocate(     wpp )
	deallocate(    fwet )
	deallocate(     sm1 )
	deallocate(     sm2 )
	deallocate(     sm4 )
	deallocate(  btran1 )
	deallocate(  btran2 )
	deallocate(  btran3 )
	deallocate(     tcx )
	deallocate(     qax )
	deallocate(     rcx )
	deallocate(   rcxdt )
	deallocate(   rcxdq )
	deallocate(     tx1 )
	deallocate(     tx2 )
	deallocate(     tx3 )
	deallocate(     qx1 )
	deallocate(     qx2 )
	deallocate(     qx3 )
	deallocate(    car1 )
	deallocate(    car2 )
	deallocate(    car4 )
	deallocate( parzone )
	deallocate(    para )
	deallocate(   parav )
	deallocate(scaled_fpar)
	deallocate(UNscaled_fpar)
	deallocate(  totwat )
	deallocate(    dayl )
	deallocate(dayl_fac )
	deallocate(     tgw )
	deallocate(     rzm )
	deallocate(    rc00 )
	deallocate(    rcdt )
	deallocate(    rcdq )
	deallocate( totcolc )
	deallocate(  wtzone )

	deallocate( psnsunx )
	deallocate( psnshax )
	deallocate( sifsunx )
	deallocate( sifshax )
	deallocate( laisunx )
	deallocate( laishax )
	deallocate(    elaz )
	deallocate(    esaz )
	deallocate(    fvez )
	deallocate(    ityz )

	deallocate(   elai )
	deallocate(   esai )
	deallocate(   fveg )
	deallocate(   tlai )
	deallocate( psnsun )
	deallocate( psnsha )
	deallocate( laisun )
	deallocate( laisha )
	deallocate(   ityp )

	deallocate(      ht )
	deallocate(      tp )
	deallocate( soilice )
        deallocate (PLSIN)

        RETURN_(ESMF_SUCCESS)

      end subroutine Driver

      ! ----------------- routines for CDF scaling -------------------
      
      REAL FUNCTION cdf2fpar (cdf, k,l, m, m1, m2)
        
        REAL, intent (in)  :: cdf, k,l,m, m1, m2
        REAL :: x, ThisCDF, ThisFPAR
        integer, parameter :: nBINS = 40
        
        x       = real (nBINS)
        ThisCDF = 1.
        
        do while (ThisCDF >= cdf)
           ThisFPAR = 1. - (real(nbins)-x)/real(nbins) - 1./2./real(nbins)
           ThisCDF  = K * betai (L, M, ThisFPAR)
           x = x - 1.
           if(x == 0) exit 
        end do
        
        cdf2fpar = ThisFPAR * m2 + m1
        if(cdf2fpar > m2) cdf2fpar = m2
        if(cdf2fpar < m1) cdf2fpar = m1
        return

      END FUNCTION cdf2fpar

      ! ---------------------------------------------------------
      
      FUNCTION betai(a,b,x)
        REAL betai,a,b,x
        REAL bt
        !external gammln
        
        if (x < 0.0125) x = 0.0125
        if (x > 0.9875) x = 0.9875
        
        if(x.lt.0..or.x.gt.1.)print *, 'bad argument x in betai',x
        if(x.lt.0..or.x.gt.1.)stop
        if(x.eq.0..or.x.eq.1.)then
           bt=0.
        else 
           bt=exp(gammln(a+b)-gammln(a)-gammln(b) &
                +a*log(x)+b*log(1.-x))
        endif
        
        if(x.lt.(a+1.)/(a+b+2.))then 
           betai=bt*betacf(a,b,x)/a
           return
        else
           betai=1.-bt*betacf(b,a,1.-x)/b 
           return
        endif
        
      END FUNCTION betai

      ! -------------------------------------------------------

      FUNCTION betacf(a,b,x)

        INTEGER MAXIT
        REAL betacf,a,b,x,EPS,FPMIN
        PARAMETER (MAXIT=100,EPS=3.e-7,FPMIN=1.e-30)
        INTEGER m,m2
        REAL aa,c,d,del,h,qab,qam,qap
        
        qab=a+b 
        qap=a+1. 
        qam=a-1.
        c=1. 
        d=1.-qab*x/qap
        
        if(abs(d).lt.FPMIN)d=FPMIN
        d=1./d
        h=d
        do m=1,MAXIT
           m2=2*m
           aa=m*(b-m)*x/((qam+m2)*(a+m2))
           d=1.+aa*d 
           if(abs(d).lt.FPMIN)d=FPMIN
           c=1.+aa/c
           if(abs(c).lt.FPMIN)c=FPMIN
           d=1./d
           h=h*d*c
           aa=-(a+m)*(qab+m)*x/((a+m2)*(qap+m2))
           d=1.+aa*d 
           if(abs(d).lt.FPMIN)d=FPMIN
           c=1.+aa/c
           if(abs(c).lt.FPMIN)c=FPMIN
           d=1./d
           del=d*c
           h=h*del
           if(abs(del-1.).lt.EPS)exit 
        enddo
        betacf=h
        return
        
      END FUNCTION betacf
      
      ! --------------------------------------------------------------
      
      FUNCTION gammln(xx)
        
        REAL gammln,xx
        INTEGER j
        DOUBLE PRECISION ser,stp,tmp,x,y,cof(6)
        
        SAVE cof,stp
        DATA cof,stp/76.18009172947146d0,-86.50532032941677d0,          &
             24.01409824083091d0,-1.231739572450155d0,.1208650973866179d-2, &
             -.5395239384953d-5,2.5066282746310005d0/
        x=xx
        y=x
        tmp=x+5.5d0
        tmp=(x+0.5d0)*log(tmp)-tmp
        ser=1.000000000190015d0
        do  j=1,6
           y=y+1.d0
           ser=ser+cof(j)/y
        enddo
        gammln=tmp+log(stp*ser/x)
        return
        
      END FUNCTION gammln

      ! --------------------------------------------------------------
      
      integer function VarID (NCFID, VNAME) 
        
        integer, intent (in)      :: NCFID
        character(*), intent (in) :: VNAME
        integer                   :: status
        
        STATUS = NF_INQ_VARID (NCFID, trim(VNAME) ,VarID); VERIFY_(STATUS)
        
      end function VarID

end subroutine RUN2

!BOP
! !IROUTINE: RUN0 -- Extra run method for the OFFLINE case, called by RUN1
! !INTERFACE:

subroutine RUN0(gc, import, export, clock, rc)
  
  ! !ARGUMENTS:
  type(ESMF_GridComp), intent(inout) :: gc     ! Gridded component
  type(ESMF_State),    intent(inout) :: import ! Import state
  type(ESMF_State),    intent(inout) :: export ! Export state
  type(ESMF_Clock),    intent(inout) :: clock  ! The clock
  integer, optional,   intent(  out) :: rc     ! Error code

  ! !DESCRIPTION: In the OFFLINE case, some diagnostic vars (INTERNAL states
  ! asnow and emis) are updated here.
  !EOP

  ! ErrLog variables
  integer :: status
  character(len=ESMF_MAXSTR) :: Iam
  character(len=ESMF_MAXSTR) :: comp_name

  ! Local variables

  !! ESMF/MAPL variables
  type(MAPL_MetaComp), pointer :: MAPL
  type(ESMF_State) :: INTERNAL

  !! IMPORT pointers
  real, pointer :: ps (:)=>null()

  !! INTERNAL pointers
  !! -asnow-emis-ww-fr-
  real, pointer :: ity(:,:)=>null()
  real, pointer :: fvg(:,:)=>null()
  real, pointer :: asnow(:)=>null()
  real, pointer :: emis(:)=>null()
  real, pointer :: ww(:,:)=>null()
  real, pointer :: fr(:,:)=>null()
  real, pointer :: DCQ(:,:)=>null()
  real, pointer :: DCH(:,:)=>null()
  !! -prognostic-variables-
  real, pointer :: tc(:,:)=>null()
  real, pointer :: qc(:,:)=>null()
  real, pointer :: htsnnn1(:)=>null()
  real, pointer :: wesnn1(:)=>null()
  real, pointer :: wesnn2(:)=>null()
  real, pointer :: wesnn3(:)=>null()
  real, pointer :: srfexc(:)=>null()
  real, pointer :: rzexc(:)=>null()
  real, pointer :: catdef(:)=>null()
  !! -parameters-
  real, pointer :: vgwmax(:)=>null()
  real, pointer :: cdcr1(:)=>null()
  real, pointer :: cdcr2(:)=>null()
  real, pointer :: psis(:)=>null()
  real, pointer :: bee(:)=>null()
  real, pointer :: poros(:)=>null()
  real, pointer :: wpwet(:)=>null()
  real, pointer :: ars1(:)=>null()
  real, pointer :: ars2(:)=>null()
  real, pointer :: ars3(:)=>null()
  real, pointer :: ara1(:)=>null()
  real, pointer :: ara2(:)=>null()
  real, pointer :: ara3(:)=>null()
  real, pointer :: ara4(:)=>null()
  real, pointer :: arw1(:)=>null()
  real, pointer :: arw2(:)=>null()
  real, pointer :: arw3(:)=>null()
  real, pointer :: arw4(:)=>null()
  real, pointer :: bf1(:)=>null()
  real, pointer :: bf2(:)=>null()

  !! Miscellaneous
  integer :: ntiles, nv, nz
  real, allocatable :: dummy(:)
  real, allocatable :: dzsf(:), ar1(:), ar2(:), wesnn(:,:)
  real, allocatable :: catdefcp(:), srfexccp(:), rzexccp(:)
  real, allocatable :: VEG1(:), VEG2(:)
  integer, allocatable :: ityp(:,:,:)
  real,    allocatable :: fveg(:,:,:), elai(:,:,:), esai(:,:,:), wtzone(:,:), lai1(:), lai2(:), wght(:)
  real, allocatable,dimension(:) :: fveg1, fveg2

  ! Begin...

  ! Get component name and setup traceback handle
  call ESMF_GridCompGet(gc, name=comp_name, rc=status)
  VERIFY_(status)
  Iam = trim(comp_name)//"::RUN0"

  ! Get MAPL object
  call MAPL_GetObjectFromGC(gc, MAPL, rc=status)
  VERIFY_(status)

  ! Get component's internal ESMF state
  call MAPL_Get(MAPL, INTERNAL_ESMF_STATE=INTERNAL, rc=status)
  VERIFY_(status)

  ! Pointers to IMPORTs
  call MAPL_GetPointer(import, ps, 'PS', rc=status)
  VERIFY_(status)

  ! Pointers to EXPORTs
  call MAPL_GetPointer(export, asnow, 'ASNOW', rc=status)
  VERIFY_(status)
  call MAPL_GetPointer(export, emis, 'EMIS', rc=status)
  VERIFY_(status)

  ! Pointers to INTERNALs
  call MAPL_GetPointer(INTERNAL, ITY, 'ITY', rc=status)
  VERIFY_(status)
  call MAPL_GetPointer(INTERNAL, FVG, 'FVG', rc=status)
  VERIFY_(status)
  call MAPL_GetPointer(INTERNAL, fr, 'FR', rc=status)
  VERIFY_(status)
  call MAPL_GetPointer(INTERNAL, ww, 'WW', rc=status)
  VERIFY_(status)
  call MAPL_GetPointer(INTERNAL, DCQ, 'DCQ', rc=status)
  VERIFY_(status)
  call MAPL_GetPointer(INTERNAL, DCH, 'DCH', rc=status)
  VERIFY_(status)
  call MAPL_GetPointer(INTERNAL, tc, 'TC', rc=status)
  VERIFY_(status)
  call MAPL_GetPointer(INTERNAL, qc, 'QC', rc=status)
  VERIFY_(status)
  call MAPL_GetPointer(INTERNAL, htsnnn1, 'HTSNNN1', rc=status)
  VERIFY_(status)
  call MAPL_GetPointer(INTERNAL, wesnn1, 'WESNN1', rc=status)
  VERIFY_(status)
  call MAPL_GetPointer(INTERNAL, wesnn2, 'WESNN2', rc=status)
  VERIFY_(status)
  call MAPL_GetPointer(INTERNAL, wesnn3, 'WESNN3', rc=status)
  VERIFY_(status)
  call MAPL_GetPointer(INTERNAL, vgwmax, 'VGWMAX', rc=status)
  VERIFY_(status)
  call MAPL_GetPointer(INTERNAL, cdcr1, 'CDCR1', rc=status)
  VERIFY_(status)
  call MAPL_GetPointer(INTERNAL, cdcr2, 'CDCR2', rc=status)
  VERIFY_(status)
  call MAPL_GetPointer(INTERNAL, psis, 'PSIS', rc=status)
  VERIFY_(status)
  call MAPL_GetPointer(INTERNAL, bee, 'BEE', rc=status)
  VERIFY_(status)
  call MAPL_GetPointer(INTERNAL, poros, 'POROS', rc=status)
  VERIFY_(status)
  call MAPL_GetPointer(INTERNAL, wpwet, 'WPWET', rc=status)
  VERIFY_(status)
  call MAPL_GetPointer(INTERNAL, ars1, 'ARS1', rc=status)
  VERIFY_(status)
  call MAPL_GetPointer(INTERNAL, ars2, 'ARS2', rc=status)
  VERIFY_(status)
  call MAPL_GetPointer(INTERNAL, ars3, 'ARS3', rc=status)
  VERIFY_(status)
  call MAPL_GetPointer(INTERNAL, ara1, 'ARA1', rc=status)
  VERIFY_(status)
  call MAPL_GetPointer(INTERNAL, ara2, 'ARA2', rc=status)
  VERIFY_(status)
  call MAPL_GetPointer(INTERNAL, ara3, 'ARA3', rc=status)
  VERIFY_(status)
  call MAPL_GetPointer(INTERNAL, ara4, 'ARA4', rc=status)
  VERIFY_(status)
  call MAPL_GetPointer(INTERNAL, arw1, 'ARW1', rc=status)
  VERIFY_(status)
  call MAPL_GetPointer(INTERNAL, arw2, 'ARW2', rc=status)
  VERIFY_(status)
  call MAPL_GetPointer(INTERNAL, arw3, 'ARW3', rc=status)
  VERIFY_(status)
  call MAPL_GetPointer(INTERNAL, arw4, 'ARW4', rc=status)
  VERIFY_(status)
  call MAPL_GetPointer(INTERNAL, bf1, 'BF1', rc=status)
  VERIFY_(status)
  call MAPL_GetPointer(INTERNAL, bf2, 'BF2', rc=status)
  VERIFY_(status)
  call MAPL_GetPointer(INTERNAL, srfexc, 'SRFEXC', rc=status)
  VERIFY_(status)
  call MAPL_GetPointer(INTERNAL, rzexc, 'RZEXC', rc=status)
  VERIFY_(status)
  call MAPL_GetPointer(INTERNAL, catdef, 'CATDEF', rc=status)
  VERIFY_(status)

  ! Number of tiles and a dummy real array
  ntiles = size(HTSNNN1)
  allocate(dummy(ntiles), stat=status)
  VERIFY_(status)
  ! Reset WW
  WW = 0.

  ! get CNLAI to compute emmissivity
  allocate(fveg1     (NTILES))
  allocate(fveg2     (NTILES))
  allocate(veg1(ntiles), stat=status)
  VERIFY_(status)
  allocate(veg2(ntiles), stat=status)
  VERIFY_(status)
  allocate(   ityp(ntiles,num_veg,num_zon) )
  allocate(   fveg(ntiles,num_veg,num_zon) )
  allocate( wtzone(ntiles,num_zon) )
  allocate(   elai(ntiles,num_veg,num_zon) )
  allocate(   esai(ntiles,num_veg,num_zon) )

  allocate ( lai1(ntiles) )
  allocate ( lai2(ntiles) )
  allocate ( wght(ntiles) )

! set CLM CN PFT & fraction, set carbon zone weights
! --------------------------------------------------
   do nz = 1,num_zon
     ityp(:,:,nz) = nint(ity(:,:))
     fveg(:,:,nz) = fvg(:,:)
     wtzone(:,nz) = CN_zone_weight(nz)
   end do

   IF(PRESCRIBE_DVG == 0) THEN
      ! obtain LAI from previous time step (from CN model)
      ! --------------------------------------------------
      call get_CN_LAI(ntiles,num_veg,num_zon,ityp,fveg,elai,esai=esai)
   ELSE
      call  read_prescribed_LAI (INTERNAL, CLOCK, GC, NTILES, elai,esai)
   ENDIF

   lai1 = 0.
   wght = 0.
   do nz = 1,num_zon
     do nv = 1,2
       lai1(:) = lai1(:) + max(elai(:,nv,nz),0.)*fveg(:,nv,nz)*wtzone(:,nz)
       wght(:) = wght(:) +                       fveg(:,nv,nz)*wtzone(:,nz)
     end do
   end do
   lai1 = lai1 / max(wght,1.e-8) ! LAI for primary vegetation type

   lai2 = 0.
   wght = 0.
   do nz = 1,num_zon
     do nv = 3,4
       lai2(:) = lai2(:) + max(elai(:,nv,nz),0.)*fveg(:,nv,nz)*wtzone(:,nz)
       wght(:) = wght(:) +                       fveg(:,nv,nz)*wtzone(:,nz)
     end do
   end do
   lai2 = lai2 / max(wght,1.e-8) ! LAI for secondary vegetation type

   deallocate ( ityp )
   deallocate ( fveg )
   deallocate ( elai )
   deallocate ( esai )
   deallocate ( wtzone )

!  Vegetation types used to index into tables
!--------------------------------------------

   where(ITY(:,1) > 0.)
     VEG1 = map_cat(nint(ITY(:,1)))  ! gkw: primary   CN PFT type mapped to catchment type; ITY should be > 0 even if FVEG=0
    elsewhere
     VEG1 = map_cat(nint(ITY(:,2)))  ! gkw: primary   CN PFT type mapped to catchment type; ITY should be > 0 even if FVEG=0
   endwhere
   where(ITY(:,3) > 0.)
     VEG2 = map_cat(nint(ITY(:,3)))  ! gkw: secondary CN PFT type mapped to catchment type; ITY should be > 0 even if FVEG=0
    elsewhere
     VEG2 = map_cat(nint(ITY(:,4)))  ! gkw: secondary CN PFT type mapped to catchment type; ITY should be > 0 even if FVEG=0
   endwhere
   _ASSERT((count(VEG1>NTYPS.or.VEG1<1)==0),'needs informative message') 
   _ASSERT((count(VEG2>NTYPS.or.VEG2<1)==0),'needs informative message')
   fveg1(:) = fvg(:,1) + fvg(:,2) ! sum veg fractions (primary)   gkw: NUM_VEG specific
   fveg2(:) = fvg(:,3) + fvg(:,4) ! sum veg fractions (secondary) gkw: fveg1+fveg2=1

  ! Compute ASNOW and EMIS
  allocate(wesnn(3,ntiles), stat=status)
  VERIFY_(status)
  wesnn(1,:) = wesnn1
  wesnn(2,:) = wesnn2
  wesnn(3,:) = wesnn3
  call StieglitzSnow_calc_asnow(3, ntiles, wesnn, asnow)

  EMIS    = fveg1*(EMSVEG(NINT(VEG1)) + (EMSBARESOIL - EMSVEG(NINT(VEG1)))*exp(-LAI1)) + &
       fveg2*(EMSVEG(NINT(VEG2)) + (EMSBARESOIL - EMSVEG(NINT(VEG2)))*exp(-LAI2))

  emis = emis*(1.-asnow) + EMSSNO*asnow

  ! Compute FR
  ! Step 1: set dzsf
  ! Step 2: compute ar1, ar2 via call to catch_calc_soil_moist()
  ! Step 3: compute fr

  ! -step-1-
  allocate(dzsf(ntiles), stat=status)
  VERIFY_(status)
  dzsf = SURFLAY

  ! -step-2-
  allocate(ar1(ntiles), stat=status)
  VERIFY_(status)
  allocate(ar2(ntiles), stat=status)
  VERIFY_(status)
  ! -we-don't-want-to-modify-srfexc-rzexc-and-catdef-
  ! -so-we-create-local-copies-
  allocate(catdefcp(ntiles), stat=status)
  VERIFY_(status)
  allocate(srfexccp(ntiles), stat=status)
  VERIFY_(status)
  allocate(rzexccp(ntiles), stat=status)
  VERIFY_(status)
  catdefcp = catdef
  srfexccp = srfexc
  rzexccp = rzexc
  call catch_calc_soil_moist(                                                   &
       ! intent(in)
       ntiles, nint(veg1), dzsf, vgwmax, cdcr1, cdcr2,                           &
       psis, bee, poros, wpwet,                                                 &
       ars1, ars2, ars3,                                                        &
       ara1, ara2, ara3, ara4,                                                  &
       arw1, arw2, arw3, arw4,                                                  &
       ! intent(inout)
       ! from process_cat
       srfexccp, rzexccp, catdefcp,                                             &
       ! use this one can match process_cat
       ! srfexc, rzexc, catdef,                                             &
       ! intent(out)
       ar1, ar2, dummy                                                          &
       )

  fr(:,FSAT) =           ar1  * (1-asnow)
  fr(:,FTRN) =           ar2  * (1-asnow)
  fr(:,FWLT) = (1.0-(ar1+ar2))* (1-asnow)
  fr(:,FSNW) =                     asnow
  fr = min(max(fr,0.0),1.0)

  ! Overwrite the top layer snow temperature tc(4) with its diagnosed value
  call StieglitzSnow_calc_tpsnow(ntiles, htsnnn1, wesnn1, tc(:,4), dummy)
  tc(:,FSNW) = tc(:,FSNW) + MAPL_TICE ! Convert to K

  ! Overwrite qc(4)
  !qc(:,FSNW) = GEOS_QSAT(tc(:,FSNW), PS, PASCALS=.true., RAMP=0.0)
   qc(:,FSNW) = MAPL_EQsat(tc(:,FSNW),PS,OverIce=.true.)

  ! Clean up
  if (allocated(catdefcp)) deallocate(catdefcp)
  if (allocated(srfexccp)) deallocate(srfexccp)
  if (allocated(rzexccp)) deallocate(rzexccp)
  if (allocated(dummy)) deallocate(dummy)
  if (allocated(dzsf)) deallocate(dzsf)
  if (allocated(ar1)) deallocate(ar1)
  if (allocated(ar2)) deallocate(ar2)
  if (allocated(wesnn)) deallocate(wesnn)
  if (allocated(fveg1)) deallocate (fveg1)
  if (allocated(fveg2)) deallocate (fveg2)
  if (allocated(veg1)) deallocate (veg1)
  if (allocated(veg2)) deallocate (veg2)
  if (allocated(lai1)) deallocate (lai1)
  if (allocated(lai2)) deallocate (lai2)
  if (allocated(wght)) deallocate (wght)

  ! All done
  RETURN_(ESMF_SUCCESS)

end subroutine RUN0

! READ PRESCRIBED LAI and SAI
! ---------------------------

SUBROUTINE read_prescribed_LAI  (INTERNAL, CLOCK, GC, NTILES, elai, esai)

  implicit none
  character(len=ESMF_MAXSTR)        :: FCAST_BEGTIME, FTIME
  type(ESMF_State)                  :: INTERNAL
  type(ESMF_Clock),   intent(inout) :: CLOCK
  type(ESMF_GridComp),intent(inout) :: GC
  character(len=ESMF_MAXSTR)        :: COMP_NAME
  type(ESMF_Time)                   :: CURRENT_TIME, TIME0
  type(MAPL_MetaComp), pointer :: MAPL=>null()
  character(len=ESMF_MAXSTR) :: Iam

  integer, parameter   :: nveg  = num_veg ! number of vegetation types
  integer, parameter   :: nzone = num_zon ! number of stress zones
  integer, intent (in) :: NTILES
  REAL                 :: TIMELAG
  INTEGER              :: BYEAR, BMON, BDAY, BHOUR, dSecs
  type(ESMF_TimeInterval) :: TIMEDIF 
  type(ESMF_VM)        :: VM
  logical, save        :: first = .true.

  real, dimension (NTILES, nveg, nzone), intent (inout) :: elai,esai
  real, dimension(:),   pointer :: CNSAI11
  real, dimension(:),   pointer :: CNSAI12 
  real, dimension(:),   pointer :: CNSAI13 
  real, dimension(:),   pointer :: CNSAI21 
  real, dimension(:),   pointer :: CNSAI22
  real, dimension(:),   pointer :: CNSAI23 
  real, dimension(:),   pointer :: CNSAI31 
  real, dimension(:),   pointer :: CNSAI32
  real, dimension(:),   pointer :: CNSAI33
  real, dimension(:),   pointer :: CNSAI41
  real, dimension(:),   pointer :: CNSAI42
  real, dimension(:),   pointer :: CNSAI43
  real, dimension(:),   pointer :: CNLAI11
  real, dimension(:),   pointer :: CNLAI12 
  real, dimension(:),   pointer :: CNLAI13 
  real, dimension(:),   pointer :: CNLAI21 
  real, dimension(:),   pointer :: CNLAI22
  real, dimension(:),   pointer :: CNLAI23 
  real, dimension(:),   pointer :: CNLAI31 
  real, dimension(:),   pointer :: CNLAI32
  real, dimension(:),   pointer :: CNLAI33
  real, dimension(:),   pointer :: CNLAI41
  real, dimension(:),   pointer :: CNLAI42
  real, dimension(:),   pointer :: CNLAI43
  
  character (len=ESMF_MAXSTR)   :: CNLAIFILE, CNSAIFILE, CNLAIlabel, CNSAILabel
  character (len=7), dimension (12) , PARAMETER :: CNSAI_VARS = (/     &
       'CNSAI11',  'CNSAI12',  'CNSAI13', &  
       'CNSAI21',  'CNSAI22',  'CNSAI23', &   
       'CNSAI31',  'CNSAI32',  'CNSAI33', &   
       'CNSAI41',  'CNSAI42',  'CNSAI43'/)   
  character (len=7), dimension (12) , PARAMETER :: CNLAI_VARS = (/     &
       'CNLAI11',  'CNLAI12',  'CNLAI13', &    
       'CNLAI21',  'CNLAI22',  'CNLAI23', &   
       'CNLAI31',  'CNLAI32',  'CNLAI33', &   
       'CNLAI41',  'CNLAI42',  'CNLAI43'/)   
  integer :: n, STATUS, NUM_ENSEMBLE, RC

  real, dimension (:,:), pointer:: ITY
  real, dimension(:),   pointer :: CNSAI11A
  real, dimension(:),   pointer :: CNSAI12A 
  real, dimension(:),   pointer :: CNSAI13A 
  real, dimension(:),   pointer :: CNSAI21A 
  real, dimension(:),   pointer :: CNSAI22A
  real, dimension(:),   pointer :: CNSAI23A 
  real, dimension(:),   pointer :: CNSAI31A 
  real, dimension(:),   pointer :: CNSAI32A
  real, dimension(:),   pointer :: CNSAI33A
  real, dimension(:),   pointer :: CNSAI41A
  real, dimension(:),   pointer :: CNSAI42A
  real, dimension(:),   pointer :: CNSAI43A
  real, dimension(:),   pointer :: CNLAI11A
  real, dimension(:),   pointer :: CNLAI12A 
  real, dimension(:),   pointer :: CNLAI13A 
  real, dimension(:),   pointer :: CNLAI21A 
  real, dimension(:),   pointer :: CNLAI22A
  real, dimension(:),   pointer :: CNLAI23A 
  real, dimension(:),   pointer :: CNLAI31A 
  real, dimension(:),   pointer :: CNLAI32A
  real, dimension(:),   pointer :: CNLAI33A
  real, dimension(:),   pointer :: CNLAI41A
  real, dimension(:),   pointer :: CNLAI42A
  real, dimension(:),   pointer :: CNLAI43A

  allocate (CNSAI11 (1:NTILES))
  allocate (CNSAI12 (1:NTILES))
  allocate (CNSAI13 (1:NTILES))
  allocate (CNSAI21 (1:NTILES))
  allocate (CNSAI22 (1:NTILES))
  allocate (CNSAI23 (1:NTILES))
  allocate (CNSAI31 (1:NTILES))
  allocate (CNSAI32 (1:NTILES))
  allocate (CNSAI33 (1:NTILES))
  allocate (CNSAI41 (1:NTILES))
  allocate (CNSAI42 (1:NTILES))
  allocate (CNSAI43 (1:NTILES))
  allocate (CNLAI11 (1:NTILES))
  allocate (CNLAI12 (1:NTILES))
  allocate (CNLAI13 (1:NTILES))
  allocate (CNLAI21 (1:NTILES))
  allocate (CNLAI22 (1:NTILES))
  allocate (CNLAI23 (1:NTILES))
  allocate (CNLAI31 (1:NTILES))
  allocate (CNLAI32 (1:NTILES))
  allocate (CNLAI33 (1:NTILES))
  allocate (CNLAI41 (1:NTILES))
  allocate (CNLAI42 (1:NTILES))
  allocate (CNLAI43 (1:NTILES))

  call MAPL_GetObjectFromGC(gc, MAPL, rc=status)               ; VERIFY_(status)
  call ESMF_GridCompGet ( GC, NAME=COMP_NAME, RC=STATUS )      ; VERIFY_(STATUS)
  call ESMF_ClockGet( CLOCK, currTime=CURRENT_TIME, RC=STATUS ); VERIFY_(STATUS)
  call MAPL_GetResource ( MAPL, NUM_ENSEMBLE, Label="NUM_LDAS_ENSEMBLE:", DEFAULT=1, RC=STATUS)
  VERIFY_(STATUS)
  Iam=trim(COMP_NAME)//"::read_prescribed_LAI"

  IF (PRESCRIBE_DVG == 3) THEN
     call MAPL_GetPointer(INTERNAL,ITY     , 'ITY'      ,   RC=STATUS); VERIFY_(STATUS)
     call MAPL_GetPointer(INTERNAL,CNSAI11A, 'CNSAI11A' ,   RC=STATUS); VERIFY_(STATUS)
     call MAPL_GetPointer(INTERNAL,CNSAI12A, 'CNSAI12A' ,   RC=STATUS); VERIFY_(STATUS)
     call MAPL_GetPointer(INTERNAL,CNSAI13A, 'CNSAI13A' ,   RC=STATUS); VERIFY_(STATUS)
     call MAPL_GetPointer(INTERNAL,CNSAI21A, 'CNSAI21A' ,   RC=STATUS); VERIFY_(STATUS)
     call MAPL_GetPointer(INTERNAL,CNSAI22A, 'CNSAI22A' ,   RC=STATUS); VERIFY_(STATUS)
     call MAPL_GetPointer(INTERNAL,CNSAI23A, 'CNSAI23A' ,   RC=STATUS); VERIFY_(STATUS)
     call MAPL_GetPointer(INTERNAL,CNSAI31A, 'CNSAI31A' ,   RC=STATUS); VERIFY_(STATUS)
     call MAPL_GetPointer(INTERNAL,CNSAI32A, 'CNSAI32A' ,   RC=STATUS); VERIFY_(STATUS)
     call MAPL_GetPointer(INTERNAL,CNSAI33A, 'CNSAI33A' ,   RC=STATUS); VERIFY_(STATUS)
     call MAPL_GetPointer(INTERNAL,CNSAI41A, 'CNSAI41A' ,   RC=STATUS); VERIFY_(STATUS)
     call MAPL_GetPointer(INTERNAL,CNSAI42A, 'CNSAI42A' ,   RC=STATUS); VERIFY_(STATUS)
     call MAPL_GetPointer(INTERNAL,CNSAI43A, 'CNSAI43A' ,   RC=STATUS); VERIFY_(STATUS)
     call MAPL_GetPointer(INTERNAL,CNLAI11A, 'CNLAI11A' ,   RC=STATUS); VERIFY_(STATUS)
     call MAPL_GetPointer(INTERNAL,CNLAI12A, 'CNLAI12A' ,   RC=STATUS); VERIFY_(STATUS)
     call MAPL_GetPointer(INTERNAL,CNLAI13A, 'CNLAI13A' ,   RC=STATUS); VERIFY_(STATUS)
     call MAPL_GetPointer(INTERNAL,CNLAI21A, 'CNLAI21A' ,   RC=STATUS); VERIFY_(STATUS)
     call MAPL_GetPointer(INTERNAL,CNLAI22A, 'CNLAI22A' ,   RC=STATUS); VERIFY_(STATUS)
     call MAPL_GetPointer(INTERNAL,CNLAI23A, 'CNLAI23A' ,   RC=STATUS); VERIFY_(STATUS)
     call MAPL_GetPointer(INTERNAL,CNLAI31A, 'CNLAI31A' ,   RC=STATUS); VERIFY_(STATUS)
     call MAPL_GetPointer(INTERNAL,CNLAI32A, 'CNLAI32A' ,   RC=STATUS); VERIFY_(STATUS)
     call MAPL_GetPointer(INTERNAL,CNLAI33A, 'CNLAI33A' ,   RC=STATUS); VERIFY_(STATUS)
     call MAPL_GetPointer(INTERNAL,CNLAI41A, 'CNLAI41A' ,   RC=STATUS); VERIFY_(STATUS)
     call MAPL_GetPointer(INTERNAL,CNLAI42A, 'CNLAI42A' ,   RC=STATUS); VERIFY_(STATUS)
     call MAPL_GetPointer(INTERNAL,CNLAI43A, 'CNLAI43A' ,   RC=STATUS); VERIFY_(STATUS)
  ENDIF
  
  DO N = 1,12 

     CNLAIlabel = trim(CNLAI_VARS(n))//' _FILE:'
     CNSAIlabel = trim(CNSAI_VARS(n))//' _FILE:'
     
     if(NUM_ENSEMBLE > 1) then
        CNLAIlabel = trim(CNLAI_VARS(n))//comp_name(6:9)//' _FILE:'
        CNSAIlabel = trim(CNSAI_VARS(n))//comp_name(6:9)//' _FILE:'          
     endif
     
     call MAPL_GetResource(MAPL, CNLAIFILE, label = trim(CNLAIlabel), default = trim(CNLAI_VARS(n))//'.data', RC=STATUS) ; VERIFY_(STATUS)
     call MAPL_GetResource(MAPL, CNSAIFILE, label = trim(CNSAIlabel), default = trim(CNSAI_VARS(n))//'.data', RC=STATUS) ; VERIFY_(STATUS)
     
     if(n ==  1) call MAPL_ReadForcing(MAPL,trim(CNLAI_VARS(n)),CNLAIFILE,CURRENT_TIME,CNLAI11,ON_TILES=.true.,RC=STATUS) ; VERIFY_(STATUS)
     if(n ==  2) call MAPL_ReadForcing(MAPL,trim(CNLAI_VARS(n)),CNLAIFILE,CURRENT_TIME,CNLAI12,ON_TILES=.true.,RC=STATUS) ; VERIFY_(STATUS)
     if(n ==  3) call MAPL_ReadForcing(MAPL,trim(CNLAI_VARS(n)),CNLAIFILE,CURRENT_TIME,CNLAI13,ON_TILES=.true.,RC=STATUS) ; VERIFY_(STATUS)
     if(n ==  4) call MAPL_ReadForcing(MAPL,trim(CNLAI_VARS(n)),CNLAIFILE,CURRENT_TIME,CNLAI21,ON_TILES=.true.,RC=STATUS) ; VERIFY_(STATUS)
     if(n ==  5) call MAPL_ReadForcing(MAPL,trim(CNLAI_VARS(n)),CNLAIFILE,CURRENT_TIME,CNLAI22,ON_TILES=.true.,RC=STATUS) ; VERIFY_(STATUS)
     if(n ==  6) call MAPL_ReadForcing(MAPL,trim(CNLAI_VARS(n)),CNLAIFILE,CURRENT_TIME,CNLAI23,ON_TILES=.true.,RC=STATUS) ; VERIFY_(STATUS)
     if(n ==  7) call MAPL_ReadForcing(MAPL,trim(CNLAI_VARS(n)),CNLAIFILE,CURRENT_TIME,CNLAI31,ON_TILES=.true.,RC=STATUS) ; VERIFY_(STATUS)
     if(n ==  8) call MAPL_ReadForcing(MAPL,trim(CNLAI_VARS(n)),CNLAIFILE,CURRENT_TIME,CNLAI32,ON_TILES=.true.,RC=STATUS) ; VERIFY_(STATUS)
     if(n ==  9) call MAPL_ReadForcing(MAPL,trim(CNLAI_VARS(n)),CNLAIFILE,CURRENT_TIME,CNLAI33,ON_TILES=.true.,RC=STATUS) ; VERIFY_(STATUS)
     if(n == 10) call MAPL_ReadForcing(MAPL,trim(CNLAI_VARS(n)),CNLAIFILE,CURRENT_TIME,CNLAI41,ON_TILES=.true.,RC=STATUS) ; VERIFY_(STATUS)
     if(n == 11) call MAPL_ReadForcing(MAPL,trim(CNLAI_VARS(n)),CNLAIFILE,CURRENT_TIME,CNLAI42,ON_TILES=.true.,RC=STATUS) ; VERIFY_(STATUS)
     if(n == 12) call MAPL_ReadForcing(MAPL,trim(CNLAI_VARS(n)),CNLAIFILE,CURRENT_TIME,CNLAI43,ON_TILES=.true.,RC=STATUS) ; VERIFY_(STATUS)
     
     if(n ==  1) call MAPL_ReadForcing(MAPL,trim(CNSAI_VARS(n)),CNSAIFILE,CURRENT_TIME,CNSAI11,ON_TILES=.true.,RC=STATUS) ; VERIFY_(STATUS)
     if(n ==  2) call MAPL_ReadForcing(MAPL,trim(CNSAI_VARS(n)),CNSAIFILE,CURRENT_TIME,CNSAI12,ON_TILES=.true.,RC=STATUS) ; VERIFY_(STATUS)
     if(n ==  3) call MAPL_ReadForcing(MAPL,trim(CNSAI_VARS(n)),CNSAIFILE,CURRENT_TIME,CNSAI13,ON_TILES=.true.,RC=STATUS) ; VERIFY_(STATUS)
     if(n ==  4) call MAPL_ReadForcing(MAPL,trim(CNSAI_VARS(n)),CNSAIFILE,CURRENT_TIME,CNSAI21,ON_TILES=.true.,RC=STATUS) ; VERIFY_(STATUS)
     if(n ==  5) call MAPL_ReadForcing(MAPL,trim(CNSAI_VARS(n)),CNSAIFILE,CURRENT_TIME,CNSAI22,ON_TILES=.true.,RC=STATUS) ; VERIFY_(STATUS)
     if(n ==  6) call MAPL_ReadForcing(MAPL,trim(CNSAI_VARS(n)),CNSAIFILE,CURRENT_TIME,CNSAI23,ON_TILES=.true.,RC=STATUS) ; VERIFY_(STATUS)
     if(n ==  7) call MAPL_ReadForcing(MAPL,trim(CNSAI_VARS(n)),CNSAIFILE,CURRENT_TIME,CNSAI31,ON_TILES=.true.,RC=STATUS) ; VERIFY_(STATUS)
     if(n ==  8) call MAPL_ReadForcing(MAPL,trim(CNSAI_VARS(n)),CNSAIFILE,CURRENT_TIME,CNSAI32,ON_TILES=.true.,RC=STATUS) ; VERIFY_(STATUS)
     if(n ==  9) call MAPL_ReadForcing(MAPL,trim(CNSAI_VARS(n)),CNSAIFILE,CURRENT_TIME,CNSAI33,ON_TILES=.true.,RC=STATUS) ; VERIFY_(STATUS)
     if(n == 10) call MAPL_ReadForcing(MAPL,trim(CNSAI_VARS(n)),CNSAIFILE,CURRENT_TIME,CNSAI41,ON_TILES=.true.,RC=STATUS) ; VERIFY_(STATUS)
     if(n == 11) call MAPL_ReadForcing(MAPL,trim(CNSAI_VARS(n)),CNSAIFILE,CURRENT_TIME,CNSAI42,ON_TILES=.true.,RC=STATUS) ; VERIFY_(STATUS)
     if(n == 12) call MAPL_ReadForcing(MAPL,trim(CNSAI_VARS(n)),CNSAIFILE,CURRENT_TIME,CNSAI43,ON_TILES=.true.,RC=STATUS) ; VERIFY_(STATUS)
     
  END DO

  IF (PRESCRIBE_DVG < 3) THEN

     ! Prescribing LAI/SAI
     esai(:,1,1) = CNSAI11
     esai(:,1,2) = CNSAI12
     esai(:,1,3) = CNSAI13
     esai(:,2,1) = CNSAI21
     esai(:,2,2) = CNSAI22
     esai(:,2,3) = CNSAI23
     esai(:,3,1) = CNSAI31
     esai(:,3,2) = CNSAI32
     esai(:,3,3) = CNSAI33
     esai(:,4,1) = CNSAI41
     esai(:,4,2) = CNSAI42
     esai(:,4,3) = CNSAI43
     elai(:,1,1) = CNLAI11
     elai(:,1,2) = CNLAI12
     elai(:,1,3) = CNLAI13
     elai(:,2,1) = CNLAI21
     elai(:,2,2) = CNLAI22
     elai(:,2,3) = CNLAI23
     elai(:,3,1) = CNLAI31
     elai(:,3,2) = CNLAI32
     elai(:,3,3) = CNLAI33
     elai(:,4,1) = CNLAI41
     elai(:,4,2) = CNLAI42
     elai(:,4,3) = CNLAI43

     where (elai < 0.) elai = 1.e-6
     where (esai < 0.) esai = 1.e-6
     
  endif

  IF (PRESCRIBE_DVG == 3) THEN

     ! Forecast mode
     call MAPL_GetResource(MAPL, FCAST_BEGTIME , label = 'FCAST_BEGTIME:', default = ''  , RC=STATUS) ; VERIFY_(STATUS)

     FTIME = ADJUSTL (FCAST_BEGTIME)
     READ (FTIME ( 1: 4), '(i4)', IOSTAT = STATUS ) BYEAR  ; VERIFY_(STATUS)
     READ (FTIME ( 5: 6), '(i2)', IOSTAT = STATUS ) BMON   ; VERIFY_(STATUS)
     READ (FTIME ( 7: 8), '(i2)', IOSTAT = STATUS ) BDAY   ; VERIFY_(STATUS)
     READ (FTIME ( 9:10),'(i2)', IOSTAT = STATUS ) BHOUR  ; VERIFY_(STATUS)
     call ESMF_TimeSet(TIME0, YY = BYEAR, MM = BMON, DD = BDAY, &
          H =  BHOUR, M =  0, S = 0, rc = STATUS) ; VERIFY_(STATUS)
     
     TIMEDIF = CURRENT_TIME - TIME0 
     call ESMF_TimeIntervalGet (TIMEDIF, s = dSecs, rc=status); VERIFY_(STATUS) 
     timelag = real (dSecs) / 86400.
!     if (MAPL_AM_I_Root(VM)) then
!        print *,'PRESCRIBE_DVG CURRENT TIME:  '
!         CALL ESMF_TimePrint ( CURRENT_TIME, OPTIONS="string", RC=STATUS )
!         print *,'PRESCRIBE_DVG TIME LAG:  ', timelag
!     endif

     IF((timelag == 0).and. (first)) THEN
        first = .false.
!        if (MAPL_AM_I_Root(VM)) then
!         print *,'PRESCRIBE_DVG FIRST ENTRY CURRENT TIME:  '
!         CALL ESMF_TimePrint ( CURRENT_TIME, OPTIONS="string", RC=STATUS )
!         print *,'PRESCRIBE_DVG FIRST ENTRY START TIME:  '
!         CALL ESMF_TimePrint ( TIME0, OPTIONS="string", RC=STATUS )
!        endif
        ! Compute initial anomaly and save with internals
        
        where (CNSAI11 > 20.) CNSAI11 = 20.
        where (CNSAI12 > 20.) CNSAI12 = 20.
        where (CNSAI13 > 20.) CNSAI13 = 20.
        where (CNSAI21 > 20.) CNSAI21 = 20.
        where (CNSAI22 > 20.) CNSAI22 = 20.
        where (CNSAI23 > 20.) CNSAI23 = 20.
        where (CNSAI31 > 20.) CNSAI31 = 20.
        where (CNSAI32 > 20.) CNSAI32 = 20.
        where (CNSAI33 > 20.) CNSAI33 = 20.
        where (CNSAI41 > 20.) CNSAI41 = 20.
        where (CNSAI42 > 20.) CNSAI42 = 20.
        where (CNSAI43 > 20.) CNSAI43 = 20.
        where (CNLAI11 > 20.) CNLAI11 = 20.
        where (CNLAI12 > 20.) CNLAI12 = 20.
        where (CNLAI13 > 20.) CNLAI13 = 20.
        where (CNLAI21 > 20.) CNLAI21 = 20.
        where (CNLAI22 > 20.) CNLAI22 = 20.
        where (CNLAI23 > 20.) CNLAI23 = 20.
        where (CNLAI31 > 20.) CNLAI31 = 20.
        where (CNLAI32 > 20.) CNLAI32 = 20.
        where (CNLAI33 > 20.) CNLAI33 = 20.
        where (CNLAI41 > 20.) CNLAI41 = 20.
        where (CNLAI42 > 20.) CNLAI42 = 20.
        where (CNLAI43 > 20.) CNLAI43 = 20.
           
        where ((CNSAI11 >= 0.) .and. (CNSAI11 <= 20.)) CNSAI11A = esai(:,1,1) - CNSAI11
        where ((CNSAI12 >= 0.) .and. (CNSAI12 <= 20.)) CNSAI12A = esai(:,1,2) - CNSAI12
        where ((CNSAI13 >= 0.) .and. (CNSAI13 <= 20.)) CNSAI13A = esai(:,1,3) - CNSAI13
        where ((CNSAI21 >= 0.) .and. (CNSAI21 <= 20.)) CNSAI21A = esai(:,2,1) - CNSAI21
        where ((CNSAI22 >= 0.) .and. (CNSAI22 <= 20.)) CNSAI22A = esai(:,2,2) - CNSAI22
        where ((CNSAI23 >= 0.) .and. (CNSAI23 <= 20.)) CNSAI23A = esai(:,2,3) - CNSAI23
        where ((CNSAI31 >= 0.) .and. (CNSAI31 <= 20.)) CNSAI31A = esai(:,3,1) - CNSAI31
        where ((CNSAI32 >= 0.) .and. (CNSAI32 <= 20.)) CNSAI32A = esai(:,3,2) - CNSAI32
        where ((CNSAI33 >= 0.) .and. (CNSAI33 <= 20.)) CNSAI33A = esai(:,3,3) - CNSAI33
        where ((CNSAI41 >= 0.) .and. (CNSAI41 <= 20.)) CNSAI41A = esai(:,4,1) - CNSAI41
        where ((CNSAI42 >= 0.) .and. (CNSAI42 <= 20.)) CNSAI42A = esai(:,4,2) - CNSAI42
        where ((CNSAI43 >= 0.) .and. (CNSAI43 <= 20.)) CNSAI43A = esai(:,4,3) - CNSAI43
        where ((CNLAI11 >= 0.) .and. (CNLAI11 <= 20.)) CNLAI11A = elai(:,1,1) - CNLAI11
        where ((CNLAI12 >= 0.) .and. (CNLAI12 <= 20.)) CNLAI12A = elai(:,1,2) - CNLAI12
        where ((CNLAI13 >= 0.) .and. (CNLAI13 <= 20.)) CNLAI13A = elai(:,1,3) - CNLAI13
        where ((CNLAI21 >= 0.) .and. (CNLAI21 <= 20.)) CNLAI21A = elai(:,2,1) - CNLAI21
        where ((CNLAI22 >= 0.) .and. (CNLAI22 <= 20.)) CNLAI22A = elai(:,2,2) - CNLAI22
        where ((CNLAI23 >= 0.) .and. (CNLAI23 <= 20.)) CNLAI23A = elai(:,2,3) - CNLAI23
        where ((CNLAI31 >= 0.) .and. (CNLAI31 <= 20.)) CNLAI31A = elai(:,3,1) - CNLAI31
        where ((CNLAI32 >= 0.) .and. (CNLAI32 <= 20.)) CNLAI32A = elai(:,3,2) - CNLAI32
        where ((CNLAI33 >= 0.) .and. (CNLAI33 <= 20.)) CNLAI33A = elai(:,3,3) - CNLAI33
        where ((CNLAI41 >= 0.) .and. (CNLAI41 <= 20.)) CNLAI41A = elai(:,4,1) - CNLAI41
        where ((CNLAI42 >= 0.) .and. (CNLAI42 <= 20.)) CNLAI42A = elai(:,4,2) - CNLAI42
        where ((CNLAI43 >= 0.) .and. (CNLAI43 <= 20.)) CNLAI43A = elai(:,4,3) - CNLAI43    
      
     ENDIF

     DO N = 1, NTILES

        esai(n,1,1)  = CNSAI11(n) + CNSAI11A(n) * EXP (-TIMELAG/TSLAI (NINT(ITY(N,1)),BMON,2))
        esai(n,1,2)  = CNSAI12(n) + CNSAI12A(n) * EXP (-TIMELAG/TSLAI (NINT(ITY(N,1)),BMON,2))
        esai(n,1,3)  = CNSAI13(n) + CNSAI13A(n) * EXP (-TIMELAG/TSLAI (NINT(ITY(N,1)),BMON,2))
        esai(n,2,1)  = CNSAI21(n) + CNSAI21A(n) * EXP (-TIMELAG/TSLAI (NINT(ITY(N,2)),BMON,2))
        esai(n,2,2)  = CNSAI22(n) + CNSAI22A(n) * EXP (-TIMELAG/TSLAI (NINT(ITY(N,2)),BMON,2))
        esai(n,2,3)  = CNSAI23(n) + CNSAI23A(n) * EXP (-TIMELAG/TSLAI (NINT(ITY(N,2)),BMON,2))
        esai(n,3,1)  = CNSAI31(n) + CNSAI31A(n) * EXP (-TIMELAG/TSLAI (NINT(ITY(N,3)),BMON,2))
        esai(n,3,2)  = CNSAI32(n) + CNSAI32A(n) * EXP (-TIMELAG/TSLAI (NINT(ITY(N,3)),BMON,2))
        esai(n,3,3)  = CNSAI33(n) + CNSAI33A(n) * EXP (-TIMELAG/TSLAI (NINT(ITY(N,3)),BMON,2))
        esai(n,4,1)  = CNSAI41(n) + CNSAI41A(n) * EXP (-TIMELAG/TSLAI (NINT(ITY(N,4)),BMON,2))
        esai(n,4,2)  = CNSAI42(n) + CNSAI42A(n) * EXP (-TIMELAG/TSLAI (NINT(ITY(N,4)),BMON,2))
        esai(n,4,3)  = CNSAI43(n) + CNSAI43A(n) * EXP (-TIMELAG/TSLAI (NINT(ITY(N,4)),BMON,2))
        elai(n,1,1)  = CNLAI11(n) + CNLAI11A(n) * EXP (-TIMELAG/TSLAI (NINT(ITY(N,1)),BMON,1))
        elai(n,1,2)  = CNLAI12(n) + CNLAI12A(n) * EXP (-TIMELAG/TSLAI (NINT(ITY(N,1)),BMON,1))
        elai(n,1,3)  = CNLAI13(n) + CNLAI13A(n) * EXP (-TIMELAG/TSLAI (NINT(ITY(N,1)),BMON,1))
        elai(n,2,1)  = CNLAI21(n) + CNLAI21A(n) * EXP (-TIMELAG/TSLAI (NINT(ITY(N,2)),BMON,1))
        elai(n,2,2)  = CNLAI22(n) + CNLAI22A(n) * EXP (-TIMELAG/TSLAI (NINT(ITY(N,2)),BMON,1))
        elai(n,2,3)  = CNLAI23(n) + CNLAI23A(n) * EXP (-TIMELAG/TSLAI (NINT(ITY(N,2)),BMON,1))
        elai(n,3,1)  = CNLAI31(n) + CNLAI31A(n) * EXP (-TIMELAG/TSLAI (NINT(ITY(N,3)),BMON,1))
        elai(n,3,2)  = CNLAI32(n) + CNLAI32A(n) * EXP (-TIMELAG/TSLAI (NINT(ITY(N,3)),BMON,1))
        elai(n,3,3)  = CNLAI33(n) + CNLAI33A(n) * EXP (-TIMELAG/TSLAI (NINT(ITY(N,3)),BMON,1))
        elai(n,4,1)  = CNLAI41(n) + CNLAI41A(n) * EXP (-TIMELAG/TSLAI (NINT(ITY(N,4)),BMON,1))
        elai(n,4,2)  = CNLAI42(n) + CNLAI42A(n) * EXP (-TIMELAG/TSLAI (NINT(ITY(N,4)),BMON,1))
        elai(n,4,3)  = CNLAI43(n) + CNLAI43A(n) * EXP (-TIMELAG/TSLAI (NINT(ITY(N,4)),BMON,1))

     END DO

     where (elai < 0.) elai = 1.e-6
     where (esai < 0.) esai = 1.e-6

  ENDIF
     
  deallocate (CNSAI11)
  deallocate (CNSAI12)
  deallocate (CNSAI13)
  deallocate (CNSAI21)
  deallocate (CNSAI22)
  deallocate (CNSAI23)
  deallocate (CNSAI31)
  deallocate (CNSAI32)
  deallocate (CNSAI33)
  deallocate (CNSAI41)
  deallocate (CNSAI42)
  deallocate (CNSAI43)
  deallocate (CNLAI11)
  deallocate (CNLAI12)
  deallocate (CNLAI13)
  deallocate (CNLAI21)
  deallocate (CNLAI22)
  deallocate (CNLAI23)
  deallocate (CNLAI31)
  deallocate (CNLAI32)
  deallocate (CNLAI33)
  deallocate (CNLAI41)
  deallocate (CNLAI42)
  deallocate (CNLAI43)

END SUBROUTINE read_prescribed_LAI

! -----------------------------------------------------------------------------------

REAL FUNCTION TSLAI (ITYP, SMONTH, LS)

  implicit none

  integer, intent (in) :: ITYP, SMONTH, LS
  
  REAL, DIMENSION (12,NUMPFT) :: LAITS, SAITS

  DATA LAITS (:, 1) /   2938.671,   1703.214,   1218.992,    989.745,   1023.555,   1243.818,   1513.783,   2135.020,   3172.746,   4790.270,   5531.646,   4010.347/
  DATA LAITS (:, 2) /  69795.391,  28537.859,   2272.353,    997.676,    723.385,   1194.828,   2072.303,   4124.063,   9919.098,  44651.770, 157986.547, 127441.438/
  DATA LAITS (:, 3) /      0.000,      0.000,      0.000,      0.000,     67.414,    200.217,    990.057,      0.000,      0.000,      0.000,      0.000,      0.000/
  DATA LAITS (:, 4) /   2015.124,   2230.545,   2539.680,   2885.798,   2460.280,   1928.556,   1523.661,   1398.747,   1492.193,   1679.618,   1911.172,   1948.057/
  DATA LAITS (:, 5) /   1392.846,   1637.630,   1853.108,   2167.965,   1997.427,   1626.865,   1516.447,   1507.380,   1408.701,   1186.145,   1045.981,   1158.065/
  DATA LAITS (:, 6) /   3411.306,   3525.113,   3349.763,   3370.082,   3189.629,   3142.207,   2573.186,   2387.604,   2470.387,   2701.219,   2850.708,   3071.156/
  DATA LAITS (:, 7) /    752.924,    404.184,    105.170,    156.788,    288.937,    747.374,   1283.792,    129.322,     85.009,    119.228,    269.670,    581.764/
  DATA LAITS (:, 8) /   1478.526,    247.332,     74.058,    144.986,    118.949,    297.104,    858.512,     39.730,    201.244,     95.643,    186.163,    469.378/
  DATA LAITS (:, 9) /    887.470,    984.260,   1197.346,    926.618,    807.862,    627.423,    532.757,    710.885,   1119.184,   1525.271,   1203.125,    955.703/
  DATA LAITS (:,10) /    427.971,     77.199,     53.993,     58.973,    155.403,    402.566,    420.889,    158.569,     53.848,     69.331,    188.980,   1133.033/
  DATA LAITS (:,11) /    693.410,    591.061,    547.764,    513.616,    501.643,    536.699,    569.842,    588.881,    588.342,    607.343,    659.312,    716.439/
  DATA LAITS (:,12) /   1576.536,    940.612,    152.002,    126.508,    108.119,    163.747,    203.844,     98.958,    123.367,    135.084,    192.891,    709.225/
  DATA LAITS (:,13) /    441.599,    144.793,     70.475,     70.259,     95.195,    210.619,    164.148,    101.477,     71.655,     63.164,     61.715,    374.426/
  DATA LAITS (:,14) /    410.132,     62.296,     55.010,     67.725,    120.379,    457.500,    564.759,    113.907,     52.373,     56.389,    273.110,    769.576/
  DATA LAITS (:,15) /    448.125,    426.954,    514.158,    617.684,    707.633,    790.015,    769.619,    650.975,    491.860,    459.468,    441.120,    473.951/
  DATA LAITS (:,16) /    420.924,     57.870,     53.152,     59.108,    111.509,    508.259,    641.459,    331.120,     53.559,     55.392,     67.203,   5811.124/
  DATA LAITS (:,17) /    760.057,    700.927,    699.944,    703.478,    767.613,    814.678,    817.368,    784.784,    759.681,    746.426,    752.065,    789.767/
  DATA LAITS (:,18) /    350.679,     75.986,     52.425,     75.525,    140.260,    441.184,    497.458,    101.640,     51.313,     64.659,    217.237,    880.205/
  DATA LAITS (:,19) /    561.547,    508.533,    605.617,    785.206,   1024.693,   1160.436,   1080.907,    914.494,    723.374,    591.482,    527.680,    601.251/
                                                                                                                                                         
  DATA SAITS (:, 1) /    230.315,    313.693,    276.539,    274.875,    258.847,    243.109,    302.606,    238.944,    225.218,    243.603,    255.616,    251.875/
  DATA SAITS (:, 2) /    229.890,    137.614,    230.997,    885.396,  34366.480,  20680.762,  15364.336, 179264.391, 106621.625,    511.944,    144.913,    112.001/
  DATA SAITS (:, 3) /    160.335,    208.863,    100.725,     61.128,      0.000,      0.000,      0.000,      0.000,      0.000,      0.000,    309.396,     67.304/
  DATA SAITS (:, 4) /    841.035,    975.111,    944.408,    825.486,    799.643,    892.486,    946.068,    891.572,    741.352,    619.509,    594.351,    667.353/
  DATA SAITS (:, 5) /    282.603,    286.126,    337.061,    405.378,    582.170,    527.165,    458.823,    379.705,    318.623,    243.743,    243.360,    261.337/
  DATA SAITS (:, 6) /    540.718,    688.578,    723.944,    581.460,    625.967,    715.761,    745.801,    668.694,    526.761,    427.821,    396.987,    432.233/
  DATA SAITS (:, 7) /   1054.735,    244.129,    339.480,      0.000,    155.208,   9998.467,   1123.824,    298.109,     70.775,      0.000,    316.463,   3617.774/
  DATA SAITS (:, 8) /    341.703,    203.918,    198.034,   2703.103,    461.967,   2371.453,    901.374,   1001.270,   1019.283,    661.610,  10851.539,   1044.646/
  DATA SAITS (:, 9) /    188.120,    225.566,    184.943,    140.310,     97.080,     95.836,     95.192,     90.871,     86.489,     89.438,    112.988,    120.050/
  DATA SAITS (:,10) /    158.143,     88.208,     50.798,     66.993,    314.352,    897.925,    525.455,     95.297,     56.950,     52.494,    225.088,    199.817/
  DATA SAITS (:,11) /    192.893,    211.842,    204.606,    188.132,    199.539,    214.700,    231.347,    218.899,    182.859,    161.711,    161.695,    174.822/
  DATA SAITS (:,12) /    187.547,    120.499,     74.804,     67.735,     63.598,     70.080,     72.900,     68.453,     68.211,     89.312,    117.100,    160.318/
  DATA SAITS (:,13) /    113.931,     79.577,     56.178,     58.576,     51.515,     61.443,     57.071,     50.591,     49.337,     62.633,     78.857,    137.055/
  DATA SAITS (:,14) /     58.460,     43.689,     45.824,     64.373,    114.885,    140.412,    119.989,     69.780,     45.139,     45.016,     69.364,     88.677/
  DATA SAITS (:,15) /    119.650,    116.520,    128.038,    132.468,    137.251,    174.534,    157.563,    109.637,     97.837,     97.455,    104.156,    106.594/
  DATA SAITS (:,16) /    114.686,     56.264,     55.149,     41.942,    279.915,   1357.135,      0.000,      0.000,     51.040,     37.574,    187.939,    250.501/
  DATA SAITS (:,17) /    170.847,    194.819,    191.218,    156.952,    150.084,    170.734,    193.540,    165.291,    142.388,    142.235,    141.490,    153.336/
  DATA SAITS (:,18) /     95.563,     33.916,     26.510,     39.303,    102.695,    784.872,    151.246,     26.027,     26.348,     53.420,    104.175,    155.367/
  DATA SAITS (:,19) /    247.494,    265.714,    277.912,    294.943,    310.404,    318.397,    333.875,    305.355,    270.572,    243.369,    236.336,    263.910/

  IF (LS == 1) TSLAI = LAITS (ITYP, SMONTH)
  IF (LS == 2) TSLAI = SAITS (ITYP, SMONTH)

  IF (TSLAI < 1.e-4) TSLAI = 1.e-4

  RETURN

END FUNCTION TSLAI


end module GEOS_CatchCNGridCompMod

