
!=============================================================================
!
! $Id$
!
! CODE DEVELOPER
!   Original data from Dorothy Koch, Hongyu Liu and Hisao Nagai.
!   LLNL implementation:  Al Franz, LLNL
!                         franz2@llnl.gov
!
! FILE
!   gmi_cosmogenic.h
!
! DESCRIPTION
!   This include file contains three tables of nuclear disintegrations per
!   gram of air, per second.  Nuclear disintegrations are called stars.  The
!   Koch values (star_cr1_be) are taken from a message from Dorothy Koch,
!   forwarded by Hongyu Liu, 1/20/00.  Dorothy received the data from D. Lal,
!   which originally was published in an article by D. Lal and B. Peters.
!   The Koch values must be multiplied by a scaling factor of 1.0d-05 and
!   then by the appropriate yield factor (Lal and Peters, fig.4).  The Nagai
!   values (star_cr2_be7 and star_cr2_be10) are taken from a message from
!   Hisao Nagai 2/04/01.  The data in the Nagai values differs from that in
!   the Koch values in that Nagai et.al. estimated the production rates for
!   Be-7 and Be-10 separately, using the neutron and proton cross-sections for
!   N-14 and O-16.  Their calculations are a modification of the nuclide
!   production model presented by Lal and Peters.  The data supplied by Nagai
!   which we use are two depth profiles at 40 deg N (one for each of Be-7 and
!   Be-10), and four latitudinal distributions (stratosphere and troposphere,
!   for each of Be-7 and Be-10).  Hence, to generate these 2D tables, the
!   latitudinal distributions were normalized to 40 deg N, and multiplied by
!   the depth profiles.  The values in the tables are subsequently bilinearly
!   interpolated by Gmimod to correspond to the current grid, and utilized in
!   calculating the amount of a cosmogenic radionuclide being produced in the
!   atmosphere in a particular gridbox.  This is currently used in the
!   calculation of the production of Be-7 and Be-10, but can be used for other
!   species as well.
!
!   References =>
!   ----------
!
!     Koch, Dorothy M., Daniel J. Jacob, and William C. Graustein, "Vertical
!     transport of tropospheric aerosols as indicated by Be-7 and Pb-210 in a
!     chemical tracer model", J. Geophys. Res., 101, 18,651-18,666, 1996.
!
!     Koch, Dorothy M., David Rind, "Beryllium 10/beryllium 7 as a tracer of
!     stratospheric transport", J. Geophys. Res., 103, 3907-3917, 1998.
!
!     Lal, D., and B. Peters, "Cosmic Ray Produced Radioactivity on the Earth",
!     Handbuch der Physik.  Berlin:  Springer-Verlag, 1967.
!
!     Nagai, Hisao, Wataru Tada, and Takayuki Kobayashi.  "Production Rates of
!     Be-7 and Be-10 in the Atmosphere."  Nuclear Instruments and Methods in
!     Physics Research B 172 (2000) 796-801.
!
!     Dorothy Koch : koch@pele.geology.yale.edu
!     Hongyu Liu   : hyl@sol.harvard.edu
!     Hisao Nagai  : hnagai@chs.nihon-u.ac.jp
!
!=============================================================================


      integer, parameter ::  &
     &  NUM_CR1_LAT  = 19,    & ! # of different lats. at which star values exist
     &  NUM_CR1_PRES = 34   ! # of different pres. at which star values exist

      integer, parameter ::  &
     &  NUM_CR2_LAT  = 19,    & ! # of different lats. at which star values exist
     &  NUM_CR2_PRES = 30   ! # of different pres. at which star values exist


!     -------------------------------------------
!     CR1_LAT  : latitude coordinate values (deg)
!     CR1_PRES : pressure coordinate values (mb)
!     -------------------------------------------

      real*8, parameter :: CR1_LAT(NUM_CR1_LAT) =  &
     &  (/ -91.0d0, -80.0d0, -70.0d0, -60.0d0, -50.0d0,  &
     &     -40.0d0, -30.0d0, -20.0d0, -10.0d0,   0.0d0,  &
     &      10.0d0,  20.0d0,  30.0d0,  40.0d0,  50.0d0,  &
     &      60.0d0,  70.0d0,  80.0d0,  91.0d0 /)

      real*8, parameter :: CR1_PRES(NUM_CR1_PRES) =  &
     &  (/   0.0d0,  50.0d0,   70.0d0,   90.0d0, 110.0d0,  &
     &     130.0d0, 150.0d0,  170.0d0,  190.0d0, 210.0d0,  &
     &     230.0d0, 250.0d0,  270.0d0,  290.0d0, 313.0d0,  &
     &     338.0d0, 364.0d0,  392.0d0,  420.0d0, 451.0d0,  &
     &     485.0d0, 518.0d0,  555.0d0,  592.0d0, 633.0d0,  &
     &     680.0d0, 725.0d0,  772.0d0,  822.0d0, 875.0d0,  &
     &     930.0d0, 985.0d0, 1030.0d0, 1200.0d0 /)


!     -------------------------------------------
!     CR2_LAT  : latitude coordinate values (deg)
!     CR2_PRES : pressure coordinate values (mb)
!     -------------------------------------------

      real*8, parameter :: CR2_LAT(NUM_CR2_LAT) =  &
     &  (/ -91.0d0, -80.0d0, -70.0d0, -60.0d0, -50.0d0,  &
     &     -40.0d0, -30.0d0, -20.0d0, -10.0d0,   0.0d0,  &
     &      10.0d0,  20.0d0,  30.0d0,  40.0d0,  50.0d0,  &
     &      60.0d0,  70.0d0,  80.0d0,  91.0d0 /)

      real*8, parameter :: CR2_PRES(NUM_CR2_PRES) =  &
     &  (/   0.000d0,   2.45225d0,  14.7135d0,   44.1405d0,   68.6630d0,  &
     &     107.899d0, 147.135d0,   186.371d0,   225.607d0,   264.843d0,  &
     &     304.079d0, 343.315d0,   382.551d0,   421.787d0,   461.023d0,  &
     &     500.259d0, 539.495d0,   578.731d0,   617.967d0,   657.203d0,  &
     &     696.439d0, 735.675d0,   774.911d0,   814.147d0,   853.383d0,  &
     &     892.619d0, 931.855d0,   971.091d0,  1006.89d0,   1200.0d0 /)


      integer :: lset


!     --------------------------------------------------------------
!     star_cr1_be   : 2D table of known nuclear disintegration values
!                     for Be-7 and Be-10 (star*10^5/g air/s)
!
!     star_cr2_be7  : 2D table of known nuclear disintegration values
!                     for Be-7  (star*10^5/g air/s)
!     star_cr2_be10 : 2D table of known nuclear disintegration values
!                     for Be-10 (star*10^5/g air/s)
!     --------------------------------------------------------------

      real*8, save :: star_cr1_be  (NUM_CR1_LAT, NUM_CR1_PRES)

      real*8, save :: star_cr2_be7 (NUM_CR2_LAT, NUM_CR2_PRES)
      real*8, save :: star_cr2_be10(NUM_CR2_LAT, NUM_CR2_PRES)


!     ------------------------------------
!     Beryllium-7 and Beryllium-10 (Koch).
!     ------------------------------------

!     0.0 mb
      data  (star_cr1_be(lset,1), lset=1,NUM_CR1_LAT) /  &

     &  3000.0d0, 3000.0d0, 3000.0d0, 1700.0d0,  910.0d0,  &
     &   500.0d0,  285.0d0,  188.0d0,  156.0d0,  150.0d0,  &
     &   156.0d0,  188.0d0,  285.0d0,  500.0d0,  910.0d0,  &
     &  1700.0d0, 3000.0d0, 3000.0d0, 3000.0d0 /

!     50.0 mb
      data  (star_cr1_be(lset,2), lset=1,NUM_CR1_LAT) /  &

     &  1800.0d0, 1800.0d0, 1800.0d0, 1800.0d0, 1390.0d0,  &
     &   880.0d0,  590.0d0,  390.0d0,  310.0d0,  280.0d0,  &
     &   310.0d0,  390.0d0,  590.0d0,  880.0d0, 1390.0d0,  &
     &  1800.0d0, 1800.0d0, 1800.0d0, 1800.0d0 /

!     70.0 mb
      data  (star_cr1_be(lset,3), lset=1,NUM_CR1_LAT) /  &

     &  1450.0d0, 1450.0d0, 1450.0d0, 1450.0d0, 1280.0d0,  &
     &   880.0d0,  620.0d0,  400.0d0,  330.0d0,  310.0d0,  &
     &   330.0d0,  400.0d0,  620.0d0,  880.0d0, 1280.0d0,  &
     &  1450.0d0, 1450.0d0, 1450.0d0, 1450.0d0 /

!     90.0 mb
      data  (star_cr1_be(lset,4), lset=1,NUM_CR1_LAT) /  &

     &  1180.0d0, 1180.0d0, 1180.0d0, 1180.0d0, 1100.0d0,  &
     &   780.0d0,  570.0d0,  375.0d0,  310.0d0,  285.0d0,  &
     &   310.0d0,  375.0d0,  570.0d0,  780.0d0, 1100.0d0,  &
     &  1180.0d0, 1180.0d0, 1180.0d0, 1180.0d0 /

!     110.0 mb
      data  (star_cr1_be(lset,5), lset=1,NUM_CR1_LAT) /  &

     &  1000.0d0, 1000.0d0, 1000.0d0, 1000.0d0,  950.0d0,  &
     &   680.0d0,  510.0d0,  330.0d0,  275.0d0,  255.0d0,  &
     &   275.0d0,  330.0d0,  510.0d0,  680.0d0,  950.0d0,  &
     &  1000.0d0, 1000.0d0, 1000.0d0, 1000.0d0 /

!     130.0 mb
      data  (star_cr1_be(lset,6), lset=1,NUM_CR1_LAT) /  &

     &   875.0d0,  875.0d0,  875.0d0,  875.0d0,  820.0d0,  &
     &   600.0d0,  450.0d0,  292.0d0,  245.0d0,  230.0d0,  &
     &   245.0d0,  292.0d0,  450.0d0,  600.0d0,  820.0d0,  &
     &   875.0d0,  875.0d0,  875.0d0,  875.0d0 /

!     150.0 mb
      data  (star_cr1_be(lset,7), lset=1,NUM_CR1_LAT) /  &

     &   750.0d0,  750.0d0,  750.0d0,  750.0d0,  730.0d0,  &
     &   530.0d0,  400.0d0,  260.0d0,  215.0d0,  205.0d0,  &
     &   215.0d0,  260.0d0,  400.0d0,  530.0d0,  730.0d0,  &
     &   750.0d0,  750.0d0,  750.0d0,  750.0d0 /

!     170.0 mb
      data  (star_cr1_be(lset,8), lset=1,NUM_CR1_LAT) /  &

     &   650.0d0,  650.0d0,  650.0d0,  650.0d0,  630.0d0,  &
     &   480.0d0,  355.0d0,  235.0d0,  195.0d0,  182.0d0,  &
     &   195.0d0,  235.0d0,  355.0d0,  480.0d0,  630.0d0,  &
     &   650.0d0,  650.0d0,  650.0d0,  650.0d0 /

!     190.0 mb
      data  (star_cr1_be(lset,9), lset=1,NUM_CR1_LAT) /  &

     &   550.0d0,  550.0d0,  550.0d0,  550.0d0,  543.0d0,  &
     &   410.0d0,  315.0d0,  208.0d0,  173.0d0,  160.0d0,  &
     &   173.0d0,  208.0d0,  315.0d0,  410.0d0,  543.0d0,  &
     &   550.0d0,  550.0d0,  550.0d0,  550.0d0 /

!     210.0 mb
      data  (star_cr1_be(lset,10), lset=1,NUM_CR1_LAT) /  &

     &   500.0d0,  500.0d0,  500.0d0,  500.0d0,  480.0d0,  &
     &   370.0d0,  280.0d0,  185.0d0,  152.0d0,  148.0d0,  &
     &   152.0d0,  185.0d0,  280.0d0,  370.0d0,  480.0d0,  &
     &   500.0d0,  500.0d0,  500.0d0,  500.0d0 /

!     230.0 mb
      data  (star_cr1_be(lset,11), lset=1,NUM_CR1_LAT) /  &

     &   430.0d0,  430.0d0,  430.0d0,  430.0d0,  425.0d0,  &
     &   320.0d0,  250.0d0,  167.0d0,  139.0d0,  130.0d0,  &
     &   139.0d0,  167.0d0,  250.0d0,  320.0d0,  425.0d0,  &
     &   430.0d0,  430.0d0,  430.0d0,  430.0d0 /

!     250.0 mb
      data  (star_cr1_be(lset,12), lset=1,NUM_CR1_LAT) /  &

     &   375.0d0,  375.0d0,  375.0d0,  375.0d0,  365.0d0,  &
     &   285.0d0,  215.0d0,  148.0d0,  123.0d0,  116.0d0,  &
     &   123.0d0,  148.0d0,  215.0d0,  285.0d0,  365.0d0,  &
     &   375.0d0,  375.0d0,  375.0d0,  375.0d0 /

!     270.0 mb
      data  (star_cr1_be(lset,13), lset=1,NUM_CR1_LAT) /  &

     &   330.0d0,  330.0d0,  330.0d0,  330.0d0,  320.0d0,  &
     &   250.0d0,  198.0d0,  130.0d0,  110.0d0,  104.0d0,  &
     &   110.0d0,  130.0d0,  198.0d0,  250.0d0,  320.0d0,  &
     &   330.0d0,  330.0d0,  330.0d0,  330.0d0 /

!     290.0 mb
      data  (star_cr1_be(lset,14), lset=1,NUM_CR1_LAT) /  &

     &   288.0d0,  288.0d0,  288.0d0,  288.0d0,  280.0d0,  &
     &   222.0d0,  170.0d0,  118.0d0,   99.0d0,   93.0d0,  &
     &    99.0d0,  118.0d0,  170.0d0,  222.0d0,  280.0d0,  &
     &   288.0d0,  288.0d0,  288.0d0,  288.0d0 /

!     313.0 mb
      data  (star_cr1_be(lset,15), lset=1,NUM_CR1_LAT) /  &

     &   250.0d0,  250.0d0,  250.0d0,  250.0d0,  235.0d0,  &
     &   190.0d0,  145.0d0,  100.0d0,   84.0d0,   80.0d0,  &
     &    84.0d0,  100.0d0,  145.0d0,  190.0d0,  235.0d0,  &
     &   250.0d0,  250.0d0,  250.0d0,  250.0d0 /

!     338.0 mb
      data  (star_cr1_be(lset,16), lset=1,NUM_CR1_LAT) /  &

     &   218.0d0,  218.0d0,  218.0d0,  218.0d0,  210.0d0,  &
     &   168.0d0,  129.0d0,   88.0d0,   74.5d0,   72.0d0,  &
     &    74.5d0,   88.0d0,  129.0d0,  168.0d0,  210.0d0,  &
     &   218.0d0,  218.0d0,  218.0d0,  218.0d0 /

!     364.0 mb
      data  (star_cr1_be(lset,17), lset=1,NUM_CR1_LAT) /  &

     &   178.0d0,  178.0d0,  178.0d0,  178.0d0,  171.0d0,  &
     &   138.0d0,  108.0d0,   73.5d0,   62.5d0,   59.5d0,  &
     &    62.5d0,   73.5d0,  108.0d0,  138.0d0,  171.0d0,  &
     &   178.0d0,  178.0d0,  178.0d0,  178.0d0 /

!     392.0 mb
      data  (star_cr1_be(lset,18), lset=1,NUM_CR1_LAT) /  &

     &   150.0d0,  150.0d0,  150.0d0,  150.0d0,  148.0d0,  &
     &   115.0d0,   90.0d0,   64.0d0,   53.0d0,   50.0d0,  &
     &    53.0d0,   64.0d0,   90.0d0,  115.0d0,  148.0d0,  &
     &   150.0d0,  150.0d0,  150.0d0,  150.0d0 /

!     420.0 mb
      data  (star_cr1_be(lset,19), lset=1,NUM_CR1_LAT) /  &

     &   128.0d0,  128.0d0,  128.0d0,  128.0d0,  122.0d0,  &
     &    98.0d0,   76.0d0,   52.5d0,   46.5d0,   45.0d0,  &
     &    46.5d0,   52.5d0,   76.0d0,   98.0d0,  122.0d0,  &
     &   128.0d0,  128.0d0,  128.0d0,  128.0d0 /

!     451.0 mb
      data  (star_cr1_be(lset,20), lset=1,NUM_CR1_LAT) /  &

     &   102.0d0,  102.0d0,  102.0d0,  102.0d0,   98.0d0,  &
     &    77.0d0,   61.0d0,   45.0d0,   37.5d0,   36.5d0,  &
     &    37.5d0,   45.0d0,   61.0d0,   77.0d0,   98.0d0,  &
     &   102.0d0,  102.0d0,  102.0d0,  102.0d0 /

!     485.0 mb
      data  (star_cr1_be(lset,21), lset=1,NUM_CR1_LAT) /  &

     &    85.0d0,   85.0d0,   85.0d0,   85.0d0,   81.0d0,  &
     &    65.0d0,   51.5d0,   37.5d0,   32.0d0,   30.8d0,  &
     &    32.0d0,   37.5d0,   51.5d0,   65.0d0,   81.0d0,  &
     &    85.0d0,   85.0d0,   85.0d0,   85.0d0 /

!     518.0 mb
      data  (star_cr1_be(lset,22), lset=1,NUM_CR1_LAT) /  &

     &    69.5d0,   69.5d0,   69.5d0,   69.5d0,   67.5d0,  &
     &    54.0d0,   40.5d0,   32.0d0,   26.5d0,   25.5d0,  &
     &    26.5d0,   32.0d0,   40.5d0,   54.0d0,   67.5d0,  &
     &    69.5d0,   69.5d0,   69.5d0,   69.5d0 /

!     555.0 mb
      data  (star_cr1_be(lset,23), lset=1,NUM_CR1_LAT) /  &

     &    55.0d0,   55.0d0,   55.0d0,   55.0d0,   53.5d0,  &
     &    42.0d0,   33.0d0,   25.5d0,   21.6d0,   20.5d0,  &
     &    21.6d0,   25.5d0,   33.0d0,   42.0d0,   53.5d0,  &
     &    55.0d0,   55.0d0,   55.0d0,   55.0d0 /

!     592.0 mb
      data  (star_cr1_be(lset,24), lset=1,NUM_CR1_LAT) /  &

     &    43.0d0,   43.0d0,   43.0d0,   43.0d0,   41.0d0,  &
     &    33.5d0,   26.0d0,   20.0d0,   17.3d0,   16.8d0,  &
     &    17.3d0,   20.0d0,   26.0d0,   33.5d0,   41.0d0,  &
     &    43.0d0,   43.0d0,   43.0d0,   43.0d0 /

!     633.0 mb
      data  (star_cr1_be(lset,25), lset=1,NUM_CR1_LAT) /  &

     &    33.5d0,   33.5d0,   33.5d0,   33.5d0,   32.5d0,  &
     &    26.8d0,   20.5d0,   15.3d0,   13.8d0,   13.0d0,  &
     &    13.8d0,   15.3d0,   20.5d0,   26.8d0,   32.5d0,  &
     &    33.5d0,   33.5d0,   33.5d0,   33.5d0 /

!     680.0 mb
      data  (star_cr1_be(lset,26), lset=1,NUM_CR1_LAT) /  &

     &    25.8d0,   25.8d0,   25.8d0,   25.8d0,   24.5d0,  &
     &    20.0d0,   15.8d0,   12.6d0,   10.6d0,   10.1d0,  &
     &    10.6d0,   12.6d0,   15.8d0,   20.0d0,   24.5d0,  &
     &    25.8d0,   25.8d0,   25.8d0,   25.8d0 /

!     725.0 mb
      data  (star_cr1_be(lset,27), lset=1,NUM_CR1_LAT) /  &

     &    18.50d0,  18.50d0,  18.50d0,  18.50d0,  17.80d0,  &
     &    14.80d0,  11.60d0,   9.40d0,   8.15d0,   7.70d0,  &
     &     8.15d0,   9.40d0,  11.60d0,  14.80d0,  17.80d0,  &
     &    18.50d0,  18.50d0,  18.50d0,  18.50d0 /

!     772.0 mb
      data  (star_cr1_be(lset,28), lset=1,NUM_CR1_LAT) /  &

     &    13.20d0,  13.20d0,  13.20d0,  13.20d0,  13.10d0,  &
     &    11.00d0,   8.22d0,   6.85d0,   5.85d0,   5.70d0,  &
     &     5.85d0,   6.85d0,   8.22d0,  11.00d0,  13.10d0,  &
     &    13.20d0,  13.20d0,  13.20d0,  13.20d0 /

!     822.0 mb
      data  (star_cr1_be(lset,29), lset=1,NUM_CR1_LAT) /  &

     &     9.20d0,   9.20d0,   9.20d0,   9.20d0,   9.00d0,  &
     &     7.60d0,   6.00d0,   4.85d0,   4.20d0,   3.90d0,  &
     &     4.20d0,   4.85d0,   6.00d0,   7.60d0,   9.00d0,  &
     &     9.20d0,   9.20d0,   9.20d0,   9.20d0 /

!     875.0 mb
      data  (star_cr1_be(lset,30), lset=1,NUM_CR1_LAT) /  &

     &     6.25d0,   6.25d0,   6.25d0,   6.25d0,   5.90d0,  &
     &     5.30d0,   4.20d0,   3.35d0,   3.05d0,   3.00d0,  &
     &     3.05d0,   3.35d0,   4.20d0,   5.30d0,   5.90d0,  &
     &     6.25d0,   6.25d0,   6.25d0,   6.25d0 /

!     930.0 mb
      data  (star_cr1_be(lset,31), lset=1,NUM_CR1_LAT) /  &

     &     4.10d0,   4.10d0,   4.10d0,   4.10d0,   3.90d0,  &
     &     3.40d0,   2.90d0,   2.32d0,   2.10d0,   2.05d0,  &
     &     2.10d0,   2.32d0,   2.90d0,   3.40d0,   3.90d0,  &
     &     4.10d0,   4.10d0,   4.10d0,   4.10d0 /

!     985.0 mb
      data  (star_cr1_be(lset,32), lset=1,NUM_CR1_LAT) /  &

     &     2.65d0,   2.65d0,   2.65d0,   2.65d0,   2.75d0,  &
     &     2.40d0,   2.03d0,   1.65d0,   1.43d0,   1.45d0,  &
     &     1.43d0,   1.65d0,   2.03d0,   2.40d0,   2.75d0,  &
     &     2.65d0,   2.65d0,   2.65d0,   2.65d0 /

!     1030.0 mb
      data  (star_cr1_be(lset,33), lset=1,NUM_CR1_LAT) /  &

     &     1.80d0,   1.80d0,   1.80d0,   1.80d0,   1.80d0,  &
     &     1.68d0,   1.50d0,   1.21d0,   1.08d0,   1.04d0,  &
     &     1.08d0,   1.21d0,   1.50d0,   1.68d0,   1.80d0,  &
     &     1.80d0,   1.80d0,   1.80d0,   1.80d0 /

!     1200.0 mb
      data  (star_cr1_be(lset,34), lset=1,NUM_CR1_LAT) /  &

     &     1.80d0,   1.80d0,   1.80d0,   1.80d0,   1.80d0,  &
     &     1.68d0,   1.50d0,   1.21d0,   1.08d0,   1.04d0,  &
     &     1.08d0,   1.21d0,   1.50d0,   1.68d0,   1.80d0,  &
     &     1.80d0,   1.80d0,   1.80d0,   1.80d0 /


!     --------------------
!     Beryllium-7 (Nagai).
!     --------------------

!     0.00000 mb
      data  (star_cr2_be7(lset,1), lset=1,NUM_CR2_LAT) /  &

     &  7.77889d-4,  7.71783d-4,  7.60182d-4,  6.65541d-4,  4.91279d-4,  &
     &  2.39900d-4,  1.14607d-4,  6.60045d-5,  4.99400d-5,  4.67955d-5,  &
     &  4.99400d-5,  6.60045d-5,  1.14607d-4,  2.39900d-4,  4.91279d-4,  &
     &  6.65541d-4,  7.60182d-4,  7.71783d-4,  7.77889d-4 /

!     2.45225 mb
      data  (star_cr2_be7(lset,2), lset=1,NUM_CR2_LAT) /  &

     &  7.77889d-4,  7.71783d-4,  7.60182d-4,  6.65541d-4,  4.91279d-4,  &
     &  2.39900d-4,  1.14607d-4,  6.60045d-5,  4.99400d-5,  4.67955d-5,  &
     &  4.99400d-5,  6.60045d-5,  1.14607d-4,  2.39900d-4,  4.91279d-4,  &
     &  6.65541d-4,  7.60182d-4,  7.71783d-4,  7.77889d-4 /

!     14.7135 mb
      data  (star_cr2_be7(lset,3), lset=1,NUM_CR2_LAT) /  &

     &  8.27500d-4,  8.21005d-4,  8.08664d-4,  7.07987d-4,  5.22611d-4,  &
     &  2.55200d-4,  1.21917d-4,  7.02141d-5,  5.31250d-5,  4.97799d-5,  &
     &  5.31250d-5,  7.02141d-5,  1.21917d-4,  2.55200d-4,  5.22611d-4,  &
     &  7.07987d-4,  8.08664d-4,  8.21005d-4,  8.27500d-4 /

!     44.1405 mb
      data  (star_cr2_be7(lset,4), lset=1,NUM_CR2_LAT) /  &

     &  8.22636d-4,  8.16179d-4,  8.03911d-4,  7.03825d-4,  5.19539d-4,  &
     &  2.53700d-4,  1.21200d-4,  6.98014d-5,  5.28127d-5,  4.94873d-5,  &
     &  5.28127d-5,  6.98014d-5,  1.21200d-4,  2.53700d-4,  5.19539d-4,  &
     &  7.03825d-4,  8.03911d-4,  8.16179d-4,  8.22636d-4 /

!     68.6630 mb
      data  (star_cr2_be7(lset,5), lset=1,NUM_CR2_LAT) /  &

     &  8.15503d-4,  8.09102d-4,  7.96939d-4,  6.97722d-4,  5.15034d-4,  &
     &  2.51500d-4,  1.20149d-4,  6.91961d-5,  5.23548d-5,  4.90582d-5,  &
     &  5.23548d-5,  6.91961d-5,  1.20149d-4,  2.51500d-4,  5.15034d-4,  &
     &  6.97722d-4,  7.96939d-4,  8.09102d-4,  8.15503d-4 /

!     107.899 mb
      data  (star_cr2_be7(lset,6), lset=1,NUM_CR2_LAT) /  &

     &  7.07850d-4,  7.02294d-4,  6.91737d-4,  6.05617d-4,  4.47045d-4,  &
     &  2.18300d-4,  1.04288d-4,  6.00617d-5,  1.32643d-4,  1.32947d-4,  &
     &  1.32643d-4,  6.00617d-5,  1.04288d-4,  2.18300d-4,  4.47045d-4,  &
     &  6.05617d-4,  6.91737d-4,  7.02294d-4,  7.07850d-4 /

!     147.135 mb
      data  (star_cr2_be7(lset,7), lset=1,NUM_CR2_LAT) /  &

     &  6.07655d-4,  6.02885d-4,  5.93823d-4,  5.19893d-4,  3.83767d-4,  &
     &  1.87400d-4,  8.95265d-5,  5.15600d-5,  1.13868d-4,  1.14128d-4,  &
     &  1.13868d-4,  5.15600d-5,  8.95265d-5,  1.87400d-4,  3.83767d-4,  &
     &  5.19893d-4,  5.93823d-4,  6.02885d-4,  6.07655d-4 /

!     186.371 mb
      data  (star_cr2_be7(lset,8), lset=1,NUM_CR2_LAT) /  &

     &  5.35346d-4,  5.31144d-4,  5.23160d-4,  4.58028d-4,  3.38100d-4,  &
     &  1.65100d-4,  7.88732d-5,  1.18591d-4,  1.00318d-4,  1.00548d-4,  &
     &  1.00318d-4,  1.18591d-4,  7.88732d-5,  1.65100d-4,  3.38100d-4,  &
     &  4.58028d-4,  5.23160d-4,  5.31144d-4,  5.35346d-4 /

!     225.607 mb
      data  (star_cr2_be7(lset,9), lset=1,NUM_CR2_LAT) /  &

     &  4.51364d-4,  4.47821d-4,  4.41089d-4,  3.86175d-4,  2.85061d-4,  &
     &  1.39200d-4,  6.65000d-5,  9.99871d-5,  8.45806d-5,  8.47742d-5,  &
     &  8.45806d-5,  9.99871d-5,  6.65000d-5,  1.39200d-4,  2.85061d-4,  &
     &  3.86175d-4,  4.41089d-4,  4.47821d-4,  4.51364d-4 /

!     264.843 mb
      data  (star_cr2_be7(lset,10), lset=1,NUM_CR2_LAT) /  &

     &  3.57654d-4,  3.54847d-4,  3.49513d-4,  3.05999d-4,  2.25878d-4,  &
     &  1.10300d-4,  1.03736d-4,  7.92283d-5,  6.70204d-5,  6.71738d-5,  &
     &  6.70204d-5,  7.92283d-5,  1.03736d-4,  1.10300d-4,  2.25878d-4,  &
     &  3.05999d-4,  3.49513d-4,  3.54847d-4,  3.57654d-4 /

!     304.079 mb
      data  (star_cr2_be7(lset,11), lset=1,NUM_CR2_LAT) /  &

     &  3.90984d-5,  4.15832d-5,  4.61630d-5,  5.21069d-5,  7.23748d-5,  &
     &  8.76000d-5,  8.23869d-5,  6.29229d-5,  5.32275d-5,  5.33493d-5,  &
     &  5.32275d-5,  6.29229d-5,  8.23869d-5,  8.76000d-5,  7.23748d-5,  &
     &  5.21069d-5,  4.61630d-5,  4.15832d-5,  3.90984d-5 /

!     343.315 mb
      data  (star_cr2_be7(lset,12), lset=1,NUM_CR2_LAT) /  &

     &  3.11404d-5,  3.31194d-5,  3.67670d-5,  4.15011d-5,  5.76437d-5,  &
     &  6.97700d-5,  6.56179d-5,  5.01157d-5,  4.23936d-5,  4.24906d-5,  &
     &  4.23936d-5,  5.01157d-5,  6.56179d-5,  6.97700d-5,  5.76437d-5,  &
     &  4.15011d-5,  3.67670d-5,  3.31194d-5,  3.11404d-5 /

!     382.551 mb
      data  (star_cr2_be7(lset,13), lset=1,NUM_CR2_LAT) /  &

     &  2.48695d-5,  2.64500d-5,  2.93630d-5,  3.31438d-5,  4.60356d-5,  &
     &  5.57200d-5,  5.24041d-5,  4.00236d-5,  3.38566d-5,  3.39340d-5,  &
     &  3.38566d-5,  4.00236d-5,  5.24041d-5,  5.57200d-5,  4.60356d-5,  &
     &  3.31438d-5,  2.93630d-5,  2.64500d-5,  2.48695d-5 /

!     421.787 mb
      data  (star_cr2_be7(lset,14), lset=1,NUM_CR2_LAT) /  &

     &  2.00268d-5,  2.12995d-5,  2.36453d-5,  2.66899d-5,  3.70714d-5,  &
     &  4.48700d-5,  4.21998d-5,  3.22300d-5,  2.72639d-5,  2.73263d-5,  &
     &  2.72639d-5,  3.22300d-5,  4.21998d-5,  4.48700d-5,  3.70714d-5,  &
     &  2.66899d-5,  2.36453d-5,  2.12995d-5,  2.00268d-5 /

!     461.023 mb
      data  (star_cr2_be7(lset,15), lset=1,NUM_CR2_LAT) /  &

     &  1.62866d-5,  1.73216d-5,  1.92293d-5,  2.17053d-5,  3.01479d-5,  &
     &  3.64900d-5,  3.43185d-5,  2.62107d-5,  2.21720d-5,  2.22228d-5,  &
     &  2.21720d-5,  2.62107d-5,  3.43185d-5,  3.64900d-5,  3.01479d-5,  &
     &  2.17053d-5,  1.92293d-5,  1.73216d-5,  1.62866d-5 /

!     500.259 mb
      data  (star_cr2_be7(lset,16), lset=1,NUM_CR2_LAT) /  &

     &  1.33051d-5,  1.41506d-5,  1.57091d-5,  1.77318d-5,  2.46289d-5,  &
     &  2.98100d-5,  2.80360d-5,  2.14125d-5,  1.81131d-5,  1.81546d-5,  &
     &  1.81131d-5,  2.14125d-5,  2.80360d-5,  2.98100d-5,  2.46289d-5,  &
     &  1.77318d-5,  1.57091d-5,  1.41506d-5,  1.33051d-5 /

!     539.495 mb
      data  (star_cr2_be7(lset,17), lset=1,NUM_CR2_LAT) /  &

     &  1.09217d-5,  1.16158d-5,  1.28951d-5,  1.45554d-5,  2.02170d-5,  &
     &  2.44700d-5,  2.30138d-5,  1.75768d-5,  1.48685d-5,  1.49025d-5,  &
     &  1.48685d-5,  1.75768d-5,  2.30138d-5,  2.44700d-5,  2.02170d-5,  &
     &  1.45554d-5,  1.28951d-5,  1.16158d-5,  1.09217d-5 /

!     578.731 mb
      data  (star_cr2_be7(lset,18), lset=1,NUM_CR2_LAT) /  &

     &  9.00246d-6,  9.57458d-6,  1.06291d-5,  1.19977d-5,  1.66644d-5,  &
     &  2.01700d-5,  1.89697d-5,  1.44881d-5,  1.22557d-5,  1.22837d-5,  &
     &  1.22557d-5,  1.44881d-5,  1.89697d-5,  2.01700d-5,  1.66644d-5,  &
     &  1.19977d-5,  1.06291d-5,  9.57458d-6,  9.00246d-6 /

!     617.967 mb
      data  (star_cr2_be7(lset,19), lset=1,NUM_CR2_LAT) /  &

     &  7.11895d-6,  7.57137d-6,  8.40524d-6,  9.48750d-6,  1.31778d-5,  &
     &  1.59500d-5,  1.50008d-5,  1.14569d-5,  9.69153d-6,  9.71371d-6,  &
     &  9.69153d-6,  1.14569d-5,  1.50008d-5,  1.59500d-5,  1.31778d-5,  &
     &  9.48750d-6,  8.40524d-6,  7.57137d-6,  7.11895d-6 /

!     657.203 mb
      data  (star_cr2_be7(lset,20), lset=1,NUM_CR2_LAT) /  &

     &  5.40058d-6,  5.74380d-6,  6.37639d-6,  7.19741d-6,  9.99697d-6,  &
     &  1.21000d-5,  1.13799d-5,  8.69141d-6,  7.35220d-6,  7.36902d-6,  &
     &  7.35220d-6,  8.69141d-6,  1.13799d-5,  1.21000d-5,  9.99697d-6,  &
     &  7.19741d-6,  6.37639d-6,  5.74380d-6,  5.40058d-6 /

!     696.439 mb
      data  (star_cr2_be7(lset,21), lset=1,NUM_CR2_LAT) /  &

     &  4.10310d-6,  4.36386d-6,  4.84448d-6,  5.46825d-6,  7.59522d-6,  &
     &  9.19300d-6,  8.64592d-6,  6.60331d-6,  5.58585d-6,  5.59863d-6,  &
     &  5.58585d-6,  6.60331d-6,  8.64592d-6,  9.19300d-6,  7.59522d-6,  &
     &  5.46825d-6,  4.84448d-6,  4.36386d-6,  4.10310d-6 /

!     735.675 mb
      data  (star_cr2_be7(lset,22), lset=1,NUM_CR2_LAT) /  &

     &  3.12475d-6,  3.32333d-6,  3.68935d-6,  4.16439d-6,  5.78420d-6,  &
     &  7.00100d-6,  6.58437d-6,  5.02881d-6,  4.25394d-6,  4.26368d-6,  &
     &  4.25394d-6,  5.02881d-6,  6.58437d-6,  7.00100d-6,  5.78420d-6,  &
     &  4.16439d-6,  3.68935d-6,  3.32333d-6,  3.12475d-6 /

!     774.911 mb
      data  (star_cr2_be7(lset,23), lset=1,NUM_CR2_LAT) /  &

     &  2.38384d-6,  2.53534d-6,  2.81457d-6,  3.17697d-6,  4.41271d-6,  &
     &  5.34100d-6,  5.02315d-6,  3.83643d-6,  3.24530d-6,  3.25272d-6,  &
     &  3.24530d-6,  3.83643d-6,  5.02315d-6,  5.34100d-6,  4.41271d-6,  &
     &  3.17697d-6,  2.81457d-6,  2.53534d-6,  2.38384d-6 /

!     814.147 mb
      data  (star_cr2_be7(lset,24), lset=1,NUM_CR2_LAT) /  &

     &  1.83084d-6,  1.94719d-6,  2.16165d-6,  2.43998d-6,  3.38905d-6,  &
     &  4.10200d-6,  3.85789d-6,  2.94646d-6,  2.49246d-6,  2.49816d-6,  &
     &  2.49246d-6,  2.94646d-6,  3.85789d-6,  4.10200d-6,  3.38905d-6,  &
     &  2.43998d-6,  2.16165d-6,  1.94719d-6,  1.83084d-6 /

!     853.383 mb
      data  (star_cr2_be7(lset,25), lset=1,NUM_CR2_LAT) /  &

     &  1.41977d-6,  1.51000d-6,  1.67631d-6,  1.89215d-6,  2.62813d-6,  &
     &  3.18100d-6,  2.99170d-6,  2.28491d-6,  1.93284d-6,  1.93726d-6,  &
     &  1.93284d-6,  2.28491d-6,  2.99170d-6,  3.18100d-6,  2.62813d-6,  &
     &  1.89215d-6,  1.67631d-6,  1.51000d-6,  1.41977d-6 /

!     892.619 mb
      data  (star_cr2_be7(lset,26), lset=1,NUM_CR2_LAT) /  &

     &  1.10824d-6,  1.17867d-6,  1.30848d-6,  1.47696d-6,  2.05144d-6,  &
     &  2.48300d-6,  2.33524d-6,  1.78353d-6,  1.50872d-6,  1.51217d-6,  &
     &  1.50872d-6,  1.78353d-6,  2.33524d-6,  2.48300d-6,  2.05144d-6,  &
     &  1.47696d-6,  1.30848d-6,  1.17867d-6,  1.10824d-6 /

!     931.855 mb
      data  (star_cr2_be7(lset,27), lset=1,NUM_CR2_LAT) /  &

     &  8.71235d-7,  9.26603d-7,  1.02865d-6,  1.16110d-6,  1.61273d-6,  &
     &  1.95200d-6,  1.83584d-6,  1.40212d-6,  1.18607d-6,  1.18879d-6,  &
     &  1.18607d-6,  1.40212d-6,  1.83584d-6,  1.95200d-6,  1.61273d-6,  &
     &  1.16110d-6,  1.02865d-6,  9.26603d-7,  8.71235d-7 /

!     971.091 mb
      data  (star_cr2_be7(lset,28), lset=1,NUM_CR2_LAT) /  &

     &  6.91364d-7,  7.35301d-7,  8.16283d-7,  9.21388d-7,  1.27978d-6,  &
     &  1.54900d-6,  1.45682d-6,  1.11264d-6,  9.41203d-7,  9.43357d-7,  &
     &  9.41203d-7,  1.11264d-6,  1.45682d-6,  1.54900d-6,  1.27978d-6,  &
     &  9.21388d-7,  8.16283d-7,  7.35301d-7,  6.91364d-7 /

!     1006.89 mb
      data  (star_cr2_be7(lset,29), lset=1,NUM_CR2_LAT) /  &

     &  5.54341d-7,  5.89570d-7,  6.54502d-7,  7.38776d-7,  1.02614d-6,  &
     &  1.24200d-6,  1.16809d-6,  8.92126d-7,  7.54664d-7,  7.56390d-7,  &
     &  7.54664d-7,  8.92126d-7,  1.16809d-6,  1.24200d-6,  1.02614d-6,  &
     &  7.38776d-7,  6.54502d-7,  5.89570d-7,  5.54341d-7 /

!     1200.00 mb
      data  (star_cr2_be7(lset,30), lset=1,NUM_CR2_LAT) /  &

     &  5.54341d-7,  5.89570d-7,  6.54502d-7,  7.38776d-7,  1.02614d-6,  &
     &  1.24200d-6,  1.16809d-6,  8.92126d-7,  7.54664d-7,  7.56390d-7,  &
     &  7.54664d-7,  8.92126d-7,  1.16809d-6,  1.24200d-6,  1.02614d-6,  &
     &  7.38776d-7,  6.54502d-7,  5.89570d-7,  5.54341d-7 /


!     ---------------------
!     Beryllium-10 (Nagai).
!     ---------------------

!     0.00000 mb
      data  (star_cr2_be10(lset,1), lset=1,NUM_CR2_LAT) /  &

     &  1.93995d-4,  1.91442d-4,  1.86785d-4,  1.68538d-4,  1.27007d-4,  &
     &  5.79500d-5,  2.53842d-5,  1.38735d-5,  1.03758d-5,  9.73422d-6,  &
     &  1.03758d-5,  1.38735d-5,  2.53842d-5,  5.79500d-5,  1.27007d-4,  &
     &  1.68538d-4,  1.86785d-4,  1.91442d-4,  1.93995d-4 /

!     2.45225 mb
      data  (star_cr2_be10(lset,2), lset=1,NUM_CR2_LAT) /  &

     &  1.93995d-4,  1.91442d-4,  1.86785d-4,  1.68538d-4,  1.27007d-4,  &
     &  5.79500d-5,  2.53842d-5,  1.38735d-5,  1.03758d-5,  9.73422d-6,  &
     &  1.03758d-5,  1.38735d-5,  2.53842d-5,  5.79500d-5,  1.27007d-4,  &
     &  1.68538d-4,  1.86785d-4,  1.91442d-4,  1.93995d-4 /

!     14.7135 mb
      data  (star_cr2_be10(lset,3), lset=1,NUM_CR2_LAT) /  &

     &  2.61918d-4,  2.58471d-4,  2.52184d-4,  2.27548d-4,  1.71476d-4,  &
     &  7.82400d-5,  3.42719d-5,  1.87310d-5,  1.40087d-5,  1.31425d-5,  &
     &  1.40087d-5,  1.87310d-5,  3.42719d-5,  7.82400d-5,  1.71476d-4,  &
     &  2.27548d-4,  2.52184d-4,  2.58471d-4,  2.61918d-4 /

!     44.1405 mb
      data  (star_cr2_be10(lset,4), lset=1,NUM_CR2_LAT) /  &

     &  3.49491d-4,  3.44893d-4,  3.36504d-4,  3.03630d-4,  2.28810d-4,  &
     &  1.04400d-4,  4.57309d-5,  2.49939d-5,  1.86926d-5,  1.75367d-5,  &
     &  1.86926d-5,  2.49939d-5,  4.57309d-5,  1.04400d-4,  2.28810d-4,  &
     &  3.03630d-4,  3.36504d-4,  3.44893d-4,  3.49491d-4 /

!     68.6630 mb
      data  (star_cr2_be10(lset,5), lset=1,NUM_CR2_LAT) /  &

     &  3.87320d-4,  3.82223d-4,  3.72926d-4,  3.36494d-4,  2.53576d-4,  &
     &  1.15700d-4,  5.06807d-5,  2.76991d-5,  2.07158d-5,  1.94348d-5,  &
     &  2.07158d-5,  2.76991d-5,  5.06807d-5,  1.15700d-4,  2.53576d-4,  &
     &  3.36494d-4,  3.72926d-4,  3.82223d-4,  3.87320d-4 /

!     107.899 mb
      data  (star_cr2_be10(lset,6), lset=1,NUM_CR2_LAT) /  &

     &  3.63886d-4,  3.59098d-4,  3.50363d-4,  3.16136d-4,  2.38234d-4,  &
     &  1.08700d-4,  4.76145d-5,  2.60233d-5,  6.13922d-5,  6.10897d-5,  &
     &  6.13922d-5,  2.60233d-5,  4.76145d-5,  1.08700d-4,  2.38234d-4,  &
     &  3.16136d-4,  3.50363d-4,  3.59098d-4,  3.63886d-4 /

!     147.135 mb
      data  (star_cr2_be10(lset,7), lset=1,NUM_CR2_LAT) /  &

     &  3.46813d-4,  3.42250d-4,  3.33925d-4,  3.01303d-4,  2.27057d-4,  &
     &  1.03600d-4,  4.53805d-5,  2.48023d-5,  5.85118d-5,  5.82235d-5,  &
     &  5.85118d-5,  2.48023d-5,  4.53805d-5,  1.03600d-4,  2.27057d-4,  &
     &  3.01303d-4,  3.33925d-4,  3.42250d-4,  3.46813d-4 /

!     186.371 mb
      data  (star_cr2_be10(lset,8), lset=1,NUM_CR2_LAT) /  &

     &  3.34762d-4,  3.30357d-4,  3.22321d-4,  2.90833d-4,  2.19167d-4,  &
     &  1.00000d-4,  4.38036d-5,  6.67726d-5,  5.64785d-5,  5.62003d-5,  &
     &  5.64785d-5,  6.67726d-5,  4.38036d-5,  1.00000d-4,  2.19167d-4,  &
     &  2.90833d-4,  3.22321d-4,  3.30357d-4,  3.34762d-4 /

!     225.607 mb
      data  (star_cr2_be10(lset,9), lset=1,NUM_CR2_LAT) /  &

     &  3.00549d-4,  2.96595d-4,  2.89380d-4,  2.61110d-4,  1.96768d-4,  &
     &  8.97800d-5,  3.93268d-5,  5.99485d-5,  5.07064d-5,  5.04566d-5,  &
     &  5.07064d-5,  5.99485d-5,  3.93268d-5,  8.97800d-5,  1.96768d-4,  &
     &  2.61110d-4,  2.89380d-4,  2.96595d-4,  3.00549d-4 /

!     264.843 mb
      data  (star_cr2_be10(lset,10), lset=1,NUM_CR2_LAT) /  &

     &  2.49197d-4,  2.45918d-4,  2.39936d-4,  2.16496d-4,  1.63148d-4,  &
     &  7.44400d-5,  6.61262d-5,  4.97056d-5,  4.20426d-5,  4.18355d-5,  &
     &  4.20426d-5,  4.97056d-5,  6.61262d-5,  7.44400d-5,  1.63148d-4,  &
     &  2.16496d-4,  2.39936d-4,  2.45918d-4,  2.49197d-4 /

!     304.079 mb
      data  (star_cr2_be10(lset,11), lset=1,NUM_CR2_LAT) /  &

     &  3.15269d-5,  3.33495d-5,  3.66746d-5,  4.08125d-5,  5.45070d-5,  &
     &  6.19700d-5,  5.50489d-5,  4.13790d-5,  3.49997d-5,  3.48273d-5,  &
     &  3.49997d-5,  4.13790d-5,  5.50489d-5,  6.19700d-5,  5.45070d-5,  &
     &  4.08125d-5,  3.66746d-5,  3.33495d-5,  3.15269d-5 /

!     343.315 mb
      data  (star_cr2_be10(lset,12), lset=1,NUM_CR2_LAT) /  &

     &  2.63479d-5,  2.78711d-5,  3.06500d-5,  3.41081d-5,  4.55530d-5,  &
     &  5.17900d-5,  4.60058d-5,  3.45816d-5,  2.92502d-5,  2.91061d-5,  &
     &  2.92502d-5,  3.45816d-5,  4.60058d-5,  5.17900d-5,  4.55530d-5,  &
     &  3.41081d-5,  3.06500d-5,  2.78711d-5,  2.63479d-5 /

!     382.551 mb
      data  (star_cr2_be10(lset,13), lset=1,NUM_CR2_LAT) /  &

     &  2.21049d-5,  2.33829d-5,  2.57142d-5,  2.86155d-5,  3.82173d-5,  &
     &  4.34500d-5,  3.85973d-5,  2.90127d-5,  2.45399d-5,  2.44190d-5,  &
     &  2.45399d-5,  2.90127d-5,  3.85973d-5,  4.34500d-5,  3.82173d-5,  &
     &  2.86155d-5,  2.57142d-5,  2.33829d-5,  2.21049d-5 /

!     421.787 mb
      data  (star_cr2_be10(lset,14), lset=1,NUM_CR2_LAT) /  &

     &  1.82893d-5,  1.93467d-5,  2.12757d-5,  2.36761d-5,  3.16206d-5,  &
     &  3.59500d-5,  3.19349d-5,  2.40048d-5,  2.03040d-5,  2.02040d-5,  &
     &  2.03040d-5,  2.40048d-5,  3.19349d-5,  3.59500d-5,  3.16206d-5,  &
     &  2.36761d-5,  2.12757d-5,  1.93467d-5,  1.82893d-5 /

!     461.023 mb
      data  (star_cr2_be10(lset,15), lset=1,NUM_CR2_LAT) /  &

     &  1.49316d-5,  1.57949d-5,  1.73697d-5,  1.93295d-5,  2.58154d-5,  &
     &  2.93500d-5,  2.60720d-5,  1.95978d-5,  1.65765d-5,  1.64948d-5,  &
     &  1.65765d-5,  1.95978d-5,  2.60720d-5,  2.93500d-5,  2.58154d-5,  &
     &  1.93295d-5,  1.73697d-5,  1.57949d-5,  1.49316d-5 /

!     500.259 mb
      data  (star_cr2_be10(lset,16), lset=1,NUM_CR2_LAT) /  &

     &  1.22353d-5,  1.29426d-5,  1.42331d-5,  1.58390d-5,  2.11537d-5,  &
     &  2.40500d-5,  2.13640d-5,  1.60588d-5,  1.35831d-5,  1.35162d-5,  &
     &  1.35831d-5,  1.60588d-5,  2.13640d-5,  2.40500d-5,  2.11537d-5,  &
     &  1.58390d-5,  1.42331d-5,  1.29426d-5,  1.22353d-5 /

!     539.495 mb
      data  (star_cr2_be10(lset,17), lset=1,NUM_CR2_LAT) /  &

     &  1.00630d-5,  1.06447d-5,  1.17060d-5,  1.30268d-5,  1.73979d-5,  &
     &  1.97800d-5,  1.75709d-5,  1.32076d-5,  1.11715d-5,  1.11164d-5,  &
     &  1.11715d-5,  1.32076d-5,  1.75709d-5,  1.97800d-5,  1.73979d-5,  &
     &  1.30268d-5,  1.17060d-5,  1.06447d-5,  1.00630d-5 /

!     578.731 mb
      data  (star_cr2_be10(lset,18), lset=1,NUM_CR2_LAT) /  &

     &  8.30779d-6,  8.78808d-6,  9.66430d-6,  1.07547d-5,  1.43634d-5,  &
     &  1.63300d-5,  1.45062d-5,  1.09040d-5,  9.22294d-6,  9.17751d-6,  &
     &  9.22294d-6,  1.09040d-5,  1.45062d-5,  1.63300d-5,  1.43634d-5,  &
     &  1.07547d-5,  9.66430d-6,  8.78808d-6,  8.30779d-6 /

!     617.967 mb
      data  (star_cr2_be10(lset,19), lset=1,NUM_CR2_LAT) /  &

     &  6.62385d-6,  7.00679d-6,  7.70540d-6,  8.57478d-6,  1.14520d-5,  &
     &  1.30200d-5,  1.15659d-5,  8.69380d-6,  7.35351d-6,  7.31728d-6,  &
     &  7.35351d-6,  8.69380d-6,  1.15659d-5,  1.30200d-5,  1.14520d-5,  &
     &  8.57478d-6,  7.70540d-6,  7.00679d-6,  6.62385d-6 /

!     657.203 mb
      data  (star_cr2_be10(lset,20), lset=1,NUM_CR2_LAT) /  &

     &  5.09762d-6,  5.39232d-6,  5.92996d-6,  6.59902d-6,  8.81330d-6,  &
     &  1.00200d-5,  8.90091d-6,  6.69062d-6,  5.65915d-6,  5.63127d-6,  &
     &  5.65915d-6,  6.69062d-6,  8.90091d-6,  1.00200d-5,  8.81330d-6,  &
     &  6.59902d-6,  5.92996d-6,  5.39232d-6,  5.09762d-6 /

!     696.439 mb
      data  (star_cr2_be10(lset,21), lset=1,NUM_CR2_LAT) /  &

     &  3.92954d-6,  4.15672d-6,  4.57116d-6,  5.08691d-6,  6.79380d-6,  &
     &  7.72400d-6,  6.86134d-6,  5.15752d-6,  4.36240d-6,  4.34091d-6,  &
     &  4.36240d-6,  5.15752d-6,  6.86134d-6,  7.72400d-6,  6.79380d-6,  &
     &  5.08691d-6,  4.57116d-6,  4.15672d-6,  3.92954d-6 /

!     735.675 mb
      data  (star_cr2_be10(lset,22), lset=1,NUM_CR2_LAT) /  &

     &  3.03669d-6,  3.21225d-6,  3.53253d-6,  3.93109d-6,  5.25016d-6,  &
     &  5.96900d-6,  5.30235d-6,  3.98566d-6,  3.37120d-6,  3.35460d-6,  &
     &  3.37120d-6,  3.98566d-6,  5.30235d-6,  5.96900d-6,  5.25016d-6,  &
     &  3.93109d-6,  3.53253d-6,  3.21225d-6,  3.03669d-6 /

!     774.911 mb
      data  (star_cr2_be10(lset,23), lset=1,NUM_CR2_LAT) /  &

     &  2.35192d-6,  2.48789d-6,  2.73595d-6,  3.04464d-6,  4.06626d-6,  &
     &  4.62300d-6,  4.10668d-6,  3.08690d-6,  2.61100d-6,  2.59814d-6,  &
     &  2.61100d-6,  3.08690d-6,  4.10668d-6,  4.62300d-6,  4.06626d-6,  &
     &  3.04464d-6,  2.73595d-6,  2.48789d-6,  2.35192d-6 /

!     814.147 mb
      data  (star_cr2_be10(lset,24), lset=1,NUM_CR2_LAT) /  &

     &  1.80095d-6,  1.90507d-6,  2.09502d-6,  2.33139d-6,  3.11368d-6,  &
     &  3.54000d-6,  3.14463d-6,  2.36375d-6,  1.99934d-6,  1.98949d-6,  &
     &  1.99934d-6,  2.36375d-6,  3.14463d-6,  3.54000d-6,  3.11368d-6,  &
     &  2.33139d-6,  2.09502d-6,  1.90507d-6,  1.80095d-6 /

!     853.383 mb
      data  (star_cr2_be10(lset,25), lset=1,NUM_CR2_LAT) /  &

     &  1.36445d-6,  1.44333d-6,  1.58724d-6,  1.76633d-6,  2.35901d-6,  &
     &  2.68200d-6,  2.38246d-6,  1.79084d-6,  1.51475d-6,  1.50729d-6,  &
     &  1.51475d-6,  1.79084d-6,  2.38246d-6,  2.68200d-6,  2.35901d-6,  &
     &  1.76633d-6,  1.58724d-6,  1.44333d-6,  1.36445d-6 /

!     892.619 mb
      data  (star_cr2_be10(lset,26), lset=1,NUM_CR2_LAT) /  &

     &  1.03631d-6,  1.09622d-6,  1.20552d-6,  1.34154d-6,  1.79169d-6,  &
     &  2.03700d-6,  1.80950d-6,  1.36016d-6,  1.15047d-6,  1.14480d-6,  &
     &  1.15047d-6,  1.36016d-6,  1.80950d-6,  2.03700d-6,  1.79169d-6,  &
     &  1.34154d-6,  1.20552d-6,  1.09622d-6,  1.03631d-6 /

!     931.855 mb
      data  (star_cr2_be10(lset,27), lset=1,NUM_CR2_LAT) /  &

     &  7.90588d-7,  8.36294d-7,  9.19677d-7,  1.02344d-6,  1.36685d-6,  &
     &  1.55400d-6,  1.38044d-6,  1.03765d-6,  8.77676d-7,  8.73353d-7,  &
     &  8.77676d-7,  1.03765d-6,  1.38044d-6,  1.55400d-6,  1.36685d-6,  &
     &  1.02344d-6,  9.19677d-7,  8.36294d-7,  7.90588d-7 /

!     971.091 mb
      data  (star_cr2_be10(lset,28), lset=1,NUM_CR2_LAT) /  &

     &  6.04897d-7,  6.39867d-7,  7.03665d-7,  7.83058d-7,  1.04581d-6,  &
     &  1.18900d-6,  1.05621d-6,  7.93927d-7,  6.71530d-7,  6.68222d-7,  &
     &  6.71530d-7,  7.93927d-7,  1.05621d-6,  1.18900d-6,  1.04581d-6,  &
     &  7.83058d-7,  7.03665d-7,  6.39867d-7,  6.04897d-7 /

!     1006.89 mb
      data  (star_cr2_be10(lset,29), lset=1,NUM_CR2_LAT) /  &

     &  4.65704d-7,  4.92628d-7,  5.41745d-7,  6.02869d-7,  8.05159d-7,  &
     &  9.15400d-7,  8.13163d-7,  6.11237d-7,  5.17005d-7,  5.14458d-7,  &
     &  5.17005d-7,  6.11237d-7,  8.13163d-7,  9.15400d-7,  8.05159d-7,  &
     &  6.02869d-7,  5.41745d-7,  4.92628d-7,  4.65704d-7 /

!     1200.00 mb
      data  (star_cr2_be10(lset,30), lset=1,NUM_CR2_LAT) /  &

     &  4.65704d-7,  4.92628d-7,  5.41745d-7,  6.02869d-7,  8.05159d-7,  &
     &  9.15400d-7,  8.13163d-7,  6.11237d-7,  5.17005d-7,  5.14458d-7,  &
     &  5.17005d-7,  6.11237d-7,  8.13163d-7,  9.15400d-7,  8.05159d-7,  &
     &  6.02869d-7,  5.41745d-7,  4.92628d-7,  4.65704d-7 /

