#include "MAPL_Generic.h"
!-------------------------------------------------------------------------
!     NASA/GSFC, Global Modeling and Assimilation Office, Code 610.1     !
!-------------------------------------------------------------------------
!BOP
!
! !MODULE:  GmiEmissionLightning_mod
!
! !INTERFACE:
!
   MODULE GmiEmissionLightning_mod

! !USES:

   USE ESMF
   USE MAPL

   IMPLICIT NONE

! !PUBLIC TYPES:
!
   PRIVATE
   PUBLIC :: emiss_lightning

#include "gmi_phys_constants.h"

! !ROUTINE
!   emiss_lightning
!
! !DESCRIPTION
!   
!  Generate NOx production rates from parameterized lightning and distribute vertically
!  using profiles from Pickering (2007).
!
! !REVISION HISTORY:
!   July 28, 2008 - Dale Allen.  First version, obsolete.  
!   December 30, 2011 - Eric Nielsen: Simplified for GEOS-5 with flash rates imported from MOIST.
!                       Target: ~4 Tg N yr^{-1}. Best advice: ~250 moles NO per flash, which 
!                       translates to numberNOperFlash = 1.5E+26.
!   January 24, 2013 - Eric Nielsen: kgNOx3D changed from N to NO production rate for export 
!                       EM_LGTNO. Purpose is to allow direct comparison with GMI NO_lgt.
!-----------------------------------------------------------------------------

CONTAINS

 SUBROUTINE emiss_lightning(i1, i2, j1, j2, k1, k2, minDeepCloudTop, ampFactor, numberNOperFlash, &
                            lwi, flashrate, cellDepth, dtrn, pNOx3D, kgNOx3D, rc)

  IMPLICIT NONE

  INTEGER, INTENT(IN)  :: i1, i2, j1, j2, k1, k2         ! Index ranges on this processor
  REAL,    INTENT(IN)  :: minDeepCloudTop                ! Minimum cloud top [km] for selecting deep convection profiles
  REAL,    INTENT(IN)  :: ampFactor                      ! > 0, for targeting the observed nitrogen production rate [3.41 Tg yr^{-1}]
  REAL,    INTENT(IN)  :: numberNOperFlash               ! NO molecules generated by each flash
  INTEGER, INTENT(IN)  :: lwi(i1:i2, j1:j2)              ! Flag: 1=water 2=land 3=ice
  REAL*8,  INTENT(IN)  :: flashrate(i1:i2, j1:j2)        ! Flash rate [km^{-2} s^{-1}]
  REAL*8,  INTENT(IN)  :: dtrn(i1:i2, j1:j2, k1:k2)      ! Detrainment [kg m^{-2} s^{-1}]
  REAL*8,  INTENT(IN)  :: cellDepth(i1:i2, j1:j2, k1:k2) ! Grid cell depth [m]
  
  REAL*8, INTENT(OUT)  :: pNOx3D(i1:i2, j1:j2, k1:k2)    ! Lightning NO production rate [m^{-3} s^{-1}]
  REAL*8, INTENT(OUT)  :: kgNOx3D(i1:i2, j1:j2, k1:k2)   ! NO production rate [kg m^{-3} s^{-1}]

! Local
! -----
  INTEGER :: k
  INTEGER :: status, rc
  REAL*8, ALLOCATABLE  :: pNOx2D(:,:)                    ! Lightning NO production [molecules NO m^{-2} s^{-1}]
  CHARACTER(LEN=*), PARAMETER :: Iam = "emiss_lightning"
  rc = 0
  status = 0

! Validate ranges for ampFactor and numberNOperFlash
! --------------------------------------------------
  IF(ampFactor <= 0.00) THEN
   IF(MAPL_AM_I_ROOT()) PRINT *,TRIM(IAm)//": Invalid ampFactor ",ampFactor
   status = 1
   VERIFY_(status)
  END IF
  IF(numberNOperFlash <= 0.00) THEN
   IF(MAPL_AM_I_ROOT()) PRINT *,TRIM(IAm)//": Invalid numberNOperFlash ",numberNOperFlash
   status = 1
   VERIFY_(status)
  END IF

! Grab some memory
! ----------------
  ALLOCATE(pNOx2D(i1:i2, j1:j2),STAT=status)
  VERIFY_(status)
  pNOx2D(:,:) = 0.00

! Calculate the NOx produdction rate [molecules NO m^{-2} s^{-1}]
! ---------------------------------------------------------------
  pNOx2D(:,:) = 1.00E-06*flashrate(:,:)*numberNOPerFlash

! Amplification/suppression factor: > 0
! -------------------------------------
  pNOx2D(:,:) = pNOx2D(:,:)*ampFactor
     
! Partition vertically without changing units
! -------------------------------------------
  CALL partition(i1, i2, j1, j2, k1, k2, minDeepCloudTop, lwi, pNOx2D, dtrn, cellDepth, pNOx3D, rc)

! Place output in useful units
! ----------------------------
  DO k = k1,k2

! Number density tendency [m^{-3} s^{-1}]
! ---------------------------------------
   pNOx3D(i1:i2,j1:j2,k) = pNOx3D(i1:i2,j1:j2,k)/cellDepth(i1:i2,j1:j2,k)

! NO density tendency [kg NO m^{-3} s^{-1}]
! -----------------------------------------
   kgNOx3D(i1:i2,j1:j2,k) = pNOx3D(i1:i2,j1:j2,k)*30.0064/(1000.00*AVOGAD)

  END DO

! Clean up
! --------
  DEALLOCATE(pNOx2D,STAT=status)
  VERIFY_(status)

  RETURN
 END SUBROUTINE emiss_lightning

!=============================================================================

 SUBROUTINE partition(i1, i2, j1, j2, k1, k2, minDeepCloudTop, lwi, pNOx2D, dtrn, cellDepth, pNOx3D, rc)

  IMPLICIT NONE

  INTEGER, INTENT(IN)  :: i1, i2, j1, j2, k1, k2    ! Index ranges on this processor
  REAL,    INTENT(IN)  :: minDeepCloudTop           ! Minimum cloud top [km] for selecting deep convection profiles
  INTEGER, INTENT(IN)  :: lwi(i1:i2, j1:j2)         ! Flag: 1=water 2=land 3=ice
  REAL*8,  INTENT(IN)  :: pNOx2D(i1:i2, j1:j2)      ! Lightning NO production rate [molecules NO m^{-2} s^{-1}]
  REAL*8,  INTENT(IN)  :: dtrn(i1:i2, j1:j2, k1:k2) ! Detrainment [kg m^{-2}s^{-1}]
  REAL*8,  INTENT(IN)  :: cellDepth(i1:i2, j1:j2, k1:k2) ! Grid cell depth [m]

  REAL*8, INTENT(OUT)  :: pNOx3D(i1:i2, j1:j2, k1:k2) ! Scaled production rate (no units conversion here)

! Local variables
! ---------------
  CHARACTER(LEN=*), PARAMETER :: Iam = "partition"

  INTEGER :: i,j,k
  INTEGER :: cl                     ! vertical index
  INTEGER :: nTop                   ! Top model level at which detrainment is non-zero
  INTEGER :: profileNumber
  INTEGER :: rc, status
  INTEGER, PARAMETER :: numKm = 17  ! Number of elements (kilometers) in each specified profile

  REAL :: zLower, zUpper

  REAL, ALLOCATABLE :: r(:,:)       ! Specified NOx distribution profiles

  REAL ::       w(k1:k2)     ! Weights applied to scaled cloud layers
  REAL ::       z(k1:k2)     ! Height above ground for top edge of grid-box
  REAL :: zScaled(k1:k2)     ! Scaled layer edge heights
   
  rc = 0
  status = 0

! Specify the percentage NOx distributions in each km for a numKm-depth cloud.
! Deep convection is arbitrarily assigned when the cloud top is greater than 7 km.
! --------------------------------------------------------------------------------
   ALLOCATE(r(numKm,3),STAT=status)
   VERIFY_(status)

! Deep convection, continental
! ----------------------------
   r(1:numKm,1) = (/ 0.23, 0.47, 0.56, 1.40, 2.70, 4.00, 5.03, 6.24, &
                     8.60,10.28,11.62,12.34,12.70,12.34, 7.63, 3.02, 0.84 /)
 
! Deep convection, marine
! -----------------------
   r(1:numKm,2) = (/ 0.60, 1.50, 2.90, 4.30, 5.40, 6.70, 7.70, 8.50, &
                     9.60,10.20,10.50,10.20, 8.20, 6.50, 4.50, 2.20, 0.50 /)

! Other
! -----
   r(1:numKm,3) = (/ 2.40, 5.00, 7.40, 9.30,10.60,11.40,11.50,11.00, &
                     9.90, 8.30, 6.30, 4.20, 2.20, 0.50, 0.00, 0.00, 0.00 /)

! Work in each column
! -------------------
   DO j = j1,j2
    DO i = i1,i2

     SeeingLightning : IF(pNOx2D(i,j) > 0.00) THEN

! Define cloud top to be highest layer with dtrn > 0, but at least 2.
! -------------------------------------------------------------------
      DO k = k2,1,-1
       IF(dtrn(i,j,k) > 0) EXIT
      END DO
      nTop = k
      IF(nTop < 2) nTop = 2

! Sum grid box thicknesses (m) to obtain layer edge heights
! ---------------------------------------------------------
      DO k = 1,k2
       z(k) = SUM(cellDepth(i,j,1:k))
      END DO

! Select NOx distribution profile. LWI flag is: 1=water 2=land 3=ice
! ------------------------------------------------------------------
      IF(z(nTop) > minDeepCloudTop*1000.00) THEN
       IF(lwi(i,j) == 2) THEN
        profileNumber = 1
!  PRINT*,'PROFILE_1  MAX WGT ', z(nTop)*13./17.
       ELSE
        profileNumber = 2
!  PRINT*,'PROFILE_2  MAX WGT ', z(nTop)*11./17.
       END IF
      ELSE
       profileNumber = 3
!  PRINT*,'PROFILE_3  MAX WGT ', z(nTop)* 7./17.
      ENDIF


! The distance from ground through cloud top is artificially
! scaled to be 0 through 17 km.
!
! The z quotient is <  1 below nTop,  1 at nTop, >  1 above nTop
! Scale factor   is < 17 below nTop, 17 at nTop, > 17 above nTop
! zScaled(i) is the top of the model level i (km)
! ---------------------------------------------
    ! zScaled(1:k2) = numKm*(z(1:k2)/z(nTop))
      zScaled(1:k2) = z(1:k2)*numKm/z(nTop)

! Intialize
! ---------
      w(:) = 0.00    ! weights - will only use indices  1:nTop
      cl = 1         ! model level index - start at the bottom
      zLower = 0.00  ! edge of gridbox or kilometer boundary (km)

! Compute the weight (w) to be applied at each level
! --------------------------------------------------
      Kilometers: DO k = 1,numKm      ! k = index into profile
                                      ! k is also the height of top edge (km)

! ... segment-by-segment
! A segment extends from zLower to zUpper
! Each of those can be the edge of a gridbox or a kilometer boundary
! ------------------------------------------------------------------
       Segment: DO

! Push up to the lesser of scaled cloud height or next km
! -------------------------------------------------------
        zUpper = MIN(zScaled(cl),k*1.)
        IF(zScaled(cl) > numKm) EXIT

! Add increment to scaled weighting for the current cloud layer
! Accumulate the weight
! Convert profile value (r) from 0-100 range, to 0-1 range
! -------------------------------------------------------------
        w(cl) = w(cl) + (zUpper-zLower) * r(k,profileNumber) * 0.01

! Advance to next cloud layer if any of it lies within this km
! ------------------------------------------------------------
        IF(zUpper == zScaled(cl)) cl = cl+1

! Shift bounds before working on the next segment
! -----------------------------------------------
        zLower = zUpper

! At top of this km. Advance to the next one
! ------------------------------------------
        IF(zUpper == k*1.) EXIT

       END DO Segment

      END DO Kilometers

! Finalize vertical distribution and clean up
! -------------------------------------------
      pNOx3D(i,j,1:nTop) = w(1:nTop)*pNOx2D(i,j)

! PRINT*,'TOTAL for w is ', SUM(w(1:nTop))  ->  this is 1.0

     END IF SeeingLightning

! Next column
! -----------
    END DO
   END DO

! Clean up
! --------
   DEALLOCATE(r,STAT=status)
   VERIFY_(status)

  RETURN
 END SUBROUTINE partition

!=============================================================================

END MODULE GmiEmissionLightning_mod
