!------------------------------------------------------------------------------
!                  GEOS-Chem Global Chemical Transport Model                  !
!------------------------------------------------------------------------------
!BOP
!
! !MODULE: diagnostics_mod.F90
!
! !DESCRIPTION: Module diagnostics\_mod.F90 contains subroutines for
!  setting State_Diag diagnostics arrays for the purposes of outputting
!  in netcdf format. Source code for setting diagnostics arrays for output
!  in binary format are not included in this module.
!
! !INTERFACE:
!
MODULE Diagnostics_mod
!
! !USES:
!
  USE ErrCode_Mod
  USE Precision_Mod

  IMPLICIT NONE
  PRIVATE
!
! !PUBLIC MEMBER FUNCTIONS:
!
  PUBLIC :: Set_Diagnostics_EndofTimestep
  PUBLIC :: Zero_Diagnostics_StartofTimestep
  PUBLIC :: Compute_Budget_Diagnostics
!
! !PRIVATE MEMBER FUNCTIONS
!
  PRIVATE :: Set_SpcConc_Diags_VVDry
!
! !REVISION HISTORY:
!  01 Feb 2018 - E. Lundgren - Initial version
!  See https://github.com/geoschem/geos-chem for complete history
!EOP
!------------------------------------------------------------------------------
!BOC
CONTAINS
!EOC
!------------------------------------------------------------------------------
!                  GEOS-Chem Global Chemical Transport Model                  !
!------------------------------------------------------------------------------
!BOP
!
! !IROUTINE: Set_Diagnostics_EndofTimestep
!
! !DESCRIPTION: Updates various diagnostics for History output at the end
!  of the GEOS-Chem "heartbeat" timestep.
!\\
!\\
! !INTERFACE:
!
  SUBROUTINE Set_Diagnostics_EndofTimestep( Input_Opt,  State_Chm, State_Diag, &
                                            State_Grid, State_Met, RC )
!
! !USES:
!
    USE Input_Opt_Mod,    ONLY : OptInput
    USE State_Met_Mod,    ONLY : MetState
    USE State_Chm_Mod,    ONLY : ChmState, Ind_
    USE State_Diag_Mod,   ONLY : DgnState
    USE State_Grid_Mod,   ONLY : GrdState
    USE PhysConstants,    ONLY : AIRMW
!
! !INPUT PARAMETERS:
!
    TYPE(OptInput),   INTENT(IN)    :: Input_Opt      ! Input Options object
    TYPE(GrdState),   INTENT(IN)    :: State_Grid     ! Grid state object
    TYPE(MetState),   INTENT(IN)    :: State_Met      ! Meteorology state object
!
! !INPUT AND OUTPUT PARAMETERS:
!
    TYPE(ChmState),   INTENT(INOUT) :: State_Chm      ! Chemistry state obj
    TYPE(DgnState),   INTENT(INOUT) :: State_Diag     ! Diagnostics state obj
!
! !OUTPUT PARAMETERS:
!
    INTEGER,              INTENT(OUT)   :: RC
!
! !REVISION HISTORY:
!  01 Feb 2018 - E. Lundgren - initial version
!  See https://github.com/geoschem/geos-chem for complete history
!EOP
!------------------------------------------------------------------------------
!BOC
!
! !LOCAL VARIABLES:
!
    ! Scalars
    INTEGER                 :: I, J, L, N, S
    REAL(fp)                :: ToPptv

    ! SAVEd scalars
    INTEGER, SAVE           :: id_Hg2 = -1
    INTEGER, SAVE           :: id_HgP = -1
    LOGICAL, SAVE           :: FIRST  = .TRUE.

    ! Strings
    CHARACTER(LEN=255)      :: ErrMsg, ThisLoc

    !=======================================================================
    ! Set_Diagnostics_EndofTimestep begins here
    !=======================================================================

    ! Initialize
    RC      = GC_SUCCESS
    ErrMsg  = ''
    ThisLoc = &
      ' -> at Set_Diagnostics_EndofTimestep (in GeosCore/diagnostics_mod.F90)'

    !-----------------------------------------------------------------------
    ! Set species concentration for diagnostics in units of
    ! v/v dry air = mol/mol dry air
    !-----------------------------------------------------------------------
    CALL Set_SpcConc_Diags_VVDry( Input_Opt,  State_Chm, State_Diag,         &
                                  State_Grid, State_Met, RC                 )

    ! Trap potential errors
    IF ( RC /= GC_SUCCESS ) THEN
       ErrMsg = 'Error encountered setting species concentration diagnostic'
       CALL GC_ERROR( ErrMsg, RC, ThisLoc )
       RETURN
    ENDIF

    !-----------------------------------------------------------------------
    ! Set total dry deposition flux
    !-----------------------------------------------------------------------
    IF ( State_Diag%Archive_DryDep ) THEN
       !$OMP PARALLEL DO           &
       !$OMP DEFAULT( SHARED     ) &
       !$OMP PRIVATE( I, J, N, S )
       DO S = 1, State_Diag%Map_DryDep%nSlots
          DO J = 1, State_Grid%NY
          DO I = 1, State_Grid%NX
             State_Diag%DryDep(I,J,S) = State_Diag%DryDepChm(I,J,S)          &
                                      + State_Diag%DryDepMix(I,J,S)
          ENDDO
          ENDDO
       ENDDO
       !$OMP END PARALLEL DO
    ENDIF

    !-----------------------------------------------------------------------
    ! Compute fraction of time each grid box spent in the troposphere
    !-----------------------------------------------------------------------
    IF ( State_Diag%Archive_FracOfTimeInTrop ) THEN
       !$OMP PARALLEL DO            &
       !$OMP DEFAULT( SHARED      ) &
       !$OMP SCHEDULE( DYNAMIC, 8 ) &
       !$OMP PRIVATE( I, J, L )
       DO L = 1, State_Grid%NZ
       DO J = 1, State_Grid%NY
       DO I = 1, State_Grid%NX
          IF ( State_Met%InTroposphere(I,J,L) ) THEN
             State_Diag%FracOfTimeInTrop(I,J,L) = 1.0_f4
          ELSE
             State_Diag%FracOfTimeInTrop(I,J,L) = 0.0_f4
          ENDIF
       ENDDO
       ENDDO
       ENDDO
       !$OMP END PARALLEL DO
    ENDIF

    !-----------------------------------------------------------------------
    ! Diagnostics for the mercury and tagged mercury simulations
    !-----------------------------------------------------------------------
    IF ( Input_Opt%ITS_A_MERCURY_SIM ) THEN

       ! Get species indices for Hg2 and HgP
       IF ( FIRST ) THEN
          id_Hg2 = Ind_('Hg2')
          id_HgP = Ind_('HgP')
          FIRST  = .FALSE.
       ENDIF

       !--------------------------------------------
       ! Ractive gaseous mercury (RGM) [pptv]
       !--------------------------------------------
       IF ( id_Hg2 > 0 .and. State_Diag%Archive_ReactiveGaseousHg ) THEN

          ! Conversion factor to pptv
          ToPptv = ( AIRMW                                  /                &
                     State_Chm%SpcData(id_Hg2)%Info%EmMW_g  *                &
                     1.0e+12_fp                               )

          ! Save into State_diag
          State_Diag%ReactiveGaseousHg = State_Chm%Species(:,:,:,id_Hg2)     &
                                       * ToPptv
       ENDIF

       !--------------------------------------------
       ! Ractive particulate mercury (RGM) [pptv]
       !--------------------------------------------
       IF ( id_HgP > 0 .and. State_Diag%Archive_ParticulateBoundHg ) THEN

          ! Conversion factor to pptv
          ToPptv = ( AIRMW                                  /                &
                     State_Chm%SpcData(id_HgP)%Info%EmMW_g  *                &
                     1.0e+12_fp                               )

          ! Save into State_Diag
          State_Diag%ParticulateBoundHg = State_Chm%Species(:,:,:,id_HgP)    &
                                        * ToPptv
       ENDIF
    ENDIF

  END SUBROUTINE Set_Diagnostics_EndofTimestep
!EOC
!------------------------------------------------------------------------------
!                  GEOS-Chem Global Chemical Transport Model                  !
!------------------------------------------------------------------------------
!BOP
!
! !IROUTINE: Zero_Diagnostics_StartofTimestep
!
! !DESCRIPTION: This routine is currently not used but is available if needed
!   in the future for diagnostics that are summed over a timestep and must
!   therefore be zeroed at the start of each time in the dynamic loop.
!\\
!\\
! !INTERFACE:
!
  SUBROUTINE Zero_Diagnostics_StartofTimestep( State_Chm, State_Diag, RC )
!
! !USES:
!
    USE State_Chm_Mod,    ONLY : ChmState
    USE State_Diag_Mod,   ONLY : DgnState
!
! !INPUT PARAMETERS:
!
    TYPE(ChmState),   INTENT(INOUT) :: State_Chm      ! Chemistry state obj
!
! !INPUT AND OUTPUT PARAMETERS:
!
    TYPE(DgnState),   INTENT(INOUT) :: State_Diag     ! Diagnostics state obj
!
! !OUTPUT PARAMETERS:
!
    INTEGER,          INTENT(OUT)   :: RC
!
! !REVISION HISTORY:
!  01 Feb 2018 - E. Lundgren - initial version
!  See https://github.com/geoschem/geos-chem for complete history
!EOP
!------------------------------------------------------------------------------
!BOC
!
! !LOCAL VARIABLES:
!
    INTEGER                 :: I, J, L, N
    CHARACTER(LEN=255)      :: ErrMsg, thisLoc

    !=======================================================================
    ! Zero_Diagnostics_StartofTimestep begins here
    !=======================================================================

    ! Initialize
    RC      = GC_SUCCESS
    ErrMsg  = ''
    ThisLoc = &
    ' -> at Zero_Diagnostics_StartofTimestep (in GeosCore/diagnostics_mod.F90)'

    ! Zero diagnostics here

  END SUBROUTINE Zero_Diagnostics_StartofTimestep
!EOC
!------------------------------------------------------------------------------
!                  GEOS-Chem Global Chemical Transport Model                  !
!------------------------------------------------------------------------------
!BOP
!
! !IROUTINE: Set_SpcConc_Diags_VVDry
!
! !DESCRIPTION: Subroutine Set_SpcConc\_DiagVVDry sets several species
!  concentration diagnostic arrays stored in State_Diag to the instantaneous
!  State_Chm%Species values (in units of "v/v, dry air").
!\\
!\\
! !INTERFACE:
!
  SUBROUTINE Set_SpcConc_Diags_VVDry( Input_Opt,  State_Chm, State_Diag,     &
                                      State_Grid, State_Met, RC            )
!
! !USES:
!
    USE Input_Opt_Mod,  ONLY : OptInput
    USE State_Met_Mod,  ONLY : MetState
    USE State_Chm_Mod,  ONLY : ChmState
    USE State_Diag_Mod, ONLY : DgnMap
    USE State_Diag_Mod, ONLY : DgnState
    USE State_Grid_Mod, ONLY : GrdState
    USE UnitConv_Mod,   ONLY : Convert_Spc_Units
!
! !INPUT PARAMETERS:
!
    TYPE(OptInput),   INTENT(IN)    :: Input_Opt    ! Input Options object
    TYPE(GrdState),   INTENT(IN)    :: State_Grid   ! Grid State object
    TYPE(MetState),   INTENT(IN)    :: State_Met    ! Meteorology State obj
!
! !INPUT/OUTPUT PARAMETERS:
!
    TYPE(ChmState),   INTENT(INOUT) :: State_Chm    ! Chemistry State object
    TYPE(DgnState),   INTENT(INOUT) :: State_Diag   ! Diagnsotics State object
!
! !OUTPUT PARAMETERS:
!
    INTEGER,          INTENT(OUT)   :: RC           ! Success or failure?
!
! !REVISION HISTORY:
!  08 Jul 2019 - R. Yantosca - Initial version
!  See https://github.com/geoschem/geos-chem for complete history
!EOP
!------------------------------------------------------------------------------
!BOC
!
! !LOCAL VARIABLES:
!
    ! Scalars
    LOGICAL               :: Found
    INTEGER               :: D, I, J, L, N, S
    REAL(fp)              :: TmpVal, Conv

    ! Strings
    CHARACTER(LEN=255)    :: ErrMsg, ThisLoc, Units, OrigUnit

    ! Objects
    TYPE(DgnMap), POINTER :: mapData

    !====================================================================
    ! Set_SpcConc_Diags_VVDry begins here!
    !====================================================================

    ! Assume success
    RC      =  GC_SUCCESS
    Found   = .FALSE.
    ThisLoc = &
         ' -> at Set_SpcConc_Diagnostics (in GeosCore/diagnostics_mod.F90)'

    ! Convert State_Chm%Species unit to [v/v dry]
    Units   = 'v/v dry'
    CALL Convert_Spc_Units( Input_Opt, State_Chm, State_Grid, State_Met, &
                            Units, RC, OrigUnit=OrigUnit )

    ! Error handling
    IF ( RC /= GC_SUCCESS ) THEN
       ErrMsg = 'Error converting species units for archiving diagnostics #1'
       CALL GC_Error( ErrMsg, RC, ThisLoc )
       RETURN
    ENDIF

    !=======================================================================
    ! Copy species to SpeciesConc (concentrations diagnostic) [v/v dry]
    !=======================================================================
    IF ( State_Diag%Archive_SpeciesConc ) THEN

       ! Point to mapping obj
       mapData => State_Diag%Map_SpeciesConc

       !$OMP PARALLEL DO       &
       !$OMP DEFAULT( SHARED ) &
       !$OMP PRIVATE( N, S   )
       DO S = 1, mapData%nSlots
          N = mapData%slot2id(S)
          State_Diag%SpeciesConc(:,:,:,S) = State_Chm%Species(:,:,:,N)
       ENDDO
       !$OMP END PARALLEL DO

       ! Free pointer
       mapData => NULL()

    ENDIF

    !=======================================================================
    ! Copy species to SpeciesBC (transport boundary conditions) [v/v dry]
    !=======================================================================
    IF ( State_Diag%Archive_SpeciesBC ) THEN

       ! Point to mapping obj
       mapData => State_Diag%Map_SpeciesBC

       !$OMP PARALLEL DO       &
       !$OMP DEFAULT( SHARED ) &
       !$OMP PRIVATE( N, S   )
       DO S = 1, mapData%nSlots
          N = mapData%slot2id(S)
          State_Diag%SpeciesBC(:,:,:,S) = State_Chm%Species(:,:,:,N)
       ENDDO
       !$OMP END PARALLEL DO

       ! Free pointer
       mapData => NULL()

    ENDIF

    !=======================================================================
    ! Copy species to SpeciesRst (restart file output) [v/v dry]
    !=======================================================================
    IF ( State_Diag%Archive_SpeciesRst ) THEN

       !$OMP PARALLEL DO        &
       !$OMP DEFAULT( SHARED  ) &
       !$OMP PRIVATE( N       )
       DO N = 1, State_Chm%nSpecies
          State_Diag%SpeciesRst(:,:,:,N) = State_Chm%Species(:,:,:,N)
       ENDDO
       !$OMP END PARALLEL DO

    ENDIF

    !=======================================================================
    ! Diagnostic for correcting species concentrations from the height
    ! of the lowest model level to the surface.
    !
    ! Use this diagnostic to correct species concentration values from
    ! (typically for O3 or HNO3) from the lowest model layer, ~60m,
    ! to the surface.
    !
    !    C(Zc) = [ 1 - Ra(Z1,Zc) * Vd(Z1) ] * C(Z1)
    !
    ! where
    !    Ra(Z1,ZC) is the aerodynamic resistance between Z1 and ZC,
    !
    !    Vd(Z1) is the ozone deposition velocity at Z1, and
    !
    !    C(Z1) is the ozone concentration at Z1.
    !
    ! Ra(Z1,Zc) is calculated to the lowest model level in drydep_mod.F90.
    ! We recalculate Ra using Z1 using a value specified in input.geos;
    ! usually 10m, which is the height of the CASTNET measurement for O3.
    ! This new Ra is stored in State_Diag%DryDepRaALT1.
    !
    ! References:
    ! (1) Travis, K.R., et al, "Resolving vertical ozone vertical gradients
    !      in air quality models, Atmos. Chem. Phys. Disc., 2017.
    ! (2) Zhang, L.,et al, "Nitrogen deposition to the United States:
    !      distribution, sources, and processes" Atmos. Chem. Phys.,
    !      12, 4,539-4,4554, 2012.
    !=======================================================================
    IF ( State_Diag%Archive_ConcAboveSfc ) THEN

       ! Loop over the number of drydep species that we wish
       ! to save at a user-specified altitude above the surface
       !$OMP PARALLEL DO                         &
       !$OMP DEFAULT( SHARED                   ) &
       !$OMP PRIVATE( D, N, I, J, TmpVal, Conv )
       DO D = 1, State_Chm%nDryAlt

          ! Get the corresponding species index and drydep index
          N = State_Chm%Map_DryAlt(D)

          ! Loop over surface locations
          DO J = 1, State_Grid%NY
          DO I = 1, State_Grid%NX

             ! Species concentration [v/v dry]
             TmpVal = State_Chm%Species(I,J,1,N)

             ! Conversion factor used to translate from
             ! lowest model layer (~60m) to the surface
             Conv = ( 1.0_fp                                              &
                  -   ( State_Diag%DryDepRaALT1(I,J) / 100.0_fp )         &
                  *   State_Diag%DryDepVelForALT1(I,J,D)                 )

             ! Do not let CONV go negative
             IF ( Conv < 0.0_fp ) Conv = 1.0_fp

             ! Save concentration at the user-defined altitude
             ! as defined in input.geos (usually 10m).
             State_Diag%SpeciesConcALT1(I,J,D) = TmpVal * Conv

          ENDDO
          ENDDO
       ENDDO
       !$OMP END PARALLEL DO

    ENDIF

    ! Convert State_Chm%Species back to original unit
    CALL Convert_Spc_Units( Input_Opt, State_Chm, State_Grid, State_Met, &
                            OrigUnit,  RC )

    ! Error handling
    IF ( RC /= GC_SUCCESS ) THEN
       ErrMsg = 'Error converting species units for archiving diagnostics #2'
       CALL GC_Error( ErrMsg, RC, ThisLoc )
       RETURN
    ENDIF

  END SUBROUTINE Set_SpcConc_Diags_VVDry
!EOC
!------------------------------------------------------------------------------
!                  GEOS-Chem Global Chemical Transport Model                  !
!------------------------------------------------------------------------------
!BOP
!
! !IROUTINE: Compute_Column_Mass
!
! !DESCRIPTION: Subroutine Compute\_Budget\_Diagnostics calculates the
!  budget diagnostics for a given component by taking the difference of the
!  final and initial kg per grid cell and dividing by the timestep in seconds
!  to get kg/s.
!\\
!\\
! !INTERFACE:
!
  SUBROUTINE Compute_Budget_Diagnostics( Input_Opt,   State_Chm, State_Grid, &
                                         State_Met,   isFull,    diagFull,   &
                                         mapDataFull, isTrop,    diagTrop,   &
                                         mapDataTrop, isPBL,     diagPBL,    &
                                         mapDataPBL,  colMass,   RC,         &
                                         timeStep,    isWetDep,  before_op  )
!
! !USES:
!
    USE Input_Opt_Mod,  Only : OptInput
    USE State_Met_Mod,  ONLY : MetState
    USE State_Chm_Mod,  ONLY : ChmState
    USE State_Diag_Mod, ONLY : DgnMap
    USE State_Diag_Mod, ONLY : DgnState
    USE State_Grid_Mod, ONLY : GrdState
!
! !INPUT PARAMETERS:
!
    TYPE(OptInput), INTENT(IN)    :: Input_Opt        ! Input options object
    TYPE(GrdState), INTENT(IN)    :: State_Grid       ! Grid state object
    TYPE(MetState), INTENT(IN)    :: State_Met        ! Meteorology state obj
    LOGICAL,        INTENT(IN)    :: isFull           ! T if full col diag on
    TYPE(DgnMap),   POINTER       :: mapDataFull      ! Map to species indexes
    LOGICAL,        INTENT(IN)    :: isTrop           ! T if trop col diag on
    TYPE(DgnMap),   POINTER       :: mapDataTrop      ! Map to species indexes
    LOGICAL,        INTENT(IN)    :: isPBL            ! T if PBL col diag on
    TYPE(DgnMap),   POINTER       :: mapDataPBL       ! Map to species indexes
    LOGICAL,        OPTIONAL      :: isWetDep         ! T = wetdep budgets
    LOGICAL,        OPTIONAL      :: before_op        ! T = before operation
    REAL(f8),       OPTIONAL      :: timestep         ! F = after operation
!
! !INPUT/OUTPUT PARAMETERS:
!
    TYPE(ChmState), INTENT(INOUT) :: State_Chm        ! Chemistry state obj
    REAL(f8),       POINTER       :: diagFull(:,:,:)  ! ptr to full col diag
    REAL(f8),       POINTER       :: diagTrop(:,:,:)  ! ptr to trop col diag
    REAL(f8),       POINTER       :: diagPBL(:,:,:)   ! ptr to pbl col diag
    REAL(f8),       POINTER       :: colMass(:,:,:,:) ! Initial column mass
                                                      ! (I,J,spc,col region)
                                                      ! 1:full, 2:trop, 3:pbl
!
! !OUTPUT PARAMETERS:
!
    INTEGER,        INTENT(OUT)   :: RC               ! Success or failure?
!
! !REVISION HISTORY:
!  28 Aug 2018 - E. Lundgren - Initial version
!  See https://github.com/geoschem/geos-chem for complete history
!EOP
!------------------------------------------------------------------------------
!BOC
!
! !LOCAL VARIABLES:
!
    ! Scalars
    LOGICAL            :: after,  before, wetDep
    INTEGER            :: I,      J,      L,       N
    INTEGER            :: numSpc, region, topLev,  S
    REAL(f8)           :: colSum, dt

    ! Arrays
    REAL(f8)           :: spcMass(State_Grid%NZ)

    ! Strings
    CHARACTER(LEN=255) :: errMsg, thisLoc

    !====================================================================
    ! Compute_Budget_Diagnostics begins here!
    !====================================================================

    ! Initialize
    RC      =  GC_SUCCESS
    errMsg  = ''
    ThisLoc = ' -> at Compute_Column_Mass (in GeosCore/diagnostics_mod.F90)'
    colSum  = 0.0_f8
    spcMass = 0.0_f8

    ! Exit if concentrations are not in [kg/kg dry]
    IF ( State_Chm%Spc_Units /= 'kg/kg dry' ) THEN
       errMsg = 'State_Chm%Species units must be kg/kg dry. ' // &
                'Incorrect units: '// TRIM( State_Chm%Spc_Units )
       CALL GC_Error( errMsg, RC, ThisLoc )
       RETURN
    ENDIF

    ! Set logicals to denote if we are calling this routine
    ! before the operation or after the operation
    IF ( PRESENT( before_op ) ) THEN
       before = before_op
    ELSE
       before = .FALSE.
    ENDIF
    after = ( .not. before )

    ! Test if the budgets are for wetdep species
    IF ( PRESENT( isWetDep ) ) THEN
       wetDep = isWetDep
    ELSE
       wetDep = .FALSE.
    ENDIF

    ! Make sure the timeStep argument is passed (if after operation)
    IF ( after .and. ( .not. PRESENT( timeStep ) ) ) THEN
       errMsg = 'The timeStep argument was not passed!'
       CALL GC_Error( errMsg, RC, thisLoc )
       RETURN
    ENDIF

    ! Make sure mapDataFull and diagFull are not undefined
    IF ( isFull ) THEN
       IF ( .not. ASSOCIATED( mapDataFull ) ) THEN
          errMsg = 'The mapDataFull object is undefined!'
          CALL GC_Error( errMsg, RC, thisLoc )
          RETURN
       ENDIF
       IF ( after .and. ( .not. ASSOCIATED( diagFull ) ) ) THEN
          errMsg = 'The diagFull array is undefined!'
          CALL GC_Error( errMsg, RC, thisLoc )
          RETURN
       ENDIF
    ENDIF

    ! Make sure mapDataTrop and diagTrop are not undefined
    IF ( isTrop ) THEN
       IF ( .not. ASSOCIATED( mapDataTrop ) ) THEN
          errMsg = 'The mapDataTrop object is undefined!'
          CALL GC_Error( errMsg, RC, thisLoc )
          RETURN
       ENDIF
       IF ( after .and. ( .not. ASSOCIATED( diagTrop ) ) ) THEN
          errMsg = 'The diagTrop array is undefined!'
          CALL GC_Error( errMsg, RC, thisLoc )
          RETURN
       ENDIF
    ENDIF

    ! Make sure mapDataPBL and diagPBL are not undefined
    IF ( isPBL ) THEN
       IF ( .not. ASSOCIATED( mapDataPBL ) ) THEN
          errMsg = 'The mapDataPBL object is undefined!'
          CALL GC_Error( errMsg, RC, thisLoc )
          RETURN
       ENDIF
       IF ( after .and. ( .not. ASSOCIATED( diagPBL ) ) ) THEN
          errMsg = 'The diagPBL array is undefined!'
          CALL GC_Error( errMsg, RC, thisLoc )
          RETURN
       ENDIF
    ENDIF

    ! Make sure the colMass array is not undefined
    IF ( .not. ASSOCIATED( colMass ) ) THEN
       errMsg = 'The colMass array is undefined!'
       CALL GC_Error( errMsg, RC, thisLoc )
       RETURN
    ENDIF

    !====================================================================
    ! Before operation: Compute column masses (full, trop, PBL)
    !
    ! After operation:  Compute column differences (final-initial)
    !                   and them update diagnostic arrays
    !====================================================================

    ! Zero out the column mass array if we are calling this routine
    ! before the desired operation.  This will let us compute initial mass.
    IF ( before ) THEN
       colMass = 0.0_f8
    ENDIF

    ! Loop over NX and NY dimensions
    !$OMP PARALLEL DO                                        &
    !$OMP DEFAULT( SHARED                                  ) &
    !$OMP PRIVATE( I, J, colSum, spcMass, topLev, S, N, L  )
    DO J = 1, State_Grid%NY
    DO I = 1, State_Grid%NX

       ! Zero column-specific variables
       colSum  = 0.0_f8
       spcMass = 0.0_f8
       topLev  = 0

       !--------------------------------------------------------------------
       ! Full-column budget for requested species
       !--------------------------------------------------------------------
       IF ( isFull ) THEN

          ! Loop over # of diagnostic slots
          DO S = 1, mapDataFull%nSlots

             ! Initialize column-specfic variables
             colSum  = 0.0_f8
             spcMass = 0.0_f8

             ! For wetdep budgets, translate wetdep ID to modelId
             ! Otherwise, get the modelId from the slotId
             IF ( wetDep ) THEN
                N = State_Chm%Map_WetDep(mapDataFull%slot2Id(S))
             ELSE
                N = mapDataFull%slot2Id(S)
             ENDIF

             ! Compute mass at each grid box in the column [kg]
             DO L = 1, State_Grid%NZ
                spcMass(L) = State_Chm%Species(I,J,L,N) * State_Met%AD(I,J,L)
             ENDDO

             ! Compute the full-atmosphere column mass [kg]
             colSum = SUM( spcMass(1:State_Grid%NZ)  )

             ! Before operation: Compute initial full-atm column mass
             ! After operation: Compute change in column mass (final-initial),
             ! convert to [kg/s], and store in the diagFull array.
             IF ( before ) THEN
                colMass(I,J,N,1) = colSum
             ELSE
                diagFull(I,J,S) = ( colSum - colMass(I,J,N,1) ) / timeStep &
                                / State_Grid%AREA_M2(I,J)
             ENDIF
          ENDDO
       ENDIF

       !---------------------------------------------------------------------
       ! Troposphere-only budget for each requested species
       !---------------------------------------------------------------------
       IF ( isTrop ) THEN

          ! Top level in the column
          topLev = State_Met%TropLev(I,J)

          ! Loop over # of diagnostic slots
          DO S = 1, mapDataTrop%nSlots

             ! Initialize column-specfic variables
             colSum  = 0.0_f8
             spcMass = 0.0_f8

             ! For wetdep budgets, translate wetdep ID to modelId
             ! Otherwise, get the modelId from the slotId
             IF ( wetDep ) THEN
                N = State_Chm%Map_WetDep(mapDataTrop%slot2Id(S))
             ELSE
                N = mapDataTrop%slot2Id(S)
             ENDIF

             ! Compute mass at each grid box in the troposphere [kg]
             DO L = 1, topLev
                spcMass(L) = State_Chm%Species(I,J,L,N) * State_Met%AD(I,J,L)
             ENDDO

             ! Compute the trop-column mass [kg]
             colSum = SUM( spcMass(1:topLev) )

             ! Before operation: Compute initial trop-column mass
             ! After operation: Compute change in column mass (final-initial),
             ! convert to [kg/s], and store in the diagTrop array.
             IF ( before ) THEN
                colMass(I,J,N,2) = colSum
             ELSE
                diagTrop(I,J,S) = ( colSum - colMass(I,J,N,2) ) / timeStep &
                                / State_Grid%AREA_M2(I,J)
             ENDIF
          ENDDO
       ENDIF

       !---------------------------------------------------------------------
       ! PBL-only budget for each requested species
       !---------------------------------------------------------------------
       IF ( isPBL ) THEN

          ! Top level of column is where PBL top occurs
          topLev = MAX( 1, FLOOR( State_Met%PBL_TOP_L(I,J) ) )

          ! Loop over # of diagnostic slots
          DO S = 1, mapDataPBL%nSlots

             ! Initialize column-specfic variables
             colSum  = 0.0_f8
             spcMass = 0.0_f8

             ! For wetdep budgets, translate wetdep ID to modelId
             ! Otherwise, get the modelId from the slotId
             IF ( wetDep ) THEN
                N = State_Chm%Map_WetDep(mapDataPBL%slot2Id(S))
             ELSE
                N = mapDataPBL%slot2Id(S)
             ENDIF

             ! Compute mass at each grid box in the column [kg]
             DO L = 1, topLev
                spcMass(L) = State_Chm%Species(I,J,L,N) * State_Met%AD(I,J,L)
             ENDDO

             ! Compute column mass in PBL region [kg]
             colSum = SUM( spcMass(1:topLev) )

             ! Before operation: Compute initial PBL-column mass
             ! After operation: Compute change in column mass (final-initial),
             ! convert to [kg/s], and store in the diagPBL array.
             IF ( before ) THEN
                colMass(I,J,N,3) = colSum
             ELSE
                diagPBL(I,J,S) = ( colSum - colMass(I,J,N,3) ) / timeStep &
                               / State_Grid%AREA_M2(I,J)
             ENDIF
          ENDDO
       ENDIF

    ENDDO
    ENDDO
    !$OMP END PARALLEL DO

  END SUBROUTINE Compute_Budget_Diagnostics
!EOC
END MODULE Diagnostics_mod
