!====================================================================

!-----------------------------------------------------------------------
!       NASA/GSFC, Data Assimilation Office, Code 910.3, GEOS/DAS      !
!-----------------------------------------------------------------------
!BOP
! !MODULE: m_ArrList -- Utility routines for generating a list of arrays
!
! !DESCRIPTION:
!     This module contains functions for generating a list of arrays with
!     dimension sizes (or indices) and would be especially useful in 
!     a message for an alloction error. The list is returned as a
!     character string where each item is separated by commas.  Each item
!     in the list is generated by the generic function, {\tt Arr}, with
!     input arguments providing the name and (dimension or bounds).  For
!     example,
! \begin{verbatim}
!        use m_TextUtil, only : Array, ArrayList
!        character (len=*), parameter :: MyName = 'Example'
!        integer M,N, stat
!        M = 20
!        N = 10 
!        stat = 1
!        print *,  ' Item 1: ' // trim ( Arr ( 'A1', M       ) )
!        print *,  ' Item 2: ' // trim ( Arr ( 'A2', 0, N    ) )
!        print *,  ' Item 3: ' // trim ( Arr ( 'A3', (/M,N/) ) )
! \end{verbatim}
!     would produce the output,
! \begin{verbatim}
!        Item 1: A1(20)
!        Item 2: A2(0:10)
!        Item 3: A3(20,10)
! \end{verbatim}
!     Note that the trim function is used to remove any trailing blanks,
!     since the function returns a string of fixed length.  Thus, the
!     statement,
! \begin{verbatim}
!        print *,  Arr ( 'A1', M       ), &
!                  Arr ( 'A2', 0, N    ), &
!                  Arr ( 'A3', (/M,N/) )
! \end{verbatim}
!     would produce the output similar to 
! \begin{verbatim}
!        A1(20)
!                A2(0:10)
!                        A3(20,10)
! \end{verbatim}
!     To remove excessive blanks and insert commas between the items, the
!     function ArrList should be used.  For example, the following line,
! \begin{verbatim}
!        print *,   'Example array list: '   &
!                    trim ( ArrList ( Arr ( 'A1', M )     &
!                                  // Arr ( 'A2', 0, N )  &
!                                  // Arr ( 'A3', (/M,N/) ) ) )
! \end{verbatim}
!     would produce the output,
! \begin{verbatim}
!        Example array list: A1(20), A2(0:10), A3(20,10)
! \end{verbatim}
!     Note that the trim function needs to be invoked only once.
!
!     The ArrList function can be used to parse more complex arrays.  For
!     example, the statement, 
! \begin{verbatim}
!        print *,   trim ( ArrList ( ' Example array list: ' &
!                                   // Array ( 'World', N )  &
!                                   // Array ( '%A1', M ) ) )
! \end{verbatim}
!     would result in the message
! \begin{verbatim}
!        Example array list: World(10)%A1(20)
! \end{verbatim}
!     Additional features exist with the ArrList function.  For further
!     details and examples see the prologue to ArrList
!
! !INTERFACE:
!
      module    m_ArrList
      use       m_TextUtil, only : ItoA
      implicit  NONE
      private ! except

!     Subroutines and functions
!     -------------------------
      public :: Arr, Array, Arrays
                           ! Parse array(s) with given dimension.
      public :: ArrList    ! Edit array list whose items are generated by Arr

      interface Arr
         module procedure  ! function parameters:
     .      ArrSize,       ! ... ( names, size  )
     .      ArrSizes,      ! ... ( names, sizes )
     .      ArrLU,         ! ... ( names, lbound,      ubound     )
     .      ArrLUs,        ! ... ( names, lbound,      ubounds (:))
     .      ArrLsU,        ! ... ( names, lbounds (:), ubound     )
     .      ArrLsUs        ! ... ( names, lbounds (:), ubounds (:))
                           ! The parametes names lbounds and ubounds
      end interface        ! correspond to rank 1 arrays 

      interface Array
         module procedure
     .      ArrSize, ArrSizes, ArrLU, ArrLUs, ArrLsU, ArrLsUs
      end interface

      interface Arrays
         module procedure
     .      ArrSize, ArrSizes, ArrLU, ArrLUs, ArrLsU, ArrLsUs
      end interface
!
! !REVISION HISTORY:
!     13Feb2001  C. Redder  Original version
!     07Aug2002  C. Redder  Removed the module procedures, ArraysSize
!                           ArraysSizes, ArraysLU, ArraysLUs, ArraysLsU
!                           ArraysLsUs from the interfaces Arr and Arrays
!
! EOP
!-----------------------------------------------------------------
!
! !DEFINED PARAMETERS: 
!
!     Special constants for general use.
!     ----------------------------------
      character ( len =  1 ), parameter ::
     .   SP        = achar (  32 ), ! blank
     .   FSlash    = achar (  92 )  ! forward slash ('\')

!     Parameters for determining size of output strings
!     ------------------------------------------------- 
      integer, parameter ::
     .   LNameAve  = 2,             ! assumed average length of array name
     .   ArrLenMax = 50,            ! max string len of each output array name
     .   MaxLen    = 1000           ! maximum length of output array.

!     Defines boundaries of logical tables for characters
!     ---------------------------------------------------
      integer, parameter :: iCharMin = 0
      integer, parameter :: iCharMax = 127

!     Logical tables used to determine if a character is a letter
!     -----------------------------------------------------------
      integer, parameter  :: iBegLetter =   65 - iCharMin
      integer, parameter  :: iEndLetter = iCharMax - 122
      integer             :: iLetter
      logical, parameter, dimension ( iCharMin : iCharMax ) ::
     .         letter = ( / ( .false., iLetter = 1, iBegLetter ),
     .                      ( .true.,  iLetter = 1, 26 ),
     .                      ( .false., iLetter = 1,  6 ),
     .                      ( .true.,  iLetter = 1, 26 ),
     .                      ( .false., iLetter = 1, iEndLetter ) / )

      contains
!...................................................................

!-------------------------------------------------------------------------
!         NASA/GSFC, Data Assimilation Office, Code 910.3, GEOS/DAS      !
!-------------------------------------------------------------------------
! BOP
!
! !ROUTINE:  ArrList() --- Edits array list generated by Arr functions
! 
! !DESCRIPTION:
!     This function returns an edited list of arrays generated by the Arr
!     functions.  In particular, this function removes excessive blanks
!     between characters and inserts a separator between closing
!     parenthesis and a following letter (unless the separator already
!     exists).  For example, the following code,
!
! \begin{verbatim}
!        use m_TextUtil, only : Array, ArrayList
!        character (len=*), parameter :: MyName = 'Example'
!        integer M,N, stat
!        M = 20
!        N = 10 
!        stat = 1
!        print *,  'Example array list: ' &
!                // Array ( 'A1', M )     &
!                // Array ( 'A2', 0, N )  &
!                // Array ( 'A3', (/M,N/)
! \end{verbatim}
!     would generate the output similar to
! \begin{verbatim}
!        Example array list: A1(20)
!                                   A2(0:10)
!                                            A3(20,10)
! \end{verbatim}
!     Invoking the function \verb|Arr| in the statement,
! \begin{verbatim}
!        print *,   trim ( ArrList ( 'Example array list: ' &
!                                  // Array ( 'A1', M )     &
!                                  // Array ( 'A2', 0, N )  &
!                                  // Array ( 'A3', (/M,N/) ) ) )
! \end{verbatim}
!     would produce the more pleasing output,
! \begin{verbatim}
!        Example array list: A1(20), A2(0:10), A3(20,10)
! \end{verbatim}
!     The same output could be generated via the statement,
! \begin{verbatim}
!        print *, ' Example array list: '                 &
!                // trim ( Array ( 'A1', M ) )    // ', ' &
!                // trim ( Array ( 'A2', 0, N ) ) // ','  &
!                // trim ( Array ( 'A3', (/M,N/) ) )
! \end{verbatim}
!     If the first nonblank character after a closing parenthesis is
!     anything but a letter or a separator (i.e. \verb|,| or \verb|;|),
!     then no comma nor blanks are inserted.  For example, the statement, 
! \begin{verbatim}
!        print *,   trim ( ArrList ( ' Example array list: ' &
!                                   // Array ( 'World', N )  &
!                                   // Array ( '%A1', M ) ) )
! \end{verbatim}
!     would result in the message
! \begin{verbatim}
!        Example array list: World(10)%A1(20)
! \end{verbatim}
!     If the first nonblank character is a forward slash (\verb|\|),
!     then the character is ignored but no comma is inserted.  For
!     example, the statement, 
! \begin{verbatim}
!        print *,   trim ( ArrList ( 'Example array list: ' &
!                                  // Array ( 'A1', M )     &
!                                  // '\ does not end with a comma' ) )
! \end{verbatim}
!     will result in the output,
! \begin{verbatim}
!        Example array list: A1(20) does not end with a comma
! \end{verbatim}
!     instead of a message with a comma after the last parenthesis.
!
! !INTERFACE:
      function ArrList ( List )
!
! !INPUT PARAMETERS: 
      implicit NONE
      character (len=*), intent (in) ::
     .   List    ! Input list
!
!
! !OUTPUT PARAMETERS: 
      character (len = 3 * Len_Trim (List) / 2 ) :: ArrList
!
! !SEE ALSO: 
!
! !REVISION HISTORY:
!     16Jan2001  Redder    Origional code.
!
! EOP
!-------------------------------------------------------------------------

      integer   :: iBOS1, iEOS1, LOS1, iBOS2, iEOS2
      integer   :: LStr1, iArr,  iVer,  iVer2
      character :: VerCh

      ArrList = SP

      LStr1   = Len_Trim ( list )
      if ( LStr1 .eq. 0 ) return              ! Entire list is blank

      iBOS1   = 1
      iBOS2   = 1
      do iArr = 1, LStr1

!        If no significant characters are left ...
!        -----------------------------------------
         LOS1 = scan ( list ( iBOS1 : ), ')' )
         if ( LOS1 .eq. 0 ) then              ! ... process remaining portion
            iEOS2  = iBOS2 + LStr1 - iBOS1
            ArrList ( iBOS2 : iEOS2 ) = List ( iBOS1 : LStr1 )
            return                            ! ... and return

         else 
            iEOS1  = iBOS1 + LOS1 - 1
            iEOS2  = iBOS2 + LOS1 - 1
            iVer   = verify ( list ( iEOS1 + 1 : LStr1 ), SP )

!           If only blanks are left beyond current segment ...
!           --------------------------------------------------
            if ( iVer .eq. 0 ) then           ! ... process last segment
               ArrList ( iBOS2 : iEOS2 ) = List ( iBOS1 : iEOS1 )
               return                         ! ... and return

!           ... else if the next non-blank character is ... 
!           -----------------------------------------------
            else
               iVer  = iEOS1 + iVer
               VerCh = list ( iVer : iVer )

!              ... a letter
!              ------------
               if ( letter ( iachar ( VerCh ) ) ) then ! ... end of array
                  iEOS2 = iEOS2 + 2                    !     is assumed
                  ArrList ( iBOS2 : iEOS2 )
     .                  = List ( iBOS1 : iEOS1 ) // ',' // SP

!              ... a forward slash
!              -------------------
               else if ( VerCh .eq. FSlash ) then      ! ... treat as an
                  iVer  = iVer  + 1                    !     escape character
                  ArrList ( iBOS2 : iEOS2 )
     .                  = List ( iBOS1 : iEOS1 )

!              ... comma or semicolon
!              ---------------------- 
               else if ( VerCh .eq. ',' .or.
     .                   VerCh .eq. ';' ) then         ! ... treat as an
                  iEOS2 = iEOS2 + 2                    !     item seperator
                  ArrList ( iBOS2 : iEOS2 )
     .                  = List ( iBOS1 : iEOS1 ) // VerCh // SP
                  iVer2 = verify ( list ( iVer + 1 : ), SP )
                  if ( iVer2 .eq. 0 ) return           ! If so, then only
                  iVer2 = iVer2 + iVer                 !   blanks are left
                  iVer  = iVer  + 1                    ! Skip over seperator
                  if ( iVer2 .gt. iVer )               !   and any following
     .               iVer = iVer + 1                   !   blank(not blanks).

!              ... any other character (e.g. %)
!              --------------------------------
               else                                    ! ... just add item 
                  ArrList ( iBOS2 : iEOS2 )            !     with no changes
     .                  = List ( iBOS1 : iEOS1 )

               end if                                  
            end if
            iBOS1 = iVer
            iBOS2 = iEOS2 + 1

         end if

      end do

      return
      end function ArrList
!...................................................................

!-------------------------------------------------------------------------
!         NASA/GSFC, Data Assimilation Office, Code 910.3, GEOS/DAS      !
!-------------------------------------------------------------------------
! BOP
!
! !ROUTINE:  ArrSize() --- Parse 1-D array name(s) with given size or index 
! 
! !DESCRIPTION: This function parses the name(s) of one-dimensional array(s)
!               with a given size or index.
!
! !INTERFACE:
      function ArrSize ( names, size )
!
! !INPUT PARAMETERS: 
      implicit NONE
      character (len=*), intent (in) :: names ! Array name(s)
      integer,           intent (in) :: size  ! ... and size (or index)
!
! !OUTPUT PARAMETERS: 
      character (len=min ( MaxLen, max (LNameAve, ArrLenMax
     .                       * (len_trim(names)
     .                       / (LNameAve + 1) + 1)))) :: ArrSize
!
! !SEE ALSO: 
!
! !REVISION HISTORY:
!     11Jan2001  Redder    Origional code.
!
! EOP
!-------------------------------------------------------------------------

      integer, parameter :: lbound = 1
      ArrSize = ArrLsUs ( names, (/ lbound /), (/ size /) )

      return
      end function ArrSize
!...................................................................

!-------------------------------------------------------------------------
!         NASA/GSFC, Data Assimilation Office, Code 910.3, GEOS/DAS      !
!-------------------------------------------------------------------------
! BOP
!
! !ROUTINE:  ArrSizes() --- Parse array name(s) with given sizes or indices 
! 
! !DESCRIPTION: This function parses the name(s) of an array with given
!               sizes or indices
!
! !INTERFACE:
      function ArrSizes ( names, sizes )
!
! !INPUT PARAMETERS: 
      implicit NONE
      character (len=*), intent (in) ::
     .   names ! Array name(s)
      integer,           intent (in), dimension (:) ::
     .   sizes ! ... and its sizes.  The rank of the parsed array is
               !   determined by the size of this input array.
!
! !OUTPUT PARAMETERS: 
      character (len=min ( MaxLen, max (LNameAve, ArrLenMax
     .                       * (len_trim(names)
     .                       / (LNameAve + 1) + 1)))) :: ArrSizes
!
! !SEE ALSO: 
!
! !REVISION HISTORY:
!     11Jan2001  Redder    Origional code.
!     07Aug2002  Redder    Enable function to process a list of array names
!                          separated by commas.
! EOP
!-------------------------------------------------------------------------

      integer, parameter :: lbounds = 1
      ArrSizes = ArrLsUs ( names, (/ lbounds /), sizes )

      return
      end function ArrSizes
!...................................................................

!-------------------------------------------------------------------------
!         NASA/GSFC, Data Assimilation Office, Code 910.3, GEOS/DAS      !
!-------------------------------------------------------------------------
! BOP
!
! !ROUTINE:  ArrLU() --- Parse 1-D array name(s) with given bounds
! 
! !DESCRIPTION: This function parses the name(s) of one-dimensional array(s)
!               with the given lower and upper bounds.
!
! !INTERFACE:
      function ArrLU ( names, lbound, ubound )
!
! !INPUT PARAMETERS: 
      implicit NONE
      character (len=*), intent (in) ::
     .   names           ! Array name(s)
      integer,           intent (in) ::
     .   lbound,         ! ... lower bounds (A value of 1 is ignored)
     .   ubound          ! ... and upper bounds
!
! !OUTPUT PARAMETERS: 
      character (len=min ( MaxLen, max (LNameAve, ArrLenMax
     .                       * (len_trim(names)
     .                       / (LNameAve + 1) + 1)))) :: ArrLU
!
! !SEE ALSO: 
!
! !REVISION HISTORY:
!     11Jan2001  Redder    Origional code.
!     07Aug2002  Redder    Enable function to process a list of array names
!                          separated by commas.
! EOP
!-------------------------------------------------------------------------

      ArrLU = ArrLsUs ( names, (/ lbound /), (/ ubound /) )
      return
      end function ArrLU
*...................................................................

!-------------------------------------------------------------------------
!         NASA/GSFC, Data Assimilation Office, Code 910.3, GEOS/DAS      !
!-------------------------------------------------------------------------
! BOP
!
! !ROUTINE:  ArrLUs() --- Parse array name(s) with given bounds
! 
! !DESCRIPTION: This function parses the name(s) of array(s) with a given
!               lower bound and one or more values for the upper bounds.
!
! !INTERFACE:
      function ArrLUs ( names, lbound, ubounds )
!
! !INPUT PARAMETERS: 
      implicit NONE
      character (len=*), intent (in) ::
     .   names       ! Array name(s)
      integer,           intent (in) ::
     .   lbound,     ! ... with the lower bound (A value of 1 is ignored.)
     .   ubounds (:) ! ... and the upper bounds.  The rank of the parsed
                     !   array is determined by the size of this input array.
!
! !OUTPUT PARAMETERS: 
      character (len=min ( MaxLen, max (LNameAve, ArrLenMax
     .                       * (len_trim(names)
     .                       / (LNameAve + 1) + 1)))) :: ArrLUs
!
! !SEE ALSO: 
!
! !REVISION HISTORY:
!     11Jan2001  Redder    Origional code.
!     07Aug2002  Redder    Enable function to process a list of array names
!                          separated by commas.
! EOP
!-------------------------------------------------------------------------

      ArrLUs = ArrLsUs ( names, (/ lbound /), ubounds )

      return
      end function ArrLUs

!...................................................................
!-------------------------------------------------------------------------
!         NASA/GSFC, Data Assimilation Office, Code 910.3, GEOS/DAS      !
!-------------------------------------------------------------------------
! BOP
!
! !ROUTINE:  ArrLsU() --- Parse array name(s) with given bounds
! 
! !DESCRIPTION: This function parses the name(s) of array(s) with a given
!               upper bound and one or more values for the lower bounds.
!
! !INTERFACE:
      function ArrLsU ( names, lbounds, ubound )
!
! !INPUT PARAMETERS: 
      implicit NONE
      character (len=*), intent (in) ::
     .   names        ! Array name(s)
      integer,           intent (in) ::
     .   lbounds (:), ! ... with the lower bounds (A value of 1 is ignored)
     .   ubound       ! ... and the upper bounds.  The rank of the parsed
                      !   array is determined by the size of the input array,
                      ! lbounds.
!
! !OUTPUT PARAMETERS: 
      character (len=min ( MaxLen, max (LNameAve, ArrLenMax
     .                       * (len_trim(names)
     .                       / (LNameAve + 1) + 1)))) :: ArrLsU
!
! !SEE ALSO: 
!
! !REVISION HISTORY:
!     11Jan2001  Redder    Origional code.
!     07Aug2002  Redder    Enable function to process a list of array names
!                          separated by commas.
! EOP
!-------------------------------------------------------------------------

      ArrLsU = ArrLsUs ( names, lbounds, (/ ubound /) )

      return
      end function ArrLsU
!...................................................................

!-------------------------------------------------------------------------
!         NASA/GSFC, Data Assimilation Office, Code 910.3, GEOS/DAS      !
!-------------------------------------------------------------------------
! BOP
!
! !ROUTINE:  ArrLsUs() --- Parse array name(s) with given bounds
! 
! !DESCRIPTION: This function parses the name(s) of array(s) with given
!               lower and upper bounds.
!
! !INTERFACE:
      function ArrLsUs ( names, lbounds, ubounds )
!
! !INPUT PARAMETERS: 
      implicit NONE
      character (len=*), intent (in) ::
     .   names    ! Array name(s)
      integer,           intent (in), dimension (:) ::
     .   lbounds, ! ... with the lower bounds (A value of 1 is ignored.)
     .   ubounds  ! ... and the upper bounds.
!
!        Note: If the size(lbounds) .ne. size(ubounds), then the maximum
!              among the pair becomes the rank of the parsed array.  The  
!              last value of the smaller input array is used for the
!              remaining dimensions.
!
! !OUTPUT PARAMETERS: 
      character (len=min ( MaxLen, max (LNameAve, ArrLenMax
     .                       * (len_trim(names)
     .                       / (LNameAve + 1) + 1)))) :: ArrLsUs
!
! !SEE ALSO: 
!
! !REVISION HISTORY:
!     11Jan2001  Redder    Origional code.
!     07Aug2002  Redder    Enable function to process a list of array names
!                          separated by commas.
! EOP
!-------------------------------------------------------------------------


      character ( len = 25 ) :: ALB, AUB
      integer :: LenALB, LenAUB, LName, LItem, LNames, LOD
      integer :: lbound, ubound
      integer :: NDim, NLB,  NUB,  iBOI, iEOI, iBON, iEON
      integer :: iCh,  iDim, iPtr, iBeg, iEnd, iBOD, iEOD
      integer :: NChar_Max

      LNames = len_trim ( names   )
      NLB    = size     ( lbounds )
      NUB    = size     ( ubounds )
      NDim   = max      ( NLB, NUB )
      NChar_Max = len ( ArrLsUs )

!     For each item in arrays ...
!     ---------------------------
      ArrLsUs  = SP
      iBOI     = 1
      iBOD     = 0
      iPtr     = 1
      EachArr : do iCh = 1, LNames

!        ... exit loop if no more items exist in the list
!        ------------------------------------------------
         if ( iBOI .gt. LNames ) exit EachArr

!        ... determine ...
!        -----------------
         LItem = scan ( names ( iBOI : ), ',') - 1        ! ... length ...
         if ( LItem .lt. 0 ) LItem = len_trim ( names ( iBOI : ))
         iEOI  = iBOI + LItem - 1                         ! ... end of item

!        Locate name to remove ...
!        -------------------------
         iBON  = max ( 1, verify ( names ( iBOI : iEOI ), SP ))
     .         + iBOI - 1                                 ! ... leading and
         LName = len_trim ( names ( iBON : iEOI ))
         iEON  = iBON + LName - 1                         ! ... trailing
                                                          !     blanks

!        If the name does not consist of only blanks ...
!        -----------------------------------------------
         if ( LName .gt. 0 ) then
            if ( iPtr .gt. NChar_Max ) exit EachArr       ! Exit loop if no
                                                          !   more space in
!           ... add array name to list if the item is ... !   output string
!           ---------------------------------------------
            iBeg = iPtr 
            if ( iBeg .eq. 1 ) then                       ! ... the first
               iEnd  = min ( iPtr + LName - 1, NChar_Max) ! -------------
               ArrLsUs ( iBeg : iEnd )
     .               = names ( iBON : iEON )
            else                                          ! ... not the first
               iEnd  = min ( iPtr + LName + 1, NChar_Max) ! Add 2 for leading
               ArrLsUs ( iBeg : iEnd )                    !   comma and blank
     .               = ',' // SP // names ( iBON : iEON ) !-----------
            end if

            iPtr = iEnd + 1
            if ( iPtr .gt. NChar_Max ) exit EachArr       ! No more space   

!           ... include the bounds/indices (if non-scalar)
!           ----------------------------------------------
            if ( NDim .gt. 0 ) then
               if ( iBOD .gt. 0 ) then            ! ... if already generated
                  iBeg = iPtr                     ! ------------------------
                  iEnd = min ( iBeg + LOD - 1, NChar_Max )
                  iEOD = iBOD + LOD - 1
                  ArrLsUs ( iBeg : iEnd )
     .                 = ArrLsUs ( iBOD : iEOD )
                  iPtr = iEnd + 1
               else                               ! ... if not already
                  if ( iPtr .gt. NChar_Max ) exit EachArr
                  ArrLsUs ( iPtr : iPtr ) = '('   !     generated
                  iBOD = iPtr                     ! -------------
                  iPtr = iPtr + 1

!                 For each array dimension ...
!                 ----------------------------
                  EachDim : do iDim = 1, NDim

                     lbound    = 1
                     if ( NLB .gt. 0 )
     .                  lbound = lbounds ( min ( iDim, NLB ) )
                     ubound    = 1
                     if ( NUB .gt. 0 )
     .                  ubound = ubounds ( min ( iDim, NUB ) )

!                    If lower bounds is 1 ...
!                    ------------------------
                     if ( lbound .eq. 1 ) then    ! ... insert only the
                        AUB    =  ItoA ( ubound ) !     size/index
                        LenAUB =  Len_Trim ( AUB )! -------------------
                        iBeg   =  iPtr
                        iEnd   =  min ( iBeg + LenAUB, NChar_Max )
                        ArrLsUs  ( iBeg : iEnd )
     .                         =  AUB ( : LenAUB ) // ','

                     else                         ! ... insert the range
                        ALB    =  ItoA ( lbound ) ! --------------------
                        AUB    =  ItoA ( ubound )
                        LenAUB =  Len_Trim ( AUB )
                        LenALB =  Len_Trim ( ALB )
                        iBeg   =  iPtr
                        iEnd   =  min ( iBeg + LenALB + LenAUB + 1,
     .                                  NChar_Max )
                        ArrLsUs  ( iBeg : iEnd )
     .                         =  ALB ( : LenALB ) // ':'
     .                         // AUB ( : LenAUB ) // ','
                     end if
                     iPtr = iEnd + 1

                  end do EachDim
                  LOD = iPtr - iBOD

               end if

!              Overwrite the last ',' with closing parenthesis
!              -----------------------------------------------
               iEnd = iPtr - 1
               if ( iEnd .le. NChar_Max ) then
                  ArrLsUs ( iEnd : iEnd ) = ')'

               else
                  exit EachArr

               end if
            end if
         end if

!        Determine the location of the first
!        character of the next item in the list
!        --------------------------------------
         iBOI = iEOI + 2
      end do EachArr
      return

      end function ArrLsUs
!...................................................................
      end module m_ArrList

!====================================================================
