#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
@date: 2018-Today
@author: jasper.bathmann@ufz.de
"""
import vtk
import numpy as np
import getopt
import sys




class BoundaryOf2DMesh:
    ## This class provides functionalities to extract the boundary line of a
    #  planar 2D grid, if the face normal is known.
    def __init__(self, source_mesh_name, boundary_normal, 
                 face_normal, deviation_angle):
        ## Constructor; some configuration
        #  @param source_mesh_name name of the 2d mesh
        #  @param boundary_normal vector normal to boundary line in 3D
        #  @param face_normal normal of the 2d input mesh
        #  @param deviation_angle angle of possible deviations from boundary
        #         normal vector
        self.source_mesh_name = source_mesh_name
        self.boundary_normal = boundary_normal
        self.face_normal = face_normal
        self.deviation_angle = deviation_angle
            
    def read2DMesh(self):
        ## Reads the 2D source mesh and converts it to a vtkPolyData
        meshReader = vtk.vtkXMLUnstructuredGridReader()
        meshReader.SetFileName(self.source_mesh_name)
        meshReader.Update()
        grid = meshReader.GetOutput()
        extract = vtk.vtkGeometryFilter()
        extract.SetInputData(grid)
        extract.SetOutputPointsPrecision(vtk.vtkAlgorithm.DOUBLE_PRECISION)
        extract.Update()
        self.source_grid = (extract.GetOutput())

    def extractBoundaryLine(self):
        ## Extracts the boundary line following the steps
        #  1.) Name point ids
        #  2.) apply featured Edges filter to extract full boundary
        #  3.) saving of boundary cells, points and the grid
        idFilter = vtk.vtkIdFilter()
        idFilter.SetInputData(self.source_grid)
        #idFilter.SetCellIdsArrayName("ids")
        idFilter.SetPointIds(True)
        idFilter.SetCellIds(False)
        # Available for vtk>=8.3:
        idFilter.SetPointIdsArrayName("ids")
        idFilter.Update()
            
        source = (idFilter.GetOutput())
        edges = vtk.vtkFeatureEdges()
        edges.SetInputData(source)
        edges.BoundaryEdgesOn()
        edges.ManifoldEdgesOff()
        edges.NonManifoldEdgesOff()
        edges.FeatureEdgesOff()
        edges.SetOutputPointsPrecision(vtk.vtkAlgorithm.DOUBLE_PRECISION)
        edges.Update()
        self.boundary = edges.GetOutput()
        self.boundary_cells = self.boundary.GetLines()
        self.boundary_points = self.boundary.GetPoints()
        
    def identifyBoundaryPoints(self):
        ## This function identifies the boundary points which are on the 
        #  boundary line perpendicular to self.boundary_normal
        #  Steps:
        #  1.) Iteration over all boundary cells
        #  2.) calculation of vector normal to boundary line and boundary normal
        #  3.) calculation of angle between the normal vector and the facenormal
        #  4.) storing of indices
        #  5.) Application of correct mapping of indices
        self.boundary_pts = vtk.vtkIdList()
        self.boundary_cells.InitTraversal()
        self.new_boundary_points = []
        self.new_boundary_cells_idx0 = []
        self.new_boundary_cells_idx1 = []
        self.cell_index_mapping = []
        for i in range(self.boundary.GetNumberOfCells()):
            self.boundary_cells.GetNextCell(self.boundary_pts)
            id0 = self.boundary_pts.GetId(0)
            id1 = self.boundary_pts.GetId(1)
            p0 = self.boundary.GetPoint(id0)
            p1 = self.boundary.GetPoint(id1)

            vector = np.array(p1, dtype=np.float128)-np.array(p0, dtype=np.float128)
            vector_length = np.sqrt(np.dot(vector, vector))

            cross_product = np.cross(self.boundary_normal,vector)
            projected_cross_product = np.dot(self.face_normal,cross_product)

            face_normal_length = np.sqrt(np.dot(self.face_normal, 
                                                self.face_normal))

            sin_angle = (projected_cross_product / (face_normal_length 
                                                    *vector_length))
            angle = np.arcsin(sin_angle)

            phi = angle/(2*np.pi) * 360
            if np.abs(phi-90) < self.deviation_angle:
                self.new_boundary_cells_idx0.append(id0)
                self.new_boundary_cells_idx1.append(id1)
                self.cell_index_mapping.append(i)
                if id0 not in self.new_boundary_points:
                    self.new_boundary_points.append(id0)
                if id1 not in self.new_boundary_points:
                    self.new_boundary_points.append(id1)
        self.createIdMapping()

    def createIdMapping(self):
        ## Mapping from old point indices to new point indices. Required for
        #  self.initializeNewGrid
        mapping = dict(zip(self.new_boundary_points, 
                           np.arange(len(self.new_boundary_points))))
        self.new_boundary_cells_idx0 = [mapping[elem] for elem 
                                        in self.new_boundary_cells_idx0]
        self.new_boundary_cells_idx1 = [mapping[elem] for elem 
                                        in self.new_boundary_cells_idx1]   

    def initializeNewGrid(self):
        ## Creates new grid using the information extracted earlier and saves 
        #  in new_boundary_points and new_boundary_cells_idx0 and ...1
        self.new_grid = vtk.vtkUnstructuredGrid()
        new_points = vtk.vtkPoints()
        for i in self.new_boundary_points:
            new_points.InsertNextPoint(self.boundary_points.GetPoint(i))
        new_cells=vtk.vtkCellArray()
        new_cells.InitTraversal()
        for i in range(len(self.new_boundary_cells_idx0)):
            cell = vtk.vtkLine()
        
            cell.GetPointIds().SetId(0, self.new_boundary_cells_idx0[i])
            cell.GetPointIds().SetId(1, self.new_boundary_cells_idx1[i])
            new_cells.InsertNextCell(cell)

        self.new_grid.SetCells(3,new_cells)
        self.new_grid.SetPoints(new_points)  

    def resamplePointData(self, array_name, data_type):
        ## Resamples point data from source mesh to new boundary line
        #  @param array_name name of the original data array
        #  @param array_type type of the original data array
        new_point_array = vtk.vtkDataArray.CreateDataArray(
            data_type)
        new_point_array.SetName(array_name)
        old_point_array = self.boundary.GetPointData().GetArray(array_name)
        for i in self.new_boundary_points:
            new_point_array.InsertNextTuple1(int(old_point_array.GetTuple(i)[0]))
        self.new_grid.GetPointData().AddArray(new_point_array)

    def resampleCellData(self, array_name, data_type):
        ## Resamples cell data from source mesh to new boundary line
        #  @param array_name name of the original data array
        #  @param array_type type of the original data array
        new_cell_array = vtk.vtkDataArray.CreateDataArray(
                    data_type)
        new_cell_array.SetName(array_name)
        old_cell_array = self.boundary.GetCellData().GetArray(array_name)
        for i in self.cell_index_mapping:
            new_cell_array.InsertNextTuple1(int(old_cell_array.GetTuple(i)[0]))
        self.new_grid.GetCellData().AddArray(new_cell_array)

    def writeOutput(self, output_name):
        ## Write output
        #  @param output_name name of resulting line-mesh
        writer = vtk.vtkXMLUnstructuredGridWriter()
        writer.SetFileName(output_name)
        writer.SetInputData(self.new_grid)
        if self.new_grid.GetNumberOfPoints() == 0:
            print("""
WARNING! Empty mesh is written by the script!
Check input configuration""")
        writer.Write()
        
        



def main(argv):
    try:
        opts, args = getopt.getopt(argv, "hm:o:a:", ["mesh_name=", 
                                                 "bnx=",
                                                 "bnz=",
                                                 "bny=",
                                                 "fnx=",
                                                 "fny=",
                                                 "fnz=",
                                                 "output_name=",
                                                 "angle="])
    except getopt.GetoptError:
        print("""wrong usage. Type "python boundary_of_2d_mesh.py -h"
  for additional help.""")
        sys.exit(0)
    for opt, arg in opts:
        if opt == '-h':
            print("""arguments:
  -m,--mesh_name <name of mesh, of which the boundary lines are extracted>
  -o, --output_name <name of output>
  --bnx <x>
  --bny <y>
  --bnz <z>
  --fnx <x>
  --fny <y>
  --fnz <z>
  -f,--face_normal <[x,y,z]>
  -a, --angle""")
            sys.exit()
        elif opt in ("-m", "--mesh_name"):
            mesh_name = str(arg)
        elif opt in ("space", "--bnx"):
            bx = float(arg)
        elif opt in ("space", "--bny"):
            by = float(arg)
        elif opt in ("space", "--bnz"):
            bz = float(arg)
        elif opt in ("space", "--fnx"):
            fx = float(arg)
        elif opt in ("space", "--fny"):
            fy = float(arg)
        elif opt in ("space", "--fnz"):
            fz = float(arg)
        elif opt in ("-o", "--output_name"):
            output_name = str(arg)
        elif opt in ("-a", "--angle"):
            deviation_angle = float(arg)


    try:
        boundary_normal = np.array([bx,by,bz])
        face_normal = np.array([fx,fy,fz])
        b2mesh = BoundaryOf2DMesh(mesh_name, boundary_normal, face_normal,
                                  deviation_angle)
        b2mesh.read2DMesh()
        b2mesh.extractBoundaryLine()
        b2mesh.identifyBoundaryPoints()
        b2mesh.initializeNewGrid()
        b2mesh.resamplePointData("bulk_node_ids", vtk.VTK_UNSIGNED_LONG)
        b2mesh.resampleCellData("bulk_face_ids", vtk.VTK_UNSIGNED_LONG)
        b2mesh.resampleCellData("bulk_element_ids", vtk.VTK_UNSIGNED_LONG)
        b2mesh.resampleCellData("MaterialIDs", vtk.VTK_INT)
        b2mesh.writeOutput(output_name)
    except UnboundLocalError:
        raise UnboundLocalError('Wrong usage. Type "python' +
                                ' boundary_of_2d_mesh.py -h" for help.')


if __name__ == "__main__":
    main(sys.argv[1:])




