'''
assign position dependent material properties

'''
import numpy as np      # for operations and data types
from scipy.interpolate import RegularGridInterpolator   # for 3D interpolation
import meshio   # read/write meshes
import sys

# TODO make usage fail-safe
inputfile = sys.argv[1]
outputfile =  sys.argv[2]



mesh = meshio.read(inputfile)
print(f'MESH: {len(mesh.points)} points')       # array
print(f'MESH: {len(mesh.cells[0].data)} cells')   # dictionary


z_min = min(mesh.points[:,2])   # TODO autodetect
z_max = max(mesh.points[:,2])  
dz = z_max - z_min

k_min = 1e-11 * 1e-7  # ignore w=0 of blocks outside domain
k_max = 1e-6 * 1e-7
dk = k_max - k_min
# add interpolated permeability values to to mesh ([0,LX]  [0,LY] [0,LZ])
interpolating_function = lambda P: k_min + (P[:,2]-z_min)*dk 

cells_count = len(mesh.cells[0][1])
centerpoints=np.zeros((cells_count, 3))
for cell_index, cellblock_cell in enumerate(mesh.cells[0][1]):
    centerpoints[cell_index] = np.sum(mesh.points[cellblock_cell], axis=0) / len(cellblock_cell)
    
w_interpolation = interpolating_function(centerpoints)  

k_cell = w_interpolation * 1e-7   # k=w_c*mu/(rho*g)

# linear interpolation (increasing) for porosity
phi_min = 0.0015
phi_max = 0.0050
dphi = phi_max - phi_min
phi_cell = phi_min + dphi*(k_cell-k_min)/dk

# linear interpolation (decreasing) for Young's modulus
E_min = 5.630e10
E_max = 7.870e10
dE = E_max - E_min
E_cell = E_max - dE*(k_cell-k_min)/dk

min_k_cell = min(k_cell)   
if min_k_cell < 0:
    print("Warning, negative value k={} found, shifted to zero.".format(min_k_cell))
    k_cell -= min_k_cell  # shift to nonnegative values

mesh.cell_data['vm_permeability'] = [np.array(k_cell)]   # append data
mesh.cell_data['vm_porosity'] = [np.array(phi_cell)]   # append data
mesh.cell_data['vm_youngs_modulus'] = [np.array(E_cell)]   # append data


# write to file
meshio.write(outputfile, mesh)
print("Appended cell data: permeability, porosity, youngs modulus")
# seems compressed, as it is smaller than original file (ogs/generatemesh) although data were added

