#!/bin/bash
# Before running this script make sure you have setup your env.sh (install directory)

short_help()
{
   # Display Help
   echo
   echo "-----------------------------------------------------------------------------------------"
   echo
   echo "3D_workflow.sh: runs deep geological repository workflow with 3D meshes (Voxel Grid)."
   echo
   echo "Brief USAGE:"
   echo "Syntax: sh 2D_workflow.sh [b|c|h|i|m|p|q|s]"
   echo '  [-i <MODEL_FILEPATHNAME>]    mandatory'
   echo "  [-m <MODEL_NAME]             mandatory"
   echo "  [-p <PROJECT_NAME>]          mandatory"
   echo "  [-c <CONTAINER_SIF_FILE>]    optional, default: defined by env.sh"
   echo "  [-q <QUAD_MESH>]             optional, default: inactive"
   echo "  [-v <VOXELSIZES>             optional, either v or u"
   echo "  [-u <UNSTRUCTUREDGRID>       optional, either v or u"
   echo "  [-e EXPERIMENT_NAME          optional, default: new folder name by parameter"
   echo "  [-l                          developer only, optional, default: automatic"
   echo "  [-n                          developer only, optional, default: no reference results"
   echo "  [-r <MODEL_REV>              developer only, optional, default: latest"
   echo "  [-t <TEST_VARIABLE>          developer only, optional, default: not-active"
   echo
   echo "sh 2D_workflow.sh -h for further help"
   echo
   echo "-----------------------------------------------------------------------------------------"

}

long_help()
{
    short_help
   echo
   echo "<CONTAINER_SIF_FILE>"
   echo "    absolute path to docker container file with 'sif' file ending"
   echo "    e.g. -c ~/dgr/container/images/ogs-dgr_latest.sif"
   echo
   echo "<MODEL_NAME>"
   echo "    Ton-Nord"
   echo "    e.g. -p Ton-Nord, Generic-Model"
   echo
   echo "<MODEL_FILENAME>"
   echo "    full path name to GoCadTSurface file that represents 2D layers in 3D structure "
   echo "    e.g. -i Basisflaechen_Ton-Nord.ts"
   echo "    OR"
   echo "    leave empty - Then it is expected that all meshes required by the project file"
   echo "    are in same folder as the project file"
   echo
   echo "<PROJECT_NAME>"
   echo "    simH_1 or simH_2a or simH_2b or simHM_1"
   echo "    A file with name equal to PROJECT_NAME and .prj extension in subfolder input"
   echo "    e.g. -m simH_1"
   echo
   echo "<QUAD_MESH>"
   echo "    0 or 1"
   echo "    e.g. -q 1"
   echo
   echo "<UNSTRUCTURED_GRID_PARAMETER_SIZE>"
   echo "    double quoted and space delimited list of 2 values representing size in RASTER_SIZE MESH_DENSITY_SCALING"
   echo "    <RASTER_SIZE> positive integer e.g. 50
   echo "    <MESH_DENSITY_SCALING> positive floating point number e.g. 0.05
   echo "    e.g. -u \"200 0.05\""
   echo
   echo "<VOXEL_SIZE>"
   echo "    double quoted and space delimited list of 3 values representing voxel size in x y z"
   echo "    e.g. -v \"500 500 250\""
}


# ========== DEFAULTS FROM GLOBAL CONFIG =======
OGS_DGR_VERSION=v0.5RC

SCRIPT=`realpath $0`
SCRIPTPATH=`dirname $SCRIPT`
DGR_ROOT=`realpath $SCRIPTPATH/../`
ENV_SCRIPT=$DGR_ROOT/install/env.sh
if [ -f "$ENV_SCRIPT" ]; then
    echo
else
    echo "Error: envh.sh not found. Expected it here: $ENV_SCRIPT."
    exit 3
fi
. $DGR_ROOT/install/env.sh

# ================ LOCAL CONFIG DEFAULTS ===================
# give a name to gocad input files the locate in $DGR_ROOT/input
MODEL_NAME=
MODEL_FILENAME=
# give a name to a ogs project file that locates in manual input folder

PROJECT_NAME=
QUADMESH=
OGS_SIMULATION_ONLY=
DOMAIN_MESH_DEFINED=
PROJECT_FILE_DEFINED=

# Default take latest revision of models
MODEL_REV=


# ================ COMMAND LINE OPTIONS ===================
# may overwrite global or local config variables
while getopts ":p::m::i::r::q::c::e::u::v::t:nhlo" opt; do
    case $opt in
        h)
            long_help
            exit
        ;;
        p)
            PROJECT_NAME=${OPTARG}
        ;;
        l)
            # Per default all layers found will be considered in vertical slice
            # It is possible to provide custom layer.txt with only a subset of layers
            AUTO_SELECT_LAYERS=0
        ;;
        m)
            MODEL_NAME=${OPTARG}
        ;;
        n)
            NEW_REFERENCE_RESULT=1
        ;;
        i)
            MODEL_FILENAME=${OPTARG}
        ;;
        r)
            MODEL_REV=${OPTARG}
        ;;
        o)
            OGS_SIMULATION_ONLY=1
        ;;
        q)
            QUADMESH=${OPTARG}
        ;;
        t)
            TEST_VARIABLE=${OPTARG}
        ;;
        v)
            VOXELSIZES=${OPTARG}
        	VOXELSIZE_X=`echo $VOXELSIZES | cut --delimiter " " --fields 1`
			VOXELSIZE_Y=`echo $VOXELSIZES | cut --delimiter " " --fields 2`
			VOXELSIZE_Z=`echo $VOXELSIZES | cut --delimiter " " --fields 3`
        ;;
        u)
            UNSTRUCTURED_SIZES=${OPTARG}
            RASTER_SIZE=`echo $UNSTRUCTURED_SIZES | cut --delimiter " " --fields 1`
			MESH_DENSITY_SCALING=`echo $UNSTRUCTURED_SIZES | cut --delimiter " " --fields 2`
            if [ $RASTER_SIZE -eq $MESH_DENSITY_SCALING ]; then
                echo "double quoted and space delimited list of 2 values must be provided"
                echo "    e.g. -u \"200 0.05\""
                exit 1
            fi


            UNSTRUCTURED="${RASTER_SIZE}_${MESH_DENSITY_SCALING}"
            echo UNSTRUCTURED
            ;;
        c)
            USE_CONTAINER=2
            OGS_CONTAINER=${OPTARG}
        ;;
        e)
            EXPERIMENT=${OPTARG}
        ;;
        \?)
            echo "Invalid option: -$OPTARG" >&2
            short_help
            exit 1
        ;;
        :)
            echo "Option -$OPTARG requires an argument." >&2
            short_help
            exit 1
        ;;
    esac
done

if [ -z "$PROJECT_NAME" ]; then
    echo "error: Process name NOT given, use -p option."
    short_help
    exit 1;
fi



if [ -z "$OGS_SIMULATION_ONLY" ]; then
    if [ -z "$MODEL_FILENAME" ]; then
        echo "error: Model filename NOT given, use -i option."
        short_help
        exit 1;
    fi

    if [ -z "$MODEL_NAME" ]; then
        echo "error: Model name NOT given, use -m option exit."
        short_help
        exit 1;
    fi

    if [ -z "$VOXELSIZES" ] && [ -z "$UNSTRUCTURED" ]; then
        echo "error: Neither Voxel sizes NOR Unstructured grid resolution given, use -v OR -u option."
        short_help
        exit 1;
    fi

else
    DOMAIN_MESH_DEFINED=1
    PROJECT_FILE_DEFINED=1
    echo "warning: Custom OGS Project file is provided - all other options will be ignored"
fi # OGS_SIMULATION_ONLY

# ================ DERIVED VARIABLES ===================

MANUAL_INPUT=$DGR_ROOT/input/models/${MODEL_NAME}
if [ -n "$VOXELSIZES" ]; then
    FOLDER_NAME="${MODEL_NAME}_${SLICE_ID}_${PROJECT_NAME}_V_${VOXELSIZE_X}_${VOXELSIZE_Y}_${VOXELSIZE_Z}_"
elif [ -n "$UNSTRUCTURED" ]; then
    FOLDER_NAME="${MODEL_NAME}_${SLICE_ID}_${PROJECT_NAME}_U_${UNSTRUCTURED}_"
else
    FOLDER_NAME="${MODEL_NAME}_${SLICE_ID}_${PROJECT_NAME}"
fi

if [ -n "$EXPERIMENT" ]; then
    FOLDER_NAME=$EXPERIMENT
fi

MODELS_DIR=$DGR_ROOT/input/models/${MODEL_NAME}/${MODEL_REV}/
MODEL_FILEPATHNAME=$MODELS_DIR/$MODEL_FILENAME

# =================== HELPER ==============================

# Using local software
if [ "$USE_CONTAINER" = "0" ]; then
    echo "Using local software"
    export run=" "
    # we assume that all other bin are already on path
    # that is why only OGSBIN is added to path
    export PATH="$PATH:$OGSBIN"
    # Using development container
elif [ "$USE_CONTAINER" = "2" ]; then
    export PATH="$PATH:$SINGULARITYBIN"
    export run="singularity -s exec $OGS_CONTAINER"
    # Using official container
else
    OGS_OFFICIAL_CONTAINER=$DGR_ROOT/container/images/ogs-dgr_$OGS_DGR_VERSION.sif
    export PATH="$PATH:$SINGULARITYBIN"
    export run="singularity -s exec $OGS_OFFICIAL_CONTAINER"

    if [ -f "$OGS_OFFICIAL_CONTAINER" ]; then
        echo "Container found on $OGS_OFFICIAL_CONTAINER."
    else
        echo "Container $OGS_OFFICIAL_CONTAINER not found."
        exit 3
    fi

fi
echo $run

# ================ WORKING DIRECTORY  ===================
if [ "$DGR_WORK" = "" ]; then
    export RESULT_DIR=$FOLDER_NAME
else
    export RESULT_DIR=$DGR_WORK/$FOLDER_NAME
fi
mkdir -p $RESULT_DIR
cd $RESULT_DIR
echo "Results will be saved to ${RESULT_DIR}"


if [ "$MODEL_NAME" = "Generic-Model" ]; then
    DOMAIN_MESH_DEFINED=1
    if [ "$PROJECT_NAME" = "ys_static_loading_onlyH" ]; then
    ELEMENT_ORDER=1
    else
    ELEMENT_ORDER=2
    fi
    echo "= Generate Generic-Model"
    cp $DGR_ROOT/input/models/Yeniseisky_Site/EO$ELEMENT_ORDER/*.vtu .

    echo "= Generate Projectfile ="
    echo $DGR_ROOT/input/models/Yeniseisky_Site/${PROJECT_NAME}.prj
    cp $DGR_ROOT/input/models/Yeniseisky_Site/${PROJECT_NAME}.prj .
else
    if [ "$PROJECT_NAME" = "simH_1" ]; then
        ELEMENT_ORDER=1
    elif [ "$PROJECT_NAME" = "simT_1" ]; then
        ELEMENT_ORDER=1
    elif [ "$QUADMESH" = "1" ]; then
        ELEMENT_ORDER=2
	else
        ELEMENT_ORDER=1
    fi
fi



if [ -z "$DOMAIN_MESH_DEFINED" ]; then
# ================ WORKFLOW  ===================
# Model depended section - needs only modification when change to other input data set
    echo "======== dos2unix ========="
    $run dos2unix -ic $MODEL_FILEPATHNAME | xargs $run dos2unix
    STEP=.gocad_STEP_$MODEL_FILENAME
    echo "=== GocadTSurfaceReader ==="
    if [ -f "$STEP" ]; then
        echo "Skip - already computed"
    else
        echo $run GocadTSurfaceReader -i $MODEL_FILEPATHNAME -o .
        $run GocadTSurfaceReader -i $MODEL_FILEPATHNAME -o .
        touch $STEP
    fi
    echo "==== Layer Selection ======"
    if [ "$AUTO_SELECT_LAYERS" = 0 ]; then
    SLICE_TXT_FILE="${MANUAL_INPUT}/${MODEL_NAME}_Layers.txt"
    else
    SLICE_TXT_FILE="${MODEL_NAME}_Layers.txt"
    fi
    if [ -z "$AUTO_SELECT_LAYERS" ]; then
        # e.g 00_KB.vtu included, 00_KB_00_Fluss.vtu excluded, convention must be fullfilled by creator of GoCAD files
        $run find ./ -regextype posix-extended -regex '.*/[0-9]{2}_[a-zA-Z]*[^_]*.(vtu|xdmf|hdf)' | sort -n  > $SLICE_TXT_FILE
    else
        echo "Read provided manual layers file"
    fi

    if [ -n "$VOXELSIZES" ]; then
        echo "=== Voxelgrid ==="
        STEP=.voxelgrid
        if [ -f "$STEP" ]; then
            echo "Skip - already computed"
        else
            echo "=== Layers2Grid ==="
            echo $run Layers2Grid -i $SLICE_TXT_FILE -o ${MODEL_NAME}.vtu -x $VOXELSIZE_X -y $VOXELSIZE_Y -z $VOXELSIZE_Z
            $run Layers2Grid -i $SLICE_TXT_FILE -o ${MODEL_NAME}.vtu -x $VOXELSIZE_X -y $VOXELSIZE_Y -z $VOXELSIZE_Z
            echo "=== Boundary Meshes ===="
            echo $MANUAL_INPUT/3D/create_boundary_meshes.sh
            cp $MANUAL_INPUT/3D/create_boundary_meshes.sh .
            $run sh create_boundary_meshes.sh $ELEMENT_ORDER
            touch $STEP
        fi

    else
        echo "=== UnstructuredGrid ==="
        STEP=".unstructuredgrid_${MODEL_NAME}_${UNSTRUCTURED}"
        echo $STEP
        if [ -f "west.vtu" ]; then
            echo "Skip - already computed"
        else
        # ToDo calculate from BoundingBox
            cp $MANUAL_INPUT/$MODEL_NAME.gml .
			find . -type f -name "*.asc" -delete
            find . -type f -name '[[:digit:]]*.vtu' | xargs -I '{}' -t $run Mesh2Raster -i {} -o {}.asc -c $RASTER_SIZE

            LAYERS=($(find . -type f -name '*.asc' | sort))
			INTERFACECOUNT=$(ls *.asc | tail -1 | cut -c1-2)
			# refers to Layer ID in top to bottom counting
            REFINDIZES=(2 10)
			# number of additonally created layers corresponding to REFINDIZES
			REFNUM=(4 2)

            YCORNER=$(sed "4q;d" ${LAYERS[0]})
            for (( i=0 ; i<$INTERFACECOUNT; i++ )); do
                sed -i '4d' ${LAYERS[i]}
                sed -i "4i $YCORNER" ${LAYERS[i]}
            done

            for i in "${!REFINDIZES[@]}"; do
                OUTNAME=${LAYERS[REFINDIZES[i]+1]}
                OUTNAME="${OUTNAME/.vtu/.InterLayer}"
                echo $run createIntermediateRasters --file1 ${LAYERS[REFINDIZES[i]]} --file2 ${LAYERS[REFINDIZES[i]+1]} -o $OUTNAME -n ${REFNUM[i]}
                $run createIntermediateRasters --file1 ${LAYERS[REFINDIZES[i]]} --file2 ${LAYERS[REFINDIZES[i]+1]} -o $OUTNAME -n ${REFNUM[i]}
            done
			MESHLAYERCOUNT=$(find . -type f -name '*.asc' | wc -l)

            $run geometryToGmshGeo -i $MODEL_NAME.gml -o Rect.geo --mesh_density_scaling_at_points $MESH_DENSITY_SCALING
            $run gmsh Rect.geo -2 -algo meshadapt -format msh22 -o $MODEL_NAME.msh -option $DGR_ROOT/ogsworkflowhelper/.gmsh-options
            $run GMSH2OGS -i $MODEL_NAME.msh -o Raster.vtu
            find .  -type f -name '[[:digit:]]*.asc' | sort > raster_layers.txt
            $run createLayeredMeshFromRasters -i Raster.vtu -r raster_layers.txt -o ${MODEL_NAME}.vtu

			EDITCOUNT=0
			for (( j=${#REFINDIZES[@]}-1; j>=0 ; j-- )); do
				for (( k=0 ; k<REFNUM[$j]; k++ )); do
					for (( i = $INTERFACECOUNT - REFINDIZES[$j]; i <= $MESHLAYERCOUNT-$EDITCOUNT; i++ )); do
						$run editMaterialID -r -i ${MODEL_NAME}.vtu -o ${MODEL_NAME}.vtu -m $i -n $(($i-1))
					done
					EDITCOUNT=$(($EDITCOUNT + 1))
				done
			done

            echo "The element order is $ELEMENT_ORDER"
            # ================ MESH NAMES ===================
            ORIGIN=${MODEL_NAME}

            echo checkMesh
            $run checkMesh ${ORIGIN}.vtu
            echo reviseMesh
            $run reviseMesh -i ${ORIGIN}.vtu -o domain.vtu

            if [ "$ELEMENT_ORDER" = "1" ]; then
                #NodeReordering -i ${ORIGIN}_domain.vtu -o ${ORIGIN}_domain.vtu -m 3
                #ExtractBoundary -i ${ORIGIN}_domain.vtu -o ${ORIGIN}_boundary.vtu
                $run ExtractSurface -i domain.vtu -o top.vtu -x 0 -y 0 -z -1 -a 80
                $run ExtractSurface -i domain.vtu -o bottom.vtu -x 0 -y 0 -z 1 -a 80
                $run ExtractSurface -i domain.vtu -o north.vtu -x 0 -y -1 -z 0 -a 1
                $run ExtractSurface -i domain.vtu -o south.vtu -x 0 -y 1 -z 0 -a 1
                $run ExtractSurface -i domain.vtu -o east.vtu -x -1 -y 0 -z 0 -a 1
                $run ExtractSurface -i domain.vtu -o west.vtu -x 1 -y 0 -z 0 -a 1

                $run identifySubdomains -m domain.vtu -f top.vtu bottom.vtu north.vtu south.vtu east.vtu west.vtu

            elif [ "$1" = "2" ]; then
                #TODO
                echo "The element order $ELEMENT_ORDER is not available"
            else
                echo "The element order $ELEMENT_ORDER is not available"
            fi
            touch $STEP
        fi
    fi

    echo "= Generate Projectfile ="
    echo $run python3 $DGR_ROOT/ogsworkflowhelper/csv2prj.py $MODEL_NAME $DGR_ROOT/input/parameters/parameterTable_rev03.csv $PROJECT_NAME $DGR_ROOT/input/processes/3D 3 $DGR_ROOT/$RESULT_DIR/${PROJECT_NAME}.prj
    $run python3 $DGR_ROOT/ogsworkflowhelper/csv2prj.py $MODEL_NAME $DGR_ROOT/input/parameters/parameterTable_rev03.csv $PROJECT_NAME $DGR_ROOT/input/processes/3D 3 $DGR_ROOT/$RESULT_DIR/${PROJECT_NAME}.prj

else
    echo "Simulation only"

fi # DOMAIN_MESH_DEFINED

echo "===== OpenGeoSys ======"
$run ogs ${PROJECT_NAME}.prj | tee ogs.log


if [ -n "$NEW_REFERENCE_RESULT" ]; then
echo "== Create new reference results"
    mkdir $DGR_ROOT/tests/$FOLDER_NAME
    find . -name "results*.vtu" -exec cp -t  $DGR_ROOT/tests/$FOLDER_NAME {} +   
    [ $? -eq 0 ] && echo "successful" || exit 1
fi

if [ -n "$TEST_VARIABLE" ]; then
echo "== Testing =="
    find $DGR_ROOT/tests/$FOLDER_NAME -name 'results*.vtu' -type f -printf "%f\n" | xargs -I '{}' -t $run vtkdiff $DGR_ROOT/tests/$FOLDER_NAME/{} {} -a ${TEST_VARIABLE} -b ${TEST_VARIABLE} --abs 1e-3 --rel 1e-3
    [ $? -eq 0 ] && echo "successful" || exit 1
fi

# TODO no files yet
if [ -z "$PARAVIEW_STATE_FILE_DEFINED" ]; then
echo "== Paraview statefile =="
#cp $MANUAL_INPUT/ParaViewStateFile_${PROJECT_NAME}.pvsm .
fi
echo "==== Workflow done ===="
echo "Results saved to ${RESULT_DIR}"
