#!/bin/sh
# Before running this script make sure you have setup your env.sh (install directory)

short_help()
{
   # Display Help
   echo 
   echo "-----------------------------------------------------------------------------------------"
   echo
   echo "2D_workflow.sh: runs deep geological repository workflow with 2D meshes (Vertical Slice)."
   echo
   echo "Brief USAGE:"
   echo "Syntax: sh 2D_workflow.sh [b|c|d|h|i|m|p|q|s]"
   echo '  [-b "<BOUNDARIES>"]          mandatory'
   echo "  [-d <RANK> (dimensions       mandatory"
   echo '  [-i <MODEL_FILEPATHNAME>]    mandatory'
   echo "  [-m <MODEL_NAME]             mandatory"
   echo "  [-p <PROJECT_NAME>]          mandatory"
   echo '  [-c <CONTAINER_SIF_FILE>]    optional, default: defined by env.sh'
   echo "  [-o                          optional, run ogs simulation only (parameters above are omitted"
   echo "  [-q <QUAD_MESH>]             optional, default: inactive"
   echo "  [-s <SLICE_ID>]              optional, default: 1"
   echo "  [-e EXPERIMENT_NAME          optional, default: new folder name by parameter"
   echo "  [-x NEXT_STAGE               optional, e.g. testslice_init_t_3e+12.vtu"
   echo "  [-y <TOY>]                   developer only, optional: default: VerticalSlice"
   echo "  [-z <TOY_RES>]               developer only, optional: default: 50"
   echo "  [-f <TOY_FACTOR>]               developer only, optional: default: 1"
   echo "  [-l <LAYERS_FILEPATHNAME>    developer only, optional, default: automatic"
   echo "  [-n                          developer only, optional, default: no reference results"
   echo "  [-r <MODEL_REV>              developer only, optional, default: latest"
   echo "  [-t <TEST_VARIABLE>          developer only, optional, default: not-active"
   echo 
   echo 
   echo "sh 2D_workflow.sh -h for further help"
   echo
   echo "-----------------------------------------------------------------------------------------"

}

long_help()
{
    short_help
   echo
   echo
   echo "<BOUNDARIES>"
   echo "    double quoted array of 4 entries delimited by single space"
   echo "    x_min y_min x_max y_max"
   echo "    e.g. -b \"9000 9200 18000 2000\" "
   echo 
   echo "<CONTAINER_SIF_FILE>"
   echo "    absolute path to docker container file with 'sif' file ending"
   echo "    e.g. -c ~/dgr/container/images/ogs-dgr_latest.sif"
   echo
   echo "<MODEL_NAME>"
   echo "    Ton_Nord"
   echo "    e.g. -p Ton_Nord"
   echo
   echo "<MODEL_FILENAME>"
   echo "    full path name to GoCadTSurface file that represents 2D layers in 3D structure "
   echo "    e.g. -i Basisflaechen_Ton_Nord.ts"
   echo "    OR"
   echo "    leave empty - Then it is expected that all meshes required by the project file"
   echo "    are in same folder as the project file"
   echo
   echo "<PROJECT_NAME>"
   echo "    simH_1 or simH_2a or simH_2b or simHM_1"
   echo "    A file with name equal to PROJECT_NAME and .prj extension in subfolder input"
   echo "    e.g. -m simH_1"
   echo
   echo "<SLICE_ID>"
   echo "    any natural number"
   echo "    e.g. -s 1"
   echo
   echo "<QUAD_MESH>"
   echo "    0 or 1"
   echo "    e.g. -q 1"
   echo
   echo "<RANK>"
   echo "    2 or 3"
   echo "   2D or 3D"
}

# ========== DEFAULTS FROM GLOBAL CONFIG =======
OGS_DGR_VERSION=v0.3
OGS_CONTAINER_CHECKSUM=72f241d614afc456a6f237c4b997a160935d0d4fc601524988aebe7c2131f6cc

SCRIPT=`realpath $0`
SCRIPTPATH=`dirname $SCRIPT`
DGR_ROOT=`realpath $SCRIPTPATH/../../`
ENV_SCRIPT=$DGR_ROOT/install/env.sh
CUSTOM_ENV_SCRIPT=$DGR_ROOT/install/custom_env.sh
if [ -f "$CUSTOM_ENV_SCRIPT" ]; then
    echo "== Custom Env script used"
    . $DGR_ROOT/install/custom_env.sh
elif [ -f "$ENV_SCRIPT" ]; then
    echo "== Stand env script used"
    . $DGR_ROOT/install/env.sh
else
    echo "Error: envh.sh not found. Expected it here: $ENV_SCRIPT."
    exit 3
fi


# ================ LOCAL CONFIG DEFAULTS ===================
# give a name to gocad input files the locate in $DGR_ROOT/input
MODEL_NAME=
MODEL_FILENAME=
SLICE_ID=
# give a name to a ogs project file that locates in manual input folder
 
PROJECT_NAME=
QUADMESH=
OGS_SIMULATION_ONLY=
DOMAIN_MESH_DEFINED=
BOUNDARIES_DEFINED=1
PROJECT_FILE_DEFINED=

# Default take latest revision of models
MODEL_REV=
TOY_RES=50
TOY_FACTOR=1

# ================ COMMAND LINE OPTIONS ===================
# may overwrite global or local config variables
while getopts ":p::m::i::r::s::b::q::c::e::d::t::x::y::z::f:nhlo" opt; do
    case $opt in
        h)  
            long_help
            exit
        ;;
        p)
            PROJECT_NAME=${OPTARG}
        ;;
        m)
            MODEL_NAME=${OPTARG}
        ;;
        i)
            MODEL_FILENAME=${OPTARG}
        ;;
        r)
            MODEL_REV=${OPTARG}
        ;;
        s)
            SLICE_ID=${OPTARG}
        ;;
        b)
            BOUNDARIES=${OPTARG}
        	SLICE_X_START=`echo $BOUNDARIES | cut --delimiter " " --fields 1`
			SLICE_Y_START=`echo $BOUNDARIES | cut --delimiter " " --fields 2`
			SLICE_X_END=`echo $BOUNDARIES | cut --delimiter " " --fields 3`
			SLICE_Y_END=`echo $BOUNDARIES | cut --delimiter " " --fields 4`
        ;;

        o)
            OGS_SIMULATION_ONLY=1
        ;;
        q)  
            QUADMESH=${OPTARG}
        ;;
        l)
            # Per default all layers found will be considered in vertical slice
            # It is possible to provide custom layer.txt with only a subset of layers
            AUTO_SELECT_LAYERS=0
        ;;
        c)
            USE_CONTAINER=2
            OGS_CONTAINER=${OPTARG}
        ;;
        e)
            EXPERIMENT=${OPTARG}
        ;;
        d)
            RANK=${OPTARG}
        ;;
        t)
            TEST_VARIABLE=${OPTARG}
        ;;
        x) 
            NEXT_STAGE=${OPTARG}
            echo $NEXT_STAGE
        ;;
        y)
            TOY=${OPTARG}
        ;;
        z)
            TOY_RES=${OPTARG}
        ;;
        f)
            TOY_FACTOR=${OPTARG}
        ;;
        n)
            NEW_REFERENCE_RESULT=1
        ;;
        \?)
            echo "Invalid option: -$OPTARG" >&2
            short_help
            exit 1
        ;;
        :)
            echo "Option -$OPTARG requires an argument." >&2
            short_help
            exit 1
        ;;
    esac
done

#echo "Project name is: $OPTARG" >&2
#echo "Model name is: $OPTARG" >&2
#echo "Model filename is: $OPTARG" >&2
#echo "Slice ID is: $OPTARG" >&2
#echo "Vertical slice: x start: $SLICE_X_START y start $SLICE_Y_START x end $SLICE_X_END y end $SLICE_Y_END"
#echo "Using Quadmesh: $OPTARG" >&2
#echo "Using specified Container: $OPTARG" >&2

if [ -z "$PROJECT_NAME" ]; then
    echo "error: Process name NOT given, use -p option exit."
    short_help
    exit 1;
fi

if [ -z "$OGS_SIMULATION_ONLY" ]; then
    if [ -z "$MODEL_FILENAME" ]; then
        echo "error: Model filename NOT given, use -i option."
        short_help
        exit 1;
    fi

    if [ -z "$MODEL_NAME" ]; then
        echo "error: Model name NOT given, use -m option exit."
        short_help
        exit 1;
    fi

    if [ -z "$RANK" ]; then
        echo "error: Rank (2D or 3D) NOT given, use -d option exit."
        short_help
        exit 1;
    fi


    if [ -z "$SLICE_ID" ]; then
        echo "warning: Slice ID NOT given, we use 1 as default."
        SLICE_ID=1
        exit 1
    fi

    if [ -z "$BOUNDARIES" ]; then
        echo "error: Boundaries NOT given, use -b option."
        echo 'xmin, ymin, xmax, ymax must be in double quotes delimited by single space'
        echo 'e.g. -b "9200 9000 18000 20000"'
        short_help
        echo "exit. "
        exit 1
    fi
else
    DOMAIN_MESH_DEFINED=1
    BOUNDARIES_DEFINED=1
    PROJECT_FILE_DEFINED=1
    echo "warning: Custom OGS Project file is provided - all other options will be ignored"
fi # OGS_SIMULATION_ONLY

# ================ DERIVED VARIABLES ===================

MANUAL_INPUT=$DGR_ROOT/input/models/${MODEL_NAME}

if [ "$AUTO_SELECT_LAYERS" = 0 ]; then
   SLICE_TXT_FILE="${MANUAL_INPUT}/${MODEL_NAME}_Layers.txt"
else
   SLICE_TXT_FILE="${MODEL_NAME}_Layers.txt"
fi
#SLICE_NAME="${MODEL_NAME}_Slice_${SLICE_ID}"
SLICE_NAME=testslice
if [ -n "$TOY" ]; then
    FOLDER_NAME="${MODEL_NAME}_${SLICE_ID}_${PROJECT_NAME}_${TOY}_${TOY_RES}_${TOY_FACTOR}"
else
    FOLDER_NAME="${MODEL_NAME}_${SLICE_ID}_${PROJECT_NAME}"
fi

if [ -n "$EXPERIMENT" ]; then
    FOLDER_NAME=$EXPERIMENT
fi
MODELS_DIR=$DGR_ROOT/input/models/${MODEL_NAME}/${MODEL_REV}/
MODEL_FILEPATHNAME=$MODELS_DIR/$MODEL_FILENAME

# =================== HELPER ==============================

python_module()
{
    local module="$1"
    local python_file="$2"
    echo "-m $module.$python_file"
}

python_script()
{
    local module="$1"
    local python_file="$2"
    echo "$MSH2VTU/$module/$python_file.py"
}
# Using local software
if [ "$USE_CONTAINER" = "0" ]; then
    echo "Using local software"
    export run=" "
    # we assume that all other bin are already on path
    # that is why only OGSBIN is added to path
    export PATH="$PATH:$OGSBIN"
    
    if [ "$USE_MSH2VTU_DIR" = "1" ]; then
        echo "USE LOCAL MSH2VTU SOURCES"
        msh2vtu=$(python_script msh2vtu msh2vtu)
    else
        echo "USE LOCAL MSH2VTU PIP INSTALLATION"
        msh2vtu=$(python_module msh2vtu msh2vtu)
    fi
    # Using development container
elif [ "$USE_CONTAINER" = "2" ]; then
    export PATH="$PATH:$SINGULARITYBIN"
    export run="singularity -s exec $OGS_CONTAINER"
    msh2vtu=$(python_module msh2vtu msh2vtu)
    # Using official container
else
    OGS_OFFICIAL_CONTAINER=$DGR_ROOT/container/images/ogs-dgr_$OGS_DGR_VERSION.sif
    export PATH="$PATH:$SINGULARITYBIN"
    export run="singularity -s exec $OGS_OFFICIAL_CONTAINER"
    msh2vtu=$(python_module msh2vtu msh2vtu)
    
    if [ -f "$OGS_OFFICIAL_CONTAINER" ]; then
        echo "Container found on $OGS_OFFICIAL_CONTAINER."
    else
        echo "Container $OGS_OFFICIAL_CONTAINER not found."
        exit 3
    fi
    
    # Disable check temporarly - checksum
    #echo "${OGS_CONTAINER_CHECKSUM} ${OGS_OFFICIAL_CONTAINER}" | sha256sum -c
    #if [ $? != 0 ]; then
    #    echo "Checksum of provided container (${OGS_OFFICIAL_CONTAINER}) does not match. Your env.sh must point to a container with version v0.3 (sif-file) with checksum ${OGS_CONTAINER_CHECKSUM}."
    #    exit 1
    #fi
fi
echo $run

RESULT_DIR=.
# ================ WORKING DIRECTORY  ===================
if [ "$DGR_WORK" = "" ]; then
    RESULT_DIR=$FOLDER_NAME/
else
    RESULT_DIR=$DGR_WORK/$FOLDER_NAME/
fi
mkdir -p $RESULT_DIR
cd $RESULT_DIR
echo "Results will be saved to ${RESULT_DIR}"




if [ -z "$DOMAIN_MESH_DEFINED" ]; then
# ================ WORKFLOW  ===================
# Model depended section - needs only modification when change to other input data set
echo "======== dos2unix ========="
$run dos2unix -ic $MODEL_FILEPATHNAME | xargs $run dos2unix
echo "=== GocadTSurfaceReader ==="
$run GocadTSurfaceReader -i $MODEL_FILEPATHNAME -o .
[ $? -eq 0 ] && echo "successful" || exit 1
echo "==== Layer Selection ======"

if [ -z "$AUTO_SELECT_LAYERS" ]; then
    # e.g 00_KB.vtu included, 00_KB_00_Fluss.vtu excluded, convention must be fullfilled by creator of GoCAD files
    $run find ./ -regextype posix-extended -regex '.*/[0-9]{2}_[a-zA-Z]*[^_]*.(vtu|xdmf|hdf)' | sort -n  > $SLICE_TXT_FILE
else
    echo "Read provided manual layers file"
fi
echo "= VerticalSliceFromLayers ="
echo $SLICE_TXT_FILE


echo "TOY" 
echo $TOY
echo $NEXT_STAGE
echo $FOLDER_NAME
if [ -n "$TOY" ]; then
    echo "= Toy = xy"
    echo $TOY_FACTOR
    echo pwd
    echo $run python3 $DGR_ROOT/ogsworkflowhelper/2D_toy/toy.py $TOY $TOY_RES $DGR_ROOT $TOY_FACTOR . ./domain.vtu 2
    $run python3 $DGR_ROOT/ogsworkflowhelper/2D_toy/toy.py $TOY $TOY_RES $DGR_ROOT $TOY_FACTOR . ./domain.vtu 2
    echo "= Toy done = "

elif [ -n "$NEXT_STAGE" ]; then
    echo $run python  $DGR_ROOT/ogsworkflowhelper/post_processing/result_as_init.py $NEXT_STAGE testslice_result.vtu -point_data epsilon -field_data epsilon_ip
    $run python $DGR_ROOT/ogsworkflowhelper/post_processing/result_as_init.py $NEXT_STAGE testslice_result.vtu -point_data epsilon -field_data epsilon_ip
else
    $run VerticalSliceFromLayers -i $SLICE_TXT_FILE -o $SLICE_NAME --start-x $SLICE_X_START --start-y $SLICE_Y_START --end-x $SLICE_X_END --end-y $SLICE_Y_END -r 200 -t -b
    [ $? -eq 0 ] && echo "successful" || exit 1
    if [ -n "$QUADMESH_OLD" ]; then
        echo "========== gmsh =========="
        $run gmsh -2 $SLICE_NAME.geo
        [ $? -eq 0 ] && echo "successful" || exit 1
        echo "======== msh2vtu ========="
        $run python3 $msh2vtu -g -s  $SLICE_NAME.msh
        [ $? -eq 0 ] && echo "successful" || exit 1
        echo "====== reviseMesh ========"
        $run reviseMesh -i ${SLICE_NAME}_domain.vtu -o ${SLICE_NAME}_domain_rev.vtu -e 0.001
        [ $? -eq 0 ] && echo "successful" || exit 1
        echo "=== createQuadraticMesh =="
        $run createQuadraticMesh -i ${SLICE_NAME}_domain_rev.vtu -o ${SLICE_NAME}.vtu
        [ $? -eq 0 ] && echo "successful" || exit 1
        $run ExtractBoundary -i ${SLICE_NAME}.vtu -o ${SLICE_NAME}_boundary.vtu
        [ $? -eq 0 ] && echo "successful" || exit 1
        # overwrite existing linear boundaries from 'VerticalSliceFromLayers -b'
        $run removeMeshElements -i ${SLICE_NAME}_boundary.vtu -o ${SLICE_NAME}_left.vtu  --x-min 12776  --x-max 27000
        $run removeMeshElements -i ${SLICE_NAME}_boundary.vtu -o ${SLICE_NAME}_right.vtu --x-min 12700 --x-max 26861
        $run removeMeshElements -i ${SLICE_NAME}_boundary.vtu -o ${SLICE_NAME}_top.vtu --y-min -5000 --y-max 51
        $run removeMeshElements -i ${SLICE_NAME}_boundary.vtu -o ${SLICE_NAME}_bottom_inter.vtu  --y-min -3636 --y-max 1000
        $run removeMeshElements -i ${SLICE_NAME}_bottom_inter.vtu -o ${SLICE_NAME}_bottom.vtu --y-min -4150 --y-max -3635 --x-min 26860 --x-max 26863
        [ $? -eq 0 ] && echo "successful" || exit 1
    fi
    if [ -n "$QUADMESH" ]; then # overwriting of existing data should be avoided in future!
        echo "=== createQuadraticMesh =="
        $run createQuadraticMesh -i ${SLICE_NAME}.vtu -o ${SLICE_NAME}.vtu
        $run createQuadraticMesh -i ${SLICE_NAME}_left.vtu -o ${SLICE_NAME}_left.vtu
        $run createQuadraticMesh -i ${SLICE_NAME}_right.vtu -o ${SLICE_NAME}_right.vtu
        $run createQuadraticMesh -i ${SLICE_NAME}_top.vtu -o ${SLICE_NAME}_top.vtu
        $run createQuadraticMesh -i ${SLICE_NAME}_bottom.vtu -o ${SLICE_NAME}_bottom.vtu
        [ $? -eq 0 ] && echo "successful" || exit 1
        echo "=== identifySubdomains ==="
        $run identifySubdomains -m ${SLICE_NAME}.vtu ${SLICE_NAME}_left.vtu ${SLICE_NAME}_right.vtu ${SLICE_NAME}_top.vtu ${SLICE_NAME}_bottom.vtu
        [ $? -eq 0 ] && echo "successful" || exit 1
    fi
    if [ "$RANK" = "2" ]; then
    echo "=== Spatial transformation"
    # If we would not copy these files will get override in subsequent steps
    #cp ${SLICE_NAME}.vtu ${SLICE_NAME}_3D.vtu   
    #cp ${SLICE_NAME}_left.vtu ${SLICE_NAME}_left_3D.vtu   
    #cp ${SLICE_NAME}_right.vtu  ${SLICE_NAME}_right_3D.vtu    
    #cp ${SLICE_NAME}_top.vtu   ${SLICE_NAME}_top_3D.vtu   
    #cp ${SLICE_NAME}_bottom.vtu  ${SLICE_NAME}_bottom_3D.vtu   
      
    export BOUNDARIES="-x $SLICE_X_START -X $SLICE_X_END -y $SLICE_Y_START -Y $SLICE_Y_END" 
    export TRAFO_PARAMETERS=" -z 0.0 -s"
    export VTU_TRAFO="$run python3 $DGR_ROOT/ogsworkflowhelper/vtu_trafo.py"
    ${VTU_TRAFO} -i ${SLICE_NAME}.vtu -o ${SLICE_NAME}.vtu ${BOUNDARIES} ${TRAFO_PARAMETERS}
    ${VTU_TRAFO} -i ${SLICE_NAME}_left.vtu -o ${SLICE_NAME}_left.vtu  ${BOUNDARIES} ${TRAFO_PARAMETERS}
    ${VTU_TRAFO} -i ${SLICE_NAME}_right.vtu -o ${SLICE_NAME}_right.vtu  ${BOUNDARIES} ${TRAFO_PARAMETERS}
    ${VTU_TRAFO} -i ${SLICE_NAME}_top.vtu -o ${SLICE_NAME}_top.vtu  ${BOUNDARIES} ${TRAFO_PARAMETERS}
    ${VTU_TRAFO} -i ${SLICE_NAME}_bottom.vtu -o ${SLICE_NAME}_bottom.vtu  ${BOUNDARIES} ${TRAFO_PARAMETERS}
    [ $? -eq 0 ] && echo "successful" || exit 1

    $run NodeReordering -m 1 -i testslice.vtu -o testslice.vtu
    [ $? -eq 0 ] && echo "successful" || exit 1
    $run identifySubdomains -m ${SLICE_NAME}.vtu ${SLICE_NAME}_left.vtu ${SLICE_NAME}_right.vtu ${SLICE_NAME}_top.vtu ${SLICE_NAME}_bottom.vtu
    [ $? -eq 0 ] && echo "successful" || exit 1



    else
        echo "=== Spatial transformation skipped"
    fi

fi

fi # DOMAIN_MESH_DEFINED

echo "= Generate Projectfile ="
if [ -n "$PROJECT_FILE_DEFINED" ]; then
    cp $MANUAL_INPUT/* .
else 
    echo $run python3 $DGR_ROOT/ogsworkflowhelper/csv2prj.py $MODEL_NAME $DGR_ROOT/input/parameters/parameterTable_rev03.csv $PROJECT_NAME $DGR_ROOT/input/processes/2D $RANK .
    $run python3 $DGR_ROOT/ogsworkflowhelper/csv2prj.py $MODEL_NAME $DGR_ROOT/input/parameters/parameterTable_rev03.csv $PROJECT_NAME $DGR_ROOT/input/processes/2D $RANK $PROJECT_NAME.prj
    [ $? -eq 0 ] && echo "successful" || exit 1
fi

echo "===== OpenGeoSys ======"
echo $run ogs ${PROJECT_NAME}.prj | tee ogs.log
$run ogs ${PROJECT_NAME}.prj | tee ogs.log
[ $? -eq 0 ] && echo "successful" || exit 1


if [ "$RANK"="2" ] && [ -z "$TOY" ]; then
    echo "=== Spatial back transformation"
    
    export VTU_TRAFO="$run python3 $DGR_ROOT/ogsworkflowhelper/vtu_trafo.py"
    echo find . -name '*_sim_*vtu' -exec ${VTU_TRAFO} -i {} -o {} -r \;
    find . -name '*_sim_*vtu' -exec ${VTU_TRAFO} -i {} -o {}  -r \;
    [ $? -eq 0 ] && echo "successful" || exit 1
fi


if [ -n "$NEW_REFERENCE_RESULT" ]; then
echo "== Create new reference results"
    mkdir $DGR_ROOT/tests/$FOLDER_NAME
    find . -name "OGS*.vtu" -exec cp -t  $DGR_ROOT/tests/$FOLDER_NAME {} +   
    [ $? -eq 0 ] && echo "successful" || exit 1

fi

if [ -n "$TEST_VARIABLE" ]; then
echo "== Testing =="
    find $DGR_ROOT/tests/$FOLDER_NAME -name '*_sim_*.vtu' -type f -printf "%f\n" | xargs -I '{}' -t $run vtkdiff $DGR_ROOT/tests/$FOLDER_NAME/{} {} -a ${TEST_VARIABLE} -b ${TEST_VARIABLE} --abs 1e-3 --rel 1e-3
    [ $? -eq 0 ] && echo "successful" || exit 1
fi


if [ -z "$PARAVIEW_STATE_FILE_DEFINED" ]; then
echo "== Paraview statefile =="
cp $MANUAL_INPUT/ParaViewStateFile_${PROJECT_NAME}.pvsm .
fi
echo "==== Workflow done ===="
echo "Results saved to ${RESULT_DIR}"
