"""General plot tools.

Used by dipole_hexagon, dipole_tubes, dipole_clusters.
last change May 1, 2024, tested with Python 3.11.7, rated 10.00/10
Ingo Rehberg, University of Bayreuth
"""
from datetime import datetime
import numpy as np
import matplotlib as mpl
import matplotlib.pyplot as plt
from matplotlib.colors import ListedColormap


class Ccc:
    """Calcuate cluster characteristics."""

    @staticmethod
    def circularity(clu):
        """Get the mean circularity squared of cp.n_s rings."""
        circ_i = np.zeros(clu.n_s)  # a circularity for each of the n_s rings
        for i in range(clu.n_s):
            beg = i*clu.l_r
            end = beg + clu.l_r
            circ_i[i] = np.mean((clu.r_vec[beg:end, 0]*clu.p_vec[beg:end, 1] -
                                clu.r_vec[beg:end, 1]*clu.p_vec[beg:end, 0])
                                / clu.lengths[beg:end]
                                )/(0.5/np.sin(np.pi/clu.l_r))
        return np.mean(circ_i*circ_i)

    @staticmethod
    def m_t(clu):
        """Get the norm of the toroidal moment."""
        moment = clu.m_tor()
        return np.sqrt(np.dot(moment, moment))

    @staticmethod
    def axial_order_parameter(clu):
        """Get the sum of the squared z-components."""
        return np.mean(clu.p_vec[:, 2]*clu.p_vec[:, 2]
                       / clu.lengths[:]/clu.lengths[:])

    @staticmethod
    def m_t_mean_abs(clu):
        """Get the mean toroidal moment squared of cp.n_s rings."""
        circ_i = np.zeros(clu.n_s)  # a circularity for each of the n_s rings
        for i in range(clu.n_s):
            beg = i*clu.l_r
            end = beg + clu.l_r
            circ_i[i] = np.sum(np.cross(clu.r_vec[beg:end, :],
                                        clu.p_vec[beg:end, :]), axis=0)[2]
        return np.mean(np.abs(circ_i))


class Tl:
    """Tools for plotting."""

    old = -1954

    @staticmethod
    def connect_conditionally(axe, x_a, y_a, fak, col):
        """Indicate jumps by arrows if the change of the slope is > fak."""
        for i in range(4, len(y_a)):
            if np.abs(y_a[i]-y_a[i-1]) > max([np.abs(y_a[i-1]-y_a[i-2]),
                                              np.abs(y_a[i-2]-y_a[i-3]),
                                              np.abs(y_a[i-3]-y_a[i-4]),
                                              1.e-3])*fak:
                del_x = x_a[i]-x_a[i-1]
                del_y = y_a[i]-y_a[i-1]
                axe.annotate("", xy=(x_a[i], y_a[i]),
                             xytext=(x_a[i-1]+del_x*0.05, y_a[i-1]+del_y*0.05),
                             arrowprops={"arrowstyle": '->', "color": col})

    @staticmethod
    def gmk(i, x_a):
        """Get marker from history."""
        marker = '.'
        if x_a[i] > Tl.old:
            marker = '->'
        elif x_a[i] < Tl.old:
            marker = '<-'
        elif x_a[i] == Tl.old:
            marker = '.'
        Tl.old = x_a[i]
        return marker

    @staticmethod
    def indicate_direction(axe, x_a, y_a, col):
        """Plot x,y-data with indicating the direction."""
        if x_a is None or len(x_a) < 1:
            Gt.msg('no data yet')
            return
        for i, xpl in enumerate(x_a):
            axe.plot(xpl, y_a[i], Tl.gmk(i, x_a), c=col, fillstyle='none')
        axe.plot(x_a[-1], y_a[-1], 's', c='r', fillstyle='none')

    @staticmethod
    def create_surface(length, width, head, headwidth):
        """Use internally by arrow3d."""
        a_h = np.array([[0, 0], [width, 0], [width, (1-head)*length],
                        [headwidth*width, (1-head)*length], [0, length]])
        radius, theta = np.meshgrid(a_h[:, 0], np.linspace(0, 2*np.pi, 30))
        z_c = np.tile(a_h[:, 1], radius.shape[0]).reshape(radius.shape)
        x_c = radius*np.sin(theta)
        y_c = radius*np.cos(theta)
        return radius, x_c, y_c, z_c

    @staticmethod
    def arrow3d(axe, **kw):
        """Plot a tube arrow."""
        theta_x = np.deg2rad(kw['theta_x'])
        theta_z = np.deg2rad(kw['theta_z'])

        radius, x_c, y_c, z_c = Tl.create_surface(kw['length'], kw['width'],
                                                  kw['head'], kw['headwidth'])

        rot_x = np.array([[1, 0, 0], [0, np.cos(theta_x), -np.sin(theta_x)],
                          [0, np.sin(theta_x), np.cos(theta_x)]])
        rot_z = np.array([[np.cos(theta_z), -np.sin(theta_z), 0],
                          [np.sin(theta_z),  np.cos(theta_z), 0], [0, 0, 1]])

        b_1 = np.dot(rot_x, np.c_[x_c.flatten(),
                                  y_c.flatten(), z_c.flatten()].T)
        b_2 = np.dot(rot_z, b_1)
        b_2 = b_2.T + kw['offset']
        axe.plot_surface(b_2[:, 0].reshape(radius.shape),
                         b_2[:, 1].reshape(radius.shape),
                         b_2[:, 2].reshape(radius.shape), color=kw['color'])

    @staticmethod
    def quiver_3d(axe, clu):
        """Plot tube arrows."""
        r_x, r_y, r_z = clu.r_vec[:, 0], clu.r_vec[:, 1], clu.r_vec[:, 2]
        leng = np.power(clu.lengths, 1/3)  # 3rd root of dipole moment
        p_u = clu.p_vec[:, 0]/clu.lengths
        p_v = clu.p_vec[:, 1]/clu.lengths
        p_w = clu.p_vec[:, 2]/clu.lengths

        t_x = np.copy(clu.p_vec[:, 2])/clu.lengths
        col = Gt.col_arr(np.clip(t_x, -1, 1))
        t_x = np.arccos(np.clip(t_x, -1, 1))  # take care of versions here...
        for i in range(clu.n_dip()):
            Tl.arrow3d(axe, offset=np.array([r_x[i]-p_u[i]*leng[i]/2,
                                             r_y[i]-p_v[i]*leng[i]/2,
                                             r_z[i]-p_w[i]*leng[i]/2]),
                       length=leng[i],
                       width=0.05*leng[i],
                       head=4*0.05*leng[i],
                       headwidth=2*leng[i],
                       theta_z=180./np.pi*np.arctan2(p_v[i], p_u[i])+90,
                       theta_x=180./np.pi*t_x[i],
                       color=col[i])

    @staticmethod
    def plot_contact(axi, c_n):
        """Plot contact lines by plotting all distances with length < 1.01."""
        prt = np.zeros((2, 3))
        for i, prt[0] in enumerate(c_n.r_vec):
            for prt[1] in c_n.r_vec[i+1:]:
                if np.sum((prt[0]-prt[1])**2) < 1.02:
                    axi.plot(prt[:, 0], prt[:, 1], prt[:, 2], '-', c='grey')

    @staticmethod
    def plot_diam(axi, c_n):
        """Plot diameter lines by plotting distances with length = diam."""
        prt = np.zeros((2, 3))
        d_max = 0
        for i, prt[0] in enumerate(c_n.r_vec):
            for prt[1] in c_n.r_vec[i+1:]:
                d_max = max(np.sum((prt[0]-prt[1])**2), d_max)
        print('d_max:', d_max)
        for i, prt[0] in enumerate(c_n.r_vec):
            for prt[1] in c_n.r_vec[i+1:]:
                if np.sum((prt[0]-prt[1])**2) > d_max*0.9999:
                    axi.plot(prt[:, 0], prt[:, 1], prt[:, 2], '--', c='b')

    @staticmethod
    def plot_sphere(axi, clu, cen, rcount, trp):
        """Plot a sphere with radius around the center cen."""
        radius = 0.5  # radius of the sphere
        u_s = np.linspace(0, 2 * np.pi, rcount)
        v_s = np.linspace(0, np.pi, rcount//2)
        x_s = radius * np.outer(np.cos(u_s), np.sin(v_s)) + cen[0]
        y_s = radius * np.outer(np.sin(u_s), np.sin(v_s)) + cen[1]
        z_s = radius * np.outer(np.ones(np.size(u_s)), np.cos(v_s)) + cen[2]
        ens = clu.b_flux_2(x_s, y_s, z_s)  # Energy density
        ens -= ens.min()
        ens /= ens.max()
        mycolors = np.empty(x_s.shape)  # the size of the color map
        mycolors = bgrs(ens)  # the type of color coding
        mycolors[:, :, 3] = trp
        axi.plot_surface(x_s, y_s, z_s, facecolors=mycolors,
                         rcount=rcount, ccount=rcount/2)

    @staticmethod
    def plot_direction(axe, geo):
        """Indicate the direction r for the lines and planes."""
        axe.quiver(geo.origin()[0], geo.origin()[1], geo.origin()[2],
                   geo.direction[0], geo.direction[1], geo.direction[2],
                   length=1.8, arrow_length_ratio=0.2, normalize=False,
                   color='c')
        t_o = geo.origin() + 0.9*geo.direction
        t_d = geo.direction
        axe.text(t_o[0], t_o[1], t_o[2], r'$\mathbf{r}$',
                 (t_d[0], t_d[1], t_d[2]),
                 verticalalignment='center', color='cyan')

    @staticmethod
    def plot_plane(axi, geo, ens, trp):
        """Plot plane, as defined by the instance geo of the class GeoPlane."""
        g_x = geo.surface[:, :, 0]
        g_y = geo.surface[:, :, 1]
        g_z = geo.surface[:, :, 2]
        ens -= ens.min()
        ens /= ens.max()
        plane_colors = np.empty(g_x.shape)  # the size of the color map
        plane_colors = bgrs(ens)  # the type of color coding
        plane_colors[:, :, 3] = trp  # transparency
        axi.plot_surface(g_x, g_y, g_z, facecolors=plane_colors,
                         lw=0, shade=False)
        axi.plot(g_x[0, :], g_y[0, :], g_z[0, :], color='gray', lw=8)
        axi.plot(g_x[geo.iii()-1, :], g_y[geo.iii()-1, :],
                 g_z[geo.iii()-1, :], color='gray', lw=8)
        axi.plot(g_x[:, 0], g_y[:, 0], g_z[:, 0], color='gray', lw=8)
        axi.plot(g_x[:, geo.iii()-1], g_y[:, geo.iii()-1],
                 g_z[:, geo.iii()-1], color='gray', lw=8)

    @staticmethod
    def plot_sur_sphere(axi, clu, rcount, trp):
        """Plot the surrounding sphere."""
        radius = clu.radius()*1.1
        u_s = np.linspace(0, 2 * np.pi, rcount)
        v_s = np.linspace(0, np.pi, rcount//2)
        x_s = radius * np.outer(np.cos(u_s), np.sin(v_s))
        y_s = radius * np.outer(np.sin(u_s), np.sin(v_s))
        z_s = radius * np.outer(np.ones(np.size(u_s)), np.cos(v_s))
        ens = clu.radial_flux(x_s, y_s, z_s)  # flux comp. radial dir.
        ens -= ens.min()
        ens /= ens.max()
        mycolors = np.empty(x_s.shape)  # the size of the color map
        mycolors = bgrs(ens)  # the type of color coding
        mycolors[:, :, 3] = trp
        axi.plot_surface(x_s, y_s, z_s, facecolors=mycolors,
                         rcount=rcount, ccount=rcount/2)

    @staticmethod
    def plot_dipole_moment(axi, clu):
        """Plot the dipole moment into axi."""
        moment = np.sum(clu.p_vec, axis=0)
        amount = np.sqrt(np.dot(moment, moment))
        if amount > 0.001:
            phi = np.arctan2(moment[1], moment[0])
            phi += 2*np.pi
            phi %= 2*np.pi
            theta = np.arcsin(moment[2]/amount)
            if abs(theta) > np.pi/2*0.99:
                phi = np.pi
            fak = 180/np.pi
            axi.plot(phi*fak, theta*fak, 'x', color=Gt.col_arr(np.sin(theta)),
                     alpha=1,
                     label=f'$m:${amount:2.1f}')
        else:
            axi.plot(370, 370, 'o', c='white', label=f'$m:${amount:2.1f}')

    @staticmethod
    def plot_toroidal_moment(axi, clu):
        """Plot the dipole moment into axi."""
        moment = clu.m_tor()
        amount = np.sqrt(np.dot(moment, moment))
        if amount > 0.001:
            phi = np.arctan2(moment[1], moment[0])
            phi += 2*np.pi
            phi %= 2*np.pi
            theta = np.arcsin(moment[2]/amount)
            if abs(theta) > np.pi/2*0.99:
                phi = np.pi
            fak = 180/np.pi
            axi.plot(phi*fak, theta*fak, '+', color=Gt.col_arr(np.sin(theta)),
                     alpha=1,
                     label=f'$m_t:${amount:2.1f}')
        else:
            axi.plot(370, 370, 'o', c='white', label=f'$m_t:${amount:2.1f}')

    @staticmethod
    def plot_theta_phi(axi, clu):
        """Plot the orientations of the clu-dipoles in 2d, also used by p2."""
        axi.cla()
        axi.grid(visible=True, which='major', axis='both')
        axi.set_ylabel(r'$\Theta _\mathrm{elev}$', labelpad=-12)
        axi.set_xlabel(r'$\varphi _\mathrm{azim}$', labelpad=-10)
        axi.set_xlim(-2, 362)
        axi.set_ylim(-99, 99)
        axi.set_xticks([0, 90, 180, 270, 360],
                       labels=[' ', '90°', ' ', '270°', ' '])
        axi.set_yticks([-90, -45, 0, 45, 90],
                       labels=['-90°', ' ', ' ', ' ', '90°'])
        phi = np.arctan2(clu.p_vec[:, 1], clu.p_vec[:, 0])
        phi += 2*np.pi
        phi %= 2*np.pi
        theta = np.arcsin(np.clip(clu.p_vec[:, 2]/clu.lengths, -1, 1))
        for i, thet in enumerate(theta):
            if abs(thet) > np.pi/2*0.99:
                phi[i] = np.pi
        col = Gt.col_arr(np.sin(theta))  # broader range of red and blue
        fak = 180/np.pi
        axi.scatter(phi*fak, theta*fak, s=clu.lengths*20, c=col, alpha=0.9)
        Tl.plot_dipole_moment(axi, clu)
        Tl.plot_toroidal_moment(axi, clu)
        axi.legend(handletextpad=-0.5, draggable=True)

    @staticmethod
    def plot_u_steps(axi, num, pot):
        """Plot the magnetic potential energy U(relaxation steps)."""
        axi.cla()
        # axi.xaxis.tick_top()  # seems too unusual here
        axi.xaxis.label.set_size(10)
        axi.set_xlabel(r'step number', labelpad=-13)
        axi.set_xticks([-100, -75, -50,  -25, 0],
                       labels=['-100', ' ', ' ', ' ', '0'])
        axi.yaxis.label.set_size(10)
        axi.set_ylabel(r'$\mathrm{log}(\frac{u}{u_{min}})$', labelpad=-8)
        axi.grid(visible=True, which='major', axis='both')
        ypl = np.log(pot/pot.min())
        axi.plot(num, ypl, 'b.')
        pos = np.argmax(num)  # the latest one
        axi.plot(num[pos], ypl[pos], 'rx', label=f'{pot[pos]:.5f}')
        axi.legend(handletextpad=-0.5)

    @staticmethod
    def pot_plt(axi, clu, geo, subtr=False):
        """Plot the potential into pic1 in linear scale."""
        psi = clu.potential(geo.p_x, geo.p_y, geo.p_z)
        if subtr:
            psi = psi - clu.potential(0, 0, 0)  # mind broadcasting here
        axi.set_ylabel(r"Scalar potential $\Phi$")
        axi.plot(geo.p_s, psi, color="red", label=r'$\Phi$')
        axi.legend(loc='best', shadow=False, fontsize='small')

    @staticmethod
    def pot_plt_log(axi, clu, geo, subtr=False):
        """Plot the potential into pic1 as loglog."""
        psi = clu.potential(geo.p_x, geo.p_y, geo.p_z)
        if subtr:
            psi = psi - clu.potential(0, 0, 0)  # mind broadcasting here
        psi = np.abs(psi)
        x_abs, psi_p = Gt.strip(geo.p_s, psi, 1.e-20, 1.e-12, 1.e6)
        axi.set_ylabel(r"$|\Phi|$")

        # psi = np.clip(psi, 1.e-25, 1.e+25)  # avoid trouble with autosc
        _, l_end, slp = Gt.get_slopes(x_abs, psi_p)
        axi.loglog(x_abs, psi_p, color='r', label=r'$|\Phi|$')
        hhstr = r'$\psi\propto r^{' + f'{slp[1]:2.1f}' + '}$'
        axi.loglog(x_abs[-5*l_end: -1],
                   psi_p[-1]*pow(x_abs[-5*l_end: -1]/x_abs[-1], slp[1]),
                   color="black", linestyle='dashed', label=hhstr)
        hhstr = r'$\psi\propto r^{' + f'{slp[0]:2.1f}' + '}$'
        axi.loglog(x_abs[1: 5*l_end],
                   psi_p[1]*pow(x_abs[1: 5*l_end]/x_abs[1], slp[0]),
                   color="m", linestyle='dashdot', label=hhstr)
        axi.legend(loc='best', shadow=False, fontsize='small')

    @staticmethod
    def get_b(clu, geo, subtr):
        """Get the field, subtract conditionally."""
        b_x, b_y, b_z = clu.b_flux(geo.p_x, geo.p_y, geo.p_z)
        if subtr:
            z_x, z_y, z_z = clu.b_flux(0, 0, 0)
            b_x = b_x - z_x  # not the same as -=, broadcasting?
            b_y = b_y - z_y
            b_z = b_z - z_z
        b_abs = np.sqrt(b_x*b_x + b_y*b_y + b_z*b_z)
        return b_x, b_y, b_z, b_abs

    @staticmethod
    def b_plt(axi, clu, geo, subtr=False):
        """Plot the magnetic induction B in pic1 on a linear scale."""
        b_x, b_y, b_z, b_abs = Tl.get_b(clu, geo, subtr)

        supermax = max(b_x.max(), b_y.max(), b_z.max(), b_abs.max())
        supermin = min(b_x.min(), b_y.min(), b_z.min(), b_abs.min())
        axi.set_ylim(supermin-(supermax-supermin)/20,
                     supermax+(supermax-supermin)/20)
        axi.plot(geo.p_s, b_x, 'm:', label=r"$B_\mathrm{x}$")
        axi.plot(geo.p_s, b_y, 'g:', label=r"$B_\mathrm{y}$")
        axi.plot(geo.p_s, b_z, 'b:', label=r"$B_\mathrm{z}$")
        axi.plot(geo.p_s, b_abs, 'r-', label=r"$B$")
        axi.legend(loc='best', shadow=False, fontsize='small')

    @staticmethod
    def b_plt_log(axi, clu, geo, subtr=False):
        """Plot pic1 as loglog."""
        b_x, b_y, b_z, b_abs = Tl.get_b(clu, geo, subtr)
        x_abs, b_absp = Gt.strip(geo.p_s, b_abs, 1.e-20, 1.e-9, 1.e6)
        # supermax = 1.05*max(b_x.max(), b_y.max(), b_z.max(), -min(b_x),
        #                     -min(b_y), -min(b_z))
        axi.set_ylim(min(b_absp[-1], b_absp[0]), b_absp.max()*2.)
        axi.set_xlim(x_abs[0], x_abs[-1])
        axi.loglog(x_abs, b_absp, 'r-', label=r"$B$")

        axi.loglog(geo.p_s[1:-1], abs(b_x[1:-1]), 'm:',
                   label=r"$|B_\mathrm{x}|$")
        axi.loglog(geo.p_s[1:-1], abs(b_y[1:-1]), 'g:',
                   label=r"$|B_\mathrm{y}|$")
        axi.loglog(geo.p_s[1:-1], abs(b_z[1:-1]), 'b:',
                   label=r"$|B_\mathrm{z}|$")
        _, n_end, slp = Gt.get_slopes(x_abs, b_absp)
        hhstr = r'$B\propto r^{' + f'{slp[1]:2.0f}' + '}$'
        axi.loglog(x_abs[-5*n_end: -1],
                   b_absp[-1]*pow(x_abs[-5*n_end: -1]/x_abs[-1], slp[1]),
                   color="black", linestyle='dashed', label=hhstr)
        hhstr = r'$B\propto r^{' + f'{slp[0]:2.0f}' + '}$'
        axi.loglog(x_abs[1: 5*n_end],
                   b_absp[1]*pow(x_abs[1: 5*n_end]/x_abs[1], slp[0]),
                   color='grey', linestyle='dashdot', label=hhstr)
        axi.legend(loc='best', shadow=False, fontsize='small')


class Gt:
    """General tools."""

    comment = True

    @staticmethod
    def set_my_rc():
        """My prefered rc settings."""
        plt.rc('axes', fc='white', edgecolor='black', titlesize=20,
               labelsize=12, grid=True)  # set the format of the plotaxes
        plt.rc('legend', fontsize=12, frameon=True, numpoints=1,
               labelspacing=0.3, columnspacing=0)    # format of the legend
        plt.rc('xtick', labelsize=12, top=True, direction='in')  # ticklabels
        plt.rc('ytick', labelsize=12, right=True, direction='in')
        plt.rc('font', family="sans-serif", size=12)
        plt.rc('text', usetex=False)  # using Latex seems slower
        # params = {'text.latex.preamble': r'\usepackage{amsmath}'}
        # plt.rcParams.update(params)

    @staticmethod
    def enable_messages(on_off):
        """Conditional printing for performance checks."""
        Gt.comment = on_off

    @staticmethod
    def msg(string, cond=True):
        """Conditional printing for performance checks."""
        if Gt.comment and cond:
            print(string)

    @staticmethod
    def strip(x_in, y_in, bound_x, bound_y, bound_yh):
        """Prepare data for log-log presention by stripping off tiny values."""
        x_r = x_in[y_in > bound_y]  # somew. arbitrary..
        y_r = y_in[y_in > bound_y]
        x_r = x_r[y_r < bound_yh]
        y_r = y_r[y_r < bound_yh]
        y_r = y_r[x_r > bound_x]  # avoid zeros for the x_axis
        x_r = x_r[x_r > bound_x]  # reduce the x_axis
        return x_r, y_r

    @staticmethod
    def get_slopes(xxx, yyy):
        """Get slopes and ranges for the asymptotes."""
        l_end = int(len(yyy)/10)  # 1/10 of the line
        slp = [None, None]
        slp[1] = np.log(yyy[-1] / yyy[-l_end])
        slp[1] /= np.log(xxx[-1]/xxx[-l_end])
        slp[0] = np.log(yyy[1] / yyy[l_end])   # slope for the inner part
        slp[0] /= np.log(xxx[1]/xxx[l_end])
        return 1, l_end, slp

    @staticmethod
    def create_bgr(stripes=False):
        """Create a colormap from blue over green to red, black stripes opt."""
        num = 1024  # even preferred?
        nu1 = num-1
        n_4 = num//4
        n_3_4 = num//4 * 3
        n_ha = num//2
        n_h1 = n_ha - 1  # should be -1
        n_41 = n_4 + 100
        viridis = mpl.colormaps['viridis'].resampled(num)  # listed colormap
        newcolors = viridis(np.linspace(0, 1, num))
        for i in range(n_4):
            newcolors[i] = np.array([0, 0, 1-i/n_h1, 1])
        for i in range(n_4, n_ha):
            newcolors[i] = np.array([0, (i-n_4)/n_41, 1-i/n_h1, 1])
        for i in range(n_ha, n_3_4):
            newcolors[i] = np.array([(i-n_ha)/n_h1, (n_3_4-i)/n_41, 0, 1])
        for i in range(n_3_4, num):
            newcolors[i] = np.array([(i-n_ha)/n_h1, 0, 0, 1])
        if stripes:
            for i in range(num):
                newcolors[i, 0:3] *= 1 - pow(np.sin(i/nu1*2*np.pi*3), 128)
        cmp = ListedColormap(newcolors)
        return cmp

    @staticmethod
    def create_bwr_s(stripes=True):
        """Modify the blue to red colormap, black stripes opt."""
        num = 1024  # even preferred?
        nu1 = num-1
        hhh = mpl.colormaps['bwr'].resampled(num)  # listed colormap
        newcolors = hhh(np.linspace(0, 1, num))
        if stripes:
            for i in range(num):
                newcolors[i, 0:3] *= 1 - pow(np.sin(i/nu1*2*np.pi*3), 128)
        return ListedColormap(newcolors)

    @staticmethod
    def col_arr(t_x):
        """Continuos color change."""
        ens = (t_x + 1) / 2
        return bgr(ens)  # the type of color coding

    @staticmethod
    def save_configuration(clu, clu_type):
        """Save the current figure, write cluster configuration to file."""
        name = datetime.now().isoformat('_', 'seconds')
        name = name.replace(':', '-')
        name = 'config/' + name
        Gt.msg('saving files to: ' + name)
        plt.savefig(name+'.pdf')
        d_s = np.concatenate((clu.r_vec, clu.p_vec), axis=1)
        h_s = ' Dipole cluster configuration of '  # generate header for csv
        h_s += clu_type + '\n'
        h_s += ' Positions ;    :; vec(r); Magnetic; moment:; vec(p)\n'
        h_s += ' r_x /d ;r_y /d ;r_z /d ;p_x /p ;p_y /p ;p_z /p '
        with open(name+'.csv', 'w', encoding='ascii') as fil:
            np.savetxt(fil, d_s, delimiter=';', fmt='%f',
                       header=h_s, comments='#')


bgrs = Gt.create_bgr(stripes=True)
bgr = Gt.create_bgr(stripes=False)


def n_e(dlist):
    """Count number of elements in a list of lists."""
    cnt = 0
    for dum in dlist:
        cnt += len(dum)
    return cnt


if __name__ == '__main__':
    print(dir(Tl))
