"""Shared parameters and plots for the first figure of dipole_clusters.

Version 1.2.1,  May 1, 2024, tested with Python 3.11.7, rated 10.00/10
"""
import sys
import random
import numpy as np
import matplotlib.pyplot as plt
from matplotlib.widgets import Slider, RadioButtons, Button, CheckButtons

try:
    import cluster_cluster_plots as ccp
    from shared import Mg
except ImportError:
    import dipole_clusters_modules.cluster_cluster_plots as ccp
    from dipole_clusters_modules.shared import Mg
try:
    from dipole_clusters_modules import dipole_cluster as dct
    from dipole_clusters_modules.plot_tools import Tl, Gt, bgrs, n_e
except ImportError:
    sys.path.append(r'D:\Py\general')
    import dipole_cluster as dct
    from plot_tools import Tl, Gt, bgrs, n_e


class Cb:
    """CheckButtons and their functions."""

    profile = {'-y(0)': False}
    disp_box, prof_box, col = [None]*3

    @staticmethod
    def place_display(options):
        """Create and connect checkbox with colors display options."""
        Cb.disp_box = CheckButtons(
            plt.axes([0.005, 0.005, 0.06, 0.14], fc='lightgoldenrodyellow'),
            labels=options.keys(), actives=options.values(), useblit=False)
        Cb.set_col(options)
        Cb.disp_box.on_clicked(Cb.change_options)

        Cb.prof_box = CheckButtons(
            plt.axes([0.72, 0.4, 0.03, 0.03], fc='lightgoldenrodyellow'),
            labels=Cb.profile.keys(), actives=Cb.profile.values(),
            useblit=False)
        Cb.prof_box.on_clicked(Cb.change_prof)

    @staticmethod
    def set_col(options):
        """Set colors for the list used by disp_box."""
        Cb.col = ['g' if x else 'r' for x in list(options.values())]
        Cb.disp_box.set_label_props({'color': Cb.col})
        Cb.disp_box.set_frame_props({'edgecolor': Cb.col})
        Cb.disp_box.set_check_props({'facecolor': Cb.col})

    @staticmethod
    def change_prof(_):
        """React to click on prof_box."""
        Gt.msg('change_prof started')
        status = Cb.prof_box.get_status()
        Cb.profile = dict(zip(Cb.profile.keys(), status))
        Pi.pic[1](Pl.ax[1], Mg.g)
        Pl.flush_figs()
        Gt.msg('change_prof finished')

    @staticmethod
    def change_options(_):
        """React to click on disp_box."""
        Gt.msg('change_options started')
        status = Cb.disp_box.get_status()
        Mg.disp = dict(zip(Mg.disp.keys(), status))
        Pi.pic[0](Pl.ax[0], Mg.g)
        Cb.set_col(Mg.disp)
        Sc.trp_control()
        Sc.res_control()
        Gt.comment = Mg.disp.get('msg')
        Pl.flush_figs()
        Gt.msg('change_options   finished')


class Rb:
    """RadioButtons and their functions."""

    log_lin_sel, tri_cub_sel, dst_mom_sel, rot_sel = [None]*4  # 4 RadioButtons
    logplt, dst, clu_choice, fix, rot = 1, True, None, True, False

    @staticmethod
    def check_clu_choice(name):
        """Check for confusion concerning the choice of the cluster."""
        act_name = Rb.clu_choice.value_selected
        o_k = act_name == name
        Gt.msg(f'Warning: {name} not selected, but:{act_name}!', cond=not o_k)
        return o_k

    @staticmethod
    def mrb(x_p, y_p, lab, a_n):
        """General form for small radiobuttons."""
        axi = plt.axes([x_p, y_p, 0.032, 0.08], fc='lightgoldenrodyellow')
        return RadioButtons(axi, lab, active=a_n)

    @staticmethod
    def dst_mom_update(label):
        """Choose distance or moment as controlp. for ring stack cluster."""
        Gt.msg('try dst_mom_update')
        if not Rb.check_clu_choice('Tubes'):
            return
        Rb.dst = label == 'dst'
        Pi.pic[1](Pl.ax[1], Mg.g)
        Pl.flush_figs()
        Gt.msg('dst_mom_update finished')

    @staticmethod
    def log_lin_update(label):
        """Toggle between linear and loglog presentation of B and psi."""
        Gt.msg('log_lin_update started')
        Rb.logplt = Rb.log_lin_options.index(label)
        Pi.pic[1](Pl.ax[1], Mg.g)
        Pl.flush_figs()
        Gt.msg('log_lin_update finished')

    @staticmethod
    def rotation_update(label):
        """Start 1 out of 3 rotation modes in Pl.rotate() , or stop them."""
        Gt.msg('rotation_update started with: '+label)

        def rot_on():
            """For local use."""
            return Rb.rot
        if label in ['xy', 'xz', 'yz', 'st', 'fix', 'exit']:
            _ = Pl.ax[0].view_init(90, -90, 0) if label == 'xy' else None
            _ = Pl.ax[0].view_init(0,  -90, 0) if label == 'xz' else None
            _ = Pl.ax[0].view_init(0,    0, 0) if label == 'yz' else None
            _ = Pl.ax[0].view_init(30, -60, 0) if label == 'st' else None
            plt.draw()
            _ = plt.close(Pl.fig) if label == 'exit' else None
        if label in ['elev', 'azim', 'roll'] and not rot_on():
            Rb.rotate()  # leaves 2 callbacks open, make sure to exit properly
        Gt.msg('rotation_update finished')

    @staticmethod
    def rotate():
        """Rotate viewing angle, be sure to exit via value_selected."""
        Gt.msg('rotation started')
        ele, azi, rol = Pl.ax[0].elev, Pl.ax[0].azim, Pl.ax[0].roll
        Rb.rot = True

        def go_on():
            """For local use."""
            return Rb.rot_sel.value_selected in ['elev', 'azim', 'roll']
        while go_on():
            ele = (ele+1) % 360 if Rb.rot_sel.value_selected == 'elev' else ele
            azi = (azi-1) % 360 if Rb.rot_sel.value_selected == 'azim' else azi
            rol = (rol-1) % 360 if Rb.rot_sel.value_selected == 'roll' else rol
            Pl.ax[0].view_init(ele, azi, rol)
            plt.draw()
            plt.pause(0.01)
        Rb.rot = False
        Gt.msg('rotation finished')

    @staticmethod
    def select_cluster(label):
        """Select a cluster from the RadioButton 'clu_choice'."""
        Gt.msg('select_cluster started')
        Pl.flush_figs()
        Mg.select_start_cluster(cluster=label)
        Bu.rlx_clr()  # the memory for relaxation steps
        Sc.set_conditional_controls(new_cluster=True)
        Gt.msg('choosing:' + str(Rb.clu_choice.value_selected) +
               ' selected type:' + str(type(Mg.clu_act).__name__))
        Pl.replt_ax(new_cluster=True, new_config=True)
        Gt.msg('select_cluster finished, selected:'+label)


class Bu:
    """Buttons and their functions."""

    rlx_but,  rlxm_but,  sw_plt_but, sav_but = [None]*4
    cnv_but, dst_but, dst_but_small, star_but, \
        hal_d_but, halb_but, hal_i_but = [None]*7
    mem, tim_stp, k_turn, tau = 100, 0, 0, 0.
    pot, num = np.ones(mem),  np.array(range(-mem+1, 1))

    @staticmethod
    def sav(_):
        """Reaction to sav-button."""
        Gt.save_configuration(Mg.clu_act, Rb.clu_choice.value_selected)
        Gt.msg("sav finished")

    @staticmethod
    def set_halbach_text():
        """Set the text of the Halbach button."""
        Bu.halb_but.label.set_text(r'H$_{' + f'{Bu.k_turn}' + r'}$')

    @staticmethod
    def set_hedgehog():
        """Set hedgehog configuration."""
        for i, vec in enumerate(Mg.clu_act.r_vec):
            Mg.clu_act.p_vec[i] = vec if \
                np.dot(vec, vec) > 1e-7 else [1, 0, 0]
        Mg.clu_act.scale_p()

    @staticmethod
    def set_ha(turn=k_turn):
        """Set halbach configuation."""
        Bu.k_turn = turn
        for dpn, r_v in enumerate(Mg.clu_act.r_vec):
            ang = np.arctan2(r_v[1], r_v[0])*(1+Bu.k_turn)
            Mg.clu_act.p_vec[dpn] = (np.cos(ang), np.sin(ang), 0)
        Mg.clu_act.scale_p()
        Bu.set_halbach_text()
        Pl.replt_ax(new_cluster=False, new_config=True)
        Gt.msg(f'k_turn = {Bu.k_turn}')

    @staticmethod
    def halb(_):
        """Set all direction in Halbach configuration, keep the length."""
        Gt.msg('halb started')
        Bu.set_ha(turn=Bu.k_turn)
        Gt.msg('halb finished')

    @staticmethod
    def hal_d(_):
        """Decrease k and set direction in Halbach configuration."""
        Gt.msg('hal_d started')
        Bu.set_ha(turn=max(Bu.k_turn-1, -9))
        Gt.msg('hal_d finished')

    @staticmethod
    def hal_i(_):
        """Increase k and set direction in Halbach configuration."""
        Gt.msg('hal_i started')
        Bu.set_ha(turn=min(Bu.k_turn+1, 9))
        Gt.msg('hal_i finished')

    @staticmethod
    def star(_):
        """Disturb by resetting all directions radially, keep the length."""
        Gt.msg('star started')
        Bu.set_hedgehog()
        Pl.replt_ax(new_cluster=False, new_config=True)
        Gt.msg('star finished')

    @staticmethod
    def dst(_):
        """Disturb by resetting all directions randomly."""
        Gt.msg('dst started')
        Mg.clu_act.create_rnd()
        Pl.replt_ax(new_cluster=False, new_config=True)
        Gt.msg('dst finished')

    @staticmethod
    def dst_small(_):
        """Disturb current orientation slightly, adding +-10% randomly."""
        Gt.msg('dst_small started')
        for i, l_e in enumerate(Mg.clu_act.lengths):
            distort = np.array([random.random()*2-1, random.random()*2-1,
                               random.random()*2-1])
            Mg.clu_act.p_vec[i] += distort*l_e*0.1
        Mg.clu_act.scale_p()  # scale according to lengths
        Pl.replt_ax(new_cluster=False, new_config=True)
        Gt.msg('dst_small   finished')

    @staticmethod
    def rlxm(_):
        """Search a solution with lower energy."""
        Gt.msg('searching...')
        p_vec = np.copy(Mg.clu_act.p_vec)  # save the old one
        u_old = Mg.clu_act.w_total_fast()
        failed = True
        for k in range(1, 100):
            Mg.clu_act.create_rnd()
            for j, _ in enumerate(Bu.pot):
                Mg.clu_act.relaxation()
                Bu.pot[j] = Mg.clu_act.w_total_fast()
            if Bu.pot[-1] < u_old:
                failed = False
                Gt.msg('found'+str(Bu.pot[-1])+'after KICK '+str(k))
                break
        if failed:
            Gt.msg('failed')
            Mg.clu_act.p_vec = p_vec
            Bu.rlx(0)
        else:
            Pl.replt_ax(new_cluster=False, new_config=True)
        Gt.msg('rlxm   finished')

    @staticmethod
    def rlx(_):
        """Do len(Bu.pot) relaxation steps and store the history."""
        Gt.msg('rlx started')
        if Pl.fig is not None:
            Pl.flush_figs()
        if Mg.clu_act.n_dip() == 1:
            Mg.clu_act.p_vec[0] = (1, 0, 0)
            Bu.pot *= 0
        else:
            for _ in range(Bu.mem):
                Mg.clu_act.relaxation()
                Bu.store_hist()
        Pl.replt_ax(new_cluster=False, new_config=False)
        Gt.msg('rlx finished')

    @staticmethod
    def store_hist():
        """Store an energy value at the next location."""
        Bu.tim_stp += 1
        Bu.tim_stp %= Bu.mem
        Bu.num -= 1  # decrease all numbers
        Bu.pot[Bu.tim_stp] = Mg.clu_act.w_total_fast()
        Bu.num[Bu.tim_stp] = 0  # the last one stored is zero

    @staticmethod
    def cnv(_):
        """Repeat relaxation steps untils some convergence is reached."""
        Gt.msg('cnv started')
        if Pl.fig is not None:
            Pl.flush_figs()
        if Mg.clu_act.n_dip() == 1:
            Mg.clu_act.p_vec[0] = (1, 0, 0)
            Bu.pot *= 0
        else:
            for _ in range(Bu.mem):
                Mg.clu_act.relaxation()
                Bu.store_hist()
            while (Bu.pot.max()-Bu.pot.min()) > 1.e-14:  # compromise...
                Mg.clu_act.relaxation()
                Bu.store_hist()
        Pl.replt_ax(new_cluster=False, new_config=False)
        Gt.msg('cnv finished')

    @staticmethod
    def rlx_clr():
        """Clear the relaxation history."""
        if Mg.clu_act.n_dip() == 1:
            Mg.clu_act.p_vec[0] = (1, 0, 0)
            Bu.pot *= 0
        else:
            Bu.pot.fill(Mg.clu_act.w_total_fast())

    @staticmethod
    def newplt(_):
        """Switch to another plot."""
        Gt.msg('newplt of 2. page started')
        Pl.flush_figs()
        if ccp.Mg.fig is None:
            ccp.plot_layout_pair()
        else:
            plt.figure(ccp.Mg.fig)
            ccp.plt_7_axes()
        plt.show()
        Gt.msg('newplt finished')


class Sc:
    """13 conditional sliders, their positions, functions and parameters."""

    cube_adj = [None]*4  # sliders for lengths and tau of the cube clusters
    ring_st_adj = [None]*3  # length of a ring, number of rings, m
    hexagon_adj = [None]*2  # number of filled hexgons stacked, centre moment
    fcc_adj = [None]*5  # sliders for adjusting parameters of the fcc-cluster
    tpl_adj = [None]*1
    trp_adj, res_adj = None, None  # slider sphere transparency & resolution
    clu_cond_sl = [cube_adj, ring_st_adj, hexagon_adj, fcc_adj, tpl_adj]
    y = [0.98 - i*0.015 for i in range(n_e(clu_cond_sl))]
    cube_p = [[0.11, y[0], 0.15, 0.013],
              [0.11, y[2], 0.15, 0.013],
              [0.11, y[4], 0.15, 0.013]]
    ring_st_p = [[0.11, y[3], 0.15, 0.013],
                 [0.11, y[6], 0.15, 0.013],
                 [0.11, y[10], 0.15, 0.013]]  # m, also for the filled hexagon
    hexagon_p = [[0.11, y[8], 0.15, 0.013],  # no. of stacked filled hexagons
                 [0.11, y[1], 0.15, 0.013]]  # m, strength of centre dipol
    fcc_p = [[0.11, y[5], 0.15, 0.013],
             [0.11, y[7], 0.15, 0.013],
             [0.11, y[9], 0.15, 0.013],
             [0.11, y[11], 0.15, 0.013],
             [0.11, y[12], 0.15, 0.013]]
    tpl_p = [[0.11, y[13], 0.15, 0.013]]
    trp_p = [0.13, 0.05, 0.18, 0.02]
    res_p = [0.13, 0.02, 0.18, 0.02]
    tri_cub_p = [0.3, y[6]]  # position for a conditional radio button
    tau = 0

    @staticmethod
    def reset_conditional_sliders():
        """Set all cluster_specific conditional sliders to initial value."""
        for out in Sc.clu_cond_sl:
            for sli in out:
                if sli is not None:
                    sli.reset()
        Pl.flush_figs()

    @staticmethod
    def set_conditional_controls(new_cluster=True):
        """Install if selected & set the visibility of conditional sliders."""
        Sc.trp_control()
        Sc.res_control()
        Sc.ring_st_control()
        Sc.hexagon_control()
        Sc.cube_control()
        Sc.fcc_control()
        Sc.tpl_control()
        if new_cluster:
            Sc.reset_conditional_sliders()

    @staticmethod
    def cube_control():
        """Control the edge length of the cuboid family."""
        if Rb.clu_choice is None:
            Gt.msg('Rb.clu_choice not yet defined')
            return
        vis = Rb.clu_choice.value_selected == 'Cube'
        if Sc.cube_adj[0] is None and vis:
            for i, lab in enumerate([r"$e_x$", r"$e_y$", r"$e_z$"]):
                Sc.cube_adj[i] = Slider(plt.axes(Sc.cube_p[i]),
                                        lab, 1, 6, valfmt='%1.0f',
                                        valstep=np.array([1, 2, 3, 4, 5, 6]),
                                        valinit=Mg.clu_act.edge[i])
            _ = [sli.on_changed(Sc.cube_update) for sli in Sc.cube_adj[0:3]]
            Sc.cube_adj[3] = Slider(plt.axes([0.37, 0.65, 0.15, 0.02]),
                                    r'$\tau$', 0, 360, valfmt='%1.0f',
                                    valinit=Sc.tau)
            Sc.cube_adj[3].on_changed(Sc.cube_tau_update)

        if Sc.cube_adj[0] is not None:
            _ = [slider.ax.set_visible(vis) for slider in Sc.cube_adj[0:3]]
            vis_2 = vis and Mg.clu_act.edge == [2, 2, 2]
            Sc.cube_adj[3].ax.set_visible(vis_2)

    @staticmethod
    def cube_update(_):
        """Adjust all figures with respect to new edge."""
        Gt.msg('cube_update')
        if not Rb.check_clu_choice('Cube'):
            return
        Gt.msg('started')
        Mg.clu_act = dct.Cube(edge=[s.val for s in Sc.cube_adj[0: 3]])
        Pl.replt_ax(new_cluster=False, new_config=True)
        Gt.msg('cube_update   finished')

    @staticmethod
    def cube_tau_update(value):
        """Adjust all figures with respect to new edge."""
        Gt.msg('cube_tau_update')
        if not Rb.check_clu_choice('Cube'):
            return
        Sc.tau = value
        Mg.clu_act.set_ang(Sc.tau)

        Pl.replt_ax(new_cluster=False, new_config=True)
        Gt.msg('cube_tau_update   finished')

    @staticmethod
    def ring_st_control():
        """Control the length of a ring, the stack number, and mom/dst."""
        if Rb.clu_choice is None:
            Gt.msg('Rb.clu_choice not yet defined')
            return
        vis = Rb.clu_choice.value_selected == 'Tubes'
        typ = type(Mg.clu_act).__name__ == 'Tube'
        Gt.msg('vis!=typ error!'+str(vis)+str(typ), cond=vis != typ)
        if vis and (Sc.ring_st_adj[0] is None):  # initialize only if selected
            Sc.ring_st_adj[0] = Slider(plt.axes(Sc.ring_st_p[0]),
                                       r'$L_\mathrm{r}$', 2, 20,
                                       valfmt='%3.0f',
                                       valstep=np.array(range(2, 21)),
                                       valinit=Mg.clu_act.l_r)
            Sc.ring_st_adj[1] = Slider(plt.axes(Sc.ring_st_p[1]),
                                       r'$n_\mathrm{s}$', 1, 8, valfmt='%3.0f',
                                       valstep=np.array(range(1, 9)),
                                       valinit=Mg.clu_act.n_s)
            lab = r'$m$' if not Rb.dst else r'$d$'
            v_ini = Mg.clu_act.dst if Rb.dst else Mg.clu_act.mom
            Sc.ring_st_adj[2] = Slider(plt.axes(Sc.ring_st_p[2]), lab, 0, 3,
                                       valstep=np.linspace(0, 3, 301),
                                       valfmt='%3.2f', valinit=v_ini)
            Sc.ring_st_adj[0].on_changed(Sc.ring_st_update)
            Sc.ring_st_adj[1].on_changed(Sc.ring_st_update)
            Sc.ring_st_adj[2].on_changed(Sc.ring_st_outer_update)
            Rb.tri_cub_sel = Rb.mrb(Sc.tri_cub_p[0], Sc.tri_cub_p[1],
                                    ('sqr', 'tri'), int(Mg.clu_act.tri))
            Rb.tri_cub_sel.on_clicked(Sc.ring_st_update)
            Rb.dst_mom_sel = Rb.mrb(Sc.tri_cub_p[0], Sc.tri_cub_p[1]-0.1,
                                    ('mom', 'dst'), int(Rb.dst))
            Rb.dst_mom_sel.on_clicked(Sc.ring_st_dst_mom_update)

        if Sc.ring_st_adj[0] is not None:
            _ = [sli.ax.set_visible(vis) for sli in Sc.ring_st_adj]
            _ = [circ.set_visible(vis) for circ in Rb.tri_cub_sel.circles]
            Rb.tri_cub_sel.ax.set_visible(vis)
            _ = [circ.set_visible(vis) for circ in Rb.dst_mom_sel.circles]
            Rb.dst_mom_sel.ax.set_visible(vis)

    @staticmethod
    def ring_st_dst_mom_update(_):
        """Select the length of a ring, and number of rings in the stack."""
        Gt.msg('ring_st_dst_mom_update')
        if not Rb.check_clu_choice('Tubes'):
            return
        Rb.dst = Rb.dst_mom_sel.value_selected == 'dst'
        if Rb.dst:
            Sc.ring_st_adj[2].set_val(Mg.clu_act.dst)
        else:
            Sc.ring_st_adj[2].set_val(Mg.clu_act.mom)
        Sc.ring_st_outer_update(0)
        Gt.msg('ring_st_dst_mom_update   finished')

    @staticmethod
    def ring_st_update(_):
        """Select the length of a ring, and number of rings in the stack."""
        Gt.msg('ring_st_update')
        if not Rb.check_clu_choice('Tubes'):
            return
        Gt.msg('started')
        l_r = Sc.ring_st_adj[0].val
        n_s = Sc.ring_st_adj[1].val
        tri = Rb.tri_cub_sel.value_selected == 'tri'
        dst = Mg.clu_act.dst  # leave this value as is
        mom = Mg.clu_act.mom  # leave this value as is
        del Mg.clu_act
        Mg.clu_act = dct.Tube(tri=tri, n_rz=(l_r, n_s), dst=dst, mom=mom)
        Pl.replt_ax(new_cluster=False, new_config=True)
        Gt.msg('ring_st_update   finished')

    @staticmethod
    def ring_st_outer_update(_):
        """Set the dipole strength m in the outer rings, or its height."""
        Gt.msg('ring_st_outer_update')
        if not Rb.check_clu_choice('Tubes'):
            return
        Gt.msg('started')
        Rb.dst = Rb.dst_mom_sel.value_selected == 'dst'  # does no harm
        if Rb.dst:
            Mg.clu_act.adjust_dst(Sc.ring_st_adj[2].val)
            Sc.ring_st_adj[2].label.set_text('$d$')
        else:
            Mg.clu_act.adjust_m(max(Sc.ring_st_adj[2].val, 0.001))  # avoid 0
            Sc.ring_st_adj[2].label.set_text('$m$')
        Pl.replt_ax(new_cluster=False, new_config=True)
        Gt.msg('ring_st_outer_update   finished')

    @staticmethod
    def hexagon_control():
        """Control the number of filled hexagons in the stack of filled hex."""
        if Rb.clu_choice is None:
            print('Rb.clu_choice not yet defined')
            return
        vis = Rb.clu_choice.value_selected == 'Hexagon'
        if vis and Sc.hexagon_adj[0] is None:
            Sc.hexagon_adj[0] = Slider(plt.axes(Sc.hexagon_p[0]),
                                       r'$n_\mathrm{s}$', 1, 8, valfmt='%3.0f',
                                       valstep=np.array(range(1, 9)),
                                       valinit=Mg.clu_act.n_s)
            Sc.hexagon_adj[1] = Slider(plt.axes(Sc.hexagon_p[1]),
                                       r'$m$', 0.01, 3, valfmt='%3.2f',
                                       valinit=Mg.clu_act.lengths[0])
            Sc.hexagon_adj[0].on_changed(Sc.hexagon_update)
            Sc.hexagon_adj[1].on_changed(Sc.hex_center_update)
        if Sc.hexagon_adj[0] is not None:
            _ = [slider.ax.set_visible(vis) for slider in Sc.hexagon_adj]

    @staticmethod
    def hexagon_update(_):
        """Select the number of filled hexagons forming the stack."""
        Gt.msg('hexagon_update')
        if not Rb.check_clu_choice('Hexagon'):
            return
        n_s = Sc.hexagon_adj[0].val
        del Mg.clu_act
        Mg.clu_act = dct.HexagonFilled(int(n_s))
        Pl.replt_ax(new_cluster=False, new_config=True)
        Gt.msg('hexagon_update finished')

    @staticmethod
    def hex_center_update(_):
        """Select the number of filled hexagons forming the stack."""
        Gt.msg('hex_center_update')
        if not Rb.check_clu_choice('Hexagon'):
            return
        Mg.clu_act.adjust_m(Sc.hexagon_adj[1].val)
        Pl.replt_ax(new_cluster=False, new_config=True)
        Gt.msg('hexagon_center_update   finished')

    @staticmethod
    def fcc_control():
        """Control the geometry of the face centred cubic lattices."""
        if Rb.clu_choice is None:
            print('Rb.clu_choice not yet defined')
            return
        vis = Rb.clu_choice.value_selected == 'FCC'
        if Sc.fcc_adj[0] is None and vis:
            for i, lab in enumerate([r"$e_i$", r"$e_j$", r"$e_k$"]):
                Sc.fcc_adj[i] = Slider(
                    plt.axes(Sc.fcc_p[i]), lab, 1, 6, valfmt='%1.0f',
                    valstep=np.array(range(1, 7)), valinit=Mg.clu_act.ijk[i])
            Sc.fcc_adj[3] = Slider(
                plt.axes(Sc.fcc_p[3]), r"$r_i$", -0.1, 2.1, valfmt='%2.1f',
                valinit=Mg.clu_act.r_b[0])
            Sc.fcc_adj[4] = Slider(
                plt.axes(Sc.fcc_p[4]), r"$r_o$", 0.95, 3.2, valfmt='%1.1f',
                valinit=Mg.clu_act.r_b[1])
            _ = [slider.on_changed(Sc.fcc_update) for slider in Sc.fcc_adj]
        if Sc.fcc_adj[0] is not None:
            _ = [slider.ax.set_visible(vis) for slider in Sc.fcc_adj]

    @staticmethod
    def fcc_update(_):
        """Adjust the fcc lattice with respect to 5 parameters."""
        Gt.msg('fcc_update started')
        if not Rb.check_clu_choice('FCC'):
            return
        Mg.clu_act = dct.FCC(r_b=[Sc.fcc_adj[i].val for i in (3, 4)],
                             ijk=[Sc.fcc_adj[i].val for i in range(3)])
        Pl.replt_ax(new_cluster=False, new_config=True)
        Gt.msg('fcc_update finished')

    @staticmethod
    def tpl_control():
        """Control the geometry the tetraplex."""
        if Rb.clu_choice is None:
            print('Rb.clu_choice not yet defined')
            return
        vis = Rb.clu_choice.value_selected == 'TPlex'
        if Sc.tpl_adj[0] is None and vis:
            Sc.tpl_adj[0] = Slider(
                plt.axes(Sc.tpl_p[0]), 'cut', 0, 4, valfmt='%1.0f',
                valstep=np.array(range(0, 5)), valinit=Mg.clu_act.cut)
            Sc.tpl_adj[0].on_changed(Sc.tpl_update)
        if Sc.tpl_adj[0] is not None:
            Sc.tpl_adj[0].ax.set_visible(vis)

    @staticmethod
    def tpl_update(_):
        """Adjust the tpl."""
        Gt.msg('tpl_update started')
        if not Rb.check_clu_choice('TPlex'):
            return
        Mg.clu_act = dct.Tetraplex(without=0, cut=Sc.tpl_adj[0].val)
        Pl.replt_ax(new_cluster=False, new_config=True)
        Gt.msg('tpl_update   finished')

    @staticmethod
    def trp_control():
        """Control the transparency by setting alpha."""
        if Sc.trp_adj is None:
            Sc.trp_adj = Slider(
                plt.axes(Sc.trp_p),
                '$\\alpha$', 0, 1, valfmt='%1.1f', valinit=0.3)
            Sc.trp_adj.on_changed(Sc.trp_update)
        vis = any([Mg.disp.get('plane'), Mg.disp.get('sphere'),
                   Mg.disp.get('sphrs')])
        Sc.trp_adj.ax.set_visible(vis)

    @staticmethod
    def trp_update(_):
        """Adjust the tranparency of the sphere or the plane."""
        Gt.msg('trp_update started')
        Pi.pic[0](Pl.ax[0], Mg.g)
        Pl.flush_figs()
        Gt.msg('trp_update   finished')

    @staticmethod
    def res_control():
        """Control the resolution of the surrounding sphere."""
        if Sc.res_adj is None:
            Sc.res_adj = Slider(
             plt.axes(Sc.res_p), r'$n_\mathrm{sp}$', 10, 360, valfmt='%3.0f',
             valstep=np.array([15, 30, 45, 90, 180, 360]), valinit=40)
            Sc.res_adj.on_changed(Sc.res_update)
        vis = any([Mg.disp.get('sphere'), Mg.disp.get('sphrs')])
        Sc.res_adj.ax.set_visible(vis)

    @staticmethod
    def res_update(_):
        """Select the sphere resolution (small for a speedy presentation)."""
        Gt.msg('res_update started')
        Pi.pic[0](Pl.ax[0], Mg.g)
        Pl.flush_figs()
        Gt.msg('res_update finished')


class Pl:
    """Plot and subplots."""

    fig = plt.figure('The field of dipole clusters', figsize=(10.*16/9, 10.))
    ax, refresh = [None]*9, [None]*9

    @staticmethod
    def place_ax():
        """Place the 9 axes."""
        if Pl.ax[0] is not None:
            return
        q_x, q_y, left2 = 0.15, 0.15*5./3., 0.37
        Pl.ax[0] = plt.axes([0.02, 0.2, 0.3, 0.3*5./3.], projection='3d')
        Pl.ax[0].view_init(10, 45, 0)
        Pl.ax[1] = plt.axes([left2, 0.13, 0.35, 0.3])  # B_xyz along direction
        Pl.ax[2] = plt.axes([left2, 0.7, q_x, q_y])    # field energy density
        Pl.ax[3] = plt.axes([left2+1.4*q_x, 0.7, q_x, q_y])    # Potential
        Pl.ax[4] = plt.axes([left2+2*1.4*q_x, 0.7, q_x, q_y])  # B_perpend
        Pl.ax[5] = plt.axes([left2+2*1.4*q_x, 0.4, q_x, q_y])  # B_upwards
        Pl.ax[6] = plt.axes([left2+2*1.4*q_x, 0.1, q_x, q_y])  # B_horizontal
        Pl.ax[7] = plt.axes([left2, 0.5, q_x, q_y*0.5],)  # orientations
        Pl.ax[8] = plt.axes([left2+1.4*q_x, 0.5, q_x, q_y*0.5])  # relaxation

    @staticmethod
    def msl(x_s, h_2, lab, end, init):
        """My slider."""
        return Slider(plt.axes([x_s, h_2, 0.09, 0.02]),
                      lab, 0, end, valfmt='%1.1f', valinit=init)

    @staticmethod
    def place_sl():
        """Place and connect the sliders."""
        if Sl.route_adj[0] is not None:
            return
        le_2, le_3, le_4, h_1, h_2 = 0.37, 0.51, 0.65, 0.005, 0.035
        for i, lab in enumerate(['$x $', '$y $', '$z $']):
            Sl.route_adj[i] = Pl.msl(le_2, h_1+i*.025, lab, 5, Mg.g.route()[i])
        _ = [slider.on_changed(Sl.route_update) for slider in Sl.route_adj]
        Sl.dp_adj = Pl.msl(le_4, h_2, r"$d_\mathrm{p}$", 4, Mg.g.d_p())
        Sl.dp_adj.on_changed(Sl.dp_update)
        Sl.length_adj = Pl.msl(le_4, h_1, r'lg$(L)$', 3,
                               np.log10(Mg.g.length()))
        Sl.length_adj.on_changed(Sl.length_update)
        for i, lab in enumerate([r'$O_x$ ', r'$O_y$ ', r'$O_z$ ']):
            Sl.origin_adj[i] = Pl.msl(le_3, h_1+i*.025, lab, 5,
                                      Mg.g.origin()[i])
        _ = [slider.on_changed(Sl.origin_update) for slider in Sl.origin_adj]

    @staticmethod
    def place_rb():
        """Place and connect the fixed (not the conditional) RadioButtons."""
        if Rb.log_lin_sel is not None:
            return
        col = 'lightgoldenrodyellow'
        Rb.rot_sel = RadioButtons(
         plt.axes([0.07, 0.005, 0.04, 0.14], fc=col),
         ['elev', 'azim', 'roll', 'fix', 'st', 'xy', 'xz', 'yz', 'exit'],
         active=3)
        Rb.rot_sel.on_clicked(Rb.rotation_update)
        Rb.log_lin_options = ['$B$', 'lg($B$)', r'$\Phi$', r'lg($\Phi$)']
        Rb.log_lin_sel = Rb.mrb(0.72, 0.13, Rb.log_lin_options, Rb.logplt)
        Rb.log_lin_sel.on_clicked(Rb.log_lin_update)
        Rb.clu_choice = RadioButtons(
            plt.axes([0.0, 0.79, 0.09, 0.21], fc=col),
            list(Mg.cluster_dic.keys()), active=Mg.clu_num)
        Rb.clu_choice.on_clicked(Rb.select_cluster)

    @staticmethod
    def place_bu():
        """Place and connect the Buttons."""
        if Bu.rlx_but is not None:
            return
        col, wid, left2, q_x, q_y = 'lightgoldenrodyellow', .02, .37, .15, .25

        def mbu(p_1, p_2, stri):
            """My Button for local use only."""
            pos = [p_1, p_2, wid, wid]
            button = Button(plt.axes(pos), stri, color=col, hovercolor='0.975')
            button.label.set_fontsize(10)
            return button
        Bu.rlx_but = mbu(left2+2.4*q_x-wid, 0.5+0.5*q_y, 'st$_{100}$')
        Bu.rlx_but.on_clicked(Bu.rlx)
        Bu.cnv_but = mbu(left2+1.4*q_x, 0.5+0.5*q_y, 'relax')
        Bu.cnv_but.on_clicked(Bu.cnv)
        tab = (q_x-wid)/6
        Bu.dst_but = mbu(left2, 0.5+0.5*q_y, 'KICK')
        Bu.dst_but.on_clicked(Bu.dst)  # disturb button
        Bu.dst_small_but = mbu(left2+tab, 0.5+0.5*q_y, 'kick')
        Bu.dst_small_but.on_clicked(Bu.dst_small)
        Bu.hal_d_but = mbu(left2+2*tab, 0.5+0.5*q_y, 'H$_<$')
        Bu.hal_d_but.on_clicked(Bu.hal_d)
        Bu.halb_but = mbu(left2+3*tab, 0.5+0.5*q_y, ' ')
        Bu.set_halbach_text()
        Bu.halb_but.on_clicked(Bu.halb)
        Bu.hal_i_but = mbu(left2+4*tab, 0.5+0.5*q_y, 'H$_>$')
        Bu.hal_i_but.on_clicked(Bu.hal_i)
        Bu.star_but = mbu(left2+5*tab, 0.5+0.5*q_y, 'star')
        Bu.star_but.on_clicked(Bu.star)
        Bu.rlxm_but = mbu(left2+6*tab, 0.5+0.5*q_y, r'K$_{100}$')
        Bu.rlxm_but.on_clicked(Bu.rlxm)
        Bu.sw_plt_but = mbu(0.965, 0.005, r'$\rightarrow$')
        Bu.sw_plt_but.on_clicked(Bu.newplt)
        Bu.sav_but = mbu(left2+q_x, 0.5, 'sav')
        Bu.sav_but.on_clicked(Bu.sav)  # write the configuration to csv-file

    @staticmethod
    def plot_layout_single_cluster():
        """Layout for  a single cluster with 9 sub-plts, sliders etc."""
        Gt.set_my_rc()
        Pl.place_rb()  # to define Rb.clu_choice.value_selected
        Pl.place_ax()
        Pl.place_sl()
        Pl.place_bu()
        Cb.place_display(Mg.disp)
        Pl.replt_ax(new_cluster=True, new_config=True)

    @staticmethod
    def flush_figs():
        """Update the figure."""
        Pl.fig.canvas.draw()  # essenial!
        Pl.fig.canvas.draw_idle()
        Pl.fig.canvas.flush_events()
        plt.pause(0.1)  # might not be neccessary

    @staticmethod
    def replt_ax(new_cluster=False, new_config=False):
        """Replot the 9 subplots."""
        Pl.flush_figs()
        if new_cluster or new_config:
            Bu.rlx_clr()
            Sc.set_conditional_controls(new_cluster=new_cluster)
        Pi.pic[0](Pl.ax[0], Mg.g)
        _ = [Pl.ax[i].set_visible(Mg.disp.get('refresh')) for i in range(1, 7)]
        if Mg.disp.get('refresh'):
            _ = [Pi.pic[i](Pl.ax[i], Mg.g) for i in range(1, 7)]
        _ = [Pi.pic[i](Pl.ax[i]) for i in range(7, 9)]
        Pl.flush_figs()

    @staticmethod
    def plot_clu(axi, c_n):
        """Plot a cluster with additional lines."""
        _ = axi.cla() if axi is not None else 0
        for xyz in [axi.xaxis, axi.yaxis, axi.zaxis]:
            xyz.set_major_locator(plt.MultipleLocator(1))
        for i, xyz in enumerate([axi.set_xlim, axi.set_ylim, axi.set_zlim]):
            xyz(-c_n.extension()[i]/2, c_n.extension()[i]/2)
        for i in range(3):
            [axi.set_xlabel('x'), axi.set_ylabel('y'), axi.set_zlabel('z')][i]
        _ = Tl.quiver_3d(axi, c_n) if Mg.disp.get('arrows') else 0
        axi.scatter(c_n.r_vec[:, 0], c_n.r_vec[:, 1],
                    c_n.r_vec[:, 2], color='grey', marker='o', s=50)
        _ = Tl.plot_contact(axi, c_n) if Mg.disp.get('contact') else 0
        _ = Tl.plot_diam(axi, c_n) if Mg.disp.get('diam') else 0
        if Mg.disp.get('sphrs'):  # show spheres around the vertices
            for pos in c_n.r_vec:
                Tl.plot_sphere(axi, c_n, pos, Sc.res_adj.val, Sc.trp_adj.val)


class Pi:
    """9 pictures."""

    pic = [None]*9

    @staticmethod
    def pic0(axi, geo):
        """Plot the 3d-figure on the left hand side."""
        Pl.plot_clu(axi, Mg.clu_act)
        _ = axi.set_axis_on() if Mg.disp.get('axes') else axi.set_axis_off()
        _ = Tl.plot_direction(axi, geo) if Mg.disp.get('direc') else 0
        if Mg.disp.get('plane'):
            Tl.plot_plane(axi, geo, Mg.clu_act.ind_x(
              geo.surface[:, :, 0], geo.surface[:, :, 1], geo.surface[:, :, 2],
              geo.direction), Sc.trp_adj.val)
        if Mg.disp.get('sphere'):
            Tl.plot_sur_sphere(axi, Mg.clu_act, Sc.res_adj.val, Sc.trp_adj.val)

    @staticmethod
    def pic1(axi, geo):
        """Plot the field as a function of position."""
        axi.cla()
        axi.set_ylabel(r"$B_\mathrm{x,y,z} /B_\mathrm{inside}$")
        direc, orig = geo.direction, geo.origin()
        axi.set_xlabel(
            r'Distance from ' +
            f'({orig[0]:3.1f}, {orig[1]:3.1f}, {orig[2]:3.1f})' +
            f' along ({direc[0]:3.2f}, {direc[1]:3.2f}, {direc[2]:3.2f})' +
            r'$\,L$')
        [Tl.b_plt, Tl.b_plt_log, Tl.pot_plt, Tl.pot_plt_log][Rb.logplt](
            axi, Mg.clu_act, geo, subtr=Cb.profile.get('-y(0)'))

    @staticmethod
    def func_2d(axi, geo, fnc, a_4):
        """Help function for 5 plots."""
        axi.cla()
        g_s, g_w = geo.surface, geo.width()
        h_3 = fnc(g_s[:, :, 0], g_s[:, :, 1], g_s[:, :, 2]) if a_4 is None\
            else fnc(g_s[:, :, 0], g_s[:, :, 1], g_s[:, :, 2], a_4)
        axi.contour(geo.x_pl, geo.y_pl, h_3, cmap=plt.get_cmap('gray'))
        axi.imshow(h_3, interpolation='bilinear', origin='lower', cmap=bgrs,
                   extent=(-g_w, g_w, -g_w, g_w))

    @staticmethod
    def pic2(axi, geo):
        """Plot the energy density in the plane."""
        Pi.func_2d(axi, geo, Mg.clu_act.b_flux_2, None)
        axi.text(0, geo.width()*1.05, 'Field Energy', ha='center', va='bottom')
        lab = f'upwards ({geo.perp[0]:3.2f}, '
        lab += f'{geo.perp[1]:3.2f}, {geo.perp[2]:3.2f})'
        axi.set_ylabel(lab)

    @staticmethod
    def pic3(axi, geo):
        """Plot the potential in the plane."""
        Pi.func_2d(axi, geo, Mg.clu_act.potential, None)
        axi.text(0, geo.width()*1.05, r'Scalar Potential $\Phi$', ha='center',
                 va='bottom')

    @staticmethod
    def lab_b(axi, wid, s_l, direc):
        """Return a string for the desciption of the field direction."""
        s_0 = f'{direc[0]:.2f}, '
        s_1 = f'{direc[1]:.2f}, '
        s_2 = f'{direc[2]:.2f}'
        lab = r'$B_\mathrm{' + s_l + '}=$'
        lab += r'$B_\mathrm{' + s_0 + s_1 + s_2 + '}$'
        axi.text(0., wid*1.01, lab, ha='center', va='bottom')

    @staticmethod
    def pic4(axi, geo):
        """Plot the field component perpendicular to the plane."""
        Pi.func_2d(axi, geo, Mg.clu_act.ind_x, geo.direction)
        Pi.lab_b(axi, geo.width(), 'perp', geo.direction)

    @staticmethod
    def pic5(axi, geo):
        """Plot the field component in upward direction."""
        Pi.func_2d(axi, geo, Mg.clu_act.ind_x, geo.perp)
        Pi.lab_b(axi, geo.width(), 'up', geo.perp)

    @staticmethod
    def pic6(axi, geo):
        """Plot the field component in horizontal direction."""
        Pi.func_2d(axi, geo, Mg.clu_act.ind_x, geo.d_cross_y)
        Pi.lab_b(axi, geo.width(), 'hor', geo.d_cross_y)
        dxy = geo.d_cross_y
        axi.set_xlabel(
            f'horizontal ({dxy[0]:3.2f}, {dxy[1]:3.2f}, {dxy[2]:3.2f})')

    @staticmethod
    def pic7(axi):
        """Plot the orientations of the dipoles in 2d."""
        Tl.plot_theta_phi(axi, Mg.clu_act)

    @staticmethod
    def pic8(axi):
        """Plot the magnetic potential energy U(relaxation steps)."""
        Tl.plot_u_steps(axi, Bu.num, Bu.pot)

    pic = [pic0, pic1, pic2, pic3, pic4, pic5, pic6, pic7, pic8]


class Sl:
    """6 fix sliders and their functions."""

    route_adj, origin_adj, dp_adj, length_adj = [None]*3, [None]*3, None, None

    @staticmethod
    def route_update(_):
        """Adjust the y-komponent of the display line (perp. to the plane)."""
        Gt.msg('route_update started')
        Mg.g.new_geo(route=[Sl.route_adj[0].val, Sl.route_adj[1].val,
                            Sl.route_adj[2].val])
        Pl.replt_ax(new_cluster=False, new_config=False)
        Gt.msg('route_update   finished')

    @staticmethod
    def origin_update(_):
        """Adjust the y-value of the origin the display line for x=0."""
        Gt.msg('origin_update started')
        Mg.g.new_geo(origin=[Sl.origin_adj[0].val, Sl.origin_adj[1].val,
                             Sl.origin_adj[2].val])
        Pl.replt_ax(new_cluster=False, new_config=False)
        Gt.msg('origin_update finished')

    @staticmethod
    def dp_update(_):
        """Adjust the distance of the plane."""
        Gt.msg('dp_update started')
        Mg.g.new_geo(d_p=Sl.dp_adj.val)
        Pl.replt_ax(new_cluster=False, new_config=False)
        Gt.msg('dp_update finished')

    @staticmethod
    def length_update(_):
        """Adjust the length of the line."""
        Gt.msg('length_update started')
        Mg.g.new_geo(length=10**Sl.length_adj.val)
        Pl.replt_ax(new_cluster=False, new_config=False)
        Gt.msg('length_update   finished')


if __name__ == '__main__':
    print(dir(Bu))
